/*
 * linux/include/asm-xtensa/semaphore.h
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 2001 - 2005 Tensilica Inc.
 */

#ifndef _XTENSA_SEMAPHORE_H
#define _XTENSA_SEMAPHORE_H

#include <asm/atomic.h>
#include <asm/system.h>
#include <linux/wait.h>

struct semaphore {
	atomic_t count;
	int sleepers;
	wait_queue_head_t wait;
};

#define __SEMAPHORE_INITIALIZER(name,n)					\
{									\
	.count		= ATOMIC_INIT(n),				\
	.sleepers	= 0,						\
	.wait		= __WAIT_QUEUE_HEAD_INITIALIZER((name).wait)	\
}

#define __DECLARE_SEMAPHORE_GENERIC(name,count) 			\
	struct semaphore name = __SEMAPHORE_INITIALIZER(name,count)

#define DECLARE_MUTEX(name) __DECLARE_SEMAPHORE_GENERIC(name,1)
#define DECLARE_MUTEX_LOCKED(name) __DECLARE_SEMAPHORE_GENERIC(name,0)

static __inline__ void sema_init (struct semaphore *sem, int val)
{
	atomic_set(&sem->count, val);
	sem->sleepers = 0;
	init_waitqueue_head(&sem->wait);
}

static __inline__ void init_MUTEX (struct semaphore *sem)
{
	sema_init(sem, 1);
}

static __inline__ void init_MUTEX_LOCKED (struct semaphore *sem)
{
	sema_init(sem, 0);
}

asmlinkage void __down(struct semaphore * sem);
asmlinkage int  __down_interruptible(struct semaphore * sem);
asmlinkage int  __down_trylock(struct semaphore * sem);
asmlinkage void __up(struct semaphore * sem);

extern spinlock_t semaphore_wake_lock;

static __inline__ void down(struct semaphore * sem)
{
	might_sleep();

	if (atomic_sub_return(1, &sem->count) < 0)
		__down(sem);
}

static __inline__ int down_interruptible(struct semaphore * sem)
{
	int ret = 0;

	might_sleep();

	if (atomic_sub_return(1, &sem->count) < 0)
		ret = __down_interruptible(sem);
	return ret;
}

static __inline__ int down_trylock(struct semaphore * sem)
{
	int ret = 0;

	if (atomic_sub_return(1, &sem->count) < 0)
		ret = __down_trylock(sem);
	return ret;
}

/*
 * Note! This is subtle. We jump to wake people up only if
 * the semaphore was negative (== somebody was waiting on it).
 */
static __inline__ void up(struct semaphore * sem)
{
	if (atomic_add_return(1, &sem->count) <= 0)
		__up(sem);
}

#endif /* _XTENSA_SEMAPHORE_H */
