// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef COMPONENTS_OFFLINE_PAGES_OFFLINE_PAGE_CLIENT_POLICY_H_
#define COMPONENTS_OFFLINE_PAGES_OFFLINE_PAGE_CLIENT_POLICY_H_

#include <stdint.h>

#include <string>

#include "base/time/time.h"

namespace offline_pages {

static const size_t kUnlimitedPages = 0;

// The struct describing the lifetime policy of offline pages.
// The following behaviors are controlled by policy:
//    a. Persistency of the offline page.
//    b. Expiration time of an offline page
//    c. Limit of number of pages offline.
struct LifetimePolicy {
  // Type of the client, indicating where the archived page would be saved
  // and whether it could be kept indefinitely.
  enum class LifetimeType {
    TEMPORARY,
    PERSISTENT,
  };

  // Type of the page generated by the client.
  LifetimeType lifetime_type;

  // The time after which the page expires.
  base::TimeDelta expiration_period;

  // The maximum number of pages allowed to be saved by the namespace.
  // kUnlimitedPages (defined above) means no limit set.
  size_t page_limit;
};

// The struct describing policies for various namespaces (Bookmark, Last-N etc.)
// used by offline page model. The name_space is supposed to be key, so that
// it's sufficient to compare name_space only when doing comparisons.
struct OfflinePageClientPolicy {
  // Namespace to which the policy applied.
  std::string name_space;

  // Policy to control the lifetime of a page generated by this namespace.
  LifetimePolicy lifetime_policy;

  // How many pages for the same online URL can be stored at any time.
  // kUnlimitedPages means there's no limit.
  size_t pages_allowed_per_url;
};

}  // namespace offline_pages

#endif  // COMPONENTS_OFFLINE_PAGES_OFFLINE_PAGE_CLIENT_POLICY_H_
