/****************************************************************************
**
** Copyright (C) 2013 Digia Plc and/or its subsidiary(-ies).
** Contact: http://www.qt-project.org/legal
**
** This file is part of the QtBluetooth module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Digia.  For licensing terms and
** conditions see http://qt.digia.com/licensing.  For further information
** use the contact form at http://qt.digia.com/contact-us.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Digia gives you certain additional
** rights.  These rights are described in the Digia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include "qdeclarativebluetoothservice_p.h"

#include <qbluetoothdeviceinfo.h>
#include <QtBluetooth/QBluetoothSocket>
#include <QtBluetooth/QBluetoothAddress>
#include <QtBluetooth/QBluetoothServer>

#include <QObject>

/* ==================== QDeclarativeBluetoothService ======================= */

/*!
    \qmltype BluetoothService
    \instantiates QDeclarativeBluetoothService
    \inqmlmodule QtBluetooth
    \since 5.2
    \brief Provides information about a particular Bluetooth service.

    \sa QBluetoothAddress
    \sa QBluetoothSocket

    It allows a QML project to get information about a remote service, or describe a service
    for a BluetoothSocket to connect to.
 */

/*!
    \qmlsignal BluetoothService::detailsChanged()

    This handler is called when any of the following properties changes:

    \list
        \li deviceAddress
        \li deviceName
        \li serviceDescription
        \li serviceName
        \li servicePort
        \li serviceProtocol
        \li serviceUuid
    \endlist
*/

class QDeclarativeBluetoothServicePrivate
{
public:
    QDeclarativeBluetoothServicePrivate()
        : m_componentComplete(false),
          m_service(0),
          m_server(0)
    {

    }

    ~QDeclarativeBluetoothServicePrivate()
    {
        delete m_service;
    }

    int listen();

    bool m_componentComplete;
    QBluetoothServiceInfo *m_service;
    QDeclarativeBluetoothService::Protocol m_protocol;
    QBluetoothServer *m_server;
};

QDeclarativeBluetoothService::QDeclarativeBluetoothService(QObject *parent) :
    QObject(parent)
{
    d = new QDeclarativeBluetoothServicePrivate;
    d->m_service = new QBluetoothServiceInfo();
}

QDeclarativeBluetoothService::QDeclarativeBluetoothService(const QBluetoothServiceInfo &service, QObject *parent)
    : QObject(parent)
{
    d = new QDeclarativeBluetoothServicePrivate;
    d->m_service = new QBluetoothServiceInfo(service);
}

QDeclarativeBluetoothService::~QDeclarativeBluetoothService()
{
    delete d;
}

void QDeclarativeBluetoothService::componentComplete()
{
    d->m_componentComplete = true;

    if (!d->m_service->isRegistered())
        setRegistered(true);
}


/*!
    \qmlproperty string BluetoothService::deviceName

    This property holds the name of the remote device. Changing this property emits the
    detailsChanged signal.
  */

QString QDeclarativeBluetoothService::deviceName() const
{

    return d->m_service->device().name();
}

/*!
    \qmlproperty string BluetoothService::deviceAddress

    This property holds the remote device's MAC address. It must be a valid address to
    connect to a remote device using a Bluetooth socket. Changing this property emits the
    detailsChanged signal.

  */

QString QDeclarativeBluetoothService::deviceAddress() const
{
    return d->m_service->device().address().toString();
}

void QDeclarativeBluetoothService::setDeviceAddress(const QString &newAddress)
{
    QBluetoothAddress address(newAddress);
    QBluetoothDeviceInfo device(address, QString(), QBluetoothDeviceInfo::ComputerDevice);
    d->m_service->setDevice(device);
}

/*!
    \qmlproperty string BluetoothService::serviceName

    This property holds the name of the remote service if available. Changing this property emits the
    detailsChanged signal.
  */

QString QDeclarativeBluetoothService::serviceName() const
{
    return d->m_service->serviceName();
}

void QDeclarativeBluetoothService::setServiceName(const QString &name)
{
    d->m_service->setServiceName(name);
}


/*!
    \qmlproperty string BluetoothService::serviceDescription

    This property holds the description provided by the remote service. Changing this property emits the
    detailsChanged signal.
  */
QString QDeclarativeBluetoothService::serviceDescription() const
{
    return d->m_service->serviceDescription();
}

void QDeclarativeBluetoothService::setServiceDescription(const QString &description)
{
    d->m_service->setServiceDescription(description);
    emit detailsChanged();
}

/*!
    \qmlproperty enumeration BluetoothService::serviceProtocol

    This property holds the protocol used for the service. Changing this property emits the
    detailsChanged signal.

    Possible values for this property are:
    \table
    \header \li Property \li Description
    \row \li \c BluetoothService.RfcommProtocol
         \li The Rfcomm protocol is used.
    \row \li \c BluetoothService.L2capProtocol
         \li The L2cap protocol is used.
    \row \li \c BluetoothService.UnknownProtocol
         \li The protocol is unknown.
    \endtable

    \sa QBluetoothServiceInfo::Protocol

  */

QDeclarativeBluetoothService::Protocol QDeclarativeBluetoothService::serviceProtocol() const
{
    return d->m_protocol;
}

void QDeclarativeBluetoothService::setServiceProtocol(QDeclarativeBluetoothService::Protocol protocol)
{
    d->m_protocol = protocol;
    emit detailsChanged();
}

/*!
    \qmlproperty string BluetoothService::serviceUuid

    This property holds the UUID of the remote service. Service UUID,
    and the address must be set to connect to a remote service. Changing
    this property emits the detailsChanged signal.

*/

QString QDeclarativeBluetoothService::serviceUuid() const
{
    return d->m_service->serviceUuid().toString();
}

void QDeclarativeBluetoothService::setServiceUuid(const QString &uuid)
{
    d->m_service->setServiceUuid(QBluetoothUuid(uuid));
    emit detailsChanged();
}

/*!
    \qmlproperty string BluetoothService::registered

    This property holds the registration/publication status of the service.  If true, the service
    is published during service discovery.
*/

bool QDeclarativeBluetoothService::isRegistered() const
{
    return d->m_service->isRegistered();
}


int QDeclarativeBluetoothServicePrivate::listen() {

    if (m_service->socketProtocol() == QBluetoothServiceInfo::UnknownProtocol) {
        qWarning() << "Unknown protocol, can't make service" << m_protocol;
        return -1;
    }
    QBluetoothServiceInfo::Protocol serverType = QBluetoothServiceInfo::UnknownProtocol;
    if (m_service->socketProtocol() == QBluetoothServiceInfo::L2capProtocol)
        serverType = QBluetoothServiceInfo::L2capProtocol;
    else if (m_service->socketProtocol() == QBluetoothServiceInfo::RfcommProtocol)
        serverType = QBluetoothServiceInfo::RfcommProtocol;

    QBluetoothServer *server = new QBluetoothServer(serverType);
    server->setMaxPendingConnections(1);
    server->listen(QBluetoothAddress());
    server->serverPort();
    m_server = server;

    return server->serverPort();
}

void QDeclarativeBluetoothService::setRegistered(bool registered)
{
    if (!d->m_componentComplete) {
        return;
    }

    delete d->m_server;
    d->m_server = 0;

    if (!registered) {
        d->m_service->unregisterService();
        emit registeredChanged();
        return;
    }

    d->listen();
    connect(d->m_server, SIGNAL(newConnection()), this, SLOT(new_connection()));


    d->m_service->setAttribute(QBluetoothServiceInfo::ServiceRecordHandle, (uint)0x00010010);

    QBluetoothServiceInfo::Sequence classId;
    classId << QVariant::fromValue(QBluetoothUuid(QBluetoothUuid::SerialPort));
    d->m_service->setAttribute(QBluetoothServiceInfo::ServiceClassIds, classId);

    //qDebug() << "name/uuid" << d->m_name << d->m_uuid << d->m_port;

    d->m_service->setAttribute(QBluetoothServiceInfo::BrowseGroupList,
                             QBluetoothUuid(QBluetoothUuid::PublicBrowseGroup));

    QBluetoothServiceInfo::Sequence protocolDescriptorList;
    QBluetoothServiceInfo::Sequence protocol;

    if (d->m_protocol == L2CapProtocol) {
        protocol << QVariant::fromValue(QBluetoothUuid(QBluetoothUuid::L2cap))
                 << QVariant::fromValue(quint16(d->m_server->serverPort()));
        protocolDescriptorList.append(QVariant::fromValue(protocol));
    }
    else if (d->m_protocol == RfcommProtocol) {
        protocol << QVariant::fromValue(QBluetoothUuid(QBluetoothUuid::Rfcomm))
                 << QVariant::fromValue(quint8(d->m_server->serverPort()));
        protocolDescriptorList.append(QVariant::fromValue(protocol));
    }
    else {
        qWarning() << "No protocol specified for bluetooth service";
    }
    d->m_service->setAttribute(QBluetoothServiceInfo::ProtocolDescriptorList,
                             protocolDescriptorList);

    if (d->m_service->registerService()) {
        emit registeredChanged();
    }
    else {
        qWarning() << "Register service failed";
        //TODO propaget this error to the user
    }
}

QBluetoothServiceInfo *QDeclarativeBluetoothService::serviceInfo() const
{
    return d->m_service;
}

void QDeclarativeBluetoothService::new_connection()
{
    emit newClient();
}

QDeclarativeBluetoothSocket *QDeclarativeBluetoothService::nextClient()
{
    QBluetoothServer *server = qobject_cast<QBluetoothServer *>(d->m_server);
    if (server) {
        if (server->hasPendingConnections()) {
            QBluetoothSocket *socket = server->nextPendingConnection();
            return new QDeclarativeBluetoothSocket(socket, this, 0);
        }
        else {
            qWarning() << "Socket has no pending connection, failing";
            return 0;
        }
    }
    return 0;
}

void QDeclarativeBluetoothService::assignNextClient(QDeclarativeBluetoothSocket *dbs)
{
    QBluetoothServer *server = qobject_cast<QBluetoothServer *>(d->m_server);
    if (server) {
        if (server->hasPendingConnections()) {
            QBluetoothSocket *socket = server->nextPendingConnection();
            dbs->newSocket(socket, this);
            return;
        }
        else {
            qWarning() << "Socket has no pending connection, failing";
            return;
        }
    }
    return;
}

