/*
    libzint - the open source barcode library
    Copyright (C) 2019-2024 Robin Stuart <rstuart114@gmail.com>

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions
    are met:

    1. Redistributions of source code must retain the above copyright
       notice, this list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright
       notice, this list of conditions and the following disclaimer in the
       documentation and/or other materials provided with the distribution.
    3. Neither the name of the project nor the names of its contributors
       may be used to endorse or promote products derived from this software
       without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
    ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
    IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
    ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
    FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
    DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
    OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
    HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
    LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
    OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
    SUCH DAMAGE.
 */
/* SPDX-License-Identifier: BSD-3-Clause */

#include "testcommon.h"

static void test_eanx_leading_zeroes(const testCtx *const p_ctx) {
    int debug = p_ctx->debug;

    struct item {
        int symbology;
        char *data;
        char *composite;
        int ret;

        int expected_rows;
        int expected_width;
    };
    /* s/\/\*[ 0-9]*\*\//\=printf("\/\*%3d*\/", line(".") - line("'<")): */
    static const struct item data[] = {
        /* 0*/ { BARCODE_EANX_CC, "1", "[21]A12345678", 0, 8, 72 }, /* EAN-8 */
        /* 1*/ { BARCODE_EANX_CC, "12", "[21]A12345678", 0, 8, 72 },
        /* 2*/ { BARCODE_EANX_CC, "123", "[21]A12345678", 0, 8, 72 },
        /* 3*/ { BARCODE_EANX_CC, "1234", "[21]A12345678", 0, 8, 72 },
        /* 4*/ { BARCODE_EANX_CC, "12345", "[21]A12345678", 0, 8, 72 },
        /* 5*/ { BARCODE_EANX_CC, "123456", "[21]A12345678", 0, 8, 72 },
        /* 6*/ { BARCODE_EANX_CC, "1234567", "[21]A12345678", 0, 8, 72 },
        /* 7*/ { BARCODE_EANX_CC, "12345678", "[21]A12345678", 0, 7, 99 }, /* EAN-13 */
        /* 8*/ { BARCODE_EANX_CC, "1+12", "[21]A12345678", 0, 8, 98 }, /* EAN-8 + EAN-2 */
        /* 9*/ { BARCODE_EANX_CC, "12+12", "[21]A12345678", 0, 8, 98 },
        /*10*/ { BARCODE_EANX_CC, "123+12", "[21]A12345678", 0, 8, 98 },
        /*11*/ { BARCODE_EANX_CC, "1234+12", "[21]A12345678", 0, 8, 98 },
        /*12*/ { BARCODE_EANX_CC, "12345+12", "[21]A12345678", 0, 8, 98 },
        /*13*/ { BARCODE_EANX_CC, "123456+12", "[21]A12345678", 0, 8, 98 },
        /*14*/ { BARCODE_EANX_CC, "1234567+12", "[21]A12345678", 0, 8, 98 },
        /*15*/ { BARCODE_EANX_CC, "12345678+12", "[21]A12345678", 0, 7, 125 }, /* EAN-13 + EAN-2 */
        /*16*/ { BARCODE_EANX_CC, "1+123", "[21]A12345678", 0, 8, 125 }, /* EAN-8 + EAN-5 */
        /*17*/ { BARCODE_EANX_CC, "12+123", "[21]A12345678", 0, 8, 125 },
        /*18*/ { BARCODE_EANX_CC, "123+123", "[21]A12345678", 0, 8, 125 },
        /*19*/ { BARCODE_EANX_CC, "1234+123", "[21]A12345678", 0, 8, 125 },
        /*20*/ { BARCODE_EANX_CC, "12345+123", "[21]A12345678", 0, 8, 125 },
        /*21*/ { BARCODE_EANX_CC, "123456+123", "[21]A12345678", 0, 8, 125 },
        /*22*/ { BARCODE_EANX_CC, "1234567+123", "[21]A12345678", 0, 8, 125 },
        /*23*/ { BARCODE_EANX_CC, "12345678+123", "[21]A12345678", 0, 7, 152 }, /* EAN-13 + EAN-5 */
        /*24*/ { BARCODE_EANX_CC, "1234567890128", "[21]A12345678", 0, 7, 99 }, /* EAN-13 + CHK */
        /*25*/ { BARCODE_EANX_CC, "1234567890128+12", "[21]A12345678", 0, 7, 125 }, /* EAN-13 + CHK + EAN-2 */
        /*26*/ { BARCODE_EANX_CC, "1234567890128+12345", "[21]A12345678", 0, 7, 152 }, /* EAN-13 + CHK + EAN-5 */
    };
    const int data_size = ARRAY_SIZE(data);
    int i, length, composite_length, ret;
    struct zint_symbol *symbol = NULL;

    testStartSymbol("test_eanx_leading_zeroes", &symbol);

    for (i = 0; i < data_size; i++) {

        if (testContinue(p_ctx, i)) continue;

        symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        length = testUtilSetSymbol(symbol, data[i].symbology, -1 /*input_mode*/, -1 /*eci*/, -1 /*option_1*/, -1, -1, -1 /*output_options*/, data[i].data, -1, debug);
        assert_zero(length >= 128, "i:%d length %d >= 128\n", i, length);
        strcpy(symbol->primary, data[i].data);

        composite_length = (int) strlen(data[i].composite);

        ret = ZBarcode_Encode(symbol, (const unsigned char *) data[i].composite, composite_length);
        assert_equal(ret, data[i].ret, "i:%d ZBarcode_Encode ret %d != %d (%s)\n", i, ret, data[i].ret, symbol->errtxt);

        assert_equal(symbol->rows, data[i].expected_rows, "i:%d symbol->rows %d != %d\n", i, symbol->rows, data[i].expected_rows);
        assert_equal(symbol->width, data[i].expected_width, "i:%d symbol->width %d != %d\n", i, symbol->width, data[i].expected_width);

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

static void test_helper_generate(const struct zint_symbol *symbol, int ret, int i, char *data, char *composite, int option_1, char *comment, int bwipp_cmp) {
    char esc_data[1024];
    char esc_composite[4096];

    testUtilEscape(data, (int) strlen(data), esc_data, sizeof(esc_data));
    testUtilEscape(composite, (int) strlen(composite), esc_composite, sizeof(esc_composite));

    if (ret == 0) {
        if (bwipp_cmp == -1) {
            printf("        /*%2d*/ { %s, %d, \"%s\", \"%s\", %d, %d, %d, \"%s\",\n",
                    i, testUtilBarcodeName(symbol->symbology), option_1, esc_data, esc_composite, ret, symbol->rows, symbol->width, comment);
        } else {
            printf("        /*%2d*/ { %s, %d, \"%s\", \"%s\", %d, %d, %d, %d, \"%s\",\n",
                    i, testUtilBarcodeName(symbol->symbology), option_1, esc_data, esc_composite, ret, symbol->rows, symbol->width, bwipp_cmp, comment);
        }
        testUtilModulesPrint(symbol, "                    ", "\n");
        printf("               },\n");
    } else {
        if (bwipp_cmp == -1) {
            printf("        /*%2d*/ { %s, %d, \"%s\", \"%s\", %s, %d, %d, \"%s\", \"\" },\n",
                    i, testUtilBarcodeName(symbol->symbology), option_1, esc_data, esc_composite, testUtilErrorName(ret), symbol->rows, symbol->width, comment);
        } else {
            printf("        /*%2d*/ { %s, %d, \"%s\", \"%s\", %s, %d, %d, %d, \"%s\", \"\" },\n",
                    i, testUtilBarcodeName(symbol->symbology), option_1, esc_data, esc_composite, testUtilErrorName(ret), symbol->rows, symbol->width, bwipp_cmp, comment);
        }
    }
}

/* Replicate examples from GS1 General Specifications 21.0.1 and ISO/IEC 24723:2010 */
static void test_examples(const testCtx *const p_ctx) {
    int debug = p_ctx->debug;

    struct item {
        int symbology;
        int input_mode;
        int option_1;
        char *data;
        char *composite;
        int ret;

        int expected_rows;
        int expected_width;
        int bwipp_cmp;
        char *comment;
        char *expected;
    };
    /* Verified manually against GS1 General Specifications 21.0.1 (GGS) and ISO/IEC 24723:2010, with noted exceptions, and verified via bwipp_dump.ps against BWIPP */
    static const struct item data[] = {
        /*  0*/ { BARCODE_DBAR_OMNSTK_CC, -1, 1, "0401234567890", "[17]050101[10]ABC123", 0, 11, 56, 1, "GSS Figure 5.1-5. GS1 DataBar Stacked Omnidirectional barcode with a Composite Component",
                    "01101100110101110001001100001000000110100111011110101001"
                    "01101101110110001100010100001100001000010110011100101001"
                    "01101101100111000101110001101001100011111010011101101001"
                    "01101101000110111100010011001111110100111011011101001001"
                    "01101001000111110111110100101010100000010000011101001101"
                    "00000000100110111010100000101010101100110001100000000000"
                    "01001111011001000100011111000001010011001110011010000000"
                    "00000000100110111010100000101010101100110001100000000000"
                    "00000101010101010101010101010101010101010101010000000000"
                    "00001000110000101010000000101010111011001111000000000000"
                    "10100111001111010101111111000001000100110000110101000000"
                },
        /*  1*/ { BARCODE_DBAR_LTD_CC, -1, 1, "1311234567890", "[17]010615[10]A123456", 0, 6, 79, 1, "GGS Figure 5.11.2-1. (24723:2010 Figure 1) GS1 DataBar Limited Composite symbol with CC-A",
                    "0111100010110110001010011000111110100110011101110100111100111011101001101000000"
                    "0100111110001101001010111000111101111010111101001111011100111011101011101000000"
                    "0100110011010000001010110000111001000110111101111011110010100011101011001000000"
                    "0111100010101000001010010000101110110111111001110011001110010011101010001000000"
                    "0000001100000101010011001110101011010100110010101111000101100110100111000000000"
                    "0101110011111010101100110001010100101011001101010000111010011001011000010100000"
                },
        /*  2*/ { BARCODE_GS1_128_CC, -1, 3, "[01]03812345678908", "[10]ABCD123456[410]3898765432108", 0, 7, 154, 1, "GGS Figure 5.11.2-2. GS1-128 Composite symbol with CC-C **NOT SAME** as zint uses encodation '10', same if '0' forced",
                    "1111111101010100011110101011110000111101011001111101110111110111010010000010000100010110010000101100001111011110110011011110101001111000111111101000101001"
                    "1111111101010100011111101010001110100001000111101001100101110010000011100001011000100100100111110110001011100001011111011111101010111000111111101000101001"
                    "1111111101010100011101010011111100110001111010001101000101011110000010001111101100010111101101111101001001011000111110011101010001111110111111101000101001"
                    "1111111101010100010101111001111000110010011001110001111010100111100010011110111101000110000110001000101100001011101111011111010111111010111111101000101001"
                    "1111111101010100011101011100001100101000111111011101011110001001111011111011001000100111100111011101001101101111001000011101011100110000111111101000101001"
                    "0000000001011000110000101000100110010011011011001110110100001100010010001010001001110111101001100100100001011100110110100001000100100001001001110001010000"
                    "0000000110100111001111010111011001101100100100110001001011110011101101110101110110001000010110011011011110100011001001011110111011011110110110001110101100"
                },
        /*  3*/ { BARCODE_GS1_128_CC, -1, 3, "[01]93812345678901", "[10]ABCD123456[410]3898765432108", 0, 7, 154, 1, "24723:2010 Figure 2 GS1-128 Composite symbol with 5-row CC-C **NOT SAME** ditto as above",
                    "1111111101010100011110101011110000111101011001111101110111110111010010000010000100010110010000101100001111011110110011011110101001111000111111101000101001"
                    "1111111101010100011111101010001110100001000111101001100101110010000011100001011000100100100111110110001011100001011111011111101010111000111111101000101001"
                    "1111111101010100011101010011111100110001111010001101000101011110000010001111101100010111101101111101001001011000111110011101010001111110111111101000101001"
                    "1111111101010100010101111001111000110010011001110001111010100111100010011110111101000110000110001000101100001011101111011111010111111010111111101000101001"
                    "1111111101010100011101011100001100101000111111011101011110001001111011111011001000100111100111011101001101101111001000011101011100110000111111101000101001"
                    "0000000001011000110000101000100110010011010111000010110100001100010010001010001001110111101001100100100001001100100110100001000100001001001001110001010000"
                    "0000000110100111001111010111011001101100101000111101001011110011101101110101110110001000010110011011011110110011011001011110111011110110110110001110101100"
                },
        /*  4*/ { BARCODE_EANX_CC, -1, 1, "331234567890", "[21]1234-abcd", 0, 7, 99, 1, "GGS Figure 5.11.8-1. EAN-13 symbol with a four-column CC-A component (note [21] not [99])",
                    "110110111011010000100000110100110011101100001001110100100001011001100001100111000110001011011000101"
                    "110110110011000110111100010111011001110000101001100100100000010111101001101011100010000011001000101"
                    "110110100010011010001110000111111010001100101001100110111111010001101001010000011011111011101000101"
                    "000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010"
                    "001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001"
                    "000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010"
                    "000101011110100110010011011010000100111010110001010101010000100010010010001110100111001010000101010"
                },
        /*  5*/ { BARCODE_EANX_CC, -1, 1, "331234567890", "[99]1234-abcd", 0, 7, 99, 1, "24723:2010 Figure 5 An EAN-13 composite symbol (with CC-A)",
                    "110110111011100110111011110100010100000010001001110100111011010110000001100110010000100011011000101"
                    "110110110011100010011101100111110001000101101001100100100001101011111101101011100010000011001000101"
                    "110110100010001011101111110110011100100011101001100110100000011101011001011110001001000011101000101"
                    "000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010"
                    "001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001"
                    "000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010"
                    "000101011110100110010011011010000100111010110001010101010000100010010010001110100111001010000101010"
                },
        /*  6*/ { BARCODE_UPCA_CC, -1, 2, "61414101234", "[91]abcdefghijklmnopqrstuvwxyz", 0, 14, 99, 1, "GGS Figure 5.11.8-2. UPC-A symbol with a four-column CC-B component **NOT SAME** (using [91] not [10] as length > 20 max for [10])",
                    "110001001010000001110010110110011111101100101001111010100100101111000001110101001111110011000100101"
                    "111001001011101110101000000111101101000111001011111010100011000110000101110011010000110011100100101"
                    "111101001011110110001101000111101000100000101011110010101001111001000001011111010001110011110100101"
                    "111101011011110100111100010111011111001011001011110110111110001001110101011111001110011011110101101"
                    "111101010010001100011100110111001000010011101001110110100011011010000001001100000110001011110101001"
                    "111001010010000001011110100111100110110100001001110100111001011000010001111010010100000011100101001"
                    "111011010010011111101110100101110001000011001001100100100011110010000101011110000110011011101101001"
                    "111010010011100011101000100100001011011000001001100110110001110110100001000110011101100011101001001"
                    "111010011010110111111011000101111101100111101001000110110100111100001101111100001101010011101001101"
                    "111010111010111000001000110110111110010001001001000010111100101111101101000110001000111011101011101"
                    "000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010"
                    "001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001"
                    "000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010"
                    "000101010111100110010100011001100101000110011001010101110010110011011011001000010101110010000101010"
                },
        /*  7*/ { BARCODE_UPCA_CC, GS1NOCHECK_MODE, 2, "61414101234", "[10]abcdefghijklmnopqrstuvwxyz", 0, 14, 99, 1, "GGS Figure 5.11.8-2. UPC-A symbol with a four-column CC-B component **NOT SAME** as zint uses encodation '10', same if '0' forced",
                    "110001001010000001110010110110011111101100101001111010111000101011111101011010001110000011000100101"
                    "111001001010110011100011000100010000100000101011111010111000111010010001011000111101110011100100101"
                    "111101001011000001000111010111010000110100001011110010111011001111001101110101100000010011110100101"
                    "111101011010001000100011110101111101111011101011110110100011111011001001000001010011110011110101101"
                    "111101010010111011100001000101000110000110001001110110101001100110000001000100000010100011110101001"
                    "111001010010111110110011110100010111111011101001110100110100111000001001101011100010000011100101001"
                    "111011010011101000101111110100111100010010001001100100111000010111110101110100010111111011101101001"
                    "111010010011010010001100000100001011100011101001100110100111101111010001110011100010100011101001001"
                    "111010011011110010100100000110000001011100101001000110101101011111100001110010011101100011101001101"
                    "111010111010111000100000110111110110111100101001000010101110000011011101011100001101110011101011101"
                    "000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010"
                    "001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001"
                    "000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010"
                    "000101010111100110010100011001100101000110011001010101110010110011011011001000010101110010000101010"
                },
        /*  8*/ { BARCODE_EANX_CC, -1, -1, "1234567", "[21]A12345678", 0, 8, 72, 1, "GGS Figure 5.11.8-3. (24723:2010 Figure 4) EAN-8 symbol with a three-column CC-A",
                    "101001111000001001010011000111110101110111101001101001111110011101001101"
                    "111110010011100101010111000101110011011100001111110100011001011101011101"
                    "110011001000010001010110000101000001000010001001000110110000011101011001"
                    "101011110010000001010010000111011100111101001111001001000010011101010001"
                    "000010000000000000000000000000000000000000000000000000000000000000000010"
                    "000100000000000000000000000000000000000000000000000000000000000000000001"
                    "000010000000000000000000000000000000000000000000000000000000000000000010"
                    "000010100110010010011011110101000110101010011101010000100010011100101010"
                },
        /*  9*/ { BARCODE_UPCE_CC, -1, 1, "0121230", "[15]021231", 0, 9, 55, 1, "GGS Figure 5.11.8-4. (24723:2010 Figure 3) UPC-E symbol with a two-column CC-A",
                    "1101100110111010011111010001100111100010110011110101001"
                    "1101101110110010010000110001101000011011100011100101001"
                    "1101101100111101001000000101000101111000010011101101001"
                    "1101101000111011110100011101001110011011100011101001001"
                    "1101001000111000010110111101001110001110010011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010110011001001100110010011011011110101001110101010"
                },
        /* 10*/ { BARCODE_DBAR_OMN_CC, -1, 1, "0361234567890", "[11]990102", 0, 5, 100, 1, "GGS Figure 5.11.8-5. (24723:2010 Figure 8) GS1 DataBar Omnidirectional symbol with a four-column CC-A",
                    "1101101110110000101000110001111001010111100010011101001110011101100110011001001100111000110110001010"
                    "1101101100110111011111001001000011010111111010011001001101000000111010010010111111001110110010001010"
                    "1101101000110010010111110001011001101111000010011001101111010011110010010000011001011100111010001010"
                    "0000000000010110001110100000000101001011010111111011001101010000011010000000010100101000110011110000"
                    "0000010011101001110001001111111000010100101000000100110010101111100101111111100011010111001100001101"
                },
        /* 11*/ { BARCODE_DBAR_STK_CC, -1, 1, "0341234567890", "[17]010200", 0, 9, 56, 1, "GGS Figure 5.11.8-6. (24723:2010 Figure 6) GS1 DataBar Stacked symbol with a two-column CC-A",
                    "01101100110101110011100111101010000100001111011110101001"
                    "01101101110110110001000010001110111101100100011100101001"
                    "01101101100110100001111011001111110011010110011101101001"
                    "01101101000100110000101110001011100000110111011101001001"
                    "01101001000101110111110111001001100110100000011101001101"
                    "00000001011011100010000010101010100000101101010000000000"
                    "01001110100100011101111100000001011111010010100010000000"
                    "00000011010111101010000010101010101001001101010000000000"
                    "10101100111000010101111111110111000110110011100101000000"
                },
        /* 12*/ { BARCODE_DBAR_LTD_CC, -1, 2, "0351234567890", "[91]abcdefghijklmnopqrstuv", 0, 17, 88, 1, "GGS Figure 5.11.8-7. (24723:2010 Figure 7) GS1 DataBar Limited symbol with a three-column CC-B **NOT SAME** (using [91] not [21] as length > 20 max for [21])",
                    "1101110100111011111011101001011000100100000100001000101111101101001111011011101001000000"
                    "1101100100111111010100111001011000110111101100001100101010000111100010011011001001000000"
                    "1101100110100111100001001001010000110111011100101111001101110010001110011011001101000000"
                    "1101101110111010000010011101010001110111010000100111001000011001100100011011011101000000"
                    "1101101100111101001000000101010001100100011011111001001111101100100010011011011001000000"
                    "1101101000101111100111001101010011100101111100011101101000101000011110011011010001000000"
                    "1101001000100011011010000001010011000100010000001000101101011100011110011010010001000000"
                    "1101011000101001111110011101010111000111101001010000001110101100100000011010110001000000"
                    "1101011100101111100001100101010110000110001111000101101111110100110100011010111001000000"
                    "1101011110110001000010001101010010000110001001011000001000100001010000011010111101000000"
                    "1101001110100000100111100101011010000111100101000001001111010000100010011010011101000000"
                    "1101001100110011100100111001001010000101100011101111101000011000100111011010011001000000"
                    "1101000100101111000111101001001011000110000100001011001010000110001100011010001001000000"
                    "1101000110111011011100000101001011100110100001110000101111011011100110011010001101000000"
                    "1101000010111110100111010001011011100100101100011111001011111011100011011010000101000000"
                    "0000000000000111101110001101000111010101011010100110010111010010011100011010111000000000"
                    "0000000000101000010001110010111000101010100101011001101000101101100011100101000010100000"
                },
        /* 13*/ { BARCODE_DBAR_LTD_CC, GS1NOCHECK_MODE, 2, "0351234567890", "[21]abcdefghijklmnopqrstuv", 0, 17, 88, 1, "GGS Figure 5.11.8-7. (24723:2010 Figure 7) GS1 DataBar Limited symbol with a three-column CC-B, same",
                    "1101110100111011111011101001011000100100000100001000101111010011011111011011101001000000"
                    "1101100100101111100101110001011000110111110001010011001111101100001001011011001001000000"
                    "1101100110101111000110110001010000110111011100101111001101110010001110011011001101000000"
                    "1101101110111010000010011101010001110111010000100111001000011001100100011011011101000000"
                    "1101101100111101001000000101010001100100011011111001001111101100100010011011011001000000"
                    "1101101000101111100111001101010011100101111100011101101000101000011110011011010001000000"
                    "1101001000100011011010000001010011000100010000001000101101011100011110011010010001000000"
                    "1101011000101001111110011101010111000111101001010000001110101100100000011010110001000000"
                    "1101011100110100001001111101010110000101111000111011101011100000100110011010111001000000"
                    "1101011110110001101100001101010010000100110011101100001101110001110011011010111101000000"
                    "1101001110110100001110000101011010000100100111101000001011110010001111011010011101000000"
                    "1101001100111001101011111001001010000111000101111100101011111011000001011010011001000000"
                    "1101000100110100010001100001001011000111101001001111001001001111011110011010001001000000"
                    "1101000110100010001111000101001011100111011011100000101111100110101000011010001101000000"
                    "1101000010111110010111000101011011100100001001011110001011111000110100011010000101000000"
                    "0000000000000111101110001101000111010101011010100110010111010010011100011010111000000000"
                    "0000000000101000010001110010111000101010100101011001101000101101100011100101000010100000"
                },
        /* 14*/ { BARCODE_DBAR_EXP_CC, -1, 1, "[01]93712345678904[3103]001234", "[91]1A2B3C4D5E", 0, 5, 151, 1, "GGS Figure 5.11.8-8. (24723:2010 Figure 9) GS1 DataBar Expanded symbol with a four-column CC-A, same, verified against BWIPP and tec-it",
                    "0011011011101110011010011000011100011100110110100111010011010001000011000101101110011000001101100010100000000000000000000000000000000000000000000000000"
                    "0011011011001101110111110100011010001111001100100110010010111111001001100100101111110011101100100010100000000000000000000000000000000000000000000000000"
                    "0011011010001010111011111100011111011011110010100110011011000011010011110100001011001111101110100010100000000000000000000000000000000000000000000000000"
                    "0000011011111011000100000000101001010000011101001110100110001100111101000010101000011010001110001000100001010000111001010000001010010111000110010110000"
                    "0101100100000100111011111111000010101111100010110001011001110011000010111100000011100101110001110111011110101111000110001111110000101000111001101000010"
                },
        /* 15*/ { BARCODE_GS1_128_CC, -1, 1, "[01]03212345678906", "[21]A1B2C3D4E5F6G7H8", 0, 6, 145, 1, "GGS Figure 5.11.8-9. (24723:2010 Figure 11) GS1-128 symbol with a four-column CC-A",
                    "0000000000000000000001101001000110100001000001101101011110111110010010001101010000010010000011101110100010000111011001010000000000000000000000000"
                    "0000000000000000000001101011000110101111001100001111010001101100010010000101111000011001101011100101100001000110011001010000000000000000000000000"
                    "0000000000000000000001101011100100011001100111101011000101110000010110000101001100110011110011011110011001110110111001010000000000000000000000000"
                    "0000000000000000000001101011110111000111011011001110010001011100010111000101011000011100110010000100000100010110111101010000000000000000000000000"
                    "0010110001100001010001001100100110110110011100100011011000100100010100010011101111010011001001000010110011011100010100001000100010010011100010100"
                    "1101001110011110101110110011011001001001100011011100100111011011101011101100010000101100110110111101001100100011101011110111011101101100011101011"
                },
        /* 16*/ { BARCODE_DBAR_EXPSTK_CC, -1, 1, "[01]00012345678905[10]ABCDEF", "[21]12345678", 0, 13, 102, 1, "24723:2010 Figure 10 — A GS1 DataBar Expanded Stacked Composite symbol (with CC-A) **NOT SAME** bottom 1st and top 2nd linear row separators different; zint same as BWIPP and hard to see how figure in standard could be correct",
                    "001101101110110100001000001101001100111011000010011101001000110011100110010100111011100000110110001010"
                    "001101101100101111110100011001111101101000001010011001001011111011011110011010111000100000110010001010"
                    "001101101000100101001111000001000111011101111010011001101011110110110000011010001011111000111010001010"
                    "000001000111011100010000000010100100001101100000101010000110010000010100000101010001110100111100100000"
                    "010110111000100011101111111100001011110010011111010101111001101111101011111000000110001011000011011101"
                    "000001000111011100010000000010100100001101100000101010000110010000010100000101010001110100111100100000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000000110001000010100001000000001010010000001000110101111011001110001001010000001010000101001000010000"
                    "101111001110111101011100111111110101101111110111001010000100110001110100001111110001111010110111100010"
                    "000000110001000010100001000000001010010000001000110101111011001110001001010000001010000101001000010000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000000111001111101010100001010100101011111000010100000000000000000000000000000000000000000000000000000"
                    "010111000110000010100011110000001010100000111101000100000000000000000000000000000000000000000000000000"
                },
        /* 17*/ { BARCODE_GS1_128_CC, -1, 3, "[00]030123456789012340", "[02]13012345678909[37]24[10]1234567ABCDEFG", 0, 7, 174, 1, "24723:2010 Figure 12 — A GS1-128 Composite symbol (with CC-C)",
                    "111111110101010001111010101111000011010111011110000111011111011101001000001000010001011110101100111110111010010001110001000100011000011011111010100111110111111101000101001000"
                    "111111110101010001111110101000111010000100111101000110011110101111101111010001010000011111000110010100111001011100011001001001111101100011111101010111000111111101000101001000"
                    "111111110101010001010100011110000011001111100001010110100010111110001110111101011100011000001101011110101111001000000101100001011111101011101010001111110111111101000101001000"
                    "111111110101010001010111100111100011110100001011110101011111011111001101010000110000011010011100011110101111001111010001100011101101000011101001011100000111111101000101001000"
                    "111111110101010001110101110000110011111010100011000100100001111000101110000001011001011110010110001100111100101101100001111011000110100011101011100110000111111101000101001000"
                    "000000000101100011000010100010010011001101101100111001100100110001001000101000100111011110100110010010000100110010011000100100010011101011101000010001000100001010011100010100"
                    "000000011010011100111101011101101100110010010011000110011011001110110111010111011000100001011001101101111011001101100111011011101100010100010111101110111011110101100011101011"
                },
        /* 18*/ { BARCODE_DBAR_STK_CC, -1, 1, "12345678901231", "[91]12345678901234567890", 0, 10, 56, 1, "Example with CC-A 2 cols, 6 rows",
                    "01100100010111100110100111001011101110001000011100101101"
                    "01110100010110001011101000001000111010111110011000101101"
                    "01110110010110101100111111001000111100001001011000101001"
                    "01100110010111100100010111101101100100100000011001101001"
                    "01101110010111110011010010001011111101000011011011101001"
                    "01101111010111111001011010001000001100010111011011001001"
                    "00000110001100011010100000000100101101110000100000000000"
                    "01001001110011100100011111111001010010001111011010000000"
                    "00000110001101011010101010101010101101010000100000000000"
                    "10101001111010000101100000000111000010100111011101000000"
                },
        /* 19*/ { BARCODE_DBAR_OMNSTK_CC, -1, 1, "12345678901231", "[91]1234567890123456789012", 0, 13, 56, 1, "Example with CC-A 2 cols, 7 rows",
                    "01110110110100100011111001101110001011100110011100010101"
                    "01110010110111000110101111101001111100110010011000010101"
                    "01100010110111010110011110001110001110110011011000110101"
                    "01100010100110010111101100001001011111001100011000100101"
                    "01100110100101110000001011001110111010011110011100100101"
                    "01101110100110100000010001101100011010010000011110100101"
                    "01101100100111001011000100001111000001001010011110101101"
                    "00000110001100011010100000000100101101110000100000000000"
                    "01001001110011100100011111111001010010001111011010000000"
                    "00000110001100011010100000000100101101110000100000000000"
                    "00000101010101010101010101010101010101010101010000000000"
                    "00000110000101111010010101010000111101011000100000000000"
                    "10101001111010000101100000000111000010100111011101000000"
                },
        /* 20*/ { BARCODE_UPCE_CC, -1, 1, "1234567", "[91]1234567890123456789012345678", 0, 12, 55, 1, "Example with CC-A 3 cols, 8 rows",
                    "1110111010100100011111001101110001011100110011011011101"
                    "1110011010111000110101111101001111100110010011011011001"
                    "1111011010111010110011110001110001110110011011011010001"
                    "1111001010110010111101100001101101111000010011010010001"
                    "1110001010100011111011001001000000100101111011010110001"
                    "1100001010111001111011000101110011001100111011010111001"
                    "1100011010111011100111100101111110011100101011010111101"
                    "1100010010101111100011010001111110010111011011010011101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
                },
        /* 21*/ { BARCODE_DBAR_STK_CC, -1, 1, "12345678901231", "[91]1234567890123456789012345678901", 0, 13, 56, 1, "Example with CC-A 2 cols, 9 rows",
                    "01100011010100100011111001101110001011100110011010111101"
                    "01100010010111000110101111101001111100110010011010011101"
                    "01110010010111010110011110001110001110110011011010011001"
                    "01111010010110010111101100001111001000101000011010001001"
                    "01111010110100111000010011001000111110001101011010001101"
                    "01111010100111000100010011101010000010000100011010000101"
                    "01110010100101100111111011001100010111000010011011000101"
                    "01110110100100011001000111001000010010111100011001000101"
                    "01110100100111001110011001101011000011101100011101000101"
                    "00000110001100011010100000000100101101110000100000000000"
                    "01001001110011100100011111111001010010001111011010000000"
                    "00000110001101011010101010101010101101010000100000000000"
                    "10101001111010000101100000000111000010100111011101000000"
                },
        /* 22*/ { BARCODE_DBAR_OMNSTK_CC, -1, 1, "12345678901231", "[91]1234567890123456789012345678901234567", 0, 16, 56, 1, "Example with CC-A 2 cols, 10 rows",
                    "01101001000111100110100111001011101110001000011101001101"
                    "01101011000110001011101000001000111010111110011101011101"
                    "01101011100110101100111111001000111100001001011101011001"
                    "01101011110111100100010111101000100110000110011101010001"
                    "01101001110111000101110110001111011011100110011001010001"
                    "01101001100101111100001100101000111100010100011001011001"
                    "01101000100101000001101100001111101111001011011001011101"
                    "01101000110111000101100100001111001001000001011001001101"
                    "01101000010101100111011111001000011101001100011001101101"
                    "01101100010110001101010000001000100001110111011101101101"
                    "00000110001100011010100000000100101101110000100000000000"
                    "01001001110011100100011111111001010010001111011010000000"
                    "00000110001100011010100000000100101101110000100000000000"
                    "00000101010101010101010101010101010101010101010000000000"
                    "00000110000101111010010101010000111101011000100000000000"
                    "10101001111010000101100000000111000010100111011101000000"
                },
        /* 23*/ { BARCODE_UPCE_CC, -1, 1, "1234567", "[91]123456789012345678901234567890123456789012334", 0, 16, 55, 1, "Example with CC-A 2 cols, 12 rows",
                    "1110010100100100011111001101110001011100110011011000101"
                    "1110110100111000110101111101001111100110010011001000101"
                    "1110100100111010110011110001110001110110011011101000101"
                    "1110100110110010111101100001001000011110001011101100101"
                    "1110101110100011010011100001100010100111110011001100101"
                    "1110101100111101111001001001100001101100011011011100101"
                    "1110101000111010001100010001101111010111110011011110101"
                    "1100101000100110000001011101000111100100010011001110101"
                    "1100101100100010011101110001110000110100110011101110101"
                    "1100101110111101000000100101111001010000001011100110101"
                    "1100100110111110010111101101100011100100111011110110101"
                    "1100110110100011101110100001000110011101100011110010101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
                },
        /* 24*/ { BARCODE_DBAR_LTD_CC, -1, 1, "12345678901231", "[91]1234567890123456789012345", 0, 7, 79, 1, "Example with CC-A 3 cols, 5 rows",
                    "0111100110100111001001101000101110111000100001011101110010000011000010101000000"
                    "0100011101011111001011101000111010110010000001100111101011111011000110101000000"
                    "0101100111011111001011001000110110000010111101110001011111001011000100101000000"
                    "0101100111100011101011001100110011000011001101000100011001100011100100101000000"
                    "0111110101011000001011000100110001011110001101101110111110100011110100101000000"
                    "0000101110100000101011100000101010100011011010110101011010111110001110000000000"
                    "0101010001011111010100011111010101011100100101001010100101000001110001110100000"
                },
        /* 25*/ { BARCODE_EANX_CC, -1, 1, "1234567", "[91]1234567890123456789012345678901", 0, 10, 72, 1, "Example with CC-A 3 cols, 6 rows",
                    "100100011111001101011000100111000101110011001100010111010000011110100101"
                    "100111110011001001011000110110101100111111001000111100001001011110101101"
                    "111100100010111101010000110111011000001011001000010100001000011110101001"
                    "101111110010011001010001110111100010110011001111101000110111011100101001"
                    "101110111000111101010001100100001001011110001000110010001110011101101001"
                    "101110111000010001010011100111001110011001101011000011101100011101001001"
                    "000010000000000000000000000000000000000000000000000000000000000000000010"
                    "000100000000000000000000000000000000000000000000000000000000000000000001"
                    "000010000000000000000000000000000000000000000000000000000000000000000010"
                    "000010100110010010011011110101000110101010011101010000100010011100101010"
                },
        /* 26*/ { BARCODE_DBAR_LTD_CC, -1, 1, "12345678901231", "[91]1234567890123456789012345678901234567", 0, 9, 79, 1, "Example with CC-A 3 cols, 7 rows",
                    "0100010001101111101011010000110001101000111101110001101011111011001010001000000"
                    "0110000010011011101001010000111010110011110001110001110110011011001011001000000"
                    "0110010111101100001001011000100100001111000101110001011101100011001011101000000"
                    "0110001010011111001001011100101111100001100101000111100010100011001001101000000"
                    "0101000001101100001011011100111110111100101101100001101100110011001101101000000"
                    "0110010000111010001011011110101111101001110001011000010111111011101101101000000"
                    "0101000110001111101011001110101100110011110001001101101111000011100101101000000"
                    "0000101110100000101011100000101010100011011010110101011010111110001110000000000"
                    "0101010001011111010100011111010101011100100101001010100101000001110001110100000"
                },
        /* 27*/ { BARCODE_DBAR_LTD_CC, -1, 1, "12345678901231", "[91]123456789012345678901234567890123456789012345", 0, 10, 79, 1, "Example with CC-A 3 cols, 8 rows",
                    "0100010001101111101011001110110001101000111101110001101011111011100101101000000"
                    "0110000010011011101001001110111010110011110001110001110110011011000101101000000"
                    "0110010111101100001001101110100100001111000101110001011101100011000101001000000"
                    "0110001010011111001000101110101111100001100101000111100010100011001101001000000"
                    "0101000001101100001000100110111001110001010001111011101100111011011101001000000"
                    "0110000010111000101000110110111110001010001101100000101111011011011001001000000"
                    "0110001101001111001000010110100000011010011101101011001111110011011001101000000"
                    "0111011000001001101000010010110111101110100001001100111000011011011011101000000"
                    "0000101110100000101011100000101010100011011010110101011010111110001110000000000"
                    "0101010001011111010100011111010101011100100101001010100101000001110001110100000"
                },
        /* 28*/ { BARCODE_UPCA_CC, -1, 1, "12345678901", "[91]12345678901234567890", 0, 7, 99, 1, "Example with CC-A 4 cols, 3 rows",
                    "110110111011110011010011100101110111000100001001110100101110111001000001100000100110111011011000101"
                    "110110110011101011001000000110011110101111101001100100110010111101100001110010111011000011001000101"
                    "110110100010011101000001100100111111011101001001100110111111001011010001000001100010111011101000101"
                    "000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010"
                    "001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001"
                    "000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010"
                    "000101001100100100110111101010001101100010101111010101000100100100011101001110010110011011011001010"
                },
        /* 29*/ { BARCODE_DBAR_OMN_CC, -1, 1, "12345678901231", "[91]1234567890123456789012345678", 0, 6, 100, 1, "Example with CC-A 4 cols, 4 rows",
                    "1101001000111100110100111001011101110001000010010001101011101110010000011000001001101110111011001010"
                    "1101011000111010110010000001100111101011111010010000101100101111011000011011011110000100110011001010"
                    "1101011100100011111011001001000000100101111010110000101100011111010010011000110101111000110111001010"
                    "1101011110100011011001000001110011100010001010111000101101111011110011011100101000111000110111101010"
                    "0000000001100011000110101000000001001011011100001001010110000101111010010101010000111101011000100000"
                    "0000010010011100111001000111111110010100100011110110101001111010000101100000000111000010100111011101"
                },
        /* 30*/ { BARCODE_DBAR_EXP_CC, -1, 1, "[01]12345678901231", "[91]1234567890123456789012345678901234567", 0, 7, 134, 1, "Example with CC-A 4 cols, 5 rows",
                    "00110101111011110011010011100101110111000100001011100010101110111001000001100000100110111011011110101000000000000000000000000000000000"
                    "00110100111011101011001000000110011110101111101011100110110010111101100001001000011110001011001110101000000000000000000000000000000000"
                    "00110100110010001101001110000110001010011111001011100100101111100001100101000111100010100011101110101000000000000000000000000000000000"
                    "00110100010010100000110110000111110111100101101011101100110001000100110001101100010001000011100110101000000000000000000000000000000000"
                    "00110100011011001011110110000100111001001111101001101100111110011000101001001101111110011011110110101000000000000000000000000000000000"
                    "00001011100111110001000000001010010011000000101001100011100100111011010000101010000111000011110101001101011110000010010100000010100000"
                    "01010100011000001110111111110000101100111111010110011100011011000100101111000000111000111100001010110010100001111101100011111100001010"
                },
        /* 31*/ { BARCODE_DBAR_EXPSTK_CC, -1, 1, "[01]12345678901231", "[91]123456789012345678901234567890123456789012345", 0, 12, 102, 1, "Example with CC-A 4 cols, 6 rows",
                    "001100010110111100110100111001011101110001000010011100101011101110010000011000001001101110110101111010"
                    "001100010100111010110010000001100111101011111010011110101100101111011000010010000111100010110100111010"
                    "001100110100100011010011100001100010100111110010111110101011111000011001010001111000101000110100110010"
                    "001101110100101000001101100001110011100010100010111100101111011101100111010000010110110000110100010010"
                    "001101100100111110001010001101100000101111011010111101101111000101110111010011110010111100110100011010"
                    "001101100110110101100111111001110001001111101010011101101001111000101000011001110100111000110100001010"
                    "000001101110011110010000000010100100001100110001001000111001001110110100001010100001110000111101010000"
                    "010110010001100001101111111100001011110011001110110111000110110001001011110000001110001111000010101101"
                    "000001101110011110010000000010100100001100110001001000111001001110110100001010100001110000111101010000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000000101111000001001010000001010010111100111110110000000000000000000000000000000000000000000000000000"
                    "001001010000111110110001111110000101000011000001001010000000000000000000000000000000000000000000000000"
                },
        /* 32*/ { BARCODE_EANX_CC, -1, 1, "123456789012", "[91]123456789012345678901234567890123456789012345678901234", 0, 11, 99, 1, "Example with CC-A 4 cols, 7 rows",
                    "110010111010010001111100110111000101110011001000011010110001011101000001000111010111110011011011001"
                    "110010011011010110011111100100011110000100101000111010101100111011111001011100010011000011011010001"
                    "110011011011011101110011000101100011010000001000110010111101111001001001100001101100011011010010001"
                    "111011011011101000110001000110111101011111001000100010101110010111110001110100011000001011010110001"
                    "111001011010011110001010000100111110000110101001100010111001101001111101000011110100001011010111001"
                    "110001011011001110001100010111011111011100101001110010111010000101110001110011000010110011010111101"
                    "110001010011110100110011000111101000010010001001111010111110100010011001001001111001000011010011101"
                    "000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010"
                    "001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001"
                    "000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010"
                    "000101001001101111010011101011000100001010010001010101001000111010011100101100110110110010010001010"
                },
        /* 33*/ { BARCODE_UPCE_CC, -1, 2, "1234567", "[91]1234567890123", 0, 12, 55, 1, "Example with CC-B 2 cols, 8 rows",
                    "1100100010111011111011101001000001000010001011001000101"
                    "1110100010110100001111011001100101110000100011101000101"
                    "1110110010101100100111000001011111011000001011101100101"
                    "1100110010110000110010100001010001000010000011001100101"
                    "1101110010111110110001001001100100111110111011011100101"
                    "1101111010100001111000101001110111111001001011011110101"
                    "1100111010100010001000000101110010000101110011001110101"
                    "1110111010111110001010110001111000100101000011101110101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
                },
        /* 34*/ { BARCODE_DBAR_STK_CC, -1, 2, "12345678901231", "[91]123456789012345678901234567", 0, 15, 56, 1, "Example with CC-B 2 cols, 11 rows",
                    "01100100010111011111011101001000001000010001011100110101"
                    "01110100010110100001111011001100101110000100011110110101"
                    "01110110010101100100111000001011111011000001011110010101"
                    "01100110010110000110010100001111100101011111011100010101"
                    "01101110010111011001111001101110100011000010011000010101"
                    "01101111010101100010111000001111101001110100011000110101"
                    "01100111010111011010110000001000110000111011011000100101"
                    "01110111010100010111110110001001001111010000011100100101"
                    "01110011010100111101000001001010110001111100011110100101"
                    "01111011010100010000111011101001100001100100011110101101"
                    "01111001010111101010000100001001110010111110011110101001"
                    "00000110001100011010100000000100101101110000100000000000"
                    "01001001110011100100011111111001010010001111011010000000"
                    "00000110001101011010101010101010101101010000100000000000"
                    "10101001111010000101100000000111000010100111011101000000"
                },
        /* 35*/ { BARCODE_DBAR_OMNSTK_CC, -1, 2, "12345678901231", "[91]1234567890123456789012345678901234567890123", 0, 20, 56, 1, "Example with CC-B 2 cols, 14 rows",
                    "01110111010100011111010011101101111110101110011101110101"
                    "01110011010100100101111000001001000100001111011100110101"
                    "01111011010101101111001110001111011110101000011110110101"
                    "01111001010110001110111110101110100011100011011110010101"
                    "01110001010110011001000111101001010000111100011100010101"
                    "01100001010110010001000011001101001111011111011000010101"
                    "01100011010110110111101000001111101100011011011000110101"
                    "01100010010100111100000100101011000101110000011000100101"
                    "01110010010100101110001110001111101101011110011100100101"
                    "01111010010111010000110100001001111101001110011110100101"
                    "01111010110100111100011011001011110001101100011110101101"
                    "01111010100111101110101100001111010001011110011110101001"
                    "01110010100100010111100000101101100111110011011100101001"
                    "01110110100101100100000111001000000110100111011101101001"
                    "00000110001100011010100000000100101101110000100000000000"
                    "01001001110011100100011111111001010010001111011010000000"
                    "00000110001100011010100000000100101101110000100000000000"
                    "00000101010101010101010101010101010101010101010000000000"
                    "00000110000101111010010101010000111101011000100000000000"
                    "10101001111010000101100000000111000010100111011101000000"
                },
        /* 36*/ { BARCODE_UPCE_CC, -1, 2, "1234567", "[91]123456789012345678901234567890123456789012345678901234567", 0, 21, 55, 1, "Example with CC-B 2 cols, 17 rows",
                    "1100110100100000011100101101100111111011001011001101001"
                    "1101110100111110110100111101110110010000110011011101001"
                    "1101100100111001011001000001110001101110100011011001001"
                    "1101100110110111111001101001100101100111111011011001101"
                    "1101101110100010110001100001010000011000110011011011101"
                    "1101101100100101111101100001111110100101110011011011001"
                    "1101101000100010110001111101011100100001100011011010001"
                    "1101001000111110111100101101100100010000110011010010001"
                    "1101011000111010000110010001110110111000100011010110001"
                    "1101011100101110110000111001011110110011000011010111001"
                    "1101011110110111100111010001000010010000100011010111101"
                    "1101001110101001111000100001110100110000010011010011101"
                    "1101001100111010011111101101101110010000111011010011001"
                    "1101000100110001000110011101101100111001110011010001001"
                    "1101000110100111110100011101100100000111001011010001101"
                    "1101000010101111000001101101001000000101111011010000101"
                    "1101100010110001100001101101011101110001000011011000101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
                },
        /* 37*/ { BARCODE_DBAR_STK_CC, -1, 2, "12345678901231", "[91]1234567890123456789012345678901234567890123456789012345678901234567890", 0, 24, 56, 1, "Example with CC-B 2 cols, 20 rows",
                    "01111010100111011111011101001000001000010001011110101001"
                    "01110010100110100001111011001100101110000100011100101001"
                    "01110110100101100100111000001011111011000001011101101001"
                    "01110100100110000110010100001111100101011111011101001001"
                    "01110100110111011001111001101110100011000010011101001101"
                    "01110101110101100010111000001111101001110100011101011101"
                    "01110101100101100011000010001100110110110000011101011001"
                    "01110101000100011001111100101001011111011000011101010001"
                    "01100101000101101001110000001100010100011111011001010001"
                    "01100101100100010110000011001000100001000001011001011001"
                    "01100101110111110001100010101101110011111010011001011101"
                    "01100100110100100101111000001100111001011100011001001101"
                    "01100110110111101001001111001001000010010000011001101101"
                    "01110110110100100001111010001010011111101110011101101101"
                    "01110010110111010001011111101011110111101111011100101101"
                    "01100010110100001001110111001000110110000100011000101101"
                    "01100010100111100100100001001000110011111001011000101001"
                    "01100110100110111010001110001001101001110000011001101001"
                    "01101110100101010000010000001110100010011100011011101001"
                    "01101100100101111101011100001001110100111110011011001001"
                    "00000110001100011010100000000100101101110000100000000000"
                    "01001001110011100100011111111001010010001111011010000000"
                    "00000110001101011010101010101010101101010000100000000000"
                    "10101001111010000101100000000111000010100111011101000000"
                },
        /* 38*/ { BARCODE_DBAR_OMNSTK_CC, -1, 2, "12345678901231", "[91]1234567890123456789012345678901234567890123456789012345678901234567890123456789012", 0, 29, 56, 1, "Example with CC-B 2 cols, 23 rows",
                    "01110011010100000011100101101100111111011001011110100101"
                    "01111011010111110110100111101110110010000110011110101101"
                    "01111001010111001011001000001110001101110100011110101001"
                    "01110001010110111111001101001100101100111111011100101001"
                    "01100001010100010110001100001010000011000110011101101001"
                    "01100011010100101111101100001111110100101110011101001001"
                    "01100010010100010110001111101011100100001100011101001101"
                    "01110010010111110111100101101100100010000110011101011101"
                    "01111010010111010000110010001110110111000100011101011001"
                    "01111010110101110110000111001011110110011000011101010001"
                    "01111010100110111100111010001000010010000100011001010001"
                    "01110010100110100001111011001100110111100010011001011001"
                    "01110110100111101011111001101000101000111100011001011101"
                    "01110100100111001101110111101011011000100000011001001101"
                    "01110100110111111010100011101110011001110100011001101101"
                    "01110101110100000010101111001001110111001111011101101101"
                    "01110101100111010010001110001101110011000100011100101101"
                    "01110101000100010111100000101010011110000100011000101101"
                    "01100101000101100110001111001011010000001110011000101001"
                    "01100101100110001000010110001011110111101000011001101001"
                    "01100101110111111010010011101000101111100011011011101001"
                    "01100100110110111101000001101110111101001110011011001001"
                    "01100110110110000010010011001101110001100100011011001101"
                    "00000110001100011010100000000100101101110000100000000000"
                    "01001001110011100100011111111001010010001111011010000000"
                    "00000110001100011010100000000100101101110000100000000000"
                    "00000101010101010101010101010101010101010101010000000000"
                    "00000110000101111010010101010000111101011000100000000000"
                    "10101001111010000101100000000111000010100111011101000000"
                },
        /* 39*/ { BARCODE_UPCE_CC, -1, 2, "1234567", "[91]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890", 0, 30, 55, 1, "Example with CC-B 2 cols, 26 rows",
                    "1100101000100000011100101101000001111001010011000101001"
                    "1100101100111110110100111101110110010000110011001101001"
                    "1100101110111001011001000001110001101110100011011101001"
                    "1100100110110111111001101001100101100111111011011001001"
                    "1100110110100010110001100001010000011000110011011001101"
                    "1110110110100101111101100001111110100101110011011011101"
                    "1110010110100010110001111101011100100001100011011011001"
                    "1100010110111110111100101101100100010000110011011010001"
                    "1100010100111010000110010001110110111000100011010010001"
                    "1100110100101110110000111001011110110011000011010110001"
                    "1101110100110111100111010001000010010000100011010111001"
                    "1101100100110100001111011001100110111100010011010111101"
                    "1101100110111101011111001101000101000111100011010011101"
                    "1101101110111001101110111101011011000100000011010011001"
                    "1101101100111111010100011101110011001110100011010001001"
                    "1101101000100000010101111001001110111001111011010001101"
                    "1101001000110100010110000001100110001000001011010000101"
                    "1101011000100110011111000101101011110110000011011000101"
                    "1101011100100110011011110001100111111001101011001000101"
                    "1101011110101001110001110001010011000011000011101000101"
                    "1101001110111101011100011101110101100010000011101100101"
                    "1101001100110011111000101001000011001101111011001100101"
                    "1101000100101000001001000001100011001001000011011100101"
                    "1101000110100010111111011101101000000111010011011110101"
                    "1101000010111111010001100101011001100111100011001110101"
                    "1101100010111011000110011101100001000001011011101110101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
                },
        /* 40*/ { BARCODE_DBAR_LTD_CC, -1, 2, "12345678901231", "[91]123456", 0, 8, 88, 1, "Example with CC-B 3 cols, 6 rows",
                    "1100100010111011111011101001011001110100000100001000101110100001000111011001000101000000"
                    "1110100010111010111101110001001001110101000011110010001101011100000010011101000101000000"
                    "1110110010110011010000111101001101110110111110100010001011001001110000011101100101000000"
                    "1100110010111000100100011101000101110111011110001101001110011101100110011001100101000000"
                    "1101110010111100101110011101000100110111101000100001001111010001000010011011100101000000"
                    "1101111010100111100100100001000110110100001111001101101011100111001111011011110101000000"
                    "0000000000000101110100000101011100000101010100011011010110101011010111110001110000000000"
                    "0000000000101010001011111010100011111010101011100100101001010100101000001110001110100000"
                },
        /* 41*/ { BARCODE_EANX_CC, -1, 2, "1234567", "[91]123456789012345678", 0, 12, 82, 1, "Example with CC-B 3 cols, 8 rows",
                    "1100111010111011111011101001000010110100000100001000101111101101001111011001110101"
                    "1110111010110010111000010001000010010111001011001000001110001101110100011101110101"
                    "1110011010110111111001101001000011010101000101111000001100100110111111011100110101"
                    "1111011010111011010000110001000111010110000110110011001110011010011000011110110101"
                    "1111001010111011011100100001000110010111011101111001001010011110000100011110010101"
                    "1110001010111111010110001001000100010111001011111101101001111000011011011100010101"
                    "1100001010111100100010111101001100010111001111001100101001110001110010011000010101"
                    "1100011010110011001111010001001110010111110111101110101011100111111001011000110101"
                    "0000000000000010000000000000000000000000000000000000000000000000000000000000000010"
                    "0000000000000100000000000000000000000000000000000000000000000000000000000000000001"
                    "0000000000000010000000000000000000000000000000000000000000000000000000000000000010"
                    "0000000000000010100110010010011011110101000110101010011101010000100010011100101010"
                },
        /* 42*/ { BARCODE_DBAR_LTD_CC, -1, 2, "12345678901231", "[91]12345678901234567890123456789", 0, 12, 88, 1, "Example with CC-B 3 cols, 10 rows",
                    "1100010010100000011100101101001111010110011111101100101001001011110000011000100101000000"
                    "1110010010111011001000011001011111010101101111001110001111011110101000011100100101000000"
                    "1111010010110001110111110101011110010111010001110001101110110011110011011110100101000000"
                    "1111010110100101000011110001011110110101100010111000001111101001110100011110101101000000"
                    "1111010100111011010001100001001110110110100001001100001110011110110010011110101001000000"
                    "1110010100101111110010001101001110100100110011111000101111010110000110011100101001000000"
                    "1110110100100001010000111101001100100101111010000001001001001110111111011101101001000000"
                    "1110100100110010001000011001001100110111011100110011001110111010001000011101001001000000"
                    "1110100110111000110011100101001000110110001001111011001111011000110001011101001101000000"
                    "1110101110110111110100010001001000010101111001100001101101110100111000011101011101000000"
                    "0000000000000101110100000101011100000101010100011011010110101011010111110001110000000000"
                    "0000000000101010001011111010100011111010101011100100101001010100101000001110001110100000"
                },
        /* 43*/ { BARCODE_EANX_CC, -1, 2, "1234567", "[91]12345678901234567890123456789012345678901", 0, 16, 82, 1, "Example with CC-B 3 cols, 12 rows",
                    "1110101100111011111011101001011000010100000100001000101111101101001111011101011001"
                    "1110101000110010111000010001011100010111001011001000001110001101110100011101010001"
                    "1100101000110111111001101001011100110110010110011111101100110010001111011001010001"
                    "1100101100101000001100011001011100100110010001000011001101001111011111011001011001"
                    "1100101110110110111101000001011101100111110110001101101000110011111001011001011101"
                    "1100100110101100010111000001001101100101101001110000001011010000011100011001001101"
                    "1100110110101110011111011101000101100111100111001011001100011000011011011001101101"
                    "1110110110111110100100110001000101000100111111010110001000010011111011011101101101"
                    "1110010110101101100000111101001101000101101110001111101011000000101110011100101101"
                    "1100010110101101100001000001011101000110011000100001001101110110001000011000101101"
                    "1100010100100001110101111101011001000101111011001111101111000110110001011000101001"
                    "1100110100100000110010011101011001100110000100101111101101111000101100011001101001"
                    "0000000000000010000000000000000000000000000000000000000000000000000000000000000010"
                    "0000000000000100000000000000000000000000000000000000000000000000000000000000000001"
                    "0000000000000010000000000000000000000000000000000000000000000000000000000000000010"
                    "0000000000000010100110010010011011110101000110101010011101010000100010011100101010"
                },
        /* 44*/ { BARCODE_DBAR_LTD_CC, -1, 2, "12345678901231", "[91]123456789012345678901234567890123456789012345678901234567", 0, 17, 88, 1, "Example with CC-B 3 cols, 15 rows",
                    "1101110100111011111011101001011000100100000100001000101111101101001111011011101001000000"
                    "1101100100110010111000010001011000110111001011001000001110001101110100011011001001000000"
                    "1101100110110111111001101001010000110110010110011111101100110010001111011011001101000000"
                    "1101101110101000001100011001010001110110010001000011001101001111011111011011011101000000"
                    "1101101100110110111101000001010001100111110110001101101000110011111001011011011001000000"
                    "1101101000101100010111000001010011100101101001110000001100010100011111011011010001000000"
                    "1101001000100010110000011001010011000100010000100000101101111001110100011010010001000000"
                    "1101011000110111001111101001010111000101001111000100001110100110000010011010110001000000"
                    "1101011100100000111101000101010110000100000011010011101101011011111100011010111001000000"
                    "1101011110101110011111011101010010000111100110100111001000011000111011011010111101000000"
                    "1101001110111011101111001001011010000111001000000110101101111010011111011010011101000000"
                    "1101001100111000111101011101001010000101110110001110001000001101001110011010011001000000"
                    "1101000100100110011110011101001011000100011001111011101000011011100011011010001001000000"
                    "1101000110111101000010000101001011100111000010000110101011111101000110011010001101000000"
                    "1101000010101111110011101001011011100111011010111110001100010111111010011010000101000000"
                    "0000000000000101110100000101011100000101010100011011010110101011010111110001110000000000"
                    "0000000000101010001011111010100011111010101011100100101001010100101000001110001110100000"
                },
        /* 45*/ { BARCODE_EANX_CC, -1, 2, "1234567", "[91]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[92]12345678901234567890123", 0, 30, 82, 1, "Example with CC-B 3 cols, 26 rows",
                    "1100100010111011111011101001000011010100000100001000101111101101001111011110100101"
                    "1110100010110010111000010001000111010111001011001000001110001101110100011110101101"
                    "1110110010110111111001101001000110010110010110011111101100110010001111011110101001"
                    "1100110010101000001100011001000100010110010001000011001101001111011111011100101001"
                    "1101110010110110111101000001001100010111110110001101101000110011111001011101101001"
                    "1101111010101100010111000001001110010101101001110000001100010100011111011101001001"
                    "1100111010100010110000011001001111010100010000100000101101111001110100011101001101"
                    "1110111010110111001111101001011111010110100001111011001100110111100010011101011101"
                    "1110011010111101011111001101011110010100010100011110001001100011001111011101011001"
                    "1111011010101101100010000001011110110111010100011100001011100011110110011101010001"
                    "1111001010101111010111100001001110110111100010011011001111110100100111011001010001"
                    "1110001010101111010010000001001110100110111100001011001110100101111110011001011001"
                    "1100001010110001010011000001001100100110010001100011101110111001000001011001011101"
                    "1100011010100111011111101001001100110111110011011001101000111010011111011001001101"
                    "1100010010111101000111101001001000110101100100000011101010110001111100011001101101"
                    "1110010010101000001000010001001000010110010001100011101011011100011000011101101101"
                    "1111010010111100011001101001011000010111011101111010001101100001111010011100101101"
                    "1111010110100001001011110001011100010110001110001011101110111000101111011000101101"
                    "1111010100111001011101111101011100110111101110100110001001000011011000011000101001"
                    "1110010100100111011111101001011100100101011110000100001110010001100100011001101001"
                    "1110110100101111001001000001011101100110001110100111001100111010011100011011101001"
                    "1110100100110100000010011001001101100100001001100001101011000111101110011011001001"
                    "1110100110111001000110100001000101100110101111000001101000110010111111011011001101"
                    "1110101110110000110101111001000101000101110011001110001000011100100011011011011101"
                    "1110101100110000110000100101001101000101011110011110001101000111011110011011011001"
                    "1110101000111011011110011001011101000110011011110001001111000101100110011011010001"
                    "0000000000000010000000000000000000000000000000000000000000000000000000000000000010"
                    "0000000000000100000000000000000000000000000000000000000000000000000000000000000001"
                    "0000000000000010000000000000000000000000000000000000000000000000000000000000000010"
                    "0000000000000010100110010010011011110101000110101010011101010000100010011100101010"
                },
        /* 46*/ { BARCODE_DBAR_LTD_CC, -1, 2, "12345678901231", "[91]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[92]123456789012345678901234", 0, 34, 88, 1, "Example with CC-B 3 cols, 32 rows",
                    "1110110100100000011100101101011101100110011111101100101001001011110000011011101001000000"
                    "1110100100111011001000011001001101100101101111001110001111011110101000011011001001000000"
                    "1110100110110001110111110101000101100111010001110001101110110011110011011011001101000000"
                    "1110101110100101000011110001000101000101100010111000001111101001110100011011011101000000"
                    "1110101100101100011000010001001101000110011011011000001111101111001011011011011001000000"
                    "1110101000100101111101100001011101000111010000110010001110110111000100011011010001000000"
                    "1100101000101110110000111001011001000101111011001100001000011101000110011010010001000000"
                    "1100101100100001001000010001011001100111110110100111101011100111001000011010110001000000"
                    "1100101110110101110000001001011000100111100100110001101001000011110100011010111001000000"
                    "1100100110101101110111110001011000110111010100011111101000110000101110011010111101000000"
                    "1100110110111101111001000101010000110111001100010001101101000101100000011010011101000000"
                    "1110110110111001101110010001010001110100110011111000101010111100001000011010011001000000"
                    "1110010110110111010001110001010001100101100011001111001000110010001110011010001001000000"
                    "1100010110111101111001010001010011100110111001101000001000011011101100011010001101000000"
                    "1100010100111010000111011001010011000100111010011111001100011011110100011010000101000000"
                    "1100110100100011110110001101010111000110011111010000101100100111111001011011000101000000"
                    "1101110100100010111101111001010110000101000011100011101000110000110100011001000101000000"
                    "1101100100111110001100101001010010000110101111101110001110010011110111011101000101000000"
                    "1101100110101111101100000101011010000100000011001011101000011011011110011101100101000000"
                    "1101101110110101000011000001001010000111011110011001001101110110010000011001100101000000"
                    "1101101100111110110100010001001011000111000100110100001011110010111100011011100101000000"
                    "1101101000100111111001110101001011100111011100100111101000101000001111011011110101000000"
                    "1101001000110000101110111101011011100111000111011011001101001001100000011001110101000000"
                    "1101011000100101111010000001011011110100000010111101001100110111100001011101110101000000"
                    "1101011100110011000100111101011001110111110100111001001011111011000010011100110101000000"
                    "1101011110110000110000101001001001110100000101110111001100010011101111011110110101000000"
                    "1101001110111111011100010101001101110110001101111010001010011111000110011110010101000000"
                    "1101001100100100010000111101000101110101111101100100001001101101111000011100010101000000"
                    "1101000100111011010011000001000100110110010010011000001001000110000110011000010101000000"
                    "1101000110111110011101101001000110110110000010011100101010011110100000011000110101000000"
                    "1101000010110010001011111001000010110101111000100100001100111101000110011000100101000000"
                    "1101100010111110010101111101000010010100111001110100001110010011001111011100100101000000"
                    "0000000000000101110100000101011100000101010100011011010110101011010111110001110000000000"
                    "0000000000101010001011111010100011111010101011100100101001010100101000001110001110100000"
                },
        /* 47*/ { BARCODE_DBAR_LTD_CC, -1, 2, "12345678901231", "[91]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[92]1234567890123456789012345678901234567890123456789012345678", 0, 40, 88, 1, "Example with CC-B 3 cols, 38 rows",
                    "1100010010100000011100101101000101100110011111101100101001001011110000011010011101000000"
                    "1110010010111011001000011001000101000101101111001110001111011110101000011010011001000000"
                    "1111010010110001110111110101001101000111010001110001101110110011110011011010001001000000"
                    "1111010110100101000011110001011101000101100010111000001111101001110100011010001101000000"
                    "1111010100101100011000010001011001000110011011011000001111101111001011011010000101000000"
                    "1110010100100101111101100001011001100111010000110010001110110111000100011011000101000000"
                    "1110110100101110110000111001011000100101111011001100001000011101000110011001000101000000"
                    "1110100100100001001000010001011000110111110110100111101011100111001000011101000101000000"
                    "1110100110110101110000001001010000110111100100110001101001000011110100011101100101000000"
                    "1110101110101101110111110001010001110111010100011111101000110000101110011001100101000000"
                    "1110101100111101111001000101010001100111001100010001101101000101100000011011100101000000"
                    "1110101000111001101110010001010011100100110011111000101010111100001000011011110101000000"
                    "1100101000110111010001110001010011000101100011001111001000110010001110011001110101000000"
                    "1100101100111101111001010001010111000110111001101000001000011011101100011101110101000000"
                    "1100101110111010000111011001010110000100111010011111001100011011110100011100110101000000"
                    "1100100110100011110100010001010010000101111110011100101101001000011111011110110101000000"
                    "1100110110110111011000100001011010000111000111001010001001000111011100011110010101000000"
                    "1110110110100000100011110101001010000110101110000010001011110001011110011100010101000000"
                    "1110010110110001110100111001001011000110001100010111101111011101011111011000010101000000"
                    "1100010110111011001000110001001011100111101011011111001110001010001110011000110101000000"
                    "1100010100111010011010000001011011100110000101110001001111001010000001011000100101000000"
                    "1100110100100010001001111001011011110110100000101111101110001111010111011100100101000000"
                    "1101110100110111101110001001011001110101011111011111001101010000110000011110100101000000"
                    "1101100100111101001010000001001001110111000100001101001100000100111001011110101101000000"
                    "1101100110100110110111100001001101110110111110001000101100111000100111011110101001000000"
                    "1101101110110100001001100001000101110101000001000100001000001010100000011100101001000000"
                    "1101101100111101000100000101000100110100111011111100101110101111101111011101101001000000"
                    "1101101000110111010000011101000110110100000010101111001000111001101110011101001001000000"
                    "1101001000111110110010111101000010110101101100010000001100101100111000011101001101000000"
                    "1101011000111000010011010001000010010110001100111101001100000101110001011101011101000000"
                    "1101011100100111100000100101000011010101110011001110001001111100110100011101011001000000"
                    "1101011110111000101011100001000111010110000101011000001110010100111000011101010001000000"
                    "1101001110100100011111001101000110010110110011110001001110110011100100011001010001000000"
                    "1101001100110111100001001101000100010110010010111110001111110010110010011001011001000000"
                    "1101000100100000100001000101001100010110100100000110001101110110010000011001011101000000"
                    "1101000110111100001001000101001110010110110111110011001000101111010000011001001101000000"
                    "1101000010100000101000111101001111010100110011011110001011111001110011011001101101000000"
                    "1101100010100110000110100001011111010110010000001011001100100011001110011101101101000000"
                    "0000000000000101110100000101011100000101010100011011010110101011010111110001110000000000"
                    "0000000000101010001011111010100011111010101011100100101001010100101000001110001110100000"
                },
        /* 48*/ { BARCODE_DBAR_LTD_CC, -1, 2, "12345678901231", "[91]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[92]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[93]1234567890123456789012345678901", 0, 46, 88, 1, "Example with CC-B 3 cols, 44 rows",
                    "1100100010111011111011101001011000010110001110001101001111101101001111011010001001000000"
                    "1110100010110010111000010001011100010111001011001000001110001101110100011010001101000000"
                    "1110110010110111111001101001011100110110010110011111101100110010001111011010000101000000"
                    "1100110010101000001100011001011100100110010001000011001101001111011111011011000101000000"
                    "1101110010110110111101000001011101100111110110001101101000110011111001011001000101000000"
                    "1101111010101100010111000001001101100101101001110000001100010100011111011101000101000000"
                    "1100111010100010110000011001000101100100010000100000101101111001110100011101100101000000"
                    "1110111010110111001111101001000101000110100001111011001100110111100010011001100101000000"
                    "1110011010111101011111001101001101000100010100011110001001100011001111011011100101000000"
                    "1111011010101101100010000001011101000111010100011100001011100011110110011011110101000000"
                    "1111001010101111010111100001011001000111100010011011001111110100100111011001110101000000"
                    "1110001010101111010010000001011001100110111100001011001110100101111110011101110101000000"
                    "1100001010110001010011000001011000100110010001100011101110111001000001011100110101000000"
                    "1100011010100111011111101001011000110111110011011001101000111010011111011110110101000000"
                    "1100010010111011010001111101010000110110011110001001101000111000100011011110010101000000"
                    "1110010010100011000011001001010001110110001110110000101110100110001111011100010101000000"
                    "1111010010111110001010110001010001100111000000101100101111101101000010011000010101000000"
                    "1111010110110011111000100101010011100111101011111011001000111101110111011000110101000000"
                    "1111010100110001101100001101010011000101110011110011001011000001100010011000100101000000"
                    "1110010100111001011100110001010111000111000001001100101101000011100010011100100101000000"
                    "1110110100100010000010111101010110000100011010000011101111110100110010011110100101000000"
                    "1110100100111101000101111001010010000100010011101110001011000001100100011110101101000000"
                    "1110100110111110100000101101011010000111101000001101101110100001101000011110101001000000"
                    "1110101110101100000110111101001010000100011110100001001101110100001110011100101001000000"
                    "1110101100100100000101000001001011000110110000100000101011011100011000011101101001000000"
                    "1110101000111110011101100101001011100111000010110000101110100000111011011101001001000000"
                    "1100101000111010111111011001011011100111001111001011101001111000110011011101001101000000"
                    "1100101100111101110110111001011011110110010000010110001010100000010000011101011101000000"
                    "1100101110101101011111100001011001110100111011111100101011110001011110011101011001000000"
                    "1100100110111010011111000101001001110100001010111100001110010111110001011101010001000000"
                    "1100110110100110110000010001001101110110100110111000001111011000100111011001010001000000"
                    "1110110110100100011110010001000101110101101111100001001001011110000010011001011001000000"
                    "1110010110110111111001100101000100110101111011100011101000010001011110011001011101000000"
                    "1100010110110001101100001101000110110111010110011110001110011000101100011001001101000000"
                    "1100010100100001011111001101000010110110010011100100001101101111000100011001101101000000"
                    "1100110100111010001111101001000010010101111000001101101110010010111111011101101101000000"
                    "1101110100100110001110011001000011010100001101110110001001101100000100011100101101000000"
                    "1101100100100011001111101001000111010110000101110001001111000101110111011000101101000000"
                    "1101100110110111101000110001000110010101111100001100101100101111110100011000101001000000"
                    "1101101110100001011000110001000100010110100011101111001100111001101000011001101001000000"
                    "1101101100111110010011011101001100010111101110011100101110011001110100011011101001000000"
                    "1101101000111011010011111001001110010100011110000010101101111100100010011011001001000000"
                    "1101001000100100000100100001001111010100110000110000101101101000100000011011001101000000"
                    "1101011000110000100111101101011111010111100010100001001100001011100010011011011101000000"
                    "0000000000000101110100000101011100000101010100011011010110101011010111110001110000000000"
                    "0000000000101010001011111010100011111010101011100100101001010100101000001110001110100000"
                },
        /* 49*/ { BARCODE_UPCA_CC, -1, 2, "12345678901", "[91]1234567890123", 0, 8, 99, 1, "Example with CC-B 4 cols, 4 rows",
                    "110100111010001111101001110110111111010111001001110110110100001111011001100101110000100011010010001"
                    "110100110010110010011100000101111101100000101001110100110111111001101001110100111110010011010110001"
                    "110100010011100110100001100110010000010110001001100100100111101111010001110001110100010011010111001"
                    "110100011011000101110000010111011011100100001001100110111110001010110001111000100101000011010111101"
                    "000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010"
                    "001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001"
                    "000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010"
                    "000101001100100100110111101010001101100010101111010101000100100100011101001110010110011011011001010"
                },
        /* 50*/ { BARCODE_EANX_CC, -1, 2, "123456789012", "[91]1234567890123456789012345", 0, 10, 99, 1, "Example with CC-B 4 cols, 6 rows",
                    "110010001011101111101110100110001110001101001011001110111110110100111101110110010000110011001000101"
                    "111010001011100101100100000111000110111010001001001110110001110111110101110100011100011011101000101"
                    "111011001011001100100011110100101000011110001001101110101100010111000001010011000111110011101100101"
                    "110011001011000010010110000100111101111101101000101110111110101001111101100110000101000011001100101"
                    "110111001011111001000010110111110011010000101000100110111111001010111001110100001101000011011100101"
                    "110111101010011100001000110110011100010111001000110110100000100010111101111110010011010011011110101"
                    "000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010"
                    "001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001"
                    "000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010"
                    "000101001001101111010011101011000100001010010001010101001000111010011100101100110110110010010001010"
                },
        /* 51*/ { BARCODE_GS1_128_CC, -1, 2, "[01]12345678901231", "[91]12345678901234567890123456789012345678901", 0, 10, 145, 1, "Example with CC-B 4 cols, 8 rows",
                    "0000000000000000000001100111010111011111011101001000001000010001010000101101111101101001111011101100100001100110011101010000000000000000000000000"
                    "0000000000000000000001110111010111001011001000001110001101110100010000100101100011101111101011101000111000110111011101010000000000000000000000000"
                    "0000000000000000000001110011010110011001000111101001010000111100010000110101011000101110000011111010011101000111001101010000000000000000000000000"
                    "0000000000000000000001111011010101100011000010001100110110110000010001110101111101111001011011001000100001100111101101010000000000000000000000000"
                    "0000000000000000000001111001010111010000110010001101000011100010010001100101000001001111010011100110001110010111100101010000000000000000000000000"
                    "0000000000000000000001110001010110001010111110001001010011110000010001000101011111101111011011011111100110010111000101010000000000000000000000000"
                    "0000000000000000000001100001010101000110001100001101100001000100010011000101011110011111011011000010001011000110000101010000000000000000000000000"
                    "0000000000000000000001100011010110111000111110101111100010010011010011100101000011010111111011001100001111010110001101010000000000000000000000000"
                    "0010110001100001010001001100100110100110001101110100111000111010010011110101100100001001010011000110010011100100010100001000101001110011100010100"
                    "1101001110011110101110110011011001011001110010001011000111000101101100001010011011110110101100111001101100011011101011110111010110001100011101011"
                },
        /* 52*/ { BARCODE_DBAR_OMN_CC, -1, 2, "12345678901231", "[91]123456789012345678901234567890123456789012345678901234567", 0, 12, 100, 1, "Example with CC-B 4 cols, 10 rows",
                    "1100010010100000011100101101100111111011001010011110101001001011110000010010001000011110110001001010"
                    "1110010010101101111001110001111011110101000010111110101100001100101000011111001010111110111001001010"
                    "1111010010111011001111001101110100011000010010111100101001011111011000011111101001011100111101001010"
                    "1111010110100010110001111101011100100001100010111101101001111000001001010110001011100000111101011010"
                    "1111010100100101110001110001100100001001100010011101101000101100000110010001000010000010111101010010"
                    "1110010100111110001100010101101110011111010010011101001010011110001000011101001100000100111001010010"
                    "1110110100100000111101101101101110000101110010011001001100011110100011010000100101111000111011010010"
                    "1110100100100110111000110001000100001100110010011001101000101000000100011100010010011100111010010010"
                    "1110100110110110011110000101110000100011010010010001101111101100001010010110011111000100111010011010"
                    "1110101110101111000010010001011110000010001010010000101001010000111100011011110001011000111010111010"
                    "0000000001100011000110101000000001001011011100001001010110000101111010010101010000111101011000100000"
                    "0000010010011100111001000111111110010100100011110110101001111010000101100000000111000010100111011101"
                },
        /* 53*/ { BARCODE_DBAR_EXP_CC, -1, 2, "[01]12345678901231", "[91]1234567890123456789012345678901234567890123456789012345678901234567890123", 0, 14, 134, 1, "Example with CC-B 4 cols, 12 rows",
                    "00111010110011101111101110100100000100001000101011000010111110110100111101110110010000110011101011001000000000000000000000000000000000"
                    "00111010100011100101100100000111000110111010001011100010110001110111110101110100011100011011101010001000000000000000000000000000000000"
                    "00110010100011001100100011110100101000011110001011100110101100010111000001111101001110100011001010001000000000000000000000000000000000"
                    "00110010110010110001100001000110011011011000001011100100111110111100101101100100010000110011001011001000000000000000000000000000000000"
                    "00110010111011101000011001000111011011100010001011101100110110011110000101001101111101000011001011101000000000000000000000000000000000"
                    "00110010011010000111010001100110001110101110001001101100100100101111000001100111001011100011001001101000000000000000000000000000000000"
                    "00110011011011110100100111100100100001001000001000101100111001101110111101101000011000111011001101101000000000000000000000000000000000"
                    "00111011011011101001100000010111010011000100001000101000111110111101110101001111010111100011101101101000000000000000000000000000000000"
                    "00111001011010100101111000000111010111110100001001101000101111011000011001100011001001111011100101101000000000000000000000000000000000"
                    "00110001011011101110001000010111001110010100001011101000100001000100001001100111100111010011000101101000000000000000000000000000000000"
                    "00110001010010000001011110100111100110110100001011001000111011011100010001101000111000010011000101001000000000000000000000000000000000"
                    "00110011010011111101001110110110110001011110001011001100110001111010011001000001110110111011001101001000000000000000000000000000000000"
                    "00001011100111110001000000001010010011000000101001100011100100111011010000101010000111000011110101001101011110000010010100000010100000"
                    "01010100011000001110111111110000101100111111010110011100011011000100101111000000111000111100001010110010100001111101100011111100001010"
                },
        /* 54*/ { BARCODE_UPCA_CC, -1, 2, "123456789012", "[91]12345678901234567890123456789012345678901234567890123456789012345678901234", 0, 19, 99, 1, "Example with CC-B 4 cols, 15 rows",
                    "110111010011101111101110100100000100001000101011000100111110110100111101110110010000110011011101001"
                    "110110010011100101100100000111000110111010001011000110110001110111110101110100011100011011011001001"
                    "110110011011001100100011110100101000011110001010000110101100010111000001111101001110100011011001101"
                    "110110111010110001100001000110011011011000001010001110111110111100101101100100010000110011011011101"
                    "110110110011101000011001000111011011100010001010001100110110011110000101001101111101000011011011001"
                    "110110100010000111010001100110001110101110001010011100100100101111000001100111001011100011011010001"
                    "110100100011110100100111100100100001001000001010011000111001101110111101011011000100000011010010001"
                    "110101100011111101010001110111100010001101101010111000111011000111000101111000001001010011010110001"
                    "110101110011001001111110010100010001001111001010110000110100000101111101110001111010111011010111001"
                    "110101111011011110111000100101011111011111001010010000110101000011000001011000111011000011010111101"
                    "110100111011001000011110110110100000011100101011010000110000011011110101000100011110100011010011101"
                    "110100110011110100011110010110111111011001001001010000111110101111001101100110100011110011010011001"
                    "110100010011011100011000010111011101110011101001011000100110111011000001001110111101100011010001001"
                    "110100011011101100001110100100001111010111101001011100111101010000010001111100011010100011010001101"
                    "110100001011010011001111110111001011111101101011011100101001100111110001011100110001110011010000101"
                    "000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010"
                    "001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001"
                    "000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010"
                    "000101001100100100110111101010001101100010101111010101000100100100011101001110010110011011011001010"
                },
        /* 55*/ { BARCODE_DBAR_EXPSTK_CC, -1, 2, "[01]12345678901231", "[91]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[92]1234567890123456789012345678901234567890123456", 0, 26, 102, 1, "Example with CC-B 4 cols, 20 rows",
                    "001100100010111011111011101001000001000010001010111100101111101101001111011101100100001100111001011010"
                    "001110100010111001011001000001110001101110100010111101101100011101111101011101000111000110110001011010"
                    "001110110010110011001000111101001010000111100010011101101011000101110000011111010011101000110001010010"
                    "001100110010101100011000010001100110110110000010011101001111101111001011011001000100001100110011010010"
                    "001101110010111010000110010001110110111000100010011001001101100111100001010011011111010000110111010010"
                    "001101111010100001110100011001100011101011100010011001101001001011110000011001110010111000110110010010"
                    "001100111010111101001001111001001000010010000010010001101110011011101111010110110001000000110110011010"
                    "001110111010111111010100011101110011001110100010010000101011110101111000011110001001101100110110111010"
                    "001110011010111110100111001001011110100100000010110000101101111000010110011101001011111100110110110010"
                    "001111011010110001010011000001100100011000111010111000101110111001000001011110111100101000110110100010"
                    "001111001010111110011011001101000111010011111010111001101110100001110110010011101001111100110100100010"
                    "001110001010100011100010001101000111101000100010111001001011111100111001011010010000111110110101100010"
                    "001100001010110111011000100001110001110010100010111011001001000111011100011000111011010000110101110010"
                    "001100011010111101011000011001110101111001110010011011001100001000111010011100010110001000110101111010"
                    "001100010010111001111110100101001001100011111010001011001110101011111100011101100111111010110100111010"
                    "001110010010100110000110000101000001000110011010001010001110001010111000011010010001100000110100110010"
                    "001111010010111100000101010001000110111111011010011010001001110111111010010011111000101110110100010010"
                    "001111010110111110001011101001001110011101111010111010001111110100011010010011100100110000110100011010"
                    "001111010100111110111010011101001000110000011010110010001101011100011110010000110111101110110100001010"
                    "001110010100100111111010110001010011111000011010110011001110010001110110011110010000110110110110001010"
                    "000001101110011110010000000010100100001100110001001000111001001110110100001010100001110000111101010000"
                    "010110010001100001101111111100001011110011001110110111000110110001001011110000001110001111000010101101"
                    "000001101110011110010000000010100100001100110001001000111001001110110100001010100001110000111101010000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000000101111000001001010000001010010111100111110110000000000000000000000000000000000000000000000000000"
                    "001001010000111110110001111110000101000011000001001010000000000000000000000000000000000000000000000000"
                },
        /* 56*/ { BARCODE_UPCA_CC, -1, 2, "12345678901", "[91]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[92]12345678901234567890123456789012345678901234567", 0, 30, 99, 1, "Example with CC-B 4 cols, 26 rows",
                    "110010001011101111101110100110001110001101001000011010111110110100111101110110010000110011110100101"
                    "111010001011100101100100000111000110111010001000111010110001110111110101110100011100011011110101101"
                    "111011001011001100100011110100101000011110001000110010101100010111000001111101001110100011110101001"
                    "110011001010110001100001000110011011011000001000100010111110111100101101100100010000110011100101001"
                    "110111001011101000011001000111011011100010001001100010110110011110000101001101111101000011101101001"
                    "110111101010000111010001100110001110101110001001110010100100101111000001100111001011100011101001001"
                    "110011101011110100100111100100100001001000001001111010111001101110111101011011000100000011101001101"
                    "111011101011111101010001110111001100111010001011111010101111010111100001111000100110110011101011101"
                    "111001101011111010011100100101111010010000001011110010110111100001011001110100101111110011101011001"
                    "111101101011000101001100000110010001100011101011110110111011100100000101111011110010100011101010001"
                    "111100101011111001101100110100011101001111101001110110111010000111011001001110100111110011001010001"
                    "111000101010001110001000110100011110100010001001110100101111110011100101101001000011111011001011001"
                    "110000101011011101100010000111000111001010001001100100100100011101110001100011101101000011001011101"
                    "110001101011010111000001000101111001101111101001100110111000101101000001111010001010000011001001101"
                    "110001001010111011001110000101000010111100001001000110100011101000110001000110011011110011001101101"
                    "111001001010000010011000110100010100000100001001000010101011100011100001000010110000011011101101101"
                    "111101001011111001000100110111000000100110101011000010110010011100000101111001010000001011100101101"
                    "111101011010001000100111100110100000101111101011100010111000111101011101101111100110011011000101101"
                    "111101010010110001111011100110001100101000001011100110101000011100011101100001001000011011000101001"
                    "111001010011010001111000110110101111100011101011100100111010001100000101000000101111001011001101001"
                    "111011010010011100010110000101111110011101001011101100110111110000101001100011110100110011011101001"
                    "111010010010000100110000110110100000100110001001101100100010111001110001000001001110111011011001001"
                    "111010011011100001001101000100001111010111101000101100101111110011011101100111101011111011011001101"
                    "111010111010011100011011100101101000000111001000101000100010110001111101100111110010010011011011101"
                    "111010110011000011011101110100010100001000001001101000101110011110110001010001100011000011011011001"
                    "111010100011100001001100100100101111001000001011101000111001100111001001101000011110110011011010001"
                    "000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010"
                    "001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001"
                    "000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010"
                    "000101001100100100110111101010001101100010101111010101000100100100011101001110010110011011011001010"
                },
        /* 57*/ { BARCODE_DBAR_EXP_CC, -1, 2, "[01]12345678901231", "[91]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[92]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[93]1234", 0, 34, 134, 1, "Example with CC-B 4 cols, 32 rows",
                    "00111011010010000001110010110110011111101100101011101100100100101111000001001000100001111011011101001000000000000000000000000000000000"
                    "00111010010010110111100111000111101111010100001001101100110000110010100001111100101011111011011001001000000000000000000000000000000000"
                    "00111010011011101100111100110111010001100001001000101100100101111101100001111110100101110011011001101000000000000000000000000000000000"
                    "00111010111010001011000111110101110010000110001000101000100111100000100101011000101110000011011011101000000000000000000000000000000000"
                    "00111010110010010111000111000110010000100110001001101000100010110000011001000100001000001011011011001000000000000000000000000000000000"
                    "00111010100011111000110001010110111001111101001011101000110100001111011001100110111100010011011010001000000000000000000000000000000000"
                    "00110010100011110101111100110100010100011110001011001000100110001100111101011011101111100011010010001000000000000000000000000000000000"
                    "00110010110011101010001110000101110001111011001011001100111101111001000101110011000100011011010110001000000000000000000000000000000000"
                    "00110010111011111101001001110111001101110010001011000100100110011111000101010111100001000011010111001000000000000000000000000000000000"
                    "00110010011011011101000111000101100011001111001011000110100011001000111001011111000011010011010111101000000000000000000000000000000000"
                    "00110011011011011100110100000100001101110110001010000110100101111101111101110111101110011011010011101000000000000000000000000000000000"
                    "00111011011011000110111101000110000100001110101010001110100011100101111101010111110011000011010011001000000000000000000000000000000000"
                    "00111001011010111000001101110100001111011011001010001100101100011101111101100111110001001011010001001000000000000000000000000000000000"
                    "00110001011011010010011000000110011100001101001010011100110001101100001101011100111100110011010001101000000000000000000000000000000000"
                    "00110001010011111011001100110111001011100110001010011000111000001001100101101000011100010011010000101000000000000000000000000000000000"
                    "00110011010010001000001011110100011010000011101010111000111111010011001001111101001110001011011000101000000000000000000000000000000000"
                    "00110111010010001001110111000101100000110010001010110000111011010000011001101101101100000011001000101000000000000000000000000000000000"
                    "00110110010011101000011010000111001001101000001010010000110011100111110101111001000110011011101000101000000000000000000000000000000000"
                    "00110110011011100100111110100111100100111101001011010000111011100101111001100101111110001011101100101000000000000000000000000000000000"
                    "00110110111011011110111001000111010110111100001001010000100011000111001101110110101100000011001100101000000000000000000000000000000000"
                    "00110110110011011000011110010100010001111101101001011000101001111000010001001100011111010011011100101000000000000000000000000000000000"
                    "00110110100010001100110111100101111110111000101001011100100110010111000001101011001111110011011110101000000000000000000000000000000000"
                    "00110100100010100000100001000101011111011111001011011100111011100010100001100100110011100011001110101000000000000000000000000000000000"
                    "00110101100011111000011010010100100111100000101011011110111001001100000101011000111110010011101110101000000000000000000000000000000000"
                    "00110101110010000111100001010100011000100011101011001110111101011111011001011110001000010011100110101000000000000000000000000000000000"
                    "00110101111011110110100011100101001111011110001001001110100100001000010001100100111001111011110110101000000000000000000000000000000000"
                    "00110100111011111010010001100111000101101000001001101110110011000111101001111100011010001011110010101000000000000000000000000000000000"
                    "00110100110011110100011110100101011101111110001000101110100111011101111001011110010100000011100010101000000000000000000000000000000000"
                    "00110100010011101101011000000100001100011010001000100110111110111010111001100001100010001011000010101000000000000000000000000000000000"
                    "00110100011011100010011001000111000010011000101000110110101110110011111101011001001111110011000110101000000000000000000000000000000000"
                    "00110100001011011101000111000100001110100110001000010110100011000010111001000010110011111011000100101000000000000000000000000000000000"
                    "00110110001010001110111101100111000110101100001000010010111001010011100001110101011100000011100100101000000000000000000000000000000000"
                    "00001011100111110001000000001010010011000000101001100011100100111011010000101010000111000011110101001101011110000010010100000010100000"
                    "01010100011000001110111111110000101100111111010110011100011011000100101111000000111000111100001010110010100001111101100011111100001010"
                },
        /* 58*/ { BARCODE_DBAR_EXPSTK_CC, -1, 2, "[01]12345678901231", "[91]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[92]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[93]1234567890123456789012345678901234567890123456789012", 0, 44, 102, 1, "Example with CC-B 4 cols, 38 rows",
                    "001100010010100000011100101101100111111011001010001011001001001011110000010010001000011110110100111010"
                    "001110010010101101111001110001111011110101000010001010001100001100101000011111001010111110110100110010"
                    "001111010010111011001111001101110100011000010010011010001001011111011000011111101001011100110100010010"
                    "001111010110100010110001111101011100100001100010111010001001111000001001010110001011100000110100011010"
                    "001111010100100101110001110001100100001001100010110010001000101100000110010001000010000010110100001010"
                    "001110010100111110001100010101101110011111010010110011001101000011110110011001101111000100110110001010"
                    "001110110100111101011111001101000101000111100010110001001001100011001111010110111011111000110010001010"
                    "001110100100111010100011100001011100011110110010110001101111011110010001011100110001000110111010001010"
                    "001110100110111111010010011101110011011100100010100001101001100111110001010101111000010000111011001010"
                    "001110101110110111010001110001011000110011110010100011101000110010001110010111110000110100110011001010"
                    "001110101100110111001101000001000011011101100010100011001001011111011111011101111011100110110111001010"
                    "001110101000110001101111010001100001000011101010100111001000111001011111010101111100110000110111101010"
                    "001100101000101110000011011101000011110110110010100110001011000111011111011001111100010010110011101010"
                    "001100101100110100100110000001100111000011010010101110001100011011000011010111001111001100111011101010"
                    "001100101110111110110011001101110010111001100010101100001110000010011001011010000111000100111001101010"
                    "001100100110100010000010111101000110100000111010100100001111110100110010011111010011100010111101101010"
                    "001100110110100010011101110001011000001100100010110100001110110100000110011011011011000000111100101010"
                    "001110110110111010000110100001110010011010000010010100001100111001111101011110010001100110111000101010"
                    "001110010110100110110111100001100010010111110010010110001110110111111010010011110001101100110000101010"
                    "001100010110110001100100000101101101100110000010010111001101001000011000011000010001101110110001101010"
                    "001100010100111100001000100101110010000001101010110111001111011011000010011101000001101000110001001010"
                    "001100110100100001000010111101101100110011111010110111101010110111110000010010111001111110111001001010"
                    "001101110100100001001100001101000011011010000010110011101001100111001100010010001101100000111101001010"
                    "001101100100111010111011000001111110111001010010010011101111100100010011011100000010011010111101011010"
                    "001101100110110001011111100101100100111111001010011011101000100010011110011010000010111110111101010010"
                    "001101101110100011000011010001101111011100010010001011101010011000110000010110001111011100111001010010"
                    "001101101100111000110111010001101111100101111010001001101110111001111001011110101000010000111011010010"
                    "001101101000100011010111000001000010010111100010001101101101101100111110011100101001111110111010010010"
                    "001101001000100001100011001001000100001000010010000101101110100100111000010001000011000110111010011010"
                    "001101011000111010000110001001010000011110100010000100101111100111011001011111101111011010111010111010"
                    "001101011100100101100001111101000000110101110010000110101000001010111100011111100101100100111010110010"
                    "001101011110110111011000100001101001110001111010001110101100101001100000011100110000100110111010100010"
                    "001101001110111100100011001101110010110010000010001100101101001111100111011110001010100000110010100010"
                    "001101001100100111001110111101101111110011001010001000101011110010000100010010010011110000110010110010"
                    "001101000100100100000100001001101110110010000010011000101111011001000111011100010001001110110010111010"
                    "001101000110111100001101100101000100000111101010011100101111001010010000011110011000110100110010011010"
                    "001101000010101011011111000001000010100011110010011110101100111110100001010001101000111000110011011010"
                    "001101100010111000100100011101001110011110110010111110101100011000010010011011100110000100111011011010"
                    "000001101110011110010000000010100100001100110001001000111001001110110100001010100001110000111101010000"
                    "010110010001100001101111111100001011110011001110110111000110110001001011110000001110001111000010101101"
                    "000001101110011110010000000010100100001100110001001000111001001110110100001010100001110000111101010000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000000101111000001001010000001010010111100111110110000000000000000000000000000000000000000000000000000"
                    "001001010000111110110001111110000101000011000001001010000000000000000000000000000000000000000000000000"
                },
        /* 59*/ { BARCODE_UPCA_CC, -1, 2, "12345678901", "[91]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[92]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[93]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[94]123456789012345678901234567890123456789012345678901234567", 0, 48, 99, 1, "Example with CC-B 4 cols, 44 rows",
                    "110010001011101111101110100100000100001000101011000010111110110100111101110110010000110011010001001"
                    "111010001011100101100100000111000110111010001011100010110001110111110101110100011100011011010001101"
                    "111011001011001100100011110100101000011110001011100110101100010111000001111101001110100011010000101"
                    "110011001010110001100001000110011011011000001011100100111110111100101101100100010000110011011000101"
                    "110111001011101000011001000111011011100010001011101100110110011110000101001101111101000011001000101"
                    "110111101010000111010001100110001110101110001001101100100100101111000001100111001011100011101000101"
                    "110011101011110100100111100100100001001000001000101100111001101110111101011011000100000011101100101"
                    "111011101011111101010001110111001100111010001000101000101111010111100001111000100110110011001100101"
                    "111001101011111010011100100101111010010000001001101000110111100001011001110100101111110011011100101"
                    "111101101011000101001100000110010001100011101011101000111011100100000101111011110010100011011110101"
                    "111100101011111001101100110100011101001111101011001000111010000111011001001110100111110011001110101"
                    "111000101010001110001000110100011110100010001011001100101111110011100101101001000011111011101110101"
                    "110000101011011101100010000111000111001010001011000100100100011101110001100011101101000011100110101"
                    "110001101011010111000001000101111000101111001011000110100010000111101001001101111110110011110110101"
                    "110001001011110111010111110110010001001111101010000110111001111100101101011010000011100011110010101"
                    "111001001011101110100100000110011001100001101010001110111010011101111101111010001011110011100010101"
                    "111101001010010000111110110111101100110010001010001100111110100000101101111010000011011011000010101"
                    "111101011011101101011111000101100000110111101010011100100011110100001001101110100001110011000110101"
                    "111101010010010000010100000110110000100000101010011000101101110001100001000110001101000011000100101"
                    "111001010011100001011000010111010000011101101010111000101011110010000001000110100111111011100100101"
                    "111011010010011110001100110101110000010110001010110000101100000100111001011011000011110011110100101"
                    "111010010011101110001000010111101100100011101010010000110101101110000001100101000011000011110101101"
                    "111010011011111011100011010110000010011100101011010000111101000100001001100010011110110011110101001"
                    "111010111011111001011101000111000111010111101001010000100111100111011101011101110011110011100101001"
                    "111010110010010000110000110100100001000010001001011000101100111000110001010011000110000011101101001"
                    "111010100011000010011101000110111001111101001001011100110010011100001001110001011100011011101001001"
                    "110010100011100111110101100100010100011110001011011100100011110001010001000100011011111011101001101"
                    "110010110011000100001000110110001000110011101011011110111011110110010001001110000111010011101011101"
                    "110010111011101000110001000110010111100011001011001110110010001111011001110001101110100011101011001"
                    "110010011010011000110011110110100010111110001001001110100001110001011001101110100111000011101010001"
                    "110011011011100110100011000111100111101000101001101110111110010101111101111010010011110011001010001"
                    "111011011011110100111011100110100000011100101000101110111101101110011001101111110101110011001011001"
                    "111001011010000111000100110101100100001110001000100110110011100001011101011100001011000011001011101"
                    "110001011010000010100010000110100110111000001000110110100100000110110001001000011011000011001001101"
                    "110001010010100001111100110101101111110001101000010110111110010001001101111100010110111011001101101"
                    "110011010011011010011110000110111001000111001000010010110111010000111001001111100110001011101101101"
                    "110111010011000011010010000111101100100011101000011010100001100111011001110111101100010011100101101"
                    "110110010010101111110111000110101111100111001000111010100100011111001101111000001010010011000101101"
                    "110110011010110100000011100100000100100111101000110010100001011001111101000111101001000011000101001"
                    "110110111011001111011100100110111000110010001000100010100100001100110001110001110011011011001101001"
                    "110110110010111101000011110100100001111001001001100010111001110111100101111010100000010011011101001"
                    "110110100011011100100111000110001111010110001001110010100011101101110001100010001011111011011001001"
                    "110100100010001110111010000111000100001011101001111010110000110110001101100110000100100011011001101"
                    "110101100011100000010110100111100010001010001011111010111101000101000001001111101011100011011011101"
                    "000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010"
                    "001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001"
                    "000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010"
                    "000101001100100100110111101010001101100010101111010101000100100100011101001110010110011011011001010"
                },
        /* 60*/ { BARCODE_GS1_128_CC, -1, 3, "[01]12345678901231", "[91]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[92]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[93]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[94]12345678901234567890123456789012345678901234567890123456789012345678901234567", 0, 32, 154, 1, "Example with CC-C 5 cols, 30 rows",
                    "1111111101010100010101000001000000101000001000001001110111110111010010000010000100010111110110100111101110110010000110011110101001111000111111101000101001"
                    "1111111101010100011111010100000110111001011001000001110001101110100011000111011111010111010001110001101110110011110011011111010100001100111111101000101001"
                    "1111111101010100011101010011111100100101000011110001011000101110000011111010011101000100010110001111101011100100001100011111101011010000111111101000101001"
                    "1111111101010100011101001001110000111110111100101101100100010000110010010111000111000110010000100110001000101100000110011111010111111010111111101000101001"
                    "1111111101010100011010111101100000100110111110100001111100011000101011011100111110100110100001111011001100110111100010011101011100000110111111101000101001"
                    "1111111101010100011111010111100110111101011111001101000101000111100010011000110011110101101110111110001110101000111111011101011111000100111111101000101001"
                    "1111111101010100010100111100111100101110001111011001111011110010001011100110001000110110100010110000001100110001000001011010011100111100111111101000101001"
                    "1111111101010100011111101001001110100110011111000101010111100001000011110010001101100101101001111110001111001101101000011110100100100000111111101000101001"
                    "1111111101010100011010011001111110101111100001101001000010001011110010001111000100010111011010001111101100111100010011011111010011100100111111101000101001"
                    "1111111101010100010100011110111100111100111001011001000110000110010011000111011000010111010011000111101101110110001000010100011101110000111111101000101001"
                    "1111111101010100011010011100000010111000000101100101111101101000010010000010001111010110101110000010001011110001011110011101001110000110111111101000101001"
                    "1111111101010100010100010000011110110001110100111001100011000101111011110111010111110110010001001111101110011111001011011111010001110100111111101000101001"
                    "1111111101010100011101000001001110111110110101111001110111010010000011001100110000110111010011101111101111010001011110010100000101000000111111101000101001"
                    "1111111101010100011110100011011000100100001111101101111011001100100011111010000010110111101000001101101110100001101000011110100010000010111111101000101001"
                    "1111111101010100010100000100111100101100000110111101000111101000010011011101000011100100110110111100001100010010111110011100101011111100111111101000101001"
                    "1111111101010100011001011001110000101101110001100001000110001101000011000110010000010110110110011000001101001000011000010010100000100000111111101000101001"
                    "1111111101010100010100011111101110100011010011111101111000010001001011100100000011010111101101100001001110100000110100010100011111001100111111101000101001"
                    "1111111101010100011111001011100010100001000010111101101100110011111010101101111100000100101110011111101000011001000111011100101111100100111111101000101001"
                    "1111111101010100011101101000001100110000110110011001101000001001100010011000110000100110010100011000001100001001000011011110110100011100111111101000101001"
                    "1111111101010100011110100000100010110000100111101101011001111100001010010011110100000111001001110110001111101101000100011111010000010110111111101000101001"
                    "1111111101010100010110110111100000110100100011111001011111011101100011000111010111000111000101111100101000110100000111010010010000111100111111101000101001"
                    "1111111101010100010110111000011000110111100011100101001000010010000011000011011000110111100110100111001100010000100011010110111001100000111111101000101001"
                    "1111111101010100011110010100000010110001011110011001100011011110001011110000010100100111010001100010001100101111000110011111001010000110111111101000101001"
                    "1111111101010100011110010011110010101110110000011101011111011000001010011000110011110110100010111110001000011100010110011001001111110010111111101000101001"
                    "1111111101010100011001000100110000110001010110000001110011010001100011110011110100010110110010000010001111011001011100011011001000000100111111101000101001"
                    "1111111101010100011100101111001110110001011100100001110010111101110010011110101111000111010110001000001111100100001011010010111100100000111111101000101001"
                    "1111111101010100011111100100011010100110100111000001011000110001111011000011010111100111110100111010001011100000010110011111011001111010111111101000101001"
                    "1111111101010100010010001110001110111001101000110001100100010000011010001110011110110101001100110000001000110001101000010010001110111000111111101000101001"
                    "1111111101010100011111101101101110100100111101000001010000111110011011110110011010000111110011001000101101000001110100011110110110100000111111101000101001"
                    "1111111101010100010110001110111110100111000000101101000001101000111011111001011101000101111101111011101000111011001110010010000010111100111111101000101001"
                    "0000000001011000110000101000100110010011010011000110111010011100011101001001111010110010000100101001100011001001110010100001000101110010001001110001010000"
                    "0000000110100111001111010111011001101100101100111001000101100011100010110110000101001101111011010110011100110110001101011110111010001101110110001110101100"
                },
        /* 61*/ { BARCODE_GS1_128_CC, -1, 3, "[01]12345678901231", "[91]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[92]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[93]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[94]12345678901234567890123456789012345678901234567890123456789012345678901234567890", 0, 28, 171, 1, "Example with CC-C 6 cols, 25 rows",
                    "111111110101010001101010000110000011010000001001100111011111011101001000001000010001011111011010011110111011001000011001011011110011100011111010100111110111111101000101001"
                    "111111110101010001111010100000100011100011011101000110001110111110101110100011100011011101100111100110111010001100001001001011111011000011110101000010000111111101000101001"
                    "111111110101010001010100011110000011111010011101000100010110001111101011100100001100010011110000010010101100010111000001011010011100000010101000001111000111111101000101001"
                    "111111110101010001101001001100000011001000010011000100010110000011001000100001000001011011110011101000100001001000010001111101101001111011101001011100000111111101000101001"
                    "111111110101010001101011100000010011001101111000100110101110000001001111001001100011010010000111101000101000001111010001111110101000111011010111000001000111111101000101001"
                    "111111110101010001111010111100010010001100001011100100000010101111001001110111001111011111010011100100101111010010000001101111000010110011110101111100110111111101000101001"
                    "111111110101010001101001111011111011101001101111000110001010011000001100100011000111011101110010000010111101111001010001101110011010000010100111000111000111111101000101001"
                    "111111110101010001111101001001100010001110100111110111010000111011001001110100111110011000110111101000110000100001110101000111001011111011111101001011100111111101000101001"
                    "111111110101010001111110100110001011010010000111110101110000011011101000011110110110010110001110111110110011111000100101111010111110110010100111011111100111111101000101001"
                    "111111110101010001010001110001110011001110000110100110001101100001101011100111100110010110000011000100110110010001000001000011011010000011010001110111100111111101000101001"
                    "111111110101010001101001110000010011010000111000100111001000001100101110001011100011010101111110111000111110100100011001001000011111011011010011100001000111111101000101001"
                    "111111110101010001111110100011010011101110101111000110010010001111101001001000001111011101101011111000101100000110111101000111101000010010100011000111110111111101000101001"
                    "111111110101010001101000001001100011000101000110000100100000101000001101100001000001010110111000110000100011000110100001100011001000001011010000010110000111111101000101001"
                    "111111110101010001110100011010000011101000001110110101011110010000001000110100111111011110000100010010111001000000110101111011011000010011110100010000100111111101000101001"
                    "111111110101010001010000010001111010110110000111100100001000010111101101100110011111010101101111100000100101110011111101000011001000111011001010111110000111111101000101001"
                    "111111110101010001001011001100000011000011011001100110100000100110001001100011000010011001010001100000110000100100001101110111101100010011001011011100000111111101000101001"
                    "111111110101010001010001111100011010110011111000010100100111101000001110010011101100011111011010001000110101111101110001100001001110100011010001111101110111111101000101001"
                    "111111110101010001111001011110100011000111010111000111000101111100101000110100000111011100111110101100100010100011110001000111100010100011110010111110110111111101000101001"
                    "111111110101010001101101000001000011110011010011100110001000010001101100010001100111011101111011001000100111000011101001010000011011000011011010000100000111111101000101001"
                    "111111110101010001111010000010010011001011110001100110010001111011001110001101110100010010000111101000111101001100001101000001011110001011110100000101000111111101000101001"
                    "111111110101010001001001001111000011011101001110000101110111101111101000001110010110010110011000011110101100100000011101000110100011100010110110001111000111111101000101001"
                    "111111110101010001001001100001100011011001101100000110001110110010001101011000111000011100010010111000101100011110111001001000110000110011011011100111000111111101000101001"
                    "111111110101010001111001010000010011110111001110010101001111110111001101100011110010011110110110000100100001101111101001100001001111011011110010100001000111111101000101001"
                    "111111110101010001111011011111001011010110011111100100101001111000001101110010000111010001111100011010100001001001111001101001111110100011001001111110100111111101000101001"
                    "111111110101010001101100110110000011110100010011110110001000010001101010011101110000011011100011000010111100111000101101011011000010000011001000101100000111111101000101001"
                    "111111110101010001100101111001100011011110101111100111110010000101101000010011111011011110011011100110111000100011101101100000010111010011100101111011100111111101000101001"
                    "000000000101100011000010100010011001001101001100011011101001110001110100100111101011001000010010100110001100100111001010000100010111001000100111000101000000000000000000000"
                    "000000011010011100111101011101100110110010110011100100010110001110001011011000010100110111101101011001110011011000110101111011101000110111011000111010110000000000000000000"
                },
        /* 62*/ { BARCODE_GS1_128_CC, -1, 3, "[01]12345678901231", "[91]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[92]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[93]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[94]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[95]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[96]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[97]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[98]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[99]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[91]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[92]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[93]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[94]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[95]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[96]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[97]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[98]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[99]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[91]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[92]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[93]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[94]1234567890123456789012345678901234567890", 0, 32, 528, 1, "Example with CC-C 27 cols, 30 rows",
                    "111111110101010001010100000100000010000100001000010111011111011101001000001000010001011111011010011110111011001000011001011011110011100011110111101010000110000110010100001111100101011111010001011000110000101000001100011001100100010000110011010011110111110101100011000010001100110110110000011111011110010110110010001000011001001011100011100011001000010011000100010110000011001000100001000001011011110011101000100001001000010001111101101001111010111001110010000111101001001111001001000010010000011101011100111110111111101000101001"
                    "111111110101010001111010110011000010010000111101000101000001111010001111110101000111011100110011101000101111010111100001111000100110110011111101001001110111001101110010001001100111110001010101111000010000111100100011011001011010011111100011110011011010000100111011111101001111100110110011010001110100111110111010000111011001001110100111110011000110111101000110000100001110101000111001011111010101111100110000111110001010110001110000001011001011111011010000100100000100011110101101011100000100011111010100001100111111101000101001"
                    "111111110101010001111110101110011010001111011101110110001110100111001100011000101111011110111010111110110010001001111101110011111001011010110100000111000100010000010111101000110100000111011111101001100100111110100111000101011100010001100011101110101111000110010010001111101001001000001111011101101011111000101100000110111101000111101000010011011101000011100100110110111100001100010010111110011101101111110100100111100011011001000111001000011011001001101111110111010111111011001110011110010111011111101011000100111111101000101001"
                    "111111110101010001110100100111000010011100111010000111101110110111001111011000100111011110010010111100111011100010000101111011001000111011010110111000000110010100001100001000010011000011011000011011001100110100000100110001001100011000010011001010001100000110000100100001101110111101100010011111001101011110100100001100001101001000010000100010110011100011000101001100011000001111011110100001010000100100001000110010000110011101100110011000011011011110001110010100100001001000001100001101100011011010011000111000111111101000101001"
                    "111111110101010001010111100010000010010001111100110111001101111001101100010111100110011000110111100010111100000101001001110100011000100011001011110001100110010001111011001110001101110100010010000111101000111101001100001101000001011110001011110010001000010111110110001010001110000001011010011100101110000110111001011000000101100010111001000011100101111011100100111101011110001110101100010000011111001000010110111001001111011101111110110100011011110111001110010101001111110111001110010000111011011101011100000110111111101000101001"
                    "111111110101010001010010000111100010100010011110000111001001111100101110111110100011010100110011111000110110011001111101011011110111111011001001101111110101111101100000101111100100011101010110001000001110100110110000111101000110000100111010001111001100110110010111111001001011110000010100011010001111110100100111110111011001000111101000010011100010111110100101011100111111001100011110001011011110101111010000100001100110111101111110010110010010011100110111000101100001001110001111010000111101011010111111000100111111101000101001"
                    "111111110101010001010011110011110011010100011000000111101000100111101100111000110010011000100110111000110100111001111001100100011001110011101100010001100111101100001011101000011011110111010001100011100110110110010000010001001000100010000010110111100111000110110010010000001100011000011011011001110110000010110010000110111001110100011001111011100011010000110111001110001010001010000011001100011100100100111000101100111011000001110110010000110010000010100010000110011011011000001010011110111100011101000110111100111111101000101001"
                    "111111110101010001111010010000001011110011001100010110111001111100101111000001010010011011100011111010110010111110111001110010000111011011110010100100000100011001111101001001111001001111011110100010100000101110110011111101000010111100001011110001000100010100100001111101101111100101101110011110100000010010111010000011000101001111000101111011110100010000010110100011110011001110111011110010011000101110001000111100000101101101001001111010000011001000111001000110000101111001101101000001110010011110100100100000111111101000101001"
                    "111111110101010001010001011110000010001100001001110100001110101100001111110001001101011101111000101110100100110001111101000111100001010010011111001100100110001010011111001110001011111010011010001001111100101001110111111001110111111010100011001100100111100110101100011111101011100000100110010100000010111100110001011011111101101100000101111011011000010011110100000101000111101101101001111000011001100001011110111101011111011001001100110001111011110101111001000101011000011111001100001011111101011110100111100010111111101000101001"
                    "111111110101010001010001111011110010010001110111000110010011001110001010001000010000010010011101110000110011100011010001001001111001111011101011101111100110000110101000001111011011001111010011101111001100110010110111000001001100111101110011110111100100010110001011100111101101110011000010011010001110011110111101110000101101001100001100100010011011001000000101101100010000001101100110110000011011000100000010100000101010000001111011100110111010011100111100110100100000100000101110011101000010010100001100110000111111101000101001"
                    "111111110101010001110100111100111011010001110010000100010111110110001110110001110001011111011010000010111100111011100101110011011101000011111011100011010110100001111011001111001000110110011101000011010000111101010000100001110101110000011011101011110011100111001001101000001110001100111001011111011010000100111000100110100001110100011000010010100001111000010101111001011110001011101011111000011110010011011000101100111110001001111010000110110011100001000110100101001111010000001001000011111011011101001110000110111111101000101001"
                    "111111110101010001111110100001101010111001110111100101110100000110001110010111110001011101001111100100110011100001011101000111100010001011011100100011100101011001111100001101100010011110010000010110111110111001010111111001110111000101111011000100100111110101001000001111001101100010011110010110000100111000101000010011110001010001110111111010001100010011100100011110111011101001111101110011011000100111111010101111001001000001011110100001000011000111001001110100111110110000101010000100001111011101000111110010111111101000101001"
                    "111111110101010001110100000100111010101000000100000100000100010100001110111000101000011011110001110010101000100010000001001110111101100011000101110011110110010010110000001111011110100010010100100000010000100011000110010001101100010010000010001101111011100110100001011000001010000010100000011001010011000000111000011001001101111011110100010010100000100000100111001010111000001110100000010111010110001110110000110100011011100001110001100101100011110110010111000100011011100011001000101000010000011001010011000000111111101000101001"
                    "111111110101010001110100011000010011111001000100110100010111111011101011000101111110011101001100000010100010111100000101110001000110010011111101010111000111010001110001101110110011110011011101000110000100100101111101100001111110100101110011011011110100000111110110001101101000110011111001010010111110110000111010000110010001110110111000100011011001111000010100110111110100001111100011000101011011100111110100110100001111011001100110111100010010101111001000000101110100001111101100011000111101011110100010000010111111101000101001"
                    "111111110101010001111110010110001010001000100011110100110110001111001001110011101111011000111100101100100001101101111001101110101110000011111100101110110110101011111000001111100010111001010111110001101000101100011001111001000110010001110010111110000110100100001000101111001000111100010001011101101000111110110011110001001101000111000100011010001111010001000101111110011100101101001000011111010111000001101110100001111011011001011000111011111011001111100010010111101011111011001000111101110111010010100001111000111111101000101001"
                    "111111110101010001100101100111000011000110110000110101110011110011001011000001100010011011001000100000100001101100001001110010001011100011010000010001100100001100110010001011011000100000011101010001110000101110001111011001111011110010001011100110001000110110100010110000001100110001000001011000010110011100111010010001110001000110000110100010010000010100000110110000100000101011011100011000010001100011010000110001100100000101101101100110000011010010000110000110000100011011101001110011101000011101101011000000111111101000101001"
                    "111111110101010001111010000100001011100100000011010111101101100001001110100000110100011111101110100010110010111000100001111010100000010010100011110001000111110111000110101100000100111001011110100010000100110001001111011001010001111100011011100001000110100100010000111100101000111100101111010000101111010000101011111101110001111101001000110010010000111110110111101100110010001111101000001011011110100000110110111010000110100001111110010010111011110010000100010110010011100000101001000111110011010100011111001100111111101000101001"
                    "111111110101010001011010000111000010011100101100000100011011000111101000011000010111011100111110100110100101000111100001111011001111101010111011000001110101111101100000101001100011001111011010001011111000100001110001011001101110100111000010111011110111110100000111001011001011001100001111010110010000001110100011010001110001001000110011111010000111100100010101100100001110001110010100111111011101111001001110111101011110001001001100101110000011111100010110010100100001000111101101001000111110011011010111100000111111101000101001"
                    "111111110101010001110110100000110011110111101000010100001001000010001100100001100111011101110110000110111011000010011001110100100011100011000010001000110111001000100111001011001110110000011011000011001100111101110000101101110000110100110010010111011100000100001011110111101110100001000111011000011010010000101111001111010001001000011001100010101110001110000100001101100001001010111100111100010000100111001110111001010111000001001100011100011010010001110011100111101101001110001111011101000110010110110000010000111111101000101001"
                    "111111110101010001110100000111011011101011000100000111110010000101101110010011110111011111101101000110111101110011100101010011111101110011100100001110110111010011000000101111001010000100010000101111100110111010111001100001101011100001000010111000010111110110100001110010001000100111110110011101101111001100111010011000000101111000100001001011011110101111100111101000110001101110100000110001011111011010100000110010111000010001100000101110100011111011000110110111110100101100001111001100110001011111010000010110111111101000101001"
                    "111111110101010001111100100111010010001110100110000111001111101001101100111101001100011011000101111000100110001000011101110001011111010010111100010000010101110011000011101111010011110010010000111010000110100010000101111001101001100111111011001000010111110100100000010111101100110100011110010011111001110110101111100011101101001010000111100011111001011100100101111001110011101000110100001110010001111001010000110011010111100001100110010011110011110111110101110111111011011101001000110000100111011001001101111110111111101000101001"
                    "111111110101010001011011100001100011000110000101000110010000010001101001100000110100010110110000100000111000011001001101100000100110111010110001101000000100100001100110001010001100011000010100111100011110111000011010001101000101100110000011101101011000000110110001110011101110101101111000011001110011010000101000011000001101100010000010110011001110111001100110111000111001101000100111011100011001001110011110110110110000110001110010010011100010011101111011000100011000110010001001000111011100011101100100110000111111101000101001"
                    "111111110101010001111100101100111011111100101011100111110100110111001111001010000010010011100100111110111001011000010001111010110000110011111000011010100110100000111000101111000110110010010100011110000100100010111111011101011110101111000010010001111001000111100010011001101110100111000110011101110111101000111100010001001001111011000110001011010000011110110111101001000001001001011110000010011000100011110110101110110011111101101001110000010011110010010000100111111001001011101111000100001001011111001010000110111111101000101001"
                    "111111110101010001011001101111000011011010011110000110011000010111101111010111110110010111011100111100111001111010111001000010010011110011001010011111000110111010001110001110110101111100010101000011110000111010111110010001110101111110011010110000011011110101111100011010001011000111011111010001100110011110100101000011110001001001001111000011000111101101110100111001100111001011000000101110010011001100111100111101101011111101101111100010010010100011101111110110111000100011101011101000000110011101100100111110111111101000101001"
                    "111111110101010001100100010011000011001101100011000110110100001000001001100111101110011111011110100110110001110110010001001000001000001011100111010000100101000000100000101101110111100111010011011000010000110010001110111101101100010001000010000110110010000110001001011000001000010010000001011010000100000110111101000010111101000100001110111011001110000110100110000110100010001100100000010110011101110001100110100010000100100001110001100100011010000010110000110111010000110111101010100000010000010010001100110000111111101000101001"
                    "111111110101010001001011111011000010011110110111110101110101111100001011110000101111011110100100000010111110000100101101100100011110011010000101111101100110001101111001001110110011100100011110100001010000111110100110111001100110011110001010001110100111110111101100011010001111110101101111010010111110001100111101110011101001110100011010000011101000111001100110100011101000001110010011100011011110100110001100111110000101011001110010110000001011111001100001010111101100111001101111100100010011010010111100100000111111101000101001"
                    "111111110101010001001000010111100010011101000110000110111001011100001010001001111000011001110101110000100111000110111001110101011111100011001011001111110110011001000111101001010000111100010110001011100000111110100111010001000101100011111010111001000000110101100010111000001101001100111111011001001000111110101001011110000001011111001110011010110000010111000100000100010111101010001100011111010100000100111100111111001011000101000111100001010010001110001011000100110010000011101000100010001111010110001011100000111111101000101001"
                    "111111110101010001001000111000111011001000001001100100110000111011001100110011100111011010011100111100110010011101111001000101111011110010100001110001110100011000011010001101111011100010010100110001100000101100011110111001111011110101000010000010001101100100011011001000001101010000110000011010011100011110111010110011110001100001100001010010100100000100000101100111011000001111011110110011010001101100010000111000011000101101111001110101100011011110011100010110110000100100001000011000110010011100100001011100111111101000101001"
                    "111111110101010001111011011000010011111101011011110111101010000010001111001011000110011111011101100100110100011100000101010111110001100011010111111011110100011110100111101101011110001100010110111111001100111110100010001101011010011111100010011101011111000111110110100010001110001101110001010000001011110010100101111110011101100001011110011011110100111011100111001001110001101111010001000001010111000100111110100001011110010001101110111110100011111011010001000111001011000001001100110001111001011110110110100000111111101000101001"
                    "111111110101010001000101001111000010011101001100000100001100010111001000001101000111010001011011111000101011000011111001011001100111100011011110000100110101101111011111101011111101110001011110110111110010110001111101010001001111010100000011100011110101110100110000100111001010001000111100010110111011111000111111010011001001001111001110111010111011000111000101100000100011101010010000111100010010001101111100100100110111110001100111101011000011110010001111010111011101000111101100111110100010010110000010011100111111101000101001"
                    "000000000101100011000010100010011001001101001100011011101001110001110100100111101011001000010010100110001100100111001010000100010111001000100111000101000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                    "000000011010011100111101011101100110110010110011100100010110001110001011011000010100110111101101011001110011011000110101111011101000110111011000111010110000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                },
        /* 63*/ { BARCODE_GS1_128_CC, -1, 3, "[01]12345678901231", "[91]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[92]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[93]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[94]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[95]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[96]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[97]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[98]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[99]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[91]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[92]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[93]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[94]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[95]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[96]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[97]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[98]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[99]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[91]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[92]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[93]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[94]123456789012345678901234567890123456789012345678901234567890123" "456789012345678901234567890[95]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[96]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[97]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[98]123456789012345678901234567890123456789012345", 0, 32, 579, 1, "Example with CC-C 30 cols, 30 rows (max)",
                    "111111110101010001010100000100000010001110111000100111011111011101001100011100011010011111011010011110111011001000011001011011110011100011110111101010000110000110010100001111100101011111010001011000110000101000001100011001100100010000110011010011110111110101100011000010001100110110110000011111011110010110110010001000011001001011100011100011001000010011000100010110000011001000100001000001011011110011101000100001001000010001111101101001111010111001110010000111101001001111001001000010010000011100110111011110101101100010000001110101000111000010101111011110000111111101000101001"
                    "111111110101010001111101011011100011100110011101000101111010111100001111000100110110011111101001001110111001101110010001001100111110001010101111000010000111100100011011001011010011111100011110011011010000100111011111101001111100110110011010001110100111110111010000111011001001110100111110011000110111101000110000100001110101000111001011111010101111100110000111110001010110001110000001011001011111011010000100100000100011110101101011100000100010111100010111100100010000111101001001101111110110011111011001100110111001011100110001110000010011001011111010100001100111111101000101001"
                    "111111110101010001010111000111111010110100000111000100010000010111101000110100000111011111101001100100111110100111000101011100010001100011101110101111000110010010001111101001001000001111011101101011111000101100000110111101000111101000010011011101000011100100110110111100001100010010111110011101101111110100100111100011011001000111001000011011001001101111110111010111111011001110011110010111010011110001100110101110000010110001011000001001110010110110000111100100001000010111101101100110011111010101101111100000100101110011111101000011001000111011111101011001000111111101000101001"
                    "111111110101010001110100100111000011000011011001100110100000100110001001100011000010011001010001100000110000100100001101110111101100010011111001101011110100100001100001101001000010000100010110011100011000101001100011000001111011110100001010000100100001000110010000110011101100110011000011011011110001110010100100001001000001100001101100011011110011010011100110001000010001101100010001100111011101111011001000100111000011101001010000011011000011101100110011100110001011101111001111011110101000011100110111011110101000110011000001000001011101110011010011000011100111111101000101001"
                    "111111110101010001010111100100000011110010001000010111110110001010001110000001011010011100101110000110111001011000000101100010111001000011100101111011100100111101011110001110101100010000011111001000010110111001001111011101111110110100011011110111001110010101001111110111001110010000111011011101001100000010111100101000010001000010111110011011010111111011110110010111000100001111001010100000011101000001110110111000110111010001100101111001100011111011010100000100100111111011101110111011110100011111000011010010111110100001001101011100100011111011101011100000110111111101000101001"
                    "111111110101010001111110100110100011010001111110100100111110111011001000111101000010011100010111110100101011100111111001100011110001011011110101111010000100001100110111101111110010110010010011100110111000101100001001110001111010000111101011010100011111000111101001111001001000011101000011010001000010111100110100110011111101100100001011111010010000001011110110011010001111001001111100111011010111110001110110101100110000111101111001000111101010110010011100000110010001011111001000111101001000010011110110000110111100010111101001010001001111000011101011111101100111111101000101001"
                    "111111110101010001010011110011110011100011010000110111001110001010001010000011001100011100100100111000101100111011000001110110010000110010000010100010000110011011011000001010011110111100010111000111000100100011011000010001001110000111010011110011101001100101100110000100001001101100000100011100100110011110100111001111001101000001000100001010100001100000110110001000001011001100111011100110011011100011100110100010011101110001100100111001111011011011000011000111001001001110001000111011110110011101000001001110100101000010000001100110000010010011101000110011110111111101000101001"
                    "111111110101010001111010010000100011000101110001000111100000101101101001001111010000011001000111001000110000101111001101101000001110010011101110111101000111111011110110101111101100101000011000100011101000111101000000101001000111011111101010001110101111100111101101110011001101101111000100011010011101000000111110100100110001111000011011010011001000111001000111010110001000001100100000011101011111010001000110110010011101000001111001001000010011111100100101110111100010000100101111010000110011011110010000010100110101110000010001001000111110110011110100100100000111111101000101001"
                    "111111110101010001010001001111000011110101111001000101011000011111001100001011111101010110001110111110111100100111101001110100111110010011001001111110100100001110100110001011001000011100011111101011101100100001110001001101111110010011001011001111001000110111100101111010001001110100011000010000001010111100110001000101111101111011110101111011010001101111110100011000010011101100111000101110010111001000000110101101110111110001001000110011111010001010111100000100011110010100001000010001001111011011111000010010100010011001111101100111101000110011110100111101000111111101000101001"
                    "111111110101010001010001111011110010100000010000010110111011110011101001101100001000011001000111011110101111011110100001101110011000100010000100110000110111110110100111101100010100110000010010111001110000110101000011000001110100100111000010100100000100000110110001000010001101111011110011010010001110111000111011100110011001010000011000110011011010000100000101111001111000101110111000101000011101100010000110111001101011000001100101110011110010011100111001000111101000010111101000100111011100010001001100110000111000101000111001100101100011100011010000110001110111111101000101001"
                    "111111110101010001110100111101110011111010011011100110011001111000101000111010011111011110110001101000111111010110111101001011111000110011110111001110100111101000110110001011000111110010010110111111011000110101111100111001001011111000110011111001001011000101001111000001001010011110100000010011011111100110101111000101111001111000001001010011110010001001000111111011101101101110001001110110011110000101101100100000101111001001010011111001100011111010100000110100111101101111101011101011111000010111100001011110111101001000000101111100001001011011101001110000110111111101000101001"
                    "111111110101010001111101000011101011000100010111110101101100011110001000111000100110010100101111000000101111100111001101011000001011100010000010001011110101000110001111101010000010011110011111100101100010100011110000101001000111000100110011100101001111110111111001011001001001010000011110011110001011110010110100111111000101001111010000010010110010000001110100011101100111001101110100000111011011101011100000100111010001100001101110010111000010100010011110000110011101011100001001110001101110011101010111111000110010110011111101100110010001111011110100011110100111111101000101001"
                    "111111110101010001110100000100111010100000110001100110010001000011001101001111011111010110001100001000110011011011000001111101111001011011001000100001100100101110001110001100100001001100010001011000001100100010000100000101101111001110100010000100100001000111110110100111101011100111001000011010010000110000111001100001011001000010000010001011110011010111000110010000010011001110011000010110010000010100100000100011011001000001001000000100100011100101000111000111010101110000001001000011011000011011110111100110110010001100011101110111001000001011001010001100000111111101000101001"
                    "111111110101010001111010001100110010011101111110100111110011011001101000111010011111011101000011101100100111010011111001100011011110100011000010000111010100011100101111101010111110011000011111000101011000111000000101100101111101101000010010000010001111010110101110000010001011110001011110010001000011110100100110111111011001111101100110011011100101110011000100001101111101001001011110100000011110100011011000101111101000111001010000011110100011111101010001110111001100111010001011110101111000011110001001101100111111010010011101110011011100100011110100010000010111111101000101001"
                    "111111110101010001111100101110100011011110000101100111010111110000101110001111010111010011011011110000110001001011111001110110111111010010011110001101100100011100100001101100100110111111011101011111101100111001111001011101001111000110011010111000001011000101100000100111001011011000011110010000100001011110110110011001111101010110111110000010010111001111110100001100100011101100111110010001010100000010011110110011100100111001111001011111011011011111000100100110001110100011101100011111001010010011110111001110111111010011001001111101001110001011100101001111110111111101000101001"
                    "111111110101010001100101100111000010001001110111000101100000110010001110110100000110011011011011000000100101110011100001011001111001110011100010010011100110110001110011101111001101001110011000100001000110110001000110011101110111101100100010011100001110100101000001101100001110110011001110011000101110111100111101111010100001110011011101111010100011001100000100000101110111001100010101100000011100110100011000111100111101000101101100100000100011110110010111000110001000100001101101100110110000011000011000100010100100111100111101010000010000010011101101001100000111111101000101001"
                    "111111110101010001111010000100100010011001001111110111101011100011101110010001100001011100100111011000111110110100010001101011111011100011000010011101000110111001111101001100100111000010011110001010001000110010011110001101110101111011100011000010111100110110010111100110001111101101010000010010011111101110111011101111010001111100001101001011111010000100110101110010001111101010011110001000010000110101111110110011100111110101101101111000100011101011101100000100001101111101001110101110011000011110111000111010111010001110011001000100111110011010100011111001100111111101000101001"
                    "111111110101010001101101000001111010110000100111000101101000000111001101110000101110011010110001111110101110010001100001001101001110000010110001100011110110000110101111001111101000011101010011000100001110101000100111100001110010011111001011101111101000110111101011110100001111010111100001010011110100001000101101000011100001000110001011100011110001011110100101000100111100001000001010001111010000001010011110110010100011111001111110010011010010110001000001110100100010000111101100111110110110010111001000011000101001100001111101101111001100111011100101111110110111111101000101001"
                    "111111110101010001110110100000110010001101100001000100111000011101001111001110100110010110011000010000100110110000010001001000011001100011000010100110000100010011011000001111010001001111011001110001100100110001001101110001101001110011110011001000110011100111011000100011001111011000010111010000110111101110100011000111001101010000011000110010010100001000000110011000001001001001100110000100010000010101000000100100001100001101000101100110000011000010001000110111011011001110001111011100001011011000110000101000110010000010001101001100000110100011100100110111100111111101000101001"
                    "111111110101010001110100000110010011110100000010100100011101111110101000111010111110011110110111001100110110111100010001101001110100000011111010010011000111100001101101001100100011100100011010000111001000110010011100000101111010110110000010001101111100010111101000101000001011101100111111010000101111000010111100010001000101001000011111011011111001011011100111101000000100101110100000110001011111000010010110111100111011101001111101101000010011111100101011100111110100110111001111001010000010010011100100111110111001011000010001111010110000110011111010000010110111111101000101001"
                    "111111110101010001111100100111001010000111000100110111111001001100101100111100100011011110010111101000100111010001100001000000101011110011000100010111110111101111010111101101000110111111010001100001001110110011100010111001011100100000011010010011001111100111101001111001001100100011111101010111100011001100101111001000010001010001100011111011011000001011110110110000100111101000001010001111011011010011110000110011000010111101111010111110110010111011100111100111001111010111001000010010011110011001010011111000110111010001110001110110101111100010110110000011110111111101000101001"
                    "111111110101010001011011100001100011010100001100000111010010011100001010010000010000011011000100001000110111101111001101001000111011100011101110011001100101000001100011001101101000010000010111100111100010111011100010100001110110001000011011100110101100000110010111001111001001110011100100011110100001011110110001001110111101100010100001100011001101100011000110110100001000001001100111101110011111011110100110110001110110010001001000001000001011100111010000100101000000100000101101110111100111010011011000010000110010001110111101101100010001000011101100100011000111111101000101001"
                    "111111110101010001111100101101110011000011011110100111100100001010001001100011111010010100111100000100101001111010000001001101111110011010111100010111100111100000100101001111001000100100011111101110110110111000100111011001111000010110110010000010111100100101001111100110001111101010000011010011110110111110101110101111100001011110000101111011110100100000010111110000100101101100100011110011010000101111101100110001101111001001110110011100100011110100001010000111110100110111001100110011110001010001110100111110111101100011010001111110101101111011111001010000110111111101000101001"
                    "111111110101010001011001100111100011011000100111100100000101101111101100101011111000011111100101100100100101000001111001111000101111001011010011111100010100111101000001001011001000000111010001110110011100110111010000011101101110101110000010011101000110000110111001011100001010001001111000011001110101110000100111000110111001110101011111100011001011001111110110011001000111101001010000111100010110001011100000111110100111010001000101100011111010111001000000110101100010111000001101001100111111011001001000111110101001011110000001011111001110011011101100101111100111111101000101001"
                    "111111110101010001100100010011000011011000100100000100011011110111001101000010110000010100000101000000110010100110000001110000110010011011111011110101100100010111011100001111001101011100011001000001001100111001100001011001000001010010000010001101100100000100100000010010001110010100011100011101010111000000100100001101100001101111011110011011001000110001110111011100100000101111011110010100011011100110100000100001101110110001001011111011111011101111011100110111100111001011001000110000110010011000111011000010110100111011110001000100011011000011011001110011100111111101000101001"
                    "111111110101010001001011110000100011100100011100110101110010001111101111010011001100011101101110001000110110011110000101001101111101000011111000110001010110111001111101001101000011110110011001101111000100110101110000001001111001001100011010010000111101000101000001111010001111110101000111011100110011101000101111010111100001111000100110110011111101001001110111001101110010001001100111110001011101011110111000111000110001110101001001111101100010010011110000010111110010100011001111100111011001011100001011000010111010000011101101010111100100000010010111100100000111111101000101001"
                    "111111110101010001001000010011110011100111100101110100111100011001101000010100001111011001100010111100110011111001010001100010111111001011101111100010110111010111111001101111010111110110010001111011101110110001110100111001100011000101111011110111010111110110010001001111101110011111001011010110100000111000100010000010111101000110100000111011111101001100100111110100111000101011100010001100011101110101111000110010010001111101001001000001111011101101011111000110110000100111101011100100000110011000101111110010100010001101111101001110010110000011100100011111010111111101000101001"
                    "111111110101010001001000111000111011000100011001110111011110110010001001110000111010010100000110110000111011001100111001100010111011110011001110011001000101100001100001001111011000100111011110010010111100111011100010000101111011001000111011010110111000000110010100001100001000010011000011011000011011001100110100000100110001001100011000010011001010001100000110000100100001101110111101100010011111001101011110100100001100001101001000010000100010110011100011000101001100011000001111011110100001010000100100001000110010000110011101110111011000011011001000010011000111111101000101001"
                    "111111110101010001111100100100011011001001111000110111010111101110001100001011110011011001011110011000111110110101000001001001111110111011101110111101000111110000110100101111101000010011010111001000111110111010001110011001110110111100110010110101111110000111101001100001101000001011110001011110010001000010111110110001010001110000001011010011100101110000110111001011000000101100010111001000011100101111011100100111101011110001110101100010000011111001000010110111001001111011101111110110100011011110011000110010110110000111101001001110111111001011110110110100000111111101000101001"
                    "111111110101010001100010100011111011011101000111000100010010011110001001100011001111010001101100011110110000111110100101010000110011111011000011111010100100010011011111001010001000011110010011110011000110101111000110001101111011011111010010111001000011000110010010011111001110000101111101010101110011111100111111000101100101111110100110001011111100101110110100010011011111001001100011011110010111101100000110101111011001100001111100010111001010100010111100000110100100011111001110111100101110010000111010001100111111010011001001010001000011110010110000110011110111111101000101001"
                    "000000000101100011000010100010011001001101001100011011101001110001110100100111101011001000010010100110001100100111001010000100010111001000100111000101000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                    "000000011010011100111101011101100110110010110011100100010110001110001011011000010100110111101101011001110011011000110101111011101000110111011000111010110000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                },
        /* 64*/ { BARCODE_GS1_128_CC, -1, 3, "[01]12345678901231", "[91]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[92]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[93]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[94]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[95]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[96]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[97]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[98]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[99]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[91]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[92]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[93]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[94]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[95]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[96]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEF", 0, 32, 579, 1, "Example with CC-C 30 cols, 30 rows (max)",
                    "111111110101010001010100000100000010001110111000100111011111011101001100011100011010011111011010011110110100100110000001001100110100000011001100100100000100011000001101001011001100001000010001100001100010100001011000001101110111101101000010000010100001000100100001100110001011000111000011010011000001100010111011001000011001110110100000110010001010000001000100011000110000101100110000100001011110110101110000111011001000110001100100010110000011101100101100000100101110111000001100001011001110011000010000100110110010000100110001110011101000010010101111011110000111111101000101001"
                    "111111110101010001111101011011100011010001110000100111010001100010001010011111101110011111011001100110111010000011010001111010001000100011111101110110110111100110110010001111001011000011011110001000100010101110011011111101111110011010011011110000101000010111100100100001001110100111101110010001000111100100111101011101110001110101111000111011111101100101100100010011110100001110001000110010011111100010101110101101111100001001111001010000100011110101100110000111010000111011001111101100010100011000100011110110111111011010001101111101100000101011111010100001100111111101000101001"
                    "111111110101010001010111000111111010010111011111100100001101000011101111110110011101011101011111000100110110001001111001001110011001110010011000011011110111001111100101101110110001011111010010010111100000110001101011110001101110101110000010010110001111100101001011110000001101000100111110011100110100111110111010111110000101101110011001111011100100011111010110111111001100101010010011110000011101011111010000110011000101111001001001100011111010001110110001110111001110101111001110111111010010010000011010011100110111000101110001100111000101110011111101011001000111111101000101001"
                    "111111110101010001110100100111000010001001000001000100011000110100001010001100110000010010011001100000100110011000010001100111000110010011001010000110000111101111001101101110010100011100011001001110111100110111001101000001110011010000110010111011110110000100010110110000001101000000100011010001001111011110110110001100001101111011110100100011110010110111110111010011011110001101100010100000010001001100110000110110111100111101101000111011110010000100010000010110011000110011001000001000100001010110110000010000110110001100001101001001100110000011010011000011100111111101000101001"
                    "111111110101010001010111100100000010110000101111110111101011101110001010111110011000011010111111011110111111010110111101101000000111001010011010001111110110011011110001001111010010000010011110111011101000111000100000110101001100111111011010001000111100100110100111010000001110010110000100011010000011100010100001011110001001111000100001001011111011010000100110100111100110001110010001101000010000010111101000111111011010110001111011011000010010111000010111110111000001011010001110101100010000010010011111000110111011011100001001011011111010000011101011100000110111111101000101001"
                    "111111110101010001111110100110100011100010011111010101000001101111101101110010001110010100100001111000110010010011111001011100011000111011001111110110100111010111111001101111110100111011011101111100010110110001111101010001110100011111010011111101000011010111010011111101101011100100001100011011100010001110111101101111101001100100100111110010100011011111000100110110001111001111100001011101010011101000110000100101011110000001011000010000111010001111000100100111001010111111001011100100001100010100110111110000111001001111100101111110101100010011101011111101100111111101000101001"
                    "111111110101010001010011110011110010010111110111110111001101000110001100001010000110010010001100001100111100110010111001100101101110000011000011010001000111000111001010001110110000100110011110011110100100101000001100110001110110000010011011011001100001100110001011100111101000101101100000010110110000100000111010101110000001000011001100001010110011000010000100011000011000101000010110000011011101111011010000100000101000010001001000011001100010110001110000110100110000011000101110110010000110011101101000001100100010100000010001000110001100001011101000110011110111111101000101001"
                    "111111110101010001111010010000100010011001001111110111010000111001101111001011100111011001011100000010111001011000001001111101000010011010011001111100010111100011001101001110110111000100010111000111111010110100011100001001110100011000100010100111111011100111110110011001101110100000110100011110100010001000111111011101101101111001101100100011110010110000110111100010001000101011100110111111011111100110100110111100001010000101111001001000010011110100110001100111100010011011001000010011111011010100000111100100111101001110011101111110001101011011110100100100000111111101000101001"
                    "111111110101010001010001001111000010111101000000100110000011101011101111100100111010011100010111110100100111000001011001101110000010111010001111010100000110111000100011101010001110111111011011110100110000111111000101101001000110101110000010111100010100000111010111110001001101100010011110010011100110011100100110000110111101110011111001011011101100010111110100100101111000001100011010111100011011101011100000100101100011111001010010111100000011010001001111100111001101001111101110101111100001011011100110011110111001000111110101101111110011001011110100111101000111111101000101001"
                    "111111110101010001010001111011110011110100110111110111110100101111101110011011001110010110110000100000100001001110111001101110111100111011000011010000010111101111001101101100010011000111010011000011001000100010010000010001000110001101000010100011001100000100100110011000001001100110000100011001110001100100110010100001100001111011110011011011100101000111000110010011101111001101110011010000011100110100001100111101110101100001010111000111000010000101000010000100010110001100001011001100100000011101011011110000101100111100011101000011011000001011010000110001110111111101000101001"
                    "111111110101010001110100111101110011110100011011000111011000111101101100011001111010011100100110010000100010111110011001111001100110001011100010011000010111100001101101001100100111001000011010001111011000111100000110110101000111010111110011110000010110110110101111110111101111110101101111011010000001110010100110100011111101100110111100010011110100100000100111101110111010001110001000001101010011001111110110100010001111001001101001110100000011100101100001000110100000111000101000010111100010011110001000010010111110110100001001101001111001100011101001110000110111111101000101001"
                    "111111110101010001111101000011101010111010001100000100111110110001001001000010111100010110000010011100100111101000010001100111110100001011010110001111110100110110001111001000101000001111011001101001111000111000100111110101010000011011111011011100100011100101001000011110001100100100111110010111000110001110110011111101101001110101111110011011111101001110110111011111000101101000011100010110011100101001111110101111110011100101101100100001111011001110001001110101111000001101101001110100000011010011110110110000110110000010111101111010001111010011110100011110100111111101000101001"
                    "111111110101010001110100000100111011000010100000110110101110001111001101001101110000010110000110001000101110011100010001011100011110011011001100100001000110110000100001001001001100001100011110101100111110100101111101111101110011010001100011000010100001100100100011000011001111001100101110011001011011100000110000110100010001110001110010100011101100001001100111100111101001001010000011001100011101100000100110110110011000011001100010111001111010001011011000000101101100001000001110101011100000010000110011000010101100110000100001000110000110001011001010001100000111111101000101001"
                    "111111110101010001111010001100110011111101110010100110000100111001001110000010111011011011011110001000110010011110110001100010001110010011001011100001000111110100000101101110110011100001011100111011110010100110010011111101110100001110011011110010111001110110010111000000101110010110000010011111010000100110100110011111000101111000110011010011110010100100000111010001100100001111101001000011011111010010110000111111001001011101110100011110111011110010100000010111110011011001101111101100010010010001011110001000110010001110100001111010001110111011110100010000010111111101000101001"
                    "111111110101010001111100101110100010111000110111000111110001011101001100011100010111011110010111100100111010010111111001011000011011110010111001110111100110110010011110001111110100011010011011111010000010101111010000001001100000111010111011111001001110100111000101111101001001110000010110011011100000101110100011110101000001101110001000111010100011101111110110111101001100001111110001011010010001101011100000101111000101000001110101111100010011011000100111100100111001100111001001100001101111011100111110010110111011000101111101001001011110000011100101001111110111111101000101001"
                    "111111110101010001100101100111000011100110011001110100100000010010001110010100111000010100100000010000101000110001100001101100001010000011101001000111000100110110000100001110010001000111011100011101101100111101001101111101111101001011111011100110110011100101101100001000001000010011101110011011101111001110100110000011010001110001110011011011010011000111000111011010000110001111101110101110010001110011101000101001000001000001010011110011110010111101111010000111001111011001001010000100010000010100001100110000101000100000010001100110110110000011101101001100000111111101000101001"
                    "111111110101010001111010000100100010110001111110110111100101001000001010000111110110011010011100010000100100111110001101110001011001000010001011111101110111101000100000101111001001001000010000110111110010111101000110110001110110001111011011000110011110100111001001100100001000101111100110011110011001100010111000100110000101111000011011010011001001110010000110100011110110001111000001101101010001110101111100111100000101101101101011111101111011111101011011110110100000011100101001101000111111011001101111000100111101001000001001111011101110100010100011111001100111111101000101001"
                    "111111110101010001101101000001111010001110111011110101111000010001001011110001100011011010001001111100101100100001110001111110010011001011011111010010000100000101000111101011000111011111011101000111110100101110100011000001001111101100010010010000101111000101100000100111001101110100111000010011111011000010100111100110011001011110111001110010010001101111100101100100000111001100101111110010011011110000101100101111100111011001100010100011111011001111010001100111111001011000101001010001111000011111010000111010111101110101111101011011000011110011100101111110110111111101000101001"
                    "111111110101010001110110100000110011101000001011100111011100011001101111001100010111010110111110011110110111000111001101110001101001100011110111001011000110011101100100001110110001001100011010000100110000110000101000001101101011100011110011010011011100000101100001100010001011100111000100010111000111100110110011001000010001101100001000010010010011000011000111101011001111101001011111011111011100110100011000110000101000011001001000110000110011110011001011100110010110111000001100001101000100011100011100101000111011000010011001111001111010010011100100110111100111111101000101001"
                    "111111110101010001110100000110010011110100011000110111110010001011001001011111001100010010001111001000110010001110100001111010000001010011111010101100000101111100101110001100101111101110010000100001111010111111011100101001100011011110100010111110000101110111101011011000001001100100111111011100010011001000111101000001101101001011111000011011110100000010100111100010000010101000101111101100011110000110110010101111010001111001001000011110010011110001000100100110010000111010001111100111011001011110100110000110111110010101100001100010111000100011111010000010110111111101000101001"
                    "111111110101010001111100100111001010000111010000110100101110011111101000001101001110011111100101110110110010011111100101000010001011110011101110010111100100001001101111101110011001011111010010110111110000101110001101110001111100010111010011000111000101110111100101111001001110100101111110010110000110111100101110011101111001101100100111100011111101000110100110111110100000101011110100000010011000001110101110111110010011101001110001011111010010011100000101100110111000001011101000111101010000011011100010001110101000111011111101101111010011000010110110000011110111111101000101001"
                    "111111110101010001011011100001100011011000110011000110011001100110001100110000010001011010010001100000100100010000001001110111000101000011110111010011000100001101101000001101100110000110011110110100111000100100001010000001000010011100111011110101001111000100100011101110001010000100010000011110001010111100100000101100110001001000110011000011110110001001110111001110100000101011110001111001011010110001110000110010000010011001011000110010000010110001111001110111011000001001101110100000101110011110001001011110110100110001110001110110100001100011101100100011000111111101000101001"
                    "111111110101010001111100101101110011101100001110100100111110110111101110101111001110011110100100100000111100111011100101000000101111010011010011110011000110100111110011101111101001100111011111011000110110101100011111101101111001010010000010100001111101100110100111000100001001001111100011011100010110010000100010111111011101111010001000001011110010010010000100001101111100101111010001101100011101100011110110110001100111101001110010011001000010001011111001100111100110011000101110001001100001011110000110110100110010011100100001101000111101100011111001010000110111111101000101001"
                    "111111110101010001011001100111100011000111110100100100111110011001001000111100101000010100110011111000101011001111100001111100100111001010011101101110000101111010000010001101110010001110011110111101011110101111101110001101001100100111000010111101111011110110110001011110001000011100101100010111100001000010101111101110011001100011110010011011100010111110100111001101011111001011110001000100010010001000011110110010010001111101100110010111100010011110000100100111011110010011101101101000011110010110011001111000100100011011111001011001000001110011101100101111100111111101000101001"
                    "111111110101010001100100010011000010010111011100000110000101100111001100001000010011011001000010011000111001110100001001100101001100000010100000110110000110100001100011101011000001100010011110010010111100111010000010111001110111000110011011110011000101110101101111100111101101110001110011011100011010011000111101110010110001100111011001000011101100010011000110100001001100001100001010000011011010111000111100110100110111000001011000011000100010111001110001000101110001111001101100110010000100011011000010000100100100110000110001111010110011111011011001110011100111111101000101001"
                    "111111110101010001001011110000100011101000011101100111110110001010001100010001111011011111101101000110111110110000010101010001111001000010011000111110010100000100111100101101011110110000010010111110001100101110101111100001110110000111001011100000100110010100101111100110001101000011110110011100010110000010111110010001001101101000111000001011010111100011000110100111010000001001001111001000011101011110111000111011000111000101001011110000100010000110011111010110101111000011001110101110000110011110010000110110111101000000101001111000100000101010010111100100000111111101000101001"
                    "111111110101010001001000010011110011100111010111100111011111101001001000001101001110011011100010111000110011100010111001011100110001110010011110001101100110100010111110001111101101111001010001101000011100100001110100001101001011100111111010000011010011100111111001011101101100100111111001010000100010111100111011100101111001000010011011111011100110010111110100101101111100001011100011011100011111000101110100110001110001011101111001011110010011101001011111100101100001101111001011100111011110011011001001111000111111010001101001101111101000001011100100011111010111111101000101001"
                    "111111110101010001001000111000111011001100011001100100000100010000101011011000001000011011000110000110101101111101111001010000011001100011011000100100000100001001100110001010011100111000011010110011100000111001001101111001111011100100110010111001110010000111101000100111101100110000110011010001101111001110100001011000011001100001010000011010100011000110000100100111100111101111011011001111010011101110010000111000110100001101001000111011100010100001000100000111100010101111001000001011001100010010001100110000111101100010011101110011101000001011001000010011000111111101000101001"
                    "111111110101010001111100100100011011100000101101000111010110001000001001001111100011011101101110000100101101111101000001111100100010110011110100010001000111101100011010001110101111101111010100001111101100111011000011101001001111101101111011101011110011100111101001001000001111001110111001010000001011110100110100111100110001101001111100111011111010011001110111110110001101101011000111111011011110010100100000101000011111011001101001110001000010010011111000110111000101100100001000101111110111011111010001011000100101111000010001111110011101010011110110110100000111111101000101001"
                    "111111110101010001100010100011111011011000001011110110011101000011101110101111101000010011110110110000110001110100111001100011111010001011100100101111110100011110110011001100111010111000011011111010001000110011111010001001010110011111000010011111001101000101111100011001001110110101111100011001101011110000110001111010110001110111101000111010010111001111110100111000000101101111110101101000010111100100100000110001110010011101101011111100010010011110000010010100001110100001101000001110010110010001110001101110100111010001100001110110111111001010110000110011110111111101000101001"
                    "000000000101100011000010100010011001001101001100011011101001110001110100100111101011001000010010100110001100100111001010000100010111001000100111000101000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                    "000000011010011100111101011101100110110010110011100100010110001110001011011000010100110111101101011001110011011000110101111011101000110111011000111010110000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                },
        /* 65*/ { BARCODE_GS1_128_CC, -1, 3, "[01]12345678901231[90]123", "[20]12", 0, 5, 205, 1, "Example with min CC-C input and smallest linear input to bump rows (2 to 3), CC-C 8 cols, 3 rows",
                    "1111111101010100011101010111000000111010110011110001110111110111010010000010000100010111010011011110001001100111101110010000110111001100111000101101111001000101110111000011101010001110000111111101000101001"
                    "1111111101010100011110101000010000111010111011000001111110111001010011111001000100110111000000100110101100100111000001011111010001101110110101110001000001111010100001000011111010101100000111111101000101001"
                    "1111111101010100011101010001111110100010100011110001111101011100001011000011001011110100101001111000001011111011010000011101111110010100110111100010001101011001001110000010101000011110000111111101000101001"
                    "0000000001011000110000101000100110010011010011000110111010011100011101001001111010110010000100101001100011001001110010010000100101001100011010000100010011010001100010100001001000100010011100010100000000000"
                    "0000000110100111001111010111011001101100101100111001000101100011100010110110000101001101111011010110011100110110001101101111011010110011100101111011101100101110011101011110110111011101100011101011000000000"
                },
        /* 66*/ { BARCODE_GS1_128_CC, -1, 3, "[01]12345678901231[90]123456789012345678901234567890[91]1234567890123456789012345678901234567890123456789012345678901234", "[20]12", ZINT_WARN_NONCOMPLIANT, 5, 702, 1, "Example with overlarge linear input and min CC-C input, CC-C 30 cols, 3 rows (bumped up from 1)",
                    "111111110101010001110101011100000010100010000010000111011111011101001000001000010001011101001101111000100110011110111001000011011100110011100010110111100100010111011100001010111000111000010000101100000110100100000010010001100001100010100011011000111001110110010011101111001000101111011110010100001110001110100011000011010001101111011100010010100110001100000101100011110111001111011110101000010000010001101100100011011001000001110101101111000010001100011100110111011010110000001110111010000100011000010110001110110100001000011001000010010000001010101111011110000111111101000101001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                    "111111110101010001111010100001000011110011011000100111100000100010101110010001100001011101011101100000111111011100101001111100100010011011100000010011010110010011100000101111001010000001010110011111001000111111010001011101110001100011101011111000110010100110101111101110001110010011110111011100011011101000110111110010111101110111001111001011110101101100000110011101111100101101000011100100011011000011110010100010001111101101010011110000100010011000111110100111100110110001001111000001000101011100100011000010111010111011000001111110111001010011111010101100000111111101000101001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                    "111111110101010001010111000111111011011101011100000110001111101000101100010111111001011001001111110010100010001001111001101000001011111011100011110101110110111110011001101101001000111110010011010011100000101111101100000101000000110010111010000110110111100110101101111110001011111000111011010110100001110000101110000100110001101111000010011010100110001111100110101100111111001001010011110000011110101111001000110101000111110001000110001001110010100011001111100100000111001001101000110011011110010011010000111000110111111001101001110101111101000010101000011110000111111101000101001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                    "000000000101100011000010100010011001001101001100011011101001110001110100100111101011001000010010100110001100100111001001000010010100110001101110100111000111010010011110101100100001001010011000110111010011100011101001001111010110010000100101001100011011101001110001110100100111101011001000010010000101000100001001001010011000110111010011100011101001001111010110010000100101001100011011101001110001110100100111101011001000010010100110001101110100111000111010010011110101100100001001010011000110111010011100011101001001111010110010000100101001100011011101001110001110100100111101011001000010010100110001101110100111000111010010011110101100100001001010011000110111010011101000010001001100110010011100010100"
                    "000000011010011100111101011101100110110010110011100100010110001110001011011000010100110111101101011001110011011000110110111101101011001110010001011000111000101101100001010011011110110101100111001000101100011100010110110000101001101111011010110011100100010110001110001011011000010100110111101101111010111011110110110101100111001000101100011100010110110000101001101111011010110011100100010110001110001011011000010100110111101101011001110010001011000111000101101100001010011011110110101100111001000101100011100010110110000101001101111011010110011100100010110001110001011011000010100110111101101011001110010001011000111000101101100001010011011110110101100111001000101100010111101110110011001101100011101011"
                },
        /* 67*/ { BARCODE_GS1_128_CC, -1, 3, "[01]12345678901231[90]123456789012345678901234567890[91]1234567890123456789012345678901234567890123456789012345678901234", "[20]01[90]123456789012345678901234567890[91]123456789012", ZINT_WARN_NONCOMPLIANT, 5, 702, 1, "Example with overlarge linear input and medium CC-C input, CC-C 30 cols, 3 rows (bumped up from 1 (before last digit) than 2)",
                    "111111110101010001110101011100000010100010000010000111011111011101001000001000010001011101001101111000111000101000111001100100011101111010001011000001100101100000110000101101101100110000011010010000110000110000100011011101001110011101000011110111011011100111101100010011101111001001011110011101110001000010111000101001110001011000001100010011010010011000000110011100001101001100011011000011011101110001010000110100001000001101010011000110000010110001111011100111101111010100001000001000110110010001101100100000111010110111100001000110001110011010101111011110000111111101000101001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                    "111111110101010001111010100001000011010000111001000110110000111100101000100011111011010100111100001000100110001111101001111001101100010011110000010001010111001000110000101110101110110000011111101110010100111110010001001101110000001001101011001001110000010111100101000000101011001111100100011111101000101110111000110001110101111100011001010011010111110111000111001001111011101110001101110100011011111001011110111011100111100101111010110110000011001110111110010110100001110010001101100001111001010001000111110110101001111000010001001100011111010011111010101100000111111101000101001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                    "111111110101010001010111000111111010001100110111100101111110111000101001100101110000010101110011111100101111000010000101101110101110000011000111110100010110001011111100101100100111111001010001000100111100110100000101111101110001111010111011011111001100110110100100011111001001101001110000010111110110000010100000011001011101000011011011110010101000011110000101001100011111001101011001111110010010100111100000111111010011010001000110101110000011101111100100110110100011011111101100111110100001011001111100001010100001010011110001101011111100010010101000011110000111111101000101001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                    "000000000101100011000010100010011001001101001100011011101001110001110100100111101011001000010010100110001100100111001001000010010100110001101110100111000111010010011110101100100001001010011000110111010011100011101001001111010110010000100101001100011011101001110001110100100111101011001000010010000101000100001001001010011000110111010011100011101001001111010110010000100101001100011011101001110001110100100111101011001000010010100110001101110100111000111010010011110101100100001001010011000110111010011100011101001001111010110010000100101001100011011101001110001110100100111101011001000010010100110001101110100111000111010010011110101100100001001010011000110111010011101000010001001100110010011100010100"
                    "000000011010011100111101011101100110110010110011100100010110001110001011011000010100110111101101011001110011011000110110111101101011001110010001011000111000101101100001010011011110110101100111001000101100011100010110110000101001101111011010110011100100010110001110001011011000010100110111101101111010111011110110110101100111001000101100011100010110110000101001101111011010110011100100010110001110001011011000010100110111101101011001110010001011000111000101101100001010011011110110101100111001000101100011100010110110000101001101111011010110011100100010110001110001011011000010100110111101101011001110010001011000111000101101100001010011011110110101100111001000101100010111101110110011001101100011101011"
                },
        /* 68*/ { BARCODE_GS1_128_CC, -1, 3, "[01]12345678901231[90]123456789012345678901234567890[91]1234567890123456789012345678901234567890123456789012345678901234", "[20]01[90]123456789012345678901234567890[91]1234567890123", ZINT_WARN_NONCOMPLIANT, 5, 702, 1, "Example with overlarge linear input and medium CC-C input, CC-C 30 cols, 3 rows (bumped up from 2)",
                    "111111110101010001110101011100000010100010000010000111011111011101001000001000010001011101001101111000111000101000111001100100011101111010001011000001100101100000110000101101101100110000011010010000110000110000100011011101001110011101000011110111011011100111101100010011101111001001011110011101110001000010111000101001110001011000001100010011010010011000000110011100001101001100011011000011010111001111001100110100000100110001010011000110000010110001111011100111101111010100001000001000110110010001101100100000111010110111100001000110001110011010101111011110000111111101000101001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                    "111111110101010001111010100001000011010000111001000110110000111100101000100011111011010100111100001000100110001111101001111001101100010011110000010001010111001000110000101110101110110000011111101110010100111110010001001101110000001001101011001001110000010111100101000000101011001111100100011111101000101110111000110001110101111100011001010011010111110111000111001001111011101110001101110100011011111001011110111011100111100101111010110110000011001110111110010110100001110010001101100001111001010001000111110110101001111000010001001100011111010011111010101100000111111101000101001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                    "111111110101010001010111000111111010001100110111100101111110111000101001100101110000010101110011111100101111000010000101101110101110000011000111110100010110001011111100101100100111111001010001000100111100110100000101111101110001111010111011011111001100110110100100011111001001101001110000010111110110000010100000011001011101000011011011110010101000011110000101001100011111001101011001111110010010100111100000111011110001011101101111000001011010111100000100010110100100011111001011101000011000010001101011100000101111110001110101011110110011000010101000011110000111111101000101001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                    "000000000101100011000010100010011001001101001100011011101001110001110100100111101011001000010010100110001100100111001001000010010100110001101110100111000111010010011110101100100001001010011000110111010011100011101001001111010110010000100101001100011011101001110001110100100111101011001000010010000101000100001001001010011000110111010011100011101001001111010110010000100101001100011011101001110001110100100111101011001000010010100110001101110100111000111010010011110101100100001001010011000110111010011100011101001001111010110010000100101001100011011101001110001110100100111101011001000010010100110001101110100111000111010010011110101100100001001010011000110111010011101000010001001100110010011100010100"
                    "000000011010011100111101011101100110110010110011100100010110001110001011011000010100110111101101011001110011011000110110111101101011001110010001011000111000101101100001010011011110110101100111001000101100011100010110110000101001101111011010110011100100010110001110001011011000010100110111101101111010111011110110110101100111001000101100011100010110110000101001101111011010110011100100010110001110001011011000010100110111101101011001110010001011000111000101101100001010011011110110101100111001000101100011100010110110000101001101111011010110011100100010110001110001011011000010100110111101101011001110010001011000111000101101100001010011011110110101100111001000101100010111101110110011001101100011101011"
                },
        /* 69*/ { BARCODE_GS1_128_CC, -1, 3, "[90]1234567890123456789012345678", "[20]01", 0, 5, 239, 1, "Example with linear input length triggering difference between new (BWIPP) 52 and old 53 adjustment",
                    "11111111010101000111010101110000001101011101111000011101111101110100100000100001000101110100110111100010110000011001000110001000110111001010000100001000010010111100111100101011100011100001000010110000011010101000001000000111111101000101001"
                    "11111111010101000111101010000100001111100100010011011100000010011010110010011100000101111001010000001010110011111001000111111010001011101110001100011101011111000110010100110101110001000001111010100001000011111010101100000111111101000101001"
                    "11111111010101000110101000011111001010011000111110011010110011111100110111110010001001100001111010110010111111011110110110110001011110001110010111110010011110101111100110101110011000011101100110100001111010101000011110000111111101000101001"
                    "00000000010110001100001010001001000010010100110001101110100111000111010010011110101100100001001010011000110111010011100011101001001111010110010000100101001100011011101001110001110100100111101011010000100010010000100100111000101000000000000"
                    "00000001101001110011110101110110111101101011001110010001011000111000101101100001010011011110110101100111001000101100011100010110110000101001101111011010110011100100010110001110001011011000010100101111011101101111011011000111010110000000000"
                },
        /* 70*/ { BARCODE_GS1_128_CC, -1, 3, "[01]12345678901231[90]123", "[20]12", 0, 5, 205, 1, "Example with 8 cols, left/right padding of linear (BWIPP)",
                    "1111111101010100011101010111000000111010110011110001110111110111010010000010000100010111010011011110001001100111101110010000110111001100111000101101111001000101110111000011101010001110000111111101000101001"
                    "1111111101010100011110101000010000111010111011000001111110111001010011111001000100110111000000100110101100100111000001011111010001101110110101110001000001111010100001000011111010101100000111111101000101001"
                    "1111111101010100011101010001111110100010100011110001111101011100001011000011001011110100101001111000001011111011010000011101111110010100110111100010001101011001001110000010101000011110000111111101000101001"
                    "0000000001011000110000101000100110010011010011000110111010011100011101001001111010110010000100101001100011001001110010010000100101001100011010000100010011010001100010100001001000100010011100010100000000000"
                    "0000000110100111001111010111011001101100101100111001000101100011100010110110000101001101111011010110011100110110001101101111011010110011100101111011101100101110011101011110110111011101100011101011000000000"
                },
        /* 71*/ { BARCODE_EANX_CC, -1, 1, "123456789012+12", "[91]123456789012345678901", 0, 8, 125, 1, "Example of EAN-13 with 2-digit addon, CC-A 4 cols, 4 rows",
                    "11010010001111001101001110010111011100010000100100011010111011100100000110000010011011101110110010100000000000000000000000000"
                    "11010110001110101100100000011001111010111110100100001011001011110110000110010011100000101100110010100000000000000000000000000"
                    "11010111001110101111101000011111100110111010101100001011111010111100110100110100111000001101110010100000000000000000000000000"
                    "11010111101000110011000010010000101110001110101110001010001110011100010110100010001100001101111010100000000000000000000000000"
                    "00010000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001000000000000000000000000000"
                    "00100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000100000000000000000000000000"
                    "00010000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001000000000000000000000000000"
                    "00010100100110111101001110101100010000101001000101010100100011101001110010110011011011001001000101000000010110011001010010011"
                },
        /* 72*/ { BARCODE_EANX_CC, -1, 1, "123456789012+54321", "[91]1234567890", 0, 7, 152, 1, "Example of EAN-13 with 5-digit addon, CC-B 4 cols, 3 rows",
                    "11011011101111001101001110010111011100010000100111010010111011100100000110000010011011101101100010100000000000000000000000000000000000000000000000000000"
                    "11011011001111110101001110011111000100100110100110010011010001110001000100101111110011101100100010100000000000000000000000000000000000000000000000000000"
                    "11011010001010000010111100011100111100101110100110011011001111101001000100111110110100001110100010100000000000000000000000000000000000000000000000000000"
                    "00010000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001000000000000000000000000000000000000000000000000000000"
                    "00100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000100000000000000000000000000000000000000000000000000000"
                    "00010000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001000000000000000000000000000000000000000000000000000000"
                    "00010100100110111101001110101100010000101001000101010100100011101001110010110011011011001001000101000000010110111001010100011010100001010010011010011001"
                },
        /* 73*/ { BARCODE_UPCA_CC, -1, 1, "12345678901+12", "[91]123456789", 0, 7, 127, 1, "Example of UPC-A with 2-digit addon, CC-A 4 cols, 3 rows",
                    "1101101110111100110100111001011101110001000010011101001011101110010000011000001001101110110110001010000000000000000000000000000"
                    "1101101100111001110011110101100001000111001010011001001110101100100000010010111111001110110010001010000000000000000000000000000"
                    "1101101000100001111001000101011110100000010010011001101000111000010110011011100010001110111010001010000000000000000000000000000"
                    "0001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000100000000000000000000000000000"
                    "0010000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010000000000000000000000000000"
                    "0001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000100000000000000000000000000000"
                    "0001010011001001001101111010100011011000101011110101010001001001000111010011100101100110110110010100000000010110011001010010011"
                },
        /* 74*/ { BARCODE_UPCA_CC, -1, 2, "12345678901+12121", "[91]1234567890123", 0, 8, 154, 1, "Example of UPC-A with 5-digit addon, CC-B 4 cols, 4 rows",
                    "1101001110100011111010011101101111110101110010011101101101000011110110011001011100001000110100100010000000000000000000000000000000000000000000000000000000"
                    "1101001100101100100111000001011111011000001010011101001101111110011010011101001111100100110101100010000000000000000000000000000000000000000000000000000000"
                    "1101000100111001101000011001100100000101100010011001001001111011110100011100011101000100110101110010000000000000000000000000000000000000000000000000000000"
                    "1101000110110001011100000101110110111001000010011001101111100010101100011110001001010000110101111010000000000000000000000000000000000000000000000000000000"
                    "0001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000100000000000000000000000000000000000000000000000000000000"
                    "0010000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010000000000000000000000000000000000000000000000000000000"
                    "0001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000100000000000000000000000000000000000000000000000000000000"
                    "0001010011001001001101111010100011011000101011110101010001001001000111010011100101100110110110010100000000010110011001010010011010110011010011011010011001"
                },
        /* 75*/ { BARCODE_UPCE_CC, -1, 1, "0654321+89", "[91]1", 0, 9, 81, 1, "Example of UPC-E with 2-digit addon, CC-A 2 cols, 5 rows",
                    "110110011011110111010111110101000100011110001111010100100000000000000000000000000"
                    "110110111011101100001000110111001010111000001110010100100000000000000000000000000"
                    "110110110011000101111101110111101000010010001110110100100000000000000000000000000"
                    "110110100010011101110011110111001111010111001110100100100000000000000000000000000"
                    "110100100011001100100000010111010001101111001110100110100000000000000000000000000"
                    "000100000000000000000000000000000000000000000000000001000000000000000000000000000"
                    "001000000000000000000000000000000000000000000000000000100000000000000000000000000"
                    "000100000000000000000000000000000000000000000000000001000000000000000000000000000"
                    "000101000010101100010011101011110100110110011001010101000000010110110111010010111"
                },
        /* 76*/ { BARCODE_UPCE_CC, -1, 2, "1876543+56789", "[91]12345", 0, 12, 108, 1, "Example of UPC-E with 5-digit addon, CC-B 2 cols, 8 rows",
                    "110010001011101111101110100100000100001000101100100010100000000000000000000000000000000000000000000000000000"
                    "111010001011010000111101100110010111000010001110100010100000000000000000000000000000000000000000000000000000"
                    "111011001011101100100111110100011110010000101110110010100000000000000000000000000000000000000000000000000000"
                    "110011001011001110110001000101011111011111001100110010100000000000000000000000000000000000000000000000000000"
                    "110111001011111001001101110100111111010011001101110010100000000000000000000000000000000000000000000000000000"
                    "110111101011001110000101110100110100001110001101111010100000000000000000000000000000000000000000000000000000"
                    "110011101010100010000010000101111011111011001100111010100000000000000000000000000000000000000000000000000000"
                    "111011101010111100101111000101111000100111101110111010100000000000000000000000000000000000000000000000000000"
                    "000100000000000000000000000000000000000000000000000001000000000000000000000000000000000000000000000000000000"
                    "001000000000000000000000000000000000000000000000000000100000000000000000000000000000000000000000000000000000"
                    "000100000000000000000000000000000000000000000000000001000000000000000000000000000000000000000000000000000000"
                    "000101011011100100010000101011000101000110100001010101000000010110110001010101111010010001010110111010010111"
                },
        /* 77*/ { BARCODE_EANX_CC, -1, 1, "9876543+65", "[91]1234567", 0, 8, 98, 1, "Example of EAN-8 with 2-digit addon, CC-A 3 cols, 4 rows",
                    "10010001111100110101001100011100010111001100110001011101000001110100110100000000000000000000000000"
                    "11011111100110100101011100011011110010110000111100010011110101110101110100000000000000000000000000"
                    "10000101100000110101011000010101111101111100111010010000111001110101100100000000000000000000000000"
                    "11110010001100110101001000011001001111101110101001111100011001110101000100000000000000000000000000"
                    "00001000000000000000000000000000000000000000000000000000000000000000001000000000000000000000000000"
                    "00010000000000000000000000000000000000000000000000000000000000000000000100000000000000000000000000"
                    "00001000000000000000000000000000000000000000000000000000000000000000001000000000000000000000000000"
                    "00001010001011011011101110110101111010101001110101110010000101110010101000000010110101111010111001"
                },
        /* 78*/ { BARCODE_EANX_CC, -1, 2, "9876543+74083", "[91]123456789012345678", 0, 12, 135, 1, "Example of EAN-8 with 5-digit addon, CC-B 3 cols, 8 rows",
                    "110011101011101111101110100100001011010000010000100010111110110100111101100111010100000000000000000000000000000000000000000000000000000"
                    "111011101011001011100001000100001001011100101100100000111000110111010001110111010100000000000000000000000000000000000000000000000000000"
                    "111001101011011111100110100100001101010100010111100000110010011011111101110011010100000000000000000000000000000000000000000000000000000"
                    "111101101011101101000011000100011101011000011011001100111001101001100001111011010100000000000000000000000000000000000000000000000000000"
                    "111100101011101101110010000100011001011101110111100100101001111000010001111001010100000000000000000000000000000000000000000000000000000"
                    "111000101011111101011000100100010001011100101111110110100111100001101101110001010100000000000000000000000000000000000000000000000000000"
                    "110000101011110010001011110100110001011100111100110010100111000111001001100001010100000000000000000000000000000000000000000000000000000"
                    "110001101011001100111101000100111001011111011110111010101110011111100101100011010100000000000000000000000000000000000000000000000000000"
                    "000000000000001000000000000000000000000000000000000000000000000000000000000000001000000000000000000000000000000000000000000000000000000"
                    "000000000000010000000000000000000000000000000000000000000000000000000000000000000100000000000000000000000000000000000000000000000000000"
                    "000000000000001000000000000000000000000000000000000000000000000000000000000000001000000000000000000000000000000000000000000000000000000"
                    "000000000000001010001011011011101110110101111010101001110101110010000101110010101000000010110111011010011101010001101010110111010100001"
                },
        /* 79*/ { BARCODE_EANX_CC, -1, 1, "1234567890128+65", "[91]1234567", 0, 7, 125, 1, "Example of EAN-13 + CHK with 2-digit addon, CC-A 3 cols, 4 rows",
                    "11011011101111001101001110010111011100010000100111010010111011100100000110000110010100001101100010100000000000000000000000000"
                    "11011011001001100101111110011111011001000010100110010011111101110010100110101110001000001100100010100000000000000000000000000"
                    "11011010001101011111100100011011101000011100100110011010001001001111000101000011011111001110100010100000000000000000000000000"
                    "00010000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001000000000000000000000000000"
                    "00100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000100000000000000000000000000"
                    "00010000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001000000000000000000000000000"
                    "00010100100110111101001110101100010000101001000101010100100011101001110010110011011011001001000101000000010110101111010111001"
                },
    };
    const int data_size = ARRAY_SIZE(data);
    int i, length, composite_length, ret;
    struct zint_symbol *symbol = NULL;

    char escaped[1024];
    char esc_composite[4096];

    char bwipp_buf[32768];
    char bwipp_msg[1024];

    int do_bwipp = (debug & ZINT_DEBUG_TEST_BWIPP) && testUtilHaveGhostscript(); /* Only do BWIPP test if asked, too slow otherwise */

    testStartSymbol("test_examples", &symbol);

    for (i = 0; i < data_size; i++) {

        if (testContinue(p_ctx, i)) continue;

        symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        length = testUtilSetSymbol(symbol, data[i].symbology, data[i].input_mode, -1 /*eci*/, data[i].option_1, -1, -1, -1 /*output_options*/, data[i].data, -1, debug);
        assert_zero(length >= 128, "i:%d length %d >= 128\n", i, length);
        strcpy(symbol->primary, data[i].data);

        composite_length = (int) strlen(data[i].composite);

        ret = ZBarcode_Encode(symbol, (const unsigned char *) data[i].composite, composite_length);
        assert_equal(ret, data[i].ret, "i:%d ZBarcode_Encode ret %d != %d (%s)\n", i, ret, data[i].ret, symbol->errtxt);

        if (p_ctx->generate) {
            printf("        /*%3d*/ { %s, %s, %d, \"%s\", \"%s\", %s, %d, %d, %d, \"%s\",\n",
                    i, testUtilBarcodeName(symbol->symbology), testUtilInputModeName(data[i].input_mode), data[i].option_1,
                    testUtilEscape(data[i].data, length, escaped, sizeof(escaped)), testUtilEscape(data[i].composite, composite_length, esc_composite, sizeof(esc_composite)),
                    testUtilErrorName(ret), symbol->rows, symbol->width, data[i].bwipp_cmp, data[i].comment);
            testUtilModulesPrint(symbol, "                    ", "\n");
            printf("                },\n");
        } else {
            int width, row;

            assert_equal(symbol->rows, data[i].expected_rows, "i:%d %s symbol->rows %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->rows, data[i].expected_rows, data[i].data);
            assert_equal(symbol->width, data[i].expected_width, "i:%d %s symbol->width %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->width, data[i].expected_width, data[i].data);

            ret = testUtilModulesCmp(symbol, data[i].expected, &width, &row);
            assert_zero(ret, "i:%d %s testUtilModulesCmp ret %d != 0 width %d row %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), ret, width, row, data[i].data);

            if (do_bwipp && testUtilCanBwipp(i, symbol, data[i].option_1, -1, -1, debug)) {
                if (!data[i].bwipp_cmp) {
                    if (debug & ZINT_DEBUG_TEST_PRINT) printf("i:%d %s not BWIPP compatible (%s)\n", i, testUtilBarcodeName(symbol->symbology), data[i].comment);
                } else {
                    ret = testUtilBwipp(i, symbol, data[i].option_1, -1, -1, data[i].composite, composite_length, symbol->primary, bwipp_buf, sizeof(bwipp_buf), NULL);
                    assert_zero(ret, "i:%d %s testUtilBwipp ret %d != 0\n", i, testUtilBarcodeName(symbol->symbology), ret);

                    ret = testUtilBwippCmp(symbol, bwipp_msg, bwipp_buf, data[i].expected);
                    assert_zero(ret, "i:%d %s testUtilBwippCmp %d != 0 %s\n  actual: %s\nexpected: %s\n",
                                   i, testUtilBarcodeName(symbol->symbology), ret, bwipp_msg, bwipp_buf, data[i].expected);
                }
            }
        }

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

static void test_odd_numbered_numeric(const testCtx *const p_ctx) {
    int debug = p_ctx->debug;

    struct item {
        int symbology;
        int option_1;
        char *data;
        char *composite;
        int ret;

        int expected_rows;
        int expected_width;
        char *comment;
        char *expected;
    };
    /* Verified via bwipp_dump.ps against BWIPP, and manually against tec-it.com */
    static const struct item data[] = {
        /* 0*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]1234567890123", 0, 9, 55, "Test odd-numbered numeric, 1st fit, 9-bit remainder, 7-bit final, 2-bit alphanumeric latch, no padding",
                    "1101100110111011101011110001001111100110010011110101001"
                    "1101101110110001100001000101100010000100110011100101001"
                    "1101101100111110001001001101100100111110111011101101001"
                    "1101101000110101100011111101100111110100100011101001001"
                    "1101001000110111000110001001010001100001100011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 1*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]12345678901234", 0, 9, 55, "Test even-numbered numeric, 1st fit, 2-bit remainder, 2-bit alphanumeric latch, no padding",
                    "1101100110111011101011110001001111100110010011110101001"
                    "1101101110110001100001000101100010000100110011100101001"
                    "1101101100111110001001001101100100111001000011101101001"
                    "1101101000101000010011110001101101001111000011101001001"
                    "1101001000100011101110010001011100011101000011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 2*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]123456789012345", 0, 10, 55, "Test odd-numbered numeric, 2nd fit, alphanumeric latch, padding",
                    "1100100010111100110100111001011101110001000011100101101"
                    "1110100010110001011101000001000111010111110011000101101"
                    "1110110010110101100111111001000111100001001011000101001"
                    "1100110010100100000010010001010000010000100011001101001"
                    "1101110010111111010101110001110101110000110011011101001"
                    "1101111010110000111101001101100011110110111011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 3*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]ABCD12345678901", 0, 10, 55, "Test odd-numbered numeric, 1st fit, 4-bit final, no alphanumeric latch or padding",
                    "1100100010111100110100111001110101110111110011100101101"
                    "1110100010101111100110111101111000001000101011000101101"
                    "1110110010110111100001001101101000111111001011000101001"
                    "1100110010101000110011000001111101101011110011001101001"
                    "1101110010110101111000011001111000001001010011011101001"
                    "1101111010100010000010111101011100000100011011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 4*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]ABCDE123456789", 0, 10, 55, "Test odd-numbered numeric, 1st fit, 5-bit final, no alphanumeric latch or padding",
                    "1100100010111100110100111001110101110111110011100101101"
                    "1110100010101111100110111101100000100011101011000101101"
                    "1110110010100110000010111001111101011110011011000101001"
                    "1100110010100011000110000101111101110101110011001101001"
                    "1101110010110100011111011101000111100101111011011101001"
                    "1101111010110111101000110001100011001011110011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 5*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]1234567890123456789", 0, 10, 55, "Test odd-numbered numeric, 1st fit, 7-bit final, no alphanumeric latch or padding",
                    "1100100010111100110100111001011101110001000011100101101"
                    "1110100010110001011101000001000111010111110011000101101"
                    "1110110010110101100111111001000111100001001011000101001"
                    "1100110010111100100010111101100100010110000011001101001"
                    "1101110010111110010110011101110000100110100011011101001"
                    "1101111010111111011001110101011100001001100011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 6*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]1234A", 0, 9, 55, "Test even-numbered numeric, ending in non-digit",
                    "1101100110111011101011110001001111100110010011110101001"
                    "1101101110100001000100001001110110011001110011100101001"
                    "1101101100100101111100110001000001011111011011101101001"
                    "1101101000110001001001111101011111011001000011101001001"
                    "1101001000100110001100000101110110000010110011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 7*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]12345A", 0, 9, 55, "Test odd-numbered numeric, ending in non-digit",
                    "1101100110111011101011110001001111100110010011110101001"
                    "1101101110100111011100100001100010110001110011100101001"
                    "1101101100100100011111001101111101001101110011101101001"
                    "1101101000101111100110010001011111000110001011101001001"
                    "1101001000100111000111010001011000011000100011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
    };
    const int data_size = ARRAY_SIZE(data);
    int i, length, composite_length, ret;
    struct zint_symbol *symbol = NULL;

    char bwipp_buf[8192];
    char bwipp_msg[1024];

    int do_bwipp = (debug & ZINT_DEBUG_TEST_BWIPP) && testUtilHaveGhostscript(); /* Only do BWIPP test if asked, too slow otherwise */

    testStartSymbol("test_odd_numbered_numeric", &symbol);

    for (i = 0; i < data_size; i++) {

        if (testContinue(p_ctx, i)) continue;

        symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        length = testUtilSetSymbol(symbol, data[i].symbology, -1 /*input_mode*/, -1 /*eci*/, data[i].option_1, -1, -1, -1 /*output_options*/, data[i].data, -1, debug);
        assert_zero(length >= 128, "i:%d length %d >= 128\n", i, length);
        strcpy(symbol->primary, data[i].data);

        composite_length = (int) strlen(data[i].composite);

        ret = ZBarcode_Encode(symbol, (const unsigned char *) data[i].composite, composite_length);
        assert_equal(ret, data[i].ret, "i:%d ZBarcode_Encode ret %d != %d (%s)\n", i, ret, data[i].ret, symbol->errtxt);

        if (p_ctx->generate) {
            test_helper_generate(symbol, ret, i, data[i].data, data[i].composite, data[i].option_1, data[i].comment, -1 /*bwipp_cmp*/);
        } else {
            int width, row;

            assert_equal(symbol->rows, data[i].expected_rows, "i:%d %s symbol->rows %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->rows, data[i].expected_rows, data[i].data);
            assert_equal(symbol->width, data[i].expected_width, "i:%d %s symbol->width %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->width, data[i].expected_width, data[i].data);

            ret = testUtilModulesCmp(symbol, data[i].expected, &width, &row);
            assert_zero(ret, "i:%d %s testUtilModulesCmp ret %d != 0 width %d row %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), ret, width, row, data[i].data);

            if (do_bwipp && testUtilCanBwipp(i, symbol, data[i].option_1, -1, -1, debug)) {
                ret = testUtilBwipp(i, symbol, data[i].option_1, -1, -1, data[i].composite, composite_length, symbol->primary, bwipp_buf, sizeof(bwipp_buf), NULL);
                assert_zero(ret, "i:%d %s testUtilBwipp ret %d != 0\n", i, testUtilBarcodeName(symbol->symbology), ret);

                ret = testUtilBwippCmp(symbol, bwipp_msg, bwipp_buf, data[i].expected);
                assert_zero(ret, "i:%d %s testUtilBwippCmp %d != 0 %s\n  actual: %s\nexpected: %s\n",
                               i, testUtilBarcodeName(symbol->symbology), ret, bwipp_msg, bwipp_buf, data[i].expected);
            }
        }

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

static void test_ean128_cc_shift(const testCtx *const p_ctx) {
    int debug = p_ctx->debug;

    struct item {
        int symbology;
        int option_1;
        char *data;
        char *composite;
        int ret;

        int expected_rows;
        int expected_width;
        int bwipp_cmp;
        char *comment;
        char *expected;
    };
    /* Verified via bwipp_dump.ps against BWIPP except where noted, when shift verified manually (tec-it.com seems to be off by 2 for top shifts > 1) */
    static const struct item data[] = {
        /* 0*/ { BARCODE_GS1_128_CC, -1, "[91]1", "[21]A1B2C3D4E5F6G7H8", 0, 6, 100, 1, "CC-A alignment, bottom shift 10",
                    "1101001000110100001000001101101011110111110010010001101010000010010000011101110100010000111011001010"
                    "1101011000110101111001100001111010001101100010010000101111000011001101011100101100001000110011001010"
                    "1101011100100011001100111101011000101110000010110000101001100110011110011011110011001110110111001010"
                    "1101011110111000111011011001110010001011100010111000101011000011100110010000100000100010110111101010"
                    "0000000000001011000110000101000100001001001010000100010110001100101000100001011000100110011100010100"
                    "0000000000110100111001111010111011110110110101111011101001110011010111011110100111011001100011101011"
               },
        /* 1*/ { BARCODE_GS1_128_CC, -1, "[91]12", "[21]A1B2C3D4E5F6G7H8", 0, 6, 99, 1, "CC-A alignment, bottom shift 12",
                    "110100100011010000100000110110101111011111001001000110101000001001000001110111010001000011101100101"
                    "110101100011010111100110000111101000110110001001000010111100001100110101110010110000100011001100101"
                    "110101110010001100110011110101100010111000001011000010100110011001111001101111001100111011011100101"
                    "110101111011100011101101100111001000101110001011100010101100001110011001000010000010001011011110101"
                    "000000000000001011000110000101000100001001001010011000110001010000101110110011001110001010000000000"
                    "000000000000110100111001111010111011110110110101100111001110101111010001001100110001110101100000000"
               },
        /* 2*/ { BARCODE_GS1_128_CC, -1, "[91]123", "[21]A1B2C3D4E5F6G7H8", 0, 6, 101, 1, "CC-A alignment, top shift 1",
                    "01101001000110100001000001101101011110111110010010001101010000010010000011101110100010000111011001010"
                    "01101011000110101111001100001111010001101100010010000101111000011001101011100101100001000110011001010"
                    "01101011100100011001100111101011000101110000010110000101001100110011110011011110011001110110111001010"
                    "01101011110111000111011011001110010001011100010111000101011000011100110010000100000100010110111101010"
                    "00101100011000010100010000100100101001100011010000100010011010001101000100001011001111010011100010100"
                    "11010011100111101011101111011011010110011100101111011101100101110010111011110100110000101100011101011"
               },
        /* 3*/ { BARCODE_GS1_128_CC, -1, "[91]12345", "[21]A1B2C3D4E5F6G7H8", 0, 6, 112, 1, "CC-A alignment, top shift 12",
                    "0000000000001101001000110100001000001101101011110111110010010001101010000010010000011101110100010000111011001010"
                    "0000000000001101011000110101111001100001111010001101100010010000101111000011001101011100101100001000110011001010"
                    "0000000000001101011100100011001100111101011000101110000010110000101001100110011110011011110011001110110111001010"
                    "0000000000001101011110111000111011011001110010001011100010111000101011000011100110010000100000100010110111101010"
                    "0010110001100001010001000010010010100110001101110100111010000100010010001101101000100001000110110010011100010100"
                    "1101001110011110101110111101101101011001110010001011000101111011101101110010010111011110111001001101100011101011"
               },
        /* 4*/ { BARCODE_GS1_128_CC, -1, "[91]1234567", "[21]A1B2C3D4E5F6G7H8", 0, 6, 123, 1, "CC-A alignment, top shift 10",
                    "000000000011010010001101000010000011011010111101111100100100011010100000100100000111011101000100001110110010100000000000000"
                    "000000000011010110001101011110011000011110100011011000100100001011110000110011010111001011000010001100110010100000000000000"
                    "000000000011010111001000110011001111010110001011100000101100001010011001100111100110111100110011101101110010100000000000000"
                    "000000000011010111101110001110110110011100100010111000101110001010110000111001100100001000001000101101111010100000000000000"
                    "001011000110000101000100001001001010011000110111010011100011101001010000100010001001000101000100001000110100110011100010100"
                    "110100111001111010111011110110110101100111001000101100011100010110101111011101110110111010111011110111001011001100011101011"
               },
        /* 5*/ { BARCODE_GS1_128_CC, -1, "[91]123456789", "[21]A1B2C3D4E5F6G7H8", 0, 6, 134, 1, "CC-A alignment, top shift 21",
                    "00000000000000000000011010010001101000010000011011010111101111100100100011010100000100100000111011101000100001110110010100000000000000"
                    "00000000000000000000011010110001101011110011000011110100011011000100100001011110000110011010111001011000010001100110010100000000000000"
                    "00000000000000000000011010111001000110011001111010110001011100000101100001010011001100111100110111100110011101101110010100000000000000"
                    "00000000000000000000011010111101110001110110110011100100010111000101110001010110000111001100100001000001000101101111010100000000000000"
                    "00101100011000010100010000100100101001100011011101001110001110100100111101011010000100010001101001101000100001001101110110011100010100"
                    "11010011100111101011101111011011010110011100100010110001110001011011000010100101111011101110010110010111011110110010001001100011101011"
               },
        /* 6*/ { BARCODE_GS1_128_CC, -1, "[91]12345678901", "[21]A1B2C3D4E5F6G7H8", 0, 6, 145, 1, "CC-A alignment, top shift 21",
                    "0000000000000000000001101001000110100001000001101101011110111110010010001101010000010010000011101110100010000111011001010000000000000000000000000"
                    "0000000000000000000001101011000110101111001100001111010001101100010010000101111000011001101011100101100001000110011001010000000000000000000000000"
                    "0000000000000000000001101011100100011001100111101011000101110000010110000101001100110011110011011110011001110110111001010000000000000000000000000"
                    "0000000000000000000001101011110111000111011011001110010001011100010111000101011000011100110010000100000100010110111101010000000000000000000000000"
                    "0010110001100001010001000010010010100110001101110100111000111010010011110101100100001001010000100010110001100101000100001011001011110011100010100"
                    "1101001110011110101110111101101101011001110010001011000111000101101100001010011011110110101111011101001110011010111011110100110100001100011101011"
               },
    };
    const int data_size = ARRAY_SIZE(data);
    int i, length, composite_length, ret;
    struct zint_symbol *symbol = NULL;

    char bwipp_buf[8192];
    char bwipp_msg[1024];

    int do_bwipp = (debug & ZINT_DEBUG_TEST_BWIPP) && testUtilHaveGhostscript(); /* Only do BWIPP test if asked, too slow otherwise */

    testStartSymbol("test_ean128_cc_shift", &symbol);

    for (i = 0; i < data_size; i++) {

        if (testContinue(p_ctx, i)) continue;

        symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        length = testUtilSetSymbol(symbol, data[i].symbology, -1 /*input_mode*/, -1 /*eci*/, data[i].option_1, -1, -1, -1 /*output_options*/, data[i].data, -1, debug);
        assert_zero(length >= 128, "i:%d length %d >= 128\n", i, length);
        strcpy(symbol->primary, data[i].data);

        composite_length = (int) strlen(data[i].composite);

        ret = ZBarcode_Encode(symbol, (const unsigned char *) data[i].composite, composite_length);
        assert_equal(ret, data[i].ret, "i:%d ZBarcode_Encode ret %d != %d (%s)\n", i, ret, data[i].ret, symbol->errtxt);

        if (p_ctx->generate) {
            test_helper_generate(symbol, ret, i, data[i].data, data[i].composite, data[i].option_1, data[i].comment, data[i].bwipp_cmp);
        } else {
            int width, row;

            assert_equal(symbol->rows, data[i].expected_rows, "i:%d %s symbol->rows %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->rows, data[i].expected_rows, data[i].data);
            assert_equal(symbol->width, data[i].expected_width, "i:%d %s symbol->width %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->width, data[i].expected_width, data[i].data);

            ret = testUtilModulesCmp(symbol, data[i].expected, &width, &row);
            assert_zero(ret, "i:%d %s testUtilModulesCmp ret %d != 0 width %d row %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), ret, width, row, data[i].data);

            if (do_bwipp && testUtilCanBwipp(i, symbol, data[i].option_1, -1, -1, debug)) {
                if (!data[i].bwipp_cmp) {
                    if (debug & ZINT_DEBUG_TEST_PRINT) printf("i:%d %s not BWIPP compatible (%s)\n", i, testUtilBarcodeName(symbol->symbology), data[i].comment);
                } else {
                    ret = testUtilBwipp(i, symbol, data[i].option_1, -1, -1, data[i].composite, composite_length, symbol->primary, bwipp_buf, sizeof(bwipp_buf), NULL);
                    assert_zero(ret, "i:%d %s testUtilBwipp ret %d != 0\n", i, testUtilBarcodeName(symbol->symbology), ret);

                    ret = testUtilBwippCmp(symbol, bwipp_msg, bwipp_buf, data[i].expected);
                    assert_zero(ret, "i:%d %s testUtilBwippCmp %d != 0 %s\n  actual: %s\nexpected: %s\n",
                                   i, testUtilBarcodeName(symbol->symbology), ret, bwipp_msg, bwipp_buf, data[i].expected);
                }
            }
        }

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

static void test_ean128_cc_width(const testCtx *const p_ctx) {
    int debug = p_ctx->debug;

    struct item {
        char *data;
        char *composite;
        int ret;

        int expected_rows;
        int expected_width;
        char *comment;
    };
    /* Verified manually with BWIPP (except very large tests) */
    static const struct item data[] = {
        /* 0*/ { "[91]1", "[02]13012345678909", 0, 11, 103, "" },
        /* 1*/ { "[91]12", "[02]13012345678909", 0, 20, 86, "" },
        /* 2*/ { "[91]123", "[02]13012345678909", 0, 11, 108, "" },
        /* 3*/ { "[91]123A", "[02]13012345678909", 0, 8, 120, "" },
        /* 4*/ { "[91]123A1", "[02]13012345678909", 0, 7, 137, "" },
        /* 5*/ { "[91]123A12", "[02]13012345678909", 0, 7, 141, "" },
        /* 6*/ { "[91]123A123", "[02]13012345678909", 0, 6, 154, "" },
        /* 7*/ { "[91]123A1234", "[02]13012345678909", 0, 6, 154, "" },
        /* 8*/ { "[91]123A1234A", "[02]13012345678909", 0, 5, 174, "" },
        /* 9*/ { "[91]123A1234A1", "[02]13012345678909", 0, 5, 188, "" },
        /*10*/ { "[91]123A1234A12", "[02]13012345678909", 0, 5, 205, "" },
        /*11*/ { "[00]123456789012345675", "[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345" "675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[91]1234567890", 0, 32, 579, "With composite 2372 digits == max" },
        /*12*/ { "[00]123456789012345675", "[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345" "675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[91]12345678901", ZINT_ERROR_TOO_LONG, 0, 0, "With composite 2373 digits > max" },
    };
    const int data_size = ARRAY_SIZE(data);
    int i, length, composite_length, ret;
    struct zint_symbol *symbol = NULL;

    testStartSymbol("test_ean128_cc_width", &symbol);

    for (i = 0; i < data_size; i++) {

        if (testContinue(p_ctx, i)) continue;

        symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        length = testUtilSetSymbol(symbol, BARCODE_GS1_128_CC, -1 /*input_mode*/, -1 /*eci*/, 3, -1, -1, -1 /*output_options*/, data[i].data, -1, debug);
        assert_zero(length >= 128, "i:%d length %d >= 128\n", i, length);
        strcpy(symbol->primary, data[i].data);

        composite_length = (int) strlen(data[i].composite);

        ret = ZBarcode_Encode(symbol, (const unsigned char *) data[i].composite, composite_length);
        assert_equal(ret, data[i].ret, "i:%d ZBarcode_Encode ret %d != %d (%s)\n", i, ret, data[i].ret, symbol->errtxt);

        if (p_ctx->generate) {
            printf("        /*%2d*/ { \"%s\", \"%s\", %s, %d, %d, \"%s\" },\n",
                    i, data[i].data, data[i].composite, testUtilErrorName(ret), symbol->rows, symbol->width, data[i].comment);
        } else {
            assert_equal(symbol->rows, data[i].expected_rows, "i:%d symbol->rows %d != %d (%s)\n", i, symbol->rows, data[i].expected_rows, data[i].data);
            assert_equal(symbol->width, data[i].expected_width, "i:%d symbol->width %d != %d (%s)\n", i, symbol->width, data[i].expected_width, data[i].data);
        }

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

/* Test general-purpose data compaction */
static void test_encodation_0(const testCtx *const p_ctx) {
    int debug = p_ctx->debug;

    struct item {
        int symbology;
        int option_1;
        char *data;
        char *composite;
        int ret;

        int expected_rows;
        int expected_width;
        char *comment;
        char *expected;
    };
    /* Verified via bwipp_dump.ps against BWIPP and manually against tec-it.com (with noted exception) */
    static const struct item data[] = {
        /* 0*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]1", 0, 9, 55, "Single numeric",
                    "1101100110111101110101111101010001000111100011110101001"
                    "1101101110111011000010001101110010101110000011100101001"
                    "1101101100110001011111011101111010000100100011101101001"
                    "1101101000100111011100111101110011110101110011101001001"
                    "1101001000110011001000000101110100011011110011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 1*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]12", 0, 9, 55, "2 numerics",
                    "1101100110111011101011110001101111110110010011110101001"
                    "1101101110101110011110011001001100110000100011100101001"
                    "1101101100100101111010000001101000001110100011101101001"
                    "1101101000110010010001111101011000011001111011101001001"
                    "1101001000110010000010011001101001000110000011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 2*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]123", 0, 9, 55, "Odd-numbered numeric",
                    "1101100110111011101011110001000111100010001011110101001"
                    "1101101110100111000111001001101000010001100011100101001"
                    "1101101100110101110001000001111010001000001011101101001"
                    "1101101000101000010000111101110011111100101011101001001"
                    "1101001000101100011100001101101110110010000011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 3*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]1234", 0, 9, 55, "Even-numbered numeric",
                    "1101100110111011101011110001001111100110010011110101001"
                    "1101101110110001100110011001101100100000010011100101001"
                    "1101101100101111101001110001001110010111110011101101001"
                    "1101101000111000110101111101000011110101000011101001001"
                    "1101001000110101101110000001001101110001100011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 4*/ { BARCODE_UPCE_CC, 1, "1234567", "[20]12[91]12", 0, 9, 55, "Fixed len + even-numbered numeric",
                    "1101100110111111010001100101001111010001000011110101001"
                    "1101101110100011001110011001110010000010111011100101001"
                    "1101101100101111000101111001111010010001000011101101001"
                    "1101101000110011110100110001110100011111001011101001001"
                    "1101001000111100100101111001101001101110000011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 5*/ { BARCODE_UPCE_CC, 1, "1234567", "[20]12[91]123", 0, 9, 55, "Fixed len + odd-numbered numeric",
                    "1101100110111111010001100101001111010001000011110101001"
                    "1101101110100011001110011001100001101000100011100101001"
                    "1101101100101110010001111101111000010001010011101101001"
                    "1101101000110011110010110001100111101000110011101001001"
                    "1101001000100001101100100001110010001000111011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 6*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]A1234567C", 0, 9, 55, "Alphanumeric followed by 7 digits and alphanumeric",
                    "1101100110111011101011110001011100010001100011110101001"
                    "1101101110111001001011100001110010001011100011100101001"
                    "1101101100111000001000110101001011111100111011101101001"
                    "1101101000111101011110001001001111000010001011101001001"
                    "1101001000101000001100011001010001000010000011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 7*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]A123456C", 0, 9, 55, "Alphanumeric followed by 6 digits and alphanumeric",
                    "1101100110111011101011110001011100010001100011110101001"
                    "1101101110111001001011100001110010001011100011100101001"
                    "1101101100111000011001110101111110011010110011101101001"
                    "1101101000100110100011100001000100110111110011101001001"
                    "1101001000100000100010010001000001011011000011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 8*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]A12345B", 0, 9, 55, "Alphanumeric followed by 5 digits and alphanumeric",
                    "1101100110111011101011110001011100010001100011110101001"
                    "1101101110110011101100100001110110001000110011100101001"
                    "1101101100101111100110111101111110100101110011101101001"
                    "1101101000111111010011000101100101100111111011101001001"
                    "1101001000110100010110000001000100010000001011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 9*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]A1234567", 0, 9, 55, "Alphanumeric followed by 7 digits, terminating",
                    "1101100110111011101011110001011100010001100011110101001"
                    "1101101110111001001011100001111001000101111011100101001"
                    "1101101100111101011001100001111100110001010011101101001"
                    "1101101000111110010011100101110100011111001011101001001"
                    "1101001000110101100011100001101110110000100011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*10*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]A123456", 0, 9, 55, "Alphanumeric followed by 6 digits, terminating",
                    "1101100110111011101011110001011100010001100011110101001"
                    "1101101110111001001011100001110010001011100011100101001"
                    "1101101100111100000100010101001111101000111011101101001"
                    "1101101000100110001001110001011100110111000011101001001"
                    "1101001000110110000010000101100110010010000011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*11*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]A12345", 0, 9, 55, "Alphanumeric followed by 5 digits, terminating",
                    "1101100110111011101011110001011100010001100011110101001"
                    "1101101110111001001011100001111001000101111011100101001"
                    "1101101100111000101110001101001111001011110011101101001"
                    "1101101000111010011111101101100111110010001011101001001"
                    "1101001000111000100010011101000101110011100011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*12*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]A1234", 0, 9, 55, "Alphanumeric followed by 4 digits, terminating",
                    "1101100110111011101011110001011100010001100011110101001"
                    "1101101110111001001011100001110110010011000011100101001"
                    "1101101100111100110111001101011111101011000011101101001"
                    "1101101000101111001100011001000001010001111011101001001"
                    "1101001000110111011000100001110000100100111011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*13*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]A123", 0, 9, 55, "Alphanumeric followed by 3 digits, terminating",
                    "1101100110111011101011110001011100010001100011110101001"
                    "1101101110110011101100100001001000111001110011100101001"
                    "1101101100100101111100110001011101001111100011101101001"
                    "1101101000110111110000100101001111000001010011101001001"
                    "1101001000110010100001100001110010001101111011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*14*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]A123[10]C", 0, 9, 55, "Alphanumeric followed by 3 digits, 2-digit AI (3 numerics including FNC1) and alphanumeric",
                    "1101100110111011101011110001011100010001100011110101001"
                    "1101101110111001001011100001100010101100000011100101001"
                    "1101101100110100001110010001110111101111101011101101001"
                    "1101101000101110001100111001001110010011000011101001001"
                    "1101001000101100111001100001100100111001111011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*15*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]A12[10]C", 0, 9, 55, "Alphanumeric followed by 2 digits, 2-digit AI (3 numerics including FNC1) and alphanumeric",
                    "1101100110111011101011110001011100010001100011110101001"
                    "1101101110110011101100100001000001001010000011100101001"
                    "1101101100111110110100001001111110111101101011101101001"
                    "1101101000100110010000011101100111000010111011101001001"
                    "1101001000110110001000100001101100100000010011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*16*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]A12[10]1", 0, 9, 55, "Alphanumeric followed by 2 digits, 2-digit AI (3 numerics including FNC1) and 1 digit, terminating",
                    "1101100110111011101011110001011100010001100011110101001"
                    "1101101110111100100101111001000010000110110011100101001"
                    "1101101100100100011110010001001110010111110011101101001"
                    "1101101000111101111010111101000100100111100011101001001"
                    "1101001000111101110010011001001000001101100011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*17*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]A1[10]1", 0, 9, 55, "Alphanumeric followed by 1 digit, 2-digit AI (3 numerics including FNC1) and 1 digit, terminating; **NOT SAME** as tec-it.com, which does not switch to numeric mode after 'A'; same as BWIPP",
                    "1101100110111011101011110001011100010001100011110101001"
                    "1101101110110010011001110001110010010011100011100101001"
                    "1101101100111110110001001001010111111000111011101101001"
                    "1101101000111011110010011101011101100111000011101001001"
                    "1101001000110100001101110001000001010100000011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*18*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]A[10]1", 0, 9, 55, "Alphanumeric followed by 2-digit AI (3 numerics including FNC1) and 1 digit, terminating",
                    "1101100110111011101011110001101110001000111011110101001"
                    "1101101110111001100010110001100010000010110011100101001"
                    "1101101100101111110010011001001101111110110011101101001"
                    "1101101000100100100111100001110010111110100011101001001"
                    "1101001000101110001111011001111011000100111011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*19*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]a1234ABCDEFGb", 0, 12, 55, "ISO-646 followed by 11 non-ISO-646 non-terminating, starting 4 digits",
                    "1110111010101100111111011001011111000100111011011011101"
                    "1110011010101100001100111101011101000001100011011011001"
                    "1111011010100100001001000001001110011110110011011010001"
                    "1111001010100111100100111101111110110001011011010010001"
                    "1110001010101100110011110001000111001101110011010110001"
                    "1100001010101000001000100001000010111001110011010111001"
                    "1100011010111011101111000101111100110010001011010111101"
                    "1100010010100111010000001101100001010111110011010011101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*20*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]a1234ABCDEFb", 0, 12, 55, "ISO-646 followed by 10 non-ISO-646 non-terminating, starting 4 digits",
                    "1110111010101100111111011001011111000100111011011011101"
                    "1110011010101100001100111101011101000001100011011011001"
                    "1111011010100100001001000001001110011110110011011010001"
                    "1111001010100111100100111101111110110001011011010010001"
                    "1110001010111101101011111101100111100010110011010110001"
                    "1100001010110110001000001001111001011011111011010111001"
                    "1100011010110111110010111101111100011101101011010111101"
                    "1100010010101110000010001101101110000100111011010011101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*21*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]a1234ABCDEF", 0, 11, 55, "ISO-646 followed by 10 non-ISO-646 terminating, starting 4 digits",
                    "1110110110101100111111011001011111000100111011100010101"
                    "1110010110101100001100111101011101000001100011000010101"
                    "1100010110100100001001000001001110011110110011000110101"
                    "1100010100100111100100111101001001111001000011000100101"
                    "1100110100110011111000101001011110000011011011100100101"
                    "1101110100111011101001000001110001000100111011110100101"
                    "1101100100110001011100100001110001011000100011110101101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*22*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]a1234ABCDEb", 0, 11, 55, "ISO-646 followed by 9 non-ISO-646 non-terminating, starting 4 digits",
                    "1110110110101100111111011001011111000100111011100010101"
                    "1110010110101100000010011101110111100010111011000010101"
                    "1100010110110010011100111101011000111100111011000110101"
                    "1100010100111101000110001101001011110010000011000100101"
                    "1100110100100001100001011101011110011011000011100100101"
                    "1101110100111010101110000001100101001100000011110100101"
                    "1101100100111100100100100001110001100111010011110101101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*23*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]a1234ABCDE", 0, 10, 55, "ISO-646 followed by 9 non-ISO-646 terminating, starting 4 digits",
                    "1100100010111001101001100001111001111010100011100101101"
                    "1110100010111110010010001101110110011100100011000101101"
                    "1110110010100010100011110001001111000011011011000101001"
                    "1100110010100000100010000101110110001000110011001101001"
                    "1101110010100010001111000101000011001111101011011101001"
                    "1101111010111110101111011001010001000001111011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*24*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]a123ABCDEFGb", 0, 12, 55, "ISO-646 followed by 10 non-ISO-646 non-terminating, starting 3 digits (5 alphanumerics rule applies)",
                    "1110111010101100111111011001011111000100111011011011101"
                    "1110011010100100000010111101101111001100111011011011001"
                    "1111011010111101100100011101001011110111100011011010001"
                    "1111001010111110001010110001101001110000100011010010001"
                    "1110001010110011100010111001101110100001110011010110001"
                    "1100001010110100111101111101100110100010000011010111001"
                    "1100011010101000111111011101110110111001000011010111101"
                    "1100010010100011110110110001000111100100001011010011101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*25*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]aABCDEF12345b", 0, 12, 55, "ISO-646 followed by 11 non-ISO-646 non-terminating, starting 6 letters",
                    "1110111010101100111111011001011111000100111011011011101"
                    "1110011010101100000110111101111010011111011011011011001"
                    "1111011010100001001100001101010010000010000011011010001"
                    "1111001010101111110011011101110100110100000011010010001"
                    "1110001010111110101110100001001101011100000011010110001"
                    "1100001010100010011000011001100001000010011011010111001"
                    "1100011010111000101110110001110100111000011011010111101"
                    "1100010010101100010011100001001110000110111011010011101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*26*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]aABCDEF1234b", 0, 12, 55, "ISO-646 followed by 10 non-ISO-646 non-terminating, starting 6 letters",
                    "1110111010101100111111011001011111000100111011011011101"
                    "1110011010101100000110111101111010011111011011011011001"
                    "1111011010100001001100001101010010000010000011011010001"
                    "1111001010101111110011011101110100110100000011010010001"
                    "1110001010111111010110000101110001110101111011010110001"
                    "1100001010111010001011100001100001001000011011010111001"
                    "1100011010101001111010000001111100100010011011010111101"
                    "1100010010110100111111010001011001000001110011010011101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*27*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]aABCDE12345b", 0, 12, 55, "ISO-646 followed by 10 non-ISO-646 non-terminating, starting 5 letters",
                    "1110111010101100111111011001011111000100111011011011101"
                    "1110011010101100000110111101111010011111011011011011001"
                    "1111011010100001001100001101010010000010000011011010001"
                    "1111001010111001101110010001101000011100010011010010001"
                    "1110001010111011011111101001011101000000110011010110001"
                    "1100001010100011000111011001110000100101110011010111001"
                    "1100011010110100001110100001110110011100010011010111101"
                    "1100010010111111010111001101110111100101110011010011101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*28*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]aABCDE12345", 0, 11, 55, "ISO-646 followed by 10 non-ISO-646 terminating, starting 5 letters",
                    "1110110110101100111111011001011111000100111011100010101"
                    "1110010110101100000110111101111010011111011011000010101"
                    "1100010110100001001100001101010010000010000011000110101"
                    "1100010100101100101111110001101000001111011011000100101"
                    "1100110100101000000100111101010000011011111011100100101"
                    "1101110100110011101100010001110001100101100011110100101"
                    "1101100100111100110110000101111110101001110011110101101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*29*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]aABCD012345b", 0, 12, 55, "ISO-646 followed by 10 non-ISO-646 non-terminating, starting 4 letters + numeric",
                    "1110111010101100111111011001011111000100111011011011101"
                    "1110011010101100000110111101111010011111011011011011001"
                    "1111011010100001001100001101110101000111000011011010001"
                    "1111001010110010111000100001001101011111100011010010001"
                    "1110001010110010101111100001000010010111100011010110001"
                    "1100001010111101001011110001001011110011110011010111001"
                    "1100011010111000101110110001110001011000100011010111101"
                    "1100010010111101101011111101000000110101110011010011101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*30*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]aABCD012345", 0, 10, 55, "ISO-646 followed by 10 non-ISO-646 terminating, starting 4 letters + numeric",
                    "1100100010111001101001100001111001111010100011100101101"
                    "1110100010111001001101000001111010011011000011000101101"
                    "1110110010100001100100011101110101000111111011000101001"
                    "1100110010111101100100011101110001011011110011001101001"
                    "1101110010111000100011010001110010110001000011011101001"
                    "1101111010111000110101111101000110010001110011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*31*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]aABCDE1234b", 0, 11, 55, "ISO-646 followed by 9 non-ISO-646 non-terminating, starting 5 letters",
                    "1110110110101100111111011001011111000100111011100010101"
                    "1110010110100010110001111101011000001101111011000010101"
                    "1100010110111101101100111101000110000110100011000110101"
                    "1100010100111001101110010001101000111000010011000100101"
                    "1100110100100111001110111101001111101100100011100100101"
                    "1101110100110011000100000101100101001100000011110100101"
                    "1101100100111100001010000101100111011111010011110101101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*32*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]aABCDE1234", 0, 10, 55, "ISO-646 followed by 9 non-ISO-646 terminating, starting 5 letters",
                    "1100100010111001101001100001111001111010100011100101101"
                    "1110100010111001001101000001111010011011000011000101101"
                    "1110110010100001100100011101110101111110011011000101001"
                    "1100110010100100000100001001111011010011100011001101001"
                    "1101110010111101011001100001111010011000011011011101001"
                    "1101111010111010111110100001111100101110010011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*33*/ { BARCODE_UPCE_CC, 1, "1234567", "[91]aABCDE123", 0, 10, 55, "ISO-646 followed by 8 non-ISO-646 terminating, starting 5 letters",
                    "1100100010111001101001100001111001111010100011100101101"
                    "1110100010111001001101000001111010011011000011000101101"
                    "1110110010100001100100011101110101111110011011000101001"
                    "1100110010110011000100000101010000010001000011001101001"
                    "1101110010111000111011110101100010011101000011011101001"
                    "1101111010101101100001111001010000011011111011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
    };
    const int data_size = ARRAY_SIZE(data);
    int i, length, composite_length, ret;
    struct zint_symbol *symbol = NULL;

    char bwipp_buf[8192];
    char bwipp_msg[1024];

    int do_bwipp = (debug & ZINT_DEBUG_TEST_BWIPP) && testUtilHaveGhostscript(); /* Only do BWIPP test if asked, too slow otherwise */

    testStartSymbol("test_encodation_0", &symbol);

    for (i = 0; i < data_size; i++) {

        if (testContinue(p_ctx, i)) continue;

        symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        length = testUtilSetSymbol(symbol, data[i].symbology, -1 /*input_mode*/, -1 /*eci*/, data[i].option_1, -1, -1, -1 /*output_options*/, data[i].data, -1, debug);
        assert_zero(length >= 128, "i:%d length %d >= 128\n", i, length);
        strcpy(symbol->primary, data[i].data);

        composite_length = (int) strlen(data[i].composite);

        ret = ZBarcode_Encode(symbol, (const unsigned char *) data[i].composite, composite_length);
        assert_equal(ret, data[i].ret, "i:%d ZBarcode_Encode ret %d != %d (%s)\n", i, ret, data[i].ret, symbol->errtxt);

        if (p_ctx->generate) {
            test_helper_generate(symbol, ret, i, data[i].data, data[i].composite, data[i].option_1, data[i].comment, -1 /*bwipp_cmp*/);
        } else {
            int width, row;

            assert_equal(symbol->rows, data[i].expected_rows, "i:%d %s symbol->rows %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->rows, data[i].expected_rows, data[i].data);
            assert_equal(symbol->width, data[i].expected_width, "i:%d %s symbol->width %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->width, data[i].expected_width, data[i].data);

            ret = testUtilModulesCmp(symbol, data[i].expected, &width, &row);
            assert_zero(ret, "i:%d %s testUtilModulesCmp ret %d != 0 width %d row %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), ret, width, row, data[i].data);

            if (do_bwipp && testUtilCanBwipp(i, symbol, data[i].option_1, -1, -1, debug)) {
                ret = testUtilBwipp(i, symbol, data[i].option_1, -1, -1, data[i].composite, composite_length, symbol->primary, bwipp_buf, sizeof(bwipp_buf), NULL);
                assert_zero(ret, "i:%d %s testUtilBwipp ret %d != 0\n", i, testUtilBarcodeName(symbol->symbology), ret);

                ret = testUtilBwippCmp(symbol, bwipp_msg, bwipp_buf, data[i].expected);
                assert_zero(ret, "i:%d %s testUtilBwippCmp %d != 0 %s\n  actual: %s\nexpected: %s\n",
                               i, testUtilBarcodeName(symbol->symbology), ret, bwipp_msg, bwipp_buf, data[i].expected);
            }
        }

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

static void test_encodation_10(const testCtx *const p_ctx) {
    int debug = p_ctx->debug;

    struct item {
        int symbology;
        int option_1;
        char *data;
        char *composite;
        int ret;

        int expected_rows;
        int expected_width;
        char *comment;
        char *expected;
    };
    /* Verified via bwipp_dump.ps against BWIPP, and manually, with noted exceptions, against tec-it.com */
    static const struct item data[] = {
        /* 0*/ { BARCODE_UPCE_CC, 1, "1234567", "[11]201001[10]AB1234", 0, 9, 55, "Mode '10' date + even-numbered numeric lot, 1st fit, alphanumeric latch and padding",
                    "1101100110101110000100011001001000100111100011110101001"
                    "1101101110110010100011000001100110010000001011100101001"
                    "1101101100111011011101000001110100111000110011101101001"
                    "1101101000111110101110000101000110000010111011101001001"
                    "1101001000101100011010000001100110001010000011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 1*/ { BARCODE_UPCE_CC, 1, "1234567", "[11]201001[10]AB12345", 0, 9, 55, "Mode '10' date + odd-numbered numeric lot, 1st fit, 7-bit final, no alphanumeric latch or padding",
                    "1101100110101110000100011001001000100111100011110101001"
                    "1101101110110010100011000001100110010000001011100101001"
                    "1101101100111011011101000001111010000110110011101101001"
                    "1101101000100111101110111001000001001001111011101001001"
                    "1101001000101100011100011001111101010111110011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 2*/ { BARCODE_UPCE_CC, 1, "1234567", "[11]201001", 0, 9, 55, "Mode '10' date, no lot or other data",
                    "1101100110101110000100011001001000100111100011110101001"
                    "1101101110110010100110000001010000010010000011100101001"
                    "1101101100111011000001110101000001011111011011101101001"
                    "1101101000101000100111100001011110000101000011101001001"
                    "1101001000111100111001001101000110001100001011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 3*/ { BARCODE_UPCE_CC, 1, "1234567", "[11]201001[20]12", 0, 9, 55, "Mode '10' date, no lot, other data",
                    "1101100110101110000100011001100100010011111011110101001"
                    "1101101110101110111010000001000101100001100011100101001"
                    "1101101100111010011000100001111110101001110011101101001"
                    "1101101000111111010110010001010100001111000011101001001"
                    "1101001000111001011011110001100001101010000011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 4*/ { BARCODE_UPCE_CC, 1, "1234567", "[10]1234", 0, 9, 55, "Mode '10' no date, numeric lot, no other data; **NOT SAME** as tec-it.com; same as BWIPP",
                    "1101100110100111000110011101110011100101111011110101001"
                    "1101101110110100100110000001111011100101100011100101001"
                    "1101101100100111101011110001111110011010110011101101001"
                    "1101101000100110110000111101101010111110000011101001001"
                    "1101001000110100111000111101000001000010001011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 5*/ { BARCODE_UPCE_CC, 1, "1234567", "[10]ABCD", 0, 9, 55, "Mode '10' no date, alphanumeric lot, no other data; **NOT SAME** as tec-it.com; same as BWIPP",
                    "1101100110101111001100011001111000010111101011110101001"
                    "1101101110100010111100111101100110010010000011100101001"
                    "1101101100100111101001111001000010111110110011101101001"
                    "1101101000110001100010111101011101000001100011101001001"
                    "1101001000111010000101110001101000011101111011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
    };
    const int data_size = ARRAY_SIZE(data);
    int i, length, composite_length, ret;
    struct zint_symbol *symbol = NULL;

    char bwipp_buf[8192];
    char bwipp_msg[1024];

    int do_bwipp = (debug & ZINT_DEBUG_TEST_BWIPP) && testUtilHaveGhostscript(); /* Only do BWIPP test if asked, too slow otherwise */

    testStartSymbol("test_encodation_10", &symbol);

    for (i = 0; i < data_size; i++) {

        if (testContinue(p_ctx, i)) continue;

        symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        length = testUtilSetSymbol(symbol, data[i].symbology, -1 /*input_mode*/, -1 /*eci*/, data[i].option_1, -1, -1, -1 /*output_options*/, data[i].data, -1, debug);
        assert_zero(length >= 128, "i:%d length %d >= 128\n", i, length);
        strcpy(symbol->primary, data[i].data);

        composite_length = (int) strlen(data[i].composite);

        ret = ZBarcode_Encode(symbol, (const unsigned char *) data[i].composite, composite_length);
        assert_equal(ret, data[i].ret, "i:%d ZBarcode_Encode ret %d != %d (%s)\n", i, ret, data[i].ret, symbol->errtxt);

        if (p_ctx->generate) {
            test_helper_generate(symbol, ret, i, data[i].data, data[i].composite, data[i].option_1, data[i].comment, -1 /*bwipp_cmp*/);
        } else {
            int width, row;

            assert_equal(symbol->rows, data[i].expected_rows, "i:%d %s symbol->rows %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->rows, data[i].expected_rows, data[i].data);
            assert_equal(symbol->width, data[i].expected_width, "i:%d %s symbol->width %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->width, data[i].expected_width, data[i].data);

            ret = testUtilModulesCmp(symbol, data[i].expected, &width, &row);
            assert_zero(ret, "i:%d %s testUtilModulesCmp ret %d != 0 width %d row %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), ret, width, row, data[i].data);

            if (do_bwipp && testUtilCanBwipp(i, symbol, data[i].option_1, -1, -1, debug)) {
                ret = testUtilBwipp(i, symbol, data[i].option_1, -1, -1, data[i].composite, composite_length, symbol->primary, bwipp_buf, sizeof(bwipp_buf), NULL);
                assert_zero(ret, "i:%d %s testUtilBwipp ret %d != 0\n", i, testUtilBarcodeName(symbol->symbology), ret);

                ret = testUtilBwippCmp(symbol, bwipp_msg, bwipp_buf, data[i].expected);
                assert_zero(ret, "i:%d %s testUtilBwippCmp %d != 0 %s\n  actual: %s\nexpected: %s\n",
                               i, testUtilBarcodeName(symbol->symbology), ret, bwipp_msg, bwipp_buf, data[i].expected);
            }
        }

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

static void test_encodation_11(const testCtx *const p_ctx) {
    int debug = p_ctx->debug;

    struct item {
        int symbology;
        int option_1;
        char *data;
        char *composite;
        int ret;

        int expected_rows;
        int expected_width;
        char *comment;
        char *expected;
    };
    /* Verified via bwipp_dump.ps against BWIPP, and manually against tec-it.com (with noted exception) */
    static const struct item data[] = {
        /* 0*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]A", 0, 9, 55, "Mode '11', letter prefix only",
                    "1101100110100111100000101001110110010011111011110101001"
                    "1101101110110001000101100001100011010010000011100101001"
                    "1101101100110001000011100101001110010011111011101101001"
                    "1101101000110100111111010001000001100010111011101001001"
                    "1101001000110001010110000001001100011100011011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 1*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]1A", 0, 9, 55, "Mode '11', 1 digit letter prefix only",
                    "1101100110100111100000101001011001000111000011110101001"
                    "1101101110101101111011100001011100111010000011100101001"
                    "1101101100111000010011010001101011100001000011101101001"
                    "1101101000101100011001111001000111110110001011101001001"
                    "1101001000100010001110111001001100111001100011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 2*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]12A", 0, 9, 55, "Mode '11', 2 digit letter prefix only",
                    "1101100110100111100000101001011001101111000011110101001"
                    "1101101110110001010000110001110111010000010011100101001"
                    "1101101100111110011010000101111100100110111011101101001"
                    "1101101000110111100101100001000110010001110011101001001"
                    "1101001000100100111101111001000010011100111011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 3*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]123A", 0, 9, 55, "Mode '11', 3 digit letter prefix only",
                    "1101100110100111100000101001100110101111000011110101001"
                    "1101101110100001001010000001101110110100000011100101001"
                    "1101101100111110110000101001100101111100111011101101001"
                    "1101101000111001110010111101110101111110011011101001001"
                    "1101001000110011000001001001101110111001100011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 4*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]30B", 0, 9, 55, "Mode '11', 2 digit <= 30 table3 letter prefix only",
                    "1101100110110111110000100101011100100110000011110101001"
                    "1101101110101111011111001101000011000111011011100101001"
                    "1101101100110100111000000101110000100011010011101101001"
                    "1101101000111101011111001101110001111110101011101001001"
                    "1101001000111100110010011101000100000010100011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 5*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]A1234", 0, 9, 55, "Mode '11', even-numbered numeric [90]",
                    "1101100110100111100000101001110110010011111011110101001"
                    "1101101110110001000100110001101000100110000011100101001"
                    "1101101100111100010111011101100011011110100011101101001"
                    "1101101000101001100001111101001001110111111011101001001"
                    "1101001000111011100100100001101000010000011011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 6*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]A12345", 0, 9, 55, "Mode '11', odd-numbered numeric [90]",
                    "1101100110100111100000101001110110010011111011110101001"
                    "1101101110110001000100110001110100010011100011100101001"
                    "1101101100101000011110010001110100011010000011101101001"
                    "1101101000101110000010110001001111000100001011101001001"
                    "1101001000111000100010111001011000111100111011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 7*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]1ABC4", 0, 9, 55, "Mode '11', alpha [90]",
                    "1101100110100111110110010001001110011000111011110101001"
                    "1101101110110110011000011001010011000011000011100101001"
                    "1101101100101000111100000101110000100110010011101101001"
                    "1101101000100111101100001101000001001011110011101001001"
                    "1101001000110111001101000001000010000110011011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 8*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]1AB34", 0, 9, 55, "Mode '11', alphanumeric [90] (letters <= numbers)",
                    "1101100110100011100110111001011111101110010011110101001"
                    "1101101110110000110100010001100101100011100011100101001"
                    "1101101100110010000011100101101100011110001011101101001"
                    "1101101000100111011101111001110011111101001011101001001"
                    "1101001000111100010101111001110011000100011011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 9*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]1AB.D", 0, 9, 55, "Mode '11', alphanumeric [90]",
                    "1101100110100011100110111001011111101110010011110101001"
                    "1101101110110000110100010001000010100000100011100101001"
                    "1101101100101110100001111101110111000111101011101101001"
                    "1101101000100011001000011101011011001111000011101001001"
                    "1101001000110010000010001101101000111011110011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*10*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]1AB+D", 0, 9, 55, "Mode '11', ISO-646 [90]",
                    "1101100110100011100110111001011111101110010011110101001"
                    "1101101110110000110100010001101000111001111011100101001"
                    "1101101100111110110100001001100101111101110011101101001"
                    "1101101000100001011001111101000111100001001011101001001"
                    "1101001000100010000001000101101100100010000011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*11*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]1A+BD", 0, 9, 55, "Mode '11', immediate ISO-646 [90]",
                    "1101100110100011100110111001011111101110010011110101001"
                    "1101101110101111000111100101111011101011000011100101001"
                    "1101101100101000011110010001001111001011110011101101001"
                    "1101101000100111110011001001101111101001000011101001001"
                    "1101001000110110001000001001101110001100001011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*12*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]1AB#D", ZINT_ERROR_INVALID_DATA, 0, 0, "Mode '11', invalid char [90]", "Error 441: Invalid character in input (2D component)" },
        /*13*/ { BARCODE_UPCE_CC, 2, "1234567", "[90]1AB#D", ZINT_ERROR_INVALID_DATA, 0, 0, "Mode '11', invalid char [90]", "Error 441: Invalid character in input (2D component)" },
        /*14*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]A12345[21]AB", 0, 10, 55, "Mode '11', numeric [90], with [21]",
                    "1100100010111000111011011001001100011000010011100101101"
                    "1110100010101000001111001001111100110100010011000101101"
                    "1110110010101110010000110001100111000100111011000101001"
                    "1100110010100001100011000101010000010000100011001101001"
                    "1101110010111101101110011001101000001110010011011101001"
                    "1101111010100001000100111101001010001111000011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*15*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]1ABC4[21]AB12", 0, 10, 55, "Mode '11', alpha [90], with [21]",
                    "1100100010100011100111101101011000111000110011100101101"
                    "1110100010111010000110001001110100011000001011000101101"
                    "1110110010100001100100011101110111111010100011000101001"
                    "1100110010110100100001100001001001111011110011001101001"
                    "1101110010111100101100011001111101001000110011011101001"
                    "1101111010101110100001100001011101100111000011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*16*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]1AB.D[21]AB", 0, 10, 55, "Mode '11', alphanumeric [90], with [21]",
                    "1100100010110011100011000101110001110110011011100101101"
                    "1110100010111101101100100001110010011100011011000101101"
                    "1110110010110000110101111001011100110001110011000101001"
                    "1100110010110011101100000101101001110001111011001101001"
                    "1101110010110110011110001001100110001111001011011101001"
                    "1101111010100110111011111001111001000111101011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*17*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]1AB+D[21]A.", 0, 10, 55, "Mode '11', ISO-646 [90], with [21]; **NOT SAME** as tec-it.com, which probably includes '21' in 5 alphanumeric count",
                    "1100100010110011100011000101110001110110011011100101101"
                    "1110100010111101101100100001110101110001100011000101101"
                    "1110110010100000111000101101110010111110001011000101001"
                    "1100110010110111011000100001110110000100011011001101001"
                    "1101110010111000010001100101111010001110111011011101001"
                    "1101111010110001100100111101111101001110010011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*18*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]1A+BD[21]A12", 0, 11, 55, "Mode '11', immediate ISO-646 [90], with [21]; tec-it.com same, probably since have 5 alphanumerics with or without '21'",
                    "1110110110111100011000110101100111101011111011100010101"
                    "1110010110100100001101111101000100010111100011000010101"
                    "1100010110100010100000100001110011101000001011000110101"
                    "1100010100100001011111011001010111110110000011000100101"
                    "1100110100100000011010011101110001111101011011100100101"
                    "1101110100100000100011011001100001101001000011110100101"
                    "1101100100111010111100011101011110100011110011110101101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*19*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]A12[8004]12", 0, 9, 55, "Mode '11', numeric [90], with [8004]",
                    "1101100110110001111010011001000001101101111011110101001"
                    "1101101110110100100000110001011100111101100011100101001"
                    "1101101100110100000111010001100100001110001011101101001"
                    "1101101000101110001001100001101111001011000011101001001"
                    "1101001000100100110000110001101000010000110011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*20*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]AB[8004]12", 0, 9, 55, "Mode '11', alpha [90], with [8004]",
                    "1101100110100111110011001001100101111110010011110101001"
                    "1101101110101000100000100001011000111001100011100101001"
                    "1101101100100111110100011101111000100001001011101101001"
                    "1101101000101000110011111001010001110111111011101001001"
                    "1101001000110011101100001001111011100001011011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*21*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]A.[8004]12", 0, 9, 55, "Mode '11', alphanumeric [90], with [8004]",
                    "1101100110110111110011001101111110100110001011110101001"
                    "1101101110100010001001000001100101100011100011100101001"
                    "1101101100100001011110001001100001100111101011101101001"
                    "1101101000100111100100001001011110001110111011101001001"
                    "1101001000101101111100111101100011010100000011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*22*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]A+[8004]12", 0, 9, 55, "Mode '11', ISO-646 [90], with [8004]",
                    "1101100110110111110011001101111110100110001011110101001"
                    "1101101110111100110010111001001110001110100011100101001"
                    "1101101100111001011100011001101001110000010011101101001"
                    "1101101000101100000110111101000110001001110011101001001"
                    "1101001000100011000110000101111100110101111011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*23*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]12A1[10]12", 0, 9, 55, "Mode '11', numeric [90], other data",
                    "1101100110100111100000101001011001101111000011110101001"
                    "1101101110111000101101111001110110100001100011100101001"
                    "1101101100111100001010001001110000100110010011101101001"
                    "1101101000100000111010001101001000011011111011101001001"
                    "1101001000100010111011100001100001101100011011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*24*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]123AB[10]12", 0, 9, 55, "Mode '11', alpha [90], other data",
                    "1101100110100111110110010001100011101000111011110101001"
                    "1101101110111001110100000101100110001100110011100101001"
                    "1101101100110110011111001101111010100001000011101101001"
                    "1101101000100110111011111001000001100010111011101001001"
                    "1101001000110010010000110001011110011110010011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*25*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]123AB.D[10]12", 0, 10, 55, "Mode '11', alphanumeric [90], other data",
                    "1100100010100011000110000101110110000100110011100101101"
                    "1110100010101101111100100001011111001001110011000101101"
                    "1110110010101111100110001001101111001000110011000101001"
                    "1100110010110110111100111101010111110111110011001101001"
                    "1101110010101000011111001101111100110100100011011101001"
                    "1101111010110011111100110101110101111101000011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*26*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]123AB+D[10]12", 0, 10, 55, "Mode '11', ISO-646 [90], other data",
                    "1100100010100011000110000101110110000100110011100101101"
                    "1110100010101101111100100001100100111000001011000101101"
                    "1110110010100010110011111001000110100111000011000101001"
                    "1100110010111011100100001001110010110011110011001101001"
                    "1101110010111101000000100101111001101101000011011101001"
                    "1101111010101101000001110001101111100001001011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*27*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]A123[21]AB[91]A123", 0, 12, 55, "Mode '11', numeric [90], with [21], other data",
                    "1110111010100001100111110101100010011110110011011011101"
                    "1110011010111111011011100101001110110011100011011011001"
                    "1111011010101000111011100001001011110011110011011010001"
                    "1111001010111100100100100001111010111101111011010010001"
                    "1110001010101100110111100001100111110100010011010110001"
                    "1100001010100110111001100001111101111001011011010111001"
                    "1100011010111110001011011101101001110010000011010111101"
                    "1100010010111001110010111101101111101100110011010011101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*28*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]AB[8004]12[10]12", 0, 10, 55, "Mode '11', alpha [90], with [8004], other data",
                    "1100100010110000110000101001101111011101000011100101101"
                    "1110100010110100011110001101001110001011111011000101101"
                    "1110110010100011100110111001011101101110000011000101001"
                    "1100110010100000100010001001010000010000100011001101001"
                    "1101110010111010000110000101101011100010000011011101001"
                    "1101111010110011110010001101000111101100110011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*29*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]12A1234B", 0, 9, 55, "Mode '11', alphanumeric [90], with first 4 digits (choosing NUMERIC would be better, not implemented)",
                    "1101100110100011100110111001001111100011001011110101001"
                    "1101101110100101100000110001111010110011111011100101001"
                    "1101101100111100000101001001111110111101101011101101001"
                    "1101101000100000011101011001001110100000110011101001001"
                    "1101001000111000110100011001100011000010001011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*30*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]1.A", 0, 9, 55, "Fails Mode '11' test2, starting with digits then uppercase",
                    "1101100110111001101111110101111010000111101011110101001"
                    "1101101110110010001100111001000001000010001011100101001"
                    "1101101100100010111100100001110011011101000011101101001"
                    "1101101000111001010111111001011100000100011011101001001"
                    "1101001000111010001011100001101100001001000011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*31*/ { BARCODE_UPCE_CC, 1, "1234567", "[90]01A", 0, 9, 55, "Fails Mode '11' test3, no leading zeroes",
                    "1101100110111001101111110101000101011110000011110101001"
                    "1101101110101101111011100001000100111011100011100101001"
                    "1101101100111110111011101101111000001010010011101101001"
                    "1101101000101111001101100001111011001111101011101001001"
                    "1101001000111011111011100101110001010111000011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
    };
    const int data_size = ARRAY_SIZE(data);
    int i, length, composite_length, ret;
    struct zint_symbol *symbol = NULL;

    char bwipp_buf[8192];
    char bwipp_msg[1024];

    int do_bwipp = (debug & ZINT_DEBUG_TEST_BWIPP) && testUtilHaveGhostscript(); /* Only do BWIPP test if asked, too slow otherwise */

    testStartSymbol("test_encodation_11", &symbol);

    for (i = 0; i < data_size; i++) {

        if (testContinue(p_ctx, i)) continue;

        symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        length = testUtilSetSymbol(symbol, data[i].symbology, -1 /*input_mode*/, -1 /*eci*/, data[i].option_1, -1, -1, -1 /*output_options*/, data[i].data, -1, debug);
        assert_zero(length >= 128, "i:%d length %d >= 128\n", i, length);
        strcpy(symbol->primary, data[i].data);

        composite_length = (int) strlen(data[i].composite);

        ret = ZBarcode_Encode(symbol, (const unsigned char *) data[i].composite, composite_length);
        assert_equal(ret, data[i].ret, "i:%d ZBarcode_Encode ret %d != %d (%s)\n", i, ret, data[i].ret, symbol->errtxt);

        if (p_ctx->generate) {
            test_helper_generate(symbol, ret, i, data[i].data, data[i].composite, data[i].option_1, data[i].comment, -1 /*bwipp_cmp*/);
        } else {
            assert_equal(symbol->rows, data[i].expected_rows, "i:%d %s symbol->rows %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->rows, data[i].expected_rows, data[i].data);
            assert_equal(symbol->width, data[i].expected_width, "i:%d %s symbol->width %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->width, data[i].expected_width, data[i].data);

            if (ret < ZINT_ERROR) {
                int width, row;
                ret = testUtilModulesCmp(symbol, data[i].expected, &width, &row);
                assert_zero(ret, "i:%d %s testUtilModulesCmp ret %d != 0 width %d row %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), ret, width, row, data[i].data);

                if (do_bwipp && testUtilCanBwipp(i, symbol, data[i].option_1, -1, -1, debug)) {
                    ret = testUtilBwipp(i, symbol, data[i].option_1, -1, -1, data[i].composite, composite_length, symbol->primary, bwipp_buf, sizeof(bwipp_buf), NULL);
                    assert_zero(ret, "i:%d %s testUtilBwipp ret %d != 0\n", i, testUtilBarcodeName(symbol->symbology), ret);

                    ret = testUtilBwippCmp(symbol, bwipp_msg, bwipp_buf, data[i].expected);
                    assert_zero(ret, "i:%d %s testUtilBwippCmp %d != 0 %s\n  actual: %s\nexpected: %s\n",
                                   i, testUtilBarcodeName(symbol->symbology), ret, bwipp_msg, bwipp_buf, data[i].expected);
                }
            }
        }

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

static void test_addongap(const testCtx *const p_ctx) {
    int debug = p_ctx->debug;

    struct item {
        int symbology;
        int option_1;
        int option_2;
        char *data;
        int ret;

        int expected_rows;
        int expected_width;
        char *comment;
        char *expected;
    };
    /* Verified via bwipp_dump.ps against BWIPP */
    static const struct item data[] = {
        /*  0*/ { BARCODE_EANX_CC, 1, -1, "1234567+12", 0, 8, 98, "EAN-8 default 7 gap",
                    "10010001111100110101001100011110001011001100110110011110000101110100110100000000000000000000000000"
                    "10000011100101100101011100010111100010001000100000100101111001110101110100000000000000000000000000"
                    "11001101001000000101011000011010011100011110110001100110011001110101100100000000000000000000000000"
                    "11111000101001100101001000011100001011100110111000110111010001110101000100000000000000000000000000"
                    "00001000000000000000000000000000000000000000000000000000000000000000001000000000000000000000000000"
                    "00010000000000000000000000000000000000000000000000000000000000000000000100000000000000000000000000"
                    "00001000000000000000000000000000000000000000000000000000000000000000001000000000000000000000000000"
                    "00001010011001001001101111010100011010101001110101000010001001110010101000000010110011001010010011"
                },
        /*  1*/ { BARCODE_EANX_CC, 1, 8, "1234567+12", 0, 8, 99, "EAN-8 8 gap",
                    "100100011111001101010011000111100010110011001101100111100001011101001101000000000000000000000000000"
                    "100000111001011001010111000101111000100010001000001001011110011101011101000000000000000000000000000"
                    "110011010010000001010110000110100111000111101100011001100110011101011001000000000000000000000000000"
                    "111110001010011001010010000111000010111001101110001101110100011101010001000000000000000000000000000"
                    "000010000000000000000000000000000000000000000000000000000000000000000010000000000000000000000000000"
                    "000100000000000000000000000000000000000000000000000000000000000000000001000000000000000000000000000"
                    "000010000000000000000000000000000000000000000000000000000000000000000010000000000000000000000000000"
                    "000010100110010010011011110101000110101010011101010000100010011100101010000000010110011001010010011"
                },
        /*  2*/ { BARCODE_EANX_CC, 1, -1, "123456789012+12345", 0, 7, 152, "EAN-13 default 7 gap",
                    "11011011101111001101001110011100010001001110100111010010001011000001100111100111101000101101100010100000000000000000000000000000000000000000000000000000"
                    "11011011001100010001110010011100110001110010100110010010010001111101100111101001010000001100100010100000000000000000000000000000000000000000000000000000"
                    "11011010001110111110101100010000101011110000100110011010000110001001110101111101100000101110100010100000000000000000000000000000000000000000000000000000"
                    "00010000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001000000000000000000000000000000000000000000000000000000"
                    "00100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000100000000000000000000000000000000000000000000000000000"
                    "00010000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001000000000000000000000000000000000000000000000000000000"
                    "00010100100110111101001110101100010000101001000101010100100011101001110010110011011011001001000101000000010110110011010010011010100001010100011010110001"
                },
        /*  3*/ { BARCODE_EANX_CC, 1, 9, "123456789012+12345", 0, 7, 154, "EAN-13 9 gap",
                    "1101101110111100110100111001110001000100111010011101001000101100000110011110011110100010110110001010000000000000000000000000000000000000000000000000000000"
                    "1101101100110001000111001001110011000111001010011001001001000111110110011110100101000000110010001010000000000000000000000000000000000000000000000000000000"
                    "1101101000111011111010110001000010101111000010011001101000011000100111010111110110000010111010001010000000000000000000000000000000000000000000000000000000"
                    "0001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000100000000000000000000000000000000000000000000000000000000"
                    "0010000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010000000000000000000000000000000000000000000000000000000"
                    "0001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000100000000000000000000000000000000000000000000000000000000"
                    "0001010010011011110100111010110001000010100100010101010010001110100111001011001101101100100100010100000000010110110011010010011010100001010100011010110001"
                },
        /*  4*/ { BARCODE_UPCA_CC, 1, -1, "12345678901+12345", 0, 7, 154, "UPC-A default 9 gap",
                    "1101101110111100110100111001110001000100111010011101001000101100000110011110011110100010110110001010000000000000000000000000000000000000000000000000000000"
                    "1101101100110001000111001001110011000111001010011001001001000111110110011110100101000000110010001010000000000000000000000000000000000000000000000000000000"
                    "1101101000111011111010110001000010101111000010011001101000011000100111010111110110000010111010001010000000000000000000000000000000000000000000000000000000"
                    "0001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000100000000000000000000000000000000000000000000000000000000"
                    "0010000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010000000000000000000000000000000000000000000000000000000"
                    "0001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000100000000000000000000000000000000000000000000000000000000"
                    "0001010011001001001101111010100011011000101011110101010001001001000111010011100101100110110110010100000000010110110011010010011010100001010100011010110001"
                },
        /*  5*/ { BARCODE_UPCA_CC, 1, 10, "12345678901+12345", 0, 7, 155, "UPC-A 10 gap",
                    "11011011101111001101001110011100010001001110100111010010001011000001100111100111101000101101100010100000000000000000000000000000000000000000000000000000000"
                    "11011011001100010001110010011100110001110010100110010010010001111101100111101001010000001100100010100000000000000000000000000000000000000000000000000000000"
                    "11011010001110111110101100010000101011110000100110011010000110001001110101111101100000101110100010100000000000000000000000000000000000000000000000000000000"
                    "00010000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001000000000000000000000000000000000000000000000000000000000"
                    "00100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000100000000000000000000000000000000000000000000000000000000"
                    "00010000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001000000000000000000000000000000000000000000000000000000000"
                    "00010100110010010011011110101000110110001010111101010100010010010001110100111001011001101101100101000000000010110110011010010011010100001010100011010110001"
                },
        /*  6*/ { BARCODE_UPCE_CC, 1, -1, "1234567+12", 0, 9, 81, "UPC-E default 7 gap",
                    "110110011011101110101111000110111111011001001111010100100000000000000000000000000"
                    "110110111010111001111001100100110011000010001110010100100000000000000000000000000"
                    "110110110010010111101000000110100000111010001110110100100000000000000000000000000"
                    "110110100011001001000111110101100001100111101110100100100000000000000000000000000"
                    "110100100011001000001001100110100100011000001110100110100000000000000000000000000"
                    "000100000000000000000000000000000000000000000000000001000000000000000000000000000"
                    "001000000000000000000000000000000000000000000000000000100000000000000000000000000"
                    "000100000000000000000000000000000000000000000000000001000000000000000000000000000"
                    "000101001001101111010100011011100100001010010001010101000000010110011001010010011"
                },
        /*  7*/ { BARCODE_UPCE_CC, 1, 12, "1234567+12", 0, 9, 86, "UPC-E 12 gap",
                    "11011001101110111010111100011011111101100100111101010010000000000000000000000000000000"
                    "11011011101011100111100110010011001100001000111001010010000000000000000000000000000000"
                    "11011011001001011110100000011010000011101000111011010010000000000000000000000000000000"
                    "11011010001100100100011111010110000110011110111010010010000000000000000000000000000000"
                    "11010010001100100000100110011010010001100000111010011010000000000000000000000000000000"
                    "00010000000000000000000000000000000000000000000000000100000000000000000000000000000000"
                    "00100000000000000000000000000000000000000000000000000010000000000000000000000000000000"
                    "00010000000000000000000000000000000000000000000000000100000000000000000000000000000000"
                    "00010100100110111101010001101110010000101001000101010100000000000010110011001010010011"
                },
    };
    const int data_size = ARRAY_SIZE(data);
    int i, length, composite_length, ret;
    struct zint_symbol *symbol = NULL;

    char bwipp_buf[8192];
    char bwipp_msg[1024];

    char *composite = "[91]12";

    int do_bwipp = (debug & ZINT_DEBUG_TEST_BWIPP) && testUtilHaveGhostscript(); /* Only do BWIPP test if asked, too slow otherwise */

    testStartSymbol("test_addongap", &symbol);

    for (i = 0; i < data_size; i++) {

        if (testContinue(p_ctx, i)) continue;

        symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        length = testUtilSetSymbol(symbol, data[i].symbology, -1 /*input_mode*/, -1 /*eci*/, data[i].option_1, data[i].option_2, -1, -1 /*output_options*/, data[i].data, -1, debug);
        assert_zero(length >= 128, "i:%d length %d >= 128\n", i, length);
        strcpy(symbol->primary, data[i].data);

        composite_length = (int) strlen(composite);

        ret = ZBarcode_Encode(symbol, (const unsigned char *) composite, composite_length);
        assert_equal(ret, data[i].ret, "i:%d ZBarcode_Encode ret %d != %d (%s)\n", i, ret, data[i].ret, symbol->errtxt);

        if (p_ctx->generate) {
            printf("        /*%3d*/ { %s, %d, %d, \"%s\", %s, %d, %d, \"%s\",\n",
                                i, testUtilBarcodeName(data[i].symbology), data[i].option_1, data[i].option_2, data[i].data,
                                testUtilErrorName(data[i].ret), symbol->rows, symbol->width, data[i].comment);
            testUtilModulesPrint(symbol, "                    ", "\n");
            printf("                },\n");
        } else {
            int width, row;

            assert_equal(symbol->rows, data[i].expected_rows, "i:%d %s symbol->rows %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->rows, data[i].expected_rows, data[i].data);
            assert_equal(symbol->width, data[i].expected_width, "i:%d %s symbol->width %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->width, data[i].expected_width, data[i].data);

            ret = testUtilModulesCmp(symbol, data[i].expected, &width, &row);
            assert_zero(ret, "i:%d %s testUtilModulesCmp ret %d != 0 width %d row %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), ret, width, row, data[i].data);

            if (do_bwipp && testUtilCanBwipp(i, symbol, data[i].option_1, data[i].option_2, -1, debug)) {
                ret = testUtilBwipp(i, symbol, data[i].option_1, data[i].option_2, -1, composite, composite_length, symbol->primary, bwipp_buf, sizeof(bwipp_buf), NULL);
                assert_zero(ret, "i:%d %s testUtilBwipp ret %d != 0\n", i, testUtilBarcodeName(symbol->symbology), ret);

                ret = testUtilBwippCmp(symbol, bwipp_msg, bwipp_buf, data[i].expected);
                assert_zero(ret, "i:%d %s testUtilBwippCmp %d != 0 %s\n  actual: %s\nexpected: %s\n",
                               i, testUtilBarcodeName(symbol->symbology), ret, bwipp_msg, bwipp_buf, data[i].expected);
            }
        }

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

static void test_gs1parens(const testCtx *const p_ctx) {
    int debug = p_ctx->debug;

    struct item {
        int symbology;
        int input_mode;
        char *data;
        char *composite;
        int ret;

        int expected_rows;
        int expected_width;
    };
    /* s/\/\*[ 0-9]*\*\//\=printf("\/\*%3d*\/", line(".") - line("'<")): */
    static const struct item data[] = {
        /*  0*/ { BARCODE_EANX_CC, -1, "1234567", "[21]A12345678", 0, 8, 72 }, /* EAN-8 */
        /*  1*/ { BARCODE_EANX_CC, GS1PARENS_MODE, "1234567", "(21)A12345678", 0, 8, 72 }, /* EAN-8 */
        /*  2*/ { BARCODE_EANX_CC, -1, "123456789012", "[21]A12345678", 0, 7, 99 }, /* EAN-13 */
        /*  3*/ { BARCODE_EANX_CC, GS1PARENS_MODE, "123456789012", "(21)A12345678", 0, 7, 99 }, /* EAN-13 */
        /*  4*/ { BARCODE_GS1_128_CC, -1, "[01]12345678901231", "[21]A12345678", 0, 5, 145 },
        /*  5*/ { BARCODE_GS1_128_CC, GS1PARENS_MODE, "(01)12345678901231", "(21)A12345678", 0, 5, 145 },
        /*  6*/ { BARCODE_DBAR_OMN_CC, -1, "12345678901231", "[21]A12345678", 0, 5, 100 },
        /*  7*/ { BARCODE_DBAR_OMN_CC, GS1PARENS_MODE, "12345678901231", "(21)A12345678", 0, 5, 100 },
        /*  8*/ { BARCODE_DBAR_LTD_CC, -1, "12345678901231", "[21]A12345678", 0, 6, 79 },
        /*  9*/ { BARCODE_DBAR_LTD_CC, GS1PARENS_MODE, "12345678901231", "(21)A12345678", 0, 6, 79 },
        /* 10*/ { BARCODE_DBAR_EXP_CC, -1, "[01]12345678901231[3103]001234", "[21]A12345678", 0, 5, 200 },
        /* 11*/ { BARCODE_DBAR_EXP_CC, GS1PARENS_MODE, "(01)12345678901231(3103)001234", "(21)A12345678", 0, 5, 200 },
        /* 12*/ { BARCODE_UPCA_CC, -1, "12345678901", "[21]A12345678", 0, 7, 99 },
        /* 13*/ { BARCODE_UPCA_CC, GS1PARENS_MODE, "12345678901", "(21)A12345678", 0, 7, 99 },
        /* 14*/ { BARCODE_UPCE_CC, -1, "1234567", "[21]A12345678", 0, 9, 55 },
        /* 15*/ { BARCODE_UPCE_CC, GS1PARENS_MODE, "1234567", "(21)A12345678", 0, 9, 55 },
        /* 16*/ { BARCODE_DBAR_STK_CC, -1, "12345678901231", "[21]A12345678", 0, 9, 56 },
        /* 17*/ { BARCODE_DBAR_STK_CC, GS1PARENS_MODE, "12345678901231", "(21)A12345678", 0, 9, 56 },
        /* 18*/ { BARCODE_DBAR_OMNSTK_CC, -1, "12345678901231", "[21]A12345678", 0, 11, 56 },
        /* 19*/ { BARCODE_DBAR_OMNSTK_CC, GS1PARENS_MODE, "12345678901231", "(21)A12345678", 0, 11, 56 },
        /* 20*/ { BARCODE_DBAR_EXPSTK_CC, -1, "[01]12345678901231[3103]001234", "[21]A12345678", 0, 9, 102 },
        /* 21*/ { BARCODE_DBAR_EXPSTK_CC, GS1PARENS_MODE, "(01)12345678901231(3103)001234", "(21)A12345678", 0, 9, 102 },
    };
    const int data_size = ARRAY_SIZE(data);
    int i, length, composite_length, ret;
    struct zint_symbol *symbol = NULL;

    testStartSymbol("test_gs1parens", &symbol);

    for (i = 0; i < data_size; i++) {

        if (testContinue(p_ctx, i)) continue;

        symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        length = testUtilSetSymbol(symbol, data[i].symbology, data[i].input_mode, -1 /*eci*/, -1 /*option_1*/, -1, -1, -1 /*output_options*/, data[i].data, -1, debug);
        assert_zero(length >= 128, "i:%d length %d >= 128\n", i, length);
        strcpy(symbol->primary, data[i].data);

        composite_length = (int) strlen(data[i].composite);

        ret = ZBarcode_Encode(symbol, (const unsigned char *) data[i].composite, composite_length);
        assert_equal(ret, data[i].ret, "i:%d ZBarcode_Encode ret %d != %d (%s)\n", i, ret, data[i].ret, symbol->errtxt);

        assert_equal(symbol->rows, data[i].expected_rows, "i:%d symbol->rows %d != %d\n", i, symbol->rows, data[i].expected_rows);
        assert_equal(symbol->width, data[i].expected_width, "i:%d symbol->width %d != %d\n", i, symbol->width, data[i].expected_width);

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

static void test_hrt(const testCtx *const p_ctx) {
    int debug = p_ctx->debug;

    struct item {
        int symbology;
        int input_mode;
        char *data;
        char *composite;

        int ret;
        char *expected;
    };
    /* s/\/\*[ 0-9]*\*\//\=printf("\/\*%3d*\/", line(".") - line("'<")): */
    static const struct item data[] = {
        /*  0*/ { BARCODE_EANX_CC, -1, "1234567", "[20]12", 0, "12345670" }, /* EAN-8 */
        /*  1*/ { BARCODE_EANX_CC, -1, "123456789012", "[20]12", 0, "1234567890128" }, /* EAN-13 */
        /*  2*/ { BARCODE_EANX_CC, -1, "1234567890128", "[20]12", 0, "1234567890128" },
        /*  3*/ { BARCODE_EANX_CC, -1, "1234567890123", "[20]12", ZINT_ERROR_INVALID_CHECK, "" },
        /*  4*/ { BARCODE_EANX_CC, GS1NOCHECK_MODE, "1234567890123", "[20]12", ZINT_ERROR_INVALID_CHECK, "" }, /* Still checked */
        /*  5*/ { BARCODE_EANX_CC, -1, "1234567890128", "[20]1A", ZINT_WARN_NONCOMPLIANT, "1234567890128" }, /* AI (20) should be 2 nos. */
        /*  6*/ { BARCODE_EANX_CC, GS1NOCHECK_MODE, "1234567890128", "[20]1A", 0, "1234567890128" },
        /*  7*/ { BARCODE_DBAR_OMN_CC, -1, "1234567890123", "[20]12", 0, "(01)12345678901231" },
        /*  8*/ { BARCODE_DBAR_OMN_CC, -1, "12345678901231", "[20]12", 0, "(01)12345678901231" },
        /*  9*/ { BARCODE_DBAR_OMN_CC, -1, "12345678901232", "[20]12", ZINT_ERROR_INVALID_CHECK, "" },
        /* 10*/ { BARCODE_DBAR_OMN_CC, GS1NOCHECK_MODE, "12345678901232", "[20]12", ZINT_ERROR_INVALID_CHECK, "" }, /* Still checked */
        /* 11*/ { BARCODE_DBAR_OMN_CC, -1, "12345678901231", "[20]1A", ZINT_WARN_NONCOMPLIANT, "(01)12345678901231" }, /* AI (20) should be 2 nos. */
        /* 12*/ { BARCODE_DBAR_OMN_CC, GS1NOCHECK_MODE, "12345678901231", "[20]1A", 0, "(01)12345678901231" },
        /* 13*/ { BARCODE_DBAR_LTD_CC, -1, "1234567890123", "[20]12", 0, "(01)12345678901231" },
        /* 14*/ { BARCODE_DBAR_LTD_CC, -1, "12345678901231", "[20]12", 0, "(01)12345678901231" },
        /* 15*/ { BARCODE_DBAR_LTD_CC, -1, "12345678901232", "[20]12", ZINT_ERROR_INVALID_CHECK, "" },
        /* 16*/ { BARCODE_DBAR_LTD_CC, GS1NOCHECK_MODE, "12345678901232", "[20]12", ZINT_ERROR_INVALID_CHECK, "" }, /* Still checked */
        /* 17*/ { BARCODE_DBAR_LTD_CC, -1, "12345678901231", "[20]1A", ZINT_WARN_NONCOMPLIANT, "(01)12345678901231" }, /* AI (20) should be 2 nos. */
        /* 18*/ { BARCODE_DBAR_LTD_CC, GS1NOCHECK_MODE, "12345678901231", "[20]1A", 0, "(01)12345678901231" },
        /* 19*/ { BARCODE_UPCA_CC, -1, "12345678901", "[20]12", 0, "123456789012" },
        /* 20*/ { BARCODE_UPCA_CC, -1, "123456789012", "[20]12", 0, "123456789012" },
        /* 21*/ { BARCODE_UPCA_CC, -1, "123456789013", "[20]12", ZINT_ERROR_INVALID_CHECK, "" },
        /* 22*/ { BARCODE_UPCA_CC, GS1NOCHECK_MODE, "123456789013", "[20]12", ZINT_ERROR_INVALID_CHECK, "" }, /* Still checked */
        /* 23*/ { BARCODE_UPCA_CC, -1, "123456789012", "[20]1A", ZINT_WARN_NONCOMPLIANT, "123456789012" }, /* AI (20) should be 2 nos. */
        /* 24*/ { BARCODE_UPCA_CC, GS1NOCHECK_MODE, "123456789012", "[20]1A", 0, "123456789012" },
        /* 25*/ { BARCODE_UPCE_CC, -1, "123456", "[20]12", 0, "01234565" },
        /* 26*/ { BARCODE_UPCE_CC, -1, "1234567", "[20]12", 0, "12345670" },
        /* 27*/ { BARCODE_UPCE_CC, -1, "12345670", "[20]12", 0, "12345670" },
        /* 28*/ { BARCODE_UPCE_CC, -1, "12345671", "[20]12", ZINT_ERROR_INVALID_CHECK, "" },
        /* 29*/ { BARCODE_UPCE_CC, GS1NOCHECK_MODE, "12345671", "[20]12", ZINT_ERROR_INVALID_CHECK, "" }, /* Still checked */
        /* 30*/ { BARCODE_UPCE_CC, -1, "12345670", "[20]12", 0, "12345670" }, /* Check digit can now be given for UPCE_CC, like UPCA_CC */
        /* 31*/ { BARCODE_UPCE_CC, -1, "1234567", "[20]1A", ZINT_WARN_NONCOMPLIANT, "12345670" }, /* AI (20) should be 2 nos. */
        /* 32*/ { BARCODE_UPCE_CC, GS1NOCHECK_MODE, "1234567", "[20]1A", 0, "12345670" },
        /* 33*/ { BARCODE_DBAR_STK_CC, -1, "12345678901231", "[20]12", 0, "" }, /* No HRT for stacked symbologies */
        /* 34*/ { BARCODE_DBAR_OMNSTK_CC, -1, "12345678901231", "[20]12", 0, "" },
    };
    const int data_size = ARRAY_SIZE(data);
    int i, length, composite_length, ret;
    struct zint_symbol *symbol = NULL;

    testStartSymbol("test_hrt", &symbol);

    for (i = 0; i < data_size; i++) {

        if (testContinue(p_ctx, i)) continue;

        symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        length = testUtilSetSymbol(symbol, data[i].symbology, data[i].input_mode, -1 /*eci*/, -1 /*option_1*/, -1, -1, -1 /*output_options*/, data[i].data, -1, debug);
        assert_zero(length >= 128, "i:%d length %d >= 128\n", i, length);
        strcpy(symbol->primary, data[i].data);

        composite_length = (int) strlen(data[i].composite);

        ret = ZBarcode_Encode(symbol, (const unsigned char *) data[i].composite, composite_length);
        assert_equal(ret, data[i].ret, "i:%d ZBarcode_Encode ret %d != %d (%s)\n", i, data[i].ret, ret, symbol->errtxt);

        assert_zero(strcmp((const char *) symbol->text, data[i].expected), "i:%d strcmp(%s, %s) != 0\n", i, symbol->text, data[i].expected);

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

/* See also `test_gs1nocheck_mode()` in "test.gs1.c" for GS1_128-based and DBAR_EXP-based symbologies */
static void test_input(const testCtx *const p_ctx) {
    int debug = p_ctx->debug;

    struct item {
        int symbology;
        int input_mode;
        int option_1;
        int option_2;
        int option_3;
        char *data;
        char *composite;

        int ret;
        int expected_rows;
        int expected_width;
        char *expected_errtxt;
    };
    /* s/\/\*[ 0-9]*\*\//\=printf("\/\*%3d*\/", line(".") - line("'<")): */
    static const struct item data[] = {
        /*  0*/ { BARCODE_EANX_CC, -1, -1, -1, -1, "1234567", "[20]12", 0, 8, 72, "" }, /* EAN-8 */
        /*  1*/ { BARCODE_EANX_CC, -1, -1, -1, -1, "123456A", "[20]12", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 284: Invalid character at position 7 in input (digits and \"+\" only) (linear component)" }, /* EAN-8 */
        /*  2*/ { BARCODE_EANX_CC, GS1NOCHECK_MODE, -1, -1, -1, "123456A", "[20]12", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 284: Invalid character at position 7 in input (digits and \"+\" only) (linear component)" }, /* Linear component still checked */
        /*  3*/ { BARCODE_EANX_CC, -1, -1, -1, -1, "1234567", "[20]1A", ZINT_WARN_NONCOMPLIANT, 8, 72, "Warning 261: AI (20) position 2: Non-numeric character 'A' (2D component)" },
        /*  4*/ { BARCODE_EANX_CC, GS1NOCHECK_MODE, -1, -1, -1, "1234567", "[20]1A", 0, 8, 72, "" },
        /*  5*/ { BARCODE_EANX_CC, -1, -1, -1, -1, "1234567", "[02]12345678901234", ZINT_WARN_NONCOMPLIANT, 8, 72, "Warning 261: AI (02) position 14: Bad checksum '4', expected '1' (2D component)" },
        /*  6*/ { BARCODE_EANX_CC, GS1NOCHECK_MODE, -1, -1, -1, "1234567", "[02]12345678901234", 0, 8, 72, "" },
        /*  7*/ { BARCODE_EANX_CC, -1, -1, -1, -1, "12345671", "[20]12", 0, 7, 99, "" }, /* EAN-13 for EANX_CC as length 8 only EAN-8 for EANX_CHK */
        /*  8*/ { BARCODE_EANX_CC, -1, -1, -1, -1, "123456789012", "[20]12", 0, 7, 99, "" }, /* EAN-13 */
        /*  9*/ { BARCODE_EANX_CC, -1, -1, -1, -1, "1234567890128", "[20]12", 0, 7, 99, "" }, /* EAN-13 */
        /* 10*/ { BARCODE_EANX_CC, -1, -1, -1, -1, "1234567890123", "[20]12", ZINT_ERROR_INVALID_CHECK, -1, -1, "Error 275: Invalid check digit '3', expecting '8' (linear component)" },
        /* 11*/ { BARCODE_EANX_CC, GS1NOCHECK_MODE, -1, -1, -1, "1234567890123", "[20]12", ZINT_ERROR_INVALID_CHECK, -1, -1, "Error 275: Invalid check digit '3', expecting '8' (linear component)" }, /* Linear component still checked */
        /* 12*/ { BARCODE_EANX_CC, -1, -1, -1, -1, "12345678901234", "[20]12", ZINT_ERROR_TOO_LONG, -1, -1, "Error 294: Input length 14 too long (maximum 13) (linear component)" },
        /* 13*/ { BARCODE_EANX_CC, GS1NOCHECK_MODE, -1, -1, -1, "12345678901234", "[20]12", ZINT_ERROR_TOO_LONG, -1, -1, "Error 294: Input length 14 too long (maximum 13) (linear component)" },
        /* 14*/ { BARCODE_EANX_CC, -1, -1, -1, -1, "123456789012A", "[20]12", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 284: Invalid character at position 13 in input (digits and \"+\" only) (linear component)" },
        /* 15*/ { BARCODE_EANX_CC, GS1NOCHECK_MODE, -1, -1, -1, "123456789012A", "[20]12", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 284: Invalid character at position 13 in input (digits and \"+\" only) (linear component)" },
        /* 16*/ { BARCODE_EANX_CC, -1, -1, -1, -1, "1234567890128", "[20]1A", ZINT_WARN_NONCOMPLIANT, 7, 99, "Warning 261: AI (20) position 2: Non-numeric character 'A' (2D component)" }, /* AI (20) should be 2 nos. */
        /* 17*/ { BARCODE_EANX_CC, GS1NOCHECK_MODE, -1, -1, -1, "1234567890128", "[20]1A", 0, 7, 99, "" },
        /* 18*/ { BARCODE_EANX_CC, -1, -1, -1, -1, "1234567890128", "[02]12345678901234", ZINT_WARN_NONCOMPLIANT, 7, 99, "Warning 261: AI (02) position 14: Bad checksum '4', expected '1' (2D component)" },
        /* 19*/ { BARCODE_EANX_CC, GS1NOCHECK_MODE, -1, -1, -1, "1234567890128", "[02]12345678901234", 0, 7, 99, "" },
        /* 20*/ { BARCODE_EANX_CC, -1, -1, -1, -1, "1234567890128", "[91]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[92]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[93]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[94]123456789012345678901234567890123456789012345678901234567", 0, 48, 99, "" }, /* Max CC-B for EAN-13 */
        /* 21*/ { BARCODE_EANX_CC, -1, -1, -1, -1, "1234567890128", "[91]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[92]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[93]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[94]1234567890123456789012345678901234567890123456789012345678", ZINT_ERROR_TOO_LONG, -1, -1, "Error 444: Input too long (2D component)" },
        /* 22*/ { BARCODE_EANX_CC, -1, 3, -1, -1, "1234567890128", "[20]12", ZINT_ERROR_INVALID_OPTION, -1, -1, "Error 447: Invalid mode (CC-C only valid with GS1-128 linear component)" },
        /* 23*/ { BARCODE_EANX_CC, -1, -1, -1, -1, "123456789012345678901", "[20]12", ZINT_ERROR_TOO_LONG, -1, -1, "Error 449: Input length 21 wrong (linear component)" },
        /* 24*/ { BARCODE_DBAR_OMN_CC, -1, -1, -1, -1, "1234567890123", "[20]12", 0, 5, 100, "" },
        /* 25*/ { BARCODE_DBAR_OMN_CC, -1, -1, -1, -1, "12345678901231", "[20]12", 0, 5, 100, "" },
        /* 26*/ { BARCODE_DBAR_OMN_CC, -1, -1, -1, -1, "12345678901232", "[20]12", ZINT_ERROR_INVALID_CHECK, -1, -1, "Error 388: Invalid check digit '2', expecting '1' (linear component)" },
        /* 27*/ { BARCODE_DBAR_OMN_CC, GS1NOCHECK_MODE, -1, -1, -1, "12345678901232", "[20]12", ZINT_ERROR_INVALID_CHECK, -1, -1, "Error 388: Invalid check digit '2', expecting '1' (linear component)" }, /* Linear component still checked */
        /* 28*/ { BARCODE_DBAR_OMN_CC, -1, -1, -1, -1, "123456789012312", "[20]12", ZINT_ERROR_TOO_LONG, -1, -1, "Error 380: Input length 15 too long (maximum 14) (linear component)" },
        /* 29*/ { BARCODE_DBAR_OMN_CC, GS1NOCHECK_MODE, -1, -1, -1, "123456789012312", "[20]12", ZINT_ERROR_TOO_LONG, -1, -1, "Error 380: Input length 15 too long (maximum 14) (linear component)" },
        /* 30*/ { BARCODE_DBAR_OMN_CC, -1, -1, -1, -1, "1234567890123A", "[20]12", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 381: Invalid character at position 14 in input (digits only) (linear component)" },
        /* 31*/ { BARCODE_DBAR_OMN_CC, GS1NOCHECK_MODE, -1, -1, -1, "1234567890123A", "[20]12", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 381: Invalid character at position 14 in input (digits only) (linear component)" },
        /* 32*/ { BARCODE_DBAR_OMN_CC, -1, -1, -1, -1, "12345678901231", "[20]1A", ZINT_WARN_NONCOMPLIANT, 5, 100, "Warning 261: AI (20) position 2: Non-numeric character 'A' (2D component)" }, /* AI (20) should be 2 nos. */
        /* 33*/ { BARCODE_DBAR_OMN_CC, GS1NOCHECK_MODE, -1, -1, -1, "12345678901231", "[20]1A", 0, 5, 100, "" },
        /* 34*/ { BARCODE_DBAR_OMN_CC, -1, -1, -1, -1, "12345678901231", "[02]12345678901234", ZINT_WARN_NONCOMPLIANT, 5, 100, "Warning 261: AI (02) position 14: Bad checksum '4', expected '1' (2D component)" },
        /* 35*/ { BARCODE_DBAR_OMN_CC, GS1NOCHECK_MODE, -1, -1, -1, "12345678901231", "[02]12345678901234", 0, 5, 100, "" },
        /* 36*/ { BARCODE_DBAR_LTD_CC, -1, -1, -1, -1, "1234567890123", "[20]12", 0, 6, 79, "" },
        /* 37*/ { BARCODE_DBAR_LTD_CC, -1, -1, -1, -1, "12345678901231", "[20]12", 0, 6, 79, "" },
        /* 38*/ { BARCODE_DBAR_LTD_CC, -1, -1, -1, -1, "12345678901232", "[20]12", ZINT_ERROR_INVALID_CHECK, -1, -1, "Error 389: Invalid check digit '2', expecting '1' (linear component)" },
        /* 39*/ { BARCODE_DBAR_LTD_CC, GS1NOCHECK_MODE, -1, -1, -1, "12345678901232", "[20]12", ZINT_ERROR_INVALID_CHECK, -1, -1, "Error 389: Invalid check digit '2', expecting '1' (linear component)" }, /* Linear component still checked */
        /* 40*/ { BARCODE_DBAR_LTD_CC, -1, -1, -1, -1, "123456789012345", "[20]12", ZINT_ERROR_TOO_LONG, -1, -1, "Error 382: Input length 15 too long (maximum 14) (linear component)" },
        /* 41*/ { BARCODE_DBAR_LTD_CC, GS1NOCHECK_MODE, -1, -1, -1, "123456789012345", "[20]12", ZINT_ERROR_TOO_LONG, -1, -1, "Error 382: Input length 15 too long (maximum 14) (linear component)" },
        /* 42*/ { BARCODE_DBAR_LTD_CC, -1, -1, -1, -1, "A1234567890123", "[20]12", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 383: Invalid character at position 1 in input (digits only) (linear component)" },
        /* 43*/ { BARCODE_DBAR_LTD_CC, GS1NOCHECK_MODE, -1, -1, -1, "A1234567890123", "[20]12", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 383: Invalid character at position 1 in input (digits only) (linear component)" },
        /* 44*/ { BARCODE_DBAR_LTD_CC, -1, -1, -1, -1, "12345678901231", "[20]1A", ZINT_WARN_NONCOMPLIANT, 6, 79, "Warning 261: AI (20) position 2: Non-numeric character 'A' (2D component)" }, /* AI (20) should be 2 nos. */
        /* 45*/ { BARCODE_DBAR_LTD_CC, GS1NOCHECK_MODE, -1, -1, -1, "12345678901231", "[20]1A", 0, 6, 79, "" },
        /* 46*/ { BARCODE_DBAR_LTD_CC, -1, -1, -1, -1, "12345678901231", "[02]12345678901234", ZINT_WARN_NONCOMPLIANT, 6, 79, "Warning 261: AI (02) position 14: Bad checksum '4', expected '1' (2D component)" },
        /* 47*/ { BARCODE_DBAR_LTD_CC, GS1NOCHECK_MODE, -1, -1, -1, "12345678901231", "[02]12345678901234", 0, 6, 79, "" },
        /* 48*/ { BARCODE_UPCA_CC, -1, -1, -1, -1, "12345678901", "[20]12", 0, 7, 99, "" },
        /* 49*/ { BARCODE_UPCA_CC, -1, -1, -1, -1, "123456789012", "[20]12", 0, 7, 99, "" },
        /* 50*/ { BARCODE_UPCA_CC, -1, -1, -1, -1, "123456789013", "[20]12", ZINT_ERROR_INVALID_CHECK, -1, -1, "Error 270: Invalid check digit '3', expecting '2' (linear component)" },
        /* 51*/ { BARCODE_UPCA_CC, GS1NOCHECK_MODE, -1, -1, -1, "123456789013", "[20]12", ZINT_ERROR_INVALID_CHECK, -1, -1, "Error 270: Invalid check digit '3', expecting '2' (linear component)" }, /* Linear component still checked */
        /* 52*/ { BARCODE_UPCA_CC, -1, -1, -1, -1, "1234567890123", "[20]12", ZINT_ERROR_TOO_LONG, -1, -1, "Error 289: Input length 13 wrong (11 or 12 only) (linear component)" },
        /* 53*/ { BARCODE_UPCA_CC, GS1NOCHECK_MODE, -1, -1, -1, "1234567890123", "[20]12", ZINT_ERROR_TOO_LONG, -1, -1, "Error 289: Input length 13 wrong (11 or 12 only) (linear component)" },
        /* 54*/ { BARCODE_UPCA_CC, -1, -1, -1, -1, "12345678901A", "[20]12", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 284: Invalid character at position 12 in input (digits and \"+\" only) (linear component)" },
        /* 55*/ { BARCODE_UPCA_CC, GS1NOCHECK_MODE, -1, -1, -1, "12345678901A", "[20]12", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 284: Invalid character at position 12 in input (digits and \"+\" only) (linear component)" },
        /* 56*/ { BARCODE_UPCA_CC, -1, -1, -1, -1, "123456789012", "[20]1A", ZINT_WARN_NONCOMPLIANT, 7, 99, "Warning 261: AI (20) position 2: Non-numeric character 'A' (2D component)" }, /* AI (20) should be 2 nos. */
        /* 57*/ { BARCODE_UPCA_CC, GS1NOCHECK_MODE, -1, -1, -1, "123456789012", "[20]1A", 0, 7, 99, "" },
        /* 58*/ { BARCODE_UPCA_CC, -1, -1, -1, -1, "123456789012", "[02]12345678901234", ZINT_WARN_NONCOMPLIANT, 7, 99, "Warning 261: AI (02) position 14: Bad checksum '4', expected '1' (2D component)" },
        /* 59*/ { BARCODE_UPCA_CC, GS1NOCHECK_MODE, -1, -1, -1, "123456789012", "[02]12345678901234", 0, 7, 99, "" },
        /* 60*/ { BARCODE_UPCE_CC, -1, -1, -1, -1, "123456", "[20]12", 0, 9, 55, "" },
        /* 61*/ { BARCODE_UPCE_CC, -1, -1, -1, -1, "1234567", "[20]12", 0, 9, 55, "" },
        /* 62*/ { BARCODE_UPCE_CC, -1, -1, -1, -1, "12345670", "[20]12", 0, 9, 55, "" }, /* Check digit can now be given for UPCE_CC, like UPCA_CC */
        /* 63*/ { BARCODE_UPCE_CC, -1, -1, -1, -1, "12345671", "[20]12", ZINT_ERROR_INVALID_CHECK, -1, -1, "Error 274: Invalid check digit '1', expecting '0' (linear component)" },
        /* 64*/ { BARCODE_UPCE_CC, GS1NOCHECK_MODE, -1, -1, -1, "12345671", "[20]12", ZINT_ERROR_INVALID_CHECK, -1, -1, "Error 274: Invalid check digit '1', expecting '0' (linear component)" }, /* Linear component still checked */
        /* 65*/ { BARCODE_UPCE_CC, -1, -1, -1, -1, "123456712", "[20]12", ZINT_ERROR_TOO_LONG, -1, -1, "Error 291: Input length 9 wrong (6, 7 or 8 only) (linear component)" },
        /* 66*/ { BARCODE_UPCE_CC, GS1NOCHECK_MODE, -1, -1, -1, "123456712", "[20]12", ZINT_ERROR_TOO_LONG, -1, -1, "Error 291: Input length 9 wrong (6, 7 or 8 only) (linear component)" },
        /* 67*/ { BARCODE_UPCE_CC, -1, -1, -1, -1, "1234567A", "[20]12", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 284: Invalid character at position 8 in input (digits and \"+\" only) (linear component)" },
        /* 68*/ { BARCODE_UPCE_CC, GS1NOCHECK_MODE, -1, -1, -1, "1234567A", "[20]12", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 284: Invalid character at position 8 in input (digits and \"+\" only) (linear component)" },
        /* 69*/ { BARCODE_UPCE_CC, -1, -1, -1, -1, "1234567", "[20]1A", ZINT_WARN_NONCOMPLIANT, 9, 55, "Warning 261: AI (20) position 2: Non-numeric character 'A' (2D component)" }, /* AI (20) should be 2 nos. */
        /* 70*/ { BARCODE_UPCE_CC, GS1NOCHECK_MODE, -1, -1, -1, "1234567", "[20]1A", 0, 9, 55, "" },
        /* 71*/ { BARCODE_UPCE_CC, -1, -1, -1, -1, "1234567", "[02]12345678901234", ZINT_WARN_NONCOMPLIANT, 9, 55, "Warning 261: AI (02) position 14: Bad checksum '4', expected '1' (2D component)" },
        /* 72*/ { BARCODE_UPCE_CC, GS1NOCHECK_MODE, -1, -1, -1, "1234567", "[02]12345678901234", 0, 9, 55, "" },
        /* 73*/ { BARCODE_DBAR_STK_CC, -1, -1, -1, -1, "1234567890123", "[20]12", 0, 9, 56, "" },
        /* 74*/ { BARCODE_DBAR_STK_CC, -1, -1, -1, -1, "12345678901231", "[20]12", 0, 9, 56, "" },
        /* 75*/ { BARCODE_DBAR_STK_CC, -1, -1, -1, -1, "12345678901232", "[20]12", ZINT_ERROR_INVALID_CHECK, -1, -1, "Error 388: Invalid check digit '2', expecting '1' (linear component)" },
        /* 76*/ { BARCODE_DBAR_STK_CC, GS1NOCHECK_MODE, -1, -1, -1, "12345678901232", "[20]12", ZINT_ERROR_INVALID_CHECK, -1, -1, "Error 388: Invalid check digit '2', expecting '1' (linear component)" }, /* Linear component still checked */
        /* 77*/ { BARCODE_DBAR_STK_CC, -1, -1, -1, -1, "123456789012323", "[20]12", ZINT_ERROR_TOO_LONG, -1, -1, "Error 380: Input length 15 too long (maximum 14) (linear component)" },
        /* 78*/ { BARCODE_DBAR_STK_CC, GS1NOCHECK_MODE, -1, -1, -1, "123456789012323", "[20]12", ZINT_ERROR_TOO_LONG, -1, -1, "Error 380: Input length 15 too long (maximum 14) (linear component)" },
        /* 79*/ { BARCODE_DBAR_STK_CC, -1, -1, -1, -1, "1234567890123A", "[20]12", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 381: Invalid character at position 14 in input (digits only) (linear component)" },
        /* 80*/ { BARCODE_DBAR_STK_CC, GS1NOCHECK_MODE, -1, -1, -1, "1234567890123A", "[20]12", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 381: Invalid character at position 14 in input (digits only) (linear component)" },
        /* 81*/ { BARCODE_DBAR_STK_CC, -1, -1, -1, -1, "12345678901231", "[20]1A", ZINT_WARN_NONCOMPLIANT, 9, 56, "Warning 261: AI (20) position 2: Non-numeric character 'A' (2D component)" }, /* AI (20) should be 2 nos. */
        /* 82*/ { BARCODE_DBAR_STK_CC, GS1NOCHECK_MODE, -1, -1, -1, "12345678901231", "[20]1A", 0, 9, 56, "" },
        /* 83*/ { BARCODE_DBAR_STK_CC, -1, -1, -1, -1, "12345678901231", "[02]12345678901234", ZINT_WARN_NONCOMPLIANT, 9, 56, "Warning 261: AI (02) position 14: Bad checksum '4', expected '1' (2D component)" },
        /* 84*/ { BARCODE_DBAR_STK_CC, GS1NOCHECK_MODE, -1, -1, -1, "12345678901231", "[02]12345678901234", 0, 9, 56, "" },
        /* 85*/ { BARCODE_DBAR_OMNSTK_CC, -1, -1, -1, -1, "1234567890123", "[20]12", 0, 11, 56, "" },
        /* 86*/ { BARCODE_DBAR_OMNSTK_CC, -1, -1, -1, -1, "12345678901231", "[20]12", 0, 11, 56, "" },
        /* 87*/ { BARCODE_DBAR_OMNSTK_CC, -1, -1, -1, -1, "12345678901232", "[20]12", ZINT_ERROR_INVALID_CHECK, -1, -1, "Error 388: Invalid check digit '2', expecting '1' (linear component)" },
        /* 88*/ { BARCODE_DBAR_OMNSTK_CC, GS1NOCHECK_MODE, -1, -1, -1, "12345678901232", "[20]12", ZINT_ERROR_INVALID_CHECK, -1, -1, "Error 388: Invalid check digit '2', expecting '1' (linear component)" }, /* Linear component still checked */
        /* 89*/ { BARCODE_DBAR_OMNSTK_CC, -1, -1, -1, -1, "123456789012312", "[20]12", ZINT_ERROR_TOO_LONG, -1, -1, "Error 380: Input length 15 too long (maximum 14) (linear component)" },
        /* 90*/ { BARCODE_DBAR_OMNSTK_CC, GS1NOCHECK_MODE, -1, -1, -1, "123456789012312", "[20]12", ZINT_ERROR_TOO_LONG, -1, -1, "Error 380: Input length 15 too long (maximum 14) (linear component)" },
        /* 91*/ { BARCODE_DBAR_OMNSTK_CC, -1, -1, -1, -1, "1234567890123A", "[20]12", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 381: Invalid character at position 14 in input (digits only) (linear component)" },
        /* 92*/ { BARCODE_DBAR_OMNSTK_CC, GS1NOCHECK_MODE, -1, -1, -1, "1234567890123A", "[20]12", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 381: Invalid character at position 14 in input (digits only) (linear component)" },
        /* 93*/ { BARCODE_DBAR_OMNSTK_CC, -1, -1, -1, -1, "12345678901231", "[20]1A", ZINT_WARN_NONCOMPLIANT, 11, 56, "Warning 261: AI (20) position 2: Non-numeric character 'A' (2D component)" }, /* AI (20) should be 2 nos. */
        /* 94*/ { BARCODE_DBAR_OMNSTK_CC, GS1NOCHECK_MODE, -1, -1, -1, "12345678901231", "[20]1A", 0, 11, 56, "" },
        /* 95*/ { BARCODE_DBAR_OMNSTK_CC, -1, -1, -1, -1, "12345678901231", "[02]12345678901234", ZINT_WARN_NONCOMPLIANT, 11, 56, "Warning 261: AI (02) position 14: Bad checksum '4', expected '1' (2D component)" },
        /* 96*/ { BARCODE_DBAR_OMNSTK_CC, GS1NOCHECK_MODE, -1, -1, -1, "12345678901231", "[02]12345678901234", 0, 11, 56, "" },
        /* 97*/ { BARCODE_GS1_128_CC, -1, -1, -1, -1, "", "[20]12", ZINT_ERROR_INVALID_OPTION, -1, -1, "Error 445: No primary (linear) message" },
        /* 98*/ { BARCODE_GS1_128_CC, -1, -1, -1, -1, "[01]12345678901231", "[91]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[92]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[93]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[94]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[95]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[96]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[97]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[98]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[99]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890", 0, 31, 273, "" }, /* Tries CC-A then CC-B then CC-C - ensure errtxt empty */
        /* 99*/ { BARCODE_GS1_128_CC, -1, -1, -1, -1, "[01]12345678901231[90]123456789012345678901234567890[91]1234567890123456789012345678901234567890123456789012345678901234", "[91]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[92]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[93]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[94]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[95]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[96]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[97]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[98]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[99]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[91]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[92]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[93]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[94]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[95]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[96]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[97]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[98]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[99]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[91]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[92]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[93]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890[94]123456789012345678901234567890123456789012345678901234567890123", ZINT_WARN_NONCOMPLIANT, 29, 702, "Warning 843: Input too long, requires 115 characters (maximum 48) (linear component)" }, /* Overlarge linear and CC-C input */
        /*100*/ { BARCODE_GS1_128_CC, -1, -1,  -1, -1, "[01]12345678901231[90]123456789012345678901234567890[91]1234567890123456789012345678901234567890123456789012345678901234", "[91]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[92]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[93]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[94]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[95]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[96]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[97]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[98]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[99]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[91]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[92]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[93]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[94]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[95]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[96]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL", ZINT_ERROR_TOO_LONG, -1, -1, "Error 442: Input too long (2D component)" }, /* Overlarge linear and oversized CC-C input */
        /*101*/ { BARCODE_GS1_128_CC, -1, -1, -1, -1, "[01]12345678901231", "[91]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[92]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[93]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[94]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[95]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[96]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[97]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[98]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[99]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[91]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[92]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[93]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[94]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[95]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[96]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[91]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[92]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[93]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[91]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[92]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[93]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[93]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHI" "JKLMNOPQRSTUVWXYZABCDEFGHIJKL[93]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[93]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[93]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[93]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[93]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[93]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[93]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[93]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[93]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[93]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKL[93]ABCDEFGHIJKLM", ZINT_ERROR_TOO_LONG, -1, -1, "Error 446: 2D component input too long, requires 2991 characters (maximum 2990)" }, /* Reduced length 2291 */
        /*102*/ { BARCODE_GS1_128_CC, -1, -1, -1, -1, "[01]12345678901231", "[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012" "345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[00]123456789012345675[11]121212[20]12", 0, 32, 579, "" }, /* Reduced length 2372 digits (no FNC1s) with not recommended ECC 4 > 2361 digit limit given in ISO/IEC 24723:2010 4.1 (d)(2)(iii) */
        /*103*/ { BARCODE_GS1_128_CC, -1, -1, -1, -1, "[90]ABCDEFGHIJKLMNOPQRSTUVWXYXABCD[91]ABCDEFGHIJKLMNOPQRSTUVWXYABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLM", "[20]12", ZINT_WARN_NONCOMPLIANT, 5, 1146, "Warning 843: Input too long, requires 99 characters (maximum 48) (linear component)" },
        /*104*/ { BARCODE_GS1_128_CC, -1, -1, -1, -1, "[90]ABCDEFGHIJKLMNOPQRSTUVWXYXABCD[91]ABCDEFGHIJKLMNOPQRSTUVWXYABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMN", "[20]12", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 344: Input too long, requires 103 symbol characters (maximum 102) (linear component)" },
        /*105*/ { BARCODE_DBAR_EXPSTK_CC, -1, -1, -1, -1, "[91]1234567890123456789012345678901234", "[20]12", 0, 13, 102, "" },
        /*106*/ { BARCODE_DBAR_EXPSTK_CC, -1, -1, 1, -1, "[91]1234567890123456789012345678901234", "[20]12", 0, 13, 102, "" },
        /*107*/ { BARCODE_DBAR_EXPSTK_CC, -1, -1, 2, -1, "[91]1234567890123456789012345678901234", "[20]12", 0, 13, 102, "" },
        /*108*/ { BARCODE_DBAR_EXPSTK_CC, -1, -1, 3, -1, "[91]1234567890123456789012345678901234", "[20]12", 0, 9, 151, "" },
        /*109*/ { BARCODE_DBAR_EXPSTK_CC, -1, -1, 4, -1, "[91]1234567890123456789012345678901234", "[20]12", 0, 9, 200, "" },
        /*110*/ { BARCODE_DBAR_EXPSTK_CC, -1, -1, -1, 2, "[91]1234567890123456789012345678901234", "[20]12", 0, 9, 151, "" },
        /*111*/ { BARCODE_DBAR_EXPSTK_CC, -1, -1, -1, 3, "[91]1234567890123456789012345678901234", "[20]12", 0, 13, 102, "" },
        /*112*/ { BARCODE_DBAR_EXPSTK_CC, -1, -1, -1, 4, "[91]1234567890123456789012345678901234", "[20]12", 0, 13, 102, "" },
    };
    const int data_size = ARRAY_SIZE(data);
    int i, length, composite_length, ret;
    struct zint_symbol *symbol = NULL;

    testStartSymbol("test_input", &symbol);

    for (i = 0; i < data_size; i++) {

        if (testContinue(p_ctx, i)) continue;

        symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        length = testUtilSetSymbol(symbol, data[i].symbology, data[i].input_mode, -1 /*eci*/, data[i].option_1, data[i].option_2, data[i].option_3, -1 /*output_options*/, data[i].data, -1, debug);
        assert_zero(length >= 128, "i:%d length %d >= 128\n", i, length);
        strcpy(symbol->primary, data[i].data);

        composite_length = (int) strlen(data[i].composite);

        ret = ZBarcode_Encode(symbol, (const unsigned char *) data[i].composite, composite_length);
        assert_equal(ret, data[i].ret, "i:%d ZBarcode_Encode ret %d != %d (%s)\n", i, data[i].ret, ret, symbol->errtxt);

        assert_zero(strcmp(symbol->errtxt, data[i].expected_errtxt), "i:%d strcmp(%s, %s) != 0\n", i, symbol->errtxt, data[i].expected_errtxt);

        if (ret < ZINT_ERROR) {
            assert_equal(symbol->rows, data[i].expected_rows, "i:%d symbol->rows %d != %d (symbol->width %d)\n", i, symbol->rows, data[i].expected_rows, symbol->width);
            assert_equal(symbol->width, data[i].expected_width, "i:%d symbol->width %d != %d\n", i, symbol->width, data[i].expected_width);
        }

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

/* #181 Christian Hartlage OSS-Fuzz and #300 Andre Maute */
static void test_fuzz(const testCtx *const p_ctx) {
    int debug = p_ctx->debug;

    struct item {
        int symbology;
        int input_mode;
        int option_1;
        char *data;
        int length;
        char *composite;
        int ret;
        int bwipp_cmp;
        char *expected_errtxt;
        char *comment;
    };
    /* s/\/\*[ 0-9]*\*\//\=printf("\/\*%3d*\/", line(".") - line("'<")): */
    static const struct item data[] = {
        /* 0*/ { BARCODE_EANX_CC, -1, -1, "+123456789012345678", -1, "[21]A12345678", ZINT_ERROR_TOO_LONG, 1, "Error 297: Input add-on length 18 too long (maximum 5) (linear component)", "" },
        /* 1*/ { BARCODE_UPCA_CC, -1, -1, "+123456789012345678", -1, "[21]A12345678", ZINT_ERROR_TOO_LONG, 1, "Error 297: Input add-on length 18 too long (maximum 5) (linear component)", "" },
        /* 2*/ { BARCODE_UPCE_CC, -1, -1, "+123456789012345678", -1, "[21]A12345678", ZINT_ERROR_TOO_LONG, 1, "Error 297: Input add-on length 18 too long (maximum 5) (linear component)", "" },
        /* 3*/ { BARCODE_EANX_CC, -1, -1, "+12345", -1, "[21]A12345678", 0 , 0, "", "BWIPP checks for proper EAN data" },
        /* 4*/ { BARCODE_EANX_CC, -1, -1, "+123456", -1, "[21]A12345678", ZINT_ERROR_TOO_LONG, 1, "Error 297: Input add-on length 6 too long (maximum 5) (linear component)", "" },
        /* 5*/ { BARCODE_EANX_CC, GS1PARENS_MODE | GS1NOCHECK_MODE, -1, "kks", -1, "()111%", ZINT_ERROR_INVALID_DATA, 1, "Error 284: Invalid character at position 1 in input (digits and \"+\" only) (linear component)", "" }, /* #300 (#5), Andre Maute (`dbar_date()` not checking length + other non-checks) */
        /* 6*/ { BARCODE_UPCA_CC, GS1PARENS_MODE | GS1NOCHECK_MODE, -1, "\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\225\215\153\153\153\153\153\153\263\153\153\153\153\153\153\153\153\153\153\163", -1, "()90", ZINT_ERROR_TOO_LONG, 1, "Error 283: Input length 46 too long (maximum 19) (linear component)", "" }, /* #300 (#6), Andre Maute (`dbar_date()` not checking length + other non-checks) */
        /* 7*/ { BARCODE_UPCA_CC, GS1PARENS_MODE | GS1NOCHECK_MODE, -1, "\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\225\215\153\153\153\153\153\153\263\153\153\377\002\000\000\153\153\153\153\163\000\000\000\153\153\153\153\153\153\153\060\047\047\043\047\057\153\153\153\153\153\000\000\000\000\153\153\153\161\153\153\153\153\153\153\153\153\153\153\153\153\153\167\167\167\167\167\167\167\167\167\167\167\167\167\167\167\167\001\100\000\000\000\000\000\000\000\167\167\167\167\167\167\167\167\167\167\167\167\167\167", 127, "()904OOOOO)CK0336680OOOOOOOOOOOOOO29[0kkkk%%%%(", ZINT_ERROR_INVALID_DATA, 1, "Error 253: Malformed AI in input (brackets don't match) (2D component)", "" }, /* #300 (#11), Andre Maute (`gs1_verify()` not checking length on resolve AI data loop) */
        /* 8*/ { BARCODE_EANX_CC, GS1PARENS_MODE | GS1NOCHECK_MODE, -1, "\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\225\215\153\153\153\153\153\153\263\153\153\377\002\000\000\153\153\153\153\163\000\000\000\153\153\153\153\153\153\153\060\047\047\043\047\057\153\153\153\153\153\000\000\000\000\153\153\153\161\153\153\153\153\153\153\153\153\153\153\153\153\153\167\167\167\167\167\167\167\167\167\167\167\167\167\167\167\167\001\100\000\000\000\000\000\000\000\167\167\167\167\167\167\167\167\167\167\167\167\167\167", 127, "()904OOOOO)CK0336680OOOOOOOOOOOOOO29[0kkkk%%%%(", ZINT_ERROR_INVALID_DATA, 1, "Error 253: Malformed AI in input (brackets don't match) (2D component)", "" }, /* #300 (#11 with EANX_CC) */
        /* 9*/ { BARCODE_GS1_128_CC, GS1NOCHECK_MODE, 3, "[]28", -1, "[]RRR___________________KKKRRR0000", 0, 1, "", "" }, /* #300 (#13), Andre Maute (`calc_padding_ccc()` dividing by zero when linear width == 68) */
        /*10*/ { BARCODE_GS1_128_CC, GS1NOCHECK_MODE, 3, "[]2", -1, "[]RRR___________________KKKRRR0000", 0, 1, "", "" }, /* #300 (#13 shortened to min linear input (but same linear width 68)) */
    };
    const int data_size = ARRAY_SIZE(data);
    int i, length, composite_length, ret;
    struct zint_symbol *symbol = NULL;

    char bwipp_buf[32768];
    char bwipp_msg[1024];

    int do_bwipp = (debug & ZINT_DEBUG_TEST_BWIPP) && testUtilHaveGhostscript(); /* Only do BWIPP test if asked, too slow otherwise */

    testStartSymbol("test_fuzz", &symbol);

    for (i = 0; i < data_size; i++) {

        if (testContinue(p_ctx, i)) continue;

        symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        length = testUtilSetSymbol(symbol, data[i].symbology, data[i].input_mode, -1 /*eci*/, data[i].option_1, -1, -1, -1 /*output_options*/, data[i].data, -1, debug);
        assert_zero(length >= 128, "i:%d length %d >= 128\n", i, length);
        strcpy(symbol->primary, data[i].data);

        composite_length = (int) strlen(data[i].composite);

        ret = ZBarcode_Encode(symbol, (const unsigned char *) data[i].composite, composite_length);
        assert_equal(ret, data[i].ret, "i:%d ZBarcode_Encode ret %d != %d (%s)\n", i, ret, data[i].ret, symbol->errtxt);
        assert_zero(strcmp(symbol->errtxt, data[i].expected_errtxt), "i:%d strcmp(%s, %s) != 0\n",
                        i, symbol->errtxt, data[i].expected_errtxt);

        if (ret < ZINT_ERROR) {
            if (do_bwipp && testUtilCanBwipp(i, symbol, data[i].option_1, -1, -1, debug)) {
                if (!data[i].bwipp_cmp) {
                    if (debug & ZINT_DEBUG_TEST_PRINT) printf("i:%d %s not BWIPP compatible (%s)\n", i, testUtilBarcodeName(symbol->symbology), data[i].comment);
                } else {
                    char modules_dump[32768];
                    assert_notequal(testUtilModulesDump(symbol, modules_dump, sizeof(modules_dump)), -1, "i:%d testUtilModulesDump == -1\n", i);
                    ret = testUtilBwipp(i, symbol, data[i].option_1, -1, -1, data[i].composite, composite_length, symbol->primary, bwipp_buf, sizeof(bwipp_buf), NULL);
                    assert_zero(ret, "i:%d %s testUtilBwipp ret %d != 0\n", i, testUtilBarcodeName(symbol->symbology), ret);

                    ret = testUtilBwippCmp(symbol, bwipp_msg, bwipp_buf, modules_dump);
                    assert_zero(ret, "i:%d %s testUtilBwippCmp %d != 0 %s\n  actual: %s\nexpected: %s\n",
                                   i, testUtilBarcodeName(symbol->symbology), ret, bwipp_msg, bwipp_buf, modules_dump);
                }
            }
        }

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

#include <time.h>

#define TEST_PERF_ITERATIONS    1000

/* Not a real test, just performance indicator */
static void test_perf(const testCtx *const p_ctx) {
    int debug = p_ctx->debug;

    struct item {
        int symbology;
        int option_1;
        char *data;
        char *composite;
        int ret;

        int expected_rows;
        int expected_width;
        char *comment;
    };
    static const struct item data[] = {
        /*  0*/ { BARCODE_EANX_CC, 1, "123456789012",
                    "[91]123456789012345678901234567890123456789012345678901234",
                    0, 11, 99, "58 chars CC-A" },
        /*  1*/ { BARCODE_UPCA_CC, 2, "12345678901",
                    "[91]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "[92]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "[93]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "[94]12345678901234567890123456789012345678901234567890",
                    0, 48, 99, "336 chars CC-B" },
        /*  2*/ { BARCODE_GS1_128_CC, 3, "[01]12345678901231",
                    "[91]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "[92]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "[93]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "[94]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "[95]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "[96]123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890",
                    0, 32, 205, "564 chars CC-C" },
    };
    const int data_size = ARRAY_SIZE(data);
    int i, length, composite_length, ret;

    clock_t start, total_encode = 0, total_buffer = 0, diff_encode, diff_buffer;

    if (!(debug & ZINT_DEBUG_TEST_PERFORMANCE)) { /* -d 256 */
        return;
    }

    for (i = 0; i < data_size; i++) {
        int j;

        if (testContinue(p_ctx, i)) continue;

        diff_encode = diff_buffer = 0;

        for (j = 0; j < TEST_PERF_ITERATIONS; j++) {
            struct zint_symbol *symbol = ZBarcode_Create();
            assert_nonnull(symbol, "Symbol not created\n");

            length = testUtilSetSymbol(symbol, data[i].symbology, -1 /*input_mode*/, -1 /*eci*/, data[i].option_1, -1, -1, -1 /*output_options*/, data[i].data, -1, debug);
            assert_zero(length >= 128, "i:%d length %d >= 128\n", i, length);
            strcpy(symbol->primary, data[i].data);

            composite_length = (int) strlen(data[i].composite);

            start = clock();
            ret = ZBarcode_Encode(symbol, (const unsigned char *) data[i].composite, composite_length);
            diff_encode += clock() - start;
            assert_equal(ret, data[i].ret, "i:%d ZBarcode_Encode ret %d != %d (%s)\n", i, ret, data[i].ret, symbol->errtxt);

            assert_equal(symbol->rows, data[i].expected_rows, "i:%d symbol->rows %d != %d (%s)\n", i, symbol->rows, data[i].expected_rows, data[i].data);
            assert_equal(symbol->width, data[i].expected_width, "i:%d symbol->width %d != %d (%s)\n", i, symbol->width, data[i].expected_width, data[i].data);

            start = clock();
            ret = ZBarcode_Buffer(symbol, 0 /*rotate_angle*/);
            diff_buffer += clock() - start;
            assert_zero(ret, "i:%d ZBarcode_Buffer ret %d != 0 (%s)\n", i, ret, symbol->errtxt);

            ZBarcode_Delete(symbol);
        }

        printf("%s: diff_encode %gms, diff_buffer %gms\n", data[i].comment, diff_encode * 1000.0 / CLOCKS_PER_SEC, diff_buffer * 1000.0 / CLOCKS_PER_SEC);

        total_encode += diff_encode;
        total_buffer += diff_buffer;
    }
    if (p_ctx->index != -1) {
        printf("totals: encode %gms, buffer %gms\n", total_encode * 1000.0 / CLOCKS_PER_SEC, total_buffer * 1000.0 / CLOCKS_PER_SEC);
    }
}

int main(int argc, char *argv[]) {

    testFunction funcs[] = { /* name, func */
        { "test_eanx_leading_zeroes", test_eanx_leading_zeroes },
        { "test_examples", test_examples },
        { "test_odd_numbered_numeric", test_odd_numbered_numeric },
        { "test_ean128_cc_shift", test_ean128_cc_shift },
        { "test_ean128_cc_width", test_ean128_cc_width },
        { "test_encodation_0", test_encodation_0 },
        { "test_encodation_10", test_encodation_10 },
        { "test_encodation_11", test_encodation_11 },
        { "test_addongap", test_addongap },
        { "test_gs1parens", test_gs1parens },
        { "test_hrt", test_hrt },
        { "test_input", test_input },
        { "test_fuzz", test_fuzz },
        { "test_perf", test_perf },
    };

    testRun(argc, argv, funcs, ARRAY_SIZE(funcs));

    testReport();

    return 0;
}

/* vim: set ts=4 sw=4 et : */
