/**************************************************************************
 * 
 * Copyright 2003 VMware, Inc.
 * All Rights Reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sub license, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT.
 * IN NO EVENT SHALL VMWARE AND/OR ITS SUPPLIERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 **************************************************************************/

#ifndef ST_CONTEXT_H
#define ST_CONTEXT_H

#include "main/mtypes.h"
#include "pipe/p_state.h"
#include "state_tracker/st_api.h"
#include "main/fbobject.h"

struct bitmap_cache;
struct dd_function_table;
struct draw_context;
struct draw_stage;
struct gen_mipmap_state;
struct st_context;
struct st_fragment_program;
struct u_upload_mgr;


#define ST_NEW_MESA                    (1 << 0) /* Mesa state has changed */
#define ST_NEW_FRAGMENT_PROGRAM        (1 << 1)
#define ST_NEW_VERTEX_PROGRAM          (1 << 2)
#define ST_NEW_FRAMEBUFFER             (1 << 3)
/* gap, re-use it */
#define ST_NEW_GEOMETRY_PROGRAM        (1 << 5)
#define ST_NEW_VERTEX_ARRAYS           (1 << 6)
#define ST_NEW_RASTERIZER              (1 << 7)
#define ST_NEW_UNIFORM_BUFFER          (1 << 8)


struct st_state_flags {
   GLuint mesa;
   GLuint st;
};

struct st_tracked_state {
   const char *name;
   struct st_state_flags dirty;
   void (*update)( struct st_context *st );
};



struct st_context
{
   struct st_context_iface iface;

   struct gl_context *ctx;

   struct pipe_context *pipe;

   struct u_upload_mgr *uploader, *indexbuf_uploader, *constbuf_uploader;

   struct draw_context *draw;  /**< For selection/feedback/rastpos only */
   struct draw_stage *feedback_stage;  /**< For GL_FEEDBACK rendermode */
   struct draw_stage *selection_stage;  /**< For GL_SELECT rendermode */
   struct draw_stage *rastpos_stage;  /**< For glRasterPos */
   GLboolean clamp_frag_color_in_shader;
   GLboolean clamp_vert_color_in_shader;
   boolean has_stencil_export; /**< can do shader stencil export? */
   boolean has_time_elapsed;
   boolean has_shader_model3;
   boolean prefer_blit_based_texture_transfer;

   boolean needs_texcoord_semantic;
   boolean apply_texture_swizzle_to_border_color;

   /* On old libGL's for linux we need to invalidate the drawables
    * on glViewpport calls, this is set via a option.
    */
   boolean invalidate_on_gl_viewport;

   boolean vertex_array_out_of_memory;

   /* Some state is contained in constant objects.
    * Other state is just parameter values.
    */
   struct {
      struct pipe_blend_state               blend;
      struct pipe_depth_stencil_alpha_state depth_stencil;
      struct pipe_rasterizer_state          rasterizer;
      struct pipe_sampler_state samplers[PIPE_SHADER_TYPES][PIPE_MAX_SAMPLERS];
      GLuint num_samplers[PIPE_SHADER_TYPES];
      struct pipe_sampler_view *sampler_views[PIPE_SHADER_TYPES][PIPE_MAX_SAMPLERS];
      GLuint num_sampler_views[PIPE_SHADER_TYPES];
      struct pipe_clip_state clip;
      struct {
         void *ptr;
         unsigned size;
      } constants[PIPE_SHADER_TYPES];
      struct pipe_framebuffer_state framebuffer;
      struct pipe_scissor_state scissor[PIPE_MAX_VIEWPORTS];
      struct pipe_viewport_state viewport[PIPE_MAX_VIEWPORTS];
      unsigned sample_mask;

      GLuint poly_stipple[32];  /**< In OpenGL's bottom-to-top order */

      GLuint fb_orientation;
   } state;

   char vendor[100];
   char renderer[100];

   struct st_state_flags dirty;

   GLboolean missing_textures;
   GLboolean vertdata_edgeflags;
   GLboolean edgeflag_culls_prims;

   /** Mapping from VARYING_SLOT_x to post-transformed vertex slot */
   const GLuint *vertex_result_to_slot;

   struct st_vertex_program *vp;    /**< Currently bound vertex program */
   struct st_fragment_program *fp;  /**< Currently bound fragment program */
   struct st_geometry_program *gp;  /**< Currently bound geometry program */

   struct st_vp_variant *vp_variant;
   struct st_fp_variant *fp_variant;
   struct st_gp_variant *gp_variant;

   struct gl_texture_object *default_texture;

   struct {
      struct gl_program_cache *cache;
      struct st_fragment_program *program;  /**< cur pixel transfer prog */
      GLuint xfer_prog_sn;  /**< pixel xfer program serial no. */
      GLuint user_prog_sn;  /**< user fragment program serial no. */
      struct st_fragment_program *combined_prog;
      GLuint combined_prog_sn;
      struct pipe_resource *pixelmap_texture;
      struct pipe_sampler_view *pixelmap_sampler_view;
      boolean pixelmap_enabled;  /**< use the pixelmap texture? */
   } pixel_xfer;

   /** for glBitmap */
   struct {
      struct pipe_rasterizer_state rasterizer;
      struct pipe_sampler_state samplers[2];
      enum pipe_format tex_format;
      void *vs;
      struct bitmap_cache *cache;
   } bitmap;

   /** for glDraw/CopyPixels */
   struct {
      struct gl_fragment_program *shaders[4];
      void *vert_shaders[2];   /**< ureg shaders */
   } drawpix;

   /** for glClear */
   struct {
      struct pipe_rasterizer_state raster;
      struct pipe_viewport_state viewport;
      void *vs;
      void *fs;
      void *vs_layered;
   } clear;

   /** used for anything using util_draw_vertex_buffer */
   struct pipe_vertex_element velems_util_draw[3];

   void *passthrough_fs;  /**< simple pass-through frag shader */

   enum pipe_texture_target internal_target;

   struct cso_context *cso_context;

   void *winsys_drawable_handle;

   /* The number of vertex buffers from the last call of validate_arrays. */
   unsigned last_num_vbuffers;

   int32_t draw_stamp;
   int32_t read_stamp;

   struct st_config_options options;
};


/* Need this so that we can implement Mesa callbacks in this module.
 */
static INLINE struct st_context *st_context(struct gl_context *ctx)
{
   return ctx->st;
}


/**
 * Wrapper for struct gl_framebuffer.
 * This is an opaque type to the outside world.
 */
struct st_framebuffer
{
   struct gl_framebuffer Base;
   void *Private;

   struct st_framebuffer_iface *iface;
   enum st_attachment_type statts[ST_ATTACHMENT_COUNT];
   unsigned num_statts;
   int32_t stamp;
   int32_t iface_stamp;
};


extern void st_init_driver_functions(struct dd_function_table *functions);

void st_invalidate_state(struct gl_context * ctx, GLuint new_state);



#define Y_0_TOP 1
#define Y_0_BOTTOM 2

static INLINE GLuint
st_fb_orientation(const struct gl_framebuffer *fb)
{
   if (fb && _mesa_is_winsys_fbo(fb)) {
      /* Drawing into a window (on-screen buffer).
       *
       * Negate Y scale to flip image vertically.
       * The NDC Y coords prior to viewport transformation are in the range
       * [y=-1=bottom, y=1=top]
       * Hardware window coords are in the range [y=0=top, y=H-1=bottom] where
       * H is the window height.
       * Use the viewport transformation to invert Y.
       */
      return Y_0_TOP;
   }
   else {
      /* Drawing into user-created FBO (very likely a texture).
       *
       * For textures, T=0=Bottom, so by extension Y=0=Bottom for rendering.
       */
      return Y_0_BOTTOM;
   }
}


/** clear-alloc a struct-sized object, with casting */
#define ST_CALLOC_STRUCT(T)   (struct T *) calloc(1, sizeof(struct T))


extern struct st_context *
st_create_context(gl_api api, struct pipe_context *pipe,
                  const struct gl_config *visual,
                  struct st_context *share,
                  const struct st_config_options *options);

extern void
st_destroy_context(struct st_context *st);


#endif
