/*
  This file is part of KOrganizer.

  Copyright (c) 2001 Cornelius Schumacher <schumacher@kde.org>
  Copyright (C) 2003-2004 Reinhold Kainhofer <reinhold@kainhofer.com>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

  As a special exception, permission is given to link this program
  with any edition of TQt, and distribute the resulting executable,
  without including the source code for TQt in the source distribution.
*/
#ifndef KOVIEWMANAGER_H
#define KOVIEWMANAGER_H

#include <tqobject.h>
class TQWidget;
class TQTabWidget;

class CalendarView;

class KOListView;
class KOAgendaView;
class KOMonthView;
class KOTodoView;
class KOWhatsNextView;
class KOJournalView;
class KOTimelineView;

namespace KOrg {
  class BaseView;
  class MultiAgendaView;
}
using namespace KCal;

/**
  This class manages the views of the calendar. It owns the objects and handles
  creation and selection.
*/
class KOViewManager : public TQObject
{
    Q_OBJECT
  TQ_OBJECT
  public:

    enum AgendaMode {
      AGENDA_NONE,
      AGENDA_DAY,
      AGENDA_WORK_WEEK,
      AGENDA_WEEK,
      AGENDA_NEXTX,
      AGENDA_OTHER // for example, showing 8 days
    };

    KOViewManager( CalendarView * );
    virtual ~KOViewManager();

    /** changes the view to be the currently selected view */
    void showView( KOrg::BaseView * );

    void readSettings( KConfig *config );
    void writeSettings( KConfig *config );

    /** Read which view was shown last from config file */
    void readCurrentView( KConfig * );
    /** Write which view is currently shown to config file */
    void writeCurrentView( KConfig * );

    KOrg::BaseView *currentView();

    void setDocumentId( const TQString & );

    void updateView();
    void updateView( const TQDate &start, const TQDate &end );

    void goMenu( bool enable );
    void raiseCurrentView();

    void connectView( KOrg::BaseView * );
    void addView( KOrg::BaseView * );

    Incidence *currentSelection();
    TQDate currentSelectionDate();

    KOAgendaView *agendaView() const { return mAgendaView; }
    KOrg::MultiAgendaView *multiAgendaView() const { return mAgendaSideBySideView; }
    KOTodoView *todoView() const { return mTodoView; }
    KOMonthView *monthView() const { return mMonthView; }

    void updateMultiCalendarDisplay();

    /*
    * Returns true if the agenda is the current view.
    *
    * Never use the pointer returned by agendaView()
    * to know if agenda is selected, because agenda has other modes
    * (tabbed, side by side). Use this function instead.
    */
    bool agendaIsSelected() const;

    /**
      If the agenda view is selected it returns the current range mode:
      week, work week, day or nextX days
    */
    AgendaMode agendaMode() const { return mAgendaMode; }

  public slots:
    void showWhatsNextView();
    void showListView();
    void showAgendaView();
    void showDayView();
    void showWorkWeekView();
    void showWeekView();
    void showNextXView();
    void showMonthView();
    void showTodoView();
    void showTimelineView();
    void showJournalView();

    void showEventView();

    void connectTodoView( KOTodoView *todoView );

    void zoomInHorizontally();
    void zoomOutHorizontally();
    void zoomInVertically();
    void zoomOutVertically();

    void resourcesChanged();

  private slots:
    void currentAgendaViewTabChanged( TQWidget* );
  private:
    TQWidget* widgetForView( KOrg::BaseView* ) const;
    CalendarView *mMainView;

    KOAgendaView    *mAgendaView;
    MultiAgendaView *mAgendaSideBySideView;
    KOListView      *mListView;
    KOMonthView     *mMonthView;
    KOTodoView      *mTodoView;
    KOWhatsNextView *mWhatsNextView;
    KOJournalView   *mJournalView;
    KOTimelineView  *mTimelineView;

    KOrg::BaseView *mCurrentView;

    KOrg::BaseView *mLastEventView;
    TQTabWidget *mAgendaViewTabs;
    int mAgendaViewTabIndex;

    AgendaMode mAgendaMode;

};

#endif
