// -*- C++ -*-
/*
   Copyright (c) 2000 Stefan Schimanski (1Stein@gmx.de)
                 1999-2000 Christian Esken (esken@kde.org)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#ifndef KaimanStyle_H
#define KaimanStyle_H

#include <tqstring.h>
#include <tqwidget.h>
#include <tqpixmap.h>
#include <tqbitmap.h>
#include <tqimage.h>
#include <tqevent.h>
#include <tqptrvector.h>

class KaimanStyleElement : public TQWidget
{
    Q_OBJECT
  TQ_OBJECT
public:
    KaimanStyleElement(TQWidget *parent, const char *name=0);
    ~KaimanStyleElement();

    virtual void loadPixmaps(TQString &val_s_filename);

    TQString     element;
    TQString     filename;
    TQPoint      upperLeft;
    TQSize      dimension;

    bool options[3];
    int digits;

    bool optionPrelight;
    bool optionStatuslight;

    int pixmapLines;
    int pixmapColumns;

    TQPtrVector<TQPixmap> pixmaps;

public slots:
    void setPixmap( int num );

protected:
    void paintEvent(TQPaintEvent *qpe);
    void dropEvent( TQDropEvent *event );
    void dragEnterEvent( TQDragEnterEvent *event );

    int pixmapNum;

private:
    int _currentPixmap;
};


class KaimanStyleMasked : public KaimanStyleElement
{
 Q_OBJECT
  TQ_OBJECT
public:
    KaimanStyleMasked(TQWidget *parent, const char *name=0)
        : KaimanStyleElement( parent, name ) {};

    virtual void loadPixmaps(TQString &val_s_filename)
        {
            KaimanStyleElement::loadPixmaps( val_s_filename );
            if(pixmaps[0]->mask())
                setMask(*pixmaps[0]->mask());
        };
};


class KaimanStyleButton : public KaimanStyleMasked
{
 Q_OBJECT
  TQ_OBJECT
public:
    KaimanStyleButton(TQWidget *parent, const char *name=0);
    ~KaimanStyleButton();

    // Button states.
    enum { NormalUp=0, NormalDown, LitUp, LitDown, PrelightUp, PrelightLitUp, StateListEND };

    TQPtrVector<int> I_pmIndex;

    void setLit(bool);
    void setPrelight(bool);
    void setDown(bool);
    bool lit();
    bool prelit();
    bool down();
    void updateButtonState();

signals:
    void clicked();

protected:
    void mousePressEvent(TQMouseEvent *qme);
    void mouseReleaseEvent(TQMouseEvent *qme);
    void enterEvent(TQEvent * );
    void leaveEvent ( TQEvent * );

private:
    int i_i_currentState;
    bool i_b_lit;
    bool i_b_prelit;
    bool i_b_down;
};


class KaimanStyleSlider : public KaimanStyleMasked
{
 Q_OBJECT
  TQ_OBJECT
public:
    KaimanStyleSlider(int min, int max, TQWidget *parent, const char *name=0);
    ~KaimanStyleSlider();

    int value() { return _value; };

    static const bool optionVertical;
    static const bool optionReversed;

public slots:
    void setValue( int value );
    void setValue( int value, int min, int max );

signals:
    void newValue( int value );
    void newValueDrag( int value );
    void newValueDrop( int value );

protected:
    void mouseMoveEvent(TQMouseEvent *qme);
    void mousePressEvent(TQMouseEvent *qme);
    void mouseReleaseEvent(TQMouseEvent *qme);
    void paintEvent(TQPaintEvent *qpe);
    void enterEvent(TQEvent * );
    void leaveEvent ( TQEvent * );

    int pos2value( int x, int y );

    bool _down;
    bool _lit;
    int _value;
    int _min, _max;
};


class KaimanStyleBackground : public KaimanStyleMasked
{
 Q_OBJECT
  TQ_OBJECT
public:
    KaimanStyleBackground(TQWidget *parent, const char *name=0);
    ~KaimanStyleBackground();

protected:
    void mousePressEvent(TQMouseEvent *qme);
    void mouseReleaseEvent(TQMouseEvent *qme);
    void mouseMoveEvent(TQMouseEvent *qme);

private:
    bool i_b_move;
    TQPoint i_point_dragStart;
    TQPoint i_point_lastPos;
};


class KaimanStyleValue : public KaimanStyleMasked
{
 Q_OBJECT
  TQ_OBJECT
public:
    KaimanStyleValue(int min, int max, TQWidget *parent, const char *name=0);
    ~KaimanStyleValue();

    int value() { return _value; };

public slots:
    void setValue( int value );
    void setValue( int value, int min, int max );

private:
    int _min, _max, _value;
};


class KaimanStyleState : public KaimanStyleMasked
{
 Q_OBJECT
  TQ_OBJECT
public:
    KaimanStyleState(TQWidget *parent, const char *name=0);
    ~KaimanStyleState();

    int value() { return _value; };

public slots:
    void setValue( int value );

signals:
    void clicked();

protected:
    void mousePressEvent(TQMouseEvent *qme);

private:
    int _value;
};


class KaimanStyleNumber : public KaimanStyleElement
{
 Q_OBJECT
  TQ_OBJECT
public:
    KaimanStyleNumber(TQWidget *parent, const char *name=0);
    ~KaimanStyleNumber();

    virtual void loadPixmaps(TQString &val_s_filename);

    static const bool optionCentered = 1;

    int value() { return _value; };

public slots:
    void setValue( int value );

protected:
    void paintEvent(TQPaintEvent *qpe);

private:
    int _value;
};


class KaimanStyleText : public KaimanStyleElement
{
 Q_OBJECT
  TQ_OBJECT
public:
    KaimanStyleText(TQWidget *parent, const char *name=0);
    ~KaimanStyleText();

    virtual void loadPixmaps(TQString &val_s_filename);

    static const bool optionExtended;

    TQString value() { return _value; };

    void startAnimation( int delay );
    void stopAnimation();

public slots:
    void setValue( TQString value );

protected:
    void paintEvent(TQPaintEvent *qpe);

protected slots:
    void timeout();

private:
    TQString _value;
    int _pos;
    int _direction;
    int _delay;
    TQTimer *_timer;
};


class KaimanStyleAnimation : public KaimanStyleMasked
{
 Q_OBJECT
  TQ_OBJECT
public:
    KaimanStyleAnimation(int delay, TQWidget *parent, const char *name=0);
    ~KaimanStyleAnimation();

public slots:
    void start();
    void pause();
    void stop();

protected:
    void timeout();

private:
    int _delay,_frame;
    TQTimer *_timer;
};

class KaimanStyle : public TQWidget
{
 Q_OBJECT
  TQ_OBJECT
public:
    KaimanStyle(TQWidget *parent, const char *name=0);
    ~KaimanStyle();

    enum { background, mask, play_Button, stop_Button, pause_Button, prev_Button, next_Button,  repeat_Button, shuffle_Button, playlist_Button, mixer_Button, exit_Button, Iconify_Button, Config_Button, Alt_Skin_Button, Minute_Number, Second_Number, in_Rate_Number, in_Hz_Number, song_Number, status_Item, cPU_Number, digit_Large, digit_Small_Default, title, volume_Item, volume_Slider, position_Item, position_Slider };

    enum { ParsingError=1, FileNotFound };

    /// Finds a style element, according  to it's name. Returns 0 when element is not available.
    KaimanStyleElement* find(const char* val_s_elemName);

    /// Tries to load the given style and returns success (true) or failure (false)
    bool loadStyle(const TQString &styleName, const TQString &descFile="skindata" );
    TQString skinName() { return i_skinName; };

    /// Returns the mask
    TQBitmap* Mask();

    virtual bool eventFilter( TQObject *o, TQEvent *e );

private:
    // Parses the "skindata" file and returns success (true) or failure (false)
    int parseStyleFile(TQString &l_s_tmpName);
    TQString getToken(TQString &val_s_string, char val_c_separator);
    void interpretTokens(TQStringList& ref_s_tokens);
    bool loadPixmaps();

    /// The name of the style, e.g. "k9"
    TQString     i_s_styleName;
    /// The base directory, where the style is found. For example
    /// "/opt/kde/share/apps/kaiman/Skins/k9/" or  "/opt/kde/share/apps/kaiman/Skins/k9.tgz"
    TQString     i_s_styleBase;


    // The mask of the complete style. Used for doing tqshaped windows
    TQBitmap     i_bitmap_Mask;

    /// All style elements are stored here.
    TQPtrVector<KaimanStyleElement> I_styleElem;

    // The parent window. In other words: The container that holds all the KaimanStyleElement's
    TQWidget*            i_qw_parent;

    TQPtrList<TQWidget> i_sliders;
    bool i_eventSemaphore;

    TQString i_smallFont;
    TQString i_largeFont;
    TQString i_skinName;
};


#endif
