#include "stdaction.h"
#include "app.h"
#include "player.h"
#include "stereobuttonaction.h"
#include "pluginloader.h"

#include <khelpmenu.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kpopupmenu.h>
#include <kstdaction.h>
#include <tqcursor.h>
#include <tqmap.h>
#include <kdebug.h>
#include <kstdguiitem.h>

/**
 * A namespace to have all of noatun's standard actions
 * This is treated like KStdAction
 **/
namespace NoatunStdAction
{
/////////////////////////////////////////////////////
PlayAction::PlayAction(TQObject *parent, const char *name)
	: KAction(i18n("Play"), 0, napp->player(), TQT_SLOT(playpause()), parent, name)
{
	connect(napp->player(), TQT_SIGNAL(playing()), TQT_SLOT(playing()));
	connect(napp->player(), TQT_SIGNAL(paused()), TQT_SLOT(notplaying()));
	connect(napp->player(), TQT_SIGNAL(stopped()), TQT_SLOT(notplaying()));
	if (napp->player()->isPlaying())
		playing();
	else if (napp->player()->isPaused() || napp->player()->isStopped())
		notplaying();
}

void PlayAction::playing()
{
	setIconSet(TQIconSet(SmallIcon("player_pause")));
	setText(i18n("Pause"));
}

void PlayAction::notplaying()
{
	setIconSet(TQIconSet(SmallIcon("player_play")));
	setText(i18n("Play"));
}
/////////////////////////////////////////////////////

PlaylistAction::PlaylistAction(TQObject *parent, const char *name)
	: KToggleAction(i18n("Show Playlist"), "playlist", 0, napp->player(), TQT_SLOT(toggleListView()), parent, name)
{
        setCheckedState(i18n("Hide Playlist"));
	connect(napp->player(), TQT_SIGNAL(playlistShown()),  TQT_SLOT(shown()));
	connect(napp->player(), TQT_SIGNAL(playlistHidden()), TQT_SLOT(hidden()));
	setChecked(napp->playlist()->listVisible());
}

void PlaylistAction::shown()
{
	setChecked(true);
}

void PlaylistAction::hidden()
{
	setChecked(false);
}

////////////////////////////////////////////////////

PluginActionMenu::PluginActionMenu(TQObject *parent, const char *name)
	: KActionMenu(i18n("&Actions"), parent, name)
{
// 	kdDebug(66666) << k_funcinfo << "called" << endl;
	setEnabled(false);
	mCount=0;
}

void PluginActionMenu::insert (KAction *action, int index)
{
// 	kdDebug(66666) << k_funcinfo << "called" << endl;
	KActionMenu::insert(action,index);
	setEnabled(true);
	mCount++;
}

void PluginActionMenu::remove(KAction *action)
{
// 	kdDebug(66666) << k_funcinfo << "called" << endl;
	KActionMenu::remove(action);
	mCount--;
	if(mCount==0)
		setEnabled(false);
}

int PluginActionMenu::menuAdd(const TQString &text, const TQObject *receiver, const char *member)
{
// 	kdDebug(66666) << k_funcinfo << "called, mCount is currently at " << mCount << endl;
	setEnabled(true);
	mCount++;
	return popupMenu()->insertItem(text, receiver, member);
}

void PluginActionMenu::menuRemove(int id)
{
// 	kdDebug(66666) << k_funcinfo << "called, mCount is currently at " << mCount << endl;
	popupMenu()->removeItem(id);
	mCount--;
	if(mCount==0)
		setEnabled(false);
}

////////////////////////////////////////////////////

VisActionMenu::VisActionMenu(TQObject *parent, const char *name)
	: KActionMenu(i18n("&Visualizations"), parent, name)
{
	connect(popupMenu(), TQT_SIGNAL(aboutToShow()), TQT_TQOBJECT(this), TQT_SLOT(fillPopup()));
	connect(popupMenu(), TQT_SIGNAL(activated(int)), TQT_TQOBJECT(this), TQT_SLOT(toggleVisPlugin(int)));
}

void VisActionMenu::fillPopup()
{
	int id;
	popupMenu()->clear();
	mSpecMap.clear();

	TQValueList<NoatunLibraryInfo> available = napp->libraryLoader()->available();
	TQValueList<NoatunLibraryInfo> loaded = napp->libraryLoader()->loaded();

	for(TQValueList<NoatunLibraryInfo>::Iterator i = available.begin(); i != available.end(); ++i)
	{
		if ((*i).type == "visualization")
		{
			id = popupMenu()->insertItem((*i).name);
			mSpecMap[id] = (*i).specfile;
			popupMenu()->setItemChecked(id, loaded.contains(*i));
		}
	}
}

void VisActionMenu::toggleVisPlugin(int id)
{
	if(!mSpecMap.contains(id))
		return;

	TQString specfile = mSpecMap[id];

	if(popupMenu()->isItemChecked(id))
	{
		napp->libraryLoader()->remove(specfile);
		popupMenu()->setItemChecked(id, false);
	}
	else
	{
		napp->libraryLoader()->add(specfile);
		popupMenu()->setItemChecked(id, true);
	}
}

////////////////////////////////////////////////////

LoopActionMenu::LoopActionMenu(TQObject *parent, const char *name)
	: KActionMenu(i18n("&Loop"), parent, name)
{
	mLoopNone = new KRadioAction(i18n("&None"), TQString::fromLocal8Bit("noatunloopnone"),
		0, TQT_TQOBJECT(this), TQT_SLOT(loopNoneSelected()), TQT_TQOBJECT(this), "loop_none");
	mLoopNone->setExclusiveGroup("loopType");
	insert(mLoopNone);

	mLoopSong = new KRadioAction(i18n("&Song"), TQString::fromLocal8Bit("noatunloopsong"),
		0, TQT_TQOBJECT(this), TQT_SLOT(loopSongSelected()), TQT_TQOBJECT(this), "loop_song");
	mLoopSong->setExclusiveGroup("loopType");
	insert(mLoopSong);

	mLoopPlaylist = new KRadioAction(i18n("&Playlist"), TQString::fromLocal8Bit("noatunloopplaylist"),
		0, TQT_TQOBJECT(this), TQT_SLOT(loopPlaylistSelected()), TQT_TQOBJECT(this), "loop_playlist");
	mLoopPlaylist->setExclusiveGroup("loopType");
	insert(mLoopPlaylist);

	mLoopRandom = new KRadioAction(i18n("&Random"), TQString::fromLocal8Bit("noatunlooprandom"),
		0, TQT_TQOBJECT(this), TQT_SLOT(loopRandomSelected()), TQT_TQOBJECT(this), "loop_random");
	mLoopRandom->setExclusiveGroup("loopType");
	insert(mLoopRandom);

	connect(napp->player(), TQT_SIGNAL(loopTypeChange(int)), TQT_TQOBJECT(this), TQT_SLOT(updateLooping(int)));

	updateLooping(static_cast<int>(napp->player()->loopStyle()));
}

void LoopActionMenu::updateLooping(int loopType)
{
	switch(loopType)
	{
		case Player::None:
			mLoopNone->setChecked(true);
			setIcon("noatunloopnone");
			break;
		case Player::Song:
			mLoopSong->setChecked(true);
			setIcon("noatunloopsong");
			break;
		case Player::Playlist:
			mLoopPlaylist->setChecked(true);
			setIcon("noatunloopplaylist");
			break;
		case Player::Random:
			mLoopRandom->setChecked(true);
			setIcon("noatunlooprandom");
			break;
	}
}

void LoopActionMenu::loopNoneSelected()
{
	napp->player()->loop(Player::None);
}

void LoopActionMenu::loopSongSelected()
{
	napp->player()->loop(Player::Song);
}

void LoopActionMenu::loopPlaylistSelected()
{
	napp->player()->loop(Player::Playlist);
}

void LoopActionMenu::loopRandomSelected()
{
	napp->player()->loop(Player::Random);
}

////////////////////////////////////////////////////

KAction *playpause(TQObject *parent, const char *name)
{
	return new PlayAction(parent, name);
}

KAction *effects(TQObject *parent, const char *name)
{
	return new KAction(i18n("&Effects..."), "effect", 0, TQT_TQOBJECT(napp), TQT_SLOT(effectView()), parent, name);
}

KAction *equalizer(TQObject *parent, const char *name)
{
	return new KAction(i18n("E&qualizer..."), "equalizer", 0, TQT_TQOBJECT(napp), TQT_SLOT(equalizerView()), parent, name);
}

KAction *back(TQObject *parent, const char *name)
{
	return new KAction(i18n("&Back"), "player_start", 0, TQT_TQOBJECT(napp->player()), TQT_SLOT(back()), parent, name);
}

KAction *stop(TQObject *parent, const char *name)
{
	StereoButtonAction *action = new StereoButtonAction(i18n("Stop"), "player_stop", 0, napp->player(), TQT_SLOT(stop()), parent, name);
	TQObject::connect(napp->player(), TQT_SIGNAL(playing()), action, TQT_SLOT(enable()));
	TQObject::connect(napp->player(), TQT_SIGNAL(paused()), action, TQT_SLOT(enable()));
	TQObject::connect(napp->player(), TQT_SIGNAL(stopped()), action, TQT_SLOT(disable()));
	if(napp->player()->isStopped())
		action->disable();
	else
		action->enable();
	return action;
}

KAction *forward(TQObject *parent, const char *name)
{
	return new KAction(i18n("&Forward"), "player_end", 0, napp->player(), TQT_SLOT(forward()), parent, name);
}

KAction *play(TQObject *parent, const char *name)
{
	StereoButtonAction *action = new StereoButtonAction(i18n("&Play"), "player_play", 0, napp->player(), TQT_SLOT(playpause()), parent, name);
	TQObject::connect(napp->player(), TQT_SIGNAL(playing()), action, TQT_SLOT(disable()));
	TQObject::connect(napp->player(), TQT_SIGNAL(paused()), action, TQT_SLOT(enable()));
	TQObject::connect(napp->player(), TQT_SIGNAL(stopped()), action, TQT_SLOT(enable()));
	if(napp->player()->isPlaying())
		action->disable();
	else
		action->enable();
	return action;
}

KAction *pause(TQObject *parent, const char *name)
{
	StereoButtonAction *action = new StereoButtonAction(i18n("&Pause"), "player_pause", 0, napp->player(), TQT_SLOT(playpause()), parent, name);
	TQObject::connect(napp->player(), TQT_SIGNAL(playing()), action, TQT_SLOT(enable()));
	TQObject::connect(napp->player(), TQT_SIGNAL(paused()), action, TQT_SLOT(disable()));
	TQObject::connect(napp->player(), TQT_SIGNAL(stopped()), action, TQT_SLOT(disable()));
	if(napp->player()->isPlaying())
		action->enable();
	else
		action->disable();
	return action;
}

LoopActionMenu *loop(TQObject *parent, const char *name)
{
	return new LoopActionMenu(parent, name);
}

PluginActionMenu *actions()
{
	// NoatunApp makes sure that we only have one ActionMenu around
	return napp->pluginActionMenu();
}

VisActionMenu *visualizations(TQObject *parent, const char *name)
{
	return new VisActionMenu(parent, name);
}

KToggleAction *playlist(TQObject *parent, const char *name)
{
	return new PlaylistAction(parent, name);
}

KPopupMenu *ContextMenu::mContextMenu = 0;

KPopupMenu *ContextMenu::contextMenu()
{
	if(!mContextMenu) mContextMenu = createContextMenu(0);

	return mContextMenu;
}

KPopupMenu *ContextMenu::createContextMenu(TQWidget *p)
{
	KPopupMenu *contextMenu = new KPopupMenu(p, "NoatunContextMenu");

	KHelpMenu *helpmenu = new KHelpMenu(contextMenu, kapp->aboutData(), false);
	KActionCollection* actions = new KActionCollection(helpmenu);

	KStdAction::open(TQT_TQOBJECT(napp), TQT_SLOT(fileOpen()), actions)->plug(contextMenu);
	KStdAction::quit(TQT_TQOBJECT(napp), TQT_SLOT(quit()), actions)->plug(contextMenu);
	contextMenu->insertItem(SmallIcon("help"), KStdGuiItem::help().text(), helpmenu->menu());
	contextMenu->insertSeparator();
	KStdAction::preferences(TQT_TQOBJECT(napp), TQT_SLOT(preferences()), actions)->plug(contextMenu);
	NoatunStdAction::playlist(TQT_TQOBJECT(contextMenu))->plug(contextMenu);
	NoatunStdAction::effects(TQT_TQOBJECT(contextMenu))->plug(contextMenu);
	NoatunStdAction::equalizer(TQT_TQOBJECT(napp))->plug(contextMenu);
	NoatunStdAction::visualizations(TQT_TQOBJECT(napp))->plug(contextMenu);
	napp->pluginActionMenu()->plug(contextMenu);

	return contextMenu;
}

void ContextMenu::showContextMenu(const TQPoint &p)
{
	contextMenu()->exec(p);
}

void ContextMenu::showContextMenu()
{
	showContextMenu(TQCursor::pos());
}

} // END namespace NoatunStdAction

#include "stdaction.moc"
