/*
 *  Copyright (c) 1998 Denis Perchine <dyp@perchine.com>
 *  Copyright (c) 2004 Szombathelyi György <gyurco@freemail.hu>
 *  Former maintainer: Adriaan de Groot <groot@kde.org>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public
 *  License version 2 as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 **/

#include <stdio.h>
#include <stdlib.h>

#include <tqdatetime.h>
#include <tqwhatsthis.h>
#include <tqlayout.h>
#include <tqgroupbox.h>
#include <tqfile.h>

#include <kseparator.h>
#include <kmessagebox.h>
#include <kdebug.h>

#include <tqvalidator.h>

#include "propdlg.h"
#include "pwddlg.h"
#include "kglobal_.h"
#include "misc.h"

void propdlg::addRow(TQWidget *parent, TQGridLayout *tqlayout, int row,
  TQWidget *widget, const TQString &label, const TQString &what,
  bool two_column, bool nochange)
{
   TQLabel *lab = new TQLabel(widget, label, parent);
   lab->setMinimumSize(lab->tqsizeHint());
   widget->setMinimumSize(widget->tqsizeHint());
   tqlayout->addWidget(lab, row, 0);
   if (!what.isEmpty())
   {
      TQWhatsThis::add(lab, what);
      TQWhatsThis::add(widget, what);
   }
   if (two_column)
      tqlayout->addMultiCellWidget(widget, row, row, 1, 2);
   else
      tqlayout->addWidget(widget, row, 1);

   if ( !nochange || ro ) return;
   TQCheckBox *nc = new TQCheckBox( i18n("Do not change"), parent );
   tqlayout->addWidget( nc, row, 3 );
   nc->hide();
   mNoChanges[ widget ] = nc;
}

KIntSpinBox *propdlg::addDaysGroup(TQWidget *parent, TQGridLayout *tqlayout, int row,
  const TQString &title, bool never)
{
    KIntSpinBox *days;

    TQLabel *label = new TQLabel( title, parent );
    tqlayout->addMultiCellWidget( label, row, row, 0, 1, AlignRight );

    days = new KIntSpinBox( parent );
    label->setBuddy( days );
    days->setSuffix( i18n(" days") );
    days->setMaxValue( 99999 );
    if (never)
    {
      days->setMinValue( -1 );
      days->setSpecialValueText(i18n("Never"));
    }
    else
    {
      days->setMinValue( 0 );
    }
    tqlayout->addWidget( days, row, 2 );

    connect(days, TQT_SIGNAL(valueChanged(int)), this, TQT_SLOT(changed()));

    TQCheckBox *nc = new TQCheckBox( i18n("Do not change"), parent );
    tqlayout->addWidget( nc, row, 3 );
    nc->hide();
    mNoChanges[ days ] = nc;

    return days;
}

void propdlg::initDlg()
{
  ro = kug->getUsers().getCaps() & KU::KUsers::Cap_ReadOnly;

  TQString whatstr;

  // Tab 1: User Info
  {
    TQFrame *frame = addPage(i18n("User Info"));
    TQGridLayout *tqlayout = new TQGridLayout(frame, 20, 4, marginHint(), spacingHint());
    int row = 0;

    frontpage = frame;
    fronttqlayout = tqlayout;

    lbuser = new TQLabel(frame);
//    whatstr = i18n("WHAT IS THIS: User login");
    addRow(frame, tqlayout, row++, lbuser, i18n("User login:"), whatstr, false, false);

    leid = new KLineEdit(frame);
//    whatstr = i18n("WHAT IS THIS: User Id");
    leid->setValidator(new TQIntValidator(TQT_TQOBJECT(frame)));
    addRow(frame, tqlayout, row++, leid, i18n("&User ID:"), whatstr);
    connect(leid, TQT_SIGNAL(textChanged(const TQString &)), this, TQT_SLOT(changed()));

    if ( !ro ) {
      pbsetpwd = new TQPushButton(i18n("Set &Password..."), frame);
      tqlayout->addWidget(pbsetpwd, 0, 2);
      connect(pbsetpwd, TQT_SIGNAL(clicked()), this, TQT_SLOT(setpwd()));
    }


    lefname = new KLineEdit(frame);
//    whatstr = i18n("WHAT IS THIS: Full Name");
    addRow(frame, tqlayout, row++, lefname, i18n("Full &name:"), whatstr);
    connect(lefname, TQT_SIGNAL(textChanged(const TQString &)), this, TQT_SLOT(changed()));
    lefname->setFocus();

    if ( kug->getUsers().getCaps() & KU::KUsers::Cap_InetOrg ) {
      lesurname = new KLineEdit(frame);
//    whatstr = i18n("WHAT IS THIS: Surname");
      addRow(frame, tqlayout, row++, lesurname, i18n("Surname:"), whatstr);
      connect(lesurname, TQT_SIGNAL(textChanged(const TQString &)), this, TQT_SLOT(changed()));

      lemail = new KLineEdit(frame);
//    whatstr = i18n("WHAT IS THIS: Email");
      addRow(frame, tqlayout, row++, lemail, i18n("Email address:"), whatstr);
      connect(lemail, TQT_SIGNAL(textChanged(const TQString &)), this, TQT_SLOT(changed()));
    }

    leshell = new KComboBox(true, frame);
    leshell->clear();
    leshell->insertItem(i18n("<Empty>"));

    TQStringList shells = readShells();
    shells.sort();
    leshell->insertStringList(shells);
    connect(leshell, TQT_SIGNAL(activated(const TQString &)), this, TQT_SLOT(changed()));
    connect(leshell, TQT_SIGNAL(textChanged(const TQString &)), this, TQT_SLOT(changed()));
//    whatstr = i18n("WHAT IS THIS: Login Shell");
    addRow(frame, tqlayout, row++, leshell, i18n("&Login shell:"), whatstr);

    lehome = new KLineEdit(frame);
    connect(lehome, TQT_SIGNAL(textChanged(const TQString &)), this, TQT_SLOT(changed()));
//    whatstr = i18n("WHAT IS THIS: Home Directory");
    addRow(frame, tqlayout, row++, lehome, i18n("&Home folder:"), whatstr);

    // FreeBSD appears to use the comma separated fields in the GECOS entry
    // differently than Linux.
    if ( kug->getUsers().getCaps() & KU::KUsers::Cap_BSD ) {
      leoffice = new KLineEdit(frame);
      connect(leoffice, TQT_SIGNAL(textChanged(const TQString &)), this, TQT_SLOT(changed()));
//    whatstr = i18n("WHAT IS THIS: Office");
      addRow(frame, tqlayout, row++, leoffice, i18n("&Office:"), whatstr);

      leophone = new KLineEdit(frame);
      connect(leophone, TQT_SIGNAL(textChanged(const TQString &)), this, TQT_SLOT(changed()));
//    whatstr = i18n("WHAT IS THIS: Office Phone");
      addRow(frame, tqlayout, row++, leophone, i18n("Offi&ce Phone:"), whatstr);

      lehphone = new KLineEdit(frame);
      connect(lehphone, TQT_SIGNAL(textChanged(const TQString &)), this, TQT_SLOT(changed()));
//    whatstr = i18n("WHAT IS THIS: Home Phone");
      addRow(frame, tqlayout, row++, lehphone, i18n("Ho&me Phone:"), whatstr);

      leclass = new KLineEdit(frame);
      connect(leclass, TQT_SIGNAL(textChanged(const TQString &)), this, TQT_SLOT(changed()));
//    whatstr = i18n("WHAT IS THIS: Login class");
      addRow(frame, tqlayout, row++, leclass, i18n("Login class:"), whatstr, true);
    } else {
      leoffice1 = new KLineEdit(frame);
      connect(leoffice1, TQT_SIGNAL(textChanged(const TQString &)), this, TQT_SLOT(changed()));
//    whatstr = i18n("WHAT IS THIS: Office1");
      addRow(frame, tqlayout, row++, leoffice1, i18n("&Office #1:"), whatstr);

      leoffice2 = new KLineEdit(frame);
      connect(leoffice2, TQT_SIGNAL(textChanged(const TQString &)), this, TQT_SLOT(changed()));
//    whatstr = i18n("WHAT IS THIS: Office2");
      addRow(frame, tqlayout, row++, leoffice2, i18n("O&ffice #2:"), whatstr);

      leaddress = new KLineEdit(frame);
      connect(leaddress, TQT_SIGNAL(textChanged(const TQString &)), this, TQT_SLOT(changed()));
//    whatstr = i18n("WHAT IS THIS: Address");
      addRow(frame, tqlayout, row++, leaddress, i18n("&Address:"), whatstr);
    }
    cbdisabled = new TQCheckBox(frame);
    connect(cbdisabled, TQT_SIGNAL(stateChanged(int)), this, TQT_SLOT(changed()));
    addRow(frame, tqlayout, row++, cbdisabled, i18n("Account &disabled"), whatstr);

    if ( kug->getUsers().getCaps() & KU::KUsers::Cap_Disable_POSIX ) {
      cbposix = new TQCheckBox(frame);
      connect(cbposix, TQT_SIGNAL(stateChanged(int)), this, TQT_SLOT(changed()));
      connect(cbposix, TQT_SIGNAL(stateChanged(int)), this, TQT_SLOT(cbposixChanged()));
      addRow(frame, tqlayout, row++, cbposix, i18n("Disable &POSIX account information"), whatstr);
    } else {
      cbposix = 0;
    }
    frontrow = row;
  }

  if ( kug->getUsers().getCaps() & KU::KUsers::Cap_Shadow ||
       kug->getUsers().getCaps() & KU::KUsers::Cap_Samba ||
       kug->getUsers().getCaps() & KU::KUsers::Cap_BSD ) {

  // Tab 2 : Password Management
    TQFrame *frame = addPage(i18n("Password Management"));
    TQGridLayout *tqlayout = new TQGridLayout(frame, 20, 4, marginHint(), spacingHint());
    int row = 0;

    TQDateTime time;
    leslstchg = new TQLabel(frame);
    addRow(frame, tqlayout, row++, leslstchg, i18n("Last password change:"), TQString(), true);

    tqlayout->addMultiCellWidget(new KSeparator(KSeparator::HLine, frame), row, row, 0, 3);
    row++;

    if ( kug->getUsers().getCaps() & KU::KUsers::Cap_Shadow ) {
      tqlayout->addWidget( new TQLabel( i18n("POSIX parameters:"), frame ), row++, 0 );
      lesmin = addDaysGroup(frame, tqlayout, row++, i18n("Time before password may &not be changed after last password change:"), false);
      lesmax = addDaysGroup(frame, tqlayout, row++, i18n("Time when password &expires after last password change:") );
      leswarn = addDaysGroup(frame, tqlayout, row++, i18n("Time before password expires to &issue an expire warning:"));
      lesinact = addDaysGroup(frame, tqlayout, row++, i18n("Time when account will be &disabled after expiration of password:"));
      tqlayout->addMultiCellWidget(new KSeparator(KSeparator::HLine, frame), row, row, 0, 3);
      row++;
    }
    /*
    if ( kug->getUsers().getCaps() & KU::KUsers::Cap_Samba ) {
      tqlayout->addWidget( new TQLabel( "SAMBA parameters:", frame ), row++, 0 );
      tqlayout->addMultiCellWidget(new KSeparator(KSeparator::HLine, frame), row, row, 0, 3);
      row++;
    }
    */
    TQLabel *label = new TQLabel( i18n("&Account will expire on:"), frame );
    tqlayout->addWidget( label, row, 0 );
    lesexpire = new KDateTimeWidget( frame );
    label->setBuddy( lesexpire );
    tqlayout->addMultiCellWidget( lesexpire, row, row, 1, 2);

    cbexpire = new TQCheckBox( i18n("Never"), frame );
    tqlayout->addWidget( cbexpire, row++, 3 );

    connect( lesexpire, TQT_SIGNAL(valueChanged(const TQDateTime&)), this, TQT_SLOT(changed()) );
    connect( cbexpire, TQT_SIGNAL(stateChanged(int)), this, TQT_SLOT(changed()) );
    connect( cbexpire, TQT_SIGNAL(toggled(bool)), lesexpire, TQT_SLOT(setDisabled(bool)) );
  }

  // Tab 3: Samba
  if ( kug->getUsers().getCaps() & KU::KUsers::Cap_Samba ) {
    TQFrame *frame = addPage(i18n("Samba"));
    TQGridLayout *tqlayout = new TQGridLayout(frame, 10, 4, marginHint(), spacingHint());
    int row = 0;

    lerid = new KLineEdit(frame);
//  whatstr = i18n("WHAT IS THIS: Rid");
    lerid->setValidator(new TQIntValidator(TQT_TQOBJECT(frame)));
    addRow(frame, tqlayout, row++, lerid, i18n("RID:"), whatstr);
    connect(lerid, TQT_SIGNAL(textChanged(const TQString &)), this, TQT_SLOT(changed()));

    leliscript = new KLineEdit(frame);
//  whatstr = i18n("WHAT IS THIS: Login script");
    addRow(frame, tqlayout, row++, leliscript, i18n("Login script:"), whatstr);
    connect(leliscript, TQT_SIGNAL(textChanged(const TQString &)), this, TQT_SLOT(changed()));

    leprofile = new KLineEdit(frame);
//  whatstr = i18n("WHAT IS THIS: Login script");
    addRow(frame, tqlayout, row++, leprofile, i18n("Profile path:"), whatstr);
    connect(leprofile, TQT_SIGNAL(textChanged(const TQString &)), this, TQT_SLOT(changed()));

    lehomedrive = new KLineEdit(frame);
//  whatstr = i18n("WHAT IS THIS: Login script");
    addRow(frame, tqlayout, row++, lehomedrive, i18n("Home drive:"), whatstr);
    connect(lehomedrive, TQT_SIGNAL(textChanged(const TQString &)), this, TQT_SLOT(changed()));

    lehomepath = new KLineEdit(frame);
//  whatstr = i18n("WHAT IS THIS: Login script");
    addRow(frame, tqlayout, row++, lehomepath, i18n("Home path:"), whatstr);
    connect(lehomepath, TQT_SIGNAL(textChanged(const TQString &)), this, TQT_SLOT(changed()));

    leworkstations = new KLineEdit(frame);
//  whatstr = i18n("WHAT IS THIS: Login script");
    addRow(frame, tqlayout, row++, leworkstations, i18n("User workstations:"), whatstr);
    connect(leworkstations, TQT_SIGNAL(textChanged(const TQString &)), this, TQT_SLOT(changed()));

    ledomain = new KLineEdit(frame);
//  whatstr = i18n("WHAT IS THIS: Login script");
    addRow(frame, tqlayout, row++, ledomain, i18n("Domain name:"), whatstr);
    connect(ledomain, TQT_SIGNAL(textChanged(const TQString &)), this, TQT_SLOT(changed()));

    ledomsid = new KLineEdit(frame);
//  whatstr = i18n("WHAT IS THIS: Login script");
    addRow(frame, tqlayout, row++, ledomsid, i18n("Domain SID:"), whatstr);
    connect(ledomsid, TQT_SIGNAL(textChanged(const TQString &)), this, TQT_SLOT(changed()));

    cbsamba = new TQCheckBox(frame);
    connect(cbsamba, TQT_SIGNAL(stateChanged(int)), this, TQT_SLOT(changed()));
    connect(cbsamba, TQT_SIGNAL(stateChanged(int)), this, TQT_SLOT(cbsambaChanged()));
    addRow(frame, tqlayout, row++, cbsamba, i18n("Disable &Samba account information"), whatstr);
  }

  // Tab 4: Groups
  {
    TQFrame *frame = addPage(i18n("Groups"));
    TQGridLayout *tqlayout = new TQGridLayout(frame, 2, 2, marginHint(), spacingHint());

    lstgrp = new KListView(frame);
    lstgrp->setFullWidth(true); // Single column, full widget width.
    lstgrp->addColumn( i18n("Groups") );
    if ( ro ) lstgrp->setSelectionMode( TQListView::NoSelection );
//    TQString whatstr = i18n("Select the groups that this user belongs to.");
    TQWhatsThis::add(lstgrp, whatstr);
    tqlayout->addMultiCellWidget(lstgrp, 0, 0, 0, 1);
    leprigr = new TQLabel( i18n("Primary group: "), frame );
    tqlayout->addWidget( leprigr, 1, 0 );
    if ( !ro ) {
      pbprigr = new TQPushButton( i18n("Set as Primary"), frame );
      tqlayout->addWidget( pbprigr, 1, 1 );
      connect( pbprigr, TQT_SIGNAL(clicked()), this, TQT_SLOT(setpgroup()) );
    }
    connect( lstgrp, TQT_SIGNAL(clicked(TQListViewItem *)), this, TQT_SLOT(gchanged()) );
  }

}

propdlg::propdlg( const TQPtrList<KU::KUser> &users,
  TQWidget *parent, const char *name ) :
  KDialogBase(Tabbed, i18n("User Properties"), Ok | Cancel, Ok, parent, name, true)

{
  mUsers = users;
  if ( mUsers.getFirst() != mUsers.getLast() )
    setCaption( i18n("User Properties - %1 Selected Users").tqarg( mUsers.count() ) );
  initDlg();
  loadgroups( false );
  selectuser();
  ischanged = false;
  isgchanged = false;
}

propdlg::propdlg( KU::KUser *AUser, bool fixedprivgroup,
  TQWidget *parent, const char *name ) :
  KDialogBase(Tabbed, i18n("User Properties"), Ok | Cancel, Ok, parent, name, true)

{
  mUsers.append( AUser );
  initDlg();
  loadgroups( fixedprivgroup );
  selectuser();
  ischanged = false;
  isgchanged = false;
}

propdlg::~propdlg()
{
}

void propdlg::cbposixChanged()
{
  bool posix = !( cbposix->state() == TQButton::On );
  leid->setEnabled( posix  & ( mUsers.getFirst() == mUsers.getLast() ) );
  leshell->setEnabled( posix );
  lehome->setEnabled( posix );
  if ( kug->getUsers().getCaps() & KU::KUsers::Cap_Shadow ) {
    lesmin->setEnabled( posix );
    lesmax->setEnabled( posix );
    leswarn->setEnabled( posix );
    lesinact->setEnabled( posix );
  }
}

void propdlg::cbsambaChanged()
{
  bool samba = !( cbsamba->state() == TQButton::On );
  lerid->setEnabled( samba & ( mUsers.getFirst() == mUsers.getLast() ) );
  leliscript->setEnabled( samba );
  leprofile->setEnabled( samba );
  lehomedrive->setEnabled( samba );
  lehomepath->setEnabled( samba );
  leworkstations->setEnabled( samba );
  ledomain->setEnabled( samba );
  ledomsid->setEnabled( samba );
}

void propdlg::setLE( KLineEdit *le, const TQString &val, bool first )
{
  if ( first ) {
    le->setText( val );
    if ( ro ) le->setReadOnly( true );
    return;
  }
  if ( val.isEmpty() && le->text().isEmpty() ) return;
  if ( le->text() != val ) {
    le->setText( "" );
    if ( !ro && mNoChanges.contains( le ) ) {
      mNoChanges[ le ]->show();
      mNoChanges[ le ]->setChecked( true );
    }
  }
}

void propdlg::setCB( TQCheckBox *cb, bool val, bool first )
{
  if ( first ) {
    cb->setChecked( val );
    if ( ro ) cb->setEnabled( false );
    return;
  }
  if ( cb->isChecked() != val ) {
    cb->setTristate();
    cb->setNoChange();
  }
}

void propdlg::setSB( KIntSpinBox *sb, int val, bool first )
{
  if ( first ) {
    sb->setValue( val );
    if ( ro ) sb->setEnabled( false );
    return;
  }
  if ( sb->value() != val ) {
    sb->setValue( 0 );
    if ( !ro && mNoChanges.contains( sb ) ) {
      mNoChanges[ sb ]->show();
      mNoChanges[ sb ]->setChecked( true );
    }
  }
}

void propdlg::selectuser()
{
  KU::KUser *user;
  bool first = true, one = ( mUsers.getFirst() == mUsers.getLast() );

  ismoreshells = false;
  user = mUsers.first();
  olduid = user->getUID();
  oldrid = user->getSID().getRID();
  oldshell = user->getShell();
  lstchg = user->getLastChange();
  TQDateTime datetime;
  datetime.setTime_t( lstchg );
  if ( kug->getUsers().getCaps() & KU::KUsers::Cap_Shadow ||
       kug->getUsers().getCaps() & KU::KUsers::Cap_Samba ||
       kug->getUsers().getCaps() & KU::KUsers::Cap_BSD ) {

    leslstchg->setText( KGlobal::locale()->formatDateTime( datetime, false ) );
  }

  if ( one ) {
    lbuser->setText( user->getName() );
    leid->setText( TQString::number( user->getUID() ) );
    if ( ro ) leid->setReadOnly( true );
    if ( kug->getUsers().getCaps() & KU::KUsers::Cap_Samba ) {
      lerid->setText( TQString::number( user->getSID().getRID() ) );
      if ( ro ) lerid->setReadOnly( true );
    }
  } else {
    leid->setEnabled( false );
    if ( kug->getUsers().getCaps() & KU::KUsers::Cap_Samba ) {
      lerid->setEnabled( false );
    }
  }
  if ( ro ) leshell->setEditable( false );

  while ( user ) {

    setLE( lefname, user->getFullName(), first );
    TQString home;
    home = user->getHomeDir();
    if ( !one ) home.replace( user->getName(), "%U" );
    setLE( lehome, home, first );

    TQString shell = user->getShell();
    if ( first ) {
      if ( !shell.isEmpty() ) {
        bool tested = false;
        for ( int i=0; i<leshell->count(); i++ )
          if ( leshell->text(i) == shell ) {
            tested = true;
            leshell->setCurrentItem(i);
            break;
          }
          if ( !tested ) {
            leshell->insertItem( shell );
            leshell->setCurrentItem( leshell->count()-1 );
          }
      } else
        leshell->setCurrentItem(0);
    } else {
      if ( leshell->currentText() != shell ) {
        if ( !ismoreshells ) {
          leshell->insertItem( i18n("Do Not Change"), 0 );
          ismoreshells = true;
        }
        leshell->setCurrentItem( 0 );
      }
    }

    setCB( cbdisabled, user->getDisabled(), first );
    if ( kug->getUsers().getCaps() & KU::KUsers::Cap_Disable_POSIX ) {
      setCB( cbposix, !(user->getCaps() & KU::KUser::Cap_POSIX), first );
    }

    if ( kug->getUsers().getCaps() & KU::KUsers::Cap_Samba ) {
      setLE( leliscript, user->getLoginScript(), first );
      TQString profile;
      profile = user->getProfilePath();
      if ( !one ) profile.replace( user->getName(), "%U" );
      setLE( leprofile, profile, first );
      setLE( lehomedrive, user->getHomeDrive(), first );
      home = user->getHomePath();
      if ( !one ) home.replace( user->getName(), "%U" );
      setLE( lehomepath, home, first );
      setLE( leworkstations, user->getWorkstations(), first );
      setLE( ledomain, user->getDomain(), first );
      setLE( ledomsid, user->getSID().getDOM(), first );
      setCB( cbsamba, !(user->getCaps() & KU::KUser::Cap_Samba), first );
    }

    if ( kug->getUsers().getCaps() & KU::KUsers::Cap_Shadow ||
         kug->getUsers().getCaps() & KU::KUsers::Cap_Samba ||
         kug->getUsers().getCaps() & KU::KUsers::Cap_BSD ) {

      if ( user->getLastChange() != lstchg ) {
        leslstchg->setText( "" );
        lstchg = 0;
      }

      TQDateTime expire;
      expire.setTime_t( user->getExpire() );
      kdDebug() << "expiration: " << user->getExpire() << endl;
      setCB( cbexpire, (int) expire.toTime_t() == -1, first );
      if ( (int) expire.toTime_t() == -1 ) expire.setTime_t( 0 );
      if ( first ) {
        lesexpire->setDateTime( expire );
      } else {
        if ( lesexpire->dateTime() != expire ) {
          expire.setTime_t( 0 );
          lesexpire->setDateTime( expire );
        }
      }
    }

    if ( kug->getUsers().getCaps() & KU::KUsers::Cap_Shadow ) {
      setSB( lesmin, user->getMin(), first );
      setSB( lesmax, user->getMax(), first );
      setSB( leswarn, user->getWarn(), first );
      setSB( lesinact, user->getInactive(), first );
    }

    if ( kug->getUsers().getCaps() & KU::KUsers::Cap_InetOrg ) {
      setLE( lesurname, user->getSurname(), first );
      setLE( lemail, user->getEmail(), first );
    }
    if ( kug->getUsers().getCaps() & KU::KUsers::Cap_BSD ) {
      setLE( leoffice, user->getOffice(), first );
      setLE( leophone, user->getWorkPhone(), first );
      setLE( lehphone, user->getHomePhone(), first );
      setLE( leclass, user->getClass(), first );
    } else {
      setLE( leoffice1, user->getOffice1(), first );
      setLE( leoffice2, user->getOffice2(), first );
      setLE( leaddress, user->getAddress(), first );
    }

    first = false;
    user = mUsers.next();
  }
}

void propdlg::loadgroups( bool fixedprivgroup )
{
  bool wasprivgr = false;

  primaryGroupWasOn = false;

  KU::KGroup *group = kug->getGroups().first();
  while ( group ) {
    TQString groupName = group->getName();
    TQCheckListItem *item = new TQCheckListItem(lstgrp, groupName, TQCheckListItem::CheckBox);
    KU::KUser *user = mUsers.first();
    while ( user ) {
      bool prigr =
        ( !fixedprivgroup && group->getGID() == user->getGID() ) ||
        ( fixedprivgroup && groupName == user->getName() );
      bool on = group->lookup_user( user->getName() ) || prigr;

      if ( prigr ) {
        item->setEnabled( false );
        if ( !wasprivgr )
          primaryGroup = groupName;
        else
          if ( primaryGroup != groupName ) primaryGroup = "";
//      primaryGroupWasOn = group->lookup_user(user->getName());
        wasprivgr = true;
      }

      if ( mUsers.getFirst() == user )
        item->setOn( on );
      else
        if ( item->isOn() != on ) {
          item->setTristate( true );
          item->setState( TQCheckListItem::NoChange );
        }
      user = mUsers.next();
    }
    group = kug->getGroups().next();
  }

  if ( fixedprivgroup ) {
    KU::KUser *user = mUsers.first();
    kdDebug() << "privgroup: " << user->getName() << endl;
    if ( !wasprivgr ) {
      TQCheckListItem *item = new TQCheckListItem(lstgrp, user->getName(), TQCheckListItem::CheckBox);
      item->setOn(true);
      item->setEnabled(false);
      primaryGroup = user->getName();
    }
  }
  leprigr->setText( i18n("Primary group: ") + primaryGroup );
}

void propdlg::setpgroup()
{
  isgchanged = true;
  TQCheckListItem *item;
  item = (TQCheckListItem *) lstgrp->selectedItem();
  if ( item == 0 || item->text() == primaryGroup )
     return;

  bool prevPrimaryGroupWasOn = primaryGroupWasOn;
  primaryGroup = ((TQCheckListItem *) lstgrp->selectedItem())->text();

  item = (TQCheckListItem *) lstgrp->firstChild();

  while(item)
  {
     TQString groupName = item->text();
     if ( !item->isEnabled() )
     {
        item->setEnabled(true);
        item->setOn(prevPrimaryGroupWasOn);
        item->tqrepaint();
     }
     if ( groupName == primaryGroup )
     {
        primaryGroupWasOn = item->isOn();
        item->setEnabled(false);
        item->setOn(true);
        item->tqrepaint();
     }

     item = (TQCheckListItem *) item->nextSibling();
  }
  leprigr->setText( i18n("Primary group: ") + primaryGroup );
}

void propdlg::changed()
{
  TQWidget *widget = (TQWidget*) sender();
  if ( mNoChanges.contains( widget ) ) mNoChanges[ widget ]->setChecked( false );
  ischanged = true;
  kdDebug() << "changed" << endl;
}

void propdlg::gchanged()
{
  isgchanged = true;
}

TQString propdlg::mergeLE( KLineEdit *le, const TQString &val, bool one )
{
  TQCheckBox *cb = 0;
  if ( mNoChanges.contains( le ) ) cb = mNoChanges[ le ];
  return ( one || ( cb && !cb->isChecked() ) ) ? le->text() : val;
}

int propdlg::mergeSB( KIntSpinBox *sb, int val, bool one )
{
  TQCheckBox *cb = 0;
  if ( mNoChanges.contains( sb ) ) cb = mNoChanges[ sb ];
  return ( one || ( cb && !cb->isChecked() ) ) ? sb->value() : val;
}

void propdlg::mergeUser( KU::KUser *user, KU::KUser *newuser )
{
  TQDateTime epoch ;
  epoch.setTime_t(0);
  bool one = ( mUsers.getFirst() == mUsers.getLast() );
  bool posix, samba = false;

  newuser->copy( user );

  if ( kug->getUsers().getCaps() & KU::KUsers::Cap_Disable_POSIX && cbposix->state() != TQButton::NoChange ) {
    if ( cbposix->isChecked() )
      newuser->setCaps( newuser->getCaps() & ~KU::KUser::Cap_POSIX );
    else
      newuser->setCaps( newuser->getCaps() | KU::KUser::Cap_POSIX );
  }
  posix = newuser->getCaps() & KU::KUser::Cap_POSIX;
  kdDebug() << "posix: " << posix << endl;
  if ( one ) {
//    newuser->setName( leuser->text() );
    newuser->setUID( posix ? leid->text().toInt() : 0 );
  }
  if ( !newpass.isNull() ) {
    kug->getUsers().createPassword( newuser, newpass );
    newuser->setLastChange( lstchg );
  }
  newuser->setFullName( mergeLE( lefname, user->getFullName(), one ) );
  if ( kug->getUsers().getCaps() & KU::KUsers::Cap_Samba ) {
    if ( cbsamba->state() != TQButton::NoChange ) {
      if ( cbsamba->isChecked() )
        newuser->setCaps( newuser->getCaps() & ~KU::KUser::Cap_Samba );
      else
        newuser->setCaps( newuser->getCaps() | KU::KUser::Cap_Samba );
    }
    samba = newuser->getCaps() & KU::KUser::Cap_Samba;
    kdDebug() << "user : " << newuser->getName() << " caps: " << newuser->getCaps() << " samba: " << samba << endl;

    SID sid;
    if ( samba ) {
      sid.setRID( one ? lerid->text().toUInt() : user->getSID().getRID() );
      sid.setDOM( mergeLE( ledomsid, user->getSID().getDOM(), one ) );
    }
    newuser->setSID( sid );
    newuser->setLoginScript( samba ? 
      mergeLE( leliscript, user->getLoginScript(), one ) : TQString()  );
    newuser->setProfilePath( samba ? 
      mergeLE( leprofile, user->getProfilePath(), one ).replace( "%U", newuser->getName() ) : TQString() );
    newuser->setHomeDrive( samba ? 
      mergeLE( lehomedrive, user->getHomeDrive(), one ) : TQString() );
    newuser->setHomePath( samba ? 
      mergeLE( lehomepath, user->getHomePath(), one ).replace( "%U", newuser->getName() ) : TQString() );
    newuser->setWorkstations( samba ? 
      mergeLE( leworkstations, user->getWorkstations(), one ) : TQString() );
    newuser->setDomain( samba ? 
      mergeLE( ledomain, user->getDomain(), one ) : TQString() );
  }

  if ( kug->getUsers().getCaps() & KU::KUsers::Cap_BSD ) {
    newuser->setOffice( mergeLE( leoffice, user->getOffice(), one ) );
    newuser->setWorkPhone( mergeLE( leophone, user->getWorkPhone(), one ) );
    newuser->setHomePhone( mergeLE( lehphone, user->getHomePhone(), one ) );
    newuser->setClass( mergeLE( leclass, user->getClass(), one ) );
  } else {
    newuser->setOffice1( mergeLE( leoffice1, user->getOffice1(), one ) );
    newuser->setOffice2( mergeLE( leoffice2, user->getOffice2(), one ) );
    newuser->setAddress( mergeLE( leaddress, user->getAddress(), one ) );
  }

  newuser->setHomeDir( posix ? 
    mergeLE( lehome, user->getHomeDir(), one ).replace( "%U", newuser->getName() ) : 
    TQString() );
  if ( posix ) {
    if ( leshell->currentItem() == 0 && ismoreshells ) {
      newuser->setShell( user->getShell() );
    } else if  (
      ( leshell->currentItem() == 0 && !ismoreshells ) ||
      ( leshell->currentItem() == 1 && ismoreshells ) ) {

      newuser->setShell( TQString() );
    } else {
  // TODO: Check shell.
      newuser->setShell( leshell->currentText() );
    }
  } else
    newuser->setShell( TQString() );

  newuser->setDisabled( (cbdisabled->state() == TQButton::NoChange) ? user->getDisabled() : cbdisabled->isChecked() );

  if ( kug->getUsers().getCaps() & KU::KUsers::Cap_InetOrg ) {
    newuser->setSurname( mergeLE( lesurname, user->getSurname(), one ) );
    newuser->setEmail( mergeLE( lemail, user->getEmail(), one ) );
  }

  if ( kug->getUsers().getCaps() & KU::KUsers::Cap_Shadow ) {
    newuser->setMin( posix ? mergeSB( lesmin, user->getMin(), one ) : 0 );
    newuser->setMax( posix ? mergeSB( lesmax, user->getMax(), one ) : 0 );
    newuser->setWarn( posix ? mergeSB( leswarn, user->getWarn(), one ) : 0 );
    newuser->setInactive( posix ? mergeSB( lesinact, user->getInactive(), one ) : 0 );
  }

  if ( ( (kug->getUsers().getCaps() & KU::KUsers::Cap_Shadow) && posix ) ||
       ( (kug->getUsers().getCaps() & KU::KUsers::Cap_Samba) && samba ) ||
       ( (kug->getUsers().getCaps() & KU::KUsers::Cap_BSD) && posix ) ) {

    switch ( cbexpire->state() ) {
      case TQButton::NoChange:
        newuser->setExpire( user->getExpire() );
        break;
      case TQButton::On:
        newuser->setExpire( -1 );
        break;
      case TQButton::Off:
        newuser->setExpire( !one && lesexpire->dateTime().toTime_t() == 0 ?
          user->getExpire() : lesexpire->dateTime().toTime_t() );
        break;
    }
  } else {
    newuser->setExpire( -1 );
  }

  if ( !primaryGroup.isEmpty() ) {
    KU::KGroup *group = kug->getGroups().lookup( primaryGroup );
    if ( group ) {
      newuser->setGID( group->getGID() );
      if ( kug->getUsers().getCaps() & KU::KUsers::Cap_Samba ) {
        newuser->setPGSID( group->getSID() );
      }
    }
  }
}

bool propdlg::saveg()
{
  if ( !isgchanged ) return true;

  TQCheckListItem *item = (TQCheckListItem *) lstgrp->firstChild();
  KU::KGroup *group;

  while(item)
  {
    kdDebug() << "saveg: group name: " << item->text() << endl;
    group = kug->getGroups().lookup(item->text());
    if ( group && item->state() != TQCheckListItem::NoChange ) {

      KU::KGroup newgroup( group );
      bool mod = false;
      bool on = item->isOn();
      KU::KUser *user = mUsers.first();

      while ( user ) {
        if ( on && (( !primaryGroup.isEmpty() && primaryGroup != group->getName() ) ||
                    ( primaryGroup.isEmpty() && user->getGID() != group->getGID() )) ) {
          if ( newgroup.addUser( user->getName() ) ) mod = true;
        } else {
          if ( newgroup.removeUser( user->getName() ) ) mod = true;
        }
        user = mUsers.next();
      }

      if ( mod ) kug->getGroups().mod( group, newgroup );
    }
    item = (TQCheckListItem *) item->nextSibling();
  }
  return true;
}

bool propdlg::checkShell(const TQString &shell)
{
   if (shell.isEmpty())
      return true;
   TQStringList shells = readShells();
   return shells.contains(shell);
}

bool propdlg::check()
{
  bool one = ( mUsers.getFirst() == mUsers.getLast() );
  bool posix = !( kug->getUsers().getCaps() & KU::KUsers::Cap_Disable_POSIX ) || !( cbposix->isChecked() );

  if ( one && posix && leid->text().isEmpty() ) {
    KMessageBox::sorry( 0, i18n("You need to specify an UID.") );
    return false;
  }

  if ( one && posix && lehome->text().isEmpty() ) {
    KMessageBox::sorry( 0, i18n("You must specify a home directory.") );
    return false;
  }

  if ( kug->getUsers().getCaps() & KU::KUsers::Cap_InetOrg ) {
    if ( one && lesurname->text().isEmpty() ) {
      KMessageBox::sorry( 0, i18n("You must fill the surname field.") );
      return false;
    }
  }

  if ( kug->getUsers().getCaps() & KU::KUsers::Cap_Samba ) {
    if ( one && lerid->text().isEmpty() && !( cbsamba->isChecked() ) ) {
      KMessageBox::sorry( 0, i18n("You need to specify a samba RID.") );
      return false;
    }
  }

  return true;
}

void propdlg::setpwd()
{
  pwddlg pd( this );

  if ( pd.exec() == TQDialog::Accepted ) {
    ischanged = true;
    newpass = pd.getPassword();
    lstchg = now();
    TQDateTime datetime;
    datetime.setTime_t( lstchg );
    if ( kug->getUsers().getCaps() & KU::KUsers::Cap_Shadow ||
        kug->getUsers().getCaps() & KU::KUsers::Cap_Samba ||
        kug->getUsers().getCaps() & KU::KUsers::Cap_BSD ) {

        leslstchg->setText( KGlobal::locale()->formatDateTime( datetime, false ) );
    }
    cbdisabled->setChecked( false );
  }
}

void propdlg::slotOk()
{
  if ( ro ) {
    reject();
    return;
  }

  bool one = ( mUsers.getFirst() == mUsers.getLast() );

  uid_t newuid = leid->text().toULong();

  if ( one && ( !( kug->getUsers().getCaps() & KU::KUsers::Cap_Disable_POSIX ) || !cbposix->isChecked() )
               && olduid != newuid )
  {
    if (kug->getUsers().lookup(newuid)) {
      KMessageBox::sorry( 0,
        i18n("User with UID %1 already exists").tqarg(newuid) );
      return;
    }
  }

  if ( one && ( kug->getUsers().getCaps() & KU::KUsers::Cap_Samba ) && !cbsamba->isChecked() ) {
    uint newrid = lerid->text().toInt();
    if ( oldrid != newrid ) {
      if (kug->getUsers().lookup_sam(newrid)) {
        KMessageBox::sorry( 0,
          i18n("User with RID %1 already exists").tqarg(newrid) );
        return;
      }
    }
  }

  TQString newshell;
  if (leshell->currentItem() != 0)
    newshell = leshell->currentText();

  if (oldshell != newshell)
  {
    if (!checkShell(newshell)) {
      int result = KMessageBox::warningYesNoCancel( 0,
      		i18n("<p>The shell %1 is not yet listed in the file %2. "
      		     "In order to use this shell you must add it to "
      		     "this file first."
      		     "<p>Do you want to add it now?").tqarg(newshell).tqarg(TQFile::decodeName(SHELL_FILE)),
      		i18n("Unlisted Shell"),
      		i18n("&Add Shell"),
      		i18n("Do &Not Add"));
      	if (result == KMessageBox::Cancel)
      	  return;

      	if (result == KMessageBox::Yes)
      	  addShell(newshell);
    }
  }

  if ( !ischanged && !isgchanged ) {
    reject();
  } else if ( check() ) {
    saveg();
    accept();
  }
}

#include "propdlg.moc"
