/*
 *  Copyright (c) 1998 Denis Perchine <dyp@perchine.com>
 *  Copyright (c) 2004 Szombathelyi György <gyurco@freemail.hu>
 *  Former maintainer: Adriaan de Groot <groot@kde.org>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public
 *  License version 2 as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 **/


#include <tqlabel.h>
#include <tqlayout.h>
#include <tqvbox.h>
#include <tqvalidator.h>
#include <tqwhatsthis.h>

#include <kapplication.h>
#include <klocale.h>
#include <kpushbutton.h>
#include <kdebug.h>
#include <kmessagebox.h>

#include "kglobal_.h"
#include "editGroup.h"

editGroup::editGroup(KU::KGroup *akg, bool samba, bool add,
   TQWidget* parent, const char* name)
  : KDialogBase(parent, name, true, i18n("Group Properties"), Ok | Cancel, Ok, true)
{
  kg = akg;
  mAdd = add;
  mSamba = samba;
  mOldName = kg->getName();
  SID sid = kg->getSID();
  ro = kug->getGroups().getCaps() & KU::KGroups::Cap_ReadOnly;

  RID rid;
  rid.rid = 512; rid.name = i18n("Domain Admins"); rid.desc = i18n("Admins"); mRids.append( rid );
  rid.rid = 513; rid.name = i18n("Domain Users"); rid.desc = i18n("Users");  mRids.append( rid );
  rid.rid = 514; rid.name = i18n("Domain Guests"); rid.desc = i18n("Guests"); mRids.append( rid );

  TQFrame *page = makeMainWidget();
  TQGridLayout *tqlayout = new TQGridLayout( page, 10, 3, marginHint(), spacingHint() );
  TQLabel *lb;

  lb = new TQLabel( page );
  lb->setText(i18n("Group number:"));
  legid = new KLineEdit(page);
  // ensure it fits at least 20 characters
  legid->setText( "XXXXXXXXXXXXXXXXXXX" );
  legid->setText( TQString::number(kg->getGID()) );
  legid->setValidator( new TQIntValidator(TQT_TQOBJECT(this)) );
  legid->setEnabled( mAdd );
  legid->setReadOnly( ro );
  lb->setBuddy( legid );
  tqlayout->addWidget( lb, 0, 0 );
  tqlayout->addMultiCellWidget( legid, 0, 0, 1, 2 );

  if ( mSamba ) {
    lb = new TQLabel( page );
    lb->setText(i18n("Group rid:"));
    lerid = new KComboBox( page );
    lerid->setEditable( !ro );
    TQValueList<RID>::Iterator it;
    for ( it = mRids.begin(); it != mRids.end(); ++it ) {
      lerid->insertItem( TQString::number( (*it).rid ) + " - " + (*it).name );
    }

    lerid->setCurrentText( TQString::number( sid.getRID() ) );
    lerid->setValidator (new TQIntValidator(TQT_TQOBJECT(this)) );
    lerid->setEnabled( mAdd );
    connect( lerid, TQT_SIGNAL(activated(int)), TQT_SLOT(ridSelected(int)) );
    lb->setBuddy( lerid );
    tqlayout->addWidget( lb, 1, 0 );
    tqlayout->addMultiCellWidget( lerid, 1, 1, 1, 2 );
  }

  lb = new TQLabel( page );
  lb->setText(i18n("Group name:"));

  legrpname = new KLineEdit( page );
  // ensure it fits at least 20 characters
  legrpname->setText( "XXXXXXXXXXXXXXXXXXX" );
  legrpname->setText( kg->getName() );
  legrpname->setReadOnly( ro );
  legrpname->setFocus();
  lb->setBuddy( legrpname );
  tqlayout->addWidget( lb, 2, 0 );
  tqlayout->addMultiCellWidget( legrpname, 2, 2, 1, 2 );

  if ( mSamba ) {
    lb = new TQLabel( page );
    lb->setText(i18n("Description:"));
    ledesc = new KLineEdit(page);
    ledesc->setText( kg->getDesc() );
    ledesc->setReadOnly( ro );
    lb->setBuddy( ledesc );
    tqlayout->addWidget( lb, 3, 0 );
    tqlayout->addMultiCellWidget( ledesc, 3, 3, 1, 2 );

    lb = new TQLabel( page );
    lb->setText(i18n("Display name:"));
    ledispname = new KLineEdit(page);
    ledispname->setText( kg->getDisplayName() );
    ledispname->setReadOnly( ro );
    lb->setBuddy( ledispname );
    tqlayout->addWidget( lb, 4, 0 );
    tqlayout->addMultiCellWidget( ledispname, 4, 4, 1, 2 );

    lb = new TQLabel( page );
    lb->setText(i18n("Type:"));
    letype = new KComboBox( page );
    letype->insertItem( i18n("Domain") );
    letype->insertItem( i18n("Local") );
    letype->insertItem( i18n("Builtin") );
    switch ( kg->getType() ) {
      case 2:
        letype->setCurrentItem( 0 );
        break;
      case 4:
        letype->setCurrentItem( 1 );
        break;
      case 5:
        letype->setCurrentItem( 2 );
        break;
    }
    lb->setBuddy( letype );
    tqlayout->addWidget( lb, 5, 0 );
    tqlayout->addMultiCellWidget( letype, 5, 5, 1, 2 );

    lb = new TQLabel( page );
    lb->setText(i18n("Domain SID:"));
    ledomsid = new KLineEdit(page);
    ledomsid->setText( sid.getDOM() );
    ledomsid->setReadOnly( ro );
    lb->setBuddy( ledomsid );
    tqlayout->addWidget( lb, 6, 0 );
    tqlayout->addMultiCellWidget( ledomsid, 6, 6, 1, 2 );

    cbsamba = new TQCheckBox( i18n("Disable Samba group information"), page );
    tqlayout->addMultiCellWidget( cbsamba, 7, 7, 0, 2 );
    connect( cbsamba, TQT_SIGNAL(toggled(bool)), ledesc, TQT_SLOT(setDisabled(bool)) );
    connect( cbsamba, TQT_SIGNAL(toggled(bool)), ledispname, TQT_SLOT(setDisabled(bool)) );
    connect( cbsamba, TQT_SIGNAL(toggled(bool)), letype, TQT_SLOT(setDisabled(bool)) );
    connect( cbsamba, TQT_SIGNAL(toggled(bool)), ledomsid, TQT_SLOT(setDisabled(bool)) );
    if ( mAdd ) connect( cbsamba, TQT_SIGNAL(toggled(bool)), lerid, TQT_SLOT(setDisabled(bool)) );
    if ( !mAdd ) cbsamba->setChecked( !( kg->getCaps() & KU::KGroup::Cap_Samba ) );
  }

  m_list_in = new KListView(page);
  m_list_in->setFullWidth(true); // Single column, full widget width.
  m_list_in->addColumn(i18n("Users in Group"));
  m_list_in->setSelectionMode( TQListView::Extended );
  tqlayout->addWidget( m_list_in, 8, 0 );

  TQVBox *vbox = new TQVBox(page);
  TQPushButton *btadd = new TQPushButton(i18n("Add <-"), vbox);
  TQPushButton *btdel = new TQPushButton(i18n("Remove ->"), vbox);
  tqlayout->addWidget( vbox, 8, 1 );

  m_list_notin = new KListView(page);
  m_list_notin->setFullWidth(true); // Single column, full widget width.
  m_list_notin->addColumn(i18n("Users NOT in Group"));
  m_list_notin->setSelectionMode(TQListView::Extended);
  tqlayout->addWidget( m_list_notin, 8, 2 );
//  TQString whatstr = i18n("Select the users that should be in this kg->");
//  TQWhatsThis::add(m_list, whatstr);
//  connect(this,TQT_SIGNAL(okClicked(void)),
          //this,TQT_SLOT(okClicked()));


  for (unsigned int i = 0; i<kug->getUsers().count(); i++) {
    KU::KUser *user;
    user = kug->getUsers()[i];
    TQString userName = user->getName();
    if ( kg->lookup_user(userName) || user->getGID() == kg->getGID() ) {
      KListViewItem *item = new KListViewItem(m_list_in, userName);
      if ( user->getGID() == kg->getGID() ) item->setSelectable( false );
    } else {
      new KListViewItem(m_list_notin, userName);
    }
  }

  connect(btadd, TQT_SIGNAL(clicked()), TQT_SLOT(addClicked()));
  connect(btdel, TQT_SIGNAL(clicked()), TQT_SLOT(delClicked()));

  if ( ro ) {
    btadd->setEnabled( false );
    btdel->setEnabled( false );
  }
}

editGroup::~editGroup() 
{
}

void editGroup::ridSelected( int index )
{
  lerid->setCurrentText( TQString::number( mRids[ index ].rid ) );
  legrpname->setText( mRids[ index ].name );
  ledesc->setText( mRids[ index ].desc );
  ledispname->setText( mRids[ index ].name );
}

void editGroup::addClicked()
{
  TQListViewItem *item, *next;
  TQString name;

  item = m_list_notin->firstChild();
  while ( item ) {
    next = item->nextSibling();
    if ( item->isSelected() ) {
      name = item->text( 0 );
      delete item;
      item = new KListViewItem( m_list_in, name );
    }
    item = next;
  }
}

void editGroup::delClicked()
{
  TQListViewItem *item, *next;
  TQString name;

  item = m_list_in->firstChild();
  while ( item ) {
    next = item->nextSibling();
    if ( item->isSelected() ) {
      name = item->text( 0 );
      delete item;
      item = new KListViewItem( m_list_notin, name );
    }
    item = next;
  }
}

void editGroup::slotOk()
{
  if ( ro ) {
    reject();
    return;
  }

  SID sid;
  kg->clear();
  TQString s;
  s = legid->text();

  if ( mSamba && !cbsamba->isChecked() ) {
    sid.setDOM( ledomsid->text() );
    sid.setRID( lerid->currentText() );
  }

  if ( legrpname->text().isEmpty() ) {
    KMessageBox::sorry( 0,
      i18n("You need to type a group name.") );
    return;
  }

  if ( legrpname->text() != mOldName && 
    kug->getGroups().lookup( legrpname->text() ) ) {

    KMessageBox::sorry( 0,
      i18n("Group with name %1 already exists.").tqarg(legrpname->text()) );
    return;
  }

  if ( mAdd ) {
    if ( mSamba && !cbsamba->isChecked() && kug->getGroups().lookup_sam( sid ) ) {
      KMessageBox::sorry( 0,
        i18n("Group with SID %1 already exists.").tqarg( sid.getSID() ) );
      return;
    }
    if (kug->getGroups().lookup(s.toInt())) {
      KMessageBox::sorry( 0,
        i18n("Group with gid %1 already exists.").tqarg(s.toInt()) );
      return;
    }
  }

  kg->setName(legrpname->text());
  kg->setGID(s.toInt());
  if ( mSamba && !cbsamba->isChecked() ) {
    kg->setCaps ( KU::KGroup::Cap_Samba );
    kg->setSID( sid );
    switch ( letype->currentItem() ) {
      case 0:
        kg->setType( 2 );
        break;
      case 1:
        kg->setType( 4 );
        break;
      case 2:
        kg->setType( 5 );
        break;
    }
    kg->setDesc( ledesc->text() );
    kg->setDisplayName( ledispname->text() );
  } else {
    kg->setCaps( 0 );
    kg->setSID( TQString() );
    kg->setDesc( TQString() );
    kg->setDisplayName( TQString() );
    kg->setType( 0 );
  }

  TQListViewItem *item;
  item = m_list_in->firstChild();
  while ( item ) {
    kg->addUser( item->text( 0 ) );
    item = item->nextSibling();
  }
  accept();
}

#include "editGroup.moc"
