// KDat - a tar-based DAT archiver
// Copyright (C) 1998-2000  Sean Vyain, svyain@mail.tds.net
// Copyright (C) 2001-2002  Lawrence Widman, kdat@cardiothink.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

#include <stdlib.h>
#include <time.h>

#include <tqcombobox.h>
#include <tqlabel.h>
#include <tqlayout.h>
#include <tqlineedit.h>

#include <kapplication.h>
#include <kglobal.h>
#include <klocale.h>
#include <kpushbutton.h>
#include <kstdguiitem.h>

#include "Options.h"
#include "Tape.h"
#include "TapeInfoWidget.h"
#include "Util.h"

#include "TapeInfoWidget.moc"

TapeInfoWidget::TapeInfoWidget( TQWidget* parent, const char* name )
        : TQWidget( parent, name ),
          _tape( 0 )
{
    TQLabel* lbl1 = new TQLabel( i18n( "Tape name:" ), this );
    TQLabel* lbl2 = new TQLabel( i18n( "Tape size:" ), this );
    TQLabel* lbl3 = new TQLabel( i18n( "Tape ID:" ), this );
    TQLabel* lbl4 = new TQLabel( i18n( "Created on:" ), this );
    TQLabel* lbl5 = new TQLabel( i18n( "Last modified:" ), this );
    TQLabel* lbl6 = new TQLabel( i18n( "Archive count:" ), this );
    TQLabel* lbl7 = new TQLabel( i18n( "Space used:" ), this );

    int max = lbl1->tqsizeHint().width();
    if ( lbl2->tqsizeHint().width() > max ) max = lbl2->tqsizeHint().width();
    if ( lbl3->tqsizeHint().width() > max ) max = lbl3->tqsizeHint().width();
    if ( lbl4->tqsizeHint().width() > max ) max = lbl4->tqsizeHint().width();
    if ( lbl5->tqsizeHint().width() > max ) max = lbl5->tqsizeHint().width();
    if ( lbl6->tqsizeHint().width() > max ) max = lbl6->tqsizeHint().width();
    if ( lbl7->tqsizeHint().width() > max ) max = lbl7->tqsizeHint().width();

    lbl1->setFixedSize( max, lbl1->tqsizeHint().height() );
    lbl2->setFixedSize( max, lbl2->tqsizeHint().height() );
    lbl3->setFixedSize( max, lbl3->tqsizeHint().height() );
    lbl4->setFixedSize( max, lbl4->tqsizeHint().height() );
    lbl5->setFixedSize( max, lbl5->tqsizeHint().height() );
    lbl6->setFixedSize( max, lbl6->tqsizeHint().height() );
    lbl7->setFixedSize( max, lbl7->tqsizeHint().height() );

    _tapeName = new TQLineEdit( this );
    _tapeName->setFixedHeight( _tapeName->tqsizeHint().height() );

    _tapeSize = new TQLineEdit( this );
    _tapeSize->setFixedSize( 48, _tapeSize->tqsizeHint().height() );

    _tapeSizeUnits = new TQComboBox( this );
    _tapeSizeUnits->setFixedSize( 48, _tapeSizeUnits->tqsizeHint().height() );
    _tapeSizeUnits->insertItem( "MB" );
    _tapeSizeUnits->insertItem( "GB" );

    _tapeID = new TQLabel( "???", this );
    _tapeID->setFixedHeight( _tapeID->tqsizeHint().height() );

    _ctime = new TQLabel( "???", this );
    _ctime->setFixedHeight( _ctime->tqsizeHint().height() );

    _mtime = new TQLabel( "???", this );
    _mtime->setFixedHeight( _mtime->tqsizeHint().height() );

    _archiveCount = new TQLabel( "???", this );
    _archiveCount->setFixedHeight( _archiveCount->tqsizeHint().height() );

    _spaceUsed = new TQLabel( "???", this );
    _spaceUsed->setFixedHeight( _spaceUsed->tqsizeHint().height() );

    _apply = new KPushButton( KStdGuiItem::apply(), this );
    _apply->setFixedSize( 80, _apply->tqsizeHint().height() );
    _apply->setEnabled( FALSE );

    TQVBoxLayout* l1 = new TQVBoxLayout( this, 4, 4 );

    TQHBoxLayout* l1_1 = new TQHBoxLayout();
    l1->addLayout( l1_1 );
    l1_1->addWidget( lbl1 );
    l1_1->addWidget( _tapeName, 1 );

    TQHBoxLayout* l1_2 = new TQHBoxLayout();
    l1->addLayout( l1_2 );
    l1_2->addWidget( lbl2 );
    l1_2->addWidget( _tapeSize );
    l1_2->addWidget( _tapeSizeUnits );
    l1_2->addStretch( 1 );

    TQHBoxLayout* l1_3 = new TQHBoxLayout();
    l1->addLayout( l1_3 );
    l1_3->addWidget( lbl3 );
    l1_3->addWidget( _tapeID );

    TQHBoxLayout* l1_4 = new TQHBoxLayout();
    l1->addLayout( l1_4 );
    l1_4->addWidget( lbl4 );
    l1_4->addWidget( _ctime );

    TQHBoxLayout* l1_5 = new TQHBoxLayout();
    l1->addLayout( l1_5 );
    l1_5->addWidget( lbl5 );
    l1_5->addWidget( _mtime );

    TQHBoxLayout* l1_6 = new TQHBoxLayout();
    l1->addLayout( l1_6 );
    l1_6->addWidget( lbl6 );
    l1_6->addWidget( _archiveCount );

    TQHBoxLayout* l1_7 = new TQHBoxLayout();
    l1->addLayout( l1_7 );
    l1_7->addWidget( lbl7 );
    l1_7->addWidget( _spaceUsed );

    l1->addStretch( 1 );

    TQHBoxLayout* l1_8 = new TQHBoxLayout();
    l1->addLayout( l1_8 );
    l1_8->addStretch( 1 );
    l1_8->addWidget( _apply );

    connect( _tapeName     , TQT_SIGNAL( textChanged( const TQString & ) ), this, TQT_SLOT( slotTextChanged( const TQString & ) ) );
    connect( _tapeSize     , TQT_SIGNAL( textChanged( const TQString & ) ), this, TQT_SLOT( slotTextChanged( const TQString & ) ) );
    connect( _tapeSizeUnits, TQT_SIGNAL( activated( int ) )          , this, TQT_SLOT( slotActivated( int ) ) );
    connect( _apply        , TQT_SIGNAL( clicked() )                 , this, TQT_SLOT( slotApply() ) );
}

TapeInfoWidget::~TapeInfoWidget()
{
}

void TapeInfoWidget::setTape( Tape* tape )
{
    _tape = tape;

    if ( !_tape ) {
        return;
    }

    _tapeName->setText( _tape->getName() );

    int size = _tape->getSize();
    if ( ( size >= 1024*1024 ) && ( size % ( 1024*1024 ) == 0 ) ) {
        // GB
        size /= 1024*1024;
        _tapeSizeUnits->setCurrentItem( 1 );
    } else {
        // MB
        size /= 1024;
        _tapeSizeUnits->setCurrentItem( 0 );
    }
    TQString tmp;
    tmp.setNum( size );
    _tapeSize->setText( tmp );

    _tapeID->setText( _tape->getID() );

    time_t tm = _tape->getCTime();
    tmp = ctime( &tm );
    tmp = tmp.stripWhiteSpace();
    _ctime->setText( tmp );

    tm = _tape->getMTime();
    tmp = ctime( &tm );
    tmp = tmp.stripWhiteSpace();
    _mtime->setText( tmp );

    tmp.setNum( _tape->getChildren().count() );
    _archiveCount->setText( tmp );

    TQPtrListIterator<Archive> i( _tape->getChildren() );
    int used = 1;
    for ( ; i.current(); ++i ) {
        used += i.current()->getEndBlock();
    }
    int blockSize = Options::instance()->getTapeBlockSize();
    if ( blockSize < 1024 ) {
        used /= 1024 / blockSize;
    } else if ( blockSize > 1024 ) {
        used *= blockSize / 1024;
    }
    if ( _tape->getSize() > 0 ) {
        tmp = TQString::tqfromLatin1( "%1 / %2 (%3%)")
          .tqarg(Util::kbytesToString( used ))
          .tqarg(Util::kbytesToString( _tape->getSize() ))
          .tqarg(used * 100 / _tape->getSize() );
    } else {
        tmp = Util::kbytesToString( used );
    }
    _spaceUsed->setText( tmp );
}

bool TapeInfoWidget::isModified()
{
    if ( _tape->getName() != _tapeName->text() ) {
        return TRUE;
    }

    int size = (int)KGlobal::locale()->readNumber( _tapeSize->text() );
    if ( _tapeSizeUnits->currentItem() == 0 ) {
        // MB
        size *= 1024;
    } else {
        // GB
        size *= 1024*1024;
    }

    return _tape->getSize() != size;
}

void TapeInfoWidget::slotTextChanged( const TQString & )
{
    if ( !_tape ) {
        return;
    }

    _apply->setEnabled( isModified() );
}

void TapeInfoWidget::slotActivated( int )
{
    if ( !_tape ) {
        return;
    }

    _apply->setEnabled( isModified() );
}

void TapeInfoWidget::slotApply()
{
    if ( !_tape ) {
        return;
    }

    int size = (int)KGlobal::locale()->readNumber( _tapeSize->text() );
    if ( _tapeSizeUnits->currentItem() == 0 ) {
        // MB
        size *= 1024;
    } else {
        // GB
        size *= 1024*1024;
    }

    if ( _tape->getName() != _tapeName->text() ) {
        _tape->setName( _tapeName->text() );
    }

    if ( size != _tape->getSize() ) {
        _tape->setSize( size );
    }

    _apply->setEnabled( FALSE );
}
