"""
Tests of various shape generation against reference examples.
"""

from pathlib import Path
import subprocess
import sys

# Import helper function to compare graphs from tests/regressions_tests
sys.path.insert(0, str(Path(__file__).resolve().parent.parent))
from regression_test_helpers import compare_graphs \
  # pylint: disable=import-error,wrong-import-position

shapes = [
    "box",
    "polygon",
    "ellipse",
    "oval",
    "circle",
    "point",
    "egg",
    "triangle",
    "none",
    "plaintext",
    "plain",
    "diamond",
    "trapezium",
    "parallelogram",
    "house",
    "pentagon",
    "hexagon",
    "septagon",
    "octagon",
    "note",
    "tab",
    "folder",
    "box3d",
    "component",
    "cylinder",
    "rect",
    "rectangle",
    "square",
    "star",
    "doublecircle",
    "doubleoctagon",
    "tripleoctagon",
    "invtriangle",
    "invtrapezium",
    "invhouse",
    "underline",
    "Mdiamond",
    "Msquare",
    "Mcircle",
    # biological circuit shapes
    # gene expression symbols
    "promoter",
    "cds",
    "terminator",
    "utr",
    "insulator",
    "ribosite",
    "rnastab",
    "proteasesite",
    "proteinstab",
    # dna construction symbols
    "primersite",
    "restrictionsite",
    "fivepoverhang",
    "threepoverhang",
    "noverhang",
    "assembly",
    "signature",
    "rpromoter",
    "larrow",
    "rarrow",
    "lpromoter"
]

output_types = [
    "gv",
    "svg",
    "xdot"
]

def generate_shape_graph(shape, output_type):
  """
  Generate a graph of the given shape and output format.
  """
  if not Path("output").exists():
    Path("output").mkdir(parents=True)

  output_file = Path("output") / f"{shape}.{output_type}"
  input_graph = f'graph G {{ a [label="" shape={shape}] }}'
  try:
    subprocess.run(["dot", f"-T{output_type}", "-o", output_file],
                   input=input_graph.encode("utf_8"), check=True)
  except subprocess.CalledProcessError:
    print(f"An error occurred while generating: {output_file}")
    sys.exit(1)

  if output_type == "svg":
    # Remove the number in "Generated by graphviz version <number>"
    # to able to compare the output to the reference. This version
    # number is different for every Graphviz compilation.
    with open(output_file, "rt", encoding="utf-8") as file:
      lines = file.readlines()

    with open(output_file, "wt", encoding="utf-8") as file:
      for line in lines:
        if "<!-- Generated by graphviz version " in line:
          file.write("<!-- Generated by graphviz version\n")
        else:
          file.write(line)

failures = 0
for s in shapes:
  for o in output_types:
    generate_shape_graph(s, o)
    if not compare_graphs(s, o):
      failures += 1

print("")
print('Results for "shapes" regression test:')
print(f"    Number of tests: {len(shapes) * len(output_types)}")
print(f"    Number of failures: {failures}")

if not failures == 0:
  sys.exit(1)
