/* LIBGIMP - The GIMP Library
 * Copyright (C) 1995-2003 Peter Mattis and Spencer Kimball
 *
 * gimppath_pdb.c
 *
 * This library is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library.  If not, see
 * <https://www.gnu.org/licenses/>.
 */

/* NOTE: This file is auto-generated by pdbgen.pl */

#include "config.h"

#include "stamp-pdbgen.h"

#include "gimp.h"


/**
 * SECTION: gimppath
 * @title: gimppath
 * @short_description: Functions for querying and manipulating path.
 *
 * Functions for querying and manipulating path.
 **/


/**
 * gimp_path_new:
 * @image: The image.
 * @name: the name of the new path object.
 *
 * Creates a new empty path object.
 *
 * Creates a new empty path object. The path object needs to be added
 * to the image using gimp_image_insert_path().
 *
 * Returns: (transfer none):
 *          the current path object, 0 if no path exists in the image.
 *
 * Since: 2.4
 **/
GimpVectors *
gimp_path_new (GimpImage   *image,
               const gchar *name)
{
  GimpValueArray *args;
  GimpValueArray *return_vals;
  GimpVectors *path = NULL;

  args = gimp_value_array_new_from_types (NULL,
                                          GIMP_TYPE_IMAGE, image,
                                          G_TYPE_STRING, name,
                                          G_TYPE_NONE);

  return_vals = _gimp_pdb_run_procedure_array (gimp_get_pdb (),
                                               "gimp-path-new",
                                               args);
  gimp_value_array_unref (args);

  if (GIMP_VALUES_GET_ENUM (return_vals, 0) == GIMP_PDB_SUCCESS)
    path = GIMP_VALUES_GET_VECTORS (return_vals, 1);

  gimp_value_array_unref (return_vals);

  return path;
}

/**
 * gimp_path_new_from_text_layer:
 * @image: The image.
 * @layer: The text layer.
 *
 * Creates a new path object from a text layer.
 *
 * Creates a new path object from a text layer. The path object needs
 * to be added to the image using gimp_image_insert_path().
 *
 * Returns: (transfer none): The path of the text layer.
 *
 * Since: 2.6
 **/
GimpVectors *
gimp_path_new_from_text_layer (GimpImage *image,
                               GimpLayer *layer)
{
  GimpValueArray *args;
  GimpValueArray *return_vals;
  GimpVectors *path = NULL;

  args = gimp_value_array_new_from_types (NULL,
                                          GIMP_TYPE_IMAGE, image,
                                          GIMP_TYPE_LAYER, layer,
                                          G_TYPE_NONE);

  return_vals = _gimp_pdb_run_procedure_array (gimp_get_pdb (),
                                               "gimp-path-new-from-text-layer",
                                               args);
  gimp_value_array_unref (args);

  if (GIMP_VALUES_GET_ENUM (return_vals, 0) == GIMP_PDB_SUCCESS)
    path = GIMP_VALUES_GET_VECTORS (return_vals, 1);

  gimp_value_array_unref (return_vals);

  return path;
}

/**
 * gimp_path_copy:
 * @path: The path object to copy.
 *
 * Copy a path object.
 *
 * This procedure copies the specified path object and returns the
 * copy.
 *
 * Returns: (transfer none): The newly copied path object.
 *
 * Since: 2.6
 **/
GimpVectors *
gimp_path_copy (GimpVectors *path)
{
  GimpValueArray *args;
  GimpValueArray *return_vals;
  GimpVectors *path_copy = NULL;

  args = gimp_value_array_new_from_types (NULL,
                                          GIMP_TYPE_VECTORS, path,
                                          G_TYPE_NONE);

  return_vals = _gimp_pdb_run_procedure_array (gimp_get_pdb (),
                                               "gimp-path-copy",
                                               args);
  gimp_value_array_unref (args);

  if (GIMP_VALUES_GET_ENUM (return_vals, 0) == GIMP_PDB_SUCCESS)
    path_copy = GIMP_VALUES_GET_VECTORS (return_vals, 1);

  gimp_value_array_unref (return_vals);

  return path_copy;
}

/**
 * gimp_path_get_strokes:
 * @path: The path object.
 * @num_strokes: (out): The number of strokes returned.
 *
 * List the strokes associated with the passed path.
 *
 * Returns an Array with the stroke-IDs associated with the passed
 * path.
 *
 * Returns: (array length=num_strokes) (element-type gint32) (transfer full):
 *          List of the strokes belonging to the path.
 *          The returned value must be freed with g_free().
 *
 * Since: 2.4
 **/
gint *
gimp_path_get_strokes (GimpVectors *path,
                       gint        *num_strokes)
{
  GimpValueArray *args;
  GimpValueArray *return_vals;
  gint *stroke_ids = NULL;

  args = gimp_value_array_new_from_types (NULL,
                                          GIMP_TYPE_VECTORS, path,
                                          G_TYPE_NONE);

  return_vals = _gimp_pdb_run_procedure_array (gimp_get_pdb (),
                                               "gimp-path-get-strokes",
                                               args);
  gimp_value_array_unref (args);

  *num_strokes = 0;

  if (GIMP_VALUES_GET_ENUM (return_vals, 0) == GIMP_PDB_SUCCESS)
    {
      *num_strokes = GIMP_VALUES_GET_INT (return_vals, 1);
      stroke_ids = GIMP_VALUES_DUP_INT32_ARRAY (return_vals, 2);
    }

  gimp_value_array_unref (return_vals);

  return stroke_ids;
}

/**
 * gimp_path_stroke_get_length:
 * @path: The path object.
 * @stroke_id: The stroke ID.
 * @precision: The precision used for approximating straight portions of the stroke.
 *
 * Measure the length of the given stroke.
 *
 * Measure the length of the given stroke.
 *
 * Returns: The length (in pixels) of the given stroke.
 *
 * Since: 2.4
 **/
gdouble
gimp_path_stroke_get_length (GimpVectors *path,
                             gint         stroke_id,
                             gdouble      precision)
{
  GimpValueArray *args;
  GimpValueArray *return_vals;
  gdouble length = 0.0;

  args = gimp_value_array_new_from_types (NULL,
                                          GIMP_TYPE_VECTORS, path,
                                          G_TYPE_INT, stroke_id,
                                          G_TYPE_DOUBLE, precision,
                                          G_TYPE_NONE);

  return_vals = _gimp_pdb_run_procedure_array (gimp_get_pdb (),
                                               "gimp-path-stroke-get-length",
                                               args);
  gimp_value_array_unref (args);

  if (GIMP_VALUES_GET_ENUM (return_vals, 0) == GIMP_PDB_SUCCESS)
    length = GIMP_VALUES_GET_DOUBLE (return_vals, 1);

  gimp_value_array_unref (return_vals);

  return length;
}

/**
 * gimp_path_stroke_get_point_at_dist:
 * @path: The path object.
 * @stroke_id: The stroke ID.
 * @dist: The given distance.
 * @precision: The precision used for the approximation.
 * @x_point: (out): The x position of the point.
 * @y_point: (out): The y position of the point.
 * @slope: (out): The slope (dy / dx) at the specified point.
 * @valid: (out): Indicator for the validity of the returned data.
 *
 * Get point at a specified distance along the stroke.
 *
 * This will return the x,y position of a point at a given distance
 * along the stroke. The distance will be obtained by first digitizing
 * the curve internally and then walking along the curve. For a closed
 * stroke the start of the path is the first point on the path that was
 * created. This might not be obvious. If the stroke is not long
 * enough, a \"valid\" flag will be FALSE.
 *
 * Returns: TRUE on success.
 *
 * Since: 2.4
 **/
gboolean
gimp_path_stroke_get_point_at_dist (GimpVectors *path,
                                    gint         stroke_id,
                                    gdouble      dist,
                                    gdouble      precision,
                                    gdouble     *x_point,
                                    gdouble     *y_point,
                                    gdouble     *slope,
                                    gboolean    *valid)
{
  GimpValueArray *args;
  GimpValueArray *return_vals;
  gboolean success = TRUE;

  args = gimp_value_array_new_from_types (NULL,
                                          GIMP_TYPE_VECTORS, path,
                                          G_TYPE_INT, stroke_id,
                                          G_TYPE_DOUBLE, dist,
                                          G_TYPE_DOUBLE, precision,
                                          G_TYPE_NONE);

  return_vals = _gimp_pdb_run_procedure_array (gimp_get_pdb (),
                                               "gimp-path-stroke-get-point-at-dist",
                                               args);
  gimp_value_array_unref (args);

  *x_point = 0.0;
  *y_point = 0.0;
  *slope = 0.0;
  *valid = FALSE;

  success = GIMP_VALUES_GET_ENUM (return_vals, 0) == GIMP_PDB_SUCCESS;

  if (success)
    {
      *x_point = GIMP_VALUES_GET_DOUBLE (return_vals, 1);
      *y_point = GIMP_VALUES_GET_DOUBLE (return_vals, 2);
      *slope = GIMP_VALUES_GET_DOUBLE (return_vals, 3);
      *valid = GIMP_VALUES_GET_BOOLEAN (return_vals, 4);
    }

  gimp_value_array_unref (return_vals);

  return success;
}

/**
 * gimp_path_remove_stroke:
 * @path: The path object.
 * @stroke_id: The stroke ID.
 *
 * remove the stroke from a path object.
 *
 * Remove the stroke from a path object.
 *
 * Returns: TRUE on success.
 *
 * Since: 2.4
 **/
gboolean
gimp_path_remove_stroke (GimpVectors *path,
                         gint         stroke_id)
{
  GimpValueArray *args;
  GimpValueArray *return_vals;
  gboolean success = TRUE;

  args = gimp_value_array_new_from_types (NULL,
                                          GIMP_TYPE_VECTORS, path,
                                          G_TYPE_INT, stroke_id,
                                          G_TYPE_NONE);

  return_vals = _gimp_pdb_run_procedure_array (gimp_get_pdb (),
                                               "gimp-path-remove-stroke",
                                               args);
  gimp_value_array_unref (args);

  success = GIMP_VALUES_GET_ENUM (return_vals, 0) == GIMP_PDB_SUCCESS;

  gimp_value_array_unref (return_vals);

  return success;
}

/**
 * gimp_path_stroke_close:
 * @path: The path object.
 * @stroke_id: The stroke ID.
 *
 * closes the specified stroke.
 *
 * Closes the specified stroke.
 *
 * Returns: TRUE on success.
 *
 * Since: 2.4
 **/
gboolean
gimp_path_stroke_close (GimpVectors *path,
                        gint         stroke_id)
{
  GimpValueArray *args;
  GimpValueArray *return_vals;
  gboolean success = TRUE;

  args = gimp_value_array_new_from_types (NULL,
                                          GIMP_TYPE_VECTORS, path,
                                          G_TYPE_INT, stroke_id,
                                          G_TYPE_NONE);

  return_vals = _gimp_pdb_run_procedure_array (gimp_get_pdb (),
                                               "gimp-path-stroke-close",
                                               args);
  gimp_value_array_unref (args);

  success = GIMP_VALUES_GET_ENUM (return_vals, 0) == GIMP_PDB_SUCCESS;

  gimp_value_array_unref (return_vals);

  return success;
}

/**
 * gimp_path_stroke_reverse:
 * @path: The path object.
 * @stroke_id: The stroke ID.
 *
 * reverses the specified stroke.
 *
 * Reverses the specified stroke.
 *
 * Returns: TRUE on success.
 *
 * Since: 3.0
 **/
gboolean
gimp_path_stroke_reverse (GimpVectors *path,
                          gint         stroke_id)
{
  GimpValueArray *args;
  GimpValueArray *return_vals;
  gboolean success = TRUE;

  args = gimp_value_array_new_from_types (NULL,
                                          GIMP_TYPE_VECTORS, path,
                                          G_TYPE_INT, stroke_id,
                                          G_TYPE_NONE);

  return_vals = _gimp_pdb_run_procedure_array (gimp_get_pdb (),
                                               "gimp-path-stroke-reverse",
                                               args);
  gimp_value_array_unref (args);

  success = GIMP_VALUES_GET_ENUM (return_vals, 0) == GIMP_PDB_SUCCESS;

  gimp_value_array_unref (return_vals);

  return success;
}

/**
 * gimp_path_stroke_translate:
 * @path: The path object.
 * @stroke_id: The stroke ID.
 * @off_x: Offset in x direction.
 * @off_y: Offset in y direction.
 *
 * translate the given stroke.
 *
 * Translate the given stroke.
 *
 * Returns: TRUE on success.
 *
 * Since: 2.4
 **/
gboolean
gimp_path_stroke_translate (GimpVectors *path,
                            gint         stroke_id,
                            gdouble      off_x,
                            gdouble      off_y)
{
  GimpValueArray *args;
  GimpValueArray *return_vals;
  gboolean success = TRUE;

  args = gimp_value_array_new_from_types (NULL,
                                          GIMP_TYPE_VECTORS, path,
                                          G_TYPE_INT, stroke_id,
                                          G_TYPE_DOUBLE, off_x,
                                          G_TYPE_DOUBLE, off_y,
                                          G_TYPE_NONE);

  return_vals = _gimp_pdb_run_procedure_array (gimp_get_pdb (),
                                               "gimp-path-stroke-translate",
                                               args);
  gimp_value_array_unref (args);

  success = GIMP_VALUES_GET_ENUM (return_vals, 0) == GIMP_PDB_SUCCESS;

  gimp_value_array_unref (return_vals);

  return success;
}

/**
 * gimp_path_stroke_scale:
 * @path: The path object.
 * @stroke_id: The stroke ID.
 * @scale_x: Scale factor in x direction.
 * @scale_y: Scale factor in y direction.
 *
 * scales the given stroke.
 *
 * Scale the given stroke.
 *
 * Returns: TRUE on success.
 *
 * Since: 2.4
 **/
gboolean
gimp_path_stroke_scale (GimpVectors *path,
                        gint         stroke_id,
                        gdouble      scale_x,
                        gdouble      scale_y)
{
  GimpValueArray *args;
  GimpValueArray *return_vals;
  gboolean success = TRUE;

  args = gimp_value_array_new_from_types (NULL,
                                          GIMP_TYPE_VECTORS, path,
                                          G_TYPE_INT, stroke_id,
                                          G_TYPE_DOUBLE, scale_x,
                                          G_TYPE_DOUBLE, scale_y,
                                          G_TYPE_NONE);

  return_vals = _gimp_pdb_run_procedure_array (gimp_get_pdb (),
                                               "gimp-path-stroke-scale",
                                               args);
  gimp_value_array_unref (args);

  success = GIMP_VALUES_GET_ENUM (return_vals, 0) == GIMP_PDB_SUCCESS;

  gimp_value_array_unref (return_vals);

  return success;
}

/**
 * gimp_path_stroke_rotate:
 * @path: The path object.
 * @stroke_id: The stroke ID.
 * @center_x: X coordinate of the rotation center.
 * @center_y: Y coordinate of the rotation center.
 * @angle: angle to rotate about.
 *
 * rotates the given stroke.
 *
 * Rotates the given stroke around given center by angle (in degrees).
 *
 * Returns: TRUE on success.
 *
 * Since: 2.4
 **/
gboolean
gimp_path_stroke_rotate (GimpVectors *path,
                         gint         stroke_id,
                         gdouble      center_x,
                         gdouble      center_y,
                         gdouble      angle)
{
  GimpValueArray *args;
  GimpValueArray *return_vals;
  gboolean success = TRUE;

  args = gimp_value_array_new_from_types (NULL,
                                          GIMP_TYPE_VECTORS, path,
                                          G_TYPE_INT, stroke_id,
                                          G_TYPE_DOUBLE, center_x,
                                          G_TYPE_DOUBLE, center_y,
                                          G_TYPE_DOUBLE, angle,
                                          G_TYPE_NONE);

  return_vals = _gimp_pdb_run_procedure_array (gimp_get_pdb (),
                                               "gimp-path-stroke-rotate",
                                               args);
  gimp_value_array_unref (args);

  success = GIMP_VALUES_GET_ENUM (return_vals, 0) == GIMP_PDB_SUCCESS;

  gimp_value_array_unref (return_vals);

  return success;
}

/**
 * gimp_path_stroke_flip:
 * @path: The path object.
 * @stroke_id: The stroke ID.
 * @flip_type: Flip orientation, either vertical or horizontal.
 * @axis: axis coordinate about which to flip, in pixels.
 *
 * flips the given stroke.
 *
 * Rotates the given stroke around given center by angle (in degrees).
 *
 * Returns: TRUE on success.
 *
 * Since: 2.4
 **/
gboolean
gimp_path_stroke_flip (GimpVectors         *path,
                       gint                 stroke_id,
                       GimpOrientationType  flip_type,
                       gdouble              axis)
{
  GimpValueArray *args;
  GimpValueArray *return_vals;
  gboolean success = TRUE;

  args = gimp_value_array_new_from_types (NULL,
                                          GIMP_TYPE_VECTORS, path,
                                          G_TYPE_INT, stroke_id,
                                          GIMP_TYPE_ORIENTATION_TYPE, flip_type,
                                          G_TYPE_DOUBLE, axis,
                                          G_TYPE_NONE);

  return_vals = _gimp_pdb_run_procedure_array (gimp_get_pdb (),
                                               "gimp-path-stroke-flip",
                                               args);
  gimp_value_array_unref (args);

  success = GIMP_VALUES_GET_ENUM (return_vals, 0) == GIMP_PDB_SUCCESS;

  gimp_value_array_unref (return_vals);

  return success;
}

/**
 * gimp_path_stroke_flip_free:
 * @path: The path object.
 * @stroke_id: The stroke ID.
 * @x1: X coordinate of the first point of the flipping axis.
 * @y1: Y coordinate of the first point of the flipping axis.
 * @x2: X coordinate of the second point of the flipping axis.
 * @y2: Y coordinate of the second point of the flipping axis.
 *
 * flips the given stroke about an arbitrary axis.
 *
 * Flips the given stroke about an arbitrary axis. Axis is defined by
 * two coordinates in the image (in pixels), through which the flipping
 * axis passes.
 *
 * Returns: TRUE on success.
 *
 * Since: 2.4
 **/
gboolean
gimp_path_stroke_flip_free (GimpVectors *path,
                            gint         stroke_id,
                            gdouble      x1,
                            gdouble      y1,
                            gdouble      x2,
                            gdouble      y2)
{
  GimpValueArray *args;
  GimpValueArray *return_vals;
  gboolean success = TRUE;

  args = gimp_value_array_new_from_types (NULL,
                                          GIMP_TYPE_VECTORS, path,
                                          G_TYPE_INT, stroke_id,
                                          G_TYPE_DOUBLE, x1,
                                          G_TYPE_DOUBLE, y1,
                                          G_TYPE_DOUBLE, x2,
                                          G_TYPE_DOUBLE, y2,
                                          G_TYPE_NONE);

  return_vals = _gimp_pdb_run_procedure_array (gimp_get_pdb (),
                                               "gimp-path-stroke-flip-free",
                                               args);
  gimp_value_array_unref (args);

  success = GIMP_VALUES_GET_ENUM (return_vals, 0) == GIMP_PDB_SUCCESS;

  gimp_value_array_unref (return_vals);

  return success;
}

/**
 * gimp_path_stroke_get_points:
 * @path: The path object.
 * @stroke_id: The stroke ID.
 * @num_points: (out): The number of floats returned.
 * @controlpoints: (out) (array length=num_points) (element-type gdouble) (transfer full): List of the control points for the stroke (x0, y0, x1, y1, ...).
 * @closed: (out): Whether the stroke is closed or not.
 *
 * returns the control points of a stroke.
 *
 * returns the control points of a stroke. The interpretation of the
 * coordinates returned depends on the type of the stroke. For Gimp 2.4
 * this is always a bezier stroke, where the coordinates are the
 * control points.
 *
 * Returns: type of the stroke (always GIMP_VECTORS_STROKE_TYPE_BEZIER for now).
 *
 * Since: 2.4
 **/
GimpVectorsStrokeType
gimp_path_stroke_get_points (GimpVectors  *path,
                             gint          stroke_id,
                             gint         *num_points,
                             gdouble     **controlpoints,
                             gboolean     *closed)
{
  GimpValueArray *args;
  GimpValueArray *return_vals;
  GimpVectorsStrokeType type = 0;

  args = gimp_value_array_new_from_types (NULL,
                                          GIMP_TYPE_VECTORS, path,
                                          G_TYPE_INT, stroke_id,
                                          G_TYPE_NONE);

  return_vals = _gimp_pdb_run_procedure_array (gimp_get_pdb (),
                                               "gimp-path-stroke-get-points",
                                               args);
  gimp_value_array_unref (args);

  *num_points = 0;

  if (GIMP_VALUES_GET_ENUM (return_vals, 0) == GIMP_PDB_SUCCESS)
    {
      type = GIMP_VALUES_GET_ENUM (return_vals, 1);
      *num_points = GIMP_VALUES_GET_INT (return_vals, 2);
      *controlpoints = GIMP_VALUES_DUP_FLOAT_ARRAY (return_vals, 3);
      *closed = GIMP_VALUES_GET_BOOLEAN (return_vals, 4);
    }

  gimp_value_array_unref (return_vals);

  return type;
}

/**
 * gimp_path_stroke_new_from_points:
 * @path: The path object.
 * @type: type of the stroke (always GIMP_VECTORS_STROKE_TYPE_BEZIER for now).
 * @num_points: The number of elements in the array, i.e. the number of controlpoints in the stroke * 2 (x- and y-coordinate).
 * @controlpoints: (array length=num_points) (element-type gdouble): List of the x- and y-coordinates of the control points.
 * @closed: Whether the stroke is to be closed or not.
 *
 * Adds a stroke of a given type to the path object.
 *
 * Adds a stroke of a given type to the path object. The coordinates of
 * the control points can be specified. For now only strokes of the
 * type GIMP_VECTORS_STROKE_TYPE_BEZIER are supported. The control
 * points are specified as a pair of float values for the x- and
 * y-coordinate. The Bezier stroke type needs a multiple of three
 * control points. Each Bezier segment endpoint (anchor, A) has two
 * additional control points (C) associated. They are specified in the
 * order CACCACCAC...
 *
 * Returns: The stroke ID of the newly created stroke.
 *
 * Since: 2.4
 **/
gint
gimp_path_stroke_new_from_points (GimpVectors           *path,
                                  GimpVectorsStrokeType  type,
                                  gint                   num_points,
                                  const gdouble         *controlpoints,
                                  gboolean               closed)
{
  GimpValueArray *args;
  GimpValueArray *return_vals;
  gint stroke_id = 0;

  args = gimp_value_array_new_from_types (NULL,
                                          GIMP_TYPE_VECTORS, path,
                                          GIMP_TYPE_VECTORS_STROKE_TYPE, type,
                                          G_TYPE_INT, num_points,
                                          GIMP_TYPE_FLOAT_ARRAY, NULL,
                                          G_TYPE_BOOLEAN, closed,
                                          G_TYPE_NONE);
  gimp_value_set_float_array (gimp_value_array_index (args, 3), controlpoints, num_points);

  return_vals = _gimp_pdb_run_procedure_array (gimp_get_pdb (),
                                               "gimp-path-stroke-new-from-points",
                                               args);
  gimp_value_array_unref (args);

  if (GIMP_VALUES_GET_ENUM (return_vals, 0) == GIMP_PDB_SUCCESS)
    stroke_id = GIMP_VALUES_GET_INT (return_vals, 1);

  gimp_value_array_unref (return_vals);

  return stroke_id;
}

/**
 * gimp_path_stroke_interpolate:
 * @path: The path object.
 * @stroke_id: The stroke ID.
 * @precision: The precision used for the approximation.
 * @num_coords: (out): The number of floats returned.
 * @closed: (out): Whether the stroke is closed or not.
 *
 * returns polygonal approximation of the stroke.
 *
 * returns polygonal approximation of the stroke.
 *
 * Returns: (array length=num_coords) (element-type gdouble) (transfer full):
 *          List of the coords along the path (x0, y0, x1, y1, ...).
 *          The returned value must be freed with g_free().
 *
 * Since: 2.4
 **/
gdouble *
gimp_path_stroke_interpolate (GimpVectors *path,
                              gint         stroke_id,
                              gdouble      precision,
                              gint        *num_coords,
                              gboolean    *closed)
{
  GimpValueArray *args;
  GimpValueArray *return_vals;
  gdouble *coords = NULL;

  args = gimp_value_array_new_from_types (NULL,
                                          GIMP_TYPE_VECTORS, path,
                                          G_TYPE_INT, stroke_id,
                                          G_TYPE_DOUBLE, precision,
                                          G_TYPE_NONE);

  return_vals = _gimp_pdb_run_procedure_array (gimp_get_pdb (),
                                               "gimp-path-stroke-interpolate",
                                               args);
  gimp_value_array_unref (args);

  *num_coords = 0;

  if (GIMP_VALUES_GET_ENUM (return_vals, 0) == GIMP_PDB_SUCCESS)
    {
      *num_coords = GIMP_VALUES_GET_INT (return_vals, 1);
      coords = GIMP_VALUES_DUP_FLOAT_ARRAY (return_vals, 2);
      *closed = GIMP_VALUES_GET_BOOLEAN (return_vals, 3);
    }

  gimp_value_array_unref (return_vals);

  return coords;
}

/**
 * gimp_path_bezier_stroke_new_moveto:
 * @path: The path object.
 * @x0: The x-coordinate of the moveto.
 * @y0: The y-coordinate of the moveto.
 *
 * Adds a bezier stroke with a single moveto to the path object.
 *
 * Adds a bezier stroke with a single moveto to the path object.
 *
 * Returns: The resulting stroke.
 *
 * Since: 2.4
 **/
gint
gimp_path_bezier_stroke_new_moveto (GimpVectors *path,
                                    gdouble      x0,
                                    gdouble      y0)
{
  GimpValueArray *args;
  GimpValueArray *return_vals;
  gint stroke_id = 0;

  args = gimp_value_array_new_from_types (NULL,
                                          GIMP_TYPE_VECTORS, path,
                                          G_TYPE_DOUBLE, x0,
                                          G_TYPE_DOUBLE, y0,
                                          G_TYPE_NONE);

  return_vals = _gimp_pdb_run_procedure_array (gimp_get_pdb (),
                                               "gimp-path-bezier-stroke-new-moveto",
                                               args);
  gimp_value_array_unref (args);

  if (GIMP_VALUES_GET_ENUM (return_vals, 0) == GIMP_PDB_SUCCESS)
    stroke_id = GIMP_VALUES_GET_INT (return_vals, 1);

  gimp_value_array_unref (return_vals);

  return stroke_id;
}

/**
 * gimp_path_bezier_stroke_lineto:
 * @path: The path object.
 * @stroke_id: The stroke ID.
 * @x0: The x-coordinate of the lineto.
 * @y0: The y-coordinate of the lineto.
 *
 * Extends a bezier stroke with a lineto.
 *
 * Extends a bezier stroke with a lineto.
 *
 * Returns: TRUE on success.
 *
 * Since: 2.4
 **/
gboolean
gimp_path_bezier_stroke_lineto (GimpVectors *path,
                                gint         stroke_id,
                                gdouble      x0,
                                gdouble      y0)
{
  GimpValueArray *args;
  GimpValueArray *return_vals;
  gboolean success = TRUE;

  args = gimp_value_array_new_from_types (NULL,
                                          GIMP_TYPE_VECTORS, path,
                                          G_TYPE_INT, stroke_id,
                                          G_TYPE_DOUBLE, x0,
                                          G_TYPE_DOUBLE, y0,
                                          G_TYPE_NONE);

  return_vals = _gimp_pdb_run_procedure_array (gimp_get_pdb (),
                                               "gimp-path-bezier-stroke-lineto",
                                               args);
  gimp_value_array_unref (args);

  success = GIMP_VALUES_GET_ENUM (return_vals, 0) == GIMP_PDB_SUCCESS;

  gimp_value_array_unref (return_vals);

  return success;
}

/**
 * gimp_path_bezier_stroke_conicto:
 * @path: The path object.
 * @stroke_id: The stroke ID.
 * @x0: The x-coordinate of the control point.
 * @y0: The y-coordinate of the control point.
 * @x1: The x-coordinate of the end point.
 * @y1: The y-coordinate of the end point.
 *
 * Extends a bezier stroke with a conic bezier spline.
 *
 * Extends a bezier stroke with a conic bezier spline. Actually a cubic
 * bezier spline gets added that realizes the shape of a conic bezier
 * spline.
 *
 * Returns: TRUE on success.
 *
 * Since: 2.4
 **/
gboolean
gimp_path_bezier_stroke_conicto (GimpVectors *path,
                                 gint         stroke_id,
                                 gdouble      x0,
                                 gdouble      y0,
                                 gdouble      x1,
                                 gdouble      y1)
{
  GimpValueArray *args;
  GimpValueArray *return_vals;
  gboolean success = TRUE;

  args = gimp_value_array_new_from_types (NULL,
                                          GIMP_TYPE_VECTORS, path,
                                          G_TYPE_INT, stroke_id,
                                          G_TYPE_DOUBLE, x0,
                                          G_TYPE_DOUBLE, y0,
                                          G_TYPE_DOUBLE, x1,
                                          G_TYPE_DOUBLE, y1,
                                          G_TYPE_NONE);

  return_vals = _gimp_pdb_run_procedure_array (gimp_get_pdb (),
                                               "gimp-path-bezier-stroke-conicto",
                                               args);
  gimp_value_array_unref (args);

  success = GIMP_VALUES_GET_ENUM (return_vals, 0) == GIMP_PDB_SUCCESS;

  gimp_value_array_unref (return_vals);

  return success;
}

/**
 * gimp_path_bezier_stroke_cubicto:
 * @path: The path object.
 * @stroke_id: The stroke ID.
 * @x0: The x-coordinate of the first control point.
 * @y0: The y-coordinate of the first control point.
 * @x1: The x-coordinate of the second control point.
 * @y1: The y-coordinate of the second control point.
 * @x2: The x-coordinate of the end point.
 * @y2: The y-coordinate of the end point.
 *
 * Extends a bezier stroke with a cubic bezier spline.
 *
 * Extends a bezier stroke with a cubic bezier spline.
 *
 * Returns: TRUE on success.
 *
 * Since: 2.4
 **/
gboolean
gimp_path_bezier_stroke_cubicto (GimpVectors *path,
                                 gint         stroke_id,
                                 gdouble      x0,
                                 gdouble      y0,
                                 gdouble      x1,
                                 gdouble      y1,
                                 gdouble      x2,
                                 gdouble      y2)
{
  GimpValueArray *args;
  GimpValueArray *return_vals;
  gboolean success = TRUE;

  args = gimp_value_array_new_from_types (NULL,
                                          GIMP_TYPE_VECTORS, path,
                                          G_TYPE_INT, stroke_id,
                                          G_TYPE_DOUBLE, x0,
                                          G_TYPE_DOUBLE, y0,
                                          G_TYPE_DOUBLE, x1,
                                          G_TYPE_DOUBLE, y1,
                                          G_TYPE_DOUBLE, x2,
                                          G_TYPE_DOUBLE, y2,
                                          G_TYPE_NONE);

  return_vals = _gimp_pdb_run_procedure_array (gimp_get_pdb (),
                                               "gimp-path-bezier-stroke-cubicto",
                                               args);
  gimp_value_array_unref (args);

  success = GIMP_VALUES_GET_ENUM (return_vals, 0) == GIMP_PDB_SUCCESS;

  gimp_value_array_unref (return_vals);

  return success;
}

/**
 * gimp_path_bezier_stroke_new_ellipse:
 * @path: The path object.
 * @x0: The x-coordinate of the center.
 * @y0: The y-coordinate of the center.
 * @radius_x: The radius in x direction.
 * @radius_y: The radius in y direction.
 * @angle: The angle the x-axis of the ellipse (radians, counterclockwise).
 *
 * Adds a bezier stroke describing an ellipse the path object.
 *
 * Adds a bezier stroke describing an ellipse on the path object.
 *
 * Returns: The resulting stroke.
 *
 * Since: 2.4
 **/
gint
gimp_path_bezier_stroke_new_ellipse (GimpVectors *path,
                                     gdouble      x0,
                                     gdouble      y0,
                                     gdouble      radius_x,
                                     gdouble      radius_y,
                                     gdouble      angle)
{
  GimpValueArray *args;
  GimpValueArray *return_vals;
  gint stroke_id = 0;

  args = gimp_value_array_new_from_types (NULL,
                                          GIMP_TYPE_VECTORS, path,
                                          G_TYPE_DOUBLE, x0,
                                          G_TYPE_DOUBLE, y0,
                                          G_TYPE_DOUBLE, radius_x,
                                          G_TYPE_DOUBLE, radius_y,
                                          G_TYPE_DOUBLE, angle,
                                          G_TYPE_NONE);

  return_vals = _gimp_pdb_run_procedure_array (gimp_get_pdb (),
                                               "gimp-path-bezier-stroke-new-ellipse",
                                               args);
  gimp_value_array_unref (args);

  if (GIMP_VALUES_GET_ENUM (return_vals, 0) == GIMP_PDB_SUCCESS)
    stroke_id = GIMP_VALUES_GET_INT (return_vals, 1);

  gimp_value_array_unref (return_vals);

  return stroke_id;
}

/**
 * gimp_path_import_from_file:
 * @image: The image.
 * @file: The SVG file to import.
 * @merge: Merge paths into a single path object.
 * @scale: Scale the SVG to image dimensions.
 * @num_paths: (out): The number of newly created path.
 * @path: (out) (array length=num_paths) (element-type GimpVectors) (transfer container): The list of newly created path.
 *
 * Import paths from an SVG file.
 *
 * This procedure imports paths from an SVG file. SVG elements other
 * than paths and basic shapes are ignored.
 *
 * Returns: TRUE on success.
 *
 * Since: 2.4
 **/
gboolean
gimp_path_import_from_file (GimpImage     *image,
                            GFile         *file,
                            gboolean       merge,
                            gboolean       scale,
                            gint          *num_paths,
                            GimpVectors ***path)
{
  GimpValueArray *args;
  GimpValueArray *return_vals;
  gboolean success = TRUE;

  args = gimp_value_array_new_from_types (NULL,
                                          GIMP_TYPE_IMAGE, image,
                                          G_TYPE_FILE, file,
                                          G_TYPE_BOOLEAN, merge,
                                          G_TYPE_BOOLEAN, scale,
                                          G_TYPE_NONE);

  return_vals = _gimp_pdb_run_procedure_array (gimp_get_pdb (),
                                               "gimp-path-import-from-file",
                                               args);
  gimp_value_array_unref (args);

  *num_paths = 0;
  *path = NULL;

  success = GIMP_VALUES_GET_ENUM (return_vals, 0) == GIMP_PDB_SUCCESS;

  if (success)
    {
      *num_paths = GIMP_VALUES_GET_INT (return_vals, 1);
      { GimpObjectArray *a = g_value_get_boxed (gimp_value_array_index (return_vals, 2)); if (a) *path = g_memdup2 (a->data, a->length * sizeof (gpointer)); };
    }

  gimp_value_array_unref (return_vals);

  return success;
}

/**
 * gimp_path_import_from_string:
 * @image: The image.
 * @string: A string that must be a complete and valid SVG document.
 * @length: Number of bytes in string or -1 if the string is NULL terminated.
 * @merge: Merge paths into a single path object.
 * @scale: Scale the SVG to image dimensions.
 * @num_paths: (out): The number of newly created path.
 * @path: (out) (array length=num_paths) (element-type GimpVectors) (transfer container): The list of newly created path.
 *
 * Import paths from an SVG string.
 *
 * This procedure works like gimp_vectors_import_from_file() but takes
 * a string rather than reading the SVG from a file. This allows you to
 * write scripts that generate SVG and feed it to GIMP.
 *
 * Returns: TRUE on success.
 *
 * Since: 2.4
 **/
gboolean
gimp_path_import_from_string (GimpImage     *image,
                              const gchar   *string,
                              gint           length,
                              gboolean       merge,
                              gboolean       scale,
                              gint          *num_paths,
                              GimpVectors ***path)
{
  GimpValueArray *args;
  GimpValueArray *return_vals;
  gboolean success = TRUE;

  args = gimp_value_array_new_from_types (NULL,
                                          GIMP_TYPE_IMAGE, image,
                                          G_TYPE_STRING, string,
                                          G_TYPE_INT, length,
                                          G_TYPE_BOOLEAN, merge,
                                          G_TYPE_BOOLEAN, scale,
                                          G_TYPE_NONE);

  return_vals = _gimp_pdb_run_procedure_array (gimp_get_pdb (),
                                               "gimp-path-import-from-string",
                                               args);
  gimp_value_array_unref (args);

  *num_paths = 0;
  *path = NULL;

  success = GIMP_VALUES_GET_ENUM (return_vals, 0) == GIMP_PDB_SUCCESS;

  if (success)
    {
      *num_paths = GIMP_VALUES_GET_INT (return_vals, 1);
      { GimpObjectArray *a = g_value_get_boxed (gimp_value_array_index (return_vals, 2)); if (a) *path = g_memdup2 (a->data, a->length * sizeof (gpointer)); };
    }

  gimp_value_array_unref (return_vals);

  return success;
}

/**
 * gimp_path_export_to_file:
 * @image: The image.
 * @file: The SVG file to create.
 * @path: (nullable): The path object to export, or %NULL for all in the image.
 *
 * save a path as an SVG file.
 *
 * This procedure creates an SVG file to save a Path object, that is, a
 * path. The resulting file can be edited using a vector graphics
 * application, or later reloaded into GIMP. Pass %NULL as the 'path'
 * argument to export all paths in the image.
 *
 * Returns: TRUE on success.
 *
 * Since: 2.6
 **/
gboolean
gimp_path_export_to_file (GimpImage   *image,
                          GFile       *file,
                          GimpVectors *path)
{
  GimpValueArray *args;
  GimpValueArray *return_vals;
  gboolean success = TRUE;

  args = gimp_value_array_new_from_types (NULL,
                                          GIMP_TYPE_IMAGE, image,
                                          G_TYPE_FILE, file,
                                          GIMP_TYPE_VECTORS, path,
                                          G_TYPE_NONE);

  return_vals = _gimp_pdb_run_procedure_array (gimp_get_pdb (),
                                               "gimp-path-export-to-file",
                                               args);
  gimp_value_array_unref (args);

  success = GIMP_VALUES_GET_ENUM (return_vals, 0) == GIMP_PDB_SUCCESS;

  gimp_value_array_unref (return_vals);

  return success;
}

/**
 * gimp_path_export_to_string:
 * @image: The image.
 * @path: (nullable): The path object to export, or %NULL for all in the image.
 *
 * Save a path as an SVG string.
 *
 * This procedure works like gimp_path_export_to_file() but creates a
 * string rather than a file. The string is NULL-terminated and holds a
 * complete XML document. Pass %NULL as the 'path' argument to export
 * all paths in the image.
 *
 * Returns: (transfer full):
 *          A string whose contents are a complete SVG document.
 *          The returned value must be freed with g_free().
 *
 * Since: 2.6
 **/
gchar *
gimp_path_export_to_string (GimpImage   *image,
                            GimpVectors *path)
{
  GimpValueArray *args;
  GimpValueArray *return_vals;
  gchar *string = NULL;

  args = gimp_value_array_new_from_types (NULL,
                                          GIMP_TYPE_IMAGE, image,
                                          GIMP_TYPE_VECTORS, path,
                                          G_TYPE_NONE);

  return_vals = _gimp_pdb_run_procedure_array (gimp_get_pdb (),
                                               "gimp-path-export-to-string",
                                               args);
  gimp_value_array_unref (args);

  if (GIMP_VALUES_GET_ENUM (return_vals, 0) == GIMP_PDB_SUCCESS)
    string = GIMP_VALUES_DUP_STRING (return_vals, 1);

  gimp_value_array_unref (return_vals);

  return string;
}
