/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim: set ts=8 sts=2 et sw=2 tw=80: */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#include "PersistenceType.h"

namespace mozilla {
namespace dom {
namespace quota {

namespace {

constexpr auto kPersistentCString = NS_LITERAL_CSTRING("persistent");
constexpr auto kTemporaryCString = NS_LITERAL_CSTRING("temporary");
constexpr auto kDefaultCString = NS_LITERAL_CSTRING("default");

static_assert(PERSISTENCE_TYPE_PERSISTENT == 0 &&
                  PERSISTENCE_TYPE_TEMPORARY == 1 &&
                  PERSISTENCE_TYPE_DEFAULT == 2 &&
                  PERSISTENCE_TYPE_INVALID == 3,
              "Incorrect enum values!");

template <PersistenceType type>
struct PersistenceTypeTraits;

template <>
struct PersistenceTypeTraits<PERSISTENCE_TYPE_PERSISTENT> {
  template <typename T>
  static T To();

  static bool From(const nsACString& aString) {
    return aString == kPersistentCString;
  }

  static bool From(const StorageType aStorageType) {
    return aStorageType == StorageType::Persistent;
  }

  static bool From(const int32_t aInt32) { return aInt32 == 0; }
};

template <>
nsLiteralCString
PersistenceTypeTraits<PERSISTENCE_TYPE_PERSISTENT>::To<nsLiteralCString>() {
  return kPersistentCString;
}

template <>
StorageType
PersistenceTypeTraits<PERSISTENCE_TYPE_PERSISTENT>::To<StorageType>() {
  return StorageType::Persistent;
}

template <>
struct PersistenceTypeTraits<PERSISTENCE_TYPE_TEMPORARY> {
  template <typename T>
  static T To();

  static bool From(const nsACString& aString) {
    return aString == kTemporaryCString;
  }

  static bool From(const StorageType aStorageType) {
    return aStorageType == StorageType::Temporary;
  }

  static bool From(const int32_t aInt32) { return aInt32 == 1; }
};

template <>
nsLiteralCString
PersistenceTypeTraits<PERSISTENCE_TYPE_TEMPORARY>::To<nsLiteralCString>() {
  return kTemporaryCString;
}

template <>
StorageType
PersistenceTypeTraits<PERSISTENCE_TYPE_TEMPORARY>::To<StorageType>() {
  return StorageType::Temporary;
}

template <>
struct PersistenceTypeTraits<PERSISTENCE_TYPE_DEFAULT> {
  template <typename T>
  static T To();

  static bool From(const nsACString& aString) {
    return aString == kDefaultCString;
  }

  static bool From(const StorageType aStorageType) {
    return aStorageType == StorageType::Default;
  }

  static bool From(const int32_t aInt32) { return aInt32 == 2; }
};

template <>
nsLiteralCString
PersistenceTypeTraits<PERSISTENCE_TYPE_DEFAULT>::To<nsLiteralCString>() {
  return kDefaultCString;
}

template <>
StorageType PersistenceTypeTraits<PERSISTENCE_TYPE_DEFAULT>::To<StorageType>() {
  return StorageType::Default;
}

template <typename T>
Maybe<T> TypeTo_impl(const PersistenceType aPersistenceType) {
  switch (aPersistenceType) {
    case PERSISTENCE_TYPE_PERSISTENT:
      return Some(PersistenceTypeTraits<PERSISTENCE_TYPE_PERSISTENT>::To<T>());

    case PERSISTENCE_TYPE_TEMPORARY:
      return Some(PersistenceTypeTraits<PERSISTENCE_TYPE_TEMPORARY>::To<T>());

    case PERSISTENCE_TYPE_DEFAULT:
      return Some(PersistenceTypeTraits<PERSISTENCE_TYPE_DEFAULT>::To<T>());

    default:
      return Nothing();
  }
}

template <typename T>
Maybe<PersistenceType> TypeFrom_impl(const T& aData) {
  if (PersistenceTypeTraits<PERSISTENCE_TYPE_PERSISTENT>::From(aData)) {
    return Some(PERSISTENCE_TYPE_PERSISTENT);
  }

  if (PersistenceTypeTraits<PERSISTENCE_TYPE_TEMPORARY>::From(aData)) {
    return Some(PERSISTENCE_TYPE_TEMPORARY);
  }

  if (PersistenceTypeTraits<PERSISTENCE_TYPE_DEFAULT>::From(aData)) {
    return Some(PERSISTENCE_TYPE_DEFAULT);
  }

  return Nothing();
}

void BadPersistenceType() { MOZ_CRASH("Bad persistence type value!"); }

}  // namespace

bool IsValidPersistenceType(const PersistenceType aPersistenceType) {
  switch (aPersistenceType) {
    case PERSISTENCE_TYPE_PERSISTENT:
    case PERSISTENCE_TYPE_TEMPORARY:
    case PERSISTENCE_TYPE_DEFAULT:
      return true;

    default:
      return false;
  }
}

nsLiteralCString PersistenceTypeToString(
    const PersistenceType aPersistenceType) {
  const auto maybeString = TypeTo_impl<nsLiteralCString>(aPersistenceType);
  if (maybeString.isNothing()) {
    BadPersistenceType();
  }
  return maybeString.value();
}

Maybe<PersistenceType> PersistenceTypeFromString(const nsACString& aString,
                                                 const fallible_t&) {
  return TypeFrom_impl(aString);
}

PersistenceType PersistenceTypeFromString(const nsACString& aString) {
  const auto maybePersistenceType = TypeFrom_impl(aString);
  if (maybePersistenceType.isNothing()) {
    BadPersistenceType();
  }
  return maybePersistenceType.value();
}

StorageType PersistenceTypeToStorageType(
    const PersistenceType aPersistenceType) {
  const auto maybeStorageType = TypeTo_impl<StorageType>(aPersistenceType);
  if (maybeStorageType.isNothing()) {
    BadPersistenceType();
  }
  return maybeStorageType.value();
}

PersistenceType PersistenceTypeFromStorageType(const StorageType aStorageType) {
  const auto maybePersistenceType = TypeFrom_impl(aStorageType);
  if (maybePersistenceType.isNothing()) {
    BadPersistenceType();
  }
  return maybePersistenceType.value();
}

Maybe<PersistenceType> PersistenceTypeFromInt32(const int32_t aInt32,
                                                const fallible_t&) {
  return TypeFrom_impl(aInt32);
}

}  // namespace quota
}  // namespace dom
}  // namespace mozilla
