/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim: set ts=8 sts=2 et sw=2 tw=80: */
// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

// This provides a wrapper around system calls which may be interrupted by a
// signal and return EINTR. See man 7 signal.
//
// On Windows, this wrapper macro does nothing because there are no
// signals.
//
// Don't wrap close calls in HANDLE_EINTR. Use IGNORE_EINTR if the return
// value of close is significant. See http://crbug.com/269623.

#ifndef BASE_EINTR_WRAPPER_H_
#define BASE_EINTR_WRAPPER_H_

#include "build/build_config.h"

#if defined(OS_POSIX)

#  include <errno.h>

#  define HANDLE_EINTR(x)                                     \
    ({                                                        \
      decltype(x) eintr_wrapper_result;                       \
      do {                                                    \
        eintr_wrapper_result = (x);                           \
      } while (eintr_wrapper_result == -1 && errno == EINTR); \
      eintr_wrapper_result;                                   \
    })

#  define IGNORE_EINTR(x)                                   \
    ({                                                      \
      decltype(x) eintr_wrapper_result;                     \
      do {                                                  \
        eintr_wrapper_result = (x);                         \
        if (eintr_wrapper_result == -1 && errno == EINTR) { \
          eintr_wrapper_result = 0;                         \
        }                                                   \
      } while (0);                                          \
      eintr_wrapper_result;                                 \
    })

#  define HANDLE_RV_EINTR(x)                   \
    ({                                         \
      decltype(x) eintr_wrapper_result;        \
      do {                                     \
        eintr_wrapper_result = (x);            \
      } while (eintr_wrapper_result == EINTR); \
      eintr_wrapper_result;                    \
    })

#else

#  define HANDLE_EINTR(x) (x)
#  define IGNORE_EINTR(x) (x)
#  define HANDLE_RV_EINTR(x) (x)

#endif  // OS_POSIX

#endif  // !BASE_EINTR_WRAPPER_H_
