/*
 * Enhanced Variable Rate Codec, Service Option 3 decoder
 * Copyright (c) 2013 Paul B Mahol
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef AVCODEC_EVRCDATA_H
#define AVCODEC_EVRCDATA_H

/**
 * @file
 * Data tables for the EVRC decoder
 * @author Paul B Mahol
 */

#include "libavutil/common.h"

/**
 * Rate 1/8 frame energy quantization
 *
 * TIA/IS-127 table 8-18
 */
static const float evrc_energy_quant[][3] = {
{-0.2464E-01,-0.4005E-02,-0.1107E+00 }, { 0.8734E+00, 0.1004E+01, 0.9930E+00 },
{ 0.4222E+00, 0.3894E+00, 0.5020E+00 }, { 0.1450E+01, 0.1328E+01, 0.1278E+01 },
{ 0.1957E+00, 0.2169E+00, 0.2735E+00 }, { 0.1142E+01, 0.1240E+01, 0.1157E+01 },
{ 0.7881E+00, 0.6778E+00, 0.4185E+00 }, { 0.1504E+01, 0.1468E+01, 0.1534E+01 },
{ 0.3173E+00, 0.2693E+00,-0.9526E-01 }, { 0.1141E+01, 0.1154E+01, 0.1044E+01 },
{ 0.5147E+00, 0.5784E+00, 0.8802E+00 }, { 0.1502E+01, 0.1407E+01, 0.1409E+01 },
{ 0.3163E+00, 0.3592E+00, 0.2830E+00 }, { 0.1217E+01, 0.1213E+01, 0.1216E+01 },
{ 0.1023E+01, 0.1139E+01,-0.9526E-01 }, { 0.1619E+01, 0.1655E+01, 0.1642E+01 },
{ 0.1437E+00, 0.1505E+00, 0.6838E-01 }, { 0.9794E+00, 0.1021E+01, 0.1117E+01 },
{ 0.4701E+00, 0.6426E+00, 0.5519E+00 }, { 0.1366E+01, 0.1397E+01, 0.1406E+01 },
{ 0.2918E+00, 0.3022E+00, 0.2420E+00 }, { 0.1309E+01, 0.1241E+01, 0.1220E+01 },
{ 0.7989E+00, 0.7654E+00, 0.7391E+00 }, { 0.1612E+01, 0.1502E+01, 0.1447E+01 },
{ 0.2594E+00, 0.1948E+00, 0.2555E+00 }, { 0.1091E+01, 0.1150E+01, 0.1272E+01 },
{ 0.3423E+00, 0.4150E+00, 0.1294E+01 }, { 0.1729E+01, 0.1377E+01, 0.1065E+01 },
{ 0.4103E+00, 0.3287E+00, 0.3228E+00 }, { 0.1144E+01, 0.1281E+01, 0.1416E+01 },
{ 0.1047E+01, 0.1117E+01, 0.6188E+00 }, { 0.1914E+01, 0.1777E+01, 0.1516E+01 },
{-0.2117E-01, 0.2159E+00, 0.2351E+00 }, { 0.1093E+01, 0.1088E+01, 0.1026E+01 },
{ 0.5567E+00, 0.5092E+00, 0.4654E+00 }, { 0.1510E+01, 0.1449E+01, 0.1201E+01 },
{ 0.2362E+00, 0.3426E+00, 0.2549E+00 }, { 0.1340E+01, 0.1225E+01, 0.1117E+01 },
{ 0.1203E+01, 0.3819E+00, 0.2269E+00 }, { 0.1373E+01, 0.1404E+01, 0.1830E+01 },
{ 0.2570E+00, 0.2668E+00, 0.1636E+00 }, { 0.1219E+01, 0.1098E+01, 0.1122E+01 },
{ 0.6985E+00, 0.8456E+00, 0.1069E+01 }, { 0.1550E+01, 0.1501E+01, 0.1388E+01 },
{ 0.2870E+00, 0.3060E+00, 0.3599E+00 }, { 0.1178E+01, 0.1345E+01, 0.1302E+01 },
{ 0.1270E+01, 0.1215E+01, 0.1812E+00 }, { 0.1725E+01, 0.1777E+01, 0.1693E+01 },
{ 0.2074E+00, 0.2104E+00, 0.1539E+00 }, { 0.1105E+01, 0.1034E+01, 0.1104E+01 },
{ 0.6683E+00, 0.6646E+00, 0.6639E+00 }, { 0.1403E+01, 0.1462E+01, 0.1435E+01 },
{ 0.3389E+00, 0.3754E+00, 0.2150E+00 }, { 0.1288E+01, 0.1325E+01, 0.1257E+01 },
{ 0.8933E+00, 0.8253E+00, 0.8133E+00 }, { 0.1555E+01, 0.1579E+01, 0.1565E+01 },
{ 0.3264E+00, 0.2434E+00, 0.2852E+00 }, { 0.1242E+01, 0.1180E+01, 0.1202E+01 },
{ 0.1314E+00, 0.1698E+00, 0.1646E+01 }, { 0.1797E+01, 0.1597E+01, 0.1241E+01 },
{ 0.4721E+00, 0.5346E+00, 0.3066E+00 }, { 0.1274E+01, 0.1401E+01, 0.1351E+01 },
{ 0.1455E+01, 0.1386E+01, 0.6430E+00 }, { 0.1828E+01, 0.1867E+01, 0.1825E+01 },
{-0.3265E+00,-0.2956E+00,-0.2462E+00 }, { 0.1035E+01, 0.1020E+01, 0.1003E+01 },
{ 0.3702E+00, 0.4307E+00, 0.7072E+00 }, { 0.1424E+01, 0.1345E+01, 0.1352E+01 },
{ 0.2267E+00, 0.2680E+00, 0.3037E+00 }, { 0.1235E+01, 0.1249E+01, 0.1146E+01 },
{ 0.9944E+00, 0.6485E+00, 0.5248E+00 }, { 0.1539E+01, 0.1492E+01, 0.1612E+01 },
{ 0.3815E+00, 0.3360E+00,-0.9526E-01 }, { 0.1163E+01, 0.1144E+01, 0.1117E+01 },
{ 0.6734E+00, 0.7656E+00, 0.1014E+01 }, { 0.1568E+01, 0.1438E+01, 0.1455E+01 },
{ 0.3409E+00, 0.3317E+00, 0.3856E+00 }, { 0.1180E+01, 0.1284E+01, 0.1284E+01 },
{ 0.1244E+01, 0.1214E+01,-0.9526E-01 }, { 0.1753E+01, 0.1598E+01, 0.1744E+01 },
{ 0.1548E+00, 0.1388E+00, 0.2020E+00 }, { 0.1027E+01, 0.1133E+01, 0.1093E+01 },
{ 0.3906E+00, 0.7505E+00, 0.5705E+00 }, { 0.1420E+01, 0.1357E+01, 0.1543E+01 },
{ 0.3252E+00, 0.3136E+00, 0.2804E+00 }, { 0.1351E+01, 0.1309E+01, 0.1224E+01 },
{ 0.8781E+00, 0.8095E+00, 0.7109E+00 }, { 0.1614E+01, 0.1580E+01, 0.1433E+01 },
{ 0.3222E+00, 0.2298E+00, 0.2157E+00 }, { 0.1216E+01, 0.1077E+01, 0.1247E+01 },
{ 0.1363E+01, 0.1280E+01, 0.1317E+01 }, { 0.1751E+01, 0.1457E+01, 0.1182E+01 },
{ 0.4428E+00, 0.4082E+00, 0.3181E+00 }, { 0.1157E+01, 0.1227E+01, 0.1604E+01 },
{ 0.1286E+01, 0.1268E+01, 0.8167E+00 }, { 0.1994E+01, 0.2018E+01, 0.1307E+01 },
{ 0.2671E-01, 0.2594E+00, 0.3397E+00 }, { 0.1164E+01, 0.1080E+01, 0.9321E+00 },
{ 0.5998E+00, 0.6076E+00, 0.5081E+00 }, { 0.1442E+01, 0.1442E+01, 0.1375E+01 },
{ 0.2390E+00, 0.3554E+00, 0.3426E+00 }, { 0.1287E+01, 0.1307E+01, 0.1144E+01 },
{ 0.1200E+01, 0.7495E+00, 0.3967E+00 }, { 0.1561E+01, 0.1517E+01, 0.1898E+01 },
{ 0.3598E+00, 0.3463E+00, 0.1200E+00 }, { 0.1298E+01, 0.1125E+01, 0.1062E+01 },
{ 0.7577E+00, 0.1013E+01, 0.1194E+01 }, { 0.1537E+01, 0.1513E+01, 0.1464E+01 },
{ 0.4041E+00, 0.4038E+00, 0.3897E+00 }, { 0.1293E+01, 0.1219E+01, 0.1378E+01 },
{ 0.1250E+01, 0.1391E+01, 0.2451E+00 }, { 0.1558E+01, 0.1764E+01, 0.1728E+01 },
{ 0.2700E+00, 0.1894E+00, 0.1924E+00 }, { 0.1111E+01, 0.1112E+01, 0.1173E+01 },
{ 0.7579E+00, 0.8342E+00, 0.4781E+00 }, { 0.1464E+01, 0.1477E+01, 0.1469E+01 },
{ 0.4001E+00, 0.3104E+00, 0.2217E+00 }, { 0.1346E+01, 0.1421E+01, 0.1312E+01 },
{ 0.1071E+01, 0.8967E+00, 0.7511E+00 }, { 0.1616E+01, 0.1551E+01, 0.1574E+01 },
{ 0.3329E+00, 0.2785E+00, 0.3140E+00 }, { 0.1281E+01, 0.1209E+01, 0.1239E+01 },
{ 0.2805E+00, 0.2687E+00, 0.1646E+01 }, { 0.1814E+01, 0.1514E+01, 0.1510E+01 },
{ 0.6231E+00, 0.4200E+00, 0.3701E+00 }, { 0.1255E+01, 0.1429E+01, 0.1454E+01 },
{ 0.1642E+01, 0.1581E+01, 0.7112E+00 }, { 0.1844E+01, 0.1963E+01, 0.1895E+01 },
{-0.4208E-01,-0.1491E+00,-0.7639E-01 }, { 0.1046E+01, 0.9598E+00, 0.9176E+00 },
{ 0.4478E+00, 0.4605E+00, 0.5111E+00 }, { 0.1521E+01, 0.1292E+01, 0.1342E+01 },
{ 0.2220E+00, 0.2549E+00, 0.2510E+00 }, { 0.1186E+01, 0.1254E+01, 0.1171E+01 },
{ 0.8999E+00, 0.4960E+00, 0.4943E+00 }, { 0.1423E+01, 0.1484E+01, 0.1620E+01 },
{ 0.2796E+00, 0.2778E+00,-0.2820E+00 }, { 0.1170E+01, 0.1181E+01, 0.1076E+01 },
{ 0.4068E+00, 0.8541E+00, 0.9352E+00 }, { 0.1584E+01, 0.1416E+01, 0.1387E+01 },
{ 0.3325E+00, 0.3655E+00, 0.3340E+00 }, { 0.1224E+01, 0.1257E+01, 0.1245E+01 },
{ 0.1061E+01, 0.1138E+01,-0.9526E-01 }, { 0.1681E+01, 0.1704E+01, 0.1673E+01 },
{ 0.1932E+00, 0.1489E+00, 0.1258E+00 }, { 0.1023E+01, 0.1088E+01, 0.1145E+01 },
{ 0.5190E+00, 0.6873E+00, 0.5172E+00 }, { 0.1380E+01, 0.1405E+01, 0.1474E+01 },
{ 0.3393E+00, 0.3100E+00, 0.2231E+00 }, { 0.1354E+01, 0.1249E+01, 0.1270E+01 },
{ 0.7363E+00, 0.8508E+00, 0.8247E+00 }, { 0.1612E+01, 0.1537E+01, 0.1509E+01 },
{ 0.2952E+00, 0.2053E+00, 0.2590E+00 }, { 0.1138E+01, 0.1219E+01, 0.1262E+01 },
{ 0.1345E+01, 0.1289E+01, 0.1338E+01 }, { 0.1437E+01, 0.1360E+01, 0.1442E+01 },
{ 0.4826E+00, 0.3298E+00, 0.3842E+00 }, { 0.1219E+01, 0.1311E+01, 0.1413E+01 },
{ 0.1212E+01, 0.1186E+01, 0.6357E+00 }, { 0.1873E+01, 0.1939E+01, 0.1674E+01 },
{ 0.1260E+01, 0.1306E+01, 0.1368E+01 }, { 0.1146E+01, 0.1077E+01, 0.1025E+01 },
{ 0.6029E+00, 0.5039E+00, 0.5781E+00 }, { 0.1514E+01, 0.1420E+01, 0.1324E+01 },
{ 0.2652E+00, 0.3192E+00, 0.3042E+00 }, { 0.1368E+01, 0.1198E+01, 0.1200E+01 },
{ 0.1234E+01, 0.4910E+00, 0.3464E-01 }, { 0.1347E+01, 0.1560E+01, 0.1861E+01 },
{ 0.2766E+00, 0.2887E+00, 0.2029E+00 }, { 0.1257E+01, 0.1105E+01, 0.1145E+01 },
{ 0.1351E+01, 0.1353E+01, 0.1406E+01 }, { 0.1506E+01, 0.1580E+01, 0.1362E+01 },
{ 0.2794E+00, 0.3868E+00, 0.4277E+00 }, { 0.1234E+01, 0.1334E+01, 0.1336E+01 },
{ 0.1280E+01, 0.1252E+01, 0.1805E+00 }, { 0.1387E+01, 0.1396E+01, 0.1434E+01 },
{ 0.2902E+00, 0.1170E+00, 0.1698E+00 }, { 0.1134E+01, 0.1077E+01, 0.1117E+01 },
{ 0.6986E+00, 0.7177E+00, 0.7366E+00 }, { 0.1370E+01, 0.1491E+01, 0.1495E+01 },
{ 0.4031E+00, 0.5144E+00, 0.1751E+00 }, { 0.1333E+01, 0.1377E+01, 0.1257E+01 },
{ 0.9212E+00, 0.8934E+00, 0.8897E+00 }, { 0.1589E+01, 0.1614E+01, 0.1523E+01 },
{ 0.3152E+00, 0.2164E+00, 0.3230E+00 }, { 0.1300E+01, 0.1145E+01, 0.1212E+01 },
{ 0.1269E+01, 0.1245E+01, 0.1497E+01 }, { 0.1763E+01, 0.1716E+01, 0.1311E+01 },
{ 0.4702E+00, 0.5422E+00, 0.4306E+00 }, { 0.1342E+01, 0.1433E+01, 0.1423E+01 },
{ 0.1472E+01, 0.1404E+01, 0.8371E+00 }, { 0.1936E+01, 0.1883E+01, 0.1838E+01 },
{ 0.1266E+01, 0.1295E+01, 0.1302E+01 }, { 0.1074E+01, 0.1002E+01, 0.1023E+01 },
{ 0.5206E+00, 0.4045E+00, 0.6549E+00 }, { 0.1457E+01, 0.1378E+01, 0.1363E+01 },
{ 0.2715E+00, 0.2629E+00, 0.2841E+00 }, { 0.1264E+01, 0.1271E+01, 0.1175E+01 },
{ 0.1337E+01, 0.1305E+01, 0.1306E+01 }, { 0.1555E+01, 0.1571E+01, 0.1657E+01 },
{ 0.3341E+00, 0.4147E+00,-0.3648E+00 }, { 0.1188E+01, 0.1185E+01, 0.1161E+01 },
{ 0.6198E+00, 0.7208E+00, 0.1157E+01 }, { 0.1582E+01, 0.1465E+01, 0.1513E+01 },
{ 0.3839E+00, 0.3651E+00, 0.3814E+00 }, { 0.1214E+01, 0.1256E+01, 0.1292E+01 },
{ 0.1361E+01, 0.1363E+01, 0.1312E+01 }, { 0.1793E+01, 0.1693E+01, 0.1669E+01 },
{ 0.1889E+00, 0.1275E+00, 0.2534E+00 }, { 0.1066E+01, 0.1174E+01, 0.1133E+01 },
{ 0.4999E+00, 0.8207E+00, 0.5813E+00 }, { 0.1478E+01, 0.1416E+01, 0.1497E+01 },
{ 0.3814E+00, 0.3138E+00, 0.2889E+00 }, { 0.1396E+01, 0.1265E+01, 0.1233E+01 },
{ 0.9458E+00, 0.9161E+00, 0.5875E+00 }, { 0.1672E+01, 0.1632E+01, 0.1553E+01 },
{ 0.3505E+00, 0.2525E+00, 0.2364E+00 }, { 0.1211E+01, 0.1138E+01, 0.1235E+01 },
{ 0.1391E+01, 0.1231E+01, 0.1355E+01 }, { 0.1783E+01, 0.1510E+01, 0.1199E+01 },
{ 0.4227E+00, 0.4548E+00, 0.3671E+00 }, { 0.1281E+01, 0.1254E+01, 0.1661E+01 },
{ 0.1338E+01, 0.1379E+01, 0.9531E+00 }, { 0.2148E+01, 0.1965E+01, 0.1584E+01 },
{ 0.9324E-01, 0.3575E+00, 0.3522E+00 }, { 0.1212E+01, 0.1086E+01, 0.1044E+01 },
{ 0.6128E+00, 0.6136E+00, 0.6060E+00 }, { 0.1484E+01, 0.1507E+01, 0.1396E+01 },
{ 0.2820E+00, 0.3848E+00, 0.3156E+00 }, { 0.1368E+01, 0.1287E+01, 0.1128E+01 },
{ 0.1369E+01, 0.1352E+01, 0.1358E+01 }, { 0.1381E+01, 0.1765E+01, 0.2113E+01 },
{ 0.1314E+01, 0.1345E+01, 0.1334E+01 }, { 0.1290E+01, 0.1172E+01, 0.1119E+01 },
{ 0.1304E+01, 0.1377E+01, 0.1427E+01 }, { 0.1490E+01, 0.1540E+01, 0.1536E+01 },
{ 0.3994E+00, 0.4402E+00, 0.4173E+00 }, { 0.1323E+01, 0.1307E+01, 0.1392E+01 },
{ 0.1400E+01, 0.1388E+01, 0.1369E+01 }, { 0.1669E+01, 0.1818E+01, 0.1834E+01 },
{ 0.2742E+00, 0.2235E+00, 0.1986E+00 }, { 0.1137E+01, 0.1139E+01, 0.1201E+01 },
{ 0.1324E+01, 0.1385E+01, 0.1349E+01 }, { 0.1455E+01, 0.1574E+01, 0.1454E+01 },
{ 0.5019E+00, 0.3255E+00, 0.2555E+00 }, { 0.1388E+01, 0.1438E+01, 0.1300E+01 },
{ 0.1394E+01, 0.1349E+01, 0.1411E+01 }, { 0.1639E+01, 0.1580E+01, 0.1681E+01 },
{ 0.3920E+00, 0.2498E+00, 0.3523E+00 }, { 0.1301E+01, 0.1221E+01, 0.1285E+01 },
{ 0.1318E+01, 0.1342E+01, 0.1494E+01 }, { 0.1910E+01, 0.1680E+01, 0.1470E+01 },
{ 0.6082E+00, 0.5270E+00, 0.4173E+00 }, { 0.1255E+01, 0.1477E+01, 0.1503E+01 },
{ 0.1807E+01, 0.1742E+01, 0.6553E+00 }, { 0.2000E+01, 0.2072E+01, 0.2051E+01 }};

/**
 * LSP vector quantization tables
 *
 * TIA/IS-127 tables 8-1 through 8-9
 */

static const float evrc_lspq_full_codebook1[64][2] = {
{1.42016308E-2, 1.93881616E-2}, {2.91667543E-2, 6.51749149E-2},
{2.06693150E-2, 4.97564934E-2}, {3.94719802E-2, 9.55850929E-2},
{2.27012448E-2, 3.96625809E-2}, {5.38789518E-2, 6.28347769E-2},
{2.90525518E-2, 5.73435798E-2}, {4.48280610E-2, 1.15364626E-1},
{1.94110647E-2, 3.46889682E-2}, {4.37502973E-2, 6.75228462E-2},
{3.55497338E-2, 4.94086780E-2}, {6.99219853E-2, 8.67279768E-2},
{2.77880151E-2, 4.65748496E-2}, {5.79111017E-2, 6.74542487E-2},
{4.74664383E-2, 5.50271496E-2}, {7.88898915E-2, 1.22443043E-1},
{2.21715886E-2, 3.02628800E-2}, {3.39134485E-2, 7.17703998E-2},
{3.17989141E-2, 4.98996116E-2}, {6.11555986E-2, 8.73361230E-2},
{2.67506503E-2, 3.96735854E-2}, {4.44100983E-2, 8.26731324E-2},
{3.89172547E-2, 5.65788932E-2}, {6.04800619E-2, 1.04536951E-1},
{2.69156620E-2, 3.57168876E-2}, {4.11117189E-2, 7.33322948E-2},
{4.12660725E-2, 4.85165231E-2}, {7.18049556E-2, 1.06202349E-1},
{3.38037871E-2, 4.24300395E-2}, {5.91818243E-2, 7.97467977E-2},
{4.70107906E-2, 6.28563762E-2}, {9.42011923E-2, 1.30053163E-1},
{1.94244273E-2, 2.72732340E-2}, {3.70831676E-2, 6.64898157E-2},
{2.80136354E-2, 5.15984930E-2}, {5.34461029E-2, 9.25904214E-2},
{2.54959203E-2, 4.32844795E-2}, {5.51860742E-2, 7.36182332E-2},
{3.39851119E-2, 6.05329126E-2}, {6.18182123E-2, 1.34581268E-1},
{2.35669166E-2, 3.55242006E-2}, {5.10804243E-2, 6.79562539E-2},
{3.83464955E-2, 5.23469411E-2}, {7.44275749E-2, 9.66108292E-2},
{3.18591148E-2, 4.62123118E-2}, {6.18909821E-2, 7.33231753E-2},
{4.41718437E-2, 5.79240918E-2}, {7.93596208E-2, 1.41177371E-1},
{2.47412287E-2, 3.23629379E-2}, {3.36563922E-2, 8.04650635E-2},
{3.37943695E-2, 5.44977151E-2}, {6.53648973E-2, 9.52775925E-2},
{2.93364152E-2, 4.28411029E-2}, {5.27870469E-2, 8.16159397E-2},
{4.00724895E-2, 6.18144684E-2}, {6.75848573E-2, 1.17196076E-1},
{3.03064957E-2, 3.86914052E-2}, {4.83106263E-2, 7.42383003E-2},
{4.37548272E-2, 5.22842295E-2}, {8.32310021E-2, 1.09881967E-1},
{3.75600643E-2, 4.53217216E-2}, {6.60113171E-2, 7.97580183E-2},
{5.03225066E-2, 5.90176322E-2}, {8.77133310E-2, 1.63187444E-1}};

static const float evrc_lspq_full_codebook2[64][2] = {
{5.21959551E-2, 8.38445649E-2}, {1.05874076E-1, 1.28694162E-1},
{5.48323877E-2, 1.33842856E-1}, {1.17768474E-1, 1.94037274E-1},
{5.36086522E-2, 1.11398734E-1}, {1.19989693E-1, 1.47474691E-1},
{8.00373554E-2, 1.42999724E-1}, {1.64086595E-1, 2.09821835E-1},
{5.21059223E-2, 9.95229408E-2}, {8.67567956E-2, 1.85966507E-1},
{7.77341127E-2, 1.31506845E-1}, {1.60545513E-1, 1.81930289E-1},
{7.42243677E-2, 1.10437103E-1}, {1.18635088E-1, 1.75306752E-1},
{6.61557764E-2, 1.64441928E-1}, {1.96810856E-1, 2.16682002E-1},
{6.05317838E-2, 9.45408568E-2}, {1.06271386E-1, 1.48013934E-1},
{5.87486550E-2, 1.47724584E-1}, {1.34816468E-1, 2.01517954E-1},
{6.59698322E-2, 1.16447397E-1}, {1.32297173E-1, 1.53267249E-1},
{9.26660746E-2, 1.46725491E-1}, {1.79285541E-1, 2.19705954E-1},
{7.06458464E-2, 9.99924466E-2}, {1.06500491E-1, 1.79443434E-1},
{8.79249722E-2, 1.25287697E-1}, {1.53640196E-1, 1.97852716E-1},
{8.88430104E-2, 1.12465657E-1}, {1.48286715E-1, 1.67517021E-1},
{8.16568136E-2, 1.69274017E-1}, {2.07810536E-1, 2.31033549E-1},
{6.14927970E-2, 8.36263224E-2}, {1.14473253E-1, 1.36779979E-1},
{6.87129870E-2, 1.38099059E-1}, {1.10511415E-1, 2.15352878E-1},
{5.55652268E-2, 1.22242786E-1}, {1.20557591E-1, 1.61072448E-1},
{8.32249671E-2, 1.55475482E-1}, {1.61638483E-1, 2.28268847E-1},
{6.29152283E-2, 1.06229566E-1}, {8.29186887E-2, 2.06774518E-1},
{8.84756893E-2, 1.35799959E-1}, {1.69772223E-1, 1.93773940E-1},
{7.77297840E-2, 1.20287232E-1}, {1.30648017E-1, 1.84331819E-1},
{6.91939592E-2, 1.84218004E-1}, {2.03904077E-1, 2.49715164E-1},
{7.07671717E-2, 9.03186128E-2}, {1.08471557E-1, 1.61966518E-1},
{7.16886371E-2, 1.51093170E-1}, {1.38779536E-1, 2.18801782E-1},
{6.75907061E-2, 1.26740307E-1}, {1.33412346E-1, 1.68838874E-1},
{9.61822569E-2, 1.58728704E-1}, {1.86485633E-1, 2.36560926E-1},
{8.23447108E-2, 1.02126025E-1}, {1.00336641E-1, 1.94918498E-1},
{9.95981991E-2, 1.36425093E-1}, {1.82448462E-1, 2.03655198E-1},
{9.78890732E-2, 1.21145472E-1}, {1.45453140E-1, 1.83604524E-1},
{9.58395451E-2, 1.72194853E-1}, {2.23295853E-1, 2.46418610E-1}};

static const float evrc_lspq_full_codebook3[512][3] = {
{1.36425778E-1, 1.68651849E-1, 2.04688221E-1},
{1.85717627E-1, 2.28756160E-1, 2.51958042E-1},
{1.22760192E-1, 1.85950696E-1, 2.79446691E-1},
{1.96468458E-1, 2.64484435E-1, 2.89318889E-1},
{1.25653744E-1, 1.50529265E-1, 2.76144296E-1},
{1.96301565E-1, 2.41699994E-1, 2.88230687E-1},
{1.40099391E-1, 2.22365588E-1, 2.74666578E-1},
{2.59952307E-1, 2.75394946E-1, 3.10975939E-1},
{1.58452198E-1, 1.88591003E-1, 2.07339197E-1},
{1.95616230E-1, 2.21379519E-1, 2.87022918E-1},
{1.69424579E-1, 2.01614648E-1, 2.75669187E-1},
{2.12393746E-1, 2.64250666E-1, 3.17967504E-1},
{1.82965085E-1, 1.99547559E-1, 2.29538843E-1},
{2.15200707E-1, 2.62409419E-1, 2.82432705E-1},
{1.46404549E-1, 2.36966729E-1, 2.90067106E-1},
{2.45338634E-1, 3.03358108E-1, 3.42260152E-1},
{1.37478963E-1, 1.58276558E-1, 2.39217222E-1},
{2.01999024E-1, 2.20102608E-1, 2.69546896E-1},
{1.18350029E-1, 2.30206400E-1, 2.83554822E-1},
{2.25519255E-1, 2.72272140E-1, 3.06072980E-1},
{1.35661438E-1, 1.91633970E-1, 2.65912026E-1},
{1.95733085E-1, 2.31926173E-1, 3.14376086E-1},
{1.67998984E-1, 2.27706313E-1, 2.76947826E-1},
{2.50170559E-1, 3.01627070E-1, 3.21084231E-1},
{1.33492306E-1, 2.01223105E-1, 2.33893991E-1},
{2.06442133E-1, 2.38704175E-1, 2.77560145E-1},
{1.79048792E-1, 1.95776582E-1, 2.80656606E-1},
{2.06193641E-1, 2.64055401E-1, 3.33098441E-1},
{1.75185278E-1, 1.91166341E-1, 2.57540315E-1},
{2.28398636E-1, 2.45296657E-1, 3.08980793E-1},
{1.80859819E-1, 2.43579060E-1, 2.96631068E-1},
{2.76152968E-1, 3.08256060E-1, 3.46822590E-1},
{1.37115732E-1, 1.80057764E-1, 2.20953465E-1},
{1.81370094E-1, 2.26770103E-1, 2.70392686E-1},
{1.25246510E-1, 1.79606944E-1, 3.10376436E-1},
{1.90708354E-1, 2.87734240E-1, 3.13476235E-1},
{1.30486086E-1, 1.60435289E-1, 3.00243706E-1},
{1.97318628E-1, 2.56378502E-1, 2.78474301E-1},
{1.58597067E-1, 2.37381399E-1, 2.62910336E-1},
{2.61825919E-1, 2.77717203E-1, 3.31382245E-1},
{1.64160743E-1, 1.85841531E-1, 2.35615849E-1},
{2.09486142E-1, 2.21452802E-1, 2.92153865E-1},
{1.66807845E-1, 2.13641763E-1, 2.70675927E-1},
{2.29834273E-1, 2.88374633E-1, 3.06238323E-1},
{1.82154253E-1, 2.00822473E-1, 2.40169376E-1},
{2.24944726E-1, 2.69813925E-1, 2.91401237E-1},
{1.63940564E-1, 2.50341147E-1, 2.78307766E-1},
{2.56727993E-1, 2.95103759E-1, 3.53297085E-1},
{1.40218839E-1, 1.76687688E-1, 2.46773273E-1},
{2.15291306E-1, 2.29216009E-1, 2.64283627E-1},
{1.21002659E-1, 2.18333840E-1, 3.22341293E-1},
{2.54243195E-1, 2.73986191E-1, 2.96262473E-1},
{1.60385415E-1, 1.83762908E-1, 2.81598717E-1},
{1.87832162E-1, 2.37420350E-1, 3.29777509E-1},
{1.77788362E-1, 2.26703495E-1, 3.02322537E-1},
{2.75108218E-1, 2.93730587E-1, 3.12373787E-1},
{1.70116410E-1, 1.85232103E-1, 2.46125028E-1},
{2.21754774E-1, 2.39912242E-1, 2.86891907E-1},
{1.95083722E-1, 2.08337873E-1, 2.88349718E-1},
{2.37536535E-1, 2.75004476E-1, 3.39786023E-1},
{1.88369319E-1, 2.04371840E-1, 2.57375032E-1},
{2.47250155E-1, 2.60551840E-1, 3.02137524E-1},
{1.66944191E-1, 2.46912360E-1, 3.18894416E-1},
{2.78118610E-1, 3.13011140E-1, 3.65329295E-1},
{1.45213529E-1, 1.63051456E-1, 2.24912614E-1},
{2.05692515E-1, 2.20831484E-1, 2.52817810E-1},
{1.21125661E-1, 1.96374118E-1, 3.00122708E-1},
{2.15566799E-1, 2.65657336E-1, 2.99202889E-1},
{1.09134212E-1, 1.78472102E-1, 2.88323194E-1},
{2.03508541E-1, 2.40347922E-1, 2.96309739E-1},
{1.53101787E-1, 2.25415319E-1, 2.84843713E-1},
{2.50233442E-1, 2.77736932E-1, 3.24840695E-1},
{1.66308925E-1, 1.94173396E-1, 2.11635381E-1},
{2.01289460E-1, 2.26062179E-1, 2.93246478E-1},
{1.49518773E-1, 2.14201719E-1, 2.83894747E-1},
{2.21836135E-1, 2.85231501E-1, 3.20082635E-1},
{1.89573213E-1, 2.06577629E-1, 2.30332345E-1},
{2.31247649E-1, 2.46864259E-1, 2.89846569E-1},
{1.39116928E-1, 2.59189934E-1, 2.98019558E-1},
{2.44512573E-1, 2.82671362E-1, 3.61258298E-1},
{1.22530967E-1, 1.68514788E-1, 2.70879298E-1},
{2.04372838E-1, 2.30398357E-1, 2.71792918E-1},
{1.42643943E-1, 2.22405583E-1, 2.92057186E-1},
{2.42643669E-1, 2.77429372E-1, 2.97135502E-1},
{1.52048603E-1, 1.96921080E-1, 2.61013240E-1},
{2.17875019E-1, 2.45840371E-1, 3.08138579E-1},
{1.90109268E-1, 2.31099129E-1, 2.80178159E-1},
{2.54314184E-1, 2.94079810E-1, 3.39649171E-1},
{1.56698599E-1, 2.08597451E-1, 2.28010774E-1},
{2.25088730E-1, 2.50014484E-1, 2.76250154E-1},
{1.78219035E-1, 1.98228240E-1, 3.04198891E-1},
{2.08567217E-1, 2.92395383E-1, 3.46786886E-1},
{1.71052113E-1, 2.03438759E-1, 2.62644321E-1},
{2.30275467E-1, 2.58817524E-1, 3.11986536E-1},
{1.85333565E-1, 2.45760202E-1, 3.10553998E-1},
{2.89413869E-1, 3.11095625E-1, 3.46476167E-1},
{1.50332406E-1, 1.67538226E-1, 2.40182847E-1},
{1.79971650E-1, 2.37168610E-1, 2.60899693E-1},
{1.49866179E-1, 1.97890073E-1, 3.07916552E-1},
{2.10799649E-1, 2.88180083E-1, 3.29747230E-1},
{1.31711140E-1, 1.65906459E-1, 3.22898000E-1},
{2.14832023E-1, 2.52822131E-1, 2.97547072E-1},
{1.83760419E-1, 2.37523615E-1, 2.74610013E-1},
{2.55575180E-1, 2.75439233E-1, 3.46021861E-1},
{1.82662204E-1, 1.99470907E-1, 2.16051653E-1},
{2.09240332E-1, 2.22406715E-1, 3.02382857E-1},
{1.84088245E-1, 2.11327791E-1, 2.82538086E-1},
{2.41171077E-1, 2.97036022E-1, 3.15979272E-1},
{1.96804658E-1, 2.11815894E-1, 2.41647676E-1},
{2.42761984E-1, 2.58586556E-1, 2.93204397E-1},
{1.58905461E-1, 2.65077025E-1, 2.89881319E-1},
{2.58060575E-1, 3.18903178E-1, 3.47846836E-1},
{1.48766384E-1, 1.66853935E-1, 2.66827434E-1},
{2.15942249E-1, 2.29938298E-1, 2.76041597E-1},
{1.38410494E-1, 2.39283442E-1, 3.27972382E-1},
{2.43765280E-1, 2.88408488E-1, 3.06048721E-1},
{1.70157120E-1, 1.89986289E-1, 2.81219155E-1},
{2.19117031E-1, 2.58005291E-1, 3.26571971E-1},
{1.92163572E-1, 2.23614186E-1, 2.98683077E-1},
{2.73545444E-1, 3.12078089E-1, 3.30766588E-1},
{1.62452087E-1, 2.04930902E-1, 2.53337711E-1},
{2.23855302E-1, 2.37671077E-1, 3.03202003E-1},
{1.93955287E-1, 2.12335557E-1, 3.07566851E-1},
{2.29912683E-1, 2.97581047E-1, 3.37499231E-1},
{1.89335391E-1, 2.04148144E-1, 2.78609782E-1},
{2.42303565E-1, 2.73163110E-1, 3.15361649E-1},
{1.55009672E-1, 2.88095146E-1, 3.35996419E-1},
{2.73716152E-1, 3.31215471E-1, 3.62539083E-1},
{1.52389362E-1, 1.72619134E-1, 1.90585673E-1},
{1.96988270E-1, 2.26309747E-1, 2.46197492E-1},
{1.20555148E-1, 2.06369758E-1, 2.81199783E-1},
{1.93709418E-1, 2.71900505E-1, 3.01332921E-1},
{1.36701152E-1, 1.54093146E-1, 2.82258362E-1},
{1.97299168E-1, 2.53656298E-1, 2.90315062E-1},
{1.43463776E-1, 2.43872911E-1, 2.75533706E-1},
{2.58477271E-1, 2.73279876E-1, 3.21119100E-1},
{1.54406175E-1, 1.93793535E-1, 2.15884149E-1},
{2.05979452E-1, 2.24277020E-1, 2.85732359E-1},
{1.74535319E-1, 2.08482355E-1, 2.79668540E-1},
{2.18844578E-1, 2.72486299E-1, 3.27095598E-1},
{1.77609727E-1, 2.12990195E-1, 2.39119649E-1},
{2.29163751E-1, 2.59165913E-1, 2.83514649E-1},
{1.57353148E-1, 2.39961296E-1, 3.04263145E-1},
{2.45613828E-1, 3.16824526E-1, 3.42909366E-1},
{1.42953232E-1, 1.61905348E-1, 2.53710240E-1},
{2.10192814E-1, 2.22847700E-1, 2.71103770E-1},
{1.26843944E-1, 2.16709048E-1, 2.97734648E-1},
{2.31000140E-1, 2.80109137E-1, 2.99707443E-1},
{1.52980462E-1, 1.93996876E-1, 2.72895664E-1},
{2.12860718E-1, 2.41545349E-1, 3.16518754E-1},
{1.71154693E-1, 2.22469687E-1, 2.93786496E-1},
{2.51988232E-1, 3.04254979E-1, 3.31269950E-1},
{1.33188918E-1, 2.07924992E-1, 2.55362093E-1},
{2.12044910E-1, 2.42189646E-1, 2.88903743E-1},
{1.84612468E-1, 2.01143622E-1, 2.86360770E-1},
{2.18286708E-1, 2.76752442E-1, 3.44581515E-1},
{1.83562174E-1, 1.99478507E-1, 2.62156576E-1},
{2.33130530E-1, 2.49596909E-1, 3.15842837E-1},
{1.89898983E-1, 2.46874869E-1, 2.97132462E-1},
{2.75022447E-1, 3.22490305E-1, 3.46977681E-1},
{1.42305329E-1, 1.92689180E-1, 2.16155857E-1},
{1.95676163E-1, 2.22268641E-1, 2.76587397E-1},
{1.33241490E-1, 1.97791785E-1, 3.22897941E-1},
{1.84865132E-1, 2.97106177E-1, 3.26105148E-1},
{1.50203660E-1, 1.76781267E-1, 2.91536182E-1},
{2.03144446E-1, 2.59616166E-1, 2.99156040E-1},
{1.65488973E-1, 2.38342047E-1, 2.87493914E-1},
{2.71071255E-1, 2.89544493E-1, 3.19521040E-1},
{1.68598369E-1, 1.98825568E-1, 2.30347604E-1},
{2.13811651E-1, 2.34471768E-1, 2.90959626E-1},
{1.74605444E-1, 2.17256010E-1, 2.85688072E-1},
{2.28503481E-1, 2.96190292E-1, 3.16534668E-1},
{1.87172607E-1, 2.20547438E-1, 2.39688724E-1},
{2.28884771E-1, 2.63583153E-1, 3.01329464E-1},
{1.77897051E-1, 2.58131474E-1, 2.81487674E-1},
{2.59513617E-1, 3.07204396E-1, 3.48793596E-1},
{1.45224437E-1, 1.78715974E-1, 2.59186983E-1},
{2.19062313E-1, 2.38223523E-1, 2.60461539E-1},
{1.43650874E-1, 2.09760785E-1, 3.15830201E-1},
{2.50127465E-1, 2.79182345E-1, 3.05153579E-1},
{1.48986444E-1, 2.01226771E-1, 2.82543689E-1},
{2.08387777E-1, 2.35603899E-1, 3.45363885E-1},
{1.85830340E-1, 2.21607298E-1, 3.10773641E-1},
{2.80904710E-1, 2.95469791E-1, 3.25499445E-1},
{1.72967300E-1, 1.97078109E-1, 2.45801106E-1},
{2.19495699E-1, 2.44767100E-1, 2.93587774E-1},
{1.83909580E-1, 2.15004295E-1, 3.00334543E-1},
{2.45338634E-1, 2.68595248E-1, 3.48330349E-1},
{1.92957386E-1, 2.06625074E-1, 2.67336398E-1},
{2.54845560E-1, 2.68642277E-1, 3.03547889E-1},
{1.76853105E-1, 2.59330958E-1, 3.16200763E-1},
{2.90929139E-1, 3.15634757E-1, 3.68723541E-1},
{1.57116994E-1, 1.73552901E-1, 2.28736520E-1},
{2.12509260E-1, 2.30501205E-1, 2.52217978E-1},
{1.42521843E-1, 2.01979935E-1, 2.93012232E-1},
{2.14919671E-1, 2.78065056E-1, 3.14176053E-1},
{1.35947272E-1, 1.81055903E-1, 2.75475413E-1},
{1.98416695E-1, 2.41673797E-1, 3.05173427E-1},
{1.59517333E-1, 2.31580108E-1, 2.95412451E-1},
{2.58203626E-1, 2.87348121E-1, 3.20351988E-1},
{1.74840674E-1, 1.92883253E-1, 2.11250007E-1},
{2.02168509E-1, 2.27025688E-1, 3.04884046E-1},
{1.69532105E-1, 2.11826235E-1, 2.97355384E-1},
{2.30033740E-1, 2.91504353E-1, 3.26589435E-1},
{1.95046112E-1, 2.11709172E-1, 2.27705747E-1},
{2.37926885E-1, 2.52411634E-1, 2.97752172E-1},
{1.53762922E-1, 2.46541560E-1, 3.14768940E-1},
{2.36075714E-1, 3.03568929E-1, 3.70624453E-1},
{1.38660327E-1, 1.67949975E-1, 2.73515254E-1},
{2.13806167E-1, 2.27267206E-1, 2.86276251E-1},
{1.25080630E-1, 2.44098395E-1, 3.02548796E-1},
{2.35714868E-1, 2.81208843E-1, 3.08903724E-1},
{1.51691392E-1, 2.10877746E-1, 2.63812989E-1},
{2.20730439E-1, 2.52777904E-1, 3.16413730E-1},
{1.84924737E-1, 2.39424765E-1, 2.85120815E-1},
{2.59548545E-1, 3.09809893E-1, 3.26423734E-1},
{1.62930742E-1, 2.19900876E-1, 2.36148626E-1},
{2.34194234E-1, 2.49944329E-1, 2.77549058E-1},
{1.70870200E-1, 1.98291600E-1, 3.21412593E-1},
{2.31566861E-1, 2.75015086E-1, 3.69710356E-1},
{1.80002406E-1, 2.06701040E-1, 2.71204919E-1},
{2.38075271E-1, 2.54006237E-1, 3.23827595E-1},
{1.99148253E-1, 2.54273921E-1, 3.07479709E-1},
{2.87428617E-1, 3.25045079E-1, 3.48634571E-1},
{1.45285025E-1, 1.91359162E-1, 2.49691397E-1},
{1.94659308E-1, 2.40821242E-1, 2.77302653E-1},
{1.53150991E-1, 1.94375664E-1, 3.27550441E-1},
{2.04085842E-1, 2.98595697E-1, 3.21480066E-1},
{1.56009689E-1, 1.81012720E-1, 3.00931662E-1},
{2.10962430E-1, 2.55770296E-1, 3.08086127E-1},
{1.85444072E-1, 2.49021322E-1, 2.74029821E-1},
{2.74493456E-1, 2.89441973E-1, 3.38794917E-1},
{1.76941887E-1, 1.94476932E-1, 2.22077265E-1},
{2.16377512E-1, 2.30735779E-1, 3.03689271E-1},
{1.89683452E-1, 2.14660764E-1, 2.88445383E-1},
{2.40827337E-1, 2.98141748E-1, 3.27378422E-1},
{2.01787844E-1, 2.19441772E-1, 2.39327446E-1},
{2.48812512E-1, 2.65865892E-1, 2.93382376E-1},
{1.82027832E-1, 2.68279046E-1, 2.93991417E-1},
{2.56498635E-1, 3.19984466E-1, 3.62663239E-1},
{1.58799276E-1, 1.75433666E-1, 2.67389864E-1},
{2.24259302E-1, 2.36668259E-1, 2.77639121E-1},
{1.49203405E-1, 2.26585329E-1, 3.45255584E-1},
{2.50655770E-1, 2.92264849E-1, 3.13574284E-1},
{1.58096299E-1, 2.02193201E-1, 2.98711687E-1},
{2.28820905E-1, 2.48557344E-1, 3.44726473E-1},
{1.87972054E-1, 2.34109432E-1, 3.04235607E-1},
{2.85657108E-1, 3.14878136E-1, 3.36931497E-1},
{1.62680015E-1, 2.17820048E-1, 2.57436782E-1},
{2.24049792E-1, 2.46739820E-1, 3.00795883E-1},
{2.01354548E-1, 2.18286663E-1, 3.13036293E-1},
{2.38028511E-1, 2.98103482E-1, 3.53503793E-1},
{1.98829994E-1, 2.12877125E-1, 2.72980839E-1},
{2.50616491E-1, 2.67659992E-1, 3.20611864E-1},
{1.70901820E-1, 2.69330353E-1, 3.34428221E-1},
{3.04988861E-1, 3.36196691E-1, 3.65235358E-1},
{1.47624031E-1, 1.81272805E-1, 2.04707921E-1},
{1.93751350E-1, 2.20973969E-1, 2.61775166E-1},
{1.32089809E-1, 1.94851607E-1, 2.83547610E-1},
{2.07739428E-1, 2.70596832E-1, 2.92264789E-1},
{1.27733424E-1, 1.66896015E-1, 2.83891350E-1},
{2.05309406E-1, 2.47807533E-1, 2.83632785E-1},
{1.54211894E-1, 2.25014091E-1, 2.70082027E-1},
{2.67574131E-1, 2.84426898E-1, 3.09334785E-1},
{1.68846920E-1, 1.87004536E-1, 2.02433169E-1},
{2.02441111E-1, 2.16733068E-1, 2.93079227E-1},
{1.63621262E-1, 2.15616465E-1, 2.82792896E-1},
{2.25509301E-1, 2.66283005E-1, 3.17886561E-1},
{1.89110294E-1, 2.05609441E-1, 2.22113580E-1},
{2.21240178E-1, 2.60288864E-1, 2.92541057E-1},
{1.55563369E-1, 2.46850818E-1, 2.89648801E-1},
{2.48406157E-1, 3.05291861E-1, 3.55316669E-1},
{1.27122149E-1, 1.58053726E-1, 2.54164368E-1},
{2.04998836E-1, 2.19476849E-1, 2.78342038E-1},
{1.33302316E-1, 2.29614019E-1, 2.86947161E-1},
{2.36777052E-1, 2.67918199E-1, 3.08230907E-1},
{1.40853569E-1, 2.03414679E-1, 2.73257107E-1},
{2.07684264E-1, 2.34520018E-1, 3.24583262E-1},
{1.77181646E-1, 2.29595393E-1, 2.83539146E-1},
{2.61378348E-1, 3.01160187E-1, 3.21707100E-1},
{1.48595735E-1, 2.07772017E-1, 2.46946126E-1},
{2.14334831E-1, 2.48061299E-1, 2.72259146E-1},
{1.76380262E-1, 1.96897894E-1, 2.92286903E-1},
{1.98193476E-1, 2.75483340E-1, 3.49037558E-1},
{1.76153168E-1, 1.93248957E-1, 2.69548506E-1},
{2.36968622E-1, 2.50065804E-1, 3.06820840E-1},
{1.76060721E-1, 2.54037619E-1, 3.03566784E-1},
{2.82952905E-1, 3.01765054E-1, 3.53956312E-1},
{1.45353720E-1, 1.83678836E-1, 2.34750062E-1},
{1.93842635E-1, 2.30635554E-1, 2.67817765E-1},
{1.38958976E-1, 1.86760783E-1, 3.13113242E-1},
{1.99944481E-1, 2.77624756E-1, 3.25046331E-1},
{1.42966077E-1, 1.71310842E-1, 3.03013414E-1},
{2.07741663E-1, 2.58691758E-1, 2.88766950E-1},
{1.71776935E-1, 2.40246087E-1, 2.73284525E-1},
{2.71046638E-1, 2.85170943E-1, 3.27401131E-1},
{1.69854626E-1, 1.87545776E-1, 2.24484712E-1},
{2.15221986E-1, 2.27339745E-1, 2.95008808E-1},
{1.75596640E-1, 2.17936546E-1, 2.74879605E-1},
{2.34665439E-1, 2.89530903E-1, 3.16494375E-1},
{1.89946994E-1, 2.04953820E-1, 2.46955171E-1},
{2.37297818E-1, 2.68316716E-1, 2.90684313E-1},
{1.69963166E-1, 2.53367484E-1, 2.92533010E-1},
{2.70659864E-1, 2.97146112E-1, 3.56183976E-1},
{1.52539685E-1, 1.70138955E-1, 2.52703935E-1},
{2.19119206E-1, 2.35900700E-1, 2.69739121E-1},
{1.42245665E-1, 2.18184620E-1, 3.28218073E-1},
{2.61472821E-1, 2.78025657E-1, 3.02375883E-1},
{1.53526023E-1, 1.90727741E-1, 2.92820841E-1},
{2.09240988E-1, 2.49808684E-1, 3.24709088E-1},
{1.75176397E-1, 2.38646746E-1, 3.06392699E-1},
{2.73218870E-1, 3.03954989E-1, 3.20513874E-1},
{1.63911596E-1, 1.89611584E-1, 2.56272525E-1},
{2.26953760E-1, 2.40120232E-1, 2.92728513E-1},
{1.95565715E-1, 2.11956203E-1, 2.97374696E-1},
{2.41045550E-1, 2.88497001E-1, 3.36352319E-1},
{1.94948331E-1, 2.09475279E-1, 2.56309658E-1},
{2.47884631E-1, 2.63356417E-1, 3.11270863E-1},
{1.69189706E-1, 2.35864580E-1, 3.36249381E-1},
{2.86001563E-1, 3.25423747E-1, 3.59607369E-1},
{1.56258598E-1, 1.76704943E-1, 2.14393437E-1},
{2.08996847E-1, 2.23968685E-1, 2.60886759E-1},
{1.35765389E-1, 2.03580052E-1, 3.05503219E-1},
{2.18961373E-1, 2.79463500E-1, 2.99450845E-1},
{1.34064749E-1, 1.78332120E-1, 2.90169626E-1},
{2.13298395E-1, 2.40031511E-1, 3.00345927E-1},
{1.64373413E-1, 2.26438701E-1, 2.87171155E-1},
{2.50739604E-1, 2.80812472E-1, 3.35349351E-1},
{1.63649514E-1, 1.97108001E-1, 2.21165180E-1},
{2.08139613E-1, 2.30869800E-1, 2.96137065E-1},
{1.59113124E-1, 2.18189180E-1, 2.95531958E-1},
{2.39883497E-1, 2.81831235E-1, 3.26045603E-1},
{1.89394727E-1, 2.08127141E-1, 2.38446414E-1},
{2.32995704E-1, 2.59603471E-1, 2.93427974E-1},
{1.60558835E-1, 2.55164832E-1, 3.02872926E-1},
{2.53509283E-1, 2.96028465E-1, 3.67721587E-1},
{1.30124375E-1, 1.74838990E-1, 2.60486037E-1},
{2.10203990E-1, 2.33570784E-1, 2.83061892E-1},
{1.52365491E-1, 2.25338757E-1, 3.03720981E-1},
{2.40558609E-1, 2.77192205E-1, 3.05891901E-1},
{1.63728818E-1, 1.94779396E-1, 2.69253582E-1},
{2.25709423E-1, 2.40902692E-1, 3.18060607E-1},
{1.92055091E-1, 2.29857832E-1, 2.89826721E-1},
{2.62759686E-1, 3.04292172E-1, 3.35680574E-1},
{1.66071162E-1, 2.06819177E-1, 2.39712462E-1},
{2.23915562E-1, 2.50106871E-1, 2.85296232E-1},
{1.88402340E-1, 2.03793734E-1, 3.03041130E-1},
{2.30698988E-1, 2.87044138E-1, 3.49802762E-1},
{1.82025358E-1, 2.14073509E-1, 2.63470024E-1},
{2.37297758E-1, 2.65025407E-1, 3.17815512E-1},
{1.89278707E-1, 2.58802205E-1, 3.04866165E-1},
{2.97243059E-1, 3.17153066E-1, 3.56583923E-1},
{1.58607468E-1, 1.78659767E-1, 2.41919369E-1},
{1.94887385E-1, 2.41695851E-1, 2.62176663E-1},
{1.58124432E-1, 2.11753070E-1, 3.11352164E-1},
{2.16902718E-1, 2.98796803E-1, 3.20994049E-1},
{1.49272785E-1, 1.74964130E-1, 3.15334409E-1},
{2.21622273E-1, 2.56179065E-1, 3.03902954E-1},
{1.75979599E-1, 2.43505448E-1, 2.85801739E-1},
{2.64590383E-1, 2.85541564E-1, 3.45107764E-1},
{1.80137083E-1, 2.05279350E-1, 2.22255990E-1},
{2.10796222E-1, 2.26315439E-1, 3.14426929E-1},
{1.79151163E-1, 2.09439725E-1, 2.93280870E-1},
{2.49719024E-1, 2.91257650E-1, 3.27162296E-1},
{1.98700234E-1, 2.15896755E-1, 2.49960214E-1},
{2.40726396E-1, 2.64857739E-1, 2.99639553E-1},
{1.71249732E-1, 2.68166155E-1, 3.03572744E-1},
{2.69555569E-1, 3.16100627E-1, 3.56570691E-1},
{1.50564745E-1, 1.84190869E-1, 2.68674821E-1},
{2.16941193E-1, 2.40813971E-1, 2.78942198E-1},
{1.35399476E-1, 2.60586530E-1, 3.32604855E-1},
{2.56150961E-1, 2.87822872E-1, 3.06156367E-1},
{1.66398838E-1, 1.88721806E-1, 2.93023735E-1},
{2.29214087E-1, 2.61565417E-1, 3.27494055E-1},
{1.98266640E-1, 2.32970506E-1, 2.99134284E-1},
{2.87046254E-1, 3.07103783E-1, 3.27298075E-1},
{1.75898686E-1, 2.11898595E-1, 2.51332909E-1},
{2.32067421E-1, 2.44622201E-1, 2.99443692E-1},
{1.90780059E-1, 2.12090015E-1, 3.25059265E-1},
{2.31531218E-1, 3.14166099E-1, 3.42735857E-1},
{1.95099846E-1, 2.09554315E-1, 2.79483467E-1},
{2.40416065E-1, 2.69604772E-1, 3.28015476E-1},
{1.71800867E-1, 2.82233089E-1, 3.14749271E-1},
{2.69243777E-1, 3.38462502E-1, 3.79935652E-1},
{1.59934625E-1, 1.77966774E-1, 2.00818628E-1},
{2.01979712E-1, 2.30668545E-1, 2.56773323E-1},
{1.34024277E-1, 2.10961610E-1, 2.84687728E-1},
{2.03712896E-1, 2.83053070E-1, 3.03309411E-1},
{1.44528881E-1, 1.64728075E-1, 2.85079390E-1},
{2.06285611E-1, 2.48649031E-1, 2.96383053E-1},
{1.58138171E-1, 2.34317720E-1, 2.79650003E-1},
{2.64995635E-1, 2.79900700E-1, 3.18619400E-1},
{1.66537479E-1, 1.84279412E-1, 2.14547485E-1},
{2.03051880E-1, 2.35110492E-1, 2.88755983E-1},
{1.68422714E-1, 2.03946173E-1, 2.87478894E-1},
{2.31727019E-1, 2.74086386E-1, 3.24755162E-1},
{1.85356215E-1, 2.14113116E-1, 2.29030401E-1},
{2.42482558E-1, 2.60655493E-1, 2.83030301E-1},
{1.67562261E-1, 2.42027491E-1, 2.99461991E-1},
{2.38809898E-1, 3.19003850E-1, 3.58415872E-1},
{1.37908265E-1, 1.54787809E-1, 2.65611202E-1},
{2.11019263E-1, 2.24607319E-1, 2.79954702E-1},
{1.37569889E-1, 2.25128531E-1, 3.09312850E-1},
{2.29239866E-1, 2.76150972E-1, 3.15241843E-1},
{1.60487458E-1, 1.95461214E-1, 2.83169478E-1},
{2.18505666E-1, 2.38197207E-1, 3.30340117E-1},
{1.81991324E-1, 2.33026952E-1, 2.93276042E-1},
{2.54552305E-1, 3.14394146E-1, 3.36392254E-1},
{1.44095764E-1, 2.26640165E-1, 2.50595063E-1},
{2.15188012E-1, 2.51417249E-1, 2.85043985E-1},
{1.87674388E-1, 2.04458863E-1, 2.94168979E-1},
{2.30494842E-1, 2.68452436E-1, 3.52370054E-1},
{1.85022101E-1, 1.99075252E-1, 2.71930546E-1},
{2.42569372E-1, 2.55389154E-1, 3.11399311E-1},
{1.95166096E-1, 2.49102056E-1, 2.98998445E-1},
{2.83654153E-1, 3.14600259E-1, 3.55619401E-1},
{1.51490018E-1, 1.97729796E-1, 2.32467473E-1},
{2.00029895E-1, 2.30101258E-1, 2.81933933E-1},
{1.38711318E-1, 1.91816628E-1, 3.45780402E-1},
{1.96580395E-1, 3.04714769E-1, 3.40553433E-1},
{1.38154253E-1, 1.88543141E-1, 2.99461216E-1},
{2.05666468E-1, 2.68904895E-1, 3.05537194E-1},
{1.72447845E-1, 2.33558387E-1, 2.93625206E-1},
{2.70145416E-1, 2.98654765E-1, 3.28556389E-1},
{1.75489411E-1, 1.91361547E-1, 2.35585332E-1},
{2.20548794E-1, 2.34773993E-1, 2.95397669E-1},
{1.85652360E-1, 2.22349137E-1, 2.79883891E-1},
{2.29456946E-1, 3.04546326E-1, 3.24684292E-1},
{1.86900780E-1, 2.15469390E-1, 2.51856804E-1},
{2.34910533E-1, 2.71217376E-1, 2.99894661E-1},
{1.85142443E-1, 2.56071001E-1, 2.93291301E-1},
{2.63883710E-1, 3.07127446E-1, 3.62546653E-1},
{1.60997644E-1, 1.78937852E-1, 2.55808324E-1},
{2.25671068E-1, 2.43735075E-1, 2.68624991E-1},
{1.55076161E-1, 2.30396181E-1, 3.21005553E-1},
{2.51760483E-1, 2.79653400E-1, 3.14202160E-1},
{1.56988814E-1, 2.07466930E-1, 2.89933950E-1},
{2.17479482E-1, 2.59626418E-1, 3.40659052E-1},
{1.76811531E-1, 2.31087089E-1, 3.17562491E-1},
{2.82952607E-1, 2.99844354E-1, 3.36822897E-1},
{1.82060316E-1, 1.98734730E-1, 2.51980305E-1},
{2.25874200E-1, 2.52469152E-1, 2.93356389E-1},
{2.00799957E-1, 2.17786849E-1, 3.02210063E-1},
{2.47423753E-1, 2.86882848E-1, 3.47820610E-1},
{2.01128140E-1, 2.14746892E-1, 2.62269646E-1},
{2.53963351E-1, 2.69477993E-1, 3.12133819E-1},
{1.91034868E-1, 2.55738169E-1, 3.32559615E-1},
{2.91053712E-1, 3.31458420E-1, 3.68588477E-1},
{1.57229915E-1, 1.85374141E-1, 2.25361317E-1},
{2.08051339E-1, 2.38350868E-1, 2.64212936E-1},
{1.46848336E-1, 2.13000089E-1, 3.00192565E-1},
{2.18630567E-1, 2.90263802E-1, 3.09045762E-1},
{1.43699184E-1, 1.87815160E-1, 2.83769876E-1},
{2.07328036E-1, 2.45088696E-1, 3.08956414E-1},
{1.64228097E-1, 2.27826655E-1, 3.08907896E-1},
{2.61919737E-1, 2.91333705E-1, 3.31527978E-1},
{1.70648888E-1, 2.02157527E-1, 2.17827827E-1},
{2.07796112E-1, 2.34704822E-1, 3.06783766E-1},
{1.72118798E-1, 2.14057386E-1, 3.10151786E-1},
{2.29116157E-1, 2.80949861E-1, 3.33774298E-1},
{1.96622208E-1, 2.16653049E-1, 2.33279720E-1},
{2.37789229E-1, 2.58971304E-1, 3.04609209E-1},
{1.55182019E-1, 2.63032585E-1, 3.18943053E-1},
{2.49388829E-1, 3.16970855E-1, 3.77762467E-1},
{1.51363596E-1, 1.75010651E-1, 2.78245836E-1},
{2.19810233E-1, 2.32360214E-1, 2.85034925E-1},
{1.42630622E-1, 2.40602851E-1, 3.04125100E-1},
{2.42764875E-1, 2.83762127E-1, 3.15481216E-1},
{1.57467470E-1, 2.07524061E-1, 2.75674909E-1},
{2.28758618E-1, 2.49092206E-1, 3.28139395E-1},
{1.90872714E-1, 2.38125205E-1, 2.94894546E-1},
{2.66389251E-1, 3.14321429E-1, 3.38669509E-1},
{1.70644209E-1, 2.25980043E-1, 2.47372389E-1},
{2.36442789E-1, 2.53003448E-1, 2.88220435E-1},
{1.85423777E-1, 2.04888850E-1, 3.14608842E-1},
{2.17379019E-1, 2.94553548E-1, 3.67831022E-1},
{1.88563988E-1, 2.15174288E-1, 2.72999734E-1},
{2.45102122E-1, 2.59770364E-1, 3.21885556E-1},
{1.98444173E-1, 2.61160702E-1, 3.17097872E-1},
{2.99013853E-1, 3.28965336E-1, 3.56681198E-1},
{1.58248767E-1, 1.92205697E-1, 2.46059090E-1},
{2.02385351E-1, 2.47965842E-1, 2.71749645E-1},
{1.61710784E-1, 2.13708103E-1, 3.27384740E-1},
{2.14419708E-1, 3.05552453E-1, 3.33721548E-1},
{1.61819980E-1, 1.89897299E-1, 3.10501546E-1},
{2.19436333E-1, 2.65029579E-1, 3.09288830E-1},
{1.88303933E-1, 2.49633163E-1, 2.85499543E-1},
{2.69325376E-1, 2.99807042E-1, 3.41722459E-1},
{1.72406003E-1, 2.10977256E-1, 2.27773219E-1},
{2.20281526E-1, 2.34015763E-1, 3.12846094E-1},
{1.83257267E-1, 2.22061962E-1, 2.91052371E-1},
{2.42531225E-1, 3.09527606E-1, 3.30389649E-1},
{2.07546696E-1, 2.24662632E-1, 2.44420141E-1},
{2.45858207E-1, 2.70285994E-1, 3.05132121E-1},
{1.84840545E-1, 2.72096783E-1, 3.12531084E-1},
{2.74252594E-1, 3.21252435E-1, 3.74658197E-1},
{1.66425839E-1, 1.84491634E-1, 2.68278092E-1},
{2.28423670E-1, 2.43025422E-1, 2.81184882E-1},
{1.60091296E-1, 2.52953321E-1, 3.35822314E-1},
{2.62109995E-1, 2.95581907E-1, 3.13354105E-1},
{1.67702749E-1, 2.01536924E-1, 3.01801592E-1},
{2.37822965E-1, 2.59894758E-1, 3.38231117E-1},
{1.97206214E-1, 2.45490909E-1, 3.17895442E-1},
{2.98455298E-1, 3.19209784E-1, 3.40971738E-1},
{1.71195343E-1, 2.24327832E-1, 2.62736112E-1},
{2.30626896E-1, 2.53310233E-1, 3.01206797E-1},
{2.04814211E-1, 2.21881568E-1, 3.25966567E-1},
{2.22987518E-1, 3.06339115E-1, 3.50717157E-1},
{2.00855389E-1, 2.15359926E-1, 2.84143478E-1},
{2.50951648E-1, 2.66189247E-1, 3.33360583E-1},
{1.75610259E-1, 2.93791324E-1, 3.40326935E-1},
{2.91745067E-1, 3.40602487E-1, 3.81397158E-1}};

static const float evrc_lspq_full_codebook4[128][3] = {
{2.77461529E-1, 3.16972077E-1, 3.95498335E-1},
{3.36560428E-1, 3.60156953E-1, 3.81473005E-1},
{3.10509324E-1, 3.31732392E-1, 3.66864383E-1},
{3.37470949E-1, 3.96795273E-1, 4.12356317E-1},
{2.79660404E-1, 3.66520107E-1, 3.85313451E-1},
{3.16038966E-1, 3.85609329E-1, 4.01304781E-1},
{3.09960425E-1, 3.43410730E-1, 4.24745500E-1},
{3.54243636E-1, 4.08699274E-1, 4.22167957E-1},
{2.95587242E-1, 3.33741128E-1, 3.87421668E-1},
{3.33446383E-1, 3.86974752E-1, 4.01353061E-1},
{3.23412836E-1, 3.65269661E-1, 3.85193288E-1},
{3.42731953E-1, 4.03192520E-1, 4.19920385E-1},
{2.77681828E-1, 3.82494986E-1, 4.04274166E-1},
{3.18247974E-1, 3.95985305E-1, 4.31353152E-1},
{3.03711414E-1, 3.80319715E-1, 4.37173545E-1},
{3.78288805E-1, 4.07077312E-1, 4.22679126E-1},
{2.38116503E-1, 3.42454314E-1, 4.24624741E-1},
{3.45615685E-1, 3.68681073E-1, 4.00817335E-1},
{3.17688107E-1, 3.41902673E-1, 4.05601799E-1},
{3.66368949E-1, 3.89039934E-1, 4.06154454E-1},
{2.99398005E-1, 3.52021694E-1, 3.99955690E-1},
{3.24991941E-1, 3.90028834E-1, 4.19478714E-1},
{3.23025763E-1, 3.68114293E-1, 4.02087748E-1},
{3.62326264E-1, 4.16927993E-1, 4.32773650E-1},
{2.72696435E-1, 3.59205008E-1, 4.26880658E-1},
{3.46539855E-1, 3.69616628E-1, 4.15621221E-1},
{3.34109128E-1, 3.55736315E-1, 3.96749556E-1},
{3.37468982E-1, 4.10392702E-1, 4.25986826E-1},
{2.99468994E-1, 3.80648255E-1, 4.18284118E-1},
{3.21378171E-1, 4.11198020E-1, 4.28792536E-1},
{3.27841163E-1, 3.69345129E-1, 4.34395611E-1},
{3.80669057E-1, 4.26086366E-1, 4.42754567E-1},
{2.68943667E-1, 3.42942953E-1, 3.98681462E-1},
{3.38102877E-1, 3.76338840E-1, 3.92043173E-1},
{3.23593497E-1, 3.48742068E-1, 3.72551978E-1},
{3.47550809E-1, 3.92885387E-1, 4.21169937E-1},
{3.04182827E-1, 3.59816670E-1, 3.81633341E-1},
{3.14221382E-1, 4.02108550E-1, 4.20085251E-1},
{3.01306546E-1, 3.62662733E-1, 4.29262817E-1},
{3.71770263E-1, 3.98696363E-1, 4.31438982E-1},
{2.74591267E-1, 3.35595489E-1, 4.20079648E-1},
{3.44540834E-1, 3.90451789E-1, 4.06412065E-1},
{3.25239837E-1, 3.78344476E-1, 3.94673288E-1},
{3.56683493E-1, 3.90574157E-1, 4.33851063E-1},
{2.63501287E-1, 3.95260096E-1, 4.23116386E-1},
{3.37520659E-1, 3.92563462E-1, 4.43415821E-1},
{3.14522266E-1, 3.80968630E-1, 4.22676384E-1},
{3.76235068E-1, 4.17298734E-1, 4.31451261E-1},
{2.61855006E-1, 3.68646085E-1, 4.04260576E-1},
{3.55580151E-1, 3.77994478E-1, 3.95868242E-1},
{3.27742815E-1, 3.53872776E-1, 4.11040604E-1},
{3.62960637E-1, 3.99466991E-1, 4.14690197E-1},
{3.09410870E-1, 3.73796046E-1, 3.92672479E-1},
{3.31016302E-1, 4.00801599E-1, 4.31759298E-1},
{3.23573053E-1, 3.68619561E-1, 4.17455137E-1},
{3.49115849E-1, 4.26840067E-1, 4.43913996E-1},
{2.89738595E-1, 3.63759339E-1, 4.10511792E-1},
{3.55286479E-1, 3.89331281E-1, 4.13432419E-1},
{3.36565912E-1, 3.60222459E-1, 4.24179018E-1},
{3.39932680E-1, 4.09228802E-1, 4.40184891E-1},
{3.00889730E-1, 4.00081098E-1, 4.17955697E-1},
{3.17052066E-1, 4.22288120E-1, 4.42229569E-1},
{3.27336788E-1, 3.84311676E-1, 4.30288613E-1},
{3.98990929E-1, 4.29498434E-1, 4.43475187E-1},
{2.49110118E-1, 3.25696886E-1, 4.11728263E-1},
{3.45929205E-1, 3.68577540E-1, 3.88473272E-1},
{3.13219666E-1, 3.39229465E-1, 3.87597919E-1},
{3.51453960E-1, 3.98730278E-1, 4.12656188E-1},
{2.93487132E-1, 3.75763118E-1, 3.94488096E-1},
{3.24470758E-1, 3.94202888E-1, 4.08882737E-1},
{3.12710822E-1, 3.57720256E-1, 4.14061189E-1},
{3.66507173E-1, 4.08171296E-1, 4.23891425E-1},
{2.99965680E-1, 3.31993401E-1, 4.07860160E-1},
{3.34925175E-1, 3.86143029E-1, 4.11538124E-1},
{3.34788024E-1, 3.66196156E-1, 3.93347144E-1},
{3.47847939E-1, 4.05926466E-1, 4.30507302E-1},
{2.85952926E-1, 3.95283282E-1, 4.16119337E-1},
{3.23867381E-1, 4.06476676E-1, 4.42482829E-1},
{3.16716671E-1, 3.84451628E-1, 4.39411044E-1},
{3.86772931E-1, 4.11824584E-1, 4.27831531E-1},
{2.38072395E-1, 3.62342358E-1, 4.30931687E-1},
{3.46450031E-1, 3.79082918E-1, 4.06567812E-1},
{3.16576600E-1, 3.56468618E-1, 3.96218300E-1},
{3.66539180E-1, 3.89590919E-1, 4.21055555E-1},
{3.08291376E-1, 3.71324301E-1, 4.07867432E-1},
{3.36435199E-1, 3.91514421E-1, 4.22977090E-1},
{3.23035538E-1, 3.80447328E-1, 4.09550190E-1},
{3.65228057E-1, 4.27910388E-1, 4.43691254E-1},
{2.72038043E-1, 3.76596808E-1, 4.33685899E-1},
{3.57665777E-1, 3.77761602E-1, 4.09178972E-1},
{3.36498559E-1, 3.64215910E-1, 4.09255505E-1},
{3.48082423E-1, 4.17631805E-1, 4.33284521E-1},
{3.02754521E-1, 3.95974755E-1, 4.33717251E-1},
{3.31676304E-1, 4.17587161E-1, 4.36239839E-1},
{3.33287597E-1, 3.80799115E-1, 4.39620733E-1},
{3.88112009E-1, 4.36933577E-1, 4.50829268E-1},
{2.56026626E-1, 3.48015189E-1, 4.22922611E-1},
{3.45773995E-1, 3.81725788E-1, 3.96794081E-1},
{3.25623751E-1, 3.50391924E-1, 3.87330651E-1},
{3.56868088E-1, 3.98574769E-1, 4.23177242E-1},
{3.01226199E-1, 3.86906981E-1, 4.03335571E-1},
{3.28178406E-1, 4.02090192E-1, 4.19389248E-1},
{3.14385355E-1, 3.69043887E-1, 4.34375286E-1},
{3.72321129E-1, 4.11672413E-1, 4.40518737E-1},
{2.90479720E-1, 3.48121881E-1, 4.26216483E-1},
{3.44438791E-1, 3.82666349E-1, 4.17321086E-1},
{3.34866822E-1, 3.76235664E-1, 4.04475212E-1},
{3.59025359E-1, 4.04721916E-1, 4.34838414E-1},
{2.79127955E-1, 4.11106586E-1, 4.35360551E-1},
{3.48125517E-1, 3.98732066E-1, 4.46927428E-1},
{3.27018857E-1, 3.90107334E-1, 4.41707492E-1},
{3.90858352E-1, 4.19813931E-1, 4.35153484E-1},
{2.55319297E-1, 3.70405972E-1, 4.32188630E-1},
{3.54651988E-1, 3.88332665E-1, 4.02956128E-1},
{3.21608186E-1, 3.54489803E-1, 4.28299785E-1},
{3.75163496E-1, 3.98833990E-1, 4.14177418E-1},
{3.11953604E-1, 3.91430676E-1, 4.12552476E-1},
{3.42528820E-1, 3.96365345E-1, 4.32497382E-1},
{3.33744347E-1, 3.76422405E-1, 4.20536995E-1},
{3.53529096E-1, 4.29231048E-1, 4.59699273E-1},
{2.88017929E-1, 3.77999961E-1, 4.34011698E-1},
{3.55683446E-1, 3.80780041E-1, 4.23145533E-1},
{3.44358265E-1, 3.72184873E-1, 4.31265354E-1},
{3.53966117E-1, 4.14166689E-1, 4.42941308E-1},
{3.04770231E-1, 4.12517488E-1, 4.34183121E-1},
{3.35913360E-1, 4.24590766E-1, 4.46378469E-1},
{3.43738198E-1, 3.84766221E-1, 4.35271382E-1},
{4.10941303E-1, 4.40662980E-1, 4.52113390E-1}};

static const float evrc_lspq_half_codebook1[128][3] = {
{1.35226343E-2, 1.82081293E-2, 3.93940695E-2},
{2.29392890E-2, 3.57831158E-2, 1.05352886E-1},
{2.09106486E-2, 3.04159056E-2, 8.93941075E-2},
{1.88909005E-2, 3.82722206E-2, 1.37820408E-1},
{2.05143820E-2, 2.85481159E-2, 7.39762187E-2},
{4.69510332E-2, 6.84031919E-2, 1.09123811E-1},
{3.15557197E-2, 5.69139980E-2, 8.57057571E-2},
{3.81181911E-2, 7.77784660E-2, 1.92532852E-1},
{2.16297153E-2, 2.92908940E-2, 6.25042021E-2},
{3.11414022E-2, 5.99079318E-2, 1.02860682E-1},
{3.02799307E-2, 5.35012372E-2, 7.80925751E-2},
{6.50846213E-2, 9.06624720E-2, 1.42850950E-1},
{3.27340364E-2, 5.04027791E-2, 6.26492277E-2},
{5.27439862E-2, 6.22574277E-2, 1.22198336E-1},
{3.48840356E-2, 6.42222390E-2, 9.16024595E-2},
{4.88984436E-2, 1.05058022E-1, 1.68813452E-1},
{2.35791076E-2, 3.21034677E-2, 5.60899563E-2},
{2.77252812E-2, 4.87281792E-2, 1.01224191E-1},
{2.74348017E-2, 4.04965915E-2, 9.34926122E-2},
{4.38360050E-2, 6.03261292E-2, 1.52400866E-1},
{2.68994924E-2, 4.52906378E-2, 6.49800375E-2},
{5.16058952E-2, 6.08312152E-2, 1.08799636E-1},
{4.20064926E-2, 6.11845106E-2, 8.54474008E-2},
{7.13502690E-2, 1.01972111E-1, 1.74640998E-1},
{2.88906675E-2, 4.13964354E-2, 5.25928028E-2},
{3.16364467E-2, 6.63532093E-2, 1.24950245E-1},
{4.30289507E-2, 5.14023267E-2, 7.96877742E-2},
{5.70970774E-2, 1.08444504E-1, 1.44075617E-1},
{3.38840261E-2, 5.04746847E-2, 7.29765445E-2},
{6.54265657E-2, 7.90987685E-2, 1.15570590E-1},
{3.85423526E-2, 7.33125433E-2, 1.02307513E-1},
{6.57824501E-2, 1.02909811E-1, 2.11874440E-1},
{1.54727865E-2, 2.04559695E-2, 5.46121262E-2},
{2.27950197E-2, 3.90954204E-2, 1.19443826E-1},
{3.06889173E-2, 4.54540215E-2, 8.20418894E-2},
{2.25957241E-2, 4.79101725E-2, 1.71844408E-1},
{2.71088015E-2, 4.01739590E-2, 7.01922849E-2},
{4.95789349E-2, 7.92963281E-2, 1.04862511E-1},
{3.06095853E-2, 5.64059429E-2, 9.49584097E-2},
{6.34224564E-2, 9.11655501E-2, 1.84724405E-1},
{2.43342388E-2, 3.91998328E-2, 6.31406233E-2},
{3.38011980E-2, 6.60846457E-2, 1.11031540E-1},
{3.51784080E-2, 5.79397269E-2, 7.20702857E-2},
{6.49054050E-2, 8.65831897E-2, 1.54648736E-1},
{2.91934665E-2, 5.16204573E-2, 6.94437325E-2},
{5.94522804E-2, 7.19829276E-2, 1.27434507E-1},
{5.31888530E-2, 6.38182089E-2, 9.88218486E-2},
{8.68290961E-2, 1.41135350E-1, 1.91728458E-1},
{2.49991138E-2, 3.62556018E-2, 5.03724031E-2},
{2.82246377E-2, 5.44572286E-2, 1.12663500E-1},
{3.62618119E-2, 4.59073223E-2, 9.43343639E-2},
{5.70455343E-2, 7.46300444E-2, 1.59157172E-1},
{2.72987466E-2, 4.56625856E-2, 7.52529651E-2},
{5.12860194E-2, 8.51126984E-2, 1.23587973E-1},
{4.91451994E-2, 5.93483113E-2, 9.22686011E-2},
{7.06961900E-2, 1.05451979E-1, 1.92602143E-1},
{2.80733760E-2, 4.18509208E-2, 5.87159805E-2},
{4.64449003E-2, 7.06698820E-2, 1.26038432E-1},
{4.18453738E-2, 6.30445331E-2, 7.66169876E-2},
{8.42416435E-2, 1.13282882E-1, 1.43687114E-1},
{4.17615622E-2, 5.59472926E-2, 7.09872842E-2},
{5.55161387E-2, 9.50126722E-2, 1.27727196E-1},
{5.90935498E-2, 7.36730024E-2, 9.65935886E-2},
{7.84136653E-2, 1.41432360E-1, 2.17428640E-1},
{2.10490543E-2, 2.91891042E-2, 4.60035764E-2},
{3.64863276E-2, 4.62387018E-2, 1.07044168E-1},
{2.68652122E-2, 3.92937548E-2, 8.41179937E-2},
{2.72903945E-2, 5.53805046E-2, 1.41586170E-1},
{2.48476695E-2, 3.63277681E-2, 7.62430876E-2},
{5.25430813E-2, 7.75778666E-2, 1.14567965E-1},
{4.07741442E-2, 5.39923795E-2, 9.07640457E-2},
{5.73043302E-2, 7.65803084E-2, 1.79578975E-1},
{2.46032421E-2, 3.41408364E-2, 6.78990781E-2},
{4.08220068E-2, 6.29783794E-2, 9.95191261E-2},
{3.83025035E-2, 5.52857481E-2, 7.90019333E-2},
{7.24111274E-2, 1.01903863E-1, 1.46979645E-1},
{3.73902172E-2, 4.70463894E-2, 6.54684529E-2},
{5.27397543E-2, 6.72770366E-2, 1.39680430E-1},
{4.05365378E-2, 7.05081299E-2, 9.25668627E-2},
{4.43425253E-2, 1.10367171E-1, 1.99636266E-1},
{2.54920740E-2, 3.47603969E-2, 6.05902039E-2},
{4.35465500E-2, 5.32369502E-2, 1.08325966E-1},
{2.79599819E-2, 4.91324775E-2, 8.84284526E-2},
{4.98051867E-2, 8.81728902E-2, 1.52597323E-1},
{3.19346264E-2, 4.62169312E-2, 6.85206428E-2},
{5.80246300E-2, 6.84268698E-2, 1.15085281E-1},
{4.33904678E-2, 6.90575615E-2, 8.44984353E-2},
{7.39691556E-2, 1.19240515E-1, 1.77340195E-1},
{3.18767503E-2, 4.59697433E-2, 5.72372638E-2},
{4.50873822E-2, 5.66509366E-2, 1.32005826E-1},
{4.59097028E-2, 5.45580424E-2, 8.61423314E-2},
{7.44685754E-2, 1.13815404E-1, 1.61570594E-1},
{3.97509560E-2, 4.95359488E-2, 7.22542256E-2},
{6.76257759E-2, 8.31029043E-2, 1.27990112E-1},
{5.76258078E-2, 6.95326403E-2, 1.05012968E-1},
{6.85313493E-2, 1.21758826E-1, 2.20626548E-1},
{2.18480472E-2, 2.99130920E-2, 5.16208000E-2},
{3.64343151E-2, 4.91795056E-2, 1.23277210E-1},
{3.89611274E-2, 4.76634987E-2, 8.61716568E-2},
{4.14635167E-2, 6.88006952E-2, 1.69356152E-1},
{3.35514620E-2, 4.17815186E-2, 7.37159401E-2},
{5.80224693E-2, 8.70314166E-2, 1.12917498E-1},
{4.80243117E-2, 5.69486506E-2, 1.00755706E-1},
{5.98873124E-2, 8.57942328E-2, 2.01388851E-1},
{2.99309995E-2, 3.94828431E-2, 6.46376088E-2},
{3.88626605E-2, 8.07443634E-2, 1.15519784E-1},
{3.49444002E-2, 6.28911033E-2, 8.04982036E-2},
{6.88817874E-2, 9.92431119E-2, 1.60393253E-1},
{3.64237651E-2, 5.34016453E-2, 6.70152009E-2},
{5.83492741E-2, 7.85285756E-2, 1.41746715E-1},
{4.86469641E-2, 7.26736858E-2, 9.48315859E-2},
{5.85533604E-2, 1.36289746E-1, 1.98639736E-1},
{2.60888506E-2, 3.73406820E-2, 5.57853170E-2},
{4.58504409E-2, 5.60512505E-2, 1.17927872E-1},
{4.28801328E-2, 5.14739119E-2, 9.75309014E-2},
{6.37611598E-2, 8.73552933E-2, 1.68334916E-1},
{3.76709923E-2, 4.58216034E-2, 7.86528140E-2},
{6.75194561E-2, 8.98697898E-2, 1.19418114E-1},
{5.46374246E-2, 6.66805878E-2, 8.93813819E-2},
{7.73086548E-2, 1.21754415E-1, 1.99579224E-1},
{3.15621309E-2, 4.51702215E-2, 6.25768527E-2},
{3.78782675E-2, 8.03486481E-2, 1.38961688E-1},
{5.08303270E-2, 6.18740581E-2, 8.31153840E-2},
{8.96311402E-2, 1.28753766E-1, 1.64891586E-1},
{4.73503470E-2, 5.75724356E-2, 7.65264630E-2},
{7.16898590E-2, 9.89895687E-2, 1.30078360E-1},
{6.29082546E-2, 7.90778771E-2, 1.05111063E-1},
{8.80649835E-2, 1.65206164E-1, 2.13214174E-1}};

static const float evrc_lspq_half_codebook2[128][3] = {
{9.75915268E-2, 1.23701490E-1, 1.69437975E-1},
{9.49536338E-2, 2.01081768E-1, 2.26855248E-1},
{9.00496617E-2, 1.49164870E-1, 2.26532787E-1},
{1.70302704E-1, 1.97222874E-1, 2.49974832E-1},
{1.08773641E-1, 1.51972428E-1, 1.75123364E-1},
{1.30278930E-1, 2.13229164E-1, 2.29646355E-1},
{1.24917991E-1, 1.87347755E-1, 2.04712003E-1},
{2.00670198E-1, 2.28963569E-1, 2.69420803E-1},
{8.98375586E-2, 1.25332758E-1, 2.10539430E-1},
{9.62376669E-2, 2.07185850E-1, 2.54174471E-1},
{1.05694629E-1, 1.78856418E-1, 2.00121015E-1},
{1.56048968E-1, 2.19573721E-1, 2.91079402E-1},
{1.37392268E-1, 1.59993336E-1, 1.94698542E-1},
{1.07262500E-1, 2.37790957E-1, 2.70740807E-1},
{1.42976448E-1, 2.01550499E-1, 2.18468934E-1},
{2.14270487E-1, 2.71881402E-1, 3.01200211E-1},
{1.10729210E-1, 1.33688226E-1, 1.54877156E-1},
{1.06667660E-1, 1.76678821E-1, 2.62798905E-1},
{9.16352943E-2, 1.74592838E-1, 2.19329327E-1},
{1.84038624E-1, 2.27964059E-1, 2.47762203E-1},
{1.10572360E-1, 1.58207163E-1, 1.96013063E-1},
{1.33543387E-1, 2.32269660E-1, 2.51828164E-1},
{1.55922309E-1, 1.77941337E-1, 2.18096644E-1},
{1.92260072E-1, 2.49512479E-1, 2.89911509E-1},
{1.13708906E-1, 1.37872443E-1, 2.02929884E-1},
{1.02557532E-1, 1.84820071E-1, 2.92164624E-1},
{1.36595622E-1, 1.58687428E-1, 2.41399556E-1},
{1.72813818E-1, 2.49303415E-1, 3.00458610E-1},
{1.36871174E-1, 1.57249823E-1, 2.10913152E-1},
{1.28974810E-1, 2.45167866E-1, 2.67653584E-1},
{1.66812256E-1, 1.88998029E-1, 2.31345922E-1},
{2.32248470E-1, 2.63196051E-1, 3.16754937E-1},
{9.24560949E-2, 1.19977452E-1, 1.91262275E-1},
{1.13085262E-1, 2.08461538E-1, 2.29368120E-1},
{1.00716405E-1, 1.40670076E-1, 2.58062959E-1},
{1.67010382E-1, 2.18105540E-1, 2.62592494E-1},
{1.25487238E-1, 1.62686959E-1, 1.84409231E-1},
{1.52406558E-1, 2.07131729E-1, 2.47582436E-1},
{1.37441203E-1, 1.80262372E-1, 2.17698842E-1},
{2.07853511E-1, 2.49209508E-1, 2.69830108E-1},
{9.35257301E-2, 1.49197355E-1, 2.04652041E-1},
{1.11997180E-1, 2.25233063E-1, 2.47003049E-1},
{1.09315015E-1, 1.93811879E-1, 2.13802189E-1},
{1.75118580E-1, 2.52520263E-1, 2.75082767E-1},
{1.36918738E-1, 1.77440569E-1, 1.97931141E-1},
{1.36811242E-1, 2.37426177E-1, 2.84737825E-1},
{1.60759792E-1, 2.00833157E-1, 2.18084484E-1},
{2.33710244E-1, 2.66372561E-1, 2.91802049E-1},
{1.19171090E-1, 1.39703169E-1, 1.87723249E-1},
{1.31049946E-1, 1.93696663E-1, 2.60426998E-1},
{1.08267047E-1, 1.65194795E-1, 2.39523023E-1},
{2.03195021E-1, 2.25942209E-1, 2.49403238E-1},
{1.23842932E-1, 1.45794615E-1, 2.15635628E-1},
{1.71226338E-1, 2.38054529E-1, 2.57975638E-1},
{1.66923836E-1, 1.88604668E-1, 2.11124212E-1},
{2.10620746E-1, 2.62442708E-1, 2.83127964E-1},
{1.05748810E-1, 1.36286482E-1, 2.20050186E-1},
{9.72945765E-2, 2.33471528E-1, 2.96113968E-1},
{1.34298369E-1, 1.93955436E-1, 2.39148825E-1},
{1.64229318E-1, 2.70067751E-1, 2.94142485E-1},
{1.42760262E-1, 1.65033355E-1, 2.24100381E-1},
{1.46414533E-1, 2.47942328E-1, 3.00708115E-1},
{1.74778774E-1, 2.19349250E-1, 2.38162965E-1},
{2.36311123E-1, 2.90669680E-1, 3.28010976E-1},
{1.14076428E-1, 1.33071408E-1, 1.73181504E-1},
{1.13575839E-1, 1.90307274E-1, 2.41681188E-1},
{8.59165266E-2, 1.63920239E-1, 2.37934500E-1},
{1.92916945E-1, 2.15082392E-1, 2.39128128E-1},
{1.37291834E-1, 1.59423307E-1, 1.79722220E-1},
{1.40435383E-1, 2.22092256E-1, 2.40960747E-1},
{1.40387163E-1, 1.89601168E-1, 2.05635697E-1},
{2.11695507E-1, 2.36578360E-1, 2.81248927E-1},
{9.03010592E-2, 1.27157405E-1, 2.33567923E-1},
{1.10118054E-1, 2.09328398E-1, 2.72836268E-1},
{1.16710417E-1, 1.77853987E-1, 2.22808748E-1},
{1.81691542E-1, 2.32265159E-1, 2.74991214E-1},
{1.46553472E-1, 1.69474706E-1, 1.90245956E-1},
{1.09213792E-1, 2.63291955E-1, 2.88490772E-1},
{1.49815127E-1, 2.11342707E-1, 2.28899449E-1},
{1.97645500E-1, 2.83229947E-1, 3.14882278E-1},
{1.24495603E-1, 1.46097973E-1, 1.66125208E-1},
{1.34878591E-1, 1.83030054E-1, 2.89288282E-1},
{9.33032110E-2, 1.83962211E-1, 2.38543004E-1},
{1.92844257E-1, 2.39588335E-1, 2.58421540E-1},
{1.23796798E-1, 1.65556595E-1, 2.08408386E-1},
{1.51144341E-1, 2.35801116E-1, 2.59280622E-1},
{1.50657728E-1, 1.90052524E-1, 2.28362590E-1},
{1.98180959E-1, 2.56794214E-1, 3.08975637E-1},
{1.28490031E-1, 1.49084017E-1, 1.98376507E-1},
{9.20595750E-2, 2.12231293E-1, 2.92948842E-1},
{1.41698137E-1, 1.72356680E-1, 2.58454144E-1},
{1.96733460E-1, 2.29709730E-1, 2.95780182E-1},
{1.47062227E-1, 1.68918088E-1, 2.07363635E-1},
{1.36309877E-1, 2.60373056E-1, 2.82607377E-1},
{1.81041077E-1, 2.01826140E-1, 2.38867551E-1},
{2.45326266E-1, 2.80183077E-1, 3.11954319E-1},
{1.04131766E-1, 1.33040652E-1, 1.89834684E-1},
{1.23298146E-1, 2.09621087E-1, 2.47813210E-1},
{1.24040775E-1, 1.59827366E-1, 2.58856058E-1},
{1.87048867E-1, 2.12488100E-1, 2.59629130E-1},
{1.24255307E-1, 1.73768952E-1, 1.92850024E-1},
{1.58917829E-1, 2.25389823E-1, 2.43284762E-1},
{1.53421149E-1, 1.91807315E-1, 2.09249526E-1},
{2.27154449E-1, 2.51181155E-1, 2.72600353E-1},
{1.09922059E-1, 1.57100275E-1, 2.20024973E-1},
{1.32782355E-1, 2.19485506E-1, 2.67028928E-1},
{1.26857504E-1, 1.98836312E-1, 2.17928499E-1},
{1.91415027E-1, 2.52424240E-1, 2.72652745E-1},
{1.55277625E-1, 1.79573521E-1, 2.00773627E-1},
{1.17547743E-1, 2.47869864E-1, 3.08279335E-1},
{1.65706977E-1, 2.10339502E-1, 2.29199320E-1},
{2.25694910E-1, 2.84438193E-1, 3.12106073E-1},
{1.29503176E-1, 1.48420051E-1, 1.80180401E-1},
{1.54752508E-1, 1.97748467E-1, 2.67275035E-1},
{1.28590241E-1, 1.76178381E-1, 2.39905864E-1},
{2.14926764E-1, 2.37634435E-1, 2.58794010E-1},
{1.28322318E-1, 1.59338519E-1, 2.26626605E-1},
{1.55747548E-1, 2.47740522E-1, 2.73726821E-1},
{1.75741687E-1, 1.97952345E-1, 2.19115943E-1},
{2.18626365E-1, 2.45809183E-1, 3.00479650E-1},
{1.17709018E-1, 1.45512864E-1, 2.38044471E-1},
{1.18006893E-1, 2.23775521E-1, 2.94175088E-1},
{1.51349202E-1, 1.88157812E-1, 2.48743281E-1},
{1.89312205E-1, 2.69580543E-1, 2.93785989E-1},
{1.49895594E-1, 1.74537256E-1, 2.37430006E-1},
{1.39775530E-1, 2.71709383E-1, 3.07839513E-1},
{1.83945730E-1, 2.07717165E-1, 2.26722151E-1},
{2.54552156E-1, 2.96640933E-1, 3.24801445E-1}};

static const float evrc_lspq_half_codebook3[256][4] = {
{2.36904725E-1, 2.56104350E-1, 3.16955745E-1, 4.07520533E-1},
{2.97596931E-1, 3.23482454E-1, 3.47667515E-1, 3.74551237E-1},
{2.73721159E-1, 2.98297524E-1, 3.29923928E-1, 3.83599102E-1},
{3.07849586E-1, 3.32836270E-1, 3.89340341E-1, 4.05575991E-1},
{2.33803615E-1, 2.60296524E-1, 3.67351949E-1, 4.04388249E-1},
{2.97513664E-1, 3.15356553E-1, 3.85135233E-1, 4.02197123E-1},
{2.85618782E-1, 3.10872793E-1, 3.65022361E-1, 3.84816766E-1},
{3.35271597E-1, 3.55222225E-1, 3.81921113E-1, 3.98685753E-1},
{2.00265601E-1, 2.50502288E-1, 3.70398223E-1, 4.32012677E-1},
{3.07982087E-1, 3.33767712E-1, 3.58199060E-1, 3.78386796E-1},
{2.60086119E-1, 3.25520277E-1, 3.56873333E-1, 3.84737790E-1},
{3.01356375E-1, 3.41369390E-1, 4.00296748E-1, 4.17337179E-1},
{2.67080963E-1, 2.97674358E-1, 3.69702041E-1, 3.89139235E-1},
{2.72669852E-1, 3.49704087E-1, 3.91925275E-1, 4.06383276E-1},
{2.52825916E-1, 3.49636555E-1, 3.84550989E-1, 4.05930996E-1},
{3.42927098E-1, 3.74274015E-1, 4.05468166E-1, 4.20351923E-1},
{2.52408743E-1, 2.80375838E-1, 3.21436584E-1, 3.88436913E-1},
{2.96970189E-1, 3.17173600E-1, 3.65342557E-1, 4.02736843E-1},
{2.81905174E-1, 3.01479161E-1, 3.34335625E-1, 4.07633483E-1},
{3.26872945E-1, 3.47177684E-1, 3.75017703E-1, 4.05372381E-1},
{2.36371145E-1, 3.16441059E-1, 3.48707020E-1, 3.82030427E-1},
{2.87817597E-1, 3.13627005E-1, 4.05129731E-1, 4.23379660E-1},
{2.77502477E-1, 3.01843822E-1, 3.72250855E-1, 4.19212818E-1},
{3.28988850E-1, 3.61901104E-1, 4.02015507E-1, 4.19229805E-1},
{2.24960461E-1, 2.74636388E-1, 3.77016127E-1, 3.94726515E-1},
{3.01045477E-1, 3.40486169E-1, 3.74888122E-1, 4.02532160E-1},
{2.59897947E-1, 3.30334961E-1, 3.57493818E-1, 4.08657968E-1},
{3.00961852E-1, 3.56449068E-1, 4.04779494E-1, 4.22508955E-1},
{2.20979586E-1, 3.16477656E-1, 4.01744068E-1, 4.20735776E-1},
{2.79754996E-1, 3.30776095E-1, 4.11152899E-1, 4.32687044E-1},
{2.64246881E-1, 3.16610634E-1, 3.83876741E-1, 4.36683774E-1},
{3.44381154E-1, 3.85365665E-1, 4.24949467E-1, 4.41560209E-1},
{2.19488308E-1, 2.36459881E-1, 3.42465997E-1, 4.24989998E-1},
{2.91465104E-1, 3.22282016E-1, 3.72852802E-1, 3.91635895E-1},
{2.74792433E-1, 3.16536307E-1, 3.45392585E-1, 3.74555230E-1},
{3.10583472E-1, 3.35264921E-1, 3.87527227E-1, 4.23076212E-1},
{2.23211512E-1, 2.98497617E-1, 3.68426204E-1, 3.90213728E-1},
{2.89078832E-1, 3.26512754E-1, 3.76308680E-1, 4.09553707E-1},
{2.63830125E-1, 3.08977246E-1, 3.81453037E-1, 4.04660761E-1},
{3.47073615E-1, 3.64797831E-1, 3.86763453E-1, 4.04511690E-1},
{2.18452707E-1, 2.75614083E-1, 3.62711072E-1, 4.18278992E-1},
{3.15042794E-1, 3.40813220E-1, 3.78627181E-1, 3.96316767E-1},
{2.79727697E-1, 3.31259727E-1, 3.60061288E-1, 3.81175518E-1},
{3.18602443E-1, 3.38044286E-1, 4.09010768E-1, 4.30300415E-1},
{2.64196932E-1, 2.90672481E-1, 3.68595004E-1, 4.31856751E-1},
{2.72645593E-1, 3.63514841E-1, 3.96518826E-1, 4.20091212E-1},
{2.26540968E-1, 3.50055099E-1, 3.93851519E-1, 4.12597001E-1},
{3.53053868E-1, 3.69929552E-1, 4.09656048E-1, 4.26387310E-1},
{2.60788381E-1, 2.85172462E-1, 3.45943332E-1, 3.97500694E-1},
{3.01113129E-1, 3.28201890E-1, 3.56068015E-1, 4.10803795E-1},
{2.88101614E-1, 3.09559643E-1, 3.43756795E-1, 4.24872875E-1},
{3.10489357E-1, 3.51421893E-1, 3.93717408E-1, 4.15550530E-1},
{2.22308263E-1, 3.26798201E-1, 3.77981663E-1, 3.98635030E-1},
{3.02915514E-1, 3.22781920E-1, 3.98558855E-1, 4.25489604E-1},
{2.77136803E-1, 3.19992602E-1, 3.77490878E-1, 4.29177463E-1},
{3.38731766E-1, 3.58164370E-1, 4.08386350E-1, 4.25495386E-1},
{2.18726233E-1, 2.84384966E-1, 3.94053698E-1, 4.16346967E-1},
{3.01005960E-1, 3.44093680E-1, 3.69013667E-1, 4.15091276E-1},
{2.80783713E-1, 3.33053648E-1, 3.76726151E-1, 3.97526860E-1},
{3.14394057E-1, 3.62678826E-1, 4.23668981E-1, 4.41899240E-1},
{2.66453624E-1, 3.08513761E-1, 3.97407174E-1, 4.17450190E-1},
{2.94222653E-1, 3.41904402E-1, 4.12726879E-1, 4.34888899E-1},
{2.87300706E-1, 3.32434595E-1, 3.78856659E-1, 4.38234031E-1},
{3.57146621E-1, 3.98147047E-1, 4.29875731E-1, 4.44243908E-1},
{2.29671344E-1, 2.51018614E-1, 3.41046572E-1, 4.04376328E-1},
{2.94472575E-1, 3.34944606E-1, 3.60409737E-1, 3.83682847E-1},
{2.88250983E-1, 3.11722696E-1, 3.31680059E-1, 3.65104675E-1},
{3.24881613E-1, 3.45656693E-1, 3.88306379E-1, 4.05954897E-1},
{2.50829220E-1, 2.77623534E-1, 3.70799541E-1, 3.90479207E-1},
{2.93523371E-1, 3.28319192E-1, 3.92112255E-1, 4.09464061E-1},
{2.83608794E-1, 3.03885639E-1, 3.78504395E-1, 3.97310555E-1},
{3.34039807E-1, 3.52837384E-1, 3.97272944E-1, 4.14322019E-1},
{2.21891895E-1, 2.51877457E-1, 3.71723533E-1, 4.31791008E-1},
{3.13201427E-1, 3.41175437E-1, 3.65503550E-1, 3.88567209E-1},
{2.71330535E-1, 3.39163721E-1, 3.62616420E-1, 3.95735979E-1},
{3.07550132E-1, 3.47777665E-1, 4.01049614E-1, 4.32767451E-1},
{2.59387434E-1, 2.87243843E-1, 3.86817336E-1, 4.06042695E-1},
{2.85485208E-1, 3.44094992E-1, 4.02050495E-1, 4.19413745E-1},
{2.65781403E-1, 3.40084374E-1, 3.69407654E-1, 4.27031696E-1},
{3.53740931E-1, 3.84463251E-1, 4.11747813E-1, 4.26181793E-1},
{2.43866488E-1, 2.68350184E-1, 3.42201948E-1, 3.98457229E-1},
{2.93145239E-1, 3.34754169E-1, 3.61702800E-1, 3.98416638E-1},
{2.91342974E-1, 3.13155174E-1, 3.36525917E-1, 3.87748599E-1},
{3.05656791E-1, 3.62904549E-1, 3.88153434E-1, 4.05543149E-1},
{2.17492327E-1, 3.11723530E-1, 3.75984788E-1, 4.28997755E-1},
{2.91149259E-1, 3.29380929E-1, 4.03900385E-1, 4.22333181E-1},
{2.90362060E-1, 3.09530973E-1, 3.78994226E-1, 4.13688362E-1},
{3.29564869E-1, 3.77404690E-1, 4.06584859E-1, 4.24739718E-1},
{2.46461585E-1, 2.71593273E-1, 3.66338253E-1, 4.30753767E-1},
{3.14107716E-1, 3.37011874E-1, 3.80409718E-1, 4.11099434E-1},
{2.76568413E-1, 3.27320695E-1, 3.58844280E-1, 4.28949475E-1},
{3.17179084E-1, 3.58972430E-1, 4.04765844E-1, 4.40376341E-1},
{2.42777750E-1, 3.34954798E-1, 3.96943450E-1, 4.13318396E-1},
{2.88895488E-1, 3.25691164E-1, 4.22859550E-1, 4.43758667E-1},
{2.77583301E-1, 3.25479031E-1, 3.89144659E-1, 4.41075861E-1},
{3.59125674E-1, 3.90694141E-1, 4.21009541E-1, 4.35708523E-1},
{2.20172390E-1, 2.47719273E-1, 3.54381859E-1, 4.25398111E-1},
{3.06046784E-1, 3.27924728E-1, 3.66992772E-1, 3.93192589E-1},
{2.70805597E-1, 3.16826642E-1, 3.45648706E-1, 4.11717594E-1},
{3.23188901E-1, 3.45463097E-1, 3.89778793E-1, 4.21570778E-1},
{2.46136114E-1, 3.12391996E-1, 3.72188628E-1, 3.95842731E-1},
{3.03856730E-1, 3.24354768E-1, 3.85747254E-1, 4.14155006E-1},
{2.81075418E-1, 3.18608463E-1, 3.85646880E-1, 4.02703643E-1},
{3.53517115E-1, 3.72702539E-1, 3.96264613E-1, 4.13074911E-1},
{2.09221140E-1, 2.95262218E-1, 3.80314291E-1, 4.31278229E-1},
{3.25313628E-1, 3.46735477E-1, 3.70724022E-1, 3.91045630E-1},
{2.86396503E-1, 3.43560040E-1, 3.69713604E-1, 3.89867842E-1},
{3.27794671E-1, 3.47367823E-1, 4.05465066E-1, 4.24566150E-1},
{2.53054976E-1, 3.02656293E-1, 3.82165134E-1, 4.29898322E-1},
{2.94418454E-1, 3.70745420E-1, 3.95443261E-1, 4.19514775E-1},
{2.62873113E-1, 3.45069230E-1, 4.04140890E-1, 4.21902061E-1},
{3.65063488E-1, 3.82435143E-1, 4.13424790E-1, 4.31241691E-1},
{2.48788506E-1, 2.82372773E-1, 3.65772307E-1, 4.10981059E-1},
{3.07288766E-1, 3.27828944E-1, 3.77664983E-1, 4.36220944E-1},
{2.98542321E-1, 3.20627332E-1, 3.50569665E-1, 4.27620232E-1},
{3.16258013E-1, 3.62903833E-1, 3.88225138E-1, 4.25608873E-1},
{2.39077866E-1, 3.31310451E-1, 3.70317876E-1, 4.15995896E-1},
{3.03735793E-1, 3.32806051E-1, 4.10232842E-1, 4.27751064E-1},
{2.96002507E-1, 3.19014788E-1, 3.81062448E-1, 4.26954985E-1},
{3.32508922E-1, 3.62516999E-1, 4.23315108E-1, 4.40995157E-1},
{2.35128701E-1, 2.74731100E-1, 4.12070572E-1, 4.35478806E-1},
{2.98073769E-1, 3.55338752E-1, 3.79087746E-1, 4.15318787E-1},
{2.83429801E-1, 3.45264912E-1, 3.70376289E-1, 4.09900844E-1},
{3.23593080E-1, 3.65412831E-1, 4.12813127E-1, 4.31023479E-1},
{2.76626348E-1, 3.00508440E-1, 4.02236879E-1, 4.26638782E-1},
{2.94512928E-1, 3.61443222E-1, 4.19635236E-1, 4.36999202E-1},
{2.90807247E-1, 3.41689348E-1, 3.92779291E-1, 4.43490267E-1},
{3.59391451E-1, 4.03985143E-1, 4.40843761E-1, 4.53028619E-1},
{2.23295465E-1, 2.39192486E-1, 3.23768020E-1, 4.21689451E-1},
{2.94778049E-1, 3.18798721E-1, 3.53217840E-1, 3.91906381E-1},
{2.59032130E-1, 3.10240507E-1, 3.43569040E-1, 3.95064235E-1},
{3.16474676E-1, 3.38544369E-1, 3.93329024E-1, 4.12235558E-1},
{2.40108207E-1, 2.84631193E-1, 3.60280991E-1, 3.79973769E-1},
{2.96909094E-1, 3.15798342E-1, 3.94964337E-1, 4.15127575E-1},
{2.85434067E-1, 3.04921508E-1, 3.61974716E-1, 4.05767262E-1},
{3.37407053E-1, 3.56672168E-1, 3.85155082E-1, 4.11186695E-1},
{2.24014923E-1, 2.60116160E-1, 3.94772530E-1, 4.19585884E-1},
{3.00647914E-1, 3.41640651E-1, 3.70223522E-1, 3.89520049E-1},
{2.65946031E-1, 3.25039148E-1, 3.74339938E-1, 3.92346144E-1},
{3.16029310E-1, 3.40491295E-1, 4.02355313E-1, 4.20484245E-1},
{2.69841492E-1, 2.94562399E-1, 3.62341762E-1, 4.06415462E-1},
{2.78897285E-1, 3.59831035E-1, 3.82025838E-1, 4.10577476E-1},
{2.60760844E-1, 3.31088543E-1, 3.88826251E-1, 4.05486643E-1},
{3.43372285E-1, 3.82647038E-1, 4.14716601E-1, 4.31592941E-1},
{2.47998103E-1, 2.73393154E-1, 3.31160426E-1, 4.18943226E-1},
{3.03579569E-1, 3.25202465E-1, 3.70984435E-1, 4.14420485E-1},
{2.76896894E-1, 3.00499499E-1, 3.54178190E-1, 4.28807020E-1},
{3.23655546E-1, 3.59816968E-1, 3.89525414E-1, 4.09288704E-1},
{2.38927796E-1, 3.09919238E-1, 3.53915572E-1, 4.16634321E-1},
{2.81171739E-1, 3.07520270E-1, 4.16264892E-1, 4.38523829E-1},
{2.88858652E-1, 3.09810817E-1, 3.67845178E-1, 4.36035573E-1},
{3.38423491E-1, 3.70634377E-1, 4.15449977E-1, 4.31534529E-1},
{2.41260394E-1, 2.73617864E-1, 3.89554620E-1, 4.12539542E-1},
{2.98046708E-1, 3.40122104E-1, 3.86183739E-1, 4.13826346E-1},
{2.82436430E-1, 3.31597507E-1, 3.57941389E-1, 4.12115216E-1},
{3.03820193E-1, 3.70588601E-1, 4.05774951E-1, 4.31517065E-1},
{2.39077732E-1, 3.11638474E-1, 4.13935781E-1, 4.35304046E-1},
{2.67116845E-1, 3.41937900E-1, 4.17409420E-1, 4.39184844E-1},
{2.67946839E-1, 3.33343923E-1, 3.86481404E-1, 4.37462509E-1},
{3.40510964E-1, 3.90878022E-1, 4.35485125E-1, 4.49101925E-1},
{2.10069850E-1, 2.32524484E-1, 3.61781418E-1, 4.31357861E-1},
{2.94509888E-1, 3.33709776E-1, 3.82278621E-1, 3.98638904E-1},
{2.80525148E-1, 3.25905204E-1, 3.50647032E-1, 3.92873943E-1},
{3.19999635E-1, 3.43674660E-1, 3.91070545E-1, 4.37501073E-1},
{2.20581010E-1, 3.03151906E-1, 3.81765544E-1, 4.04488146E-1},
{2.86122739E-1, 3.29746544E-1, 3.88102829E-1, 4.24247742E-1},
{2.69807100E-1, 3.25332284E-1, 3.79154503E-1, 4.15138245E-1},
{3.34858894E-1, 3.69258404E-1, 3.94743145E-1, 4.11922157E-1},
{2.07109794E-1, 2.72779524E-1, 3.78566444E-1, 4.34579968E-1},
{3.06466222E-1, 3.46695721E-1, 3.87138307E-1, 4.03558314E-1},
{2.70148575E-1, 3.46654534E-1, 3.77696693E-1, 3.96434486E-1},
{3.18745911E-1, 3.40225697E-1, 4.14991558E-1, 4.41578746E-1},
{2.58592844E-1, 3.14370096E-1, 3.65083754E-1, 4.21615183E-1},
{2.82712996E-1, 3.54137123E-1, 4.06745970E-1, 4.29267883E-1},
{2.52021760E-1, 3.59105110E-1, 3.95102918E-1, 4.18148398E-1},
{3.54906201E-1, 3.74952912E-1, 4.18965995E-1, 4.36144412E-1},
{2.64841139E-1, 2.92941809E-1, 3.27751458E-1, 4.08790469E-1},
{3.07774246E-1, 3.35586190E-1, 3.62209618E-1, 4.25394237E-1},
{2.88466334E-1, 3.16075742E-1, 3.60989630E-1, 4.19551432E-1},
{3.17128420E-1, 3.55772197E-1, 4.05808747E-1, 4.23972964E-1},
{2.47089684E-1, 3.38184595E-1, 3.71859610E-1, 3.95971477E-1},
{3.07981730E-1, 3.32691789E-1, 4.00534213E-1, 4.38273668E-1},
{2.79484808E-1, 3.16183507E-1, 3.97237718E-1, 4.34746623E-1},
{3.44490469E-1, 3.66153181E-1, 4.10959423E-1, 4.41727102E-1},
{2.35741779E-1, 2.94587255E-1, 3.98072541E-1, 4.16833401E-1},
{3.14038455E-1, 3.52272034E-1, 3.79138887E-1, 4.10969079E-1},
{2.83002496E-1, 3.38136256E-1, 3.88641894E-1, 4.06193316E-1},
{3.23625326E-1, 3.50243390E-1, 4.28089559E-1, 4.46630359E-1},
{2.61252105E-1, 3.24970961E-1, 4.00214493E-1, 4.25321758E-1},
{3.05284500E-1, 3.42164159E-1, 4.24475133E-1, 4.43830967E-1},
{2.87374794E-1, 3.32500637E-1, 3.94308269E-1, 4.42538500E-1},
{3.74075353E-1, 4.02026355E-1, 4.30933535E-1, 4.44160044E-1},
{2.34503999E-1, 2.56218612E-1, 3.41238797E-1, 4.23045278E-1},
{3.05492580E-1, 3.29156995E-1, 3.52709830E-1, 3.92439067E-1},
{2.81323552E-1, 3.03292334E-1, 3.48925412E-1, 3.93163860E-1},
{3.21893454E-1, 3.50419939E-1, 3.97317469E-1, 4.14560318E-1},
{2.39684582E-1, 2.92451501E-1, 3.78937423E-1, 3.96535456E-1},
{3.07307243E-1, 3.29127908E-1, 3.98455560E-1, 4.16143298E-1},
{2.85274565E-1, 3.08774531E-1, 3.92916501E-1, 4.14437652E-1},
{3.44446361E-1, 3.62201869E-1, 3.97619784E-1, 4.17743623E-1},
{2.32083067E-1, 2.67807961E-1, 3.78075659E-1, 4.34560895E-1},
{3.04738700E-1, 3.51865292E-1, 3.75973165E-1, 3.95293653E-1},
{2.61990905E-1, 3.46207321E-1, 3.71296942E-1, 4.12438929E-1},
{3.11080933E-1, 3.51040900E-1, 4.16082799E-1, 4.34340119E-1},
{2.74980426E-1, 2.96631455E-1, 3.87520492E-1, 4.09243762E-1},
{2.90939093E-1, 3.54455590E-1, 3.93426955E-1, 4.08220291E-1},
{2.71871865E-1, 3.45510781E-1, 3.87125313E-1, 4.22590613E-1},
{3.63245904E-1, 3.81932199E-1, 4.04114902E-1, 4.18370664E-1},
{2.45770738E-1, 2.72909343E-1, 3.48317921E-1, 4.25161839E-1},
{3.14139009E-1, 3.37872326E-1, 3.65195215E-1, 4.04423416E-1},
{2.94075787E-1, 3.16935539E-1, 3.43047202E-1, 4.06130373E-1},
{3.14627469E-1, 3.72413397E-1, 4.00660694E-1, 4.17930841E-1},
{2.34014243E-1, 3.14007223E-1, 3.83003533E-1, 4.34829175E-1},
{2.93635666E-1, 3.20529997E-1, 4.10837352E-1, 4.36393142E-1},
{2.89505839E-1, 3.11828852E-1, 3.86311471E-1, 4.38771248E-1},
{3.26317430E-1, 3.80858183E-1, 4.19721425E-1, 4.38795507E-1},
{2.50809520E-1, 2.83018053E-1, 3.82247388E-1, 4.34244394E-1},
{3.18994045E-1, 3.44855130E-1, 3.72690141E-1, 4.23067033E-1},
{2.88380086E-1, 3.36622238E-1, 3.69742334E-1, 4.25057590E-1},
{3.06107700E-1, 3.81856918E-1, 4.18206155E-1, 4.32868361E-1},
{2.33898312E-1, 3.44861805E-1, 4.12176549E-1, 4.29216206E-1},
{2.85980880E-1, 3.42903793E-1, 4.25112903E-1, 4.44299698E-1},
{2.79858828E-1, 3.38789344E-1, 3.92085373E-1, 4.40541029E-1},
{3.64509344E-1, 3.82202744E-1, 4.29830611E-1, 4.45818365E-1},
{2.34392300E-1, 2.57377386E-1, 3.59567046E-1, 4.30088580E-1},
{3.05031896E-1, 3.27589393E-1, 3.78305554E-1, 4.01026130E-1},
{2.77522624E-1, 3.18130314E-1, 3.67794275E-1, 4.01543021E-1},
{3.33035767E-1, 3.55820954E-1, 3.87548923E-1, 4.24628675E-1},
{2.45021001E-1, 3.12560678E-1, 3.91147614E-1, 4.08762813E-1},
{2.97059119E-1, 3.40246916E-1, 3.92919302E-1, 4.28899705E-1},
{2.77839303E-1, 3.25019777E-1, 3.97436380E-1, 4.15920913E-1},
{3.49465251E-1, 3.70362461E-1, 3.95482540E-1, 4.31923389E-1},
{2.31485590E-1, 2.91023374E-1, 3.77909541E-1, 4.32259738E-1},
{3.19283485E-1, 3.53671074E-1, 3.80982876E-1, 3.97843361E-1},
{2.89689243E-1, 3.50265682E-1, 3.80729675E-1, 3.97969365E-1},
{3.28987300E-1, 3.52005422E-1, 4.12557244E-1, 4.37597930E-1},
{2.76273251E-1, 3.02267194E-1, 3.81723404E-1, 4.34989095E-1},
{2.79627264E-1, 3.73727322E-1, 4.12374616E-1, 4.30626333E-1},
{2.53442764E-1, 3.65940034E-1, 4.14937019E-1, 4.32743609E-1},
{3.76107216E-1, 3.95142019E-1, 4.16787744E-1, 4.33023572E-1},
{2.62815833E-1, 2.88270533E-1, 3.47397208E-1, 4.24182594E-1},
{3.01931322E-1, 3.43652546E-1, 3.77031326E-1, 4.34204459E-1},
{2.97834277E-1, 3.23495388E-1, 3.64492416E-1, 4.33550835E-1},
{3.31774473E-1, 3.64324927E-1, 3.98243546E-1, 4.35078323E-1},
{2.49049723E-1, 3.27870786E-1, 3.83587003E-1, 4.35558081E-1},
{3.04653406E-1, 3.27671230E-1, 4.18484688E-1, 4.41378772E-1},
{2.96960890E-1, 3.23898911E-1, 3.90463710E-1, 4.39915955E-1},
{3.43923748E-1, 3.67100477E-1, 4.29523230E-1, 4.45214987E-1},
{2.59399652E-1, 2.91602671E-1, 4.04372454E-1, 4.31413233E-1},
{2.97537506E-1, 3.57573807E-1, 3.88991833E-1, 4.30006981E-1},
{2.84068942E-1, 3.49574566E-1, 3.81042838E-1, 4.29712772E-1},
{3.25716257E-1, 3.74875903E-1, 4.31959271E-1, 4.47290838E-1},
{2.65302956E-1, 3.14745963E-1, 4.16703463E-1, 4.37294722E-1},
{3.00398588E-1, 3.54147255E-1, 4.28538084E-1, 4.60336387E-1},
{2.98077166E-1, 3.49304914E-1, 4.00429249E-1, 4.48213518E-1},
{3.75576198E-1, 4.16657329E-1, 4.42136765E-1, 4.52728629E-1}};

static const float evrc_lspq_quant_codebook1[16][5] = {
{0.42091064E-1, 0.69474973E-1, 0.11168948E+0, 0.14571965E+0, 0.20893581E+0},
{0.54944664E-1, 0.98242261E-1, 0.11007882E+0, 0.15890779E+0, 0.20548241E+0},
{0.45188572E-1, 0.75199433E-1, 0.11423391E+0, 0.15469728E+0, 0.19746706E+0},
{0.49474996E-1, 0.79667501E-1, 0.12571351E+0, 0.16944779E+0, 0.20775315E+0},
{0.41789379E-1, 0.63459560E-1, 0.12068028E+0, 0.15850765E+0, 0.20406815E+0},
{0.47159236E-1, 0.79129547E-1, 0.12183110E+0, 0.15650047E+0, 0.22309226E+0},
{0.54539919E-1, 0.80343045E-1, 0.12947764E+0, 0.15186153E+0, 0.20171718E+0},
{0.55852082E-1, 0.94114847E-1, 0.14016025E+0, 0.17807084E+0, 0.22955489E+0},
{0.45443531E-1, 0.73541410E-1, 0.11937657E+0, 0.15442030E+0, 0.21010752E+0},
{0.63178010E-1, 0.95231488E-1, 0.12364983E+0, 0.17672543E+0, 0.21743731E+0},
{0.52765369E-1, 0.84351443E-1, 0.11589085E+0, 0.15790924E+0, 0.20732352E+0},
{0.51865745E-1, 0.81328541E-1, 0.13756232E+0, 0.18322878E+0, 0.21640070E+0},
{0.44419531E-1, 0.68874463E-1, 0.13115251E+0, 0.16263582E+0, 0.21659100E+0},
{0.49378436E-1, 0.81882551E-1, 0.13067168E+0, 0.16821896E+0, 0.23136081E+0},
{0.55909779E-1, 0.90783298E-1, 0.13348848E+0, 0.16298474E+0, 0.20961523E+0},
{0.61378211E-1, 0.98602772E-1, 0.14793332E+0, 0.19283190E+0, 0.23156509E+0}};

static const float evrc_lspq_quant_codebook2[16][5] = {
{0.26822963, 0.30585295, 0.31110349, 0.36823335, 0.40774474},
{0.24418014, 0.28970167, 0.32573757, 0.39021483, 0.41345838},
{0.23341830, 0.30078292, 0.32893899, 0.38557330, 0.41068462},
{0.25905868, 0.29756859, 0.34196618, 0.38531172, 0.41295227},
{0.24290450, 0.29223618, 0.32718554, 0.37788135, 0.40332928},
{0.24674191, 0.29749370, 0.33631226, 0.39426059, 0.42258954},
{0.21377595, 0.33140418, 0.34067687, 0.38222077, 0.40939021},
{0.26673481, 0.30791649, 0.34419721, 0.39611506, 0.42387524},
{0.26121426, 0.30492544, 0.32997236, 0.38486803, 0.42023736},
{0.24954870, 0.29372856, 0.33382735, 0.37850669, 0.41714057},
{0.24158891, 0.30173415, 0.34128246, 0.38428575, 0.41619650},
{0.25818908, 0.31736413, 0.34904337, 0.38769925, 0.41551358},
{0.24450587, 0.30673453, 0.33579323, 0.37844428, 0.40557048},
{0.25164026, 0.31225079, 0.33847794, 0.39554194, 0.42396802},
{0.22787990, 0.31779197, 0.33831909, 0.40044111, 0.41185561},
{0.27896860, 0.32261974, 0.35658112, 0.40206763, 0.42370448}};

static const float * const evrc_lspq_full_codebooks[] = {
    evrc_lspq_full_codebook1[0], evrc_lspq_full_codebook2[0],
    evrc_lspq_full_codebook3[0], evrc_lspq_full_codebook4[0],
};

static const float * const evrc_lspq_half_codebooks[] = {
    evrc_lspq_half_codebook1[0], evrc_lspq_half_codebook2[0],
    evrc_lspq_half_codebook3[0],
};

static const float * const evrc_lspq_quant_codebooks[] = {
    evrc_lspq_quant_codebook1[0], evrc_lspq_quant_codebook2[0],
};

static const float * const *evrc_lspq_codebooks[] = {
    0,
    evrc_lspq_quant_codebooks,
    0,
    evrc_lspq_half_codebooks,
    evrc_lspq_full_codebooks,
};

static const uint8_t evrc_lspq_nb_codebooks[] = {
    0,
    FF_ARRAY_ELEMS(evrc_lspq_quant_codebooks),
    0,
    FF_ARRAY_ELEMS(evrc_lspq_half_codebooks),
    FF_ARRAY_ELEMS(evrc_lspq_full_codebooks),
};

static const uint8_t evrc_lspq_full_codebooks_row_sizes[] = {
    FF_ARRAY_ELEMS(evrc_lspq_full_codebook1[0]),
    FF_ARRAY_ELEMS(evrc_lspq_full_codebook2[0]),
    FF_ARRAY_ELEMS(evrc_lspq_full_codebook3[0]),
    FF_ARRAY_ELEMS(evrc_lspq_full_codebook4[0]),
};

static const uint8_t evrc_lspq_half_codebooks_row_sizes[] = {
    FF_ARRAY_ELEMS(evrc_lspq_half_codebook1[0]),
    FF_ARRAY_ELEMS(evrc_lspq_half_codebook2[0]),
    FF_ARRAY_ELEMS(evrc_lspq_half_codebook3[0]),
};

static const uint8_t evrc_lspq_quant_codebooks_row_sizes[] = {
    FF_ARRAY_ELEMS(evrc_lspq_quant_codebook1[0]),
    FF_ARRAY_ELEMS(evrc_lspq_quant_codebook2[0]),
};

static const uint8_t* const evrc_lspq_codebooks_row_sizes[] = {
    NULL,
    evrc_lspq_quant_codebooks_row_sizes,
    NULL,
    evrc_lspq_half_codebooks_row_sizes,
    evrc_lspq_full_codebooks_row_sizes,
};

static const float pitch_gain_vq[] = { 0, 0.3, 0.55, 0.7, 0.8, 0.9, 1, 1.2 };
static const float estimation_delay[] = { 55.0, 80.0, 39.0, 71.0, 33.0 }; // 5.2.3.4
static const uint8_t subframe_sizes[] = { 53, 53, 54 };
#endif /* AVCODEC_EVRCDATA_H */
