/***
  This file is part of cups-filters.

  This file is free software; you can redistribute it and/or modify it
  under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2.1 of the
  License, or (at your option) any later version.

  This file is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General
  Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with avahi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
  USA.
***/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <assert.h>
#include <stdlib.h>
#include <time.h>
#include <signal.h>

#include <avahi-client/client.h>
#include <avahi-client/lookup.h>

#include <avahi-common/simple-watch.h>
#include <avahi-common/malloc.h>
#include <avahi-common/error.h>

#include <cups/cups.h>

/* Attribute to mark a CUPS queue as created by us */
#define CUPS_BROWSED_MARK "cups-browsed"

/* Timeout values in sec */
#define TIMEOUT_IMMEDIATELY -1
#define TIMEOUT_CONFIRM     10
#define TIMEOUT_RETRY       10
#define TIMEOUT_REMOVE      -1
#define TIMEOUT_CHECK_LIST   2

/* Interval for checking our CUPS queue list periodically in sec */
#define CHECK_INTERVAL       2

/* Status of remote printer */
typedef enum printer_status_e {
  STATUS_UNCONFIRMED = 0,
  STATUS_CONFIRMED,
  STATUS_TO_BE_CREATED,
  STATUS_DISAPPEARED
} printer_status_t;

/* Data structure for remote printers */
typedef struct remote_printer_s {
  char *name;
  char *uri;
  printer_status_t status;
  time_t timeout;
  int duplicate;
  char *host;
  char *service_name;
  char *type;
  char *domain;
} remote_printer_t;

cups_array_t *remote_printers;

static AvahiSimplePoll *simple_poll = NULL;

static int debug = 0;

void debug_printf(const char *format, ...) {
  if (debug) {
    va_list arglist;
    va_start(arglist, format);
    vfprintf(stderr, format, arglist);
    fflush(stderr);
    va_end(arglist);
  }
}

void handle_cups_queues() {
  remote_printer_t *p;
  http_t *http;
  char uri[HTTP_MAX_URI];
  int num_options;
  cups_option_t *options;
  int num_jobs;
  cups_job_t *jobs;
  ipp_t *request;
  time_t current_time = time(NULL);

  debug_printf("cups-browsed: Processing printer list ...\n");
  for (p = (remote_printer_t *)cupsArrayFirst(remote_printers);
       p; p = (remote_printer_t *)cupsArrayNext(remote_printers)) {
    switch (p->status) {

    /* Print queue generated by us in a previous session */
    case STATUS_UNCONFIRMED:

      /* Only act if the timeout has passed */
      if (p->timeout > current_time)
	break;

      /* Queue not reported again by Bonjour, remove it */
      p->status = STATUS_DISAPPEARED;
      p->timeout = current_time + TIMEOUT_IMMEDIATELY;

      debug_printf("cups-browsed: No remote printer named %s available, removing entry from previous session.\n",
		   p->name);

    /* Bonjour has reported this printer as disappeared or we have replaced
       this printer by another one */
    case STATUS_DISAPPEARED:

      /* Only act if the timeout has passed */
      if (p->timeout > current_time)
	break;

      debug_printf("cups-browsed: Removing entry %s%s.\n", p->name,
		   (p->duplicate ? "" : " and its CUPS queue"));

      /* Remove the CUPS queue */
      if (!p->duplicate) { /* Duplicates do not have a CUPS queue */
	if ((http = httpConnectEncrypt(cupsServer(), ippPort(),
				       cupsEncryption())) == NULL) {
	  debug_printf("cups-browsed: Unable to connect to CUPS!\n");
	  p->timeout = current_time + TIMEOUT_RETRY;
	  break;
	}

	/* Check whether there are still jobs and do not remove the queue
	   then */
	num_jobs = 0;
	jobs = NULL;
	num_jobs = cupsGetJobs2(http, &jobs, p->name, 0, CUPS_WHICHJOBS_ACTIVE);
	if (num_jobs != 0) { /* error or jobs */
	  debug_printf("cups-browsed: Queue has still jobs or CUPS error!\n");
	  cupsFreeJobs(num_jobs, jobs);
	  httpClose(http);
	  /* Schedule the removal of the queue for later */
	  p->timeout = current_time + TIMEOUT_RETRY;
	  break;
	}

	/* No jobs, remove the CUPS queue */
	request = ippNewRequest(CUPS_DELETE_PRINTER);
	/* Printer URI: ipp://localhost:631/printers/<queue name> */
	httpAssembleURIf(HTTP_URI_CODING_ALL, uri, sizeof(uri), "ipp", NULL,
			 "localhost", 0, "/printers/%s", p->name);
	ippAddString(request, IPP_TAG_OPERATION, IPP_TAG_URI,
		     "printer-uri", NULL, uri);
	/* Default user */
	ippAddString(request, IPP_TAG_OPERATION, IPP_TAG_NAME,
		     "requesting-user-name", NULL, cupsUser());
	/* Do it */
	ippDelete(cupsDoRequest(http, request, "/admin/"));
	if (cupsLastError() > IPP_OK_CONFLICT) {
	  debug_printf("cups-browsed: Unable to remove CUPS queue!\n");
	  p->timeout = current_time + TIMEOUT_RETRY;
	  httpClose(http);
	  break;
	}
	httpClose(http);
      }

      /* CUPS queue removed, remove the list entry */
      cupsArrayRemove(remote_printers, p);
      if (p->name) free (p->name);
      if (p->uri) free (p->uri);
      if (p->host) free (p->host);
      if (p->service_name) free (p->service_name);
      if (p->type) free (p->type);
      if (p->domain) free (p->domain);
      free(p);
      break;

    /* Bonjour has reported a new remote printer, create a CUPS queue for it,
       or upgrade an existing queue, or update a queue to use a backup host
       when it has disappeared on the currently used host */
    case STATUS_TO_BE_CREATED:

      /* Do not create a queue for duplicates */
      if (p->duplicate)
	break;

      /* Only act if the timeout has passed */
      if (p->timeout > current_time)
	break;

      debug_printf("cups-browsed: Creating/Updating CUPS queue for %s\n",
		   p->name);

      /* Create a new CUPS queue or modify the existing queue */
      if ((http = httpConnectEncrypt(cupsServer(), ippPort(),
				     cupsEncryption())) == NULL) {
	debug_printf("cups-browsed: Unable to connect to CUPS!\n");
	p->timeout = current_time + TIMEOUT_RETRY;
	break;
      }
      request = ippNewRequest(CUPS_ADD_MODIFY_PRINTER);
      /* Printer URI: ipp://localhost:631/printers/<queue name> */
      httpAssembleURIf(HTTP_URI_CODING_ALL, uri, sizeof(uri), "ipp", NULL,
		       "localhost", ippPort(), "/printers/%s", p->name);
      ippAddString(request, IPP_TAG_OPERATION, IPP_TAG_URI,
               "printer-uri", NULL, uri);
      /* Default user */
      ippAddString(request, IPP_TAG_OPERATION, IPP_TAG_NAME,
		   "requesting-user-name", NULL, cupsUser());
      /* Queue should be enabled ... */
      ippAddInteger(request, IPP_TAG_PRINTER, IPP_TAG_ENUM, "printer-state",
		    IPP_PRINTER_IDLE);
      /* ... and accepting jobs */
      ippAddBoolean(request, IPP_TAG_PRINTER, "printer-is-accepting-jobs", 1);
      num_options = 0;
      options = NULL;
      /* Device URI: ipp(s)://<remote host>:631/printers/<remote queue> */
      num_options = cupsAddOption("device-uri", p->uri,
				  num_options, &options);
      /* Option cups-browsed=true, marking that we have created this queue */
      num_options = cupsAddOption("cups-browsed-default", "true",
				  num_options, &options);
      /* Do not share a queue which serves only to point to a remote printer */
      num_options = cupsAddOption("printer-is-shared", "false",
				  num_options, &options);
      /* Description: <Bonjour service name> */
      num_options = cupsAddOption("printer-info", p->service_name,
				  num_options, &options);
      /* Location: <Remote host name> */
      num_options = cupsAddOption("printer-location", p->host,
				  num_options, &options);
      cupsEncodeOptions2(request, num_options, options, IPP_TAG_PRINTER);
      /* Do it */
      ippDelete(cupsDoRequest(http, request, "/admin/"));
      cupsFreeOptions(num_options, options);
      if (cupsLastError() > IPP_OK_CONFLICT) {
	debug_printf("cups-browsed: Unable to create CUPS queue!\n");
	p->timeout = current_time + TIMEOUT_RETRY;
	httpClose(http);
	break;
      }
      httpClose(http);

      p->status = STATUS_CONFIRMED;
      break;

    /* Nothing to do */
    case STATUS_CONFIRMED:
      break;

    }
  }
}

static void resolve_callback(
  AvahiServiceResolver *r,
  AVAHI_GCC_UNUSED AvahiIfIndex interface,
  AVAHI_GCC_UNUSED AvahiProtocol protocol,
  AvahiResolverEvent event,
  const char *name,
  const char *type,
  const char *domain,
  const char *host_name,
  const AvahiAddress *address,
  uint16_t port,
  AvahiStringList *txt,
  AvahiLookupResultFlags flags,
  AVAHI_GCC_UNUSED void* userdata) {

  assert(r);

  /* Called whenever a service has been resolved successfully or timed out */

  switch (event) {

  /* Resolver error */
  case AVAHI_RESOLVER_FAILURE:
    debug_printf("cups-browsed: Avahi-Resolver: Failed to resolve service '%s' of type '%s' in domain '%s': %s\n",
		 name, type, domain,
		 avahi_strerror(avahi_client_errno(avahi_service_resolver_get_client(r))));
    break;

  /* New remote printer found */
  case AVAHI_RESOLVER_FOUND: {
    AvahiStringList *rp_entry, *adminurl_entry;
    char *rp_key, *rp_value, *adminurl_key, *adminurl_value,
      *remote_queue, *remote_host;
    remote_printer_t *p, *q;
    char *backup_queue_name, *local_queue_name = NULL;
    cups_dest_t *dests, *dest;
    int i, num_dests;
    const char *val;

    debug_printf("cups-browsed: Avahi Resolver: Service '%s' of type '%s' in domain '%s'.\n",
		 name, type, domain);

    /* Check if we have a remote CUPS queue, other remote printers are not
       handled by us */
    rp_entry = avahi_string_list_find(txt, "rp");
    adminurl_entry = avahi_string_list_find(txt, "adminurl");
    if (rp_entry && adminurl_entry) {
      avahi_string_list_get_pair(rp_entry, &rp_key, &rp_value, NULL);
      avahi_string_list_get_pair(adminurl_entry, &adminurl_key,
				 &adminurl_value, NULL);

      /* Check by "rp" and "adminurl" TXT record fields whether
	 the discovered printer is a CUPS queue */
      if (rp_key && rp_value && adminurl_key && adminurl_value &&
	  !strcmp(rp_key, "rp") && !strncmp(rp_value, "printers/", 9) &&
	  !strcmp(adminurl_key, "adminurl") &&
	  !strcmp(adminurl_value + strlen(adminurl_value) -
		  strlen(rp_value), rp_value)) {

	/* This IS a remote CUPS queue, find queue name and host name */
	remote_queue = rp_value + 9;
	remote_host = strdup(host_name);
	if (!strcmp(remote_host + strlen(remote_host) - 6, ".local"))
	  remote_host[strlen(remote_host) - 6] = '\0';
	if (!strcmp(remote_host + strlen(remote_host) - 7, ".local."))
	  remote_host[strlen(remote_host) - 7] = '\0';
	debug_printf("cups-browsed: Found CUPS queue: %s on host %s.\n",
		     remote_queue, remote_host);

	/* Check if there exists already a CUPS queue with the
	   requested name Try name@host in such a case and if
	   this is also taken, ignore the printer */
	if ((backup_queue_name = malloc((strlen(remote_queue) + 
					 strlen(remote_host) + 2) *
					sizeof(char))) == NULL) {
	  debug_printf("cups-browsed: ERROR: Unable to allocate memory.\n");
	  exit(1);
	}
	sprintf(backup_queue_name, "%s@%s", remote_queue, remote_host);

	/* Get available CUPS queues */
	num_dests = cupsGetDests(&dests);

	local_queue_name = remote_queue;
	if (num_dests > 0) {
	  /* Is there a local queue with the name of the remote queue? */
	  for (i = num_dests, dest = dests; i > 0; i --, dest ++)
	    /* Only consider CUPS queues not created by us */
	    if ((((val =
		   cupsGetOption(CUPS_BROWSED_MARK, dest->num_options,
				 dest->options)) == NULL) ||
		 (strcasecmp(val, "yes") != 0 &&
		  strcasecmp(val, "on") != 0 &&
		  strcasecmp(val, "true") != 0)) &&
		!strcmp(local_queue_name, dest->name))
	      break;
	  if (i > 0) {
	    /* Found local queue with same name as remote queue */
	    /* Is there a local queue with the name <queue>@<host>? */
	    local_queue_name = backup_queue_name;
	    debug_printf("cups-browsed: %s already taken, using fallback name: %s\n",
			 remote_queue, local_queue_name);
	    for (i = num_dests, dest = dests; i > 0; i --, dest ++)
	      /* Only consider CUPS queues not created by us */
	      if ((((val =
		     cupsGetOption(CUPS_BROWSED_MARK, dest->num_options,
				   dest->options)) == NULL) ||
		   (strcasecmp(val, "yes") != 0 &&
		    strcasecmp(val, "on") != 0 &&
		    strcasecmp(val, "true") != 0)) &&
		  !strcmp(local_queue_name, dest->name))
		break;
	    if (i > 0) {
	      /* Found also a local queue with name <queue>@<host>, so
		 ignore this remote printer */
	      debug_printf("cups-browsed: %s also taken, printer ignored.\n",
			   local_queue_name);
	      break;
	    }
	  }
	  cupsFreeDests(num_dests, dests);
	}

	/* Check if we have already created a queue for the discovered
	   printer */
	for (p = (remote_printer_t *)cupsArrayFirst(remote_printers);
	     p; p = (remote_printer_t *)cupsArrayNext(remote_printers))
	  if (!strcmp(p->name, local_queue_name) &&
	      !strcmp(p->host, remote_host))
	    break;

	if (p) {
	  /* We have already created a local queue, check whether the
	     discovered service allows us to upgrade the queue to IPPS */
	  if (strcasestr(type, "_ipps") &&
	      !strncmp(p->uri, "ipp:", 4)) {

	    /* Schedule local queue for upgrade to ipps: */
	    if ((p->uri = realloc(p->uri, strlen(p->uri) + 2)) == NULL){
	      debug_printf("cups-browsed: ERROR: Unable to allocate memory.\n");
	      exit(1);
	    }
	    memmove((void *)(p->uri + 4), (const void *)(p->uri + 3),
		    strlen(p->uri) - 2);
	    p->uri[3] = 's';
	    p->status = STATUS_TO_BE_CREATED;
	    p->timeout = time(NULL) + TIMEOUT_IMMEDIATELY;
	    debug_printf("cups-browsed: Upgrading printer %s (Host: %s) to IPPS. New URI: %s\n",
			 p->name, p->host, p->uri);

	  } else {

	    /* Nothing to do, mark queue entry as confirmed */
	    p->status = STATUS_CONFIRMED;
	    debug_printf("cups-browsed: Entry for %s (Host: %s, URI: %s) already exists, marking confirmed.\n",
			 p->name, p->host, p->uri);

	  }

	} else {

	  /* We need to create a local queue pointing to the
	     discovered printer */
	  if ((p = (remote_printer_t *)malloc(sizeof(remote_printer_t)))
	      == NULL) {
	    debug_printf("cups-browsed: ERROR: Unable to allocate memory.\n");
	    exit(1);
	  }

	  /* Queue name */
	  p->name = strdup(local_queue_name);
	  /* Device URI: ipp(s)://<remote host>:631/printers/<remote queue> */
	  if ((p->uri = malloc(strlen(host_name) +
			       strlen(remote_queue) + 34)) == NULL){
	    debug_printf("cups-browsed: ERROR: Unable to allocate memory.\n");
	    exit(1);
	  }
	  sprintf(p->uri, "ipp%s://%s:%u/printers/%s",
		  (strcasestr(type, "_ipps") ? "s" : ""), host_name,
		  port, remote_queue);
	  /* Schedule for immediate creation of the CUPS queue */
	  p->status = STATUS_TO_BE_CREATED;
	  p->timeout = time(NULL) + TIMEOUT_IMMEDIATELY;
	  /* Check whether we have an equally named queue already from another
	     server */
	  for (q = (remote_printer_t *)cupsArrayFirst(remote_printers);
	       q;
	       q = (remote_printer_t *)cupsArrayNext(remote_printers))
	    if (!strcmp(q->name, p->name))
	      break;
	  p->duplicate = q ? 1 : 0;
	  /* Add the new remote printer entry */
	  cupsArrayAdd(remote_printers, p);

	  debug_printf("cups-browsed: New remote printer %s found. Host: %s, URI: %s\n",
		       p->name, remote_host, p->uri);
	  if (p->duplicate)
	    debug_printf("cups-browsed: Printer already available through host %s.\n",
			 q->host);
	}

	/* Record Bonjour service parameters to identify print queue
	   entry for removal when service disappears */
	p->host = strdup(remote_host);
	p->service_name = strdup(name);
	p->type = strdup(type);
	p->domain = strdup(domain);
	debug_printf("cups-browsed: Bonjour IDs: Service name: \"%s\", Service type: \"%s\", Domain: \"%s\"\n",
		     p->service_name, p->type, p->domain);
      }

      /* Clean up */
      avahi_free(rp_key);
      avahi_free(rp_value);
      avahi_free(adminurl_key);
      avahi_free(adminurl_value);
    }
    break;
  }
  }

  avahi_service_resolver_free(r);
}

static void browse_callback(
  AvahiServiceBrowser *b,
  AvahiIfIndex interface,
  AvahiProtocol protocol,
  AvahiBrowserEvent event,
  const char *name,
  const char *type,
  const char *domain,
  AvahiLookupResultFlags flags,
  void* userdata) {

  AvahiClient *c = userdata;
  assert(b);

  /* Called whenever a new services becomes available on the LAN or
     is removed from the LAN */

  switch (event) {

  /* Avah browser error */
  case AVAHI_BROWSER_FAILURE:

    debug_printf("cups-browsed: Avahi Browser: ERROR: %s\n",
		 avahi_strerror(avahi_client_errno(avahi_service_browser_get_client(b))));
    avahi_simple_poll_quit(simple_poll);
    return;

  /* New service (remote printer) */
  case AVAHI_BROWSER_NEW:

    /* Ignore events from the local machine */
    if (flags & AVAHI_LOOKUP_RESULT_LOCAL)
      break;

    debug_printf("cups-browsed: Avahi Browser: NEW: service '%s' of type '%s' in domain '%s'\n",
		 name, type, domain);

    /* We ignore the returned resolver object. In the callback
       function we free it. If the server is terminated before
       the callback function is called the server will free
       the resolver for us. */

    if (!(avahi_service_resolver_new(c, interface, protocol, name, type, domain, AVAHI_PROTO_UNSPEC, 0, resolve_callback, c)))
      debug_printf("Failed to resolve service '%s': %s\n",
		   name, avahi_strerror(avahi_client_errno(c)));
    break;

  /* A service (remote printer) has disappeared */
  case AVAHI_BROWSER_REMOVE: {
    remote_printer_t *p, *q;

    /* Ignore events from the local machine */
    if (flags & AVAHI_LOOKUP_RESULT_LOCAL)
      break;

    debug_printf("cups-browsed: Avahi Browser: REMOVE: service '%s' of type '%s' in domain '%s'\n",
		 name, type, domain);

    /* Check whether we have listed this printer */
    for (p = (remote_printer_t *)cupsArrayFirst(remote_printers);
	 p; p = (remote_printer_t *)cupsArrayNext(remote_printers))
      if (!strcmp(p->service_name, name) &&
	  !strcmp(p->type, type) &&
	  !strcmp(p->domain, domain))
	break;
    if (p) {
      /* Check whether this queue has a duplicate from another server */
      q = NULL;
      if (!p->duplicate) {
	for (q = (remote_printer_t *)cupsArrayFirst(remote_printers);
	     q;
	     q = (remote_printer_t *)cupsArrayNext(remote_printers))
	  if (!strcmp(q->name, p->name) &&
	      strcmp(q->host, p->host) &&
	      q->duplicate)
	    break;
      }
      if (q) {
	/* Remove the data of the disappeared remote printer */
	free (p->uri);
	free (p->host);
	free (p->service_name);
	free (p->type);
	free (p->domain);
	/* Replace the data with the data of the duplicate printer */
	p->uri = strdup(q->uri);
	p->host = strdup(q->host);
	p->service_name = strdup(q->service_name);
	p->type = strdup(q->type);
	p->domain = strdup(q->domain);
	/* Schedule this printer for updating the CUPS queue */
	p->status = STATUS_TO_BE_CREATED;
	p->timeout = time(NULL) + TIMEOUT_IMMEDIATELY;
	/* Schedule the remote printer for removal */
	q->status = STATUS_DISAPPEARED;
	q->timeout = time(NULL) + TIMEOUT_IMMEDIATELY;

	debug_printf("cups-browsed: Printer %s diasappeared, replacing by backup on host %s with URI %s.\n",
		     p->name, p->host, p->uri);
      } else {

	/* Schedule CUPS queue for removal */
	p->status = STATUS_DISAPPEARED;
	p->timeout = time(NULL) + TIMEOUT_REMOVE;

	debug_printf("cups-browsed: Printer %s (Host: %s, URI: %s) disappeared and no backup available, removing entry.\n",
		     p->name, p->host, p->uri);

      }

      debug_printf("cups-browsed: Bonjour IDs: Service name: \"%s\", Service type: \"%s\", Domain: \"%s\"\n",
		   p->service_name, p->type, p->domain);

    }
    break;
  }

  /* All cached Avahi events are treated now */
  case AVAHI_BROWSER_ALL_FOR_NOW:
  case AVAHI_BROWSER_CACHE_EXHAUSTED:
    debug_printf("cups-browsed: Avahi Browser: %s\n",
		 event == AVAHI_BROWSER_CACHE_EXHAUSTED ?
		 "CACHE_EXHAUSTED" : "ALL_FOR_NOW");
    break;
  }

}

static void client_callback(AvahiClient *c, AvahiClientState state, AVAHI_GCC_UNUSED void * userdata) {
  assert(c);

  /* Called whenever the client or server state changes */

  if (state == AVAHI_CLIENT_FAILURE) {
    debug_printf("cups-browsed: ERROR: Avahi server connection failure: %s\n",
		 avahi_strerror(avahi_client_errno(c)));
    avahi_simple_poll_quit(simple_poll);
  }

}

int compare_remote_printers (remote_printer_t *a, remote_printer_t *b) {
  return strcmp(a->name, b->name);
}

static void
sigterm_handler(int sig) {
  (void)sig;    /* remove compiler warnings... */

  /* Flag that we should stop and return... */
  avahi_simple_poll_quit(simple_poll);
  debug_printf("cups-browsed: Caught signal %d, shutting down ...\n", sig);
}

int main(int argc, char*argv[]) {
  AvahiClient *client = NULL;
  AvahiServiceBrowser *sb1 = NULL, *sb2 = NULL;
  int error;
  int ret = 1;
  http_t *http;
  cups_dest_t *dests,
              *dest;
  int i,
      num_dests;
  const char *val;
  remote_printer_t *p;

  /* Turn on debug mode if requested */
  if (argc >= 2 &&
      (!strcmp(argv[1], "--debug") || !strcmp(argv[1], "-d") ||
       !strncmp(argv[1], "-v", 2)))
    debug = 1;

  /* Wait for CUPS daemon to start */
  while ((http = httpConnectEncrypt(cupsServer(), ippPort(),
				    cupsEncryption())) == NULL)
    sleep(1);

  /* Read out the currently defined CUPS queues and find the ones which we
     have added in an earlier session */
  num_dests = cupsGetDests(&dests);
  remote_printers = cupsArrayNew((cups_array_func_t)compare_remote_printers,
				 NULL);
  if (num_dests > 0) {
    for (i = num_dests, dest = dests; i > 0; i --, dest ++) {
      if ((val = cupsGetOption(CUPS_BROWSED_MARK, dest->num_options,
			       dest->options)) != NULL) {
	if (strcasecmp(val, "no") != 0 && strcasecmp(val, "off") != 0 &&
	    strcasecmp(val, "false") != 0) {
	  /* Queue found, add to our list */
	  p = (remote_printer_t *)malloc(sizeof(remote_printer_t));
	  if (p) {
	    p->name = strdup(dest->name);
	    p->host = strdup("");
	    p->service_name = strdup("");
	    p->type = strdup("");
	    p->domain = strdup("");
	    p->uri = strdup(cupsGetOption("device-uri",
					  dest->num_options, dest->options));
	    /* Mark as unconfirmed, if no Avahi report of this queue appears
	       in a certain time frame, we will remove the queue */
	    p->status = STATUS_UNCONFIRMED;
	    p->timeout = time(NULL) + TIMEOUT_CONFIRM;
	    p->duplicate = 0;
	    cupsArrayAdd(remote_printers, p);
	    debug_printf("cups-browsed: Found CUPS queue %s (URI: %s) from previous session.\n",
			 p->name, p->uri);
	  } else {
	    debug_printf("cups-browsed: ERROR: Unable to allocate memory.\n");
	    exit(1);
	  }
	}
      }
    }
    cupsFreeDests(num_dests, dests);
  }
  httpClose(http);

  /* Redirect SIGINT and SIGTERM so that we do a proper shutdown, removing
     the CUPS queues which we have created */
#ifdef HAVE_SIGSET /* Use System V signals over POSIX to avoid bugs */
  sigset(SIGTERM, sigterm_handler);
  sigset(SIGINT, sigterm_handler);
  debug_printf("cups-browsed: Using signal handler SIGSET\n");
#elif defined(HAVE_SIGACTION)
  struct sigaction action; /* Actions for POSIX signals */
  memset(&action, 0, sizeof(action));
  sigemptyset(&action.sa_mask);
  sigaddset(&action.sa_mask, SIGTERM);
  action.sa_handler = sigterm_handler;
  sigaction(SIGTERM, &action, NULL);
  sigemptyset(&action.sa_mask);
  sigaddset(&action.sa_mask, SIGINT);
  action.sa_handler = sigterm_handler;
  sigaction(SIGINT, &action, NULL);
  debug_printf("cups-browsed: Using signal handler SIGACTION\n");
#else
  signal(SIGTERM, sigterm_handler);
  signal(SIGINT, sigterm_handler);
  debug_printf("cups-browsed: Using signal handler SIGNAL\n");
#endif /* HAVE_SIGSET */

  /* Allocate main loop object */
  if (!(simple_poll = avahi_simple_poll_new())) {
    debug_printf("cups-browsed: ERROR: Failed to create simple poll object.\n");
    goto fail;
  }

  /* Allocate a new client */
  client = avahi_client_new(avahi_simple_poll_get(simple_poll), 0,
			    client_callback, NULL, &error);

  /* Check wether creating the client object succeeded */
  if (!client) {
    debug_printf("cups-browsed: ERROR: Failed to create client: %s\n",
		 avahi_strerror(error));
    goto fail;
  }

  /* Create the service browsers */
  if (!(sb1 =
	avahi_service_browser_new(client, AVAHI_IF_UNSPEC, AVAHI_PROTO_UNSPEC,
				  "_ipp._tcp", NULL, 0, browse_callback,
				  client))) {
    debug_printf("cups-browsed: ERROR: Failed to create service browser for IPP: %s\n",
		 avahi_strerror(avahi_client_errno(client)));
    goto fail;
  }
  if (!(sb2 =
	avahi_service_browser_new(client, AVAHI_IF_UNSPEC, AVAHI_PROTO_UNSPEC,
				  "_ipps._tcp", NULL, 0, browse_callback,
				  client))) {
    debug_printf("cups-browsed: ERROR: Failed to create service browser for IPPS: %s\n",
		 avahi_strerror(avahi_client_errno(client)));
    goto fail;
  }

  /* Run the main loop */
  while((ret = avahi_simple_poll_iterate(simple_poll, CHECK_INTERVAL * 1000))
	== 0)
    handle_cups_queues();

  debug_printf("cups-browsed: Avahi main loop exited with status: %d\n", ret);
  if (ret == 1)
    ret = 0;

fail:

  /* Clean up things */

  /* Remove all queues which we have set up */
  for (p = (remote_printer_t *)cupsArrayFirst(remote_printers);
       p; p = (remote_printer_t *)cupsArrayNext(remote_printers)) {
    p->status = STATUS_DISAPPEARED;
    p->timeout = time(NULL) + TIMEOUT_IMMEDIATELY;
  }
  handle_cups_queues();
    
  /* Free the data structures for Bonjour browsing */
  if (sb1)
    avahi_service_browser_free(sb1);
  if (sb2)
    avahi_service_browser_free(sb2);

  if (client)
    avahi_client_free(client);

  if (simple_poll)
    avahi_simple_poll_free(simple_poll);

  return ret;
}
