      SUBROUTINE ARY_NEWP( FTYPE, NDIM, UBND, PLACE, IARY, STATUS ) 
*+
*  Name:
*     ARY_NEWP

*  Purpose:
*     Create a new primitive array.

*  Language:
*     Starlink Fortran 77

*  Invocation:
*     CALL ARY_NEWP( FTYPE, NDIM, UBND, PLACE, IARY, STATUS )

*  Description:
*     The routine creates a new primitive array and returns an
*     identifier for it. The array may subsequently be manipulated with
*     the ARY_ routines.

*  Arguments:
*     FTYPE = CHARACTER * ( * ) (Given)
*        Data type of the array (e.g. '_REAL'). Note that complex types
*        are not allowed for primitive arrays.
*     NDIM = INTEGER (Given)
*        Number of array dimensions.
*     UBND( NDIM ) = INTEGER (Given)
*        Upper pixel-index bounds of the array (the lower bound of each
*        dimension is taken to be 1).
*     PLACE = INTEGER (Given and Returned)
*        An array placeholder (e.g. generated by the ARY_PLACE routine)
*        which indicates the position in the data system where the new
*        array will reside. The placeholder is annulled by this
*        routine, and a value of ARY__NOPL will be returned (as defined
*        in the include file ARY_PAR).
*     IARY = INTEGER (Returned)
*        Identifier for the new array.
*     STATUS = INTEGER (Given and Returned)
*        The global status.

*  Notes:
*     -  If this routine is called with STATUS set, then a value of
*     ARY__NOID will be returned for the IARY argument, although no
*     further processing will occur. The same value will also be
*     returned if the routine should fail for any reason. In either
*     event, the placeholder will still be annulled. The ARY__NOID
*     constant is defined in the include file ARY_PAR.

*  Algorithm:
*     -  Set an initial value for the IARY argument.
*     -  Save the error context on entry.
*     -  Import the placeholder identifier.
*     -  If no errors have occurred, then check the data type
*     specification for validity.
*     -  If a complex data type was specified then report an error.
*     -  Check the array upper bounds for validity.
*     -  Create the new array structure with a DCB entry to refer to it.
*     -  Create a new base array entry in the ACB to refer to the DCB
*     entry.
*     -  Export an identifier for the array.
*     -  Annul the placeholder, erasing the associated object if any
*     error occurred.
*     -  Restore the error context, reporting context information if
*     appropriate.

*  Authors:
*     RFWS: R.F. Warren-Smith (STARLINK)
*     BKM:  B.K. McIlwrath    (STARLINK)
*     {enter_new_authors_here}

*  History:
*     13-FEB-1990 (RFWS):
*        Original, derived from the ARY_NEW routine.
*     15-JAN-1996 (BKM):
*        Bug in initialisation of LBND corrected.
*     {enter_changes_here}

*  Bugs:
*     {note_any_bugs_here}

*-
      
*  Type Definitions:
      IMPLICIT NONE              ! No implicit typing

*  Global Constants:
      INCLUDE 'SAE_PAR'          ! Standard SAE constants
      INCLUDE 'DAT_PAR'          ! DAT_ public constants
      INCLUDE 'ARY_PAR'          ! ARY_ public constants
      INCLUDE 'ARY_CONST'        ! ARY_ private constants
      INCLUDE 'ARY_ERR'          ! ARY_ error codes

*  Global Variables:
      INCLUDE 'ARY_PCB'          ! ARY_ Placeholder Control Block
*        PCB_LOC( ARY__MXPCB ) = CHARACTER * ( DAT__SZLOC ) (Read and
*        Write)
*           Locator to placeholder object.
*        PCB_TMP( ARY__MXPCB ) = LOGICAL (Read)
*           Whether the object which replaces the placeholder object
*           should be temporary.

*  Arguments Given:
      CHARACTER * ( * ) FTYPE
      INTEGER NDIM
      INTEGER UBND( * )
      INTEGER PLACE

*  Arguments Returned:
      INTEGER IARY

*  Status:
      INTEGER STATUS             ! Global status

*  Local Variables:
      CHARACTER * ( ARY__SZTYP ) VTYPE ! Numeric type of the array
      INTEGER I                  ! Loop counter for dimensions
      INTEGER IACB               ! Index of array entry in the ACB
      INTEGER IDCB               ! Index of data object entry in the DCB
      INTEGER IPCB               ! Index to placeholder entry in the PCB
      INTEGER LBND( ARY__MXDIM ) ! Lower bounds of array
      INTEGER TSTAT              ! Temporary status variable
      LOGICAL CMPLX              ! Whether a complex array is required
      LOGICAL ERASE              ! Whether to erase placeholder object

*.

*  Set an initial value for the IARY argument.
      IARY = ARY__NOID
       
*  Save the STATUS value and mark the error stack.
      TSTAT = STATUS
      CALL ERR_MARK
       
*  Import the array placeholder, converting it to a PCB index.
      STATUS = SAI__OK
      IPCB = 0
      CALL ARY1_IMPPL( PLACE, IPCB, STATUS )

*  If there has been no error at all so far, then check the full type
*  specification for validity.
      IF ( ( STATUS .EQ. SAI__OK ) .AND. ( TSTAT .EQ. SAI__OK ) ) THEN
         CALL ARY1_VFTP( FTYPE, VTYPE, CMPLX, STATUS )

*  If the type is complex, then report an error.
         IF ( CMPLX ) THEN
            STATUS = ARY__FTPIN
            CALL MSG_SETC( 'BADTYPE', FTYPE )
            CALL ERR_REP( 'ARY_NEWP_TYPE',
     :      'The complex type ''^BADTYPE'' is not valid for a ' //
     :      'primitive array (possible programming error).', STATUS )
         END IF

*  Check the array upper bounds for validity.
         IF ( STATUS .EQ. SAI__OK ) THEN
            DO 1 I = 1, MIN( NDIM, ARY__MXDIM )
               LBND( I ) = 1
1           CONTINUE
            CALL ARY1_VBND( NDIM, LBND, UBND, STATUS )

*  Create a new simple array structure in place of the placeholder
*  object, obtaining a DCB entry which refers to it.
            IF ( STATUS .EQ. SAI__OK ) THEN
               CALL ARY1_DCREP( VTYPE, NDIM, UBND, PCB_TMP( IPCB ),
     :                          PCB_LOC( IPCB ), IDCB, STATUS )
            END IF
         END IF

*  Create a base array entry in the ACB to refer to the DCB entry.
         CALL ARY1_CRNBA( IDCB, IACB, STATUS )

*  Export an identifier for the array.
         CALL ARY1_EXPID( IACB, IARY, STATUS )
      END IF
       
*  Annul the placeholder, erasing the associated object if any error has
*  occurred.
      IF ( IPCB .NE. 0 ) THEN
         ERASE = ( STATUS .NE. SAI__OK ) .OR. ( TSTAT .NE. SAI__OK )
         CALL ARY1_ANNPL( ERASE, IPCB, STATUS )
      END IF

*  Reset the PLACE argument.
      PLACE = ARY__NOPL

*  Annul any error if STATUS was previously bad, otherwise let the new
*  error report stand.
      IF ( STATUS .NE. SAI__OK ) THEN
         IF ( TSTAT .NE. SAI__OK ) THEN
            CALL ERR_ANNUL( STATUS )
            STATUS = TSTAT

*  If appropriate, report the error context and call the error tracing
*  routine.
         ELSE
            IARY = ARY__NOID
            CALL ERR_REP( 'ARY_NEWP_ERR',
     :      'ARY_NEWP: Error creating a new primitive array.', STATUS )
            CALL ARY1_TRACE( 'ARY_NEWP', STATUS )
         END IF
      ELSE
         STATUS = TSTAT
      END IF

*  Release error stack.
      CALL ERR_RLSE

      END
