      SUBROUTINE SUBPAR_PROMPT ( PARAM, PROMPT, DFAULT, HELP, HLPKEY,
     :                           ERRMES, VALUE, STATUS )
*+
*  Name:
*     SUBPAR_PROMPT

*  Purpose:
*     Get input with prompt.

*  Language:
*     Starlink Fortran 77

*  Invocation:
*     CALL SUBPAR_PROMPT ( PARAM, PROMPT, DFAULT, HELP, HLPKEY,
*                          ERRMES, VALUE, STATUS )

*  Description:
*     Prompts a user for input, giving a prompt string and an
*     acceptable default value. Requests for help are handled, as are
*     instructions to return null or abort status.
*     This routine is called instead of SUBPAR_PROMPTCL by SUBPAR_REQUEST
*     when the task is connected directly to the terminal.

*  Arguments:
*     PARAM=CHARACTER*(*) (given)
*        Name of parameter
*     PROMPT=CHARACTER*(*) (given)
*        prompt string
*     DFAULT=CHARACTER*(*) (given)
*        default value
*     HELP=CHARACTER*(*) (given)
*        one-line help specifier
*     HLPKEY=CHARACTER*(*) (given)
*        fullhelp specifier
*     ERRMES=CHARACTER*(*) (given)
*        one-line error message
*     VALUE=CHARACTER*(*) (returned)
*     STATUS=INTEGER

*  Algorithm:
*     The string to be shown to the user is generated by taking the
*     parameter's prompt string, or its name if the prompt string is
*     undefined, concatenating the character-encoded default, and adding
*     a '>'.
*     The user's reply is checked for the special characters:
*          ' '  => user typed carriage return
*                  if there is a default value, use it
*                  otherwise prompt again
*          '?'  => use the 'help' specifier or, if there isn't one,
*                  use the 'helpkey' specifier and return.
*          '??' => use the 'helpkey' specifier if there is one and stay
*                  for help library search. If there isn't a 'helpkey'
*                  specifier, report and give one-line help, if any.
*          '!' and '!!' - set appropriate status.
*           Anything else => accept it as the value.
*     If ERRMES exists, then it is output first.
*     A maximum number of 5 CRs in response to a prompt without a prompt
*     value are allowed in order to trap the case of insufficient input
*     supplied with a batch job.

*  Implementation Deficiencies:
*     Uses VMS specific LBR$OUTPUT_HELP

*  Authors:
*     BDK: B D Kelly (ROE)
*     AJC: A J Chipperfield (STARLINK)
*     JAB: J A Bailey (JACH)
*     {enter_new_authors_here}

*  History:
*     24-SEP-1984 (BDK):
*        Original
*     08-SEP-1986 (BDK):
*        Add space to end of prompt
*     09-JUN-1987 (BDK):
*        Allow acceptance of NULL prompt defaulty
*     13-SEP-1988 (JAB):
*        Use ICL_READA for input and form prompt strings
*        compatible with ICL
*     25-SEP-1989 (AJC):
*        Re-prompt if carriage return with no default.
*     18-JAN-1990 (AJC):
*        Trap repeated CR when no prompt value.
*     03-SEP-1990 (AJC):
*        Add multi-line help - use SUBPAR_SELHELP
*        Shorten names
*     16-SEP-1991 (AJC):
*        Use CHR_LEN not STR$TRIM
*     14-JAN-1992 (AJC):
*        Report when setting STATUS
*     04-AUG-1991 (AJC):
*        Remove %REF in call to ICL_READA for UNIX
*     09-AUG-1993 (AJC):
*        INCLUDE SUBPAR_PARERR not PAR_ERR
*     28-JUN-1995 (AJC):
*        Remove illegal concatenation '*'//ERRMES in call to ICL_WRITEA
*        - it's redundant now anyway.
*      5-FEB-1996 (AJC):
*        Allow longer prompt string (was 200)
*      5-AUG-1998 (AJC):
*        Ensure DEFLEN set even when 0
*     17-AUG-2001 (AJC)
*        Make env var ADAM_NOPROMPT prevent prompting
*     {enter_further_changes_here}

*  Bugs:
*     {note_any_bugs_here}

*-

*  Type Definitions:
      IMPLICIT NONE


*  Global Constants:
      INCLUDE 'SAE_PAR'
      INCLUDE 'SUBPAR_PARERR'

*  Local Constants:
      INTEGER MCLENGTH
      PARAMETER ( MCLENGTH = 444 )

*  Arguments Given:
      CHARACTER*(*) PARAM               ! Name of parameter

      CHARACTER*(*) PROMPT              ! prompt string

      CHARACTER*(*) DFAULT              ! default value

      CHARACTER*(*) HELP                ! one-line help specifier

      CHARACTER*(*) HLPKEY              ! fulhelp specifier

      CHARACTER*(*) ERRMES              ! error message


*  Arguments Returned:
      CHARACTER*(*) VALUE               ! value obtained


*  Status:
      INTEGER STATUS


*  External References:
      INTEGER CHR_LEN
      EXTERNAL CHR_LEN


*  Local Variables:
      CHARACTER*(MCLENGTH) PRTEXT           ! generated prompt string
      CHARACTER*(MCLENGTH) P2STR            ! alternative prompt string
      INTEGER LENGTH                 ! length of generated prompt string
      INTEGER PARLEN                 ! length of parameter name
      INTEGER PRLEN                  ! length of prompt string
      INTEGER DEFLEN                 ! length of deafult string
      INTEGER HELPLEN                ! length of help string
      INTEGER HLPKEYLEN              ! length of fullhelp specifier
      INTEGER P2LEN                  ! length of alternative prompt string
      INTEGER LENERR                 ! length of error message
      INTEGER ENDLINE                ! location of CR in input
      LOGICAL FINISHED               ! loop controller
      INTEGER NCRS                   ! count CRs
      CHARACTER*132 HELPLB           ! help library name
      INTEGER LIBLEN                 ! used length of HELPLB
      CHARACTER*132 STRING           ! key string or error mess
      INTEGER STRLEN                 ! used length of STRING
      LOGICAL LBSRCH                 ! whether to stay in help
      INTEGER SRCHFG                 ! flag for LBR$OUTPUT_HELP

*.


      IF ( STATUS .NE. SAI__OK ) RETURN

*   If ADAM_NOPROMPT is set, do not prompt
      CALL PSX_GETENV( 'ADAM_NOPROMPT', STRING, STATUS )
      IF ( STATUS .EQ. SAI__OK ) THEN
         STATUS = PAR__NOUSR
         CALL EMS_SETC( 'NAME', PARAM )
         CALL EMS_REP( 'SUP_PROMPTCL1',
     :   'SUBPAR: Parameter ^NAME has no value - prompting disallowed.',
     :    STATUS )

      ELSE
         CALL EMS_ANNUL( STATUS )

*
*   Assemble the prompt string
*
         IF ( PROMPT(1:1) .EQ. ' ' ) THEN
            PARLEN = CHR_LEN ( PARAM )
            PRTEXT = PARAM(1:PARLEN)
            PRLEN = PARLEN
         ELSE
            PARLEN = CHR_LEN ( PARAM )
            LENGTH = CHR_LEN ( PROMPT )
            PRTEXT = PARAM(1:PARLEN)//' - '//PROMPT(1:LENGTH)
            PRLEN = 3 + LENGTH + PARLEN
         ENDIF

         P2STR = PRTEXT
         P2LEN = PRLEN
         P2STR(P2LEN+1:) = ' > '
         P2LEN = P2LEN + 3

         DEFLEN = CHR_LEN( DFAULT )
         IF ( DEFLEN .NE. 0 ) THEN
            PRTEXT = PRTEXT(1:PRLEN) // ' /' // DFAULT
            PRLEN = PRLEN + 2 + DEFLEN
            PRTEXT = PRTEXT(1:PRLEN) // '/' // ' > '
            PRLEN = PRLEN + 4
         ELSE
            PRTEXT = PRTEXT(1:PRLEN) // ' > '
            PRLEN = PRLEN + 3
         ENDIF

*
*   If there is an error message, output it first.
*
         IF ( ERRMES .NE. ' ' ) THEN
            LENERR = CHR_LEN ( ERRMES )
            CALL ICL_WRITEA(ERRMES(1:LENERR))
         ENDIF
*
*   Now loop asking for the value until a reply is received.
*
         NCRS = 0
         FINISHED = .FALSE.

         DO WHILE ( .NOT. FINISHED )

            VALUE = ' '
!         CALL ICL_READA(PRTEXT,PRLEN,P2STR,P2LEN,%REF(VALUE),
            CALL ICL_READA(PRTEXT,PRLEN,P2STR,P2LEN,VALUE,
     :          LEN(VALUE),DFAULT,DEFLEN)

            ENDLINE = INDEX( VALUE, CHAR(13) )
            IF (ENDLINE .NE. 0) VALUE(ENDLINE:) = ' '

            IF ( VALUE(1:1) .EQ. '?' ) THEN
               NCRS = 0
*         Help is required
               ENDLINE = MAX(1, CHR_LEN( VALUE ))
               HELPLEN = MAX(1, CHR_LEN( HELP ))
               HLPKEYLEN = MAX(1, CHR_LEN( HLPKEY ))
*         Determine required help
               CALL SUBPAR_SELHELP( PARAM(1:PARLEN), VALUE(1:ENDLINE),
     :                        HELP(1:HELPLEN), HLPKEY(1:HLPKEYLEN),
     :                        HELPLB, LIBLEN, STRING, STRLEN, LBSRCH,
     :                        STATUS )

*         Output required help - or error message
               IF (( HELPLB.EQ.' ') .AND. (STRING.EQ.' ')) THEN
*            Output single-line help
                  CALL ICL_WRITEA( HELP(1:HELPLEN) )

               ELSEIF (HELPLB.NE.' ') THEN
*            Output full help
*            set SRCHFG =1 to stay in help
                  IF (LBSRCH) THEN
                     SRCHFG = 1
                  ELSE
                     SRCHFG = 0
                  ENDIF
                  CALL SUBPAR_WRHELP (STRING(1:STRLEN),
     :              HELPLB(1:LIBLEN), SRCHFG, STATUS )

               ELSE
*            Output error message
                  CALL ICL_WRITEA( STRING(1:STRLEN) )

               ENDIF

            ELSE IF ( VALUE .EQ. ' ' ) THEN
*         If there is a default value, use it; otherwise re-prompt
               IF ( DFAULT .NE. ' ' ) THEN
                  VALUE = DFAULT
                  FINISHED = .TRUE.
               ELSE
                  NCRS = NCRS + 1
                  IF ( NCRS .EQ. 5 ) THEN
                     STATUS = PAR__NULL
                     CALL EMS_SETC( 'NAME', PARAM )
                     CALL EMS_REP( 'SUP_PROMPT1',
     :               'SUBPAR: Failed to get value after 5 prompts '//
     :               'for parameter ^NAME', STATUS )
                     FINISHED = .TRUE.
                  ENDIF
               ENDIF
            ELSE
               FINISHED = .TRUE.
            ENDIF

         ENDDO
*
*   Check for special characters
*
         IF ( STATUS .EQ. SAI__OK ) THEN
            IF ( VALUE(1:2) .EQ. '!!' ) THEN
               STATUS = PAR__ABORT
               CALL EMS_SETC( 'NAME', PARAM )
               CALL EMS_REP( 'SUP_PROMPT2', 'SUBPAR: ' //
     :         'Abort (!!) response to prompt for parameter ^NAME',
     :          STATUS)
            ELSE IF ( VALUE(1:1) .EQ. '!' ) THEN
               STATUS = PAR__NULL
               CALL EMS_SETC( 'NAME', PARAM )
               CALL EMS_REP( 'SUP_PROMPT3', 'SUBPAR: '//
     :         'Null (!) response to prompt for parameter ^NAME',
     :          STATUS)
            ENDIF
         ENDIF

      ENDIF

      END
