      SUBROUTINE NDF1_QMAB( EL, QUAL, BADBIT, NARRAY, ARRAY1, ARRAY2,
     :                        ARRAY3, ARRAY4, BAD, STATUS )
*+
*  Name:
*     NDF1_QMAB

*  Purpose:
*     Perform quality masking on vectorised BYTE arrays.

*  Language:
*     Starlink Fortran 77

*  Invocation:
*     CALL NDF1_QMAB( EL, QUAL, BADBIT, NARRAY, ARRAY1, ARRAY2,
*                       ARRAY3, ARRAY4, BAD, STATUS )

*  Description:
*     The routine converts a vectorised quality array QUAL into a set
*     of "bad pixel" flags which are inserted into a matching set of up
*     to 4 vectorised BYTE arrays.  This is done by performing a
*     bit-wise "AND" operation between elements of the QUAL array and
*     the bitmask BADBIT. Where the result of this operation is
*     non-zero, the corresponding elements of ARRAY1...ARRAY4 are set
*     to the "bad" value VAL__BADB (the precise number of arrays
*     processed is determined by the NARRAY argument).  Other array
*     elements are unchanged.  A logical value BAD is also returned
*     indicating whether any "bad" pixels were actually generated by
*     this quality masking process.

*  Arguments:
*     EL = INTEGER (Given)
*        The number of elements to process in each vectorised array.
*     QUAL( EL ) = BYTE (Given)
*        The quality array.
*     BADBIT = BYTE (Given)
*        The bad-bits mask to be applied to the quality array.
*     NARRAY = INTEGER (Given)
*        Number of vectorised arrays to be processed in the range 1 to
*        4. The routine returns without action if this value is out of
*        range.
*     ARRAY1...ARRAY4( EL ) = BYTE (Given and Returned)
*        The vectorised BYTE arrays to be processed. Up to four of
*        these will be processed, according to the value of the NARRAY
*        argument.  Any remaining arrays are not used.
*     BAD = LOGICAL (Returned)
*        Whether any bad pixels were generated as a result of the
*        quality masking process.
*     STATUS = INTEGER (Given and Returned)
*        The global status.

*  Notes:
*     -  This routine loops through the arrays in a backward direction.
*     This is to minimise paging on a virtual memory machine, since
*     this routine will usually be followed by a processing loop which
*     passes forward through the same arrays.

*  Algorithm:
*     -  Initialise.
*     -  Check the badbits value is not zero (there is nothing to do if
*     it is).
*     -  If one array is being processed, then loop to process each
*     array element.
*     -  Evaluate the quality masking function.
*     -  Set affected array elements to the bad value and note this has
*     happened.
*     -  Repeat this process for each value of NARRAY, setting bad
*     pixels in the appropriate number of output arrays in each case.

*  Authors:
*     RFWS: R.F. Warren-Smith (STARLINK)
*     {enter_new_authors_here}

*  History:
*     21-NOV-1989 (RFWS):
*        Original version.
*     6-FEB-1990 (RFWS):
*        Extended to handle up to four arrays at once.
*     20-MAR-1990 (RFWS):
*        Changed to loop through the arrays backwards.
*     20-MAR-1990 (RFWS):
*        Changed to define the quality masking operation by means of a
*        statement function defined in an include file.
*     {enter_further_changes_here}

*  Bugs:
*     {note_any_bugs_here}

*-
      
*  Type Definitions:
      IMPLICIT NONE              ! No implicit typing

*  Global Constants:
      INCLUDE 'SAE_PAR'          ! Standard SAE constants
      INCLUDE 'DAT_PAR'          ! DAT_ public_constants
      INCLUDE 'PRM_PAR'          ! PRIMDAT primitive data constants

*  Arguments Given:
      INTEGER EL
      BYTE QUAL( EL )
      BYTE BADBIT
      INTEGER NARRAY

*  Arguments Given and Returned:
      BYTE ARRAY1( * )
      BYTE ARRAY2( * )
      BYTE ARRAY3( * )
      BYTE ARRAY4( * )

*  Arguments Returned:
      LOGICAL BAD

*  Status:
      INTEGER STATUS             ! Global status

*  Local Constants:
      BYTE ZEROUB                ! Zero as an unsigned byte value
      PARAMETER ( ZEROUB = 0 )

*  Local Variables:
      INTEGER I                  ! Loop counter for array elements

*  Internal References:
      INCLUDE 'NDF_FUNC_DEC'     ! Declare NDF_ statement functions
      INCLUDE 'NDF_FUNC_DEF'     ! Define NDF_ statement functions

*.

*  Check inherited global status.
      IF ( STATUS .NE. SAI__OK ) RETURN

*  Initialise.
      BAD = .FALSE.

*  There is nothing to do unless the mask is non-zero.
      IF ( BADBIT .NE. ZEROUB ) THEN

*  1 array to process:
*  ==================
*  Loop to process each array element.
         IF ( NARRAY .EQ. 1 ) THEN
            DO 1 I = EL, 1, -1

*  Evaluate the quality masking function.
               IF ( .NOT. NDF_QMASK( QUAL( I ), BADBIT ) ) THEN

*  Set affected array elements to the bad value and note this has
*  happened.
                  ARRAY1( I ) = VAL__BADB
                  BAD = .TRUE.
               END IF
1           CONTINUE

*  2 arrays to process:
*  ===================
*  Loop to process each array element.
         ELSE IF ( NARRAY .EQ. 2 ) THEN
            DO 2 I = EL, 1, -1

*  Evaluate the quality masking function.
               IF ( .NOT. NDF_QMASK( QUAL( I ), BADBIT ) ) THEN

*  Set affected array elements to the bad value and note this has
*  happened.
                  ARRAY1( I ) = VAL__BADB
                  ARRAY2( I ) = VAL__BADB
                  BAD = .TRUE.
               END IF
2           CONTINUE

*  3 arrays to process:
*  ===================
*  Loop to process each array element.
         ELSE IF ( NARRAY .EQ. 3 ) THEN
            DO 3 I = EL, 1, -1

*  Evaluate the quality masking function.
               IF ( .NOT. NDF_QMASK( QUAL( I ), BADBIT ) ) THEN

*  Set affected array elements to the bad value and note this has
*  happened.
                  ARRAY1( I ) = VAL__BADB
                  ARRAY2( I ) = VAL__BADB
                  ARRAY3( I ) = VAL__BADB
                  BAD = .TRUE.
               END IF
3           CONTINUE

*  4 arrays to process:
*  ===================
*  Loop to process each array element.
         ELSE IF ( NARRAY .EQ. 4 ) THEN
            DO 4 I = EL, 1, -1

*  Evaluate the quality masking function.
               IF ( .NOT. NDF_QMASK( QUAL( I ), BADBIT ) ) THEN

*  Set affected array elements to the bad value and note this has
*  happened.
                  ARRAY1( I ) = VAL__BADB
                  ARRAY2( I ) = VAL__BADB
                  ARRAY3( I ) = VAL__BADB
                  ARRAY4( I ) = VAL__BADB
                  BAD = .TRUE.
               END IF
4           CONTINUE
         END IF
      END IF
       
*  Call error tracing routine and exit.
      IF ( STATUS .NE. SAI__OK ) CALL NDF1_TRACE( 'NDF1_QMAB',
     :                                            STATUS )

      END
      SUBROUTINE NDF1_QMAUB( EL, QUAL, BADBIT, NARRAY, ARRAY1, ARRAY2,
     :                        ARRAY3, ARRAY4, BAD, STATUS )
*+
*  Name:
*     NDF1_QMAUB

*  Purpose:
*     Perform quality masking on vectorised UNSIGNED BYTE arrays.

*  Language:
*     Starlink Fortran 77

*  Invocation:
*     CALL NDF1_QMAUB( EL, QUAL, BADBIT, NARRAY, ARRAY1, ARRAY2,
*                       ARRAY3, ARRAY4, BAD, STATUS )

*  Description:
*     The routine converts a vectorised quality array QUAL into a set
*     of "bad pixel" flags which are inserted into a matching set of up
*     to 4 vectorised UNSIGNED BYTE arrays.  This is done by performing a
*     bit-wise "AND" operation between elements of the QUAL array and
*     the bitmask BADBIT. Where the result of this operation is
*     non-zero, the corresponding elements of ARRAY1...ARRAY4 are set
*     to the "bad" value VAL__BADUB (the precise number of arrays
*     processed is determined by the NARRAY argument).  Other array
*     elements are unchanged.  A logical value BAD is also returned
*     indicating whether any "bad" pixels were actually generated by
*     this quality masking process.

*  Arguments:
*     EL = INTEGER (Given)
*        The number of elements to process in each vectorised array.
*     QUAL( EL ) = BYTE (Given)
*        The quality array.
*     BADBIT = BYTE (Given)
*        The bad-bits mask to be applied to the quality array.
*     NARRAY = INTEGER (Given)
*        Number of vectorised arrays to be processed in the range 1 to
*        4. The routine returns without action if this value is out of
*        range.
*     ARRAY1...ARRAY4( EL ) = BYTE (Given and Returned)
*        The vectorised UNSIGNED BYTE arrays to be processed. Up to four of
*        these will be processed, according to the value of the NARRAY
*        argument.  Any remaining arrays are not used.
*     BAD = LOGICAL (Returned)
*        Whether any bad pixels were generated as a result of the
*        quality masking process.
*     STATUS = INTEGER (Given and Returned)
*        The global status.

*  Notes:
*     -  This routine loops through the arrays in a backward direction.
*     This is to minimise paging on a virtual memory machine, since
*     this routine will usually be followed by a processing loop which
*     passes forward through the same arrays.

*  Algorithm:
*     -  Initialise.
*     -  Check the badbits value is not zero (there is nothing to do if
*     it is).
*     -  If one array is being processed, then loop to process each
*     array element.
*     -  Evaluate the quality masking function.
*     -  Set affected array elements to the bad value and note this has
*     happened.
*     -  Repeat this process for each value of NARRAY, setting bad
*     pixels in the appropriate number of output arrays in each case.

*  Authors:
*     RFWS: R.F. Warren-Smith (STARLINK)
*     {enter_new_authors_here}

*  History:
*     21-NOV-1989 (RFWS):
*        Original version.
*     6-FEB-1990 (RFWS):
*        Extended to handle up to four arrays at once.
*     20-MAR-1990 (RFWS):
*        Changed to loop through the arrays backwards.
*     20-MAR-1990 (RFWS):
*        Changed to define the quality masking operation by means of a
*        statement function defined in an include file.
*     {enter_further_changes_here}

*  Bugs:
*     {note_any_bugs_here}

*-
      
*  Type Definitions:
      IMPLICIT NONE              ! No implicit typing

*  Global Constants:
      INCLUDE 'SAE_PAR'          ! Standard SAE constants
      INCLUDE 'DAT_PAR'          ! DAT_ public_constants
      INCLUDE 'PRM_PAR'          ! PRIMDAT primitive data constants

*  Arguments Given:
      INTEGER EL
      BYTE QUAL( EL )
      BYTE BADBIT
      INTEGER NARRAY

*  Arguments Given and Returned:
      BYTE ARRAY1( * )
      BYTE ARRAY2( * )
      BYTE ARRAY3( * )
      BYTE ARRAY4( * )

*  Arguments Returned:
      LOGICAL BAD

*  Status:
      INTEGER STATUS             ! Global status

*  Local Constants:
      BYTE ZEROUB                ! Zero as an unsigned byte value
      PARAMETER ( ZEROUB = 0 )

*  Local Variables:
      INTEGER I                  ! Loop counter for array elements

*  Internal References:
      INCLUDE 'NDF_FUNC_DEC'     ! Declare NDF_ statement functions
      INCLUDE 'NDF_FUNC_DEF'     ! Define NDF_ statement functions

*.

*  Check inherited global status.
      IF ( STATUS .NE. SAI__OK ) RETURN

*  Initialise.
      BAD = .FALSE.

*  There is nothing to do unless the mask is non-zero.
      IF ( BADBIT .NE. ZEROUB ) THEN

*  1 array to process:
*  ==================
*  Loop to process each array element.
         IF ( NARRAY .EQ. 1 ) THEN
            DO 1 I = EL, 1, -1

*  Evaluate the quality masking function.
               IF ( .NOT. NDF_QMASK( QUAL( I ), BADBIT ) ) THEN

*  Set affected array elements to the bad value and note this has
*  happened.
                  ARRAY1( I ) = VAL__BADUB
                  BAD = .TRUE.
               END IF
1           CONTINUE

*  2 arrays to process:
*  ===================
*  Loop to process each array element.
         ELSE IF ( NARRAY .EQ. 2 ) THEN
            DO 2 I = EL, 1, -1

*  Evaluate the quality masking function.
               IF ( .NOT. NDF_QMASK( QUAL( I ), BADBIT ) ) THEN

*  Set affected array elements to the bad value and note this has
*  happened.
                  ARRAY1( I ) = VAL__BADUB
                  ARRAY2( I ) = VAL__BADUB
                  BAD = .TRUE.
               END IF
2           CONTINUE

*  3 arrays to process:
*  ===================
*  Loop to process each array element.
         ELSE IF ( NARRAY .EQ. 3 ) THEN
            DO 3 I = EL, 1, -1

*  Evaluate the quality masking function.
               IF ( .NOT. NDF_QMASK( QUAL( I ), BADBIT ) ) THEN

*  Set affected array elements to the bad value and note this has
*  happened.
                  ARRAY1( I ) = VAL__BADUB
                  ARRAY2( I ) = VAL__BADUB
                  ARRAY3( I ) = VAL__BADUB
                  BAD = .TRUE.
               END IF
3           CONTINUE

*  4 arrays to process:
*  ===================
*  Loop to process each array element.
         ELSE IF ( NARRAY .EQ. 4 ) THEN
            DO 4 I = EL, 1, -1

*  Evaluate the quality masking function.
               IF ( .NOT. NDF_QMASK( QUAL( I ), BADBIT ) ) THEN

*  Set affected array elements to the bad value and note this has
*  happened.
                  ARRAY1( I ) = VAL__BADUB
                  ARRAY2( I ) = VAL__BADUB
                  ARRAY3( I ) = VAL__BADUB
                  ARRAY4( I ) = VAL__BADUB
                  BAD = .TRUE.
               END IF
4           CONTINUE
         END IF
      END IF
       
*  Call error tracing routine and exit.
      IF ( STATUS .NE. SAI__OK ) CALL NDF1_TRACE( 'NDF1_QMAUB',
     :                                            STATUS )

      END
      SUBROUTINE NDF1_QMAD( EL, QUAL, BADBIT, NARRAY, ARRAY1, ARRAY2,
     :                        ARRAY3, ARRAY4, BAD, STATUS )
*+
*  Name:
*     NDF1_QMAD

*  Purpose:
*     Perform quality masking on vectorised DOUBLE PRECISION arrays.

*  Language:
*     Starlink Fortran 77

*  Invocation:
*     CALL NDF1_QMAD( EL, QUAL, BADBIT, NARRAY, ARRAY1, ARRAY2,
*                       ARRAY3, ARRAY4, BAD, STATUS )

*  Description:
*     The routine converts a vectorised quality array QUAL into a set
*     of "bad pixel" flags which are inserted into a matching set of up
*     to 4 vectorised DOUBLE PRECISION arrays.  This is done by performing a
*     bit-wise "AND" operation between elements of the QUAL array and
*     the bitmask BADBIT. Where the result of this operation is
*     non-zero, the corresponding elements of ARRAY1...ARRAY4 are set
*     to the "bad" value VAL__BADD (the precise number of arrays
*     processed is determined by the NARRAY argument).  Other array
*     elements are unchanged.  A logical value BAD is also returned
*     indicating whether any "bad" pixels were actually generated by
*     this quality masking process.

*  Arguments:
*     EL = INTEGER (Given)
*        The number of elements to process in each vectorised array.
*     QUAL( EL ) = BYTE (Given)
*        The quality array.
*     BADBIT = BYTE (Given)
*        The bad-bits mask to be applied to the quality array.
*     NARRAY = INTEGER (Given)
*        Number of vectorised arrays to be processed in the range 1 to
*        4. The routine returns without action if this value is out of
*        range.
*     ARRAY1...ARRAY4( EL ) = DOUBLE PRECISION (Given and Returned)
*        The vectorised DOUBLE PRECISION arrays to be processed. Up to four of
*        these will be processed, according to the value of the NARRAY
*        argument.  Any remaining arrays are not used.
*     BAD = LOGICAL (Returned)
*        Whether any bad pixels were generated as a result of the
*        quality masking process.
*     STATUS = INTEGER (Given and Returned)
*        The global status.

*  Notes:
*     -  This routine loops through the arrays in a backward direction.
*     This is to minimise paging on a virtual memory machine, since
*     this routine will usually be followed by a processing loop which
*     passes forward through the same arrays.

*  Algorithm:
*     -  Initialise.
*     -  Check the badbits value is not zero (there is nothing to do if
*     it is).
*     -  If one array is being processed, then loop to process each
*     array element.
*     -  Evaluate the quality masking function.
*     -  Set affected array elements to the bad value and note this has
*     happened.
*     -  Repeat this process for each value of NARRAY, setting bad
*     pixels in the appropriate number of output arrays in each case.

*  Authors:
*     RFWS: R.F. Warren-Smith (STARLINK)
*     {enter_new_authors_here}

*  History:
*     21-NOV-1989 (RFWS):
*        Original version.
*     6-FEB-1990 (RFWS):
*        Extended to handle up to four arrays at once.
*     20-MAR-1990 (RFWS):
*        Changed to loop through the arrays backwards.
*     20-MAR-1990 (RFWS):
*        Changed to define the quality masking operation by means of a
*        statement function defined in an include file.
*     {enter_further_changes_here}

*  Bugs:
*     {note_any_bugs_here}

*-
      
*  Type Definitions:
      IMPLICIT NONE              ! No implicit typing

*  Global Constants:
      INCLUDE 'SAE_PAR'          ! Standard SAE constants
      INCLUDE 'DAT_PAR'          ! DAT_ public_constants
      INCLUDE 'PRM_PAR'          ! PRIMDAT primitive data constants

*  Arguments Given:
      INTEGER EL
      BYTE QUAL( EL )
      BYTE BADBIT
      INTEGER NARRAY

*  Arguments Given and Returned:
      DOUBLE PRECISION ARRAY1( * )
      DOUBLE PRECISION ARRAY2( * )
      DOUBLE PRECISION ARRAY3( * )
      DOUBLE PRECISION ARRAY4( * )

*  Arguments Returned:
      LOGICAL BAD

*  Status:
      INTEGER STATUS             ! Global status

*  Local Constants:
      BYTE ZEROUB                ! Zero as an unsigned byte value
      PARAMETER ( ZEROUB = 0 )

*  Local Variables:
      INTEGER I                  ! Loop counter for array elements

*  Internal References:
      INCLUDE 'NDF_FUNC_DEC'     ! Declare NDF_ statement functions
      INCLUDE 'NDF_FUNC_DEF'     ! Define NDF_ statement functions

*.

*  Check inherited global status.
      IF ( STATUS .NE. SAI__OK ) RETURN

*  Initialise.
      BAD = .FALSE.

*  There is nothing to do unless the mask is non-zero.
      IF ( BADBIT .NE. ZEROUB ) THEN

*  1 array to process:
*  ==================
*  Loop to process each array element.
         IF ( NARRAY .EQ. 1 ) THEN
            DO 1 I = EL, 1, -1

*  Evaluate the quality masking function.
               IF ( .NOT. NDF_QMASK( QUAL( I ), BADBIT ) ) THEN

*  Set affected array elements to the bad value and note this has
*  happened.
                  ARRAY1( I ) = VAL__BADD
                  BAD = .TRUE.
               END IF
1           CONTINUE

*  2 arrays to process:
*  ===================
*  Loop to process each array element.
         ELSE IF ( NARRAY .EQ. 2 ) THEN
            DO 2 I = EL, 1, -1

*  Evaluate the quality masking function.
               IF ( .NOT. NDF_QMASK( QUAL( I ), BADBIT ) ) THEN

*  Set affected array elements to the bad value and note this has
*  happened.
                  ARRAY1( I ) = VAL__BADD
                  ARRAY2( I ) = VAL__BADD
                  BAD = .TRUE.
               END IF
2           CONTINUE

*  3 arrays to process:
*  ===================
*  Loop to process each array element.
         ELSE IF ( NARRAY .EQ. 3 ) THEN
            DO 3 I = EL, 1, -1

*  Evaluate the quality masking function.
               IF ( .NOT. NDF_QMASK( QUAL( I ), BADBIT ) ) THEN

*  Set affected array elements to the bad value and note this has
*  happened.
                  ARRAY1( I ) = VAL__BADD
                  ARRAY2( I ) = VAL__BADD
                  ARRAY3( I ) = VAL__BADD
                  BAD = .TRUE.
               END IF
3           CONTINUE

*  4 arrays to process:
*  ===================
*  Loop to process each array element.
         ELSE IF ( NARRAY .EQ. 4 ) THEN
            DO 4 I = EL, 1, -1

*  Evaluate the quality masking function.
               IF ( .NOT. NDF_QMASK( QUAL( I ), BADBIT ) ) THEN

*  Set affected array elements to the bad value and note this has
*  happened.
                  ARRAY1( I ) = VAL__BADD
                  ARRAY2( I ) = VAL__BADD
                  ARRAY3( I ) = VAL__BADD
                  ARRAY4( I ) = VAL__BADD
                  BAD = .TRUE.
               END IF
4           CONTINUE
         END IF
      END IF
       
*  Call error tracing routine and exit.
      IF ( STATUS .NE. SAI__OK ) CALL NDF1_TRACE( 'NDF1_QMAD',
     :                                            STATUS )

      END
      SUBROUTINE NDF1_QMAI( EL, QUAL, BADBIT, NARRAY, ARRAY1, ARRAY2,
     :                        ARRAY3, ARRAY4, BAD, STATUS )
*+
*  Name:
*     NDF1_QMAI

*  Purpose:
*     Perform quality masking on vectorised INTEGER arrays.

*  Language:
*     Starlink Fortran 77

*  Invocation:
*     CALL NDF1_QMAI( EL, QUAL, BADBIT, NARRAY, ARRAY1, ARRAY2,
*                       ARRAY3, ARRAY4, BAD, STATUS )

*  Description:
*     The routine converts a vectorised quality array QUAL into a set
*     of "bad pixel" flags which are inserted into a matching set of up
*     to 4 vectorised INTEGER arrays.  This is done by performing a
*     bit-wise "AND" operation between elements of the QUAL array and
*     the bitmask BADBIT. Where the result of this operation is
*     non-zero, the corresponding elements of ARRAY1...ARRAY4 are set
*     to the "bad" value VAL__BADI (the precise number of arrays
*     processed is determined by the NARRAY argument).  Other array
*     elements are unchanged.  A logical value BAD is also returned
*     indicating whether any "bad" pixels were actually generated by
*     this quality masking process.

*  Arguments:
*     EL = INTEGER (Given)
*        The number of elements to process in each vectorised array.
*     QUAL( EL ) = BYTE (Given)
*        The quality array.
*     BADBIT = BYTE (Given)
*        The bad-bits mask to be applied to the quality array.
*     NARRAY = INTEGER (Given)
*        Number of vectorised arrays to be processed in the range 1 to
*        4. The routine returns without action if this value is out of
*        range.
*     ARRAY1...ARRAY4( EL ) = INTEGER (Given and Returned)
*        The vectorised INTEGER arrays to be processed. Up to four of
*        these will be processed, according to the value of the NARRAY
*        argument.  Any remaining arrays are not used.
*     BAD = LOGICAL (Returned)
*        Whether any bad pixels were generated as a result of the
*        quality masking process.
*     STATUS = INTEGER (Given and Returned)
*        The global status.

*  Notes:
*     -  This routine loops through the arrays in a backward direction.
*     This is to minimise paging on a virtual memory machine, since
*     this routine will usually be followed by a processing loop which
*     passes forward through the same arrays.

*  Algorithm:
*     -  Initialise.
*     -  Check the badbits value is not zero (there is nothing to do if
*     it is).
*     -  If one array is being processed, then loop to process each
*     array element.
*     -  Evaluate the quality masking function.
*     -  Set affected array elements to the bad value and note this has
*     happened.
*     -  Repeat this process for each value of NARRAY, setting bad
*     pixels in the appropriate number of output arrays in each case.

*  Authors:
*     RFWS: R.F. Warren-Smith (STARLINK)
*     {enter_new_authors_here}

*  History:
*     21-NOV-1989 (RFWS):
*        Original version.
*     6-FEB-1990 (RFWS):
*        Extended to handle up to four arrays at once.
*     20-MAR-1990 (RFWS):
*        Changed to loop through the arrays backwards.
*     20-MAR-1990 (RFWS):
*        Changed to define the quality masking operation by means of a
*        statement function defined in an include file.
*     {enter_further_changes_here}

*  Bugs:
*     {note_any_bugs_here}

*-
      
*  Type Definitions:
      IMPLICIT NONE              ! No implicit typing

*  Global Constants:
      INCLUDE 'SAE_PAR'          ! Standard SAE constants
      INCLUDE 'DAT_PAR'          ! DAT_ public_constants
      INCLUDE 'PRM_PAR'          ! PRIMDAT primitive data constants

*  Arguments Given:
      INTEGER EL
      BYTE QUAL( EL )
      BYTE BADBIT
      INTEGER NARRAY

*  Arguments Given and Returned:
      INTEGER ARRAY1( * )
      INTEGER ARRAY2( * )
      INTEGER ARRAY3( * )
      INTEGER ARRAY4( * )

*  Arguments Returned:
      LOGICAL BAD

*  Status:
      INTEGER STATUS             ! Global status

*  Local Constants:
      BYTE ZEROUB                ! Zero as an unsigned byte value
      PARAMETER ( ZEROUB = 0 )

*  Local Variables:
      INTEGER I                  ! Loop counter for array elements

*  Internal References:
      INCLUDE 'NDF_FUNC_DEC'     ! Declare NDF_ statement functions
      INCLUDE 'NDF_FUNC_DEF'     ! Define NDF_ statement functions

*.

*  Check inherited global status.
      IF ( STATUS .NE. SAI__OK ) RETURN

*  Initialise.
      BAD = .FALSE.

*  There is nothing to do unless the mask is non-zero.
      IF ( BADBIT .NE. ZEROUB ) THEN

*  1 array to process:
*  ==================
*  Loop to process each array element.
         IF ( NARRAY .EQ. 1 ) THEN
            DO 1 I = EL, 1, -1

*  Evaluate the quality masking function.
               IF ( .NOT. NDF_QMASK( QUAL( I ), BADBIT ) ) THEN

*  Set affected array elements to the bad value and note this has
*  happened.
                  ARRAY1( I ) = VAL__BADI
                  BAD = .TRUE.
               END IF
1           CONTINUE

*  2 arrays to process:
*  ===================
*  Loop to process each array element.
         ELSE IF ( NARRAY .EQ. 2 ) THEN
            DO 2 I = EL, 1, -1

*  Evaluate the quality masking function.
               IF ( .NOT. NDF_QMASK( QUAL( I ), BADBIT ) ) THEN

*  Set affected array elements to the bad value and note this has
*  happened.
                  ARRAY1( I ) = VAL__BADI
                  ARRAY2( I ) = VAL__BADI
                  BAD = .TRUE.
               END IF
2           CONTINUE

*  3 arrays to process:
*  ===================
*  Loop to process each array element.
         ELSE IF ( NARRAY .EQ. 3 ) THEN
            DO 3 I = EL, 1, -1

*  Evaluate the quality masking function.
               IF ( .NOT. NDF_QMASK( QUAL( I ), BADBIT ) ) THEN

*  Set affected array elements to the bad value and note this has
*  happened.
                  ARRAY1( I ) = VAL__BADI
                  ARRAY2( I ) = VAL__BADI
                  ARRAY3( I ) = VAL__BADI
                  BAD = .TRUE.
               END IF
3           CONTINUE

*  4 arrays to process:
*  ===================
*  Loop to process each array element.
         ELSE IF ( NARRAY .EQ. 4 ) THEN
            DO 4 I = EL, 1, -1

*  Evaluate the quality masking function.
               IF ( .NOT. NDF_QMASK( QUAL( I ), BADBIT ) ) THEN

*  Set affected array elements to the bad value and note this has
*  happened.
                  ARRAY1( I ) = VAL__BADI
                  ARRAY2( I ) = VAL__BADI
                  ARRAY3( I ) = VAL__BADI
                  ARRAY4( I ) = VAL__BADI
                  BAD = .TRUE.
               END IF
4           CONTINUE
         END IF
      END IF
       
*  Call error tracing routine and exit.
      IF ( STATUS .NE. SAI__OK ) CALL NDF1_TRACE( 'NDF1_QMAI',
     :                                            STATUS )

      END
      SUBROUTINE NDF1_QMAR( EL, QUAL, BADBIT, NARRAY, ARRAY1, ARRAY2,
     :                        ARRAY3, ARRAY4, BAD, STATUS )
*+
*  Name:
*     NDF1_QMAR

*  Purpose:
*     Perform quality masking on vectorised REAL arrays.

*  Language:
*     Starlink Fortran 77

*  Invocation:
*     CALL NDF1_QMAR( EL, QUAL, BADBIT, NARRAY, ARRAY1, ARRAY2,
*                       ARRAY3, ARRAY4, BAD, STATUS )

*  Description:
*     The routine converts a vectorised quality array QUAL into a set
*     of "bad pixel" flags which are inserted into a matching set of up
*     to 4 vectorised REAL arrays.  This is done by performing a
*     bit-wise "AND" operation between elements of the QUAL array and
*     the bitmask BADBIT. Where the result of this operation is
*     non-zero, the corresponding elements of ARRAY1...ARRAY4 are set
*     to the "bad" value VAL__BADR (the precise number of arrays
*     processed is determined by the NARRAY argument).  Other array
*     elements are unchanged.  A logical value BAD is also returned
*     indicating whether any "bad" pixels were actually generated by
*     this quality masking process.

*  Arguments:
*     EL = INTEGER (Given)
*        The number of elements to process in each vectorised array.
*     QUAL( EL ) = BYTE (Given)
*        The quality array.
*     BADBIT = BYTE (Given)
*        The bad-bits mask to be applied to the quality array.
*     NARRAY = INTEGER (Given)
*        Number of vectorised arrays to be processed in the range 1 to
*        4. The routine returns without action if this value is out of
*        range.
*     ARRAY1...ARRAY4( EL ) = REAL (Given and Returned)
*        The vectorised REAL arrays to be processed. Up to four of
*        these will be processed, according to the value of the NARRAY
*        argument.  Any remaining arrays are not used.
*     BAD = LOGICAL (Returned)
*        Whether any bad pixels were generated as a result of the
*        quality masking process.
*     STATUS = INTEGER (Given and Returned)
*        The global status.

*  Notes:
*     -  This routine loops through the arrays in a backward direction.
*     This is to minimise paging on a virtual memory machine, since
*     this routine will usually be followed by a processing loop which
*     passes forward through the same arrays.

*  Algorithm:
*     -  Initialise.
*     -  Check the badbits value is not zero (there is nothing to do if
*     it is).
*     -  If one array is being processed, then loop to process each
*     array element.
*     -  Evaluate the quality masking function.
*     -  Set affected array elements to the bad value and note this has
*     happened.
*     -  Repeat this process for each value of NARRAY, setting bad
*     pixels in the appropriate number of output arrays in each case.

*  Authors:
*     RFWS: R.F. Warren-Smith (STARLINK)
*     {enter_new_authors_here}

*  History:
*     21-NOV-1989 (RFWS):
*        Original version.
*     6-FEB-1990 (RFWS):
*        Extended to handle up to four arrays at once.
*     20-MAR-1990 (RFWS):
*        Changed to loop through the arrays backwards.
*     20-MAR-1990 (RFWS):
*        Changed to define the quality masking operation by means of a
*        statement function defined in an include file.
*     {enter_further_changes_here}

*  Bugs:
*     {note_any_bugs_here}

*-
      
*  Type Definitions:
      IMPLICIT NONE              ! No implicit typing

*  Global Constants:
      INCLUDE 'SAE_PAR'          ! Standard SAE constants
      INCLUDE 'DAT_PAR'          ! DAT_ public_constants
      INCLUDE 'PRM_PAR'          ! PRIMDAT primitive data constants

*  Arguments Given:
      INTEGER EL
      BYTE QUAL( EL )
      BYTE BADBIT
      INTEGER NARRAY

*  Arguments Given and Returned:
      REAL ARRAY1( * )
      REAL ARRAY2( * )
      REAL ARRAY3( * )
      REAL ARRAY4( * )

*  Arguments Returned:
      LOGICAL BAD

*  Status:
      INTEGER STATUS             ! Global status

*  Local Constants:
      BYTE ZEROUB                ! Zero as an unsigned byte value
      PARAMETER ( ZEROUB = 0 )

*  Local Variables:
      INTEGER I                  ! Loop counter for array elements

*  Internal References:
      INCLUDE 'NDF_FUNC_DEC'     ! Declare NDF_ statement functions
      INCLUDE 'NDF_FUNC_DEF'     ! Define NDF_ statement functions

*.

*  Check inherited global status.
      IF ( STATUS .NE. SAI__OK ) RETURN

*  Initialise.
      BAD = .FALSE.

*  There is nothing to do unless the mask is non-zero.
      IF ( BADBIT .NE. ZEROUB ) THEN

*  1 array to process:
*  ==================
*  Loop to process each array element.
         IF ( NARRAY .EQ. 1 ) THEN
            DO 1 I = EL, 1, -1

*  Evaluate the quality masking function.
               IF ( .NOT. NDF_QMASK( QUAL( I ), BADBIT ) ) THEN

*  Set affected array elements to the bad value and note this has
*  happened.
                  ARRAY1( I ) = VAL__BADR
                  BAD = .TRUE.
               END IF
1           CONTINUE

*  2 arrays to process:
*  ===================
*  Loop to process each array element.
         ELSE IF ( NARRAY .EQ. 2 ) THEN
            DO 2 I = EL, 1, -1

*  Evaluate the quality masking function.
               IF ( .NOT. NDF_QMASK( QUAL( I ), BADBIT ) ) THEN

*  Set affected array elements to the bad value and note this has
*  happened.
                  ARRAY1( I ) = VAL__BADR
                  ARRAY2( I ) = VAL__BADR
                  BAD = .TRUE.
               END IF
2           CONTINUE

*  3 arrays to process:
*  ===================
*  Loop to process each array element.
         ELSE IF ( NARRAY .EQ. 3 ) THEN
            DO 3 I = EL, 1, -1

*  Evaluate the quality masking function.
               IF ( .NOT. NDF_QMASK( QUAL( I ), BADBIT ) ) THEN

*  Set affected array elements to the bad value and note this has
*  happened.
                  ARRAY1( I ) = VAL__BADR
                  ARRAY2( I ) = VAL__BADR
                  ARRAY3( I ) = VAL__BADR
                  BAD = .TRUE.
               END IF
3           CONTINUE

*  4 arrays to process:
*  ===================
*  Loop to process each array element.
         ELSE IF ( NARRAY .EQ. 4 ) THEN
            DO 4 I = EL, 1, -1

*  Evaluate the quality masking function.
               IF ( .NOT. NDF_QMASK( QUAL( I ), BADBIT ) ) THEN

*  Set affected array elements to the bad value and note this has
*  happened.
                  ARRAY1( I ) = VAL__BADR
                  ARRAY2( I ) = VAL__BADR
                  ARRAY3( I ) = VAL__BADR
                  ARRAY4( I ) = VAL__BADR
                  BAD = .TRUE.
               END IF
4           CONTINUE
         END IF
      END IF
       
*  Call error tracing routine and exit.
      IF ( STATUS .NE. SAI__OK ) CALL NDF1_TRACE( 'NDF1_QMAR',
     :                                            STATUS )

      END
      SUBROUTINE NDF1_QMAW( EL, QUAL, BADBIT, NARRAY, ARRAY1, ARRAY2,
     :                        ARRAY3, ARRAY4, BAD, STATUS )
*+
*  Name:
*     NDF1_QMAW

*  Purpose:
*     Perform quality masking on vectorised WORD arrays.

*  Language:
*     Starlink Fortran 77

*  Invocation:
*     CALL NDF1_QMAW( EL, QUAL, BADBIT, NARRAY, ARRAY1, ARRAY2,
*                       ARRAY3, ARRAY4, BAD, STATUS )

*  Description:
*     The routine converts a vectorised quality array QUAL into a set
*     of "bad pixel" flags which are inserted into a matching set of up
*     to 4 vectorised WORD arrays.  This is done by performing a
*     bit-wise "AND" operation between elements of the QUAL array and
*     the bitmask BADBIT. Where the result of this operation is
*     non-zero, the corresponding elements of ARRAY1...ARRAY4 are set
*     to the "bad" value VAL__BADW (the precise number of arrays
*     processed is determined by the NARRAY argument).  Other array
*     elements are unchanged.  A logical value BAD is also returned
*     indicating whether any "bad" pixels were actually generated by
*     this quality masking process.

*  Arguments:
*     EL = INTEGER (Given)
*        The number of elements to process in each vectorised array.
*     QUAL( EL ) = BYTE (Given)
*        The quality array.
*     BADBIT = BYTE (Given)
*        The bad-bits mask to be applied to the quality array.
*     NARRAY = INTEGER (Given)
*        Number of vectorised arrays to be processed in the range 1 to
*        4. The routine returns without action if this value is out of
*        range.
*     ARRAY1...ARRAY4( EL ) = INTEGER*2 (Given and Returned)
*        The vectorised WORD arrays to be processed. Up to four of
*        these will be processed, according to the value of the NARRAY
*        argument.  Any remaining arrays are not used.
*     BAD = LOGICAL (Returned)
*        Whether any bad pixels were generated as a result of the
*        quality masking process.
*     STATUS = INTEGER (Given and Returned)
*        The global status.

*  Notes:
*     -  This routine loops through the arrays in a backward direction.
*     This is to minimise paging on a virtual memory machine, since
*     this routine will usually be followed by a processing loop which
*     passes forward through the same arrays.

*  Algorithm:
*     -  Initialise.
*     -  Check the badbits value is not zero (there is nothing to do if
*     it is).
*     -  If one array is being processed, then loop to process each
*     array element.
*     -  Evaluate the quality masking function.
*     -  Set affected array elements to the bad value and note this has
*     happened.
*     -  Repeat this process for each value of NARRAY, setting bad
*     pixels in the appropriate number of output arrays in each case.

*  Authors:
*     RFWS: R.F. Warren-Smith (STARLINK)
*     {enter_new_authors_here}

*  History:
*     21-NOV-1989 (RFWS):
*        Original version.
*     6-FEB-1990 (RFWS):
*        Extended to handle up to four arrays at once.
*     20-MAR-1990 (RFWS):
*        Changed to loop through the arrays backwards.
*     20-MAR-1990 (RFWS):
*        Changed to define the quality masking operation by means of a
*        statement function defined in an include file.
*     {enter_further_changes_here}

*  Bugs:
*     {note_any_bugs_here}

*-
      
*  Type Definitions:
      IMPLICIT NONE              ! No implicit typing

*  Global Constants:
      INCLUDE 'SAE_PAR'          ! Standard SAE constants
      INCLUDE 'DAT_PAR'          ! DAT_ public_constants
      INCLUDE 'PRM_PAR'          ! PRIMDAT primitive data constants

*  Arguments Given:
      INTEGER EL
      BYTE QUAL( EL )
      BYTE BADBIT
      INTEGER NARRAY

*  Arguments Given and Returned:
      INTEGER*2 ARRAY1( * )
      INTEGER*2 ARRAY2( * )
      INTEGER*2 ARRAY3( * )
      INTEGER*2 ARRAY4( * )

*  Arguments Returned:
      LOGICAL BAD

*  Status:
      INTEGER STATUS             ! Global status

*  Local Constants:
      BYTE ZEROUB                ! Zero as an unsigned byte value
      PARAMETER ( ZEROUB = 0 )

*  Local Variables:
      INTEGER I                  ! Loop counter for array elements

*  Internal References:
      INCLUDE 'NDF_FUNC_DEC'     ! Declare NDF_ statement functions
      INCLUDE 'NDF_FUNC_DEF'     ! Define NDF_ statement functions

*.

*  Check inherited global status.
      IF ( STATUS .NE. SAI__OK ) RETURN

*  Initialise.
      BAD = .FALSE.

*  There is nothing to do unless the mask is non-zero.
      IF ( BADBIT .NE. ZEROUB ) THEN

*  1 array to process:
*  ==================
*  Loop to process each array element.
         IF ( NARRAY .EQ. 1 ) THEN
            DO 1 I = EL, 1, -1

*  Evaluate the quality masking function.
               IF ( .NOT. NDF_QMASK( QUAL( I ), BADBIT ) ) THEN

*  Set affected array elements to the bad value and note this has
*  happened.
                  ARRAY1( I ) = VAL__BADW
                  BAD = .TRUE.
               END IF
1           CONTINUE

*  2 arrays to process:
*  ===================
*  Loop to process each array element.
         ELSE IF ( NARRAY .EQ. 2 ) THEN
            DO 2 I = EL, 1, -1

*  Evaluate the quality masking function.
               IF ( .NOT. NDF_QMASK( QUAL( I ), BADBIT ) ) THEN

*  Set affected array elements to the bad value and note this has
*  happened.
                  ARRAY1( I ) = VAL__BADW
                  ARRAY2( I ) = VAL__BADW
                  BAD = .TRUE.
               END IF
2           CONTINUE

*  3 arrays to process:
*  ===================
*  Loop to process each array element.
         ELSE IF ( NARRAY .EQ. 3 ) THEN
            DO 3 I = EL, 1, -1

*  Evaluate the quality masking function.
               IF ( .NOT. NDF_QMASK( QUAL( I ), BADBIT ) ) THEN

*  Set affected array elements to the bad value and note this has
*  happened.
                  ARRAY1( I ) = VAL__BADW
                  ARRAY2( I ) = VAL__BADW
                  ARRAY3( I ) = VAL__BADW
                  BAD = .TRUE.
               END IF
3           CONTINUE

*  4 arrays to process:
*  ===================
*  Loop to process each array element.
         ELSE IF ( NARRAY .EQ. 4 ) THEN
            DO 4 I = EL, 1, -1

*  Evaluate the quality masking function.
               IF ( .NOT. NDF_QMASK( QUAL( I ), BADBIT ) ) THEN

*  Set affected array elements to the bad value and note this has
*  happened.
                  ARRAY1( I ) = VAL__BADW
                  ARRAY2( I ) = VAL__BADW
                  ARRAY3( I ) = VAL__BADW
                  ARRAY4( I ) = VAL__BADW
                  BAD = .TRUE.
               END IF
4           CONTINUE
         END IF
      END IF
       
*  Call error tracing routine and exit.
      IF ( STATUS .NE. SAI__OK ) CALL NDF1_TRACE( 'NDF1_QMAW',
     :                                            STATUS )

      END
      SUBROUTINE NDF1_QMAUW( EL, QUAL, BADBIT, NARRAY, ARRAY1, ARRAY2,
     :                        ARRAY3, ARRAY4, BAD, STATUS )
*+
*  Name:
*     NDF1_QMAUW

*  Purpose:
*     Perform quality masking on vectorised UNSIGNED WORD arrays.

*  Language:
*     Starlink Fortran 77

*  Invocation:
*     CALL NDF1_QMAUW( EL, QUAL, BADBIT, NARRAY, ARRAY1, ARRAY2,
*                       ARRAY3, ARRAY4, BAD, STATUS )

*  Description:
*     The routine converts a vectorised quality array QUAL into a set
*     of "bad pixel" flags which are inserted into a matching set of up
*     to 4 vectorised UNSIGNED WORD arrays.  This is done by performing a
*     bit-wise "AND" operation between elements of the QUAL array and
*     the bitmask BADBIT. Where the result of this operation is
*     non-zero, the corresponding elements of ARRAY1...ARRAY4 are set
*     to the "bad" value VAL__BADUW (the precise number of arrays
*     processed is determined by the NARRAY argument).  Other array
*     elements are unchanged.  A logical value BAD is also returned
*     indicating whether any "bad" pixels were actually generated by
*     this quality masking process.

*  Arguments:
*     EL = INTEGER (Given)
*        The number of elements to process in each vectorised array.
*     QUAL( EL ) = BYTE (Given)
*        The quality array.
*     BADBIT = BYTE (Given)
*        The bad-bits mask to be applied to the quality array.
*     NARRAY = INTEGER (Given)
*        Number of vectorised arrays to be processed in the range 1 to
*        4. The routine returns without action if this value is out of
*        range.
*     ARRAY1...ARRAY4( EL ) = INTEGER*2 (Given and Returned)
*        The vectorised UNSIGNED WORD arrays to be processed. Up to four of
*        these will be processed, according to the value of the NARRAY
*        argument.  Any remaining arrays are not used.
*     BAD = LOGICAL (Returned)
*        Whether any bad pixels were generated as a result of the
*        quality masking process.
*     STATUS = INTEGER (Given and Returned)
*        The global status.

*  Notes:
*     -  This routine loops through the arrays in a backward direction.
*     This is to minimise paging on a virtual memory machine, since
*     this routine will usually be followed by a processing loop which
*     passes forward through the same arrays.

*  Algorithm:
*     -  Initialise.
*     -  Check the badbits value is not zero (there is nothing to do if
*     it is).
*     -  If one array is being processed, then loop to process each
*     array element.
*     -  Evaluate the quality masking function.
*     -  Set affected array elements to the bad value and note this has
*     happened.
*     -  Repeat this process for each value of NARRAY, setting bad
*     pixels in the appropriate number of output arrays in each case.

*  Authors:
*     RFWS: R.F. Warren-Smith (STARLINK)
*     {enter_new_authors_here}

*  History:
*     21-NOV-1989 (RFWS):
*        Original version.
*     6-FEB-1990 (RFWS):
*        Extended to handle up to four arrays at once.
*     20-MAR-1990 (RFWS):
*        Changed to loop through the arrays backwards.
*     20-MAR-1990 (RFWS):
*        Changed to define the quality masking operation by means of a
*        statement function defined in an include file.
*     {enter_further_changes_here}

*  Bugs:
*     {note_any_bugs_here}

*-
      
*  Type Definitions:
      IMPLICIT NONE              ! No implicit typing

*  Global Constants:
      INCLUDE 'SAE_PAR'          ! Standard SAE constants
      INCLUDE 'DAT_PAR'          ! DAT_ public_constants
      INCLUDE 'PRM_PAR'          ! PRIMDAT primitive data constants

*  Arguments Given:
      INTEGER EL
      BYTE QUAL( EL )
      BYTE BADBIT
      INTEGER NARRAY

*  Arguments Given and Returned:
      INTEGER*2 ARRAY1( * )
      INTEGER*2 ARRAY2( * )
      INTEGER*2 ARRAY3( * )
      INTEGER*2 ARRAY4( * )

*  Arguments Returned:
      LOGICAL BAD

*  Status:
      INTEGER STATUS             ! Global status

*  Local Constants:
      BYTE ZEROUB                ! Zero as an unsigned byte value
      PARAMETER ( ZEROUB = 0 )

*  Local Variables:
      INTEGER I                  ! Loop counter for array elements

*  Internal References:
      INCLUDE 'NDF_FUNC_DEC'     ! Declare NDF_ statement functions
      INCLUDE 'NDF_FUNC_DEF'     ! Define NDF_ statement functions

*.

*  Check inherited global status.
      IF ( STATUS .NE. SAI__OK ) RETURN

*  Initialise.
      BAD = .FALSE.

*  There is nothing to do unless the mask is non-zero.
      IF ( BADBIT .NE. ZEROUB ) THEN

*  1 array to process:
*  ==================
*  Loop to process each array element.
         IF ( NARRAY .EQ. 1 ) THEN
            DO 1 I = EL, 1, -1

*  Evaluate the quality masking function.
               IF ( .NOT. NDF_QMASK( QUAL( I ), BADBIT ) ) THEN

*  Set affected array elements to the bad value and note this has
*  happened.
                  ARRAY1( I ) = VAL__BADUW
                  BAD = .TRUE.
               END IF
1           CONTINUE

*  2 arrays to process:
*  ===================
*  Loop to process each array element.
         ELSE IF ( NARRAY .EQ. 2 ) THEN
            DO 2 I = EL, 1, -1

*  Evaluate the quality masking function.
               IF ( .NOT. NDF_QMASK( QUAL( I ), BADBIT ) ) THEN

*  Set affected array elements to the bad value and note this has
*  happened.
                  ARRAY1( I ) = VAL__BADUW
                  ARRAY2( I ) = VAL__BADUW
                  BAD = .TRUE.
               END IF
2           CONTINUE

*  3 arrays to process:
*  ===================
*  Loop to process each array element.
         ELSE IF ( NARRAY .EQ. 3 ) THEN
            DO 3 I = EL, 1, -1

*  Evaluate the quality masking function.
               IF ( .NOT. NDF_QMASK( QUAL( I ), BADBIT ) ) THEN

*  Set affected array elements to the bad value and note this has
*  happened.
                  ARRAY1( I ) = VAL__BADUW
                  ARRAY2( I ) = VAL__BADUW
                  ARRAY3( I ) = VAL__BADUW
                  BAD = .TRUE.
               END IF
3           CONTINUE

*  4 arrays to process:
*  ===================
*  Loop to process each array element.
         ELSE IF ( NARRAY .EQ. 4 ) THEN
            DO 4 I = EL, 1, -1

*  Evaluate the quality masking function.
               IF ( .NOT. NDF_QMASK( QUAL( I ), BADBIT ) ) THEN

*  Set affected array elements to the bad value and note this has
*  happened.
                  ARRAY1( I ) = VAL__BADUW
                  ARRAY2( I ) = VAL__BADUW
                  ARRAY3( I ) = VAL__BADUW
                  ARRAY4( I ) = VAL__BADUW
                  BAD = .TRUE.
               END IF
4           CONTINUE
         END IF
      END IF
       
*  Call error tracing routine and exit.
      IF ( STATUS .NE. SAI__OK ) CALL NDF1_TRACE( 'NDF1_QMAUW',
     :                                            STATUS )

      END
