# Source Map Semantics

Source Maps are a specification for helping users how their original maps onto a single file.
While the mapping specifies to point-to-point relationships,
it is up to source map generators to decides the appropriate relationships in the code.

This document attempts to outline some of the existing mappings we have observed
as well as articulate some opinions on the appropriate mappings.

### Calls

The call maps the identifier and includes the opening original paren.

![](https://shipusercontent.com/554e9a9d7b74a52465156d80391625f8/Screen%20Shot%202018-03-28%20at%2010.13.15%20PM.png)

The object property of a member expression call maps the identifier, period, and generated parens.

![](https://shipusercontent.com/17b1f5147336473f9d5070d5d7b82094/Screen%20Shot%202018-03-28%20at%2010.13.11%20PM.png)

The arguments of a call attempt to map one to one. The last argument of a call will include the closing original paren.

![](https://shipusercontent.com/33ff9ea55c3cd35d55b70e64cf9326e9/Screen%20Shot%202018-03-28%20at%2010.13.01%20PM.png)

Complex arguments are also mapped one to one. In this case, the argument itself has a call that is included in the mapping.

![](https://shipusercontent.com/20f575759f31c01d13243e83a253c9ba/Screen%20Shot%202018-03-28%20at%2010.16.20%20PM.png)


### Dependencies

When Babel is involved, most dependencies are mapped line by line, but namespaces are broken up.

example: ![imported-bindings]

![](https://shipusercontent.com/c0939aecadad12902646bd687b05840f/Screen%20Shot%202018-03-28%20at%2010.00.30%20PM.png)

![](https://shipusercontent.com/fadbe965f836d304e84989f89529276e/Screen%20Shot%202018-03-28%20at%2010.07.57%20PM.png)
![](https://shipusercontent.com/cc5ff07e169e2583e9e4e8859b912810/Screen%20Shot%202018-03-28%20at%2010.07.52%20PM.png)

When Webpack is left to its own devices then dependencies are not mapped at all. In these cases, the original dependencies all map to the generated location `0,0`, which is equivalent to null.

- example 1: ![webpack-es-modules]  
- example 2: ![webpack-modules]

![](https://shipusercontent.com/cf65771d652824679cfe0ed8dab13f8b/Screen%20Shot%202018-03-28%20at%2010.09.36%20PM.png)

[webpack-es-modules]: https://sokra.github.io/source-map-visualization/#base64,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,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,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,aW1wb3J0IGFEZWZhdWx0IGZyb20gIi4vc3JjL21vZDEiOwppbXBvcnQgeyBhTmFtZWQgfSBmcm9tICIuL3NyYy9tb2QyIjsKaW1wb3J0IHsgb3JpZ2luYWwgYXMgYW5BbGlhc2VkIH0gZnJvbSAiLi9zcmMvbW9kMyI7CmltcG9ydCAqIGFzIGFOYW1lc3BhY2UgZnJvbSAiLi9zcmMvbW9kNCI7CgppbXBvcnQgYURlZmF1bHQyIGZyb20gIi4vc3JjL21vZDUiOwppbXBvcnQgeyBhTmFtZWQyIH0gZnJvbSAiLi9zcmMvbW9kNiI7CmltcG9ydCB7IG9yaWdpbmFsIGFzIGFuQWxpYXNlZDIgfSBmcm9tICIuL3NyYy9tb2Q3IjsKaW1wb3J0ICogYXMgYU5hbWVzcGFjZTIgZnJvbSAiLi9zcmMvbW9kOCI7CgppbXBvcnQgYURlZmF1bHQzIGZyb20gIi4vc3JjL21vZDkiOwppbXBvcnQgeyBhTmFtZWQzIH0gZnJvbSAiLi9zcmMvbW9kMTAiOwppbXBvcnQgeyBvcmlnaW5hbCBhcyBhbkFsaWFzZWQzIH0gZnJvbSAiLi9zcmMvbW9kMTEiOwppbXBvcnQgKiBhcyBhTmFtZXNwYWNlMyBmcm9tICIuL3NyYy9tb2QxMiI7CgppbXBvcnQgb3B0aW1pemVkT3V0IGZyb20gIi4vc3JjL29wdGltaXplZC1vdXQiOwpvcHRpbWl6ZWRPdXQoKTsKCmV4cG9ydCBkZWZhdWx0IGZ1bmN0aW9uIHJvb3QoKSB7CiAgY29uc29sZS5sb2coInBhdXNlIGhlcmUiLCByb290KTsKCiAgY29uc29sZS5sb2coYURlZmF1bHQpOwogIGNvbnNvbGUubG9nKGFuQWxpYXNlZCk7CiAgY29uc29sZS5sb2coYU5hbWVkKTsKICBjb25zb2xlLmxvZyhhbkFsaWFzZWQpOwogIGNvbnNvbGUubG9nKGFOYW1lc3BhY2UpOwoKICB0cnkgewogICAgLy8gTm9uZSBvZiB0aGVzZSBhcmUgY2FsbGFibGUgaW4gdGhpcyBjb2RlLCBidXQgd2Ugc3RpbGwgd2FudCB0byBtYWtlIHN1cmUKICAgIC8vIHRoZXkgbWFwIHByb3Blcmx5IGV2ZW4gaWYgdGhlIG9ubHkgcmVmZXJlbmNlIGlzIGluIGEgY2FsbCBleHByZXNzaW9ucy4KICAgIGNvbnNvbGUubG9nKGFEZWZhdWx0MigpKTsKICAgIGNvbnNvbGUubG9nKGFuQWxpYXNlZDIoKSk7CiAgICBjb25zb2xlLmxvZyhhTmFtZWQyKCkpOwogICAgY29uc29sZS5sb2coYW5BbGlhc2VkMigpKTsKICAgIGNvbnNvbGUubG9nKGFOYW1lc3BhY2UyKCkpOwoKICAgIGNvbnNvbGUubG9nKG5ldyBhRGVmYXVsdDMoKSk7CiAgICBjb25zb2xlLmxvZyhuZXcgYW5BbGlhc2VkMygpKTsKICAgIGNvbnNvbGUubG9nKG5ldyBhTmFtZWQzKCkpOwogICAgY29uc29sZS5sb2cobmV3IGFuQWxpYXNlZDMoKSk7CiAgICBjb25zb2xlLmxvZyhuZXcgYU5hbWVzcGFjZTMoKSk7CiAgfSBjYXRjaCAoZSkge30KfQoKLy8gVGhlIGJ1aWxkIGhhcm5lc3Mgc2V0cyB0aGUgd3JvbmcgZ2xvYmFsLCBzbyBqdXN0IG92ZXJyaWRlIGl0LgpQcm9taXNlLnJlc29sdmUoKS50aGVuKCgpID0+IHsKICB3aW5kb3cud2VicGFja01vZHVsZXNFczYgPSByb290Owp9KTsKCgoKLy8gV0VCUEFDSyBGT09URVIgLy8KLy8gLi9maXh0dXJlcy93ZWJwYWNrLW1vZHVsZXMtZXM2L2lucHV0Lmpz,ZXhwb3J0IGRlZmF1bHQgImEtZGVmYXVsdCI7CgoKCi8vIFdFQlBBQ0sgRk9PVEVSIC8vCi8vIC4vZml4dHVyZXMvd2VicGFjay1tb2R1bGVzLWVzNi9zcmMvbW9kMS5qcw==,ZXhwb3J0IGNvbnN0IGFOYW1lZCA9ICJhLW5hbWVkIjsKCgoKLy8gV0VCUEFDSyBGT09URVIgLy8KLy8gLi9maXh0dXJlcy93ZWJwYWNrLW1vZHVsZXMtZXM2L3NyYy9tb2QyLmpz,ZXhwb3J0IGNvbnN0IG9yaWdpbmFsID0gImFuLW9yaWdpbmFsIjsKCgoKLy8gV0VCUEFDSyBGT09URVIgLy8KLy8gLi9maXh0dXJlcy93ZWJwYWNrLW1vZHVsZXMtZXM2L3NyYy9tb2QzLmpz,ZXhwb3J0IGRlZmF1bHQgImEtZGVmYXVsdCI7CmV4cG9ydCBjb25zdCBhTmFtZWQgPSAiYS1uYW1lZCI7CgoKCi8vIFdFQlBBQ0sgRk9PVEVSIC8vCi8vIC4vZml4dHVyZXMvd2VicGFjay1tb2R1bGVzLWVzNi9zcmMvbW9kNC5qcw==,ZXhwb3J0IGRlZmF1bHQgImEtZGVmYXVsdDIiOwoKCgovLyBXRUJQQUNLIEZPT1RFUiAvLwovLyAuL2ZpeHR1cmVzL3dlYnBhY2stbW9kdWxlcy1lczYvc3JjL21vZDUuanM=,ZXhwb3J0IGNvbnN0IGFOYW1lZDIgPSAiYS1uYW1lZDIiOwoKCgovLyBXRUJQQUNLIEZPT1RFUiAvLwovLyAuL2ZpeHR1cmVzL3dlYnBhY2stbW9kdWxlcy1lczYvc3JjL21vZDYuanM=,ZXhwb3J0IGNvbnN0IG9yaWdpbmFsID0gImFuLW9yaWdpbmFsMiI7CgoKCi8vIFdFQlBBQ0sgRk9PVEVSIC8vCi8vIC4vZml4dHVyZXMvd2VicGFjay1tb2R1bGVzLWVzNi9zcmMvbW9kNy5qcw==,ZXhwb3J0IGRlZmF1bHQgImEtZGVmYXVsdDIiOwpleHBvcnQgY29uc3QgYU5hbWVkID0gImEtbmFtZWQyIjsKCgoKLy8gV0VCUEFDSyBGT09URVIgLy8KLy8gLi9maXh0dXJlcy93ZWJwYWNrLW1vZHVsZXMtZXM2L3NyYy9tb2Q4Lmpz,ZXhwb3J0IGRlZmF1bHQgImEtZGVmYXVsdDMiOwoKCgovLyBXRUJQQUNLIEZPT1RFUiAvLwovLyAuL2ZpeHR1cmVzL3dlYnBhY2stbW9kdWxlcy1lczYvc3JjL21vZDkuanM=,ZXhwb3J0IGNvbnN0IGFOYW1lZDMgPSAiYS1uYW1lZDMiOwoKCgovLyBXRUJQQUNLIEZPT1RFUiAvLwovLyAuL2ZpeHR1cmVzL3dlYnBhY2stbW9kdWxlcy1lczYvc3JjL21vZDEwLmpz,ZXhwb3J0IGNvbnN0IG9yaWdpbmFsID0gImFuLW9yaWdpbmFsMyI7CgoKCi8vIFdFQlBBQ0sgRk9PVEVSIC8vCi8vIC4vZml4dHVyZXMvd2VicGFjay1tb2R1bGVzLWVzNi9zcmMvbW9kMTEuanM=,ZXhwb3J0IGRlZmF1bHQgImEtZGVmYXVsdDMiOwpleHBvcnQgY29uc3QgYU5hbWVkID0gImEtbmFtZWQzIjsKCgoKLy8gV0VCUEFDSyBGT09URVIgLy8KLy8gLi9maXh0dXJlcy93ZWJwYWNrLW1vZHVsZXMtZXM2L3NyYy9tb2QxMi5qcw==,ZXhwb3J0IGRlZmF1bHQgZnVuY3Rpb24gb3B0aW1pemVkT3V0KCkge30KCgoKLy8gV0VCUEFDSyBGT09URVIgLy8KLy8gLi9maXh0dXJlcy93ZWJwYWNrLW1vZHVsZXMtZXM2L3NyYy9vcHRpbWl6ZWQtb3V0Lmpz
[imported-bindings]: https://sokra.github.io/source-map-visualization/#base64,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,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly8vd2VicGFjay9ib290c3RyYXAgZTAxODBjNzZhNDFiZGQ3YThlYWUiLCJ3ZWJwYWNrOi8vLy4vZml4dHVyZXMvaW1wb3J0ZWQtYmluZGluZ3MvaW5wdXQuanMiLCJ3ZWJwYWNrOi8vLy4vZml4dHVyZXMvaW1wb3J0ZWQtYmluZGluZ3Mvc3JjL21vZDEuanMiLCJ3ZWJwYWNrOi8vLy4vZml4dHVyZXMvaW1wb3J0ZWQtYmluZGluZ3Mvc3JjL21vZDIuanMiLCJ3ZWJwYWNrOi8vLy4vZml4dHVyZXMvaW1wb3J0ZWQtYmluZGluZ3Mvc3JjL21vZDMuanMiLCJ3ZWJwYWNrOi8vLy4vZml4dHVyZXMvaW1wb3J0ZWQtYmluZGluZ3Mvc3JjL21vZDQuanMiLCJ3ZWJwYWNrOi8vLy4vZml4dHVyZXMvaW1wb3J0ZWQtYmluZGluZ3Mvc3JjL21vZDUuanMiLCJ3ZWJwYWNrOi8vLy4vZml4dHVyZXMvaW1wb3J0ZWQtYmluZGluZ3Mvc3JjL21vZDYuanMiLCJ3ZWJwYWNrOi8vLy4vZml4dHVyZXMvaW1wb3J0ZWQtYmluZGluZ3Mvc3JjL21vZDcuanMiLCJ3ZWJwYWNrOi8vLy4vZml4dHVyZXMvaW1wb3J0ZWQtYmluZGluZ3Mvc3JjL21vZDguanMiLCJ3ZWJwYWNrOi8vLy4vZml4dHVyZXMvaW1wb3J0ZWQtYmluZGluZ3Mvc3JjL21vZDkuanMiLCJ3ZWJwYWNrOi8vLy4vZml4dHVyZXMvaW1wb3J0ZWQtYmluZGluZ3Mvc3JjL21vZDEwLmpzIiwid2VicGFjazovLy8uL2ZpeHR1cmVzL2ltcG9ydGVkLWJpbmRpbmdzL3NyYy9tb2QxMS5qcyIsIndlYnBhY2s6Ly8vLi9maXh0dXJlcy9pbXBvcnRlZC1iaW5kaW5ncy9zcmMvbW9kMTIuanMiLCJ3ZWJwYWNrOi8vLy4vZml4dHVyZXMvaW1wb3J0ZWQtYmluZGluZ3Mvc3JjL29wdGltaXplZC1vdXQuanMiXSwibmFtZXMiOlsicm9vdCIsImFOYW1lc3BhY2UiLCJhTmFtZXNwYWNlMiIsImFOYW1lc3BhY2UzIiwiY29uc29sZSIsImxvZyIsImUiLCJhTmFtZWQiLCJvcmlnaW5hbCIsImFOYW1lZDIiLCJhTmFtZWQzIiwib3B0aW1pemVkT3V0Il0sIm1hcHBpbmdzIjoiOztBQUFBO0FBQ0E7O0FBRUE7QUFDQTs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7QUFFQTtBQUNBOztBQUVBO0FBQ0E7O0FBRUE7QUFDQTtBQUNBOzs7QUFHQTtBQUNBOztBQUVBO0FBQ0E7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSxhQUFLO0FBQ0w7QUFDQTs7QUFFQTtBQUNBO0FBQ0E7QUFDQSxtQ0FBMkIsMEJBQTBCLEVBQUU7QUFDdkQseUNBQWlDLGVBQWU7QUFDaEQ7QUFDQTtBQUNBOztBQUVBO0FBQ0EsOERBQXNELCtEQUErRDs7QUFFckg7QUFDQTs7QUFFQTtBQUNBOzs7Ozs7Ozs7Ozs7O2tCQzNDd0JBLEk7O0FBbEJ4Qjs7OztBQUNBOztBQUNBOztBQUNBOztJQUFZQyxVOztBQUVaOzs7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0lBQVlDLFc7O0FBRVo7Ozs7QUFDQTs7QUFDQTs7QUFDQTs7SUFBWUMsVzs7QUFFWjs7Ozs7Ozs7QUFDQTs7QUFFZSxTQUFTSCxJQUFULEdBQWdCO0FBQzdCSSxVQUFRQyxHQUFSLENBQVksWUFBWixFQUEwQkwsSUFBMUI7O0FBRUFJLFVBQVFDLEdBQVI7QUFDQUQsVUFBUUMsR0FBUjtBQUNBRCxVQUFRQyxHQUFSO0FBQ0FELFVBQVFDLEdBQVI7QUFDQUQsVUFBUUMsR0FBUixDQUFZSixVQUFaOztBQUVBLE1BQUk7QUFDRjtBQUNBO0FBQ0FHLFlBQVFDLEdBQVIsQ0FBWSxvQkFBWjtBQUNBRCxZQUFRQyxHQUFSLENBQVkscUJBQVo7QUFDQUQsWUFBUUMsR0FBUixDQUFZLG9CQUFaO0FBQ0FELFlBQVFDLEdBQVIsQ0FBWSxxQkFBWjtBQUNBRCxZQUFRQyxHQUFSLENBQVlILGFBQVo7O0FBRUFFLFlBQVFDLEdBQVIsQ0FBWSxvQkFBWjtBQUNBRCxZQUFRQyxHQUFSLENBQVkscUJBQVo7QUFDQUQsWUFBUUMsR0FBUixDQUFZLG9CQUFaO0FBQ0FELFlBQVFDLEdBQVIsQ0FBWSxxQkFBWjtBQUNBRCxZQUFRQyxHQUFSLENBQVksSUFBSUYsV0FBSixFQUFaO0FBQ0QsR0FkRCxDQWNFLE9BQU9HLENBQVAsRUFBVSxDQUFFO0FBQ2Y7Ozs7Ozs7Ozs7Ozs7a0JDMUNjLFc7Ozs7Ozs7Ozs7Ozs7QUNBUixJQUFNQywwQkFBUyxTQUFmLEM7Ozs7Ozs7Ozs7OztBQ0FBLElBQU1DLDhCQUFXLGFBQWpCLEM7Ozs7Ozs7Ozs7OztrQkNBUSxXO0FBQ1IsSUFBTUQsMEJBQVMsU0FBZixDOzs7Ozs7Ozs7Ozs7a0JDRFEsWTs7Ozs7Ozs7Ozs7OztBQ0FSLElBQU1FLDRCQUFVLFVBQWhCLEM7Ozs7Ozs7Ozs7OztBQ0FBLElBQU1ELDhCQUFXLGNBQWpCLEM7Ozs7Ozs7Ozs7OztrQkNBUSxZO0FBQ1IsSUFBTUQsMEJBQVMsVUFBZixDOzs7Ozs7Ozs7Ozs7a0JDRFEsWTs7Ozs7Ozs7Ozs7OztBQ0FSLElBQU1HLDRCQUFVLFVBQWhCLEM7Ozs7Ozs7Ozs7OztBQ0FBLElBQU1GLDhCQUFXLGNBQWpCLEM7Ozs7Ozs7Ozs7OztrQkNBUSxZO0FBQ1IsSUFBTUQsMEJBQVMsVUFBZixDOzs7Ozs7Ozs7Ozs7a0JDRGlCSSxZO0FBQVQsU0FBU0EsWUFBVCxHQUF3QixDQUFFIiwiZmlsZSI6ImZpeHR1cmVzL2ltcG9ydGVkLWJpbmRpbmdzL291dHB1dC5qcyIsInNvdXJjZXNDb250ZW50IjpbIiBcdC8vIFRoZSBtb2R1bGUgY2FjaGVcbiBcdHZhciBpbnN0YWxsZWRNb2R1bGVzID0ge307XG5cbiBcdC8vIFRoZSByZXF1aXJlIGZ1bmN0aW9uXG4gXHRmdW5jdGlvbiBfX3dlYnBhY2tfcmVxdWlyZV9fKG1vZHVsZUlkKSB7XG5cbiBcdFx0Ly8gQ2hlY2sgaWYgbW9kdWxlIGlzIGluIGNhY2hlXG4gXHRcdGlmKGluc3RhbGxlZE1vZHVsZXNbbW9kdWxlSWRdKSB7XG4gXHRcdFx0cmV0dXJuIGluc3RhbGxlZE1vZHVsZXNbbW9kdWxlSWRdLmV4cG9ydHM7XG4gXHRcdH1cbiBcdFx0Ly8gQ3JlYXRlIGEgbmV3IG1vZHVsZSAoYW5kIHB1dCBpdCBpbnRvIHRoZSBjYWNoZSlcbiBcdFx0dmFyIG1vZHVsZSA9IGluc3RhbGxlZE1vZHVsZXNbbW9kdWxlSWRdID0ge1xuIFx0XHRcdGk6IG1vZHVsZUlkLFxuIFx0XHRcdGw6IGZhbHNlLFxuIFx0XHRcdGV4cG9ydHM6IHt9XG4gXHRcdH07XG5cbiBcdFx0Ly8gRXhlY3V0ZSB0aGUgbW9kdWxlIGZ1bmN0aW9uXG4gXHRcdG1vZHVsZXNbbW9kdWxlSWRdLmNhbGwobW9kdWxlLmV4cG9ydHMsIG1vZHVsZSwgbW9kdWxlLmV4cG9ydHMsIF9fd2VicGFja19yZXF1aXJlX18pO1xuXG4gXHRcdC8vIEZsYWcgdGhlIG1vZHVsZSBhcyBsb2FkZWRcbiBcdFx0bW9kdWxlLmwgPSB0cnVlO1xuXG4gXHRcdC8vIFJldHVybiB0aGUgZXhwb3J0cyBvZiB0aGUgbW9kdWxlXG4gXHRcdHJldHVybiBtb2R1bGUuZXhwb3J0cztcbiBcdH1cblxuXG4gXHQvLyBleHBvc2UgdGhlIG1vZHVsZXMgb2JqZWN0IChfX3dlYnBhY2tfbW9kdWxlc19fKVxuIFx0X193ZWJwYWNrX3JlcXVpcmVfXy5tID0gbW9kdWxlcztcblxuIFx0Ly8gZXhwb3NlIHRoZSBtb2R1bGUgY2FjaGVcbiBcdF9fd2VicGFja19yZXF1aXJlX18uYyA9IGluc3RhbGxlZE1vZHVsZXM7XG5cbiBcdC8vIGRlZmluZSBnZXR0ZXIgZnVuY3Rpb24gZm9yIGhhcm1vbnkgZXhwb3J0c1xuIFx0X193ZWJwYWNrX3JlcXVpcmVfXy5kID0gZnVuY3Rpb24oZXhwb3J0cywgbmFtZSwgZ2V0dGVyKSB7XG4gXHRcdGlmKCFfX3dlYnBhY2tfcmVxdWlyZV9fLm8oZXhwb3J0cywgbmFtZSkpIHtcbiBcdFx0XHRPYmplY3QuZGVmaW5lUHJvcGVydHkoZXhwb3J0cywgbmFtZSwge1xuIFx0XHRcdFx0Y29uZmlndXJhYmxlOiBmYWxzZSxcbiBcdFx0XHRcdGVudW1lcmFibGU6IHRydWUsXG4gXHRcdFx0XHRnZXQ6IGdldHRlclxuIFx0XHRcdH0pO1xuIFx0XHR9XG4gXHR9O1xuXG4gXHQvLyBnZXREZWZhdWx0RXhwb3J0IGZ1bmN0aW9uIGZvciBjb21wYXRpYmlsaXR5IHdpdGggbm9uLWhhcm1vbnkgbW9kdWxlc1xuIFx0X193ZWJwYWNrX3JlcXVpcmVfXy5uID0gZnVuY3Rpb24obW9kdWxlKSB7XG4gXHRcdHZhciBnZXR0ZXIgPSBtb2R1bGUgJiYgbW9kdWxlLl9fZXNNb2R1bGUgP1xuIFx0XHRcdGZ1bmN0aW9uIGdldERlZmF1bHQoKSB7IHJldHVybiBtb2R1bGVbJ2RlZmF1bHQnXTsgfSA6XG4gXHRcdFx0ZnVuY3Rpb24gZ2V0TW9kdWxlRXhwb3J0cygpIHsgcmV0dXJuIG1vZHVsZTsgfTtcbiBcdFx0X193ZWJwYWNrX3JlcXVpcmVfXy5kKGdldHRlciwgJ2EnLCBnZXR0ZXIpO1xuIFx0XHRyZXR1cm4gZ2V0dGVyO1xuIFx0fTtcblxuIFx0Ly8gT2JqZWN0LnByb3RvdHlwZS5oYXNPd25Qcm9wZXJ0eS5jYWxsXG4gXHRfX3dlYnBhY2tfcmVxdWlyZV9fLm8gPSBmdW5jdGlvbihvYmplY3QsIHByb3BlcnR5KSB7IHJldHVybiBPYmplY3QucHJvdG90eXBlLmhhc093blByb3BlcnR5LmNhbGwob2JqZWN0LCBwcm9wZXJ0eSk7IH07XG5cbiBcdC8vIF9fd2VicGFja19wdWJsaWNfcGF0aF9fXG4gXHRfX3dlYnBhY2tfcmVxdWlyZV9fLnAgPSBcIlwiO1xuXG4gXHQvLyBMb2FkIGVudHJ5IG1vZHVsZSBhbmQgcmV0dXJuIGV4cG9ydHNcbiBcdHJldHVybiBfX3dlYnBhY2tfcmVxdWlyZV9fKF9fd2VicGFja19yZXF1aXJlX18ucyA9IDApO1xuXG5cblxuLy8gV0VCUEFDSyBGT09URVIgLy9cbi8vIHdlYnBhY2svYm9vdHN0cmFwIGUwMTgwYzc2YTQxYmRkN2E4ZWFlIiwiaW1wb3J0IGFEZWZhdWx0IGZyb20gXCIuL3NyYy9tb2QxXCI7XG5pbXBvcnQgeyBhTmFtZWQgfSBmcm9tIFwiLi9zcmMvbW9kMlwiO1xuaW1wb3J0IHsgb3JpZ2luYWwgYXMgYW5BbGlhc2VkIH0gZnJvbSBcIi4vc3JjL21vZDNcIjtcbmltcG9ydCAqIGFzIGFOYW1lc3BhY2UgZnJvbSBcIi4vc3JjL21vZDRcIjtcblxuaW1wb3J0IGFEZWZhdWx0MiBmcm9tIFwiLi9zcmMvbW9kNVwiO1xuaW1wb3J0IHsgYU5hbWVkMiB9IGZyb20gXCIuL3NyYy9tb2Q2XCI7XG5pbXBvcnQgeyBvcmlnaW5hbCBhcyBhbkFsaWFzZWQyIH0gZnJvbSBcIi4vc3JjL21vZDdcIjtcbmltcG9ydCAqIGFzIGFOYW1lc3BhY2UyIGZyb20gXCIuL3NyYy9tb2Q4XCI7XG5cbmltcG9ydCBhRGVmYXVsdDMgZnJvbSBcIi4vc3JjL21vZDlcIjtcbmltcG9ydCB7IGFOYW1lZDMgfSBmcm9tIFwiLi9zcmMvbW9kMTBcIjtcbmltcG9ydCB7IG9yaWdpbmFsIGFzIGFuQWxpYXNlZDMgfSBmcm9tIFwiLi9zcmMvbW9kMTFcIjtcbmltcG9ydCAqIGFzIGFOYW1lc3BhY2UzIGZyb20gXCIuL3NyYy9tb2QxMlwiO1xuXG5pbXBvcnQgb3B0aW1pemVkT3V0IGZyb20gXCIuL3NyYy9vcHRpbWl6ZWQtb3V0XCI7XG5vcHRpbWl6ZWRPdXQoKTtcblxuZXhwb3J0IGRlZmF1bHQgZnVuY3Rpb24gcm9vdCgpIHtcbiAgY29uc29sZS5sb2coXCJwYXVzZSBoZXJlXCIsIHJvb3QpO1xuXG4gIGNvbnNvbGUubG9nKGFEZWZhdWx0KTtcbiAgY29uc29sZS5sb2coYW5BbGlhc2VkKTtcbiAgY29uc29sZS5sb2coYU5hbWVkKTtcbiAgY29uc29sZS5sb2coYW5BbGlhc2VkKTtcbiAgY29uc29sZS5sb2coYU5hbWVzcGFjZSk7XG5cbiAgdHJ5IHtcbiAgICAvLyBOb25lIG9mIHRoZXNlIGFyZSBjYWxsYWJsZSBpbiB0aGlzIGNvZGUsIGJ1dCB3ZSBzdGlsbCB3YW50IHRvIG1ha2Ugc3VyZVxuICAgIC8vIHRoZXkgbWFwIHByb3Blcmx5IGV2ZW4gaWYgdGhlIG9ubHkgcmVmZXJlbmNlIGlzIGluIGEgY2FsbCBleHByZXNzaW9ucy5cbiAgICBjb25zb2xlLmxvZyhhRGVmYXVsdDIoKSk7XG4gICAgY29uc29sZS5sb2coYW5BbGlhc2VkMigpKTtcbiAgICBjb25zb2xlLmxvZyhhTmFtZWQyKCkpO1xuICAgIGNvbnNvbGUubG9nKGFuQWxpYXNlZDIoKSk7XG4gICAgY29uc29sZS5sb2coYU5hbWVzcGFjZTIoKSk7XG5cbiAgICBjb25zb2xlLmxvZyhuZXcgYURlZmF1bHQzKCkpO1xuICAgIGNvbnNvbGUubG9nKG5ldyBhbkFsaWFzZWQzKCkpO1xuICAgIGNvbnNvbGUubG9nKG5ldyBhTmFtZWQzKCkpO1xuICAgIGNvbnNvbGUubG9nKG5ldyBhbkFsaWFzZWQzKCkpO1xuICAgIGNvbnNvbGUubG9nKG5ldyBhTmFtZXNwYWNlMygpKTtcbiAgfSBjYXRjaCAoZSkge31cbn1cblxuXG5cbi8vIFdFQlBBQ0sgRk9PVEVSIC8vXG4vLyAuL2ZpeHR1cmVzL2ltcG9ydGVkLWJpbmRpbmdzL2lucHV0LmpzIiwiZXhwb3J0IGRlZmF1bHQgXCJhLWRlZmF1bHRcIjtcblxuXG5cbi8vIFdFQlBBQ0sgRk9PVEVSIC8vXG4vLyAuL2ZpeHR1cmVzL2ltcG9ydGVkLWJpbmRpbmdzL3NyYy9tb2QxLmpzIiwiZXhwb3J0IGNvbnN0IGFOYW1lZCA9IFwiYS1uYW1lZFwiO1xuXG5cblxuLy8gV0VCUEFDSyBGT09URVIgLy9cbi8vIC4vZml4dHVyZXMvaW1wb3J0ZWQtYmluZGluZ3Mvc3JjL21vZDIuanMiLCJleHBvcnQgY29uc3Qgb3JpZ2luYWwgPSBcImFuLW9yaWdpbmFsXCI7XG5cblxuXG4vLyBXRUJQQUNLIEZPT1RFUiAvL1xuLy8gLi9maXh0dXJlcy9pbXBvcnRlZC1iaW5kaW5ncy9zcmMvbW9kMy5qcyIsImV4cG9ydCBkZWZhdWx0IFwiYS1kZWZhdWx0XCI7XG5leHBvcnQgY29uc3QgYU5hbWVkID0gXCJhLW5hbWVkXCI7XG5cblxuXG4vLyBXRUJQQUNLIEZPT1RFUiAvL1xuLy8gLi9maXh0dXJlcy9pbXBvcnRlZC1iaW5kaW5ncy9zcmMvbW9kNC5qcyIsImV4cG9ydCBkZWZhdWx0IFwiYS1kZWZhdWx0MlwiO1xuXG5cblxuLy8gV0VCUEFDSyBGT09URVIgLy9cbi8vIC4vZml4dHVyZXMvaW1wb3J0ZWQtYmluZGluZ3Mvc3JjL21vZDUuanMiLCJleHBvcnQgY29uc3QgYU5hbWVkMiA9IFwiYS1uYW1lZDJcIjtcblxuXG5cbi8vIFdFQlBBQ0sgRk9PVEVSIC8vXG4vLyAuL2ZpeHR1cmVzL2ltcG9ydGVkLWJpbmRpbmdzL3NyYy9tb2Q2LmpzIiwiZXhwb3J0IGNvbnN0IG9yaWdpbmFsID0gXCJhbi1vcmlnaW5hbDJcIjtcblxuXG5cbi8vIFdFQlBBQ0sgRk9PVEVSIC8vXG4vLyAuL2ZpeHR1cmVzL2ltcG9ydGVkLWJpbmRpbmdzL3NyYy9tb2Q3LmpzIiwiZXhwb3J0IGRlZmF1bHQgXCJhLWRlZmF1bHQyXCI7XG5leHBvcnQgY29uc3QgYU5hbWVkID0gXCJhLW5hbWVkMlwiO1xuXG5cblxuLy8gV0VCUEFDSyBGT09URVIgLy9cbi8vIC4vZml4dHVyZXMvaW1wb3J0ZWQtYmluZGluZ3Mvc3JjL21vZDguanMiLCJleHBvcnQgZGVmYXVsdCBcImEtZGVmYXVsdDNcIjtcblxuXG5cbi8vIFdFQlBBQ0sgRk9PVEVSIC8vXG4vLyAuL2ZpeHR1cmVzL2ltcG9ydGVkLWJpbmRpbmdzL3NyYy9tb2Q5LmpzIiwiZXhwb3J0IGNvbnN0IGFOYW1lZDMgPSBcImEtbmFtZWQzXCI7XG5cblxuXG4vLyBXRUJQQUNLIEZPT1RFUiAvL1xuLy8gLi9maXh0dXJlcy9pbXBvcnRlZC1iaW5kaW5ncy9zcmMvbW9kMTAuanMiLCJleHBvcnQgY29uc3Qgb3JpZ2luYWwgPSBcImFuLW9yaWdpbmFsM1wiO1xuXG5cblxuLy8gV0VCUEFDSyBGT09URVIgLy9cbi8vIC4vZml4dHVyZXMvaW1wb3J0ZWQtYmluZGluZ3Mvc3JjL21vZDExLmpzIiwiZXhwb3J0IGRlZmF1bHQgXCJhLWRlZmF1bHQzXCI7XG5leHBvcnQgY29uc3QgYU5hbWVkID0gXCJhLW5hbWVkM1wiO1xuXG5cblxuLy8gV0VCUEFDSyBGT09URVIgLy9cbi8vIC4vZml4dHVyZXMvaW1wb3J0ZWQtYmluZGluZ3Mvc3JjL21vZDEyLmpzIiwiZXhwb3J0IGRlZmF1bHQgZnVuY3Rpb24gb3B0aW1pemVkT3V0KCkge31cblxuXG5cbi8vIFdFQlBBQ0sgRk9PVEVSIC8vXG4vLyAuL2ZpeHR1cmVzL2ltcG9ydGVkLWJpbmRpbmdzL3NyYy9vcHRpbWl6ZWQtb3V0LmpzIl0sInNvdXJjZVJvb3QiOiIifQ==,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,aW1wb3J0IGFEZWZhdWx0IGZyb20gIi4vc3JjL21vZDEiOwppbXBvcnQgeyBhTmFtZWQgfSBmcm9tICIuL3NyYy9tb2QyIjsKaW1wb3J0IHsgb3JpZ2luYWwgYXMgYW5BbGlhc2VkIH0gZnJvbSAiLi9zcmMvbW9kMyI7CmltcG9ydCAqIGFzIGFOYW1lc3BhY2UgZnJvbSAiLi9zcmMvbW9kNCI7CgppbXBvcnQgYURlZmF1bHQyIGZyb20gIi4vc3JjL21vZDUiOwppbXBvcnQgeyBhTmFtZWQyIH0gZnJvbSAiLi9zcmMvbW9kNiI7CmltcG9ydCB7IG9yaWdpbmFsIGFzIGFuQWxpYXNlZDIgfSBmcm9tICIuL3NyYy9tb2Q3IjsKaW1wb3J0ICogYXMgYU5hbWVzcGFjZTIgZnJvbSAiLi9zcmMvbW9kOCI7CgppbXBvcnQgYURlZmF1bHQzIGZyb20gIi4vc3JjL21vZDkiOwppbXBvcnQgeyBhTmFtZWQzIH0gZnJvbSAiLi9zcmMvbW9kMTAiOwppbXBvcnQgeyBvcmlnaW5hbCBhcyBhbkFsaWFzZWQzIH0gZnJvbSAiLi9zcmMvbW9kMTEiOwppbXBvcnQgKiBhcyBhTmFtZXNwYWNlMyBmcm9tICIuL3NyYy9tb2QxMiI7CgppbXBvcnQgb3B0aW1pemVkT3V0IGZyb20gIi4vc3JjL29wdGltaXplZC1vdXQiOwpvcHRpbWl6ZWRPdXQoKTsKCmV4cG9ydCBkZWZhdWx0IGZ1bmN0aW9uIHJvb3QoKSB7CiAgY29uc29sZS5sb2coInBhdXNlIGhlcmUiLCByb290KTsKCiAgY29uc29sZS5sb2coYURlZmF1bHQpOwogIGNvbnNvbGUubG9nKGFuQWxpYXNlZCk7CiAgY29uc29sZS5sb2coYU5hbWVkKTsKICBjb25zb2xlLmxvZyhhbkFsaWFzZWQpOwogIGNvbnNvbGUubG9nKGFOYW1lc3BhY2UpOwoKICB0cnkgewogICAgLy8gTm9uZSBvZiB0aGVzZSBhcmUgY2FsbGFibGUgaW4gdGhpcyBjb2RlLCBidXQgd2Ugc3RpbGwgd2FudCB0byBtYWtlIHN1cmUKICAgIC8vIHRoZXkgbWFwIHByb3Blcmx5IGV2ZW4gaWYgdGhlIG9ubHkgcmVmZXJlbmNlIGlzIGluIGEgY2FsbCBleHByZXNzaW9ucy4KICAgIGNvbnNvbGUubG9nKGFEZWZhdWx0MigpKTsKICAgIGNvbnNvbGUubG9nKGFuQWxpYXNlZDIoKSk7CiAgICBjb25zb2xlLmxvZyhhTmFtZWQyKCkpOwogICAgY29uc29sZS5sb2coYW5BbGlhc2VkMigpKTsKICAgIGNvbnNvbGUubG9nKGFOYW1lc3BhY2UyKCkpOwoKICAgIGNvbnNvbGUubG9nKG5ldyBhRGVmYXVsdDMoKSk7CiAgICBjb25zb2xlLmxvZyhuZXcgYW5BbGlhc2VkMygpKTsKICAgIGNvbnNvbGUubG9nKG5ldyBhTmFtZWQzKCkpOwogICAgY29uc29sZS5sb2cobmV3IGFuQWxpYXNlZDMoKSk7CiAgICBjb25zb2xlLmxvZyhuZXcgYU5hbWVzcGFjZTMoKSk7CiAgfSBjYXRjaCAoZSkge30KfQoKCgovLyBXRUJQQUNLIEZPT1RFUiAvLwovLyAuL2ZpeHR1cmVzL2ltcG9ydGVkLWJpbmRpbmdzL2lucHV0Lmpz,ZXhwb3J0IGRlZmF1bHQgImEtZGVmYXVsdCI7CgoKCi8vIFdFQlBBQ0sgRk9PVEVSIC8vCi8vIC4vZml4dHVyZXMvaW1wb3J0ZWQtYmluZGluZ3Mvc3JjL21vZDEuanM=,ZXhwb3J0IGNvbnN0IGFOYW1lZCA9ICJhLW5hbWVkIjsKCgoKLy8gV0VCUEFDSyBGT09URVIgLy8KLy8gLi9maXh0dXJlcy9pbXBvcnRlZC1iaW5kaW5ncy9zcmMvbW9kMi5qcw==,ZXhwb3J0IGNvbnN0IG9yaWdpbmFsID0gImFuLW9yaWdpbmFsIjsKCgoKLy8gV0VCUEFDSyBGT09URVIgLy8KLy8gLi9maXh0dXJlcy9pbXBvcnRlZC1iaW5kaW5ncy9zcmMvbW9kMy5qcw==,ZXhwb3J0IGRlZmF1bHQgImEtZGVmYXVsdCI7CmV4cG9ydCBjb25zdCBhTmFtZWQgPSAiYS1uYW1lZCI7CgoKCi8vIFdFQlBBQ0sgRk9PVEVSIC8vCi8vIC4vZml4dHVyZXMvaW1wb3J0ZWQtYmluZGluZ3Mvc3JjL21vZDQuanM=,ZXhwb3J0IGRlZmF1bHQgImEtZGVmYXVsdDIiOwoKCgovLyBXRUJQQUNLIEZPT1RFUiAvLwovLyAuL2ZpeHR1cmVzL2ltcG9ydGVkLWJpbmRpbmdzL3NyYy9tb2Q1Lmpz,ZXhwb3J0IGNvbnN0IGFOYW1lZDIgPSAiYS1uYW1lZDIiOwoKCgovLyBXRUJQQUNLIEZPT1RFUiAvLwovLyAuL2ZpeHR1cmVzL2ltcG9ydGVkLWJpbmRpbmdzL3NyYy9tb2Q2Lmpz,ZXhwb3J0IGNvbnN0IG9yaWdpbmFsID0gImFuLW9yaWdpbmFsMiI7CgoKCi8vIFdFQlBBQ0sgRk9PVEVSIC8vCi8vIC4vZml4dHVyZXMvaW1wb3J0ZWQtYmluZGluZ3Mvc3JjL21vZDcuanM=,ZXhwb3J0IGRlZmF1bHQgImEtZGVmYXVsdDIiOwpleHBvcnQgY29uc3QgYU5hbWVkID0gImEtbmFtZWQyIjsKCgoKLy8gV0VCUEFDSyBGT09URVIgLy8KLy8gLi9maXh0dXJlcy9pbXBvcnRlZC1iaW5kaW5ncy9zcmMvbW9kOC5qcw==,ZXhwb3J0IGRlZmF1bHQgImEtZGVmYXVsdDMiOwoKCgovLyBXRUJQQUNLIEZPT1RFUiAvLwovLyAuL2ZpeHR1cmVzL2ltcG9ydGVkLWJpbmRpbmdzL3NyYy9tb2Q5Lmpz,ZXhwb3J0IGNvbnN0IGFOYW1lZDMgPSAiYS1uYW1lZDMiOwoKCgovLyBXRUJQQUNLIEZPT1RFUiAvLwovLyAuL2ZpeHR1cmVzL2ltcG9ydGVkLWJpbmRpbmdzL3NyYy9tb2QxMC5qcw==,ZXhwb3J0IGNvbnN0IG9yaWdpbmFsID0gImFuLW9yaWdpbmFsMyI7CgoKCi8vIFdFQlBBQ0sgRk9PVEVSIC8vCi8vIC4vZml4dHVyZXMvaW1wb3J0ZWQtYmluZGluZ3Mvc3JjL21vZDExLmpz,ZXhwb3J0IGRlZmF1bHQgImEtZGVmYXVsdDMiOwpleHBvcnQgY29uc3QgYU5hbWVkID0gImEtbmFtZWQzIjsKCgoKLy8gV0VCUEFDSyBGT09URVIgLy8KLy8gLi9maXh0dXJlcy9pbXBvcnRlZC1iaW5kaW5ncy9zcmMvbW9kMTIuanM=,ZXhwb3J0IGRlZmF1bHQgZnVuY3Rpb24gb3B0aW1pemVkT3V0KCkge30KCgoKLy8gV0VCUEFDSyBGT09URVIgLy8KLy8gLi9maXh0dXJlcy9pbXBvcnRlZC1iaW5kaW5ncy9zcmMvb3B0aW1pemVkLW91dC5qcw==
[webpack-modules]: https://sokra.github.io/source-map-visualization/#base64,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,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,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,aW1wb3J0IGFEZWZhdWx0IGZyb20gIi4vc3JjL21vZDEiOwppbXBvcnQgeyBhTmFtZWQgfSBmcm9tICIuL3NyYy9tb2QyIjsKaW1wb3J0IHsgb3JpZ2luYWwgYXMgYW5BbGlhc2VkIH0gZnJvbSAiLi9zcmMvbW9kMyI7CmltcG9ydCAqIGFzIGFOYW1lc3BhY2UgZnJvbSAiLi9zcmMvbW9kNCI7CgppbXBvcnQgYURlZmF1bHQyIGZyb20gIi4vc3JjL21vZDUiOwppbXBvcnQgeyBhTmFtZWQyIH0gZnJvbSAiLi9zcmMvbW9kNiI7CmltcG9ydCB7IG9yaWdpbmFsIGFzIGFuQWxpYXNlZDIgfSBmcm9tICIuL3NyYy9tb2Q3IjsKaW1wb3J0ICogYXMgYU5hbWVzcGFjZTIgZnJvbSAiLi9zcmMvbW9kOCI7CgppbXBvcnQgYURlZmF1bHQzIGZyb20gIi4vc3JjL21vZDkiOwppbXBvcnQgeyBhTmFtZWQzIH0gZnJvbSAiLi9zcmMvbW9kMTAiOwppbXBvcnQgeyBvcmlnaW5hbCBhcyBhbkFsaWFzZWQzIH0gZnJvbSAiLi9zcmMvbW9kMTEiOwppbXBvcnQgKiBhcyBhTmFtZXNwYWNlMyBmcm9tICIuL3NyYy9tb2QxMiI7CgppbXBvcnQgb3B0aW1pemVkT3V0IGZyb20gIi4vc3JjL29wdGltaXplZC1vdXQiOwpvcHRpbWl6ZWRPdXQoKTsKCmV4cG9ydCBkZWZhdWx0IGZ1bmN0aW9uIHJvb3QoKSB7CiAgY29uc29sZS5sb2coInBhdXNlIGhlcmUiLCByb290KTsKCiAgY29uc29sZS5sb2coYURlZmF1bHQpOwogIGNvbnNvbGUubG9nKGFuQWxpYXNlZCk7CiAgY29uc29sZS5sb2coYU5hbWVkKTsKICBjb25zb2xlLmxvZyhhbkFsaWFzZWQpOwogIGNvbnNvbGUubG9nKGFOYW1lc3BhY2UpOwoKICB0cnkgewogICAgLy8gTm9uZSBvZiB0aGVzZSBhcmUgY2FsbGFibGUgaW4gdGhpcyBjb2RlLCBidXQgd2Ugc3RpbGwgd2FudCB0byBtYWtlIHN1cmUKICAgIC8vIHRoZXkgbWFwIHByb3Blcmx5IGV2ZW4gaWYgdGhlIG9ubHkgcmVmZXJlbmNlIGlzIGluIGEgY2FsbCBleHByZXNzaW9ucy4KICAgIGNvbnNvbGUubG9nKGFEZWZhdWx0MigpKTsKICAgIGNvbnNvbGUubG9nKGFuQWxpYXNlZDIoKSk7CiAgICBjb25zb2xlLmxvZyhhTmFtZWQyKCkpOwogICAgY29uc29sZS5sb2coYW5BbGlhc2VkMigpKTsKICAgIGNvbnNvbGUubG9nKGFOYW1lc3BhY2UyKCkpOwoKICAgIGNvbnNvbGUubG9nKG5ldyBhRGVmYXVsdDMoKSk7CiAgICBjb25zb2xlLmxvZyhuZXcgYW5BbGlhc2VkMygpKTsKICAgIGNvbnNvbGUubG9nKG5ldyBhTmFtZWQzKCkpOwogICAgY29uc29sZS5sb2cobmV3IGFuQWxpYXNlZDMoKSk7CiAgICBjb25zb2xlLmxvZyhuZXcgYU5hbWVzcGFjZTMoKSk7CiAgfSBjYXRjaCAoZSkge30KfQoKLy8gVGhlIGJ1aWxkIGhhcm5lc3Mgc2V0cyB0aGUgd3JvbmcgZ2xvYmFsLCBzbyBqdXN0IG92ZXJyaWRlIGl0LgpQcm9taXNlLnJlc29sdmUoKS50aGVuKCgpID0+IHsKICB3aW5kb3cud2VicGFja01vZHVsZXMgPSByb290Owp9KTsKCgoKLy8gV0VCUEFDSyBGT09URVIgLy8KLy8gLi9maXh0dXJlcy93ZWJwYWNrLW1vZHVsZXMvaW5wdXQuanM=,ZXhwb3J0IGRlZmF1bHQgImEtZGVmYXVsdCI7CgoKCi8vIFdFQlBBQ0sgRk9PVEVSIC8vCi8vIC4vZml4dHVyZXMvd2VicGFjay1tb2R1bGVzL3NyYy9tb2QxLmpz,ZXhwb3J0IGNvbnN0IGFOYW1lZCA9ICJhLW5hbWVkIjsKCgoKLy8gV0VCUEFDSyBGT09URVIgLy8KLy8gLi9maXh0dXJlcy93ZWJwYWNrLW1vZHVsZXMvc3JjL21vZDIuanM=,ZXhwb3J0IGNvbnN0IG9yaWdpbmFsID0gImFuLW9yaWdpbmFsIjsKCgoKLy8gV0VCUEFDSyBGT09URVIgLy8KLy8gLi9maXh0dXJlcy93ZWJwYWNrLW1vZHVsZXMvc3JjL21vZDMuanM=,ZXhwb3J0IGRlZmF1bHQgImEtZGVmYXVsdCI7CmV4cG9ydCBjb25zdCBhTmFtZWQgPSAiYS1uYW1lZCI7CgoKCi8vIFdFQlBBQ0sgRk9PVEVSIC8vCi8vIC4vZml4dHVyZXMvd2VicGFjay1tb2R1bGVzL3NyYy9tb2Q0Lmpz,ZXhwb3J0IGRlZmF1bHQgImEtZGVmYXVsdDIiOwoKCgovLyBXRUJQQUNLIEZPT1RFUiAvLwovLyAuL2ZpeHR1cmVzL3dlYnBhY2stbW9kdWxlcy9zcmMvbW9kNS5qcw==,ZXhwb3J0IGNvbnN0IGFOYW1lZDIgPSAiYS1uYW1lZDIiOwoKCgovLyBXRUJQQUNLIEZPT1RFUiAvLwovLyAuL2ZpeHR1cmVzL3dlYnBhY2stbW9kdWxlcy9zcmMvbW9kNi5qcw==,ZXhwb3J0IGNvbnN0IG9yaWdpbmFsID0gImFuLW9yaWdpbmFsMiI7CgoKCi8vIFdFQlBBQ0sgRk9PVEVSIC8vCi8vIC4vZml4dHVyZXMvd2VicGFjay1tb2R1bGVzL3NyYy9tb2Q3Lmpz,ZXhwb3J0IGRlZmF1bHQgImEtZGVmYXVsdDIiOwpleHBvcnQgY29uc3QgYU5hbWVkID0gImEtbmFtZWQyIjsKCgoKLy8gV0VCUEFDSyBGT09URVIgLy8KLy8gLi9maXh0dXJlcy93ZWJwYWNrLW1vZHVsZXMvc3JjL21vZDguanM=,ZXhwb3J0IGRlZmF1bHQgImEtZGVmYXVsdDMiOwoKCgovLyBXRUJQQUNLIEZPT1RFUiAvLwovLyAuL2ZpeHR1cmVzL3dlYnBhY2stbW9kdWxlcy9zcmMvbW9kOS5qcw==,ZXhwb3J0IGNvbnN0IGFOYW1lZDMgPSAiYS1uYW1lZDMiOwoKCgovLyBXRUJQQUNLIEZPT1RFUiAvLwovLyAuL2ZpeHR1cmVzL3dlYnBhY2stbW9kdWxlcy9zcmMvbW9kMTAuanM=,ZXhwb3J0IGNvbnN0IG9yaWdpbmFsID0gImFuLW9yaWdpbmFsMyI7CgoKCi8vIFdFQlBBQ0sgRk9PVEVSIC8vCi8vIC4vZml4dHVyZXMvd2VicGFjay1tb2R1bGVzL3NyYy9tb2QxMS5qcw==,ZXhwb3J0IGRlZmF1bHQgImEtZGVmYXVsdDMiOwpleHBvcnQgY29uc3QgYU5hbWVkID0gImEtbmFtZWQzIjsKCgoKLy8gV0VCUEFDSyBGT09URVIgLy8KLy8gLi9maXh0dXJlcy93ZWJwYWNrLW1vZHVsZXMvc3JjL21vZDEyLmpz,ZXhwb3J0IGRlZmF1bHQgZnVuY3Rpb24gb3B0aW1pemVkT3V0KCkge30KCgoKLy8gV0VCUEFDSyBGT09URVIgLy8KLy8gLi9maXh0dXJlcy93ZWJwYWNrLW1vZHVsZXMvc3JjL29wdGltaXplZC1vdXQuanM=
