/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * vim: set ts=8 sts=4 et sw=4 tw=99:
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

/* JS Array interface. */

#ifndef jsarray_h
#define jsarray_h

#include "jsobj.h"
#include "jspubtd.h"

#include "vm/ArrayObject.h"

namespace js {
/* 2^32-2, inclusive */
const uint32_t MAX_ARRAY_INDEX = 4294967294u;

MOZ_ALWAYS_INLINE bool
IdIsIndex(jsid id, uint32_t* indexp)
{
    if (JSID_IS_INT(id)) {
        int32_t i = JSID_TO_INT(id);
        MOZ_ASSERT(i >= 0);
        *indexp = (uint32_t)i;
        return true;
    }

    if (MOZ_UNLIKELY(!JSID_IS_STRING(id)))
        return false;

    JSAtom* atom = JSID_TO_ATOM(id);
    if (atom->length() == 0 || !JS7_ISDEC(atom->latin1OrTwoByteChar(0)))
        return false;

    return js::StringIsArrayIndex(atom, indexp);
}

// The methods below only create dense boxed arrays.

/* Create a dense array with no capacity allocated, length set to 0. */
extern ArrayObject * JS_FASTCALL
NewDenseEmptyArray(JSContext* cx, HandleObject proto = nullptr,
                   NewObjectKind newKind = GenericObject);

/*
 * Create a dense array with a set length, but without allocating space for the
 * contents. This is useful, e.g., when accepting length from the user.
 */
extern ArrayObject * JS_FASTCALL
NewDenseUnallocatedArray(JSContext* cx, uint32_t length, HandleObject proto = nullptr,
                         NewObjectKind newKind = GenericObject);

/*
 * Create a dense array with length and capacity == |length|, initialized length set to 0,
 * but with only |EagerAllocationMaxLength| elements allocated.
 */
extern ArrayObject * JS_FASTCALL
NewDensePartlyAllocatedArray(JSContext* cx, uint32_t length, HandleObject proto = nullptr,
                             NewObjectKind newKind = GenericObject);

/* Create a dense array with length and capacity == 'length', initialized length set to 0. */
extern ArrayObject * JS_FASTCALL
NewDenseFullyAllocatedArray(JSContext* cx, uint32_t length, HandleObject proto = nullptr,
                            NewObjectKind newKind = GenericObject);

/* Create a dense array from the given array values, which must be rooted */
extern ArrayObject*
NewDenseCopiedArray(JSContext* cx, uint32_t length, const Value* values,
                    HandleObject proto = nullptr, NewObjectKind newKind = GenericObject);

/* Create a dense array based on templateObject with the given length. */
extern ArrayObject*
NewDenseFullyAllocatedArrayWithTemplate(JSContext* cx, uint32_t length, JSObject* templateObject);

/* Create a dense array with the same copy-on-write elements as another object. */
extern ArrayObject*
NewDenseCopyOnWriteArray(JSContext* cx, HandleArrayObject templateObject, gc::InitialHeap heap);

extern ArrayObject*
NewFullyAllocatedArrayTryUseGroup(JSContext* cx, HandleObjectGroup group, size_t length,
                                  NewObjectKind newKind = GenericObject);

extern ArrayObject*
NewPartlyAllocatedArrayTryUseGroup(JSContext* cx, HandleObjectGroup group, size_t length);

extern ArrayObject*
NewFullyAllocatedArrayTryReuseGroup(JSContext* cx, HandleObject obj, size_t length,
                                    NewObjectKind newKind = GenericObject);

extern ArrayObject*
NewPartlyAllocatedArrayTryReuseGroup(JSContext* cx, HandleObject obj, size_t length);

extern ArrayObject*
NewFullyAllocatedArrayForCallingAllocationSite(JSContext* cx, size_t length,
                                               NewObjectKind newKind = GenericObject);

extern ArrayObject*
NewPartlyAllocatedArrayForCallingAllocationSite(JSContext* cx, size_t length, HandleObject proto);

extern ArrayObject*
NewCopiedArrayTryUseGroup(JSContext* cx, HandleObjectGroup group,
                          const Value* vp, size_t length,
                          NewObjectKind newKind = GenericObject,
                          ShouldUpdateTypes updateTypes = ShouldUpdateTypes::Update);

extern ArrayObject*
NewCopiedArrayForCallingAllocationSite(JSContext* cx, const Value* vp, size_t length,
                                       HandleObject proto = nullptr);

extern bool
NewValuePair(JSContext* cx, const Value& val1, const Value& val2, MutableHandleValue rval);

/*
 * Determines whether a write to the given element on |obj| should fail because
 * |obj| is an Array with a non-writable length, and writing that element would
 * increase the length of the array.
 */
extern bool
WouldDefinePastNonwritableLength(HandleNativeObject obj, uint32_t index);

extern bool
GetLengthProperty(JSContext* cx, HandleObject obj, uint32_t* lengthp);

extern bool
SetLengthProperty(JSContext* cx, HandleObject obj, uint32_t length);

/*
 * Copy 'length' elements from aobj to vp.
 *
 * This function assumes 'length' is effectively the result of calling
 * GetLengthProperty on aobj. vp must point to rooted memory.
 */
extern bool
GetElements(JSContext* cx, HandleObject aobj, uint32_t length, js::Value* vp);

/* Natives exposed for optimization by the interpreter and JITs. */

extern bool
intrinsic_ArrayNativeSort(JSContext* cx, unsigned argc, js::Value* vp);

extern bool
array_push(JSContext* cx, unsigned argc, js::Value* vp);

extern bool
array_pop(JSContext* cx, unsigned argc, js::Value* vp);

extern bool
array_join(JSContext* cx, unsigned argc, js::Value* vp);

extern void
ArrayShiftMoveElements(NativeObject* obj);

extern bool
array_shift(JSContext* cx, unsigned argc, js::Value* vp);

extern bool
array_unshift(JSContext* cx, unsigned argc, js::Value* vp);

extern bool
array_slice(JSContext* cx, unsigned argc, js::Value* vp);

extern JSObject*
array_slice_dense(JSContext* cx, HandleObject obj, int32_t begin, int32_t end, HandleObject result);

extern bool
array_reverse(JSContext* cx, unsigned argc, js::Value* vp);

extern bool
array_splice(JSContext* cx, unsigned argc, js::Value* vp);

extern const JSJitInfo array_splice_info;

/*
 * Append the given (non-hole) value to the end of an array.  The array must be
 * a newborn array -- that is, one which has not been exposed to script for
 * arbitrary manipulation.  (This method optimizes on the assumption that
 * extending the array to accommodate the element will never make the array
 * sparse, which requires that the array be completely filled.)
 */
extern bool
NewbornArrayPush(JSContext* cx, HandleObject obj, const Value& v);

extern ArrayObject*
ArrayConstructorOneArg(JSContext* cx, HandleObjectGroup group, int32_t lengthInt);

#ifdef DEBUG
extern bool
ArrayInfo(JSContext* cx, unsigned argc, Value* vp);
#endif

/* Array constructor native. Exposed only so the JIT can know its address. */
extern bool
ArrayConstructor(JSContext* cx, unsigned argc, Value* vp);

// Like Array constructor, but doesn't perform GetPrototypeFromConstructor.
extern bool
array_construct(JSContext* cx, unsigned argc, Value* vp);

extern bool
IsWrappedArrayConstructor(JSContext* cx, const Value& v, bool* result);

} /* namespace js */

#endif /* jsarray_h */
