/*
 * Copyright (C) 2012 Michael Brown <mbrown@fensystems.co.uk>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

FILE_LICENCE ( GPL2_OR_LATER );

/** @file
 *
 * X.509 self-tests
 *
 */

/* Forcibly enable assertions */
#undef NDEBUG

#include <stdint.h>
#include <string.h>
#include <errno.h>
#include <ipxe/x509.h>
#include <ipxe/asn1.h>
#include <ipxe/sha256.h>
#include <ipxe/test.h>

/** Fingerprint algorithm used for X.509 test certificates */
#define x509_test_algorithm sha256_algorithm

/** An X.509 test certificate */
struct x509_test_certificate {
	/** Data */
	const void *data;
	/** Length of data */
	size_t len;
	/** Fingerprint */
	const void *fingerprint;

	/** Parsed certificate */
	struct x509_certificate *cert;
};

/** An X.509 test certificate chain */
struct x509_test_chain {
	/** Test certificates */
	struct x509_test_certificate **certs;
	/** Number of certificates */
	unsigned int count;

	/** Parsed certificate chain */
	struct x509_chain *chain;
};

/** Define inline certificate data */
#define DATA(...) { __VA_ARGS__ }

/** Define inline fingerprint data */
#define FINGERPRINT(...) { __VA_ARGS__ }

/** Define a test certificate */
#define CERTIFICATE( name, DATA, FINGERPRINT )				\
	static const uint8_t name ## _data[] = DATA;			\
	static const uint8_t name ## _fingerprint[] = FINGERPRINT;	\
	static struct x509_test_certificate name = {			\
		.data = name ## _data,					\
		.len = sizeof ( name ## _data ),			\
		.fingerprint = name ## _fingerprint,			\
	}

/** Define a test certificate chain */
#define CHAIN( name, ... )						\
	static struct x509_test_certificate * name ## _certs[] =	\
		{ __VA_ARGS__ };					\
	static struct x509_test_chain name = {				\
		.certs = name ## _certs,				\
		.count = ( sizeof ( name ## _certs ) /			\
			   sizeof ( name ## _certs[0] ) ),		\
	}

/*
 * subject	iPXE self-test root CA
 * issuer	iPXE self-test root CA
 */
CERTIFICATE ( root_crt,
	DATA ( 0x30, 0x82, 0x02, 0xb3, 0x30, 0x82, 0x02, 0x1c, 0xa0, 0x03,
	       0x02, 0x01, 0x02, 0x02, 0x09, 0x00, 0xc6, 0xb8, 0x9c, 0x58,
	       0xd2, 0xdc, 0xc9, 0x5d, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86,
	       0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x30,
	       0x81, 0x88, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04,
	       0x06, 0x13, 0x02, 0x47, 0x42, 0x31, 0x17, 0x30, 0x15, 0x06,
	       0x03, 0x55, 0x04, 0x08, 0x0c, 0x0e, 0x43, 0x61, 0x6d, 0x62,
	       0x72, 0x69, 0x64, 0x67, 0x65, 0x73, 0x68, 0x69, 0x72, 0x65,
	       0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0c,
	       0x09, 0x43, 0x61, 0x6d, 0x62, 0x72, 0x69, 0x64, 0x67, 0x65,
	       0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c,
	       0x0f, 0x46, 0x65, 0x6e, 0x20, 0x53, 0x79, 0x73, 0x74, 0x65,
	       0x6d, 0x73, 0x20, 0x4c, 0x74, 0x64, 0x31, 0x11, 0x30, 0x0f,
	       0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x08, 0x69, 0x70, 0x78,
	       0x65, 0x2e, 0x6f, 0x72, 0x67, 0x31, 0x1f, 0x30, 0x1d, 0x06,
	       0x03, 0x55, 0x04, 0x03, 0x0c, 0x16, 0x69, 0x50, 0x58, 0x45,
	       0x20, 0x73, 0x65, 0x6c, 0x66, 0x2d, 0x74, 0x65, 0x73, 0x74,
	       0x20, 0x72, 0x6f, 0x6f, 0x74, 0x20, 0x43, 0x41, 0x30, 0x1e,
	       0x17, 0x0d, 0x31, 0x32, 0x30, 0x33, 0x32, 0x32, 0x30, 0x30,
	       0x30, 0x31, 0x33, 0x33, 0x5a, 0x17, 0x0d, 0x33, 0x39, 0x30,
	       0x38, 0x30, 0x38, 0x30, 0x30, 0x30, 0x31, 0x33, 0x33, 0x5a,
	       0x30, 0x81, 0x88, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55,
	       0x04, 0x06, 0x13, 0x02, 0x47, 0x42, 0x31, 0x17, 0x30, 0x15,
	       0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0e, 0x43, 0x61, 0x6d,
	       0x62, 0x72, 0x69, 0x64, 0x67, 0x65, 0x73, 0x68, 0x69, 0x72,
	       0x65, 0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04, 0x07,
	       0x0c, 0x09, 0x43, 0x61, 0x6d, 0x62, 0x72, 0x69, 0x64, 0x67,
	       0x65, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x0a,
	       0x0c, 0x0f, 0x46, 0x65, 0x6e, 0x20, 0x53, 0x79, 0x73, 0x74,
	       0x65, 0x6d, 0x73, 0x20, 0x4c, 0x74, 0x64, 0x31, 0x11, 0x30,
	       0x0f, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x08, 0x69, 0x70,
	       0x78, 0x65, 0x2e, 0x6f, 0x72, 0x67, 0x31, 0x1f, 0x30, 0x1d,
	       0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x16, 0x69, 0x50, 0x58,
	       0x45, 0x20, 0x73, 0x65, 0x6c, 0x66, 0x2d, 0x74, 0x65, 0x73,
	       0x74, 0x20, 0x72, 0x6f, 0x6f, 0x74, 0x20, 0x43, 0x41, 0x30,
	       0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86,
	       0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8d,
	       0x00, 0x30, 0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0xaa, 0x72,
	       0xb5, 0xc1, 0x73, 0xf4, 0x95, 0x76, 0xa4, 0x27, 0xab, 0x5e,
	       0xeb, 0x1d, 0x9d, 0xd0, 0x04, 0xb2, 0x93, 0x05, 0xc7, 0xfa,
	       0x75, 0x84, 0x66, 0xe6, 0x3a, 0x26, 0x1f, 0xbc, 0x2d, 0xfd,
	       0x8f, 0x59, 0x64, 0xac, 0xcf, 0x65, 0x9d, 0x82, 0x23, 0xc3,
	       0x72, 0x93, 0xf2, 0x40, 0x68, 0x32, 0xd1, 0xb8, 0xf1, 0x47,
	       0x61, 0x50, 0xea, 0xbc, 0xcc, 0x3c, 0x6b, 0x74, 0x7a, 0xec,
	       0x2b, 0x75, 0xa6, 0xc2, 0xa2, 0xb8, 0xbf, 0x23, 0x48, 0x97,
	       0xd5, 0xaf, 0x77, 0xc1, 0x92, 0x88, 0xd7, 0x38, 0xb7, 0x9e,
	       0xda, 0xee, 0x72, 0x04, 0xcb, 0x96, 0xe5, 0xdb, 0xfd, 0x9b,
	       0x5d, 0x99, 0x4e, 0x7a, 0x60, 0x23, 0x34, 0xa4, 0x8d, 0xd7,
	       0x6c, 0xe7, 0x5d, 0x93, 0x97, 0xe1, 0xab, 0x36, 0x2c, 0x24,
	       0x16, 0x92, 0x66, 0xf6, 0x6a, 0x14, 0x23, 0x1d, 0x18, 0xb9,
	       0x44, 0x24, 0x61, 0x6b, 0xd3, 0x75, 0x02, 0x03, 0x01, 0x00,
	       0x01, 0xa3, 0x23, 0x30, 0x21, 0x30, 0x0f, 0x06, 0x03, 0x55,
	       0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x05, 0x30, 0x03, 0x01,
	       0x01, 0xff, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01,
	       0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x02, 0x04, 0x30, 0x0d,
	       0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
	       0x05, 0x05, 0x00, 0x03, 0x81, 0x81, 0x00, 0x94, 0x9e, 0xea,
	       0x17, 0x8d, 0x27, 0xa9, 0x17, 0xe5, 0xa9, 0x19, 0xbe, 0x82,
	       0x36, 0xbd, 0xac, 0x74, 0xf3, 0x6e, 0x75, 0x71, 0x30, 0x1c,
	       0x05, 0x80, 0x6d, 0x1a, 0x69, 0x37, 0x86, 0x9c, 0x77, 0x75,
	       0x29, 0xa1, 0xc6, 0xb7, 0x11, 0x0a, 0x63, 0x27, 0xee, 0xb1,
	       0xc8, 0x94, 0xa9, 0x2e, 0x56, 0x8f, 0xca, 0x9d, 0xbe, 0xf4,
	       0xdb, 0x63, 0x97, 0x68, 0x3b, 0x13, 0xf8, 0x6a, 0xa5, 0xd1,
	       0x3d, 0xed, 0xbb, 0x86, 0x9d, 0x42, 0xfc, 0x15, 0x0a, 0x04,
	       0xf8, 0x3c, 0x0e, 0xc4, 0x86, 0x05, 0x57, 0x56, 0x96, 0xf6,
	       0xc0, 0x18, 0x53, 0xb0, 0xc5, 0xf0, 0xca, 0x72, 0x77, 0x77,
	       0xc9, 0x8e, 0x90, 0xa5, 0x4b, 0xb6, 0x80, 0x4a, 0x4c, 0x34,
	       0x6f, 0xc9, 0xe8, 0x6f, 0xc2, 0x28, 0xdf, 0x93, 0xa9, 0xf5,
	       0x63, 0x18, 0xc0, 0xec, 0x9e, 0xd5, 0x19, 0x36, 0xc5, 0x94,
	       0x10, 0xd4, 0x72, 0xd2, 0xb8 ),
	FINGERPRINT ( 0x71, 0x5d, 0x51, 0x37, 0x5e, 0x18, 0xb3, 0xbc,
		      0xbb, 0x30, 0x0e, 0x8f, 0x50, 0xc7, 0x55, 0xf5,
		      0x96, 0xe7, 0xa8, 0x6d, 0x63, 0x2d, 0x32, 0x38,
		      0xaf, 0x00, 0xc4, 0x1a, 0xfc, 0xd8, 0xac, 0xc3 ) );

/*
 * subject	iPXE self-test intermediate CA
 * issuer	iPXE self-test root CA
 */
CERTIFICATE ( intermediate_crt,
	DATA ( 0x30, 0x82, 0x02, 0xb3, 0x30, 0x82, 0x02, 0x1c, 0xa0, 0x03,
	       0x02, 0x01, 0x02, 0x02, 0x01, 0x02, 0x30, 0x0d, 0x06, 0x09,
	       0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05, 0x05,
	       0x00, 0x30, 0x81, 0x88, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03,
	       0x55, 0x04, 0x06, 0x13, 0x02, 0x47, 0x42, 0x31, 0x17, 0x30,
	       0x15, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0e, 0x43, 0x61,
	       0x6d, 0x62, 0x72, 0x69, 0x64, 0x67, 0x65, 0x73, 0x68, 0x69,
	       0x72, 0x65, 0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04,
	       0x07, 0x0c, 0x09, 0x43, 0x61, 0x6d, 0x62, 0x72, 0x69, 0x64,
	       0x67, 0x65, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04,
	       0x0a, 0x0c, 0x0f, 0x46, 0x65, 0x6e, 0x20, 0x53, 0x79, 0x73,
	       0x74, 0x65, 0x6d, 0x73, 0x20, 0x4c, 0x74, 0x64, 0x31, 0x11,
	       0x30, 0x0f, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x08, 0x69,
	       0x70, 0x78, 0x65, 0x2e, 0x6f, 0x72, 0x67, 0x31, 0x1f, 0x30,
	       0x1d, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x16, 0x69, 0x50,
	       0x58, 0x45, 0x20, 0x73, 0x65, 0x6c, 0x66, 0x2d, 0x74, 0x65,
	       0x73, 0x74, 0x20, 0x72, 0x6f, 0x6f, 0x74, 0x20, 0x43, 0x41,
	       0x30, 0x1e, 0x17, 0x0d, 0x31, 0x32, 0x30, 0x33, 0x32, 0x32,
	       0x30, 0x30, 0x30, 0x31, 0x33, 0x33, 0x5a, 0x17, 0x0d, 0x31,
	       0x34, 0x31, 0x32, 0x31, 0x37, 0x30, 0x30, 0x30, 0x31, 0x33,
	       0x33, 0x5a, 0x30, 0x81, 0x90, 0x31, 0x0b, 0x30, 0x09, 0x06,
	       0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x47, 0x42, 0x31, 0x17,
	       0x30, 0x15, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0e, 0x43,
	       0x61, 0x6d, 0x62, 0x72, 0x69, 0x64, 0x67, 0x65, 0x73, 0x68,
	       0x69, 0x72, 0x65, 0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55,
	       0x04, 0x07, 0x0c, 0x09, 0x43, 0x61, 0x6d, 0x62, 0x72, 0x69,
	       0x64, 0x67, 0x65, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55,
	       0x04, 0x0a, 0x0c, 0x0f, 0x46, 0x65, 0x6e, 0x20, 0x53, 0x79,
	       0x73, 0x74, 0x65, 0x6d, 0x73, 0x20, 0x4c, 0x74, 0x64, 0x31,
	       0x11, 0x30, 0x0f, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x08,
	       0x69, 0x70, 0x78, 0x65, 0x2e, 0x6f, 0x72, 0x67, 0x31, 0x27,
	       0x30, 0x25, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x1e, 0x69,
	       0x50, 0x58, 0x45, 0x20, 0x73, 0x65, 0x6c, 0x66, 0x2d, 0x74,
	       0x65, 0x73, 0x74, 0x20, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x6d,
	       0x65, 0x64, 0x69, 0x61, 0x74, 0x65, 0x20, 0x43, 0x41, 0x30,
	       0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86,
	       0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8d,
	       0x00, 0x30, 0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0xc9, 0x3a,
	       0xee, 0xc6, 0x3c, 0xac, 0x4d, 0x81, 0xc6, 0x98, 0x5e, 0xe1,
	       0x48, 0x66, 0x1a, 0x1e, 0x60, 0x19, 0x41, 0xae, 0xca, 0x14,
	       0x97, 0xc8, 0x3a, 0x50, 0xb6, 0x48, 0xf5, 0x42, 0xac, 0x0f,
	       0xe1, 0xe3, 0x47, 0xf0, 0xbf, 0x7c, 0xd0, 0xee, 0x8f, 0xb7,
	       0xa6, 0x19, 0xad, 0xbb, 0xc5, 0x1b, 0x34, 0x38, 0xc8, 0xbd,
	       0x55, 0x84, 0x93, 0x72, 0xaf, 0x84, 0xfc, 0x9b, 0x97, 0x1d,
	       0xb5, 0x54, 0x24, 0xd6, 0x5d, 0xb7, 0x31, 0xf4, 0xbd, 0x3b,
	       0x40, 0x97, 0xc0, 0xa9, 0x5a, 0x2a, 0xcb, 0x6b, 0x98, 0x07,
	       0xdb, 0xb5, 0x9f, 0xe8, 0x31, 0x3f, 0x01, 0x46, 0x46, 0x70,
	       0x05, 0xa2, 0x0f, 0x8c, 0x7a, 0x61, 0xf3, 0xdf, 0xdb, 0xa1,
	       0x37, 0x2c, 0x88, 0x6a, 0x81, 0x21, 0x12, 0x4c, 0xf5, 0xcd,
	       0xaf, 0xc9, 0xd2, 0x36, 0x3d, 0x82, 0xd1, 0xca, 0x19, 0xaf,
	       0x4e, 0xae, 0x50, 0x71, 0x44, 0xbf, 0x02, 0x03, 0x01, 0x00,
	       0x01, 0xa3, 0x23, 0x30, 0x21, 0x30, 0x0f, 0x06, 0x03, 0x55,
	       0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x05, 0x30, 0x03, 0x01,
	       0x01, 0xff, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01,
	       0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x02, 0x04, 0x30, 0x0d,
	       0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
	       0x05, 0x05, 0x00, 0x03, 0x81, 0x81, 0x00, 0x5d, 0x3c, 0xb3,
	       0x52, 0x19, 0xa6, 0x9e, 0x4a, 0x44, 0x98, 0xbf, 0x51, 0x20,
	       0x47, 0x0a, 0xf3, 0x26, 0x1a, 0xcc, 0x35, 0x2f, 0xc9, 0xed,
	       0xe0, 0x9d, 0x46, 0xeb, 0xbc, 0x7e, 0xc9, 0xb9, 0x1d, 0x76,
	       0xa4, 0x1d, 0xc2, 0xd9, 0x16, 0x29, 0x77, 0x01, 0x40, 0xdd,
	       0xe5, 0xcb, 0x28, 0x91, 0x3a, 0x0c, 0x13, 0x01, 0x1b, 0x72,
	       0x62, 0x45, 0x27, 0xfd, 0xd7, 0x00, 0x47, 0x36, 0x09, 0x1e,
	       0x7b, 0xd2, 0xcb, 0x95, 0x3d, 0x28, 0x82, 0xce, 0x83, 0x59,
	       0x32, 0xf9, 0xe6, 0xec, 0x89, 0xac, 0x88, 0x45, 0x22, 0x88,
	       0x6f, 0x5e, 0xa2, 0x79, 0x95, 0xba, 0xb9, 0xc9, 0xb6, 0x4c,
	       0x7c, 0xb4, 0x29, 0xa1, 0x02, 0xf5, 0xac, 0x5d, 0x8e, 0x52,
	       0xeb, 0xe8, 0xb1, 0x56, 0x49, 0xb3, 0x77, 0x62, 0x7d, 0x87,
	       0x4d, 0x17, 0xf2, 0x62, 0x83, 0x08, 0x59, 0x21, 0x60, 0x0d,
	       0x84, 0x8e, 0x5a, 0x84, 0xf6 ),
	FINGERPRINT ( 0x88, 0x70, 0xbf, 0xf0, 0xd6, 0x09, 0x03, 0x3a,
		      0xe1, 0x80, 0xa7, 0xa5, 0x5c, 0x3e, 0xe1, 0x05,
		      0x38, 0x97, 0xde, 0xe1, 0xe9, 0x74, 0x55, 0xb1,
		      0x1e, 0x59, 0x69, 0x44, 0x42, 0x1b, 0xc8, 0xff ) );

/*
 * subject	iPXE self-test leaf CA
 * issuer	iPXE self-test intermediate CA
 */
CERTIFICATE ( leaf_crt,
	DATA ( 0x30, 0x82, 0x02, 0xb6, 0x30, 0x82, 0x02, 0x1f, 0xa0, 0x03,
	       0x02, 0x01, 0x02, 0x02, 0x01, 0x02, 0x30, 0x0d, 0x06, 0x09,
	       0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05, 0x05,
	       0x00, 0x30, 0x81, 0x90, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03,
	       0x55, 0x04, 0x06, 0x13, 0x02, 0x47, 0x42, 0x31, 0x17, 0x30,
	       0x15, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0e, 0x43, 0x61,
	       0x6d, 0x62, 0x72, 0x69, 0x64, 0x67, 0x65, 0x73, 0x68, 0x69,
	       0x72, 0x65, 0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04,
	       0x07, 0x0c, 0x09, 0x43, 0x61, 0x6d, 0x62, 0x72, 0x69, 0x64,
	       0x67, 0x65, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04,
	       0x0a, 0x0c, 0x0f, 0x46, 0x65, 0x6e, 0x20, 0x53, 0x79, 0x73,
	       0x74, 0x65, 0x6d, 0x73, 0x20, 0x4c, 0x74, 0x64, 0x31, 0x11,
	       0x30, 0x0f, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x08, 0x69,
	       0x70, 0x78, 0x65, 0x2e, 0x6f, 0x72, 0x67, 0x31, 0x27, 0x30,
	       0x25, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x1e, 0x69, 0x50,
	       0x58, 0x45, 0x20, 0x73, 0x65, 0x6c, 0x66, 0x2d, 0x74, 0x65,
	       0x73, 0x74, 0x20, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x6d, 0x65,
	       0x64, 0x69, 0x61, 0x74, 0x65, 0x20, 0x43, 0x41, 0x30, 0x1e,
	       0x17, 0x0d, 0x31, 0x32, 0x30, 0x33, 0x32, 0x32, 0x30, 0x30,
	       0x30, 0x31, 0x33, 0x33, 0x5a, 0x17, 0x0d, 0x31, 0x34, 0x31,
	       0x32, 0x31, 0x37, 0x30, 0x30, 0x30, 0x31, 0x33, 0x33, 0x5a,
	       0x30, 0x81, 0x88, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55,
	       0x04, 0x06, 0x13, 0x02, 0x47, 0x42, 0x31, 0x17, 0x30, 0x15,
	       0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0e, 0x43, 0x61, 0x6d,
	       0x62, 0x72, 0x69, 0x64, 0x67, 0x65, 0x73, 0x68, 0x69, 0x72,
	       0x65, 0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04, 0x07,
	       0x0c, 0x09, 0x43, 0x61, 0x6d, 0x62, 0x72, 0x69, 0x64, 0x67,
	       0x65, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x0a,
	       0x0c, 0x0f, 0x46, 0x65, 0x6e, 0x20, 0x53, 0x79, 0x73, 0x74,
	       0x65, 0x6d, 0x73, 0x20, 0x4c, 0x74, 0x64, 0x31, 0x11, 0x30,
	       0x0f, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x08, 0x69, 0x70,
	       0x78, 0x65, 0x2e, 0x6f, 0x72, 0x67, 0x31, 0x1f, 0x30, 0x1d,
	       0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x16, 0x69, 0x50, 0x58,
	       0x45, 0x20, 0x73, 0x65, 0x6c, 0x66, 0x2d, 0x74, 0x65, 0x73,
	       0x74, 0x20, 0x6c, 0x65, 0x61, 0x66, 0x20, 0x43, 0x41, 0x30,
	       0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86,
	       0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8d,
	       0x00, 0x30, 0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0xc3, 0x55,
	       0xad, 0xdf, 0x7b, 0xd1, 0x48, 0xc3, 0xd3, 0x02, 0x54, 0x6c,
	       0x92, 0x45, 0x22, 0x3d, 0x90, 0xd8, 0xc7, 0x13, 0xcd, 0xc1,
	       0x59, 0xc6, 0xe0, 0xad, 0x0e, 0xe6, 0xdb, 0x3b, 0xe8, 0x63,
	       0xea, 0x4e, 0xb6, 0xea, 0x50, 0xea, 0x6e, 0x33, 0x9d, 0x28,
	       0x25, 0x42, 0x49, 0xd0, 0xf0, 0xed, 0xc5, 0x5b, 0x6b, 0x4a,
	       0xe7, 0x45, 0xfa, 0xd3, 0x3f, 0xae, 0xde, 0x5a, 0x90, 0xab,
	       0xf1, 0x61, 0x2f, 0x40, 0x5e, 0xcf, 0x8b, 0x0b, 0x10, 0x59,
	       0xa9, 0xd0, 0x1e, 0x0f, 0x18, 0x6b, 0x92, 0xd8, 0x9f, 0x58,
	       0x10, 0x84, 0xb6, 0x15, 0xe8, 0x5b, 0xc4, 0xa0, 0x3e, 0x49,
	       0x8b, 0xea, 0xdd, 0xa9, 0x7e, 0x32, 0x26, 0x9a, 0x68, 0x44,
	       0xf0, 0x30, 0xca, 0x2a, 0xd6, 0x19, 0x7a, 0x80, 0xfd, 0xd7,
	       0xfc, 0xc7, 0x5d, 0xe7, 0x61, 0xd2, 0x3f, 0x1f, 0x2c, 0x40,
	       0x70, 0x7b, 0x34, 0xcb, 0x08, 0xa9, 0x02, 0x03, 0x01, 0x00,
	       0x01, 0xa3, 0x26, 0x30, 0x24, 0x30, 0x12, 0x06, 0x03, 0x55,
	       0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x08, 0x30, 0x06, 0x01,
	       0x01, 0xff, 0x02, 0x01, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55,
	       0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x02,
	       0x04, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
	       0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x81, 0x81, 0x00,
	       0x40, 0xd2, 0x70, 0x02, 0x08, 0x19, 0xa0, 0xb8, 0x8d, 0x9d,
	       0x3d, 0x62, 0x41, 0x90, 0x2a, 0x36, 0x4a, 0x8b, 0x21, 0x42,
	       0x9a, 0xb4, 0xc5, 0xf8, 0x79, 0x17, 0xd7, 0x64, 0x4d, 0xbf,
	       0x8f, 0x6a, 0x04, 0x54, 0x7a, 0x0b, 0xd4, 0xb5, 0x0e, 0xab,
	       0xf7, 0xb7, 0x06, 0x2b, 0xf8, 0xde, 0x87, 0xb2, 0x37, 0x3b,
	       0x95, 0x01, 0xba, 0x9f, 0x8f, 0xec, 0x0a, 0x86, 0xca, 0x51,
	       0xb6, 0x25, 0x73, 0x2f, 0xa1, 0x66, 0xc8, 0x7a, 0x5e, 0x51,
	       0xbd, 0x49, 0xb5, 0x75, 0xda, 0xea, 0xe5, 0xeb, 0x5d, 0xe3,
	       0xb0, 0xad, 0x49, 0x9f, 0x8b, 0xfd, 0x89, 0xb3, 0xb7, 0xb2,
	       0x4c, 0x7d, 0x8a, 0x29, 0xb2, 0xbe, 0x04, 0xef, 0x9c, 0x73,
	       0x3c, 0xea, 0xa3, 0x9f, 0x07, 0x66, 0x5a, 0x2f, 0x38, 0xad,
	       0x1a, 0xeb, 0xe1, 0xb0, 0x62, 0x14, 0x55, 0xdc, 0x8c, 0x83,
	       0xbb, 0xc7, 0x13, 0x04, 0x41, 0x54, 0xf1, 0x45 ),
	FINGERPRINT ( 0xca, 0xcf, 0xea, 0x98, 0x3d, 0x71, 0xb6, 0x9d,
		      0x4f, 0x5b, 0x84, 0x5e, 0xaa, 0x8e, 0xae, 0x63,
		      0x0e, 0xad, 0x52, 0xe8, 0xc7, 0x51, 0x81, 0x07,
		      0xd1, 0xa1, 0x66, 0xdb, 0xd5, 0x62, 0xe1, 0xe6 ) );

/*
 * subject	iPXE self-test useless CA
 * issuer	iPXE self-test leaf CA
 */
CERTIFICATE ( useless_crt,
	DATA ( 0x30, 0x82, 0x02, 0xae, 0x30, 0x82, 0x02, 0x17, 0xa0, 0x03,
	       0x02, 0x01, 0x02, 0x02, 0x01, 0x02, 0x30, 0x0d, 0x06, 0x09,
	       0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05, 0x05,
	       0x00, 0x30, 0x81, 0x88, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03,
	       0x55, 0x04, 0x06, 0x13, 0x02, 0x47, 0x42, 0x31, 0x17, 0x30,
	       0x15, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0e, 0x43, 0x61,
	       0x6d, 0x62, 0x72, 0x69, 0x64, 0x67, 0x65, 0x73, 0x68, 0x69,
	       0x72, 0x65, 0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04,
	       0x07, 0x0c, 0x09, 0x43, 0x61, 0x6d, 0x62, 0x72, 0x69, 0x64,
	       0x67, 0x65, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04,
	       0x0a, 0x0c, 0x0f, 0x46, 0x65, 0x6e, 0x20, 0x53, 0x79, 0x73,
	       0x74, 0x65, 0x6d, 0x73, 0x20, 0x4c, 0x74, 0x64, 0x31, 0x11,
	       0x30, 0x0f, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x08, 0x69,
	       0x70, 0x78, 0x65, 0x2e, 0x6f, 0x72, 0x67, 0x31, 0x1f, 0x30,
	       0x1d, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x16, 0x69, 0x50,
	       0x58, 0x45, 0x20, 0x73, 0x65, 0x6c, 0x66, 0x2d, 0x74, 0x65,
	       0x73, 0x74, 0x20, 0x6c, 0x65, 0x61, 0x66, 0x20, 0x43, 0x41,
	       0x30, 0x1e, 0x17, 0x0d, 0x31, 0x32, 0x30, 0x33, 0x32, 0x32,
	       0x30, 0x30, 0x30, 0x31, 0x33, 0x34, 0x5a, 0x17, 0x0d, 0x31,
	       0x34, 0x31, 0x32, 0x31, 0x37, 0x30, 0x30, 0x30, 0x31, 0x33,
	       0x34, 0x5a, 0x30, 0x81, 0x8b, 0x31, 0x0b, 0x30, 0x09, 0x06,
	       0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x47, 0x42, 0x31, 0x17,
	       0x30, 0x15, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0e, 0x43,
	       0x61, 0x6d, 0x62, 0x72, 0x69, 0x64, 0x67, 0x65, 0x73, 0x68,
	       0x69, 0x72, 0x65, 0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55,
	       0x04, 0x07, 0x0c, 0x09, 0x43, 0x61, 0x6d, 0x62, 0x72, 0x69,
	       0x64, 0x67, 0x65, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55,
	       0x04, 0x0a, 0x0c, 0x0f, 0x46, 0x65, 0x6e, 0x20, 0x53, 0x79,
	       0x73, 0x74, 0x65, 0x6d, 0x73, 0x20, 0x4c, 0x74, 0x64, 0x31,
	       0x11, 0x30, 0x0f, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x08,
	       0x69, 0x70, 0x78, 0x65, 0x2e, 0x6f, 0x72, 0x67, 0x31, 0x22,
	       0x30, 0x20, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x19, 0x69,
	       0x50, 0x58, 0x45, 0x20, 0x73, 0x65, 0x6c, 0x66, 0x2d, 0x74,
	       0x65, 0x73, 0x74, 0x20, 0x75, 0x73, 0x65, 0x6c, 0x65, 0x73,
	       0x73, 0x20, 0x43, 0x41, 0x30, 0x81, 0x9f, 0x30, 0x0d, 0x06,
	       0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01,
	       0x05, 0x00, 0x03, 0x81, 0x8d, 0x00, 0x30, 0x81, 0x89, 0x02,
	       0x81, 0x81, 0x00, 0xbe, 0x7f, 0x5a, 0x07, 0x7c, 0x61, 0xc2,
	       0x3a, 0x7e, 0xe3, 0x94, 0xcb, 0xe9, 0xc3, 0x4c, 0x6f, 0x8d,
	       0x5c, 0x4a, 0xf0, 0xc2, 0x13, 0x54, 0x09, 0x39, 0xa8, 0xf9,
	       0xc2, 0xc3, 0xdd, 0xbe, 0x42, 0x99, 0xa6, 0xe1, 0x58, 0x0a,
	       0xd5, 0x89, 0x12, 0xa6, 0xd6, 0x4e, 0xfb, 0x6c, 0xe5, 0xab,
	       0xff, 0x40, 0x52, 0xcc, 0x1e, 0x63, 0x10, 0xd7, 0xfe, 0x49,
	       0xf3, 0x86, 0x29, 0x58, 0x6a, 0x90, 0xe4, 0xe2, 0x56, 0x85,
	       0x14, 0x7d, 0xa5, 0xf8, 0xe0, 0x7e, 0x96, 0x88, 0xd9, 0x23,
	       0xe5, 0x44, 0x72, 0xa9, 0x5a, 0xbb, 0x76, 0x6b, 0x59, 0x3e,
	       0x85, 0xd4, 0xe7, 0xb2, 0x31, 0x32, 0xea, 0x40, 0x1f, 0xce,
	       0xfb, 0xb1, 0x91, 0xee, 0x86, 0x91, 0x3e, 0xa4, 0x86, 0xa4,
	       0xe9, 0x74, 0xd7, 0x14, 0x8c, 0xb6, 0xb4, 0xc0, 0x08, 0xbb,
	       0xc8, 0x38, 0xc3, 0x96, 0x3d, 0x85, 0xcf, 0xef, 0x94, 0x52,
	       0x29, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x23, 0x30, 0x21,
	       0x30, 0x0f, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff,
	       0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xff, 0x30, 0x0e, 0x06,
	       0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03,
	       0x02, 0x02, 0x04, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48,
	       0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x81,
	       0x81, 0x00, 0x50, 0x59, 0xfb, 0x9d, 0x4d, 0xfe, 0x0e, 0x5b,
	       0xc4, 0x51, 0xe9, 0xe8, 0xa4, 0xf5, 0x2f, 0x32, 0x8b, 0x06,
	       0x78, 0xbe, 0xf1, 0x18, 0xc5, 0x6f, 0xd9, 0x20, 0xee, 0xb7,
	       0x51, 0x40, 0xaf, 0xf3, 0x3c, 0xe4, 0x74, 0x00, 0xa4, 0x63,
	       0x3b, 0x37, 0xe1, 0xef, 0x80, 0xdc, 0xd5, 0x90, 0xed, 0xba,
	       0x91, 0x86, 0x7f, 0x97, 0x5d, 0x3e, 0x8f, 0x29, 0xcc, 0x57,
	       0xee, 0x79, 0x15, 0x6b, 0xe3, 0xd1, 0x25, 0x14, 0x24, 0xdf,
	       0xbf, 0x38, 0xee, 0xe3, 0x8a, 0x88, 0x19, 0x0f, 0xc8, 0x10,
	       0xae, 0x27, 0x99, 0xa8, 0x35, 0x47, 0xc9, 0xfb, 0x92, 0x47,
	       0xa2, 0x36, 0x2a, 0x8c, 0x26, 0x12, 0xb1, 0x0d, 0x46, 0xe2,
	       0xdc, 0x33, 0x29, 0x0c, 0x32, 0xcf, 0x22, 0x49, 0xde, 0xc3,
	       0x55, 0x2a, 0xba, 0xdd, 0xe3, 0x98, 0xc0, 0xe4, 0x9a, 0xa2,
	       0xe5, 0x43, 0x04, 0x32, 0xd3, 0x50, 0x7d, 0x9c, 0x71, 0x23 ),
	FINGERPRINT ( 0xda, 0xbf, 0xd3, 0x5e, 0x2e, 0x29, 0xa9, 0xfd,
		      0x4d, 0x40, 0xba, 0xb8, 0xdd, 0x66, 0x93, 0x4c,
		      0x10, 0xea, 0x5b, 0x07, 0xa6, 0xe2, 0x27, 0x63,
		      0x2e, 0xfe, 0x01, 0x63, 0x7c, 0xea, 0xc6, 0xd0 ) );

/*
 * subject	boot.test.ipxe.org
 * issuer	iPXE self-test leaf CA
 */
CERTIFICATE ( server_crt,
	DATA ( 0x30, 0x82, 0x02, 0xba, 0x30, 0x82, 0x02, 0x23, 0xa0, 0x03,
	       0x02, 0x01, 0x02, 0x02, 0x01, 0x18, 0x30, 0x0d, 0x06, 0x09,
	       0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05, 0x05,
	       0x00, 0x30, 0x81, 0x88, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03,
	       0x55, 0x04, 0x06, 0x13, 0x02, 0x47, 0x42, 0x31, 0x17, 0x30,
	       0x15, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0e, 0x43, 0x61,
	       0x6d, 0x62, 0x72, 0x69, 0x64, 0x67, 0x65, 0x73, 0x68, 0x69,
	       0x72, 0x65, 0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04,
	       0x07, 0x0c, 0x09, 0x43, 0x61, 0x6d, 0x62, 0x72, 0x69, 0x64,
	       0x67, 0x65, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04,
	       0x0a, 0x0c, 0x0f, 0x46, 0x65, 0x6e, 0x20, 0x53, 0x79, 0x73,
	       0x74, 0x65, 0x6d, 0x73, 0x20, 0x4c, 0x74, 0x64, 0x31, 0x11,
	       0x30, 0x0f, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x08, 0x69,
	       0x70, 0x78, 0x65, 0x2e, 0x6f, 0x72, 0x67, 0x31, 0x1f, 0x30,
	       0x1d, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x16, 0x69, 0x50,
	       0x58, 0x45, 0x20, 0x73, 0x65, 0x6c, 0x66, 0x2d, 0x74, 0x65,
	       0x73, 0x74, 0x20, 0x6c, 0x65, 0x61, 0x66, 0x20, 0x43, 0x41,
	       0x30, 0x1e, 0x17, 0x0d, 0x31, 0x32, 0x30, 0x33, 0x30, 0x35,
	       0x31, 0x33, 0x34, 0x35, 0x30, 0x30, 0x5a, 0x17, 0x0d, 0x31,
	       0x33, 0x30, 0x33, 0x30, 0x35, 0x31, 0x33, 0x34, 0x35, 0x30,
	       0x30, 0x5a, 0x30, 0x81, 0x84, 0x31, 0x0b, 0x30, 0x09, 0x06,
	       0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x47, 0x42, 0x31, 0x17,
	       0x30, 0x15, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0e, 0x43,
	       0x61, 0x6d, 0x62, 0x72, 0x69, 0x64, 0x67, 0x65, 0x73, 0x68,
	       0x69, 0x72, 0x65, 0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55,
	       0x04, 0x07, 0x0c, 0x09, 0x43, 0x61, 0x6d, 0x62, 0x72, 0x69,
	       0x64, 0x67, 0x65, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55,
	       0x04, 0x0a, 0x0c, 0x0f, 0x46, 0x65, 0x6e, 0x20, 0x53, 0x79,
	       0x73, 0x74, 0x65, 0x6d, 0x73, 0x20, 0x4c, 0x74, 0x64, 0x31,
	       0x11, 0x30, 0x0f, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x08,
	       0x69, 0x70, 0x78, 0x65, 0x2e, 0x6f, 0x72, 0x67, 0x31, 0x1b,
	       0x30, 0x19, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x12, 0x62,
	       0x6f, 0x6f, 0x74, 0x2e, 0x74, 0x65, 0x73, 0x74, 0x2e, 0x69,
	       0x70, 0x78, 0x65, 0x2e, 0x6f, 0x72, 0x67, 0x30, 0x81, 0x9f,
	       0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
	       0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8d, 0x00, 0x30,
	       0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0x9d, 0x87, 0xe4, 0xa7,
	       0xcf, 0x12, 0x08, 0x43, 0x4c, 0x90, 0x8b, 0x10, 0x7d, 0xcc,
	       0x94, 0x1e, 0x5e, 0xef, 0xa7, 0x90, 0xbc, 0xe8, 0xe4, 0xee,
	       0xd9, 0xb4, 0xd9, 0x63, 0x55, 0xc7, 0x03, 0x98, 0x42, 0xd7,
	       0x4e, 0xaf, 0xd7, 0xdc, 0x40, 0x83, 0x61, 0x1b, 0xcc, 0x7b,
	       0xf5, 0x1d, 0xba, 0x9f, 0x66, 0xfb, 0xe7, 0x42, 0xbd, 0xd7,
	       0xac, 0xeb, 0x3c, 0xa2, 0x99, 0x6a, 0xe4, 0x8f, 0xb4, 0x06,
	       0x4e, 0xc3, 0x3b, 0x62, 0xcd, 0x6a, 0x30, 0x0a, 0xe0, 0xb1,
	       0x50, 0x83, 0x77, 0xc4, 0x97, 0x15, 0xc4, 0x7c, 0x40, 0xb8,
	       0x60, 0x39, 0x07, 0x72, 0x4b, 0xd2, 0x61, 0x5c, 0xd0, 0xac,
	       0x21, 0x9b, 0x85, 0xba, 0x53, 0x39, 0x1d, 0xef, 0xe9, 0xb7,
	       0x69, 0xed, 0x7f, 0x1c, 0x38, 0x56, 0x0a, 0xe5, 0x24, 0xd0,
	       0x1a, 0xa5, 0x9a, 0xd2, 0x5e, 0x1b, 0x47, 0x42, 0x49, 0x08,
	       0x0d, 0x68, 0x2d, 0xc9, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3,
	       0x36, 0x30, 0x34, 0x30, 0x32, 0x06, 0x03, 0x55, 0x1d, 0x11,
	       0x04, 0x2b, 0x30, 0x29, 0x82, 0x12, 0x64, 0x65, 0x6d, 0x6f,
	       0x2e, 0x74, 0x65, 0x73, 0x74, 0x2e, 0x69, 0x70, 0x78, 0x65,
	       0x2e, 0x6f, 0x72, 0x67, 0x82, 0x13, 0x2a, 0x2e, 0x61, 0x6c,
	       0x74, 0x2e, 0x74, 0x65, 0x73, 0x74, 0x2e, 0x69, 0x70, 0x78,
	       0x65, 0x2e, 0x6f, 0x72, 0x67, 0x30, 0x0d, 0x06, 0x09, 0x2a,
	       0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05, 0x05, 0x00,
	       0x03, 0x81, 0x81, 0x00, 0x23, 0x16, 0x6a, 0x10, 0x55, 0x44,
	       0xb9, 0x9d, 0x9f, 0x9f, 0x53, 0x51, 0x3d, 0x7d, 0x33, 0xa1,
	       0x84, 0xb2, 0x5a, 0xfb, 0x1d, 0x76, 0xd5, 0xb1, 0x79, 0x66,
	       0xf5, 0xe3, 0xa6, 0x58, 0x2e, 0x3d, 0xec, 0x9f, 0xcf, 0x7d,
	       0x75, 0x3b, 0xd7, 0xe8, 0xf6, 0x96, 0xd7, 0xdd, 0x89, 0x1e,
	       0x30, 0x25, 0xd9, 0xbb, 0xc0, 0x99, 0xc0, 0x1f, 0x1b, 0x4f,
	       0xa6, 0x8e, 0xd5, 0x76, 0x50, 0x18, 0xa1, 0x7a, 0x48, 0x08,
	       0xd5, 0x75, 0xee, 0x20, 0x82, 0x12, 0xc0, 0xe8, 0xeb, 0xf1,
	       0x50, 0xee, 0x9d, 0xbd, 0x73, 0x7c, 0xb5, 0x13, 0x05, 0x91,
	       0x1f, 0xc6, 0x50, 0x08, 0xbc, 0x98, 0xde, 0x43, 0x9a, 0xa4,
	       0x9f, 0x69, 0xf7, 0x6e, 0x36, 0x20, 0x42, 0x80, 0x72, 0xba,
	       0x0d, 0x63, 0x4c, 0xc5, 0x00, 0x0d, 0x85, 0xaa, 0x14, 0x38,
	       0x28, 0x11, 0x3e, 0xa2, 0xcc, 0xc2, 0xac, 0xe8, 0xa7, 0xbe,
	       0x0a, 0xa0 ),
	FINGERPRINT ( 0x2f, 0xd3, 0xe0, 0x69, 0xde, 0xbc, 0x7c, 0x39,
		      0xa7, 0xee, 0x23, 0x3b, 0xf5, 0x92, 0xf5, 0xbe,
		      0x05, 0xab, 0xb5, 0xf8, 0x42, 0x9e, 0xf5, 0x9c,
		      0x24, 0xde, 0x9e, 0x1f, 0xeb, 0xed, 0xd1, 0x20 ) );

/*
 * subject	not.a.ca.test.ipxe.org
 * issuer	boot.test.ipxe.org
 */
CERTIFICATE ( not_ca_crt,
	DATA ( 0x30, 0x82, 0x02, 0x7d, 0x30, 0x82, 0x01, 0xe6, 0x02, 0x01,
	       0x02, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
	       0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x30, 0x81, 0x84, 0x31,
	       0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
	       0x47, 0x42, 0x31, 0x17, 0x30, 0x15, 0x06, 0x03, 0x55, 0x04,
	       0x08, 0x0c, 0x0e, 0x43, 0x61, 0x6d, 0x62, 0x72, 0x69, 0x64,
	       0x67, 0x65, 0x73, 0x68, 0x69, 0x72, 0x65, 0x31, 0x12, 0x30,
	       0x10, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x09, 0x43, 0x61,
	       0x6d, 0x62, 0x72, 0x69, 0x64, 0x67, 0x65, 0x31, 0x18, 0x30,
	       0x16, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x0f, 0x46, 0x65,
	       0x6e, 0x20, 0x53, 0x79, 0x73, 0x74, 0x65, 0x6d, 0x73, 0x20,
	       0x4c, 0x74, 0x64, 0x31, 0x11, 0x30, 0x0f, 0x06, 0x03, 0x55,
	       0x04, 0x0b, 0x0c, 0x08, 0x69, 0x70, 0x78, 0x65, 0x2e, 0x6f,
	       0x72, 0x67, 0x31, 0x1b, 0x30, 0x19, 0x06, 0x03, 0x55, 0x04,
	       0x03, 0x0c, 0x12, 0x62, 0x6f, 0x6f, 0x74, 0x2e, 0x74, 0x65,
	       0x73, 0x74, 0x2e, 0x69, 0x70, 0x78, 0x65, 0x2e, 0x6f, 0x72,
	       0x67, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x32, 0x30, 0x33, 0x32,
	       0x32, 0x30, 0x30, 0x30, 0x31, 0x33, 0x34, 0x5a, 0x17, 0x0d,
	       0x31, 0x33, 0x30, 0x33, 0x32, 0x32, 0x30, 0x30, 0x30, 0x31,
	       0x33, 0x34, 0x5a, 0x30, 0x81, 0x88, 0x31, 0x0b, 0x30, 0x09,
	       0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x47, 0x42, 0x31,
	       0x17, 0x30, 0x15, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0e,
	       0x43, 0x61, 0x6d, 0x62, 0x72, 0x69, 0x64, 0x67, 0x65, 0x73,
	       0x68, 0x69, 0x72, 0x65, 0x31, 0x12, 0x30, 0x10, 0x06, 0x03,
	       0x55, 0x04, 0x07, 0x0c, 0x09, 0x43, 0x61, 0x6d, 0x62, 0x72,
	       0x69, 0x64, 0x67, 0x65, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03,
	       0x55, 0x04, 0x0a, 0x0c, 0x0f, 0x46, 0x65, 0x6e, 0x20, 0x53,
	       0x79, 0x73, 0x74, 0x65, 0x6d, 0x73, 0x20, 0x4c, 0x74, 0x64,
	       0x31, 0x11, 0x30, 0x0f, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c,
	       0x08, 0x69, 0x70, 0x78, 0x65, 0x2e, 0x6f, 0x72, 0x67, 0x31,
	       0x1f, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x16,
	       0x6e, 0x6f, 0x74, 0x2e, 0x61, 0x2e, 0x63, 0x61, 0x2e, 0x74,
	       0x65, 0x73, 0x74, 0x2e, 0x69, 0x70, 0x78, 0x65, 0x2e, 0x6f,
	       0x72, 0x67, 0x30, 0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09, 0x2a,
	       0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00,
	       0x03, 0x81, 0x8d, 0x00, 0x30, 0x81, 0x89, 0x02, 0x81, 0x81,
	       0x00, 0xc3, 0x5b, 0x6d, 0xb3, 0x8d, 0x74, 0x9c, 0x1d, 0xbd,
	       0x94, 0x41, 0xa2, 0x42, 0x96, 0x3c, 0x41, 0x82, 0xc0, 0xf1,
	       0x95, 0xbf, 0xc5, 0x34, 0x92, 0x92, 0xa3, 0xed, 0xed, 0x5c,
	       0x07, 0xaa, 0xb4, 0xc1, 0x66, 0xbb, 0xa6, 0xd1, 0xd9, 0x78,
	       0x93, 0xf1, 0x9c, 0x3e, 0x13, 0x3a, 0xee, 0x74, 0x31, 0xeb,
	       0x55, 0x86, 0xa5, 0x43, 0x8a, 0x5d, 0x0c, 0x2c, 0x0d, 0xfb,
	       0x91, 0x9e, 0x31, 0x22, 0xbe, 0x96, 0xb5, 0x0e, 0x44, 0xc8,
	       0x5b, 0x65, 0xb2, 0xf5, 0xec, 0x2a, 0x51, 0xed, 0x8f, 0x28,
	       0xd8, 0xb2, 0x4b, 0x45, 0x39, 0x31, 0x1f, 0x11, 0xb7, 0x12,
	       0xe3, 0xc6, 0xb2, 0xd2, 0x8d, 0x50, 0xd5, 0xf4, 0xd2, 0x71,
	       0x77, 0xc9, 0x4c, 0x67, 0xee, 0xf7, 0xdc, 0xdb, 0x68, 0xa6,
	       0xac, 0x33, 0xd4, 0xb2, 0x12, 0x61, 0x5c, 0xae, 0x4c, 0x2e,
	       0x26, 0xe8, 0xdf, 0x46, 0x3a, 0x05, 0xaf, 0xeb, 0x0d, 0x02,
	       0x03, 0x01, 0x00, 0x01, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86,
	       0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03,
	       0x81, 0x81, 0x00, 0x90, 0x3e, 0x16, 0x27, 0x2f, 0x4e, 0x4b,
	       0x31, 0x0e, 0xae, 0x31, 0x9d, 0x64, 0x88, 0x9f, 0xce, 0xd8,
	       0x22, 0x51, 0x9d, 0xd9, 0x2b, 0xfe, 0xed, 0x75, 0xbe, 0xec,
	       0x5a, 0x73, 0xaf, 0x6c, 0xa5, 0x5e, 0xd1, 0x15, 0x9a, 0x08,
	       0xcf, 0x4d, 0x41, 0x78, 0x48, 0xb4, 0x29, 0xf1, 0xf7, 0x63,
	       0x9b, 0x11, 0x91, 0x16, 0x94, 0x55, 0xff, 0xeb, 0xe9, 0x6f,
	       0x0a, 0x34, 0x89, 0xed, 0xf2, 0xd1, 0x79, 0x91, 0x9d, 0xe5,
	       0x73, 0x48, 0x68, 0x7f, 0x9b, 0xf4, 0x94, 0x80, 0x29, 0xbb,
	       0x2f, 0xac, 0x6c, 0xf7, 0x6a, 0x43, 0xcc, 0x40, 0x34, 0x85,
	       0xc8, 0xa1, 0x6d, 0x16, 0x36, 0x65, 0x3f, 0x93, 0x60, 0xc1,
	       0x64, 0x33, 0x91, 0xa1, 0x8f, 0x86, 0x8c, 0xce, 0x14, 0x19,
	       0x72, 0x28, 0xef, 0x94, 0x3d, 0x09, 0xb8, 0x3b, 0x39, 0xe8,
	       0xd1, 0x66, 0x2b, 0x38, 0xb4, 0x46, 0x50, 0xf4, 0xcd, 0xc4,
	       0x9a ),
	FINGERPRINT ( 0x37, 0x6b, 0xc2, 0x20, 0xa9, 0xbc, 0xe2, 0x83,
		      0x99, 0x60, 0x06, 0x2e, 0xaf, 0x94, 0xfe, 0xb0,
		      0x1a, 0x2c, 0x17, 0x47, 0x1e, 0xc0, 0xd1, 0x66,
		      0xb6, 0x76, 0xeb, 0x1c, 0x07, 0xae, 0x72, 0xf2 ) );

/*
 * subject	bad.path.len.test.ipxe.org
 * issuer	iPXE self-test useless CA
 */
CERTIFICATE ( bad_path_len_crt,
	DATA ( 0x30, 0x82, 0x02, 0x88, 0x30, 0x82, 0x01, 0xf1, 0x02, 0x01,
	       0x02, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
	       0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x30, 0x81, 0x8b, 0x31,
	       0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
	       0x47, 0x42, 0x31, 0x17, 0x30, 0x15, 0x06, 0x03, 0x55, 0x04,
	       0x08, 0x0c, 0x0e, 0x43, 0x61, 0x6d, 0x62, 0x72, 0x69, 0x64,
	       0x67, 0x65, 0x73, 0x68, 0x69, 0x72, 0x65, 0x31, 0x12, 0x30,
	       0x10, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x09, 0x43, 0x61,
	       0x6d, 0x62, 0x72, 0x69, 0x64, 0x67, 0x65, 0x31, 0x18, 0x30,
	       0x16, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x0f, 0x46, 0x65,
	       0x6e, 0x20, 0x53, 0x79, 0x73, 0x74, 0x65, 0x6d, 0x73, 0x20,
	       0x4c, 0x74, 0x64, 0x31, 0x11, 0x30, 0x0f, 0x06, 0x03, 0x55,
	       0x04, 0x0b, 0x0c, 0x08, 0x69, 0x70, 0x78, 0x65, 0x2e, 0x6f,
	       0x72, 0x67, 0x31, 0x22, 0x30, 0x20, 0x06, 0x03, 0x55, 0x04,
	       0x03, 0x0c, 0x19, 0x69, 0x50, 0x58, 0x45, 0x20, 0x73, 0x65,
	       0x6c, 0x66, 0x2d, 0x74, 0x65, 0x73, 0x74, 0x20, 0x75, 0x73,
	       0x65, 0x6c, 0x65, 0x73, 0x73, 0x20, 0x43, 0x41, 0x30, 0x1e,
	       0x17, 0x0d, 0x31, 0x32, 0x30, 0x33, 0x32, 0x32, 0x30, 0x30,
	       0x30, 0x31, 0x33, 0x34, 0x5a, 0x17, 0x0d, 0x31, 0x33, 0x30,
	       0x33, 0x32, 0x32, 0x30, 0x30, 0x30, 0x31, 0x33, 0x34, 0x5a,
	       0x30, 0x81, 0x8c, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55,
	       0x04, 0x06, 0x13, 0x02, 0x47, 0x42, 0x31, 0x17, 0x30, 0x15,
	       0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0e, 0x43, 0x61, 0x6d,
	       0x62, 0x72, 0x69, 0x64, 0x67, 0x65, 0x73, 0x68, 0x69, 0x72,
	       0x65, 0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04, 0x07,
	       0x0c, 0x09, 0x43, 0x61, 0x6d, 0x62, 0x72, 0x69, 0x64, 0x67,
	       0x65, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x0a,
	       0x0c, 0x0f, 0x46, 0x65, 0x6e, 0x20, 0x53, 0x79, 0x73, 0x74,
	       0x65, 0x6d, 0x73, 0x20, 0x4c, 0x74, 0x64, 0x31, 0x11, 0x30,
	       0x0f, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x08, 0x69, 0x70,
	       0x78, 0x65, 0x2e, 0x6f, 0x72, 0x67, 0x31, 0x23, 0x30, 0x21,
	       0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x1a, 0x62, 0x61, 0x64,
	       0x2e, 0x70, 0x61, 0x74, 0x68, 0x2e, 0x6c, 0x65, 0x6e, 0x2e,
	       0x74, 0x65, 0x73, 0x74, 0x2e, 0x69, 0x70, 0x78, 0x65, 0x2e,
	       0x6f, 0x72, 0x67, 0x30, 0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09,
	       0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05,
	       0x00, 0x03, 0x81, 0x8d, 0x00, 0x30, 0x81, 0x89, 0x02, 0x81,
	       0x81, 0x00, 0xed, 0xf1, 0xe3, 0xb2, 0x61, 0x68, 0xa0, 0xd5,
	       0x43, 0xfe, 0xad, 0xee, 0xfb, 0x8e, 0x2c, 0xf0, 0x44, 0xaf,
	       0x0a, 0x3c, 0x87, 0xc2, 0x56, 0x9b, 0x66, 0x15, 0xc6, 0xbc,
	       0x5b, 0x96, 0xef, 0xa1, 0x49, 0xd6, 0xe7, 0xeb, 0xb8, 0xf6,
	       0x3d, 0x62, 0xf5, 0x51, 0xfd, 0xb1, 0xa5, 0x4e, 0x92, 0x7c,
	       0x7a, 0x31, 0x1b, 0xb8, 0x21, 0x5c, 0xfe, 0x0b, 0x4e, 0x58,
	       0xd6, 0xd0, 0x8b, 0x81, 0x00, 0x4a, 0xf8, 0xf7, 0x2a, 0xc9,
	       0xea, 0xfa, 0x9c, 0xc9, 0x33, 0x0b, 0xc4, 0xce, 0x96, 0x4c,
	       0x30, 0x6e, 0xf0, 0x07, 0xfa, 0x1b, 0x94, 0x1f, 0xe3, 0x3b,
	       0xb2, 0x7d, 0x31, 0x1a, 0x37, 0x64, 0xe2, 0xc3, 0xf1, 0xe5,
	       0xb9, 0xcc, 0xd1, 0x02, 0xae, 0x16, 0x39, 0x9b, 0xfc, 0x55,
	       0xca, 0xdd, 0x33, 0x92, 0xe3, 0x12, 0x40, 0xc5, 0x32, 0x51,
	       0x62, 0xac, 0x3a, 0xc0, 0x17, 0x36, 0xd0, 0x27, 0x3d, 0xbb,
	       0x02, 0x03, 0x01, 0x00, 0x01, 0x30, 0x0d, 0x06, 0x09, 0x2a,
	       0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05, 0x05, 0x00,
	       0x03, 0x81, 0x81, 0x00, 0x07, 0x53, 0x2a, 0x80, 0xd6, 0x25,
	       0x10, 0x37, 0xce, 0x3b, 0x87, 0x87, 0xfc, 0xae, 0xe2, 0x2a,
	       0x28, 0x3f, 0xf7, 0xa6, 0x32, 0x5b, 0x06, 0xbd, 0x4f, 0x34,
	       0x6b, 0x47, 0x8a, 0x4b, 0x47, 0x51, 0xe8, 0x45, 0x69, 0xe3,
	       0xf3, 0xdf, 0xa4, 0x25, 0x8f, 0x34, 0xbe, 0xe5, 0x2c, 0xa4,
	       0x6c, 0x8c, 0x6e, 0x02, 0x74, 0x23, 0x43, 0x21, 0x4d, 0xe3,
	       0x75, 0x93, 0x8e, 0xa8, 0x2c, 0x54, 0xba, 0x35, 0xe7, 0xab,
	       0x44, 0xfa, 0x07, 0x7a, 0x18, 0xb4, 0xa7, 0xce, 0xfa, 0xa6,
	       0x74, 0x5a, 0x45, 0x2c, 0x6f, 0x86, 0x34, 0x8f, 0x4a, 0x09,
	       0xe0, 0xf3, 0x4f, 0x37, 0xbb, 0xa3, 0xa0, 0xcb, 0xad, 0x6b,
	       0xc1, 0x16, 0x06, 0xdf, 0x83, 0x98, 0xaf, 0xa8, 0xc3, 0xa0,
	       0x5f, 0x33, 0x09, 0x01, 0x12, 0xbd, 0xd3, 0x45, 0x9f, 0x5f,
	       0x96, 0x93, 0xe9, 0x69, 0xe9, 0xb1, 0x8a, 0xe4, 0x94, 0xce,
	       0xe4, 0x8d ),
	FINGERPRINT ( 0xb6, 0x80, 0x84, 0xf1, 0x45, 0x55, 0x1f, 0xbc,
		      0x15, 0xa6, 0xd8, 0x4b, 0xf3, 0x19, 0x65, 0xef,
		      0x53, 0x5a, 0xc8, 0x99, 0xe5, 0xdf, 0x79, 0x07,
		      0x00, 0x2c, 0x9f, 0x49, 0x91, 0x21, 0xeb, 0xfc ) );

/** Valid certificate chain up to boot.test.ipxe.org */
CHAIN ( server_chain, &server_crt, &leaf_crt, &intermediate_crt, &root_crt );

/** Broken certificate chain up to boot.test.ipxe.org */
CHAIN ( broken_server_chain, &server_crt, &leaf_crt, &root_crt );

/** Incomplete certificate chain up to boot.test.ipxe.org */
CHAIN ( incomplete_server_chain, &server_crt, &leaf_crt, &intermediate_crt );

/** Non-functional certificate chain up to not_ca.test.ipxe.org */
CHAIN ( not_ca_chain,
	&not_ca_crt, &server_crt, &leaf_crt, &intermediate_crt, &root_crt );

/** Valid certificate chain up to iPXE self-test useless CA */
CHAIN ( useless_chain, &useless_crt, &leaf_crt,	&intermediate_crt, &root_crt );

/** Non-functional certificate chain up to bad.path.len.test.ipxe.org */
CHAIN ( bad_path_len_chain, &bad_path_len_crt, &useless_crt, &leaf_crt,
	&intermediate_crt, &root_crt );

/** Empty certificate store */
static struct x509_chain empty_store = {
	.refcnt = REF_INIT ( ref_no_free ),
	.links = LIST_HEAD_INIT ( empty_store.links ),
};

/** Root certificate list containing the iPXE self-test root CA */
static struct x509_root test_root = {
	.digest = &x509_test_algorithm,
	.count = 1,
	.fingerprints = root_crt_fingerprint,
};

/** Root certificate list containing the iPXE self-test intermediate CA */
static struct x509_root intermediate_root = {
	.digest = &x509_test_algorithm,
	.count = 1,
	.fingerprints = intermediate_crt_fingerprint,
};

/** Dummy fingerprint (not matching any certificates) */
static uint8_t dummy_fingerprint[] =
	FINGERPRINT ( 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
		      0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff,
		      0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
		      0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff );

/** Certificate store containing a dummy fingerprint */
static struct x509_root dummy_root = {
	.digest = &x509_test_algorithm,
	.count = 1,
	.fingerprints = dummy_fingerprint,
};

/** Time at which all test certificates are valid */
static time_t test_time = 1332374737ULL; /* Thu Mar 22 00:05:37 2012 */

/** Time at which end-entity test certificates are invalid */
static time_t test_expired = 1375573111ULL; /* Sat Aug  3 23:38:31 2013 */

/** Time at which CA test certificates are invalid */
static time_t test_ca_expired = 2205014905ULL; /* Wed Nov 16 00:08:25 2039 */

/**
 * Report certificate parsing test result
 *
 * @v crt		Test certificate
 * @v file		Test code file
 * @v line		Test code line
 */
static void x509_certificate_okx ( struct x509_test_certificate *crt,
				   const char *file, unsigned int line ) {

	okx ( x509_certificate ( crt->data, crt->len, &crt->cert ) == 0,
	      file, line );
}
#define x509_certificate_ok( crt ) \
	x509_certificate_okx ( crt, __FILE__, __LINE__ )

/**
 * Report cached certificate parsing test result
 *
 * @v crt		Test certificate
 * @v file		Test code file
 * @v line		Test code line
 */
static void x509_cached_okx ( struct x509_test_certificate *crt,
			      const char *file, unsigned int line ) {
	struct x509_certificate *temp;

	okx ( x509_certificate ( crt->data, crt->len, &temp ) == 0,
	      file, line );
	okx ( temp == crt->cert, file, line );
	x509_put ( temp );
}
#define x509_cached_ok( crt ) x509_cached_okx ( crt, __FILE__, __LINE__ )

/**
 * Report certificate fingerprint test result
 *
 * @v crt		Test certificate
 * @v file		Test code file
 * @v line		Test code line
 */
static void x509_fingerprint_okx ( struct x509_test_certificate *crt,
				   const char *file, unsigned int line ) {
	uint8_t fingerprint[ x509_test_algorithm.digestsize ];

	x509_fingerprint ( crt->cert, &x509_test_algorithm, fingerprint );
	okx ( memcmp ( fingerprint, crt->fingerprint,
		       sizeof ( fingerprint ) ) == 0, file, line );
}
#define x509_fingerprint_ok( crt ) \
	x509_fingerprint_okx ( crt, __FILE__, __LINE__ )

/**
 * Report certificate issuer validation test result
 *
 * @v crt		Test certificate
 * @v issuer		Test issuer
 * @v file		Test code file
 * @v line		Test code line
 */
static void x509_check_issuer_okx ( struct x509_test_certificate *crt,
				    struct x509_test_certificate *issuer,
				    const char *file, unsigned int line ) {

	okx ( x509_check_issuer ( crt->cert, issuer->cert ) == 0, file, line );
}
#define x509_check_issuer_ok( crt, issuer ) \
	x509_check_issuer_okx ( crt, issuer, __FILE__, __LINE__ )

/**
 * Report certificate issuer validation failure test result
 *
 * @v crt		Test certificate
 * @v issuer		Test issuer
 * @v file		Test code file
 * @v line		Test code line
 */
static void x509_check_issuer_fail_okx ( struct x509_test_certificate *crt,
					 struct x509_test_certificate *issuer,
					 const char *file, unsigned int line ) {

	okx ( x509_check_issuer ( crt->cert, issuer->cert ) != 0,
	      file, line );
}
#define x509_check_issuer_fail_ok( crt, issuer ) \
	x509_check_issuer_fail_okx ( crt, issuer, __FILE__, __LINE__ )

/**
 * Report certificate root validation test result
 *
 * @v crt		Test certificate
 * @v root		Test root certificate store
 * @v file		Test code file
 * @v line		Test code line
 */
static void x509_check_root_okx ( struct x509_test_certificate *crt,
				  struct x509_root *root, const char *file,
				  unsigned int line ) {

	okx ( x509_check_root ( crt->cert, root ) == 0, file, line );
}
#define x509_check_root_ok( crt, root ) \
	x509_check_root_okx ( crt, root, __FILE__, __LINE__ )

/**
 * Report certificate root validation failure test result
 *
 * @v crt		Test certificate
 * @v root		Test root certificate store
 * @v file		Test code file
 * @v line		Test code line
 */
static void x509_check_root_fail_okx ( struct x509_test_certificate *crt,
				       struct x509_root *root,
				       const char *file, unsigned int line ) {

	okx ( x509_check_root ( crt->cert, root ) != 0, file, line );
}
#define x509_check_root_fail_ok( crt, root ) \
	x509_check_root_fail_okx ( crt, root, __FILE__, __LINE__ )

/**
 * Report certificate time validation test result
 *
 * @v crt		Test certificate
 * @v time		Test time
 * @v file		Test code file
 * @v line		Test code line
 */
static void x509_check_time_okx ( struct x509_test_certificate *crt,
				  time_t time, const char *file,
				  unsigned int line ) {

	okx ( x509_check_time ( crt->cert, time ) == 0, file, line );
}
#define x509_check_time_ok( crt, time ) \
	x509_check_time_okx ( crt, time, __FILE__, __LINE__ )

/**
 * Report certificate time validation failure test result
 *
 * @v crt		Test certificate
 * @v time		Test time
 * @v file		Test code file
 * @v line		Test code line
 */
static void x509_check_time_fail_okx ( struct x509_test_certificate *crt,
				       time_t time, const char *file,
				       unsigned int line ) {

	okx ( x509_check_time ( crt->cert, time ) != 0, file, line );
}
#define x509_check_time_fail_ok( crt, time ) \
	x509_check_time_fail_okx ( crt, time, __FILE__, __LINE__ )

/**
 * Report certificate name validation test result
 *
 * @v crt		Test certificate
 * @v name		Test name
 * @v file		Test code file
 * @v line		Test code line
 */
static void x509_check_name_okx ( struct x509_test_certificate *crt,
				  const char *name, const char *file,
				  unsigned int line ) {

	okx ( x509_check_name ( crt->cert, name ) == 0, file, line );
}
#define x509_check_name_ok( crt, name ) \
	x509_check_name_okx ( crt, name, __FILE__, __LINE__ )

/**
 * Report certificate name validation failure test result
 *
 * @v crt		Test certificate
 * @v name		Test name
 * @v file		Test code file
 * @v line		Test code line
 */
static void x509_check_name_fail_okx ( struct x509_test_certificate *crt,
				       const char *name, const char *file,
				       unsigned int line ) {

	okx ( x509_check_name ( crt->cert, name ) != 0, file, line );
}
#define x509_check_name_fail_ok( crt, name ) \
	x509_check_name_fail_okx ( crt, name, __FILE__, __LINE__ )

/**
 * Report certificate chain parsing test result
 *
 * @v chn		Test certificate chain
 * @v file		Test code file
 * @v line		Test code line
 */
static void x509_chain_okx ( struct x509_test_chain *chn, const char *file,
			     unsigned int line ) {
	unsigned int i;
	struct x509_certificate *first;

	chn->chain = x509_alloc_chain();
	okx ( chn->chain != NULL, file, line );
	for ( i = 0 ; i < chn->count ; i++ ) {
		okx ( x509_append ( chn->chain, chn->certs[i]->cert ) == 0,
		      file, line );
	}
	first = x509_first ( chn->chain );
	okx ( first != NULL, file, line );
	okx ( first->raw.len == chn->certs[0]->len, file, line );
	okx ( memcmp ( first->raw.data, chn->certs[0]->data,
		       first->raw.len ) == 0, file, line );
}
#define x509_chain_ok( chn ) \
	x509_chain_okx ( chn, __FILE__, __LINE__ )

/**
 * Report certificate chain validation test result
 *
 * @v chn		Test certificate chain
 * @v time		Test certificate validation time
 * @v store		Test certificate store
 * @v root		Test root certificate list
 * @v file		Test code file
 * @v line		Test code line
 */
static void x509_validate_chain_okx ( struct x509_test_chain *chn, time_t time,
				      struct x509_chain *store,
				      struct x509_root *root, const char *file,
				      unsigned int line ) {

	x509_invalidate_chain ( chn->chain );
	okx ( x509_validate_chain ( chn->chain, time, store, root ) == 0,
	      file, line );
}
#define x509_validate_chain_ok( chn, time, store, root ) \
	x509_validate_chain_okx ( chn, time, store, root, __FILE__, __LINE__ )

/**
 * Report certificate chain validation failure test result
 *
 * @v chn		Test certificate chain
 * @v time		Test certificate validation time
 * @v store		Test certificate store
 * @v root		Test root certificate list
 * @v file		Test code file
 * @v line		Test code line
 */
static void x509_validate_chain_fail_okx ( struct x509_test_chain *chn,
					   time_t time,
					   struct x509_chain *store,
					   struct x509_root *root,
					   const char *file,
					   unsigned int line ) {

	x509_invalidate_chain ( chn->chain );
	okx ( x509_validate_chain ( chn->chain, time, store, root ) != 0,
	      file, line );
}
#define x509_validate_chain_fail_ok( chn, time, store, root )		\
	x509_validate_chain_fail_okx ( chn, time, store, root,		\
				       __FILE__, __LINE__ )

/**
 * Perform X.509 self-tests
 *
 */
static void x509_test_exec ( void ) {

	/* Parse all certificates */
	x509_certificate_ok ( &root_crt );
	x509_certificate_ok ( &intermediate_crt );
	x509_certificate_ok ( &leaf_crt );
	x509_certificate_ok ( &useless_crt );
	x509_certificate_ok ( &server_crt );
	x509_certificate_ok ( &not_ca_crt );
	x509_certificate_ok ( &bad_path_len_crt );

	/* Check cache functionality */
	x509_cached_ok ( &root_crt );
	x509_cached_ok ( &intermediate_crt );
	x509_cached_ok ( &leaf_crt );
	x509_cached_ok ( &useless_crt );
	x509_cached_ok ( &server_crt );
	x509_cached_ok ( &not_ca_crt );
	x509_cached_ok ( &bad_path_len_crt );

	/* Check all certificate fingerprints */
	x509_fingerprint_ok ( &root_crt );
	x509_fingerprint_ok ( &intermediate_crt );
	x509_fingerprint_ok ( &leaf_crt );
	x509_fingerprint_ok ( &useless_crt );
	x509_fingerprint_ok ( &server_crt );
	x509_fingerprint_ok ( &not_ca_crt );
	x509_fingerprint_ok ( &bad_path_len_crt );

	/* Check pairwise issuing */
	x509_check_issuer_ok ( &intermediate_crt, &root_crt );
	x509_check_issuer_ok ( &leaf_crt, &intermediate_crt );
	x509_check_issuer_ok ( &useless_crt, &leaf_crt );
	x509_check_issuer_ok ( &server_crt, &leaf_crt );
	x509_check_issuer_fail_ok ( &not_ca_crt, &server_crt );
	x509_check_issuer_ok ( &bad_path_len_crt, &useless_crt );

	/* Check root certificate stores */
	x509_check_root_ok ( &root_crt, &test_root );
	x509_check_root_fail_ok ( &intermediate_crt, &test_root );
	x509_check_root_ok ( &intermediate_crt, &intermediate_root );
	x509_check_root_fail_ok ( &root_crt, &intermediate_root );
	x509_check_root_fail_ok ( &root_crt, &dummy_root );

	/* Check certificate validity periods */
	x509_check_time_ok ( &server_crt, test_time );
	x509_check_time_fail_ok ( &server_crt, test_expired );
	x509_check_time_ok ( &root_crt, test_time );
	x509_check_time_ok ( &root_crt, test_expired );
	x509_check_time_fail_ok ( &root_crt, test_ca_expired );

	/* Check certificate names */
	x509_check_name_ok ( &server_crt, "boot.test.ipxe.org" );
	x509_check_name_ok ( &server_crt, "demo.test.ipxe.org" );
	x509_check_name_fail_ok ( &server_crt, "incorrect.test.ipxe.org" );
	x509_check_name_ok ( &server_crt, "anything.alt.test.ipxe.org" );
	x509_check_name_ok ( &server_crt, "wildcard.alt.test.ipxe.org" );
	x509_check_name_fail_ok ( &server_crt, "sub.domain.alt.test.ipxe.org" );
	x509_check_name_fail_ok ( &server_crt, "alt.test.ipxe.org" );
	x509_check_name_fail_ok ( &server_crt, "test.ipxe.org" );
	x509_check_name_fail_ok ( &server_crt, "ipxe.org" );
	x509_check_name_fail_ok ( &server_crt, "org" );
	x509_check_name_fail_ok ( &server_crt, "" );

	/* Parse all certificate chains */
	x509_chain_ok ( &server_chain );
	x509_chain_ok ( &broken_server_chain );
	x509_chain_ok ( &incomplete_server_chain );
	x509_chain_ok ( &not_ca_chain );
	x509_chain_ok ( &useless_chain );
	x509_chain_ok ( &bad_path_len_chain );

	/* Check certificate chains */
	x509_validate_chain_ok ( &server_chain, test_time,
				 &empty_store, &test_root );
	x509_validate_chain_ok ( &server_chain, test_time,
				 &empty_store, &intermediate_root );
	x509_validate_chain_fail_ok ( &server_chain, test_time,
				      &empty_store, &dummy_root );
	x509_validate_chain_fail_ok ( &broken_server_chain, test_time,
				      &empty_store, &test_root );
	x509_validate_chain_fail_ok ( &incomplete_server_chain, test_time,
				      &empty_store, &test_root );
	x509_validate_chain_ok ( &incomplete_server_chain, test_time,
				 &empty_store, &intermediate_root );
	x509_validate_chain_fail_ok ( &not_ca_chain, test_time,
				      &empty_store, &test_root );
	x509_validate_chain_ok ( &useless_chain, test_time,
				 &empty_store, &test_root );
	x509_validate_chain_fail_ok ( &bad_path_len_chain, test_time,
				      &empty_store, &test_root );

	/* Check certificate chain expiry times */
	x509_validate_chain_fail_ok ( &server_chain, test_expired,
				      &empty_store, &test_root );
	x509_validate_chain_ok ( &useless_chain, test_expired,
				 &empty_store, &test_root );
	x509_validate_chain_fail_ok ( &useless_chain, test_ca_expired,
				      &empty_store, &test_root );

	/* Sanity check */
	assert ( list_empty ( &empty_store.links ) );

	/* Drop chain references */
	x509_chain_put ( bad_path_len_chain.chain );
	x509_chain_put ( useless_chain.chain );
	x509_chain_put ( not_ca_chain.chain );
	x509_chain_put ( incomplete_server_chain.chain );
	x509_chain_put ( broken_server_chain.chain );
	x509_chain_put ( server_chain.chain );

	/* Drop certificate references */
	x509_put ( bad_path_len_crt.cert );
	x509_put ( not_ca_crt.cert );
	x509_put ( server_crt.cert );
	x509_put ( useless_crt.cert );
	x509_put ( leaf_crt.cert );
	x509_put ( intermediate_crt.cert );
	x509_put ( root_crt.cert );
}

/** X.509 self-test */
struct self_test x509_test __self_test = {
	.name = "x509",
	.exec = x509_test_exec,
};

/* Drag in algorithms required for tests */
REQUIRE_OBJECT ( rsa );
REQUIRE_OBJECT ( sha1 );
REQUIRE_OBJECT ( sha256 );
