      SUBROUTINE IMG1_GKEYR( NCARD, BUFFER, SCARD, NAME, NOCCUR,
     :                         THERE, VALUE, CARD, STATUS )
*+
*  Name:
*     IMG1_GKEYx

*  Purpose:
*     Get the value of a named header of type REAL from a buffer of
*     FITS header card images.

*  Language:
*     Starlink Fortran 77

*  Invocation:
*     CALL IMG1_GKEYx( NCARD, BUFFER, SCARD, NAME, NOCCUR, THERE,
*                      VALUE, CARD, STATUS )

*  Description:
*     This routine searches a buffer containing the header card images
*     from a FITS file for a keyword NAME; and returns its value, and
*     the number of the card image within the buffer array that contains
*     the named keyword.  The search ends when the next end of a header
*     block, marked by the END keyword, is encountered or the buffer is
*     exhausted.  If the keyword is present THERE is true, otherwise it
*     is false.   Since all images are in character format, type
*     conversion is performed.  An error status will be returned if the
*     conversion has failed.  If the keyword expected to be present more
*     than once then the argument NOCCUR controls which occurrece will
*     be retrieved. If a keyword is not found then no error results and
*     the argument VALUE remains unmodified.
*
*     The name may be compound to permit reading of hierarchical
*     keywords.  This routine will probably only work for HISTORY,
*     COMMENT and ' ' (blank) if there is just one value given on the
*     line, i.e. only one "keyword = value" before any comment
*     marker. An error will result otherwise.

*  Arguments:
*     NCARD = INTEGER (Given)
*        The number of card images in the buffer.
*     BUFFER( NCARD ) = CHARACTER * 80 (Given)
*        The buffer containing the header card images.
*     SCARD = INTEGER (Given)
*        The number of the card from where the search will begin.  This
*        is needed because the headers may contain a dummy header
*        prior to an extension.
*     NAME = CHARACTER * ( * ) (Given)
*        The name of the keyword whose value is required.  This may be
*        a compound name to handle hierarchical keywords, and it has
*        the form keyword1.keyword2.keyword3 etc.  The maximum number of
*        keywords per FITS card is 20.  Comparisons are performed in
*        uppercase and blanks are removed.  Each keyword must be no
*        longer than 8 characters.
*     NOCCUR = INTEGER (Given)
*        The value of this argument specifies which occurence of a
*        keyword should be used, if, multiple ones are expected. (Any
*        value less then equal to 1 indicates the first occurrence)
*     THERE = LOGICAL (Returned)
*        If true the keyword NAME is present (regardless of exit
*        status).
*     VALUE = REAL (Returned)
*        The value of the keyword.
*     CARD = INTEGER (Returned)
*        The number of the card containing the named keyword.  If the
*        card could not be found this is set to the position of the
*        last card or the 'END' card..
*     STATUS = INTEGER (Given)
*        Global status value.

*  Notes:
*     - There is a routine for each of the data types logical, integer,
*     real and double precision: replace "x" in the routine name by
*     L, I, R or D as appropriate.

*  Authors:
*     MJC: Malcolm J. Currie (STARLINK)
*     PDRAPER: Peter Draper (STARLINK - Durham University)
*     {enter_new_authors_here}

*  History:
*     1988 Sep 22 (MJC):
*        Original version.
*     1989 Jul 28 (MJC):
*        Added extra argument CARD.
*     1990 November 19 (MJC):
*        Renamed from IMG1_GKEY, and converted the prologue to the SST
*        style.
*     1991 February 28 (MJC):
*        Converted BUFFER from an assumed-size to an adjustable array
*        via the NCARD argument in case the END keyword is missing.
*     1991 July 14 (MJC):
*        Modified to handle hierarchical keywords.
*     1991 August 31 (MJC):
*        Fixed bug in comparing card keyword with the selected keyword.
*     20-JUL-1994 (PDRAPER):
*        Incorporated into IMG_ (was called FTS1_GKEYX in KAPPA). Beefed
*        up the error reports and changed returned status to IMG codes.
*     12-SEP-1994 (PDRAPER):
*        Added ability to deal with keywords with more than one
*        occurrence.
*     {enter_further_changes_here}

*  Bugs:
*     None known.
*     {note_new_bugs_here}

*-

*  Type Definitions:
      IMPLICIT  NONE           ! no default typing allowed

*  Global Constants:
      INCLUDE  'SAE_PAR'       ! SSE global definitions
      INCLUDE  'IMG_ERR'       ! IMG_ error codes

*  Arguments Given:
      INTEGER NCARD
      INTEGER SCARD
      CHARACTER * ( * ) BUFFER( NCARD )
      CHARACTER * ( * ) NAME
      INTEGER NOCCUR

*  Arguments Returned:
      LOGICAL THERE
      REAL VALUE
      INTEGER CARD

*  Status:
      INTEGER STATUS

*  External References:
      EXTERNAL CHR_LEN
      INTEGER CHR_LEN            ! Number of characters in a string
                                 ! ignoring trailing blanks

*  Local Constants:
      INTEGER MXWORD             ! Maximum number of hierarchical levels
                                 ! in a keyword
      PARAMETER ( MXWORD = 20 )

*  Local Variables:
      CHARACTER * ( 60 ) CHRVAL  ! The value in characters
      CHARACTER * ( 72 ) CMPKEY  ! Compound name
      CHARACTER * ( 8 )  WORDS( MXWORD ) ! The keywords in the current
                                         ! card image
      CHARACTER * ( 80 ) KEYWRD  ! The compound keyword
      INTEGER COMMNT             ! Column number containing the comment
                                 ! character in the current card image
      INTEGER ENDW( MXWORD )     ! End columns of each keyword in a card
                                 ! image
      INTEGER EQUALS             ! Column number containing the first
                                 ! equals sign in the current card image
      INTEGER I                  ! Loop counter
      INTEGER ISTAT              ! Local status
      INTEGER NF                 ! Number of occurrences found
      INTEGER NC                 ! Number of characters
      INTEGER NCHV               ! Number of characters in the value
      INTEGER NCK                ! Number of characters in the compound
                                 ! keyword
      INTEGER NWORD              ! Number of keywords in the current
                                 ! card image
      INTEGER STARTW( MXWORD )   ! Start columns of each keyword in a
                                 ! card image
      LOGICAL COMPND             ! Supplied name is compound

*.


*  Check for error on entry.
      IF ( STATUS .NE. SAI__OK ) RETURN

*  Initialise some variables.
      CARD = MAX( 1, SCARD )
      THERE = .FALSE.
      NF = 0

*  Remove blanks from the keyword to be searched for, and make it
*  uppercase for comparisons.  Find its effective length.
      KEYWRD = NAME
      CALL CHR_UCASE( KEYWRD )
      CALL CHR_RMBLK( KEYWRD )
      NC = CHR_LEN( KEYWRD )

*  Is it a compound name?
      COMPND = INDEX( KEYWRD, '.' ) .NE. 0

*  The simple case.
*  ================
      IF ( .NOT. COMPND ) THEN

*  Name is limited to 8 characters.
        NC = MIN( NC, 8 )

*  Now loop through the cards.  Compare the keyword on each word with
*  the given keyword, until the required card is found, or the 'END'
*  card is met, or there are no cards remaining.
 1       CONTINUE                ! Start of 'DO WHILE' loop
         IF ( ( .NOT. THERE ) .AND.
     :        ( CARD .LE. NCARD ) .AND.
     :        ( BUFFER( MIN( NCARD, CARD ) )( :3 ) .NE. 'END' ) ) THEN

*  Does the current card contain the required keyword?
            IF ( BUFFER( CARD )( :8 ) .EQ. KEYWRD( :NC ) ) THEN

*  The keyword is present. Is this the occurrence we require?
               NF = NF + 1
               IF ( NF .GE. NOCCUR ) THEN

*  Yes it is.
                  THERE = .TRUE.
                  CHRVAL = BUFFER( CARD )( 11:30 )

*  Remove leading blanks and get number of characters comprising the
*  value.
                  CALL CHR_LDBLK( CHRVAL )
                  NCHV = CHR_LEN( CHRVAL )

*  Convert the string to the correct type. Trap the occasion when the
*  value is a blank string (this may be true for COMMENT and HISTORY
*  records) and replace CHR conversion error with IMG equivalent.
                  IF ( NCHV .GT. 0 ) THEN
                     CALL ERR_MARK
                     CALL CHR_CTOR( CHRVAL( :NCHV ), VALUE, STATUS )
                     IF ( STATUS .NE. SAI__OK ) THEN
                        CALL ERR_ANNUL( STATUS )
                        STATUS = IMG__CONER
                     END IF
                     CALL ERR_RLSE
                  ELSE
                     STATUS = IMG__CONER
                  END IF

*  Report the error context.
                  IF ( STATUS .EQ. IMG__CONER ) THEN
                     CALL MSG_SETC( 'NAME', NAME( :NC ) )
                     CALL MSG_SETC( 'VALUE', CHRVAL( :NCHV ) )
                     CALL ERR_REP( 'IMG1_GKEYR_TYPCNV',
     :                    'Error converting FITS item ^NAME (with ' //
     :                    'value ^VALUE) to type REAL.', STATUS )
                  END IF
               ELSE

*  Onto the next card in the buffer.
                  CARD = CARD + 1
               END IF
            ELSE

*  Onto the next card in the buffer.
               CARD = CARD + 1
            END IF

*  Next 'WHILE'.
            GO TO 1
         END IF

*  Hierarchical-keyword case.
*  ==========================
      ELSE

*  Now loop through the cards ('END' terminates header).
 2       CONTINUE                ! Start of 'DO WHILE' loop
         IF ( ( .NOT. THERE ) .AND.
     :        ( CARD .LE. NCARD ) .AND.
     :        ( BUFFER( MIN( NCARD, CARD ) )( :3 ) .NE. 'END' ) ) THEN

*  Does the current card have a value, i.e. an equals sign. This is
*  not foolproof because of the ING format et al. uses blank fields,
*  comments and history to store data. Fortunately these are not likely
*  to be compound names.
            EQUALS = INDEX( BUFFER( CARD ), '=' )
            IF ( EQUALS .NE. 0 ) THEN

*  Extract the words from the FITS card image up to the
*  equals sign, assuming these to be keywords.
               CALL CHR_DCWRD( BUFFER( CARD )( :EQUALS-1 ), MXWORD,
     :                         NWORD, STARTW, ENDW, WORDS, ISTAT )

*  Form compound name if there is more than one supposed
*  keyword by concatenating the words via the delimeter.
               IF ( NWORD .GT. 1 ) THEN
                  NCK = 0
                  CMPKEY = ' '
                  DO I = 1, NWORD
                     NC = ENDW( I ) - STARTW( I ) + 1
                     CALL CHR_PUTC( WORDS( I )( :NC ), CMPKEY, NCK )
                     IF ( I .NE. NWORD )
     :                    CALL CHR_PUTC( '.', CMPKEY, NCK )
                  END DO

*  Merely copy the first keyword.
               ELSE
                  CMPKEY = WORDS( 1 )
                  NCK = ENDW( 1 ) - STARTW( 1 ) + 1
               END IF

*  Compare the (compound) keyword of the current card image with that
*  of the compound keyword be searched for in the buffer.
               IF ( CMPKEY( :NCK ) .EQ. KEYWRD( :NCK ) ) THEN

*  Have we located enough occurrences?
                  NF = NF + 1
                  IF ( NF .GE. NOCCUR ) THEN

*  The keyword is present.
                     THERE = .TRUE.

*  Find the upper range of columns that contains the value associated
*  with the hierarchical keyword.  This is achieved by looking for the
*  comment character.  This was not necessary for the the simple case
*  because the format is rigid.  Hierarchical keywords are non-standard
*  so it pays to be flexible.  Allow for the offset of the equals sign
*  when finding the column that last contains part of the value.
                     COMMNT = INDEX( BUFFER( CARD )( EQUALS + 1: ), '/')
                     IF ( COMMNT .EQ. 0 ) THEN
                        COMMNT = LEN( BUFFER( CARD ) )
                     ELSE
                        COMMNT = COMMNT + EQUALS - 1
                     END IF

*  Extract the value as a character string.
                     CHRVAL = BUFFER( CARD )( EQUALS + 1:COMMNT )

*  Remove leading blanks and get number of characters comprising the
*  value.
                     CALL CHR_LDBLK( CHRVAL )
                     NCHV = CHR_LEN( CHRVAL )

*  Convert the string to the correct type. Trap the occasion when the
*  value is a blank string (this may be true for COMMENT and HISTORY
*  records) and replace CHR conversion error with IMG equivalent.
                     IF ( NCHV .GT. 0 ) THEN
                        CALL ERR_MARK
                        CALL CHR_CTOR( CHRVAL( :NCHV ), VALUE,
     :                                   STATUS )
                        IF ( STATUS .NE. SAI__OK ) THEN
                           CALL ERR_ANNUL( STATUS )
                           STATUS = IMG__CONER
                        END IF
                        CALL ERR_RLSE
                     ELSE
                        STATUS = IMG__CONER
                     END IF

*  Report the error context.
                     IF ( STATUS .EQ. IMG__CONER ) THEN
                        CALL MSG_SETC( 'NAME', NAME( :NCK ) )
                        CALL MSG_SETC( 'VALUE', CHRVAL( :NCHV ) )
                        CALL ERR_REP( 'IMG1_GKEYR_TYPCNV',
     :                       'Error converting FITS item ^NAME ' //
     :                       '(with value ^VALUE) to type REAL.',
     :                       STATUS )
                     END IF
                  ELSE

*  Onto the next card in the buffer.
                     CARD = CARD + 1
                  END IF
               ELSE

*  Onto the next card in the buffer.
                  CARD = CARD + 1
               END IF
            ELSE

*  Onto the next card in the buffer.
               CARD = CARD + 1
            END IF

*  Next 'WHILE'.
            GO TO 2
         END IF
      END IF
      END
* $Id: img1_gkeyx.gsd,v 1.1 1998/06/03 13:13:57 pdraper Exp $
