/*
*class++
*  Name:
*     LutMap

*  Purpose:
*     Transform 1-dimensional coordinates using a lookup table.

*  Constructor Function:
c     astLutMap
f     AST_LUTMAP

*  Description:
*     A LutMap is a specialised form of Mapping which transforms
*     1-dimensional coordinates by using linear interpolation in a
*     lookup table.
*
*     Each input coordinate value is first scaled to give the index of
*     an entry in the table by subtracting a starting value (the input
*     coordinate corresponding to the first table entry) and dividing
*     by an increment (the difference in input coordinate value
*     between adjacent table entries).
*
*     The resulting index will usually contain a fractional part, so
*     the output coordinate value is then generated by interpolating
*     linearly between the appropriate entries in the table. If the
*     index lies outside the range of the table, linear extrapolation
*     is used based on the two nearest entries (i.e. the two entries
*     at the start or end of the table, as appropriate). If either of the
*     entries used for the interplation has a value of AST__BAD, then the 
*     interpolated value is returned as AST__BAD.
*
*     If the lookup table entries increase or decrease monotonically (and
*     if the table contains no AST__BAD values), then the inverse 
*     transformation may also be performed.

*  Inheritance:
*     The LutMap class inherits from the Mapping class.

*  Attributes:
*     The LutMap class does not define any new attributes beyond
*     those which are applicable to all Mappings.

*  Functions:
c     The LutMap class does not define any new functions beyond those
f     The LutMap class does not define any new routines beyond those
*     which are applicable to all Mappings.

*  Copyright:
*     Copyright (C) 2004 Central Laboratory of the Research Councils

*  Authors:
*     RFWS: R.F. Warren-Smith (Starlink)

*  History:
*     8-JUL-1997 (RFWS):
*        Original version.
*     10-JUL-1997 (RFWS):
*        Added the MapMerge function.
*     8-JAN-2003 (DSB):
*        Changed private InitVtab method to protected astInitLutMapVtab
*        method.
*     12-JAN-2004 (DSB):
*        Check for AST__BAD values in the supplied lut array.
*class--
*/

/* Module Macros. */
/* ============== */
/* Set the name of the class we are implementing. This indicates to
   the header files that define class interfaces that they should make
   "protected" symbols available. */
#define astCLASS LutMap

/* Include files. */
/* ============== */
/* Interface definitions. */
/* ---------------------- */
#include "error.h"               /* Error reporting facilities */
#include "memory.h"              /* Memory management facilities */
#include "object.h"              /* Base Object class */
#include "pointset.h"            /* Sets of points/coordinates */
#include "mapping.h"             /* Coordinate mappings (parent class) */
#include "winmap.h"              /* Linear mappings between windows */
#include "channel.h"             /* I/O channels */
#include "lutmap.h"              /* Interface definition for this class */

/* Error code definitions. */
/* ----------------------- */
#include "ast_err.h"             /* AST error codes */

/* C header files. */
/* --------------- */
#include <float.h>
#include <math.h>
#include <stdarg.h>
#include <stddef.h>
#include <stdio.h>
#include <string.h>

/* Module Variables. */
/* ================= */
/* Define the class virtual function table and its initialisation flag
   as static variables. */
static AstLutMapVtab class_vtab; /* Virtual function table */
static int class_init = 0;       /* Virtual function table initialised? */

/* Pointers to parent class methods which are extended by this class. */
static AstPointSet *(* parent_transform)( AstMapping *, AstPointSet *, int, AstPointSet * );

/* External Interface Function Prototypes. */
/* ======================================= */
/* The following functions have public prototypes only (i.e. no
   protected prototypes), so we must provide local prototypes for use
   within this module. */
AstLutMap *astLutMapId_( int, const double [], double, double, const char *, ... );

/* Prototypes for Private Member Functions. */
/* ======================================== */
static AstPointSet *Transform( AstMapping *, AstPointSet *, int, AstPointSet * );
static int GetLinear( AstMapping * );
static int MapMerge( AstMapping *, int, int, int *, AstMapping ***, int ** );
static void Copy( const AstObject *, AstObject * );
static void Delete( AstObject * );
static void Dump( AstObject *, AstChannel * );

/* Member functions. */
/* ================= */
static int GetLinear( AstMapping *this_mapping ) {
/*
*  Name:
*     GetLinear

*  Purpose:
*     Determine if a LutMap implements a linear coordinate transformation.

*  Type:
*     Private function.

*  Synopsis:
*     #include "lutmap.h"
*     int GetLinear( AstMapping *this )

*  Class Membership:
*     LutMap member function.

*  Description:
*     This function returns a boolean value to indicate if the LutMap
*     supplied is equivalent to a linear coordinate
*     transformation. This will be the case if the lookup table
*     elements increase or decrease linearly.

*  Parameters:
*     this
*        Pointer to the LutMap.

*  Notes:
*    - A value of zero will be returned if this function is invoked
*    with the global error status set, or if it should fail for any
*    reason.
*/

/* Local Variables: */
   AstLutMap *this;              /* Pointer to the LutMap structure */
   double *lut;                  /* Pointer to the lookup table */
   double fract;                 /* Fractional position within table */
   double hi;                    /* Largest value */
   double interp;                /* Interpolated value */
   double lo;                    /* Smallest value */
   double tol1;                  /* First tolerance estimate */
   double tol2;                  /* Second tolerance estimate */
   double tol;                   /* Tolerance value used */
   int ilut;                     /* Loop counter for table elements */
   int linear;                   /* Result to be returned */
   int nlut;                     /* Number of lookup table elements */

/* Initialise. */
   linear = 0;

/* Check the global error status. */
   if ( !astOK ) return linear;

/* Obtain a pointer to the LutMap structure. */
   this = (AstLutMap *) this_mapping;

/* Obtain the lookup table details. */
   lut = this->lut;
   nlut = this->nlut;

/* Loop to identify the largest and smallest values in the lookup
   table. */
   lo = DBL_MAX;
   hi = -DBL_MAX;
   for ( ilut = 0; ilut < nlut; ilut++ ) {
      if ( lut[ ilut ] > hi ) hi = lut[ ilut ];
      if ( lut[ ilut ] < lo ) lo = lut[ ilut ];
   }

/* Check if the values are all the same (this makes the LutMap
   linear, although it will have no inverse). */
   linear = ( hi == lo );
   if ( !linear ) {

/* Form a tolerance estimate based on the overall range of values in
   the lookup table. */
      tol1 = fabs( hi - lo ) * DBL_EPSILON;

/* Now loop to inspect all the lookup table elements except the first
   and last. */
      linear = 1;
      for ( ilut = 1; ilut < ( nlut - 1 ); ilut++ ) {

/* Calculate the fractional position of the current element within the
   table. */
         fract = ( (double) ilut ) / ( (double) ( nlut - 1 ) );

/* Calculate the value it should have if the table is linear by
   interpolating between the first and last values. */
         interp = lut[ 0 ] * ( 1.0 - fract ) + lut[ nlut - 1 ] * fract;

/* Form a second tolerance estimate from this interpolated
   value. Select whichever tolerance estimate is larger (this avoids
   problems when values are near zero). */
         tol2 = fabs( interp ) * DBL_EPSILON;
         tol = ( tol1 > tol2 ) ? tol1 : tol2;

/* Test for linearity within a small multiple of the tolerance. */
         linear = ( fabs( lut[ ilut ] - interp ) <= ( 2.0 * tol ) );
         if ( !linear ) break;
      }
   }

/* Return the result. */
   return linear;
}

void astInitLutMapVtab_(  AstLutMapVtab *vtab, const char *name ) {
/*
*+
*  Name:
*     astInitLutMapVtab

*  Purpose:
*     Initialise a virtual function table for a LutMap.

*  Type:
*     Protected function.

*  Synopsis:
*     #include "lutmap.h"
*     void astInitLutMapVtab( AstLutMapVtab *vtab, const char *name )

*  Class Membership:
*     LutMap vtab initialiser.

*  Description:
*     This function initialises the component of a virtual function
*     table which is used by the LutMap class.

*  Parameters:
*     vtab
*        Pointer to the virtual function table. The components used by
*        all ancestral classes will be initialised if they have not already
*        been initialised.
*     name
*        Pointer to a constant null-terminated character string which contains
*        the name of the class to which the virtual function table belongs (it 
*        is this pointer value that will subsequently be returned by the Object
*        astClass function).
*-
*/

/* Local Variables: */
   AstMappingVtab *mapping;      /* Pointer to Mapping component of Vtab */

/* Check the local error status. */
   if ( !astOK ) return;

/* Initialize the component of the virtual function table used by the
   parent class. */
   astInitMappingVtab( (AstMappingVtab *) vtab, name );

/* Store a unique "magic" value in the virtual function table. This
   will be used (by astIsALutMap) to determine if an object belongs
   to this class.  We can conveniently use the address of the (static)
   class_init variable to generate this unique value. */
   vtab->check = &class_init;

/* Initialise member function pointers. */
/* ------------------------------------ */
/* Store pointers to the member functions (implemented here) that
   provide virtual methods for this class. */

/* Save the inherited pointers to methods that will be extended, and
   replace them with pointers to the new member functions. */
   mapping = (AstMappingVtab *) vtab;

   parent_transform = mapping->Transform;
   mapping->Transform = Transform;

/* Store replacement pointers for methods which will be over-ridden by
   new member functions implemented here. */
   mapping->MapMerge = MapMerge;

/* Declare the class dump, copy and delete functions.*/
   astSetDump( vtab, Dump, "LutMap",
               "Map 1-d coordinates using a lookup table" );
   astSetCopy( (AstObjectVtab *) vtab, Copy );
   astSetDelete( (AstObjectVtab *) vtab, Delete );
}

static int MapMerge( AstMapping *this, int where, int series, int *nmap,
                     AstMapping ***map_list, int **invert_list ) {
/*
*  Name:
*     MapMerge

*  Purpose:
*     Simplify a sequence of Mappings containing a LutMap.

*  Type:
*     Private function.

*  Synopsis:
*     #include "mapping.h"
*     int MapMerge( AstMapping *this, int where, int series, int *nmap,
*                   AstMapping ***map_list, int **invert_list )

*  Class Membership:
*     LutMap method (over-rides the protected astMapMerge method
*     inherited from the Mapping class).

*  Description:
*     This function attempts to simplify a sequence of Mappings by
*     merging a nominated LutMap in the sequence with its neighbours,
*     so as to shorten the sequence if possible.
*
*     In many cases, simplification will not be possible and the
*     function will return -1 to indicate this, without further
*     action.
*
*     In most cases of interest, however, this function will either
*     attempt to replace the nominated LutMap with one which it
*     considers simpler, or to merge it with the Mappings which
*     immediately precede it or follow it in the sequence (both will
*     normally be considered). This is sufficient to ensure the
*     eventual simplification of most Mapping sequences by repeated
*     application of this function.
*
*     In some cases, the function may attempt more elaborate
*     simplification, involving any number of other Mappings in the
*     sequence. It is not restricted in the type or scope of
*     simplification it may perform, but will normally only attempt
*     elaborate simplification in cases where a more straightforward
*     approach is not adequate.

*  Parameters:
*     this
*        Pointer to the nominated LutMap which is to be merged with
*        its neighbours. This should be a cloned copy of the LutMap
*        pointer contained in the array element "(*map_list)[where]"
*        (see below). This pointer will not be annulled, and the
*        LutMap it identifies will not be modified by this function.
*     where
*        Index in the "*map_list" array (below) at which the pointer
*        to the nominated LutMap resides.
*     series
*        A non-zero value indicates that the sequence of Mappings to
*        be simplified will be applied in series (i.e. one after the
*        other), whereas a zero value indicates that they will be
*        applied in parallel (i.e. on successive sub-sets of the
*        input/output coordinates).
*     nmap
*        Address of an int which counts the number of Mappings in the
*        sequence. On entry this should be set to the initial number
*        of Mappings. On exit it will be updated to record the number
*        of Mappings remaining after simplification.
*     map_list
*        Address of a pointer to a dynamically allocated array of
*        Mapping pointers (produced, for example, by the astMapList
*        method) which identifies the sequence of Mappings. On entry,
*        the initial sequence of Mappings to be simplified should be
*        supplied.
*
*        On exit, the contents of this array will be modified to
*        reflect any simplification carried out. Any form of
*        simplification may be performed. This may involve any of: (a)
*        removing Mappings by annulling any of the pointers supplied,
*        (b) replacing them with pointers to new Mappings, (c)
*        inserting additional Mappings and (d) changing their order.
*
*        The intention is to reduce the number of Mappings in the
*        sequence, if possible, and any reduction will be reflected in
*        the value of "*nmap" returned. However, simplifications which
*        do not reduce the length of the sequence (but improve its
*        execution time, for example) may also be performed, and the
*        sequence might conceivably increase in length (but normally
*        only in order to split up a Mapping into pieces that can be
*        more easily merged with their neighbours on subsequent
*        invocations of this function).
*
*        If Mappings are removed from the sequence, any gaps that
*        remain will be closed up, by moving subsequent Mapping
*        pointers along in the array, so that vacated elements occur
*        at the end. If the sequence increases in length, the array
*        will be extended (and its pointer updated) if necessary to
*        accommodate any new elements.
*
*        Note that any (or all) of the Mapping pointers supplied in
*        this array may be annulled by this function, but the Mappings
*        to which they refer are not modified in any way (although
*        they may, of course, be deleted if the annulled pointer is
*        the final one).
*     invert_list
*        Address of a pointer to a dynamically allocated array which,
*        on entry, should contain values to be assigned to the Invert
*        attributes of the Mappings identified in the "*map_list"
*        array before they are applied (this array might have been
*        produced, for example, by the astMapList method). These
*        values will be used by this function instead of the actual
*        Invert attributes of the Mappings supplied, which are
*        ignored.
*
*        On exit, the contents of this array will be updated to
*        correspond with the possibly modified contents of the
*        "*map_list" array.  If the Mapping sequence increases in
*        length, the "*invert_list" array will be extended (and its
*        pointer updated) if necessary to accommodate any new
*        elements.

*  Returned Value:
*     If simplification was possible, the function returns the index
*     in the "map_list" array of the first element which was
*     modified. Otherwise, it returns -1 (and makes no changes to the
*     arrays supplied).

*  Notes:
*     - A value of -1 will be returned if this function is invoked
*     with the global error status set, or if it should fail for any
*     reason.
*/

/* Local Variables: */
   AstLutMap *map;               /* Pointer to LutMap */
   AstMapping *new;              /* Pointer to replacement Mapping */
   double a1;                    /* First input coordinate value */
   double a2;                    /* Second input coordinate value */
   double b1;                    /* First output coordinate value */
   double b2;                    /* Second output coordinate value */
   int result;                   /* Result value to return */
   int simpler;                  /* Mapping simplified? */

/* Initialise the returned result. */
   result = -1;

/* Check the global error status. */
   if ( !astOK ) return result;

/* Obtain a pointer to the nominated LutMap. */
   map = (AstLutMap *) ( *map_list )[ where ];

/* See if the LutMap is linear. If so, it can probably be
   simplified. */
   simpler = GetLinear( (AstMapping *) map );
   if ( simpler ) {

/* Obtain the range of input values corresponding to the first and
   last lookup table elements. */
      a1 = map->start;
      a2 = map->start + map->inc * ( map->nlut - 1 );

/* Obtain the corresponding range of output values and check these
   values are not the same. */
      b1 = map->lut[ 0 ];
      b2 = map->lut[ map->nlut - 1 ];
      if ( b1 != b2 ) {

/* Create a new WinMap that implements an equivalent linear Mapping,
   allowing for the invert flag associated with the LutMap. */
         if ( !( *invert_list )[ where ] ) {
            new = (AstMapping *) astWinMap( 1, &a1, &a2, &b1, &b2, "" );
         } else {
            new = (AstMapping *) astWinMap( 1, &b1, &b2, &a1, &a2, "" );
         }

/* If OK, annul the original LutMap pointer and substitute the new
   one. Also clear the associated invert flag. */
         if ( astOK ) {
            (void) astAnnul( ( *map_list )[ where ] );
            ( *map_list )[ where ] = new;
            ( *invert_list )[ where ] = 0;

/* Assign the result value. */
            result = where;
         }
      }
   }

/* If an error occurred, clear the returned result. */
   if ( !astOK ) result = -1;

/* Return the result. */
   return result;
}

static AstPointSet *Transform( AstMapping *this, AstPointSet *in,
                               int forward, AstPointSet *out ) {
/*
*  Name:
*     Transform

*  Purpose:
*     Apply a LutMap to transform a set of points.

*  Type:
*     Private function.

*  Synopsis:
*     #include "lutmap.h"
*     AstPointSet *Transform( AstMapping *this, AstPointSet *in,
*                             int forward, AstPointSet *out )

*  Class Membership:
*     LutMap member function (over-rides the astTransform protected
*     method inherited from the Mapping class).

*  Description:
*     This function takes a LutMap and a set of points encapsulated
*     in a PointSet and transforms the points so as to apply the
*     lookup table transformation.

*  Parameters:
*     this
*        Pointer to the LutMap.
*     in
*        Pointer to the PointSet holding the input coordinate data.
*     forward
*        A non-zero value indicates that the forward coordinate
*        transformation should be applied, while a zero value requests
*        the inverse transformation.
*     out
*        Pointer to a PointSet which will hold the transformed
*        (output) coordinate values. A NULL value may also be given,
*        in which case a new PointSet will be created by this
*        function.

*  Returned Value:
*     Pointer to the output (possibly new) PointSet.

*  Notes:
*     - A null pointer will be returned if this function is invoked
*     with the global error status set, or if it should fail for any
*     reason.
*     - The number of coordinate values per point in the input
*     PointSet must equal 1.
*     - If an output PointSet is supplied, it must have space for
*     sufficient number of points (with 1 coordinate value per point)
*     to accommodate the result. Any excess space will be ignored.
*/

/* Local Variables: */
   AstLutMap *map;               /* Pointer to LutMap to be applied */
   AstPointSet *result;          /* Pointer to output PointSet */
   double **ptr_in;              /* Pointer to input coordinate data */
   double **ptr_out;             /* Pointer to output coordinate data */
   double *lut;                  /* Pointer to LUT */
   double fract;                 /* Fractional interpolation distance */
   double scale;                 /* Normalising scale factor */
   double value_in;              /* Input coordinate value */
   double value_out;             /* Output coordinate value */
   double x;                     /* Value normalised to LUT increment */
   double xi;                    /* Integer value of "x" */
   int i1;                       /* Lower adjacent LUT index */
   int i2;                       /* Upper adjacent LUT index */
   int i;                        /* New LUT index */
   int ix;                       /* "x" converted to an int */
   int nlut;                     /* Number of LUT entries */
   int npoint;                   /* Number of points */
   int point;                    /* Loop counter for points */
   int up;                       /* LUT values are increasing? */

/* Check the global error status. */
   if ( !astOK ) return NULL;

/* Obtain a pointer to the LutMap. */
   map = (AstLutMap *) this;

/* Apply the parent mapping using the stored pointer to the Transform
   member function inherited from the parent Mapping class. This
   function validates all arguments and generates an output PointSet
   if necessary, but does not actually transform any coordinate
   values. */
   result = (*parent_transform)( this, in, forward, out );

/* We will now extend the parent astTransform method by performing the
   calculations needed to generate the output coordinate values. */

/* Determine the numbers of points from the input PointSet and obtain
   pointers for accessing the input and output coordinate values. */
   npoint = astGetNpoint( in );
   ptr_in = astGetPoints( in );      
   ptr_out = astGetPoints( result );

/* Determine whether to apply the forward or inverse mapping,
   according to the direction specified and whether the mapping has
   been inverted. */
   if ( astGetInvert( map ) ) forward = !forward;

/* Obtain lookup table details. */
   lut = map->lut;
   nlut = map->nlut;

/* Forward transformation. */
/* ----------------------- */
   if( astOK ){
      if ( forward ) {

/* Calculate the scale factor required. */
         scale = 1.0 / map->inc;

/* Loop to transform each input point. */
         for ( point = 0; point < npoint; point++ ) {

/* Extract the input coordinate value. */
            value_in = ptr_in[ 0 ][ point ];

/* First check if this is the same value as we transformed last. If
   so, re-use the last result. */
            if ( value_in == map->last_fwd_in ) {
               value_out = map->last_fwd_out;

/* Check for bad input coordinates and generate a bad result if
   necessary. */
            } else if ( value_in == AST__BAD ) {
               value_out = AST__BAD;

/* Otherwise, identify the lookup table entry corresponding to the
   input coordinate. */
            } else {
               x = ( value_in - map->start ) * scale;
               xi = floor( x );
               ix = (int) xi;

/* If the input value lies below the first lookup table entry,
   extrapolate using the first two table values. */
               if ( ix < 0 ) {
                  if( lut[ 0 ] != AST__BAD && lut[ 1 ] != AST__BAD ) {
                     value_out = lut[ 0 ] + x * ( lut[ 1 ] - lut[ 0 ] );
                  } else {
                     value_out = AST__BAD;
                  }

/* If the input value lies above the last lookup table entry (or
   equals it), extrapolate using the last two table values. */
               } else if ( ix >= ( nlut - 1 ) ) {
                  if( lut[ nlut - 1 ] != AST__BAD && 
                      lut[ nlut - 2 ] != AST__BAD ) {
                     value_out = lut[ nlut - 1 ] +
                                 ( x - (double) ( nlut - 1 ) ) *
                                 ( lut[ nlut - 1 ] - lut[ nlut - 2 ] );
                  } else {
                     value_out = AST__BAD;
                  }

/* Otherwise, interpolate between the adjacent entries. */
               } else {
                  if( lut[ ix ] != AST__BAD && 
                      lut[ ix + 1 ] != AST__BAD ) {
                     fract = x - xi;
                     value_out = lut[ ix ] * ( 1.0 - fract ) +
                                 lut[ ix + 1 ] * fract;
                  } else {
                     value_out = AST__BAD;
                  }
               }
            }

/* Assign the output coordinate value. */
            ptr_out[ 0 ][ point ] = value_out;

/* Retain the input and output coordinate values for possible re-use
   in future. */
            map->last_fwd_in = value_in;
            map->last_fwd_out = value_out;
         }

/* Inverse transformation. */
/* ----------------------- */
      } else {

/* Loop to transform each input point. */
         for ( point = 0; point < npoint; point++ ) {

/* Extract the input coordinate value. */
            value_in = ptr_in[ 0 ][ point ];

/* First check if this is the same value as we transformed last. If
   so, re-use the last result. */
            if ( value_in == map->last_inv_in ) {
               value_out = map->last_inv_out;

/* Check for bad input coordinates and generate a bad result if
   necessary. */
            } else if ( value_in == AST__BAD ) {
               value_out = AST__BAD;

/* Otherwise, determine if the lookup table's elements are increasing
   or decreasing. (Note the inverse transformation will not be
   defined, so will not be attempted, unless all the table entries are
   monotonically increasing or decreasing.) */
            } else {
               up = ( lut[ nlut - 1 ] > lut[ 0 ] );

/* Perform a binary search to identify two adjacent lookup table
   elements whose values bracket the input coordinate value. */
               i1 = -1;
               i2 = nlut;
               while ( i2 > ( i1 + 1 ) ) {
                  i = ( i1 + i2 ) / 2;
                  *( ( ( value_in >= lut[ i ] ) == up ) ? &i1 : &i2 ) = i;
               }

/* We are interested in the lower bracketing table element. If
   necessary, restrict this element's index to lie within the
   table. This causes extrapolation to occur (instead of
   interpolation) if the input value actually lies outside the range
   of the lookup table. */
               if ( i1 < 0 ) i1 = 0;
               if ( i1 > ( nlut - 2 ) ) i1 = nlut - 2;

/* Interpolate (or extrapolate) to derive the output coordinate
   value. */
               value_out = map->start + map->inc *
                           ( (double) i1 + ( ( value_in - lut[ i1 ] ) /
                                             ( lut[ i1 + 1 ] - lut[ i1 ] ) ) );
            }

/* Assign the output coordinate value. */
            ptr_out[ 0 ][ point ] = value_out;

/* Retain the input and output coordinate values for possible re-use
   in future. */
            map->last_inv_in = value_in;
            map->last_inv_out = value_out;
         }
      }
   }

/* Return a pointer to the output PointSet. */
   return result;
}

/* Copy constructor. */
/* ----------------- */
static void Copy( const AstObject *objin, AstObject *objout ) {
/*
*  Name:
*     Copy

*  Purpose:
*     Copy constructor for LutMap objects.

*  Type:
*     Private function.

*  Synopsis:
*     void Copy( const AstObject *objin, AstObject *objout )

*  Description:
*     This function implements the copy constructor for LutMap objects.

*  Parameters:
*     objin
*        Pointer to the LutMap to be copied.
*     objout
*        Pointer to the LutMap being constructed.
*/

/* Local Variables: */
   AstLutMap *out;               /* Pointer to output LutMap */
   AstLutMap *in;                /* Pointer to input LutMap */

/* Check the global error status. */
   if ( !astOK ) return;

/* Obtain a pointer to the input and output LutMaps. */
   in= (AstLutMap *) objin;
   out = (AstLutMap *) objout;

/* Allocate memory and store a copy of the lookup table data. */
   out->lut = astStore( NULL, in->lut, 
                        sizeof( double ) * (size_t) in->nlut );
}

/* Destructor. */
/* ----------- */
static void Delete( AstObject *obj ) {
/*
*  Name:
*     Delete

*  Purpose:
*     Destructor for LutMap objects.

*  Type:
*     Private function.

*  Synopsis:
*     void Delete( AstObject *obj )

*  Description:
*     This function implements the destructor for LutMap objects.

*  Parameters:
*     obj
*        Pointer to the LutMap to be deleted.
*/

/* Local Variables: */
   AstLutMap *this;              /* Pointer to LutMap */

/* Obtain a pointer to the LutMap structure. */
   this = (AstLutMap *) obj;

/* Free the memory holding the lookup table. */
   if ( this->lut ) this->lut = astFree( this->lut );
}

/* Dump function. */
/* -------------- */
static void Dump( AstObject *this_object, AstChannel *channel ) {
/*
*  Name:
*     Dump

*  Purpose:
*     Dump function for LutMap objects.

*  Type:
*     Private function.

*  Synopsis:
*     void Dump( AstObject *this, AstChannel *channel )

*  Description:
*     This function implements the Dump function which writes out data
*     for the LutMap class to an output Channel.

*  Parameters:
*     this
*        Pointer to the LutMap whose data are being written.
*     channel
*        Pointer to the Channel to which the data are being written.
*/

/* Local Constants: */
#define KEY_LEN 50               /* Maximum length of a keyword */

/* Local Variables: */
   AstLutMap *this;              /* Pointer to the LutMap structure */
   char buff[ KEY_LEN + 1 ];     /* Buffer for keyword string */
   int ilut;                     /* Loop counter for table elements */

/* Check the global error status. */
   if ( !astOK ) return;

/* Obtain a pointer to the LutMap structure. */
   this = (AstLutMap *) this_object;

/* Write out values representing the instance variables for the LutMap
   class.  Accompany these with appropriate comment strings, possibly
   depending on the values being written. */

/* Number of lookup table elements. */
   astWriteInt( channel, "Nlut", 1, 1, this->nlut,
                "Number of lookup table elements" );

/* Input coordinate at first element centre. */
   astWriteDouble( channel, "Start", ( this->start != 0.0 ), 1, this->start,
                   "Input value at first element" );

/* Element spacing. */
   astWriteDouble( channel, "Incr", ( this->inc != 1.0 ), 1, this->inc,
                   "Input value increment between elements" );

/* Lookup table contents. */
   for ( ilut = 0; ilut < this->nlut; ilut++ ) {
      (void) sprintf( buff, "L%d", ilut + 1 );
      astWriteDouble( channel, buff, 1, 1, this->lut[ ilut ], 
                      ilut ? "" : "Lookup table elements..." );
   }

/* Undefine macros local to this function. */
#undef KEY_LEN
}

/* Standard class functions. */
/* ========================= */
/* Implement the astIsALutMap and astCheckLutMap functions using the
   macros defined for this purpose in the "object.h" header file. */
astMAKE_ISA(LutMap,Mapping,check,&class_init)
astMAKE_CHECK(LutMap)

AstLutMap *astLutMap_( int nlut, const double lut[],
                       double start, double inc,
                       const char *options, ... ) {
/*
*++
*  Name:
c     astLutMap
f     AST_LUTMAP

*  Purpose:
*     Create a LutMap.

*  Type:
*     Public function.

*  Synopsis:
c     #include "lutmap.h"
c     AstLutMap *astLutMap( int nlut, const double lut[],
c                           double start, double inc,
c                           const char *options, ... )
f     RESULT = AST_LUTMAP( NLUT, LUT, START, INC, OPTIONS, STATUS )

*  Class Membership:
*     LutMap constructor.

*  Description:
*     This function creates a new LutMap and optionally initialises
*     its attributes.
*
*     A LutMap is a specialised form of Mapping which transforms
*     1-dimensional coordinates by using linear interpolation in a
*     lookup table.  Each input coordinate value is first scaled to
*     give the index of an entry in the table by subtracting a
*     starting value (the input coordinate corresponding to the first
*     table entry) and dividing by an increment (the difference in
*     input coordinate value between adjacent table entries).
*
*     The resulting index will usually contain a fractional part, so
*     the output coordinate value is then generated by interpolating
*     linearly between the appropriate entries in the table. If the
*     index lies outside the range of the table, linear extrapolation
*     is used based on the two nearest entries (i.e. the two entries
*     at the start or end of the table, as appropriate).
*
*     If the lookup table entries increase or decrease monotonically,
*     then the inverse transformation may also be performed.

*  Parameters:
c     nlut
f     NLUT = INTEGER (Given)
*        The number of entries in the lookup table. This value must be
*        at least 2.
c     lut
f     LUT( NLUT ) = DOUBLE PRECISION (Given)
c        An array containing the "nlut"
f        An array containing the
*        lookup table entries.
c     start
f     START = DOUBLE PRECISION (Given)
*        The input coordinate value which corresponds with the first
*        lookup table entry.
c     inc
f     INC = DOUBLE PRECISION (Given)
*        The lookup table spacing (the increment in input coordinate
*        value between successive lookup table entries). This value
*        may be positive or negative, but must not be zero.
c     options
f     OPTIONS = CHARACTER * ( * ) (Given)
c        Pointer to a null-terminated string containing an optional
c        comma-separated list of attribute assignments to be used for
c        initialising the new LutMap. The syntax used is identical to
c        that for the astSet function and may include "printf" format
c        specifiers identified by "%" symbols in the normal way.
f        A character string containing an optional comma-separated
f        list of attribute assignments to be used for initialising the
f        new LutMap. The syntax used is identical to that for the
f        AST_SET routine.
c     ...
c        If the "options" string contains "%" format specifiers, then
c        an optional list of additional arguments may follow it in
c        order to supply values to be substituted for these
c        specifiers. The rules for supplying these are identical to
c        those for the astSet function (and for the C "printf"
c        function).
f     STATUS = INTEGER (Given and Returned)
f        The global status.

*  Returned Value:
c     astLutMap()
f     AST_LUTMAP = INTEGER
*        A pointer to the new LutMap.

*  Notes:
*     - If the entries in the lookup table either increase or decrease
*     monotonically, then the new LutMap's TranInverse attribute will
*     have a value of one, indicating that the inverse transformation
*     can be performed. Otherwise, it will have a value of zero, so
*     that any attempt to use the inverse transformation will result
*     in an error.
*     - A null Object pointer (AST__NULL) will be returned if this
c     function is invoked with the AST error status set, or if it
f     function is invoked with STATUS set to an error value, or if it
*     should fail for any reason.
*--
*/

/* Local Variables: */
   AstLutMap *new;               /* Pointer to new LutMap */
   va_list args;                 /* Variable argument list */

/* Check the global status. */
   if ( !astOK ) return NULL;

/* Initialise the LutMap, allocating memory and initialising the
   virtual function table as well if necessary. */
   new = astInitLutMap( NULL, sizeof( AstLutMap ), !class_init, &class_vtab,
                        "LutMap", nlut, lut, start, inc );

/* If successful, note that the virtual function table has been
   initialised. */
   if ( astOK ) {
      class_init = 1;

/* Obtain the variable argument list and pass it along with the
   options string to the astVSet method to initialise the new
   LutMap's attributes. */
      va_start( args, options );
      astVSet( new, options, args );
      va_end( args );

/* If an error occurred, clean up by deleting the new object. */
      if ( !astOK ) new = astDelete( new );
   }

/* Return a pointer to the new LutMap. */
   return new;
}

AstLutMap *astLutMapId_( int nlut, const double lut[],
                         double start, double inc,
                         const char *options, ... ) {
/*
*  Name:
*     astLutMapId_

*  Purpose:
*     Create a LutMap.

*  Type:
*     Private function.

*  Synopsis:
*     #include "lutmap.h"
*     AstLutMap *astLutMapId( int nlut, const double lut[],
*                             double start, double inc,
*                             const char *options, ... )

*  Class Membership:
*     LutMap constructor.

*  Description:
*     This function implements the external (public) interface to the
*     astLutMap constructor function. It returns an ID value (instead
*     of a true C pointer) to external users, and must be provided
*     because astLutMap_ has a variable argument list which cannot be
*     encapsulated in a macro (where this conversion would otherwise
*     occur).
*
*     The variable argument list also prevents this function from
*     invoking astLutMap_ directly, so it must be a re-implementation
*     of it in all respects, except for the final conversion of the
*     result to an ID value.

*  Parameters:
*     As for astLutMap_.

*  Returned Value:
*     The ID value associated with the new LutMap.
*/

/* Local Variables: */
   AstLutMap *new;               /* Pointer to new LutMap */
   va_list args;                 /* Variable argument list */

/* Check the global status. */
   if ( !astOK ) return NULL;

/* Initialise the LutMap, allocating memory and initialising the
   virtual function table as well if necessary. */
   new = astInitLutMap( NULL, sizeof( AstLutMap ), !class_init, &class_vtab,
                        "LutMap", nlut, lut, start, inc );

/* If successful, note that the virtual function table has been
   initialised. */
   if ( astOK ) {
      class_init = 1;

/* Obtain the variable argument list and pass it along with the
   options string to the astVSet method to initialise the new LutMap's
   attributes. */
      va_start( args, options );
      astVSet( new, options, args );
      va_end( args );

/* If an error occurred, clean up by deleting the new object. */
      if ( !astOK ) new = astDelete( new );
   }

/* Return an ID value for the new LutMap. */
   return astMakeId( new );
}

AstLutMap *astInitLutMap_( void *mem, size_t size, int init,
                           AstLutMapVtab *vtab, const char *name,
                           int nlut, const double lut[],
                           double start, double inc ) {
/*
*+
*  Name:
*     astInitLutMap

*  Purpose:
*     Initialise a LutMap.

*  Type:
*     Protected function.

*  Synopsis:
*     #include "lutmap.h"
*     AstLutMap *astInitLutMap( void *mem, size_t size, int init,
*                               AstLutMapVtab *vtab, const char *name,
*                               int nlut, const double lut[],
*                               double start, double inc )

*  Class Membership:
*     LutMap initialiser.

*  Description:
*     This function is provided for use by class implementations to
*     initialise a new LutMap object. It allocates memory (if
*     necessary) to accommodate the LutMap plus any additional data
*     associated with the derived class.  It then initialises a LutMap
*     structure at the start of this memory. If the "init" flag is
*     set, it also initialises the contents of a virtual function
*     table for a LutMap at the start of the memory passed via the
*     "vtab" parameter.

*  Parameters:
*     mem
*        A pointer to the memory in which the LutMap is to be
*        initialised.  This must be of sufficient size to accommodate
*        the LutMap data (sizeof(LutMap)) plus any data used by the
*        derived class. If a value of NULL is given, this function
*        will allocate the memory itself using the "size" parameter to
*        determine its size.
*     size
*        The amount of memory used by the LutMap (plus derived class
*        data).  This will be used to allocate memory if a value of
*        NULL is given for the "mem" parameter. This value is also
*        stored in the LutMap structure, so a valid value must be
*        supplied even if not required for allocating memory.
*     init
*        A logical flag indicating if the LutMap's virtual function
*        table is to be initialised. If this value is non-zero, the
*        virtual function table will be initialised by this function.
*     vtab
*        Pointer to the start of the virtual function table to be
*        associated with the new LutMap.
*     name
*        Pointer to a constant null-terminated character string which
*        contains the name of the class to which the new object
*        belongs (it is this pointer value that will subsequently be
*        returned by the astGetClass method).
*     nlut
*        The number of elements in the lookup table. This value must
*        be at least 2.
*     lut
*        An array containing the "nlut" lookup table elements.
*     start
*        The input coordinate value which corresponds with the first
*        lookup table element.
*     inc
*        The lookup table element spacing (i.e. the increment in input
*        coordinate value between successive lookup table elements).

*  Returned Value:
*     A pointer to the new LutMap.

*  Notes:
*     - A null pointer will be returned if this function is invoked
*     with the global error status set, or if it should fail for any
*     reason.
*-
*/

/* Local Variables: */
   AstLutMap *new;               /* Pointer to new LutMap */
   double *p;                    /* Pointer to next lut element */
   int down;                     /* Values are decreasing? */
   int ilut;                     /* Loop counter for LUT elements */
   int monotonic;                /* LUT elements distinct and monotonic? */
   int up;                       /* Values are increasing? */

/* Initialise. */
   new = NULL;

/* Check the global status. */
   if ( !astOK ) return new;

/* If necessary, initialise the virtual function table. */
   if ( init ) astInitLutMapVtab( vtab, name );

/* Check that the number of lookup table elements is valid. */
   if ( nlut < 2 ) {
      astError( AST__LUTIN, "astInitLutMap(%s): Invalid number of lookup "
                "table elements (%d).", name, nlut );
      astError( AST__LUTIN, "This value should be at least 2." );

/* Also check that the input value increment is not zero. */
   } else if ( inc == 0.0 ) {
      astError( AST__LUTII, "astInitLutMap(%s): An input value increment of "
                "zero between lookup table elements is not allowed.", name );

/* Determine if the element values are all good, distinct and increase 
   or decrease monotonically. We can only implement the inverse
   transformation if this is so. Use the astISBAD macro to include NaNs
   in the checking as well as AST__BAD values. */
   } else {
      if( astISBAD(lut[ 0 ]) ) {
         monotonic = 0;

      } else {
         up = ( lut[ nlut - 1 ] > lut[ 0 ] );
         down = ( lut[ nlut - 1 ] < lut[ 0 ] );
         monotonic = up || down;
         if ( monotonic ) {
            for ( ilut = 0; ilut < ( nlut - 1 ); ilut++ ) {
               if( astISBAD(lut[ ilut  + 1 ]) ) {
                  monotonic = 0;
               } else {
                  monotonic = up ? ( lut[ ilut + 1 ] > lut[ ilut ] ) :
                                   ( lut[ ilut + 1 ] < lut[ ilut ] );
               }
               if ( !monotonic ) break;
            }
         }
      }

/* Initialise a Mapping structure (the parent class) as the first
   component within the LutMap structure, allocating memory if
   necessary. Specify that the Mapping should be defined in the
   forward direction, and conditionally in the inverse direction. */
      new = (AstLutMap *) astInitMapping( mem, size, 0,
                                         (AstMappingVtab *) vtab, name,
                                          1, 1, 1, monotonic );

      if ( astOK ) {

/* Initialise the LutMap data. */
/* ---------------------------- */
         new->nlut = nlut;
         new->start = start;
         new->inc = inc;

/* Allocate memory and store the lookup table. */
         new->lut = astStore( NULL, lut, sizeof( double ) * (size_t) nlut );

/* Replace an NaN values by AST__BAD */
         p = new->lut;
         for ( ilut = 0; ilut < nlut; ilut++, p++ ) {
            if( astISNAN(*p) ) *p = AST__BAD;
         }

/* Initialise the retained input and output coordinate values. */
         new->last_fwd_in = AST__BAD;
         new->last_fwd_out = AST__BAD;
         new->last_inv_in = AST__BAD;
         new->last_inv_out = AST__BAD;
      }
         
/* If an error occurred, clean up by deleting the new LutMap. */
      if ( !astOK ) new = astDelete( new );
   }

/* Return a pointer to the new LutMap. */
   return new;
}

AstLutMap *astLoadLutMap_( void *mem, size_t size,
                           AstLutMapVtab *vtab, const char *name,
                           AstChannel *channel ) {
/*
*+
*  Name:
*     astLoadLutMap

*  Purpose:
*     Load a LutMap.

*  Type:
*     Protected function.

*  Synopsis:
*     #include "lutmap.h"
*     AstLutMap *astLoadLutMap( void *mem, size_t size,
*                               AstLutMapVtab *vtab, const char *name,
*                               AstChannel *channel )

*  Class Membership:
*     LutMap loader.

*  Description:
*     This function is provided to load a new LutMap using data read
*     from a Channel. It first loads the data used by the parent class
*     (which allocates memory if necessary) and then initialises a
*     LutMap structure in this memory, using data read from the input
*     Channel.
*
*     If the "init" flag is set, it also initialises the contents of a
*     virtual function table for a LutMap at the start of the memory
*     passed via the "vtab" parameter.


*  Parameters:
*     mem
*        A pointer to the memory into which the LutMap is to be
*        loaded.  This must be of sufficient size to accommodate the
*        LutMap data (sizeof(LutMap)) plus any data used by derived
*        classes. If a value of NULL is given, this function will
*        allocate the memory itself using the "size" parameter to
*        determine its size.
*     size
*        The amount of memory used by the LutMap (plus derived class
*        data).  This will be used to allocate memory if a value of
*        NULL is given for the "mem" parameter. This value is also
*        stored in the LutMap structure, so a valid value must be
*        supplied even if not required for allocating memory.
*
*        If the "vtab" parameter is NULL, the "size" value is ignored
*        and sizeof(AstLutMap) is used instead.
*     vtab
*        Pointer to the start of the virtual function table to be
*        associated with the new LutMap. If this is NULL, a pointer
*        to the (static) virtual function table for the LutMap class
*        is used instead.
*     name
*        Pointer to a constant null-terminated character string which
*        contains the name of the class to which the new object
*        belongs (it is this pointer value that will subsequently be
*        returned by the astGetClass method).
*
*        If the "vtab" parameter is NULL, the "name" value is ignored
*        and a pointer to the string "LutMap" is used instead.

*  Returned Value:
*     A pointer to the new LutMap.

*  Notes:
*     - A null pointer will be returned if this function is invoked
*     with the global error status set, or if it should fail for any
*     reason.
*-
*/

/* Local Constants: */
#define KEY_LEN 50               /* Maximum length of a keyword */

/* Local Variables: */
   AstLutMap *new;               /* Pointer to the new LutMap */
   char buff[ KEY_LEN + 1 ];     /* Buffer for keyword string */
   int ilut;                     /* Loop counter for table elements */

/* Initialise. */
   new = NULL;

/* Check the global error status. */
   if ( !astOK ) return new;

/* If a NULL virtual function table has been supplied, then this is
   the first loader to be invoked for this LutMap. In this case the
   LutMap belongs to this class, so supply appropriate values to be
   passed to the parent class loader (and its parent, etc.). */
   if ( !vtab ) {
      size = sizeof( AstLutMap );
      vtab = &class_vtab;
      name = "LutMap";

/* If required, initialise the virtual function table for this class. */
      if ( !class_init ) {
         astInitLutMapVtab( vtab, name );
         class_init = 1;
      }
   }

/* Invoke the parent class loader to load data for all the ancestral
   classes of the current one, returning a pointer to the resulting
   partly-built LutMap. */
   new = astLoadMapping( mem, size, (AstMappingVtab *) vtab, name,
                         channel );

   if ( astOK ) {

/* Read input data. */
/* ================ */
/* Request the input Channel to read all the input data appropriate to
   this class into the internal "values list". */
      astReadClassData( channel, "LutMap" );

/* Now read each individual data item from this list and use it to
   initialise the appropriate instance variable(s) for this class. */

/* Number of lookup table elements. */
      new->nlut = astReadInt( channel, "nlut", 2 );

/* Starting input coordinate value. */
      new->start = astReadDouble( channel, "start", 0.0 );

/* Input coordinate value increment. */
      new->inc = astReadDouble( channel, "incr", 1.0 );

/* Allocate memory to hold the lookup table elements. */
      new->lut = astMalloc( sizeof( double ) * (size_t) new->nlut );

/* If OK, loop to read each element. */
      if ( astOK ) {
         for ( ilut = 0; ilut < new->nlut; ilut++ ) {
            (void) sprintf( buff, "l%d", ilut + 1 );
            new->lut[ ilut ] = astReadDouble( channel, buff, 0.0 );
         }

/* Initialise the retained input and output coordinate values. */
         new->last_fwd_in = AST__BAD;
         new->last_fwd_out = AST__BAD;
         new->last_inv_in = AST__BAD;
         new->last_inv_out = AST__BAD;
      }
   }

/* If an error occurred, clean up by deleting the new LutMap. */
   if ( !astOK ) new = astDelete( new );

/* Return the new LutMap pointer. */
   return new;

/* Undefine macros local to this function. */
#undef KEY_LEN
}

/* Virtual function interfaces. */
/* ============================ */
/* These provide the external interface to the virtual functions
   defined by this class. Each simply checks the global error status
   and then locates and executes the appropriate member function,
   using the function pointer stored in the object's virtual function
   table (this pointer is located using the astMEMBER macro defined in
   "object.h").

   Note that the member function may not be the one defined here, as
   it may have been over-ridden by a derived class. However, it should
   still have the same interface. */
