      SUBROUTINE NDF_NEWP( FTYPE, NDIM, UBND, PLACE, INDF, STATUS ) 
*+
*  Name:
*     NDF_NEWP

*  Purpose:
*     Create a new primitive NDF.

*  Language:
*     Starlink Fortran 77

*  Invocation:
*     CALL NDF_NEWP( FTYPE, NDIM, UBND, PLACE, INDF, STATUS )

*  Description:
*     The routine creates a new primitive NDF and returns an identifier
*     for it. The NDF may subsequently be manipulated with the NDF_
*     routines.

*  Arguments:
*     FTYPE = CHARACTER * ( * ) (Given)
*        Data type of the NDF's DATA component (e.g. '_REAL'). Note that
*        complex types are not permitted when creating a primitive NDF.
*     NDIM = INTEGER (Given)
*        Number of NDF dimensions.
*     UBND( NDIM ) = INTEGER (Given)
*        Upper pixel-index bounds of the NDF (the lower bound of each
*        dimension is taken to be 1).
*     PLACE = INTEGER (Given and Returned)
*        An NDF placeholder (e.g. generated by the NDF_PLACE routine)
*        which indicates the position in the data system where the new
*        NDF will reside. The placeholder is annulled by this routine,
*        and a value of NDF__NOPL will be returned (as defined in the
*        include file NDF_PAR).
*     INDF = INTEGER (Returned)
*        Identifier for the new NDF.
*     STATUS = INTEGER (Given and Returned)
*        The global status.

*  Notes:
*     -  This routine creates a "primitive" NDF, i.e. one whose array
*     components are stored in "primitive" form by default (see
*     SGP/38).
*     -  The full type of the DATA component is specified via the FTYPE
*     argument and the type of the VARIANCE component defaults to the
*     same value. These types may be set individually with the
*     NDF_STYPE routine if required.
*     -  If this routine is called with STATUS set, then a value of
*     NDF__NOID will be returned for the INDF argument, although no
*     further processing will occur. The same value will also be
*     returned if the routine should fail for any reason. In either
*     event, the placeholder will still be annulled. The NDF__NOID
*     constant is defined in the include file NDF_PAR.

*  Copyright:
*     Copyright (C) 1993 Science & Engineering Research Council

*  Authors:
*     RFWS: R.F. Warren-Smith (STARLINK)
*     {enter_new_authors_here}

*  History:
*     15-FEB-1990 (RFWS):
*        Original, derived from the NDF_NEW routine.
*     4-NOV-1993 (RFWS):
*        Changed to support foreign format files.
*     {enter_further_changes_here}

*  Bugs:
*     {note_any_bugs_here}

*-
      
*  Type Definitions:
      IMPLICIT NONE              ! No implicit typing

*  Global Constants:
      INCLUDE 'SAE_PAR'          ! Standard SAE constants
      INCLUDE 'DAT_PAR'          ! DAT_ public constants
      INCLUDE 'NDF_PAR'          ! NDF_ public constants
      INCLUDE 'NDF_CONST'        ! NDF_ private constants
      INCLUDE 'NDF_ERR'          ! NDF_ error codes

*  Arguments Given:
      CHARACTER * ( * ) FTYPE
      INTEGER NDIM
      INTEGER UBND( * )
      INTEGER PLACE

*  Arguments Returned:
      INTEGER INDF

*  Status:
      INTEGER STATUS             ! Global status

*  Local Variables:
      CHARACTER * ( NDF__SZTYP ) TYPE ! Numeric data type
      INTEGER I                  ! Loop counter for dimensions
      INTEGER IACB               ! Index of NDF entry in the ACB
      INTEGER IPCB               ! Index to placeholder entry in the PCB
      INTEGER LBND( NDF__MXDIM ) ! Lower bounds of NDF
      INTEGER TSTAT              ! Temporary status variable
      LOGICAL CMPLX              ! Whether the data type is complex
      LOGICAL ERASE              ! Whether to erase placeholder object

*.

*  Set an initial value for the INDF argument.
      INDF = NDF__NOID
       
*  Save the STATUS value and mark the error stack.
      TSTAT = STATUS
      CALL ERR_MARK
       
*  Import the NDF placeholder, converting it to a PCB index.
      STATUS = SAI__OK
      IPCB = 0
      CALL NDF1_IMPPL( PLACE, IPCB, STATUS )

*  If there has been no error at all so far, then check the data type
*  for validity. Report an error if a complex data type has been
*  specified.
      IF ( ( STATUS .EQ. SAI__OK ) .AND. ( TSTAT .EQ. SAI__OK ) ) THEN
         CALL NDF1_CHFTP( FTYPE, TYPE, CMPLX, STATUS )
         IF ( CMPLX ) THEN
            STATUS = NDF__FTPIN
            CALL MSG_SETC( 'BADTYPE', FTYPE )
            CALL ERR_REP( 'NDF_NEWP_TYPE',
     :      'The complex type ''^BADTYPE'' is not valid for a ' //
     :      'primitive NDF (possible programming error).', STATUS )
         END IF

*  Check the NDF upper bounds for validity.
         IF ( STATUS .EQ. SAI__OK ) THEN
            DO 1 I = 1, MIN( NDIM, NDF__MXDIM )
               LBND( I ) = 1
1           CONTINUE
            CALL NDF1_VBND( NDIM, LBND, UBND, STATUS )
         END IF

*  Create a new primitive NDF in place of the placeholder object,
*  obtaining an ACB entry which refers to it.
         IF ( STATUS .EQ. SAI__OK ) THEN
            CALL NDF1_DCREP( FTYPE, NDIM, UBND, IPCB, IACB, STATUS )

*  Export an identifier for the NDF.
            CALL NDF1_EXPID( IACB, INDF, STATUS )

*  If an error occurred, then annul any ACB entry which may have been
*  acquired.
            IF ( STATUS .NE. SAI__OK ) THEN
               CALL NDF1_ANL( IACB, STATUS )
            END IF
         END IF
      END IF
       
*  Annul the placeholder, erasing the associated object if any error has
*  occurred.
      IF ( IPCB .NE. 0 ) THEN
         ERASE = ( STATUS .NE. SAI__OK ) .OR. ( TSTAT .NE. SAI__OK )
         CALL NDF1_ANNPL( ERASE, IPCB, STATUS )
      END IF

*  Reset the PLACE argument.
      PLACE = NDF__NOPL

*  Annul any error if STATUS was previously bad, otherwise let the new
*  error report stand.
      IF ( STATUS .NE. SAI__OK ) THEN
         IF ( TSTAT .NE. SAI__OK ) THEN
            CALL ERR_ANNUL( STATUS )
            STATUS = TSTAT

*  If appropriate, report the error context and call the error tracing
*  routine.
         ELSE
            INDF = NDF__NOID
            CALL ERR_REP( 'NDF_NEWP_ERR',
     :      'NDF_NEWP: Error creating a new primitive NDF.', STATUS )
            CALL NDF1_TRACE( 'NDF_NEWP', STATUS )
         END IF
      ELSE
         STATUS = TSTAT
      END IF

*  Release error stack.
      CALL ERR_RLSE

      END
