      SUBROUTINE NDF_SCOPY( INDF1, CLIST, PLACE, INDF2, STATUS )
*+
*  Name:
*     NDF_SCOPY

*  Purpose:
*     Selectively copy NDF components to a new location.

*  Language:
*     Starlink Fortran 77

*  Invocation:
*     CALL NDF_SCOPY( INDF1, CLIST, PLACE, INDF2, STATUS )

*  Description:
*     The routine propagates (copies) selected components of an NDF to
*     a new location and returns an identifier for the resulting new
*     base NDF.

*  Arguments:
*     INDF1 = INTEGER (Given)
*        Identifier for the NDF (or NDF section) to be copied.
*     CLIST = CHARACTER * ( * ) (Given)
*        A comma-separated list of the NDF components which are to be
*        propagated to the new data structure. By default, the HISTORY,
*        LABEL and TITLE components and all extensions are copied.  See
*        the "Component Propagation" section for further details.
*     PLACE = INTEGER (Given and Returned)
*        An NDF placeholder (e.g. generated by the NDF_PLACE routine)
*        which indicates the position in the data system where the new
*        NDF will reside. The placeholder is annulled by this routine,
*        and a value of NDF__NOPL will be returned (as defined in the
*        include file NDF_PAR).
*     INDF2 = INTEGER (Returned)
*        Identifier for the new NDF.
*     STATUS = INTEGER (Given and Returned)
*        The global status.

*  Notes:
*     -  If this routine is called with STATUS set, then a value of
*     NDF__NOID will be returned for the INDF2 argument, although no
*     further processing will occur. The same value will also be
*     returned if the routine should fail for any reason.  In either
*     event the placeholder will still be annulled.  The NDF__NOID
*     constant is defined in the include file NDF_PAR.

*  Component Propagation:
*     -  The components whose values are to be propagated (copied) from
*     the existing NDF to the new data structure are specified via the
*     CLIST argument. Thus CLIST='DATA,QUALITY' would cause the DATA
*     and QUALITY components to be propagated (if available) from the
*     existing NDF to the new structure, in addition to those
*     propagated by default.  Component propagation may be suppressed
*     by supplying a component name with the prefix 'NO'. Thus
*     CLIST='DATA,NOHISTORY' would propagate the DATA component, but
*     suppress propagation of HISTORY. If component names appear more
*     than once in the CLIST value, then the last occurrence takes
*     precedence.
*     -  Propagation of specific NDF extensions may be suppressed by
*     using 'NOEXTENSION()' as one of the items in the CLIST argument;
*     a list of the extensions to be suppressed should appear between
*     the parentheses. Thus CLIST='AXIS,NOEXTENSION(IRAS,ASTERIX)'
*     would propagate the AXIS component, but suppress propagation of
*     the IRAS and ASTERIX extensions (if present). Propagation of
*     suppressed extensions may be re-enabled by specifying
*     'EXTENSION()' in a similar manner at a later point in the CLIST
*     value.
*     -  Component names in the CLIST argument may be abbreviated to 3
*     characters, but extension names must appear in full.

*  Copyright:
*     Copyright (C) 1993 Science & Engineering Research Council

*  Authors:
*     DSB: David Berry (STARLINK)
*     RFWS: R.F. Warren-Smith (STARLINK, RAL)
*     {enter_new_authors_here}

*  History:
*     9-JUN-1993 (DSB):
*        Original version, derived from NDF_COPY.
*     4-NOV-1993 (RFWS):
*        Changed to support foreign format files.
*     {enter_further_changes_here}

*  Bugs:
*     {note_any_bugs_here}

*-
      
*  Type Definitions:
      IMPLICIT NONE              ! No implicit typing

*  Global Constants:
      INCLUDE 'SAE_PAR'          ! Standard SAE constants
      INCLUDE 'DAT_PAR'          ! DAT_ public constants
      INCLUDE 'NDF_PAR'          ! NDF_ public constants
      INCLUDE 'NDF_CONST'        ! NDF_ private constants

*  Arguments Given:
      INTEGER INDF1
      CHARACTER * ( * ) CLIST
      INTEGER PLACE

*  Arguments Returned:
      INTEGER INDF2

*  Status:
      INTEGER STATUS             ! Global status

*  Local Variables:
      CHARACTER * ( DAT__SZNAM ) EXTN( NDF__MXEXT ) ! Excluded ext. list
      INTEGER IACB1              ! Index to input NDF entry in ACB
      INTEGER IACB2              ! Index to output NDF entry in ACB
      INTEGER IPCB               ! Index to placeholder entry in the PCB
      INTEGER NEXTN              ! Number of excluded extensions
      INTEGER TSTAT              ! Temporary status variable
      LOGICAL CPF( NDF__MXCPF )  ! Component propagation flags
      LOGICAL ERASE              ! Whether to erase placeholder object

*.

*  Set an initial value for the INDF2 argument.
      INDF2 = NDF__NOID

*  Save the STATUS value and mark the error stack.
      TSTAT = STATUS
      CALL ERR_MARK
       
*  Import the NDF placeholder, converting it to a PCB index.
      STATUS = SAI__OK
      IPCB = 0
      CALL NDF1_IMPPL( PLACE, IPCB, STATUS )

*  Parse the component propagation expression.
      CALL NDF1_PSCPX( CLIST, NDF__MXEXT, EXTN, NEXTN, CPF, STATUS )

*  If there has been no error at all so far, then import the input
*  NDF identifier.
      IF ( ( STATUS .EQ. SAI__OK ) .AND. ( TSTAT .EQ. SAI__OK ) ) THEN
         CALL NDF1_IMPID( INDF1, IACB1, STATUS )

*  Selectively propagate the components of the input NDF to create a
*  new base NDF and an ACB entry to describe it.
         CALL NDF1_PRP( IACB1, NEXTN, EXTN, CPF, IPCB, IACB2, STATUS )

*  Export an identifier for the new NDF.
         CALL NDF1_EXPID( IACB2, INDF2, STATUS )

*  If an error occurred, then annul any ACB entry which may have been
*  acquired.
         IF ( STATUS .NE. SAI__OK ) THEN
            CALL NDF1_ANL( IACB2, STATUS )
         END IF
      END IF
       
*  Annul the placeholder, erasing the associated object if any error has
*  occurred.
      IF ( IPCB .NE. 0 ) THEN
         ERASE = ( STATUS .NE. SAI__OK ) .OR. ( TSTAT .NE. SAI__OK )
         CALL NDF1_ANNPL( ERASE, IPCB, STATUS )
      END IF

*  Reset the PLACE argument.
      PLACE = NDF__NOPL

*  Annul any error if STATUS was previously bad, otherwise let the new
*  error report stand.
      IF ( STATUS .NE. SAI__OK ) THEN
         IF ( TSTAT .NE. SAI__OK ) THEN
            CALL ERR_ANNUL( STATUS )
            STATUS = TSTAT
             
*  If appropriate, reset the INDF2 argument, report the error context
*  and call the error tracing routine.
         ELSE
            INDF2 = NDF__NOID
            CALL ERR_REP( 'NDF_SCOPY_ERR',
     :      'NDF_SCOPY: Error selectively copying NDF components to ' //
     :      'a new location.', STATUS )
            CALL NDF1_TRACE( 'NDF_SCOPY', STATUS )
         END IF
      ELSE
         STATUS = TSTAT
      END IF

*  Release error stack.
      CALL ERR_RLSE

      END
