/***************************************************************************
 *
 * gnome-mount.c : GNOME mount, unmount and eject wrappers using HAL
 *
 * Copyright (C) 2006 David Zeuthen, <david@fubar.dk>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 **************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <gnome.h>
#include <glib/gi18n.h>
#include <gconf/gconf-client.h>
#include <gdk/gdkx.h>
#include <libgnomeui/gnome-password-dialog.h>
#include <gnome-keyring.h>

#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <libhal.h>
#include <libhal-storage.h>

#include <mntent.h>

static DBusConnection *dbus_connection;
static LibHalContext *hal_ctx;


static int fds[2];
static int rc;
static gboolean opt_noui = FALSE;
static gboolean opt_block = FALSE;
static gboolean opt_nodisplay = FALSE;

static void 
notify_parent (gboolean success)
{
	static gboolean already_notified = FALSE;

	if (opt_block) {
		rc = success ? 0 : 1;
	} else {
		if (!already_notified) {
			char buf;
			already_notified = TRUE;
			buf = success ? '1' : '0';
			write (fds[1], &buf, 1);
		}
	}
}

static void
show_error_dialog_no_media (const char *udi, LibHalVolume *volume, LibHalDrive *drive)
{
	GtkWidget *w;
	if (!opt_noui) {
		w = gtk_message_dialog_new (NULL, 
					    GTK_DIALOG_MODAL,
					    GTK_MESSAGE_ERROR,
					    GTK_BUTTONS_CLOSE,
					    _("Unable to mount media."));
		/* TODO: use icons, text from gnome-vfs, libhal-storage, e.g. s/drive/floppy drive/ */
		gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (w),
							  "There is probably no media in the drive.");
		gtk_dialog_run (GTK_DIALOG (w));
		gtk_widget_unref (w);
	}
}

static void
show_error_dialog_eject (const char *udi, LibHalVolume *volume, LibHalDrive *drive, 
			 const char *error_name, const char *error_detail)
{
	GtkWidget *w;
	if (!opt_noui) {
		const char *volume_name;

		/* We need to handle the following errors 
		 *
		 *   org.freedesktop.Hal.Device.Volume.PermissionDenied
		 *   org.freedesktop.Hal.Device.Volume.UnsupportedEjectOption
		 *   org.freedesktop.Hal.Device.Volume.InvalidEjectOption
		 *   org.freedesktop.Hal.Device.Volume.Busy
		 *   org.freedesktop.Hal.Device.Volume.UnknownFailure
		 *
		 *
		 * in a sane way.
		 */

		w = gtk_message_dialog_new (NULL, 
					    GTK_DIALOG_MODAL,
					    GTK_MESSAGE_ERROR,
					    0,
					    _("Cannot eject volume"));

		if (volume != NULL)
			volume_name = libhal_volume_get_label (volume);
		else
			volume_name = NULL;

		if (strcmp (error_name, "org.freedesktop.Hal.Device.Volume.PermissionDenied") == 0) {
			gtk_message_dialog_format_secondary_text (
				GTK_MESSAGE_DIALOG (w),
				volume_name != NULL ? 
				_("You are not privileged to eject the volume '%s'.") :
				_("You are not privileged to eject this volume."),
				volume_name);
		} else if (strcmp (error_name, "org.freedesktop.Hal.Device.Volume.Busy") == 0) {
			/* TODO: figure out exactly which application and find localized name and icon via
			 * desktop files */
			gtk_message_dialog_format_secondary_text (
				GTK_MESSAGE_DIALOG (w),
				volume_name != NULL ? 
				_("An application is preventing the volume '%s' from being ejected.") :
				_("An application is preventing the volume from being ejected."),
				volume_name);
			/* TODO: could add 'Lazy Unmount' button */
		} else if (strcmp (error_name, "org.freedesktop.Hal.Device.Volume.InvalidUnmountOption") == 0 ||
			   strcmp (error_name, "org.freedesktop.Hal.Device.Volume.UnsupportedEjectOption") == 0 ||
			   strcmp (error_name, "org.freedesktop.Hal.Device.Volume.UnknownFailure") == 0) {
			gtk_message_dialog_format_secondary_text (
				GTK_MESSAGE_DIALOG (w),
				volume_name != NULL ? 
				_("Cannot eject the volume '%s'.") :
				_("Cannot eject the volume."),
				volume_name);
		}

		gtk_dialog_add_buttons (GTK_DIALOG (w),
					GTK_STOCK_OK,
					GTK_RESPONSE_NONE,
					NULL);

		gtk_dialog_run (GTK_DIALOG (w));
		gtk_widget_unref (w);
	}
}

static void
show_error_dialog_unmount (const char *udi, LibHalVolume *volume, LibHalDrive *drive, 
			   const char *error_name, const char *error_detail)
{
	GtkWidget *w;
	if (!opt_noui) {
		const char *volume_name;

		/* We need to handle the following errors 
		 *
		 *   org.freedesktop.Hal.Device.Volume.PermissionDenied
		 *   org.freedesktop.Hal.Device.Volume.UnsupportedUnmountOption
		 *   org.freedesktop.Hal.Device.Volume.InvalidUnmountOption
		 *   org.freedesktop.Hal.Device.Volume.Busy
		 *   org.freedesktop.Hal.Device.Volume.NotMounted
		 *   org.freedesktop.Hal.Device.Volume.UnknownFailure
		 *
		 *
		 * in a sane way.
		 */

		w = gtk_message_dialog_new (NULL, 
					    GTK_DIALOG_MODAL,
					    GTK_MESSAGE_ERROR,
					    0,
					    _("Cannot unmount volume"));

		if (volume != NULL)
			volume_name = libhal_volume_get_label (volume);
		else
			volume_name = NULL;

		if (strcmp (error_name, "org.freedesktop.Hal.Device.Volume.PermissionDenied") == 0) {
			gtk_message_dialog_format_secondary_text (
				GTK_MESSAGE_DIALOG (w),
				volume_name != NULL ? 
				_("You are not privileged to unmount the volume '%s'.") :
				_("You are not privileged to unmount this volume."),
				volume_name);
		} else if (strcmp (error_name, "org.freedesktop.Hal.Device.Volume.Busy") == 0) {
			/* TODO: figure out exactly which application and find localized name and icon via
			 * desktop files */
			gtk_message_dialog_format_secondary_text (
				GTK_MESSAGE_DIALOG (w),
				volume_name != NULL ? 
				_("An application is preventing the volume '%s' from being unmounted.") :
				_("An application is preventing the volume from being unmounted."),
				volume_name);
			/* TODO: could add 'Lazy Unmount' button */
		} else if (strcmp (error_name, "org.freedesktop.Hal.Device.Volume.NotMounted") == 0 ) {
			gtk_message_dialog_format_secondary_text (
				GTK_MESSAGE_DIALOG (w),
				volume_name != NULL ? 
				_("The volume '%s' is not mounted.") :
				_("The volume is not mounted."),
				volume_name);
		} else if (strcmp (error_name, "org.freedesktop.Hal.Device.Volume.InvalidUnmountOption") == 0 ||
			   strcmp (error_name, "org.freedesktop.Hal.Device.Volume.UnsupportedUnmountOption") == 0 ||
			   strcmp (error_name, "org.freedesktop.Hal.Device.Volume.UnknownFailure") == 0) {
			gtk_message_dialog_format_secondary_text (
				GTK_MESSAGE_DIALOG (w),
				volume_name != NULL ? 
				_("Cannot unmount the volume '%s'.") :
				_("Cannot unmount the volume."),
				volume_name);
		}

		gtk_dialog_add_buttons (GTK_DIALOG (w),
					GTK_STOCK_OK,
					GTK_RESPONSE_NONE,
					NULL);

		gtk_dialog_run (GTK_DIALOG (w));
		gtk_widget_unref (w);
	}
}

/* #define MOUNT_ERROR_DIALOG_RESPONSE_INSTALL_DRIVER 10 see below */

static void
show_error_dialog_mount (LibHalVolume *volume, const char *error_name, const char *error_detail, 
			 const char *fstype_requested)
{
	GtkWidget *w;
	if (!opt_noui && 
	    strcmp (error_name, "org.freedesktop.Hal.Device.Volume.AlreadyMounted") != 0) {
		int response;
		const char *volume_name;

		/* We need to handle the following errors 
		 *
		 *   org.freedesktop.Hal.Device.Volume.PermissionDenied
		 *   org.freedesktop.Hal.Device.Volume.InvalidMountOption
		 *   org.freedesktop.Hal.Device.Volume.FailedToCreateMountpoint
		 *   org.freedesktop.Hal.Device.Volume.UnknownFilesystemType
		 *   org.freedesktop.Hal.Device.Volume.UnknownFailure
		 *
		 * in a sane way. We ignore 
		 *
		 *   org.freedesktop.Hal.Device.Volume.AlreadyMounted
		 */

		w = gtk_message_dialog_new (NULL, 
					    GTK_DIALOG_MODAL,
					    GTK_MESSAGE_ERROR,
					    0,
					    _("Cannot mount volume"));

		volume_name = libhal_volume_get_label (volume);

		if (strcmp (error_name, "org.freedesktop.Hal.Device.Volume.PermissionDenied") == 0) {
			gtk_message_dialog_format_secondary_text (
				GTK_MESSAGE_DIALOG (w),
				volume_name != NULL ? 
				_("You are not privileged to mount the volume '%s'.") :
				_("You are not privileged to mount this volume."),
				volume_name);
		} else if (strcmp (error_name, "org.freedesktop.Hal.Device.Volume.InvalidMountOption") == 0) {
			/* TODO: slim down mount options to what is allowed, cf. volume.mount.valid_options */
			gtk_message_dialog_format_secondary_text (
				GTK_MESSAGE_DIALOG (w),
				volume_name != NULL ? 
				_("Invalid mount option when attempting to mount the volume '%s'.") :
				_("Invalid mount option when attempting to mount the volume."),
				volume_name);
		} else if (strcmp (error_name, "org.freedesktop.Hal.Device.Volume.UnknownFilesystemType") == 0) {
			gtk_message_dialog_format_secondary_markup (
				GTK_MESSAGE_DIALOG (w),
				volume_name != NULL ? 
				_("The volume '%s' uses the <i>%s</i> file system which is not supported by your system.") :
				_("The volume uses the <i>%s</i> file system which is not supported by your system."),
				volume_name != NULL ? volume_name : fstype_requested,
				volume_name != NULL ? fstype_requested : "");

			/* some day.. :-)
			gtk_dialog_add_buttons (GTK_DIALOG (w),
						_("Install Driver..."),
						MOUNT_ERROR_DIALOG_RESPONSE_INSTALL_DRIVER,
						NULL);
			*/
		} else if (strcmp (error_name, "org.freedesktop.Hal.Device.Volume.FailedToCreateMountPoint") == 0 ||
			   strcmp (error_name, "org.freedesktop.Hal.Device.Volume.UnknownFailure") == 0) {
			gtk_message_dialog_format_secondary_text (
				GTK_MESSAGE_DIALOG (w),
				volume_name != NULL ? 
				_("Unable to mount the volume '%s'.") :
				_("Unable to mount the volume."),
				volume_name);
		}

		gtk_dialog_add_buttons (GTK_DIALOG (w),
					GTK_STOCK_OK,
					GTK_RESPONSE_NONE,
					NULL);

		response = gtk_dialog_run (GTK_DIALOG (w));
		gtk_widget_unref (w);

		switch (response) {
			/* case MOUNT_ERROR_DIALOG_RESPONSE_INSTALL_DRIVER:
			g_message ("install driver!");
			break; */
		default:
			break;
		}

	}
}


/** Integrate a dbus mainloop. 
 *
 *  @param  ctx                 LibHal context
 *  @param  error     		pointer to a D-BUS error object
 *
 *  @return 			TRUE if we connected to the bus
 */
static dbus_bool_t 
hal_mainloop_integration (LibHalContext *ctx, DBusError *error)
{
	return FALSE;
}

/** Internal HAL initialization function
 *
 * @return			The LibHalContext of the HAL connection or
 *				NULL on error.
 */
static LibHalContext *
do_hal_init (void)
{
	LibHalContext *ctx;
	DBusError error;
	char **devices;
	int nr;
	
	ctx = libhal_ctx_new ();
	if (ctx == NULL) {
		g_warning ("Failed to get libhal context");
		goto error;
	}
	
	dbus_error_init (&error);
	dbus_connection = dbus_bus_get (DBUS_BUS_SYSTEM, &error);
	if (dbus_error_is_set (&error)) {
		g_warning ("Cannot connect to system bus: %s : %s", error.name, error.message);
		dbus_error_free (&error);
		goto error;
	}
	
        dbus_connection_setup_with_g_main (dbus_connection, NULL);	
	libhal_ctx_set_dbus_connection (ctx, dbus_connection);
	
	if (!libhal_ctx_init (ctx, &error)) {
		g_warning ("Failed to initialize libhal context: %s : %s", error.name, error.message);
		dbus_error_free (&error);
		goto error;
	}

	return ctx;

error:
	if (ctx != NULL)
		libhal_ctx_free (ctx);
	return NULL;
}

static const char *
get_dev_file (LibHalVolume *volume, LibHalDrive *drive)
{
	if (volume != NULL) {
		return libhal_volume_get_device_file (volume);
	} else if (drive != NULL) {
		return libhal_drive_get_device_file (drive);
	} else
		return NULL;
}


static gboolean
volume_mount_with_options (const char *udi, LibHalVolume *volume, LibHalDrive *drive,
			   const char *mount_point, const char *fstype, GPtrArray *options)
{
	DBusMessage *reply = NULL;
	DBusMessage *dmesg = NULL;
	gboolean ret = FALSE;
	DBusError error;
	unsigned int i;
	
	if (mount_point == NULL)
		mount_point = "";

	if (fstype == NULL)
		fstype = "";
	
	g_debug ("Mounting %s with mount_point='%s', fstype='%s', num_options=%d", 
		   udi,
		   mount_point, 
		   fstype,
		   options->len);
	for (i = 0; i < options->len; i++)
		g_debug ("  option='%s'", (char *) options->pdata[i]);
	
	if (!(dmesg = dbus_message_new_method_call ("org.freedesktop.Hal", udi,
						    "org.freedesktop.Hal.Device.Volume",
						    "Mount"))) {
		g_warning ("Could not create dbus message for %s", udi);
		return FALSE;
	}
	
	if (!dbus_message_append_args (dmesg, DBUS_TYPE_STRING, &mount_point, DBUS_TYPE_STRING, &fstype,
				       DBUS_TYPE_ARRAY, DBUS_TYPE_STRING, &options->pdata, options->len,
				       DBUS_TYPE_INVALID)) {
		g_warning ("Could not append args to dbus message for %s", udi);
		dbus_message_unref (dmesg);
		return FALSE;
	}

	dbus_error_init (&error);
	if (!(reply = dbus_connection_send_with_reply_and_block (dbus_connection, dmesg, -1, &error)) ||
	    dbus_error_is_set (&error)) {
		g_warning ("Mount failed for %s\n%s : %s\n", udi, error.name, error.message);

		if (strcmp (error.name, "org.freedesktop.Hal.Device.Volume.InvalidMountpoint") == 0 &&
		    mount_point != NULL) {

			dbus_message_unref (dmesg);
			if (reply != NULL)
				dbus_message_unref (reply);

			g_warning ("Given mount point name '%s' was invalid, trying without this...", mount_point);

			/* possibly illegal mount point name; try without that mount point name... */
			ret = volume_mount_with_options (udi, volume, drive, NULL, fstype, options);
			if (ret) {
				/* TODO: log to syslog that given mount_point was illegal */
			}

			dbus_error_free (&error);

			return ret;
		}

		notify_parent (FALSE);

		show_error_dialog_mount (volume, error.name, error.message, 
					 (fstype != NULL && strlen (fstype) > 0) ? 
					 fstype : libhal_volume_get_fstype (volume));

		dbus_error_free (&error);
		goto out;
	}

	{
		char *mount_point;
		const char *device_file;

		mount_point = NULL;
		device_file = get_dev_file (volume, drive);

		if (device_file != NULL) {
			FILE *f;
			struct mntent mnt;
			struct mntent *mnte;
			char buf[512];

			if ((f = setmntent ("/proc/mounts", "r")) != NULL) {
				
				while ((mnte = getmntent_r (f, &mnt, buf, sizeof(buf))) != NULL) {
					if (strcmp (mnt.mnt_fsname, device_file) == 0) {
						if (mnt.mnt_dir != NULL) {
							mount_point = g_strdup (mnt.mnt_dir);
						}
						break;
					}
				}
				endmntent (f);
			}
		}

		g_print (_("Mounted %s at \"%s\"\n"), device_file, mount_point);
		g_free (mount_point);
	}

	ret = TRUE;

out:
	if (dmesg != NULL)
		dbus_message_unref (dmesg);
	if (reply != NULL)
		dbus_message_unref (reply);
	
	return ret;
}


enum {
	MOUNT_CODEPAGE   = (1 << 0),
	MOUNT_DATA       = (1 << 1),
	MOUNT_DIRSYNC    = (1 << 2),
	MOUNT_DMASK      = (1 << 3),
	MOUNT_FLUSH      = (1 << 4),
	MOUNT_IOCHARSET  = (1 << 5),
	MOUNT_MODE       = (1 << 6),
	MOUNT_NOATIME    = (1 << 7),
	MOUNT_NODIRATIME = (1 << 8),
	MOUNT_NOEXEC     = (1 << 9),
	MOUNT_QUIET      = (1 << 10),
	MOUNT_READ_ONLY  = (1 << 11),
	MOUNT_SHORTNAME  = (1 << 12),
	MOUNT_SYNC       = (1 << 13),
	MOUNT_UID        = (1 << 14),
	MOUNT_UMASK      = (1 << 15),
	MOUNT_UTF8       = (1 << 16),
};

static struct {
	const char *name;
	guint32 flag;
} mount_options[] = {
	{ "codepage=",  MOUNT_CODEPAGE   },
	{ "data=",      MOUNT_DATA       },
	{ "dirsync",    MOUNT_DIRSYNC    },
	{ "dmask=",     MOUNT_DMASK      },
	{ "flush",      MOUNT_FLUSH      },
	{ "iocharset=", MOUNT_IOCHARSET  },
	{ "mode=",      MOUNT_MODE       },
	{ "noatime",    MOUNT_NOATIME    },
	{ "nodiratime", MOUNT_NODIRATIME },
	{ "noexec",     MOUNT_NOEXEC     },
	{ "quiet",      MOUNT_QUIET      },
	{ "ro",         MOUNT_READ_ONLY  },
	{ "shortname=", MOUNT_SHORTNAME  },
	{ "sync",       MOUNT_SYNC       },
	{ "uid=",       MOUNT_UID        },
	{ "umask=",     MOUNT_UMASK      },
	{ "utf8",       MOUNT_UTF8       },
};

static guint32
mount_option_flags (char **strlist)
{
	guint32 flags = 0;
	size_t i, j;
	
	for (i = 0; strlist[i]; i++) {
		for (j = 0; j < G_N_ELEMENTS (mount_options); j++) {
			if (!strcmp (strlist[i], mount_options[j].name))
				flags |= mount_options[j].flag;
		}
	}
	
	return flags;
}

static gboolean
volume_mount (const char *udi, LibHalVolume *volume, LibHalDrive *drive)
{
	char **strlist;
	char uidbuf[64];
	char *mount_point;
	char *fstype_override;
	GSList *list, *l, *n;
	GPtrArray *options;
	guint32 user_opts;
	guint32 opts = 0;
	char *key;
	gboolean ret;
	const char *fstype;

	ret = FALSE;

	if (volume != NULL) {
		fstype = libhal_volume_get_fstype (volume);
	} else {
		fstype = "";
	}
	
	if ((strlist = libhal_device_get_property_strlist (hal_ctx, udi, "volume.mount.valid_options", NULL))) {
		opts = mount_option_flags (strlist);
		libhal_free_string_array (strlist);
	}
			
	/* TODO: read per-volume settings from gconf and adjust mount options */
#if 0
	/* TODO: cannot assume volume!=NULL (either volume!=NULL or drive!=NULL)
	 *       Should use hal UDI instead (it contains the UUID anyway)
	 */
	key = g_strdup_printf ("/system/storage/volumes/%s/mount_options", libhal_volume_get_uuid (volume));
	if ((list = gconf_client_get_list (client, key, GCONF_VALUE_STRING, NULL))) {
		options = g_ptr_array_new ();
		for (l = list; l; l = n) {
			n = l->next;
			g_ptr_array_add (options, l->data);
			g_slist_free_1 (l);
		}
		g_ptr_array_add (options, NULL);
		strlist = options->pdata;
		g_ptr_array_free (options, FALSE);
		
		user_opts = mount_option_flags (strlist);
		g_strfreev (strlist);
	}
	g_free (key);
#else
	user_opts = opts;
#endif
			
	/* find the union of options to pass to mount */
	opts &= user_opts;
	
	options = g_ptr_array_new ();
	
	if (volume == NULL) {
		/* volume from a non-pollable drive, just set uid.. */
		
		snprintf (uidbuf, sizeof (uidbuf) - 1, "uid=%u", getuid ());
		g_ptr_array_add (options, uidbuf);
		
	} else if (strcmp (fstype, "vfat") == 0) {
/*
 * Ugh, flush is not upstream yet.. better not add it...
 *
 if (opts & MOUNT_FLUSH)
 g_ptr_array_add (options, "flush");
*/
		if (opts & MOUNT_SHORTNAME)
			g_ptr_array_add (options, "shortname=winnt");

		if (opts & MOUNT_UID) {
			snprintf (uidbuf, sizeof (uidbuf) - 1, "uid=%u", getuid ());
			g_ptr_array_add (options, uidbuf);
		}
	} else if (strcmp (fstype, "iso9660") == 0) {
		if (opts & MOUNT_UID) {
			snprintf (uidbuf, sizeof (uidbuf) - 1, "uid=%u", getuid ());
			g_ptr_array_add (options, uidbuf);
		}
	} else if (strcmp (fstype, "udf") == 0) {
		if (opts & MOUNT_UID) {
			snprintf (uidbuf, sizeof (uidbuf) - 1, "uid=%u", getuid ());
			g_ptr_array_add (options, uidbuf);
		}
	}
	
	
	/* TODO: read mount point from gconf */
	mount_point = NULL;
	
	/* If we don't have a mount point yet... Use the label of the file system if available */
	if (mount_point == NULL) {
		if (volume != NULL) {
			const char *label;
			label = libhal_volume_get_label (volume);
			if (label != NULL)
				mount_point = g_strdup (label);
		}
		
		/* for testing:
		 * mount_point = g_strdup ("foobar der æøå サイトの"); */
	}
	
	
	/* TODO: read fstype override from gconf */
	fstype_override = NULL;
	
	ret = volume_mount_with_options (udi, volume, drive, mount_point, fstype_override, options);
	
	g_ptr_array_free (options, TRUE);
	
	g_free (mount_point);

	return ret;
}


static gboolean
volume_unmount (const char *udi, LibHalVolume *volume, LibHalDrive *drive)
{
	gboolean ret = FALSE;
	DBusMessage *msg = NULL;
	DBusMessage *reply = NULL;
	DBusError error;
	char **options = NULL;
	unsigned int num_options = 0;

	g_debug ("Unmounting %s", udi);

	msg = dbus_message_new_method_call ("org.freedesktop.Hal", udi,
					    "org.freedesktop.Hal.Device.Volume",
					    "Unmount");
	if (msg == NULL) {
		g_warning ("Could not create dbus message for %s", udi);
		goto out;
	}

	if (!dbus_message_append_args (msg, 
				       DBUS_TYPE_ARRAY, DBUS_TYPE_STRING, &options, num_options,
				       DBUS_TYPE_INVALID)) {
		g_warning ("Could not append args to dbus message for %s", udi);
		goto out;
	}
		
	dbus_error_init (&error);
	if (!(reply = dbus_connection_send_with_reply_and_block (dbus_connection, msg, -1, &error)) ||
	    dbus_error_is_set (&error)) {
		g_warning ("Unmount failed for %s: %s : %s\n", udi, error.name, error.message);
		show_error_dialog_unmount (udi, volume, drive, error.name, error.message);
		dbus_error_free (&error);
		goto out;
	}

	g_print (_("Unmounted %s.\n"), get_dev_file (volume, drive));

	ret = TRUE;

out:
	if (msg != NULL)
		dbus_message_unref (msg);
	if (reply != NULL)
		dbus_message_unref (reply);

	return ret;
}

static gboolean
volume_eject (const char *udi, LibHalVolume *volume, LibHalDrive *drive)
{
	gboolean ret = FALSE;
	DBusMessage *msg = NULL;
	DBusMessage *reply = NULL;
	DBusError error;
	char **options = NULL;
	unsigned int num_options = 0;

	g_debug ("Ejecting %s", udi);
	
	msg = dbus_message_new_method_call ("org.freedesktop.Hal", udi,
					    "org.freedesktop.Hal.Device.Volume",
					    "Eject");
	if (msg == NULL) {
		g_warning ("Could not create dbus message for %s", udi);
		goto out;
	}

	if (!dbus_message_append_args (msg, 
				       DBUS_TYPE_ARRAY, DBUS_TYPE_STRING, &options, num_options,
				       DBUS_TYPE_INVALID)) {
		g_warning ("Could not append args to dbus message for %s", udi);
		goto out;
	}
	
	dbus_error_init (&error);
	if (!(reply = dbus_connection_send_with_reply_and_block (dbus_connection, msg, -1, &error))) {
		g_warning ("Eject failed for %s: %s : %s\n", udi, error.name, error.message);
		dbus_error_free (&error);
		goto out;
	}

	if (dbus_error_is_set (&error)) {
		g_warning ("Eject failed for %s\n%s : %s\n", udi, error.name, error.message);

		show_error_dialog_eject (udi, volume, drive, error.name, error.message);

		dbus_error_free (&error);
		goto out;
	}

	g_print (_("Ejected %s.\n"), get_dev_file (volume, drive));

	ret = TRUE;

out:
	if (msg != NULL)
		dbus_message_unref (msg);
	if (reply != NULL)
		dbus_message_unref (reply);
	return ret;
}


static LibHalVolume *
volume_findby (LibHalContext *hal_ctx, const char *property, const char *value)
{
	int i;
	char **hal_udis;
	int num_hal_udis;
	LibHalVolume *result = NULL;
	char *found_udi = NULL;
	DBusError error;

	dbus_error_init (&error);
	if ((hal_udis = libhal_manager_find_device_string_match (hal_ctx, property, 
								 value, &num_hal_udis, &error)) == NULL)
		goto out;

	for (i = 0; i < num_hal_udis; i++) {
		char *udi;
		udi = hal_udis[i];
		if (libhal_device_query_capability (hal_ctx, udi, "volume", &error)) {
			found_udi = strdup (udi);
			break;
		}
	}

	libhal_free_string_array (hal_udis);

	if (found_udi != NULL)
		result = libhal_volume_from_udi (hal_ctx, found_udi);

	free (found_udi);
out:
	return result;
}

static LibHalVolume *
volume_from_nickname (LibHalContext *hal_ctx, const char *name)
{
	int name_len;
	char *path;
	LibHalVolume *result = NULL;

	if (name[0] == '/') {
		result = volume_findby (hal_ctx, "volume.mount_point", name);
	}

	name_len = strlen (name);
	if ((result == NULL) && ((path = calloc (1, sizeof ("/media/") + name_len + 1)) != NULL)) {
		strcat(path, "/media/");
		strcat(path, name);
		result = volume_findby (hal_ctx, "volume.mount_point", path);
		free(path);
	}

	if (result == NULL) {
		result = volume_findby (hal_ctx, "volume.label", name);
	}

	return (result);
}


static char *
lookup_password (const char *udi, gboolean *is_session)
{
	char *password;
	GList *keyring_result;

	password = NULL;

	if (gnome_keyring_find_network_password_sync (g_get_user_name (),     /* user */
						      NULL,                   /* domain */
						      udi,                    /* server */
						      "password",             /* object */
						      "org.gnome.Mount",      /* protocol */
						      NULL,                   /* authtype */
						      0,                      /* port */
						      &keyring_result) != GNOME_KEYRING_RESULT_OK)
		return FALSE;
	
	if (keyring_result != NULL && g_list_length (keyring_result) == 1) {
		GnomeKeyringNetworkPasswordData *data1 = keyring_result->data;

		if (strcmp (data1->object, "password") == 0) {
			password = g_strdup (data1->password);
		}

		if (password != NULL) {
			if (strcmp (data1->keyring, "session") == 0)
				*is_session = TRUE;
			else
				*is_session = FALSE;
		}

		gnome_keyring_network_password_list_free (keyring_result);
	}

	return password;
}

static void 
save_password (const char *udi, const char *keyring, const char *password)
{
	guint32 item_id;
	GnomeKeyringResult keyring_result;

	keyring_result = gnome_keyring_set_network_password_sync (keyring,            /* keyring  */
								  g_get_user_name (), /* user     */
								  NULL,               /* domain   */
								  udi,                /* server   */
								  "password",         /* object   */
								  "org.gnome.Mount",  /* prtocol  */
								  NULL,               /* authtype */
								  0,                  /* port     */
								  password,           /* password */
								  &item_id);          /* item_id  */
	if (keyring_result != GNOME_KEYRING_RESULT_OK)
	{
		g_warning ("Couldn't store password in keyring, code %d", (int) keyring_result);
	}
}


static char *
get_password (const char *udi, LibHalVolume *volume, LibHalDrive *drive, gboolean retry)
{
	char            *drive_name;
	char            *result;
	char            *prompt;
	GtkWidget	*dialog;
	GnomePasswordDialogRemember remember;
	char            *keyring_password;
	gboolean         keyring_is_session;
	LibHalDrive     *drv;
	char            *model;
	char            *vendor;


	keyring_password = NULL;
	keyring_is_session = FALSE;

	result = NULL;

	model = NULL;
	vendor = NULL;
	if (drive != NULL) {
		model = g_strdup (libhal_drive_get_model (drive));
		vendor = g_strdup (libhal_drive_get_vendor (drive));
	} else if (volume != NULL) {
		drive = libhal_drive_from_udi (hal_ctx, 
					       libhal_volume_get_storage_device_udi (volume));
		if (drive != NULL) {
			model = g_strdup (libhal_drive_get_model (drive));
			vendor = g_strdup (libhal_drive_get_vendor (drive));
			libhal_drive_free (drive);
			drive = NULL;
		}
	}

	if (vendor != NULL && model != NULL)
		drive_name = g_strdup_printf (" %s %s", vendor, model);
	else if (vendor != NULL)
		drive_name = g_strdup_printf (" %s", vendor);
	else if (model != NULL)
		drive_name = g_strdup_printf (" %s", model);
	else
		drive_name = g_strdup ("");
		
	keyring_password = lookup_password (udi, &keyring_is_session);

	if (!retry)
	{
		if (keyring_password != NULL) {
			result = g_strdup (keyring_password);
			goto out;
		}
	}

	/* ask on console if we don't have a display */
	if (opt_nodisplay) {
		char *password;
		char *prompt;

		prompt = g_strdup_printf (_("Enter password to unlock encrypted data for %s: "), 
					  get_dev_file (volume, drive));

		password = getpass (prompt);
		if (password != NULL)
			result = g_strdup (password);
		goto out;
	}

	if (opt_noui) {
		g_warning ("Not showing password dialog since invoked with --no-ui");
		goto out;
	}

	/* We don't really want to block - tell parent it's going to
	 * be alright even though we don't know yet.. 
	 *
	 * If we don't do this Nautilus will put up a "press cancel"
	 * dialog when we ask for the password..
	 */
	notify_parent (TRUE);

	prompt = g_strdup_printf (
		_("The storage device%s contains encrypted data. Enter a password to unlock."), drive_name);
	
	dialog = gnome_password_dialog_new (_("Unlock Encrypted Data"), prompt, NULL, NULL, FALSE);
	g_free (prompt);

	gnome_password_dialog_set_show_username (GNOME_PASSWORD_DIALOG (dialog), FALSE);
	gnome_password_dialog_set_show_userpass_buttons (GNOME_PASSWORD_DIALOG (dialog), FALSE);
	gnome_password_dialog_set_show_domain (GNOME_PASSWORD_DIALOG (dialog), FALSE);
	gnome_password_dialog_set_show_remember (GNOME_PASSWORD_DIALOG (dialog), TRUE);
	/* use the same keyring storage options as from the items we put in the entry boxes */
	remember = GNOME_PASSWORD_DIALOG_REMEMBER_NOTHING;

	/* use the same keyring storage options as from the items we put in the entry boxes */
	remember = GNOME_PASSWORD_DIALOG_REMEMBER_NOTHING;
	if (keyring_password != NULL) {
		if (keyring_is_session)
			remember = GNOME_PASSWORD_DIALOG_REMEMBER_SESSION;
		else
			remember = GNOME_PASSWORD_DIALOG_REMEMBER_FOREVER;				
	}

	gnome_password_dialog_set_remember (GNOME_PASSWORD_DIALOG (dialog), remember);

	/* if retrying, put in the passwords from the keyring */
	if (keyring_password != NULL) {
		gnome_password_dialog_set_password (GNOME_PASSWORD_DIALOG (dialog), keyring_password);
	}

	gtk_widget_show (dialog);

	if (gnome_password_dialog_run_and_block (GNOME_PASSWORD_DIALOG (dialog)))
	{
		char *password;

		password = gnome_password_dialog_get_password (GNOME_PASSWORD_DIALOG (dialog));
		result = password;

		switch (gnome_password_dialog_get_remember (GNOME_PASSWORD_DIALOG (dialog)))
		{
			case GNOME_PASSWORD_DIALOG_REMEMBER_SESSION:
				save_password (udi, "session", password);
				break;
			case GNOME_PASSWORD_DIALOG_REMEMBER_FOREVER:
				save_password (udi, NULL, password);
				break;
			default:
				break;
		}

	}

	gtk_widget_destroy (dialog);

out:
	g_free (keyring_password);
	g_free (drive_name);
	return result;
}

static gboolean
setup_crypto (const char *udi, LibHalVolume *volume, LibHalDrive *drive, 
 	      const char *password, gboolean *password_error)
{
	gboolean ret = FALSE;
	DBusMessage *msg = NULL;
	DBusMessage *reply = NULL;
	DBusError error;

	*password_error = FALSE;

	g_debug ("Setting up %s for crypto", udi);
	
	msg = dbus_message_new_method_call ("org.freedesktop.Hal", udi,
					    "org.freedesktop.Hal.Device.Volume.Crypto",
					    "Setup");
	if (msg == NULL) {
		g_warning ("Could not create dbus message for %s", udi);
		goto out;
	}

	if (!dbus_message_append_args (msg, 
				       DBUS_TYPE_STRING, &password,
				       DBUS_TYPE_INVALID)) {
		g_warning ("Could not append args to dbus message for %s", udi);
		goto out;
	}
	
	dbus_error_init (&error);
	if (!(reply = dbus_connection_send_with_reply_and_block (dbus_connection, msg, -1, &error)) || 
	    dbus_error_is_set (&error)) {
		g_warning ("Setup failed for %s: %s : %s\n", udi, error.name, error.message);
		if (strcmp (error.name, "org.freedesktop.Hal.Device.Volume.Crypto.SetupPasswordError") == 0) {
			*password_error = TRUE;
		}
		dbus_error_free (&error);
		goto out;
	}


	ret = TRUE;

	g_print (_("Setup clear-text device for %s.\n"), get_dev_file (volume, drive));

out:
	if (msg != NULL)
		dbus_message_unref (msg);
	if (reply != NULL)
		dbus_message_unref (reply);

	return ret;
}

static gboolean
teardown_crypto (const char *udi, LibHalVolume *volume, LibHalDrive *drive)
{
	gboolean ret = FALSE;
	DBusMessage *msg = NULL;
	DBusMessage *reply = NULL;
	DBusError error;
	char *clear_udi;

	g_debug ("Tearing down %s for crypto", udi);
	
	clear_udi = libhal_volume_crypto_get_clear_volume_udi (hal_ctx, volume);
	if (clear_udi != NULL) {
		LibHalVolume *clear_volume;
		gboolean unmount_child_failed;

		unmount_child_failed = FALSE;

		clear_volume = libhal_volume_from_udi (hal_ctx, clear_udi);
		if (clear_volume != NULL) {
			if (libhal_volume_is_mounted (clear_volume)) {
				if (!volume_unmount (clear_udi, clear_volume, NULL))
					unmount_child_failed = TRUE;
			}
			libhal_volume_free (clear_volume);
		}
		free (clear_udi);

		if (unmount_child_failed)
			goto out;
	}


	msg = dbus_message_new_method_call ("org.freedesktop.Hal", udi,
					    "org.freedesktop.Hal.Device.Volume.Crypto",
					    "Teardown");
	if (msg == NULL) {
		g_warning ("Could not create dbus message for %s", udi);
		goto out;
	}

	if (!dbus_message_append_args (msg, 
				       DBUS_TYPE_INVALID)) {
		g_warning ("Could not append args to dbus message for %s", udi);
		goto out;
	}
	
	dbus_error_init (&error);
	if (!(reply = dbus_connection_send_with_reply_and_block (dbus_connection, msg, -1, &error)) || 
	    dbus_error_is_set (&error)) {
		g_warning ("Teardown failed for %s: %s : %s\n", udi, error.name, error.message);
		dbus_error_free (&error);
		goto out;
	}

	g_print (_("Teared down clear-text device for %s.\n"), get_dev_file (volume, drive));

	ret = TRUE;

out:
	if (msg != NULL)
		dbus_message_unref (msg);
	if (reply != NULL)
		dbus_message_unref (reply);

	return ret;
}

static const char *crypto_setup_backing_udi = NULL;
static int crypto_setup_rc = 1;

static void
crypto_setup_device_removed (LibHalContext *ctx, const char *udi)
{
	g_debug ("In crypto_setup_device_removed for %s", udi);

	if (strcmp (udi, crypto_setup_backing_udi) == 0) {
		g_debug ("Backing volume removed! Exiting..");
		gtk_exit (1);		
	}
}

static void
crypto_setup_device_added (LibHalContext *ctx, const char *udi)
{
	const char *backing_udi;
	DBusError error;
	LibHalVolume *volume;

	g_debug ("In crypto_setup_device_added for %s", udi);

	backing_udi = NULL;
	volume = libhal_volume_from_udi (hal_ctx, udi);
	if (volume != NULL) {
		backing_udi = libhal_volume_crypto_get_backing_volume_udi (volume);
		if (backing_udi != NULL) {
			if (strcmp (backing_udi, crypto_setup_backing_udi) == 0) {
				LibHalVolumeUsage fsuage;

				g_debug ("%s is backed by %s - will mount", udi, backing_udi);

				g_print (_("Clear text device is %s. Mounting.\n"), get_dev_file (volume, NULL));
					
				fsuage = libhal_volume_get_fsusage (volume);
					
				if (fsuage == LIBHAL_VOLUME_USAGE_MOUNTABLE_FILESYSTEM) {
					if (volume_mount (udi, volume, NULL))
						crypto_setup_rc = 0;
				} else {
					g_warning ("%s does not have a mountable filesystem", udi);
				}
				
				notify_parent (crypto_setup_rc == 0);
				gtk_main_quit ();				
			}
		}
		
		libhal_volume_free (volume);
	}

}

static gboolean
crypto_setup_timeout (gpointer data)
{
	g_warning ("Timeout for waiting for cleartext device... Exiting.");
	notify_parent (FALSE);
	gtk_main_quit ();
}


static void 
my_empty_log_handler (const gchar *log_domain,
		      GLogLevelFlags log_level,
		      const gchar *message,
		      gpointer user_data)
{
}


/* borrowed from gtk/gtkfilesystemunix.c in GTK+ on 02/23/2006 */
static void
canonicalize_filename (gchar *filename)
{
	gchar *p, *q;
	gboolean last_was_slash = FALSE;
	
	p = filename;
	q = filename;
	
	while (*p)
	{
		if (*p == G_DIR_SEPARATOR)
		{
			if (!last_was_slash)
				*q++ = G_DIR_SEPARATOR;
			
			last_was_slash = TRUE;
		}
		else
		{
			if (last_was_slash && *p == '.')
			{
				if (*(p + 1) == G_DIR_SEPARATOR ||
				    *(p + 1) == '\0')
				{
					if (*(p + 1) == '\0')
						break;
					
					p += 1;
				}
				else if (*(p + 1) == '.' &&
					 (*(p + 2) == G_DIR_SEPARATOR ||
					  *(p + 2) == '\0'))
				{
					if (q > filename + 1)
					{
						q--;
						while (q > filename + 1 &&
						       *(q - 1) != G_DIR_SEPARATOR)
							q--;
					}
					
					if (*(p + 2) == '\0')
						break;
					
					p += 2;
				}
				else
				{
					*q++ = *p;
					last_was_slash = FALSE;
				}
			}
			else
			{
				*q++ = *p;
				last_was_slash = FALSE;
			}
		}
		
		p++;
	}
	
	if (q > filename + 1 && *(q - 1) == G_DIR_SEPARATOR)
		q--;
	
	*q = '\0';
}

static char *
resolve_symlink (const char *file)
{
	GError *error;
	char *dir;
	char *link;
	char *f;
	char *f1;

	f = g_strdup (file);

	while (g_file_test (f, G_FILE_TEST_IS_SYMLINK)) {
		link = g_file_read_link (f, &error);
		if (link == NULL) {
			g_warning ("Cannot resolve symlink %s: %s", f, error->message);
			g_error_free (error);
			g_free (f);
			f = NULL;
			goto out;
		}
		
		dir = g_path_get_dirname (f);
		f1 = g_strdup_printf ("%s/%s", dir, link);
		g_free (dir);
		g_free (link);
		g_free (f);
		f = f1;
	}

out:
	if (f != NULL)
		canonicalize_filename (f);
	return f;
}

int
main (int argc, char *argv[])
{
	pid_t pid;
	char buf;
	int dev_null_fd;
	LibHalDrive *drive = NULL;
	LibHalVolume *volume = NULL;
	const char *udi;
	char *resolved_device_file = NULL;
	static gboolean opt_verbose = FALSE;
	static gboolean opt_unmount = FALSE;
	static gboolean opt_eject = FALSE;
	static gchar *opt_hal_udi = NULL;
	static gchar *opt_device_file = NULL;
	static gchar *opt_nickname = NULL;
        GError *error = NULL;
        GOptionContext *context = NULL;
        static GOptionEntry entries[] =
                {
                        { "verbose", 'v', 0, G_OPTION_ARG_NONE, &opt_verbose, "Verbose operation", NULL},
                        { "no-ui", 'n', 0, G_OPTION_ARG_NONE, &opt_noui, "Don't show any dialogs", NULL},
                        { "block", 'b', 0, G_OPTION_ARG_NONE, &opt_block, "Allow gnome-mount to block for UI", NULL},
                        { "unmount", 'u', 0, G_OPTION_ARG_NONE, &opt_unmount, "Unmount rather than mount", NULL},
                        { "eject", 'e', 0, G_OPTION_ARG_NONE, &opt_eject, "Eject rather than mount", NULL},
                        { "hal-udi", 'h', 0, G_OPTION_ARG_STRING, &opt_hal_udi, "Mount by HAL UDI", NULL},
                        { "device", 'd', 0, G_OPTION_ARG_STRING, &opt_device_file, "Mount by device file", NULL},
                        { "pseudonym", 'p', 0, G_OPTION_ARG_STRING, &opt_nickname, "Mount by one of device's nicknames: mountpoint, label, with or without directory prefix", NULL},
			{ "text", 't', 0, G_OPTION_ARG_NONE, &opt_nodisplay, "Text-based operation", NULL},
                        { NULL, 0, 0, 0, NULL, NULL, NULL }
                };

	rc = 1;

	bindtextdomain (PACKAGE, GNOMELOCALEDIR);
	bind_textdomain_codeset (PACKAGE, "UTF-8");
	textdomain (PACKAGE);

        context = g_option_context_new ("- GNOME mount");
        g_option_context_add_main_entries (context, entries, PACKAGE);
        g_option_context_add_group (context, gtk_get_option_group (FALSE));
        g_option_context_parse (context, &argc, &argv, &error);

	g_print ("%s\n", PACKAGE_STRING);

	if (!gtk_init_check (&argc, &argv)) {
		g_print (_("X display not available - using text-based operation.\n"));
		opt_nodisplay = TRUE;
	}

	if (opt_nodisplay) {
		opt_block = TRUE;
		opt_noui = TRUE;
	}

	/* If we want to show dialogs, show that in a child process and let the parent exit immediately
	 * so we don't block
	 */
	if (!opt_block) {
		if (pipe (fds) != 0) {
			fprintf (stderr, "Cannot create pipe\n");
			goto error;
		}
		
		pid = fork();
		switch (pid) {
		case 0:
			/* child; do our real work and notify parent when we know the exit code */
			break;
		case -1:
			/* failure */
			fprintf (stderr, "Cannot fork\n");
			goto error;
			
		default:
			/* parent; wait on rc over pipe from child */
			close (fds[1]);
			read (fds[0], &buf, 1);
			if (buf == '1')
				rc = 0;
			goto error;
		}

		/* close unused descriptor */
		close (fds[0]);

		/* we have to complete daemonize, because otherwise the caller
		 * might wait for the to finish too (e.g.  processes using
		 * libgnomevfs such as Nautilus or any application using the
		 * gnomevfs backend of the filechooser).
		 */
		dev_null_fd = open ("/dev/null", O_RDWR);
		/* ignore if we can't open /dev/null */
		if (dev_null_fd >= 0) {
			/* attach /dev/null to stdout, stdin, stderr */
			dup2 (dev_null_fd, 0);
			dup2 (dev_null_fd, 1);
			dup2 (dev_null_fd, 2);
			close (dev_null_fd);
		}
		setsid ();
	}

	if (!opt_noui) {
		gnome_program_init ("gnome-mount", VERSION, LIBGNOMEUI_MODULE,
				    argc, argv, 
				    GNOME_PARAM_NONE);
	}

	if (!opt_verbose) {
		g_log_set_handler (NULL, G_LOG_LEVEL_DEBUG, my_empty_log_handler, NULL);
	}


	hal_ctx = do_hal_init ();
	if (hal_ctx == NULL)
		goto out;

	if (opt_device_file != NULL) {
		resolved_device_file = resolve_symlink (opt_device_file);
		if (resolved_device_file == NULL) {
			goto out;
		}
		if (strcmp (resolved_device_file, opt_device_file) != 0) {
			g_print (_("Resolved device file %s -> %s\n"), opt_device_file, resolved_device_file);
		}
	}
		
	if (opt_hal_udi != NULL) {
		volume = libhal_volume_from_udi (hal_ctx, opt_hal_udi);
	} else if (resolved_device_file != NULL) {
		volume = libhal_volume_from_device_file (hal_ctx, resolved_device_file);
	} else if (opt_nickname != NULL) {
		volume = volume_from_nickname (hal_ctx, opt_nickname);
		if (volume != NULL) {
			g_print (_("Resolved pseudonym \"%s\" -> %s\n"), opt_nickname, get_dev_file (volume, NULL));
		} else {
			g_warning (_("Cannot resolve pseudonym \"%s\" to a volume\n"), opt_nickname);
			goto out;
		}
	} else {
		g_print (_("Use --hal-udi, --device or --pseudonym to specify volume\n"));
		goto out;
	}

	if (volume == NULL) {
		if (opt_hal_udi != NULL) {
			drive = libhal_drive_from_udi (hal_ctx, opt_hal_udi);
		} else if (resolved_device_file != NULL) {
			drive = libhal_drive_from_device_file (hal_ctx, resolved_device_file);
		}

		if (drive != NULL) {
			char **ifs;
			gboolean found;

			found = FALSE;

			ifs = libhal_device_get_property_strlist (hal_ctx,
								  libhal_drive_get_udi (drive),
								  "info.interfaces",
								  NULL);
			if (ifs != NULL) {
				unsigned int i;

				for (i = 0; ifs[i] != NULL; i++) {
					if (strcmp (ifs[i], "org.freedesktop.Hal.Device.Volume") == 0) {
						found = TRUE;
						g_debug ("Will attempts methods on drive object");
						break;
					}
				}
				libhal_free_string_array (ifs);
				
			} 

			if (found) {
				goto try_drive;
			} else {
				g_warning (_("Drive %s does not contain media."), 
					 resolved_device_file != NULL ? resolved_device_file : opt_hal_udi);
				
				notify_parent (FALSE);
				
				show_error_dialog_no_media (NULL, NULL, NULL);
			}

		} else {
			/* Silently fail */
			g_warning (_("Given device '%s' is not a volume or a drive."), 
				 resolved_device_file != NULL ? resolved_device_file : opt_hal_udi);
		}

		goto out;
	}

try_drive:

	/* from here on either volume!=NULL or drive!=NULL */
	if (volume != NULL)
		udi = libhal_volume_get_udi (volume);
	else
		udi = libhal_drive_get_udi (drive);


	/* infer from commandline */
	if (strcmp (g_path_get_basename (argv[0]), "gnome-umount") == 0) {
		opt_unmount = TRUE;
	} else if (strcmp (g_path_get_basename (argv[0]), "gnome-eject") == 0) {
		opt_eject = TRUE;
	}

	if (opt_unmount && opt_eject) {
		g_warning (_("Cannot unmount and eject simultaneously"));
		goto out;
	}

	if (opt_unmount) {
		const char *fstype;
		LibHalVolumeUsage fsusage;

		if (volume != NULL) {
			fstype = libhal_volume_get_fstype (volume);
			fsusage = libhal_volume_get_fsusage (volume);
		} else {
			fstype = "";
			fsusage = LIBHAL_VOLUME_USAGE_UNKNOWN;
		}

		if (fsusage == LIBHAL_VOLUME_USAGE_CRYPTO) {
			if (teardown_crypto (udi, volume, drive))
				rc = 0;
		} else {
			if (volume_unmount (udi, volume, drive)) {

				/* as a convenience also tear down the crypto backing volume if appropriate */
				if (volume != NULL) {
					const char *backing_udi;
					backing_udi = libhal_volume_crypto_get_backing_volume_udi (volume);
					if (backing_udi != NULL) {
						LibHalVolume *backing_volume;
						backing_volume = libhal_volume_from_udi (hal_ctx, backing_udi);
						if (backing_volume != NULL) {
							if (teardown_crypto (backing_udi, backing_volume, NULL))
								rc = 0;
							libhal_volume_free (backing_volume);
						}
					} else {
						rc = 0;
					}
				}
			}
		}

	} else if (opt_eject) {
		if (volume_eject (udi, volume, drive))
			rc = 0;
	} else {
		LibHalVolumeUsage fsusage;

		if (volume != NULL) {
			fsusage = libhal_volume_get_fsusage (volume);
		} else {
			fsusage = LIBHAL_VOLUME_USAGE_UNKNOWN;
		}

		if (fsusage == LIBHAL_VOLUME_USAGE_CRYPTO) {
			char *password;
			gboolean password_error;
			gboolean setup_success;
			char *clear_udi;
			gboolean password_retry;
			int password_num_tries;

			/* see if we're already setup */
			clear_udi = libhal_volume_crypto_get_clear_volume_udi (hal_ctx, volume);
			if (clear_udi != NULL) {
				g_warning (_("Crypto volume '%s' is already setup with clear volume '%s'"), 
					 udi, clear_udi);
				
				free (clear_udi);
				goto out;
			}
			
			g_debug ("Crypto volume - UDI '%s'", udi);
			
			setup_success = FALSE;
			
			/* we need this to catch when the cleartext device is added */
			crypto_setup_backing_udi = udi;
			libhal_ctx_set_device_added (hal_ctx, crypto_setup_device_added);
			/* we need to catch this if the backing device is removed (to remove the password dialog) */
			libhal_ctx_set_device_removed (hal_ctx, crypto_setup_device_removed);

			password_retry = FALSE;
			password_num_tries = 0;
		retry_password:			
			password = get_password (udi, volume, drive, password_retry);
			password_num_tries++;
			if (password != NULL) {
				setup_success = setup_crypto (udi, volume, drive, password, &password_error);
				if (!setup_success && password_error) {
					g_warning (_("Bad crypto password"));
					if (!opt_nodisplay && password_num_tries < 3) {
						g_free (password);
						password_retry = TRUE;
						goto retry_password;
					} else {
						g_warning (_("Bailing out..."));
					}
				}
			}
			
			if (setup_success) {
				
				/* wait as we try to mount the cleartext volume */
				g_debug ("Waiting for cleartext volume backed by %s..", udi);
				/* add a timeout as we don't want to wait forever */
				g_timeout_add (10 * 1000, crypto_setup_timeout, NULL);
				gtk_main();
				
				rc = crypto_setup_rc;
			}
			
			g_free (password);
			
			goto out;

		} else if (fsusage == LIBHAL_VOLUME_USAGE_MOUNTABLE_FILESYSTEM) {
			if (volume_mount (udi, volume, drive))
				rc = 0;
		}

	}

	notify_parent (rc == 0);

out:	
 	if (drive != NULL)
		libhal_volume_free (volume);

 	if (volume != NULL)
		libhal_volume_free (volume);

	if (hal_ctx != NULL)
		libhal_ctx_free (hal_ctx);

	if (resolved_device_file != NULL)
		g_free (resolved_device_file);

error:
	if (context != NULL)
		g_option_context_free (context);

	return rc;
}
