/*
 *  filter_tc_audio.c
 *
 *  Copyright (C) Tilmann Bitterberg - August 2002
 *
 *  This file is part of transcode, a linux video stream processing tool
 *      
 *  transcode is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  transcode is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#define MOD_NAME    "filter_tc_audio.so"
#define MOD_VERSION "v0.1 (2002-08-13)"
#define MOD_CAP     "audio 23.9 -> 29.9 telecide filter"

#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>

/* -------------------------------------------------
 *
 * mandatory include files
 *
 *-------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "transcode.h"
#include "framebuffer.h"
#include "optstr.h"

/*-------------------------------------------------
 *
 * single function interface
 *
 *-------------------------------------------------*/


// telecine filter

int tc_filter(aframe_list_t *ptr, char *options)
{

  static vob_t *vob=NULL;
  static char *audio_buf[2] = {NULL, NULL};
  double fch;
  int leap_bytes1, leap_bytes2;

  // API explanation:
  // ================
  //
  // (1) need more infos, than get pointer to transcode global 
  //     information structure vob_t as defined in transcode.h.
  //
  // (2) 'tc_get_vob' and 'verbose' are exported by transcode.
  //
  // (3) filter is called first time with TC_FILTER_INIT flag set.
  //
  // (4) make sure to exit immediately if context (video/audio) or 
  //     placement of call (pre/post) is not compatible with the filters 
  //     intended purpose, since the filter is called 4 times per frame.
  //
  // (5) see framebuffer.h for a complete list of frame_list_t variables.
  //
  // (6) filter is last time with TC_FILTER_CLOSE flag set


  //----------------------------------
  //
  // filter init
  //
  //----------------------------------


  if (ptr->tag & TC_VIDEO)
	  return (0);

  if(ptr->tag & TC_FILTER_INIT) {
    
    if((vob = tc_get_vob())==NULL) return(-1);
    
    // filter init ok.
    
    if(verbose) printf("[%s] %s %s\n", MOD_NAME, MOD_VERSION, MOD_CAP);
    
    if(verbose) printf("[%s] options=%s\n", MOD_NAME, options);

    /* thank god there is no write protection for variables */
    
    // if keep_ifps is supplied, do NOT change import_audio size 

    if (options && optstr_lookup (options, "keep_ifps")) {
      ;
    } else {

	// copied verbatim from transcode.c
	fch = vob->a_rate/NTSC_FILM;

	// bytes per audio frame
	vob->im_a_size = (int)(fch * (vob->dm_bits/8) * vob->dm_chan);
	vob->im_a_size =  (vob->im_a_size>>2)<<2;

	// rest:
	fch *= (vob->dm_bits/8) * vob->dm_chan;

	leap_bytes1 = TC_LEAP_FRAME * (fch - vob->im_a_size);
	leap_bytes2 = - leap_bytes1 + TC_LEAP_FRAME * (vob->dm_bits/8) * vob->dm_chan;
	leap_bytes1 = (leap_bytes1 >>2)<<2;
	leap_bytes2 = (leap_bytes2 >>2)<<2;

	if(leap_bytes1<leap_bytes2) {
	  vob->a_leap_bytes = leap_bytes1;
	} else {
	  vob->a_leap_bytes = -leap_bytes2;
	  vob->im_a_size += (vob->dm_bits/8) * vob->dm_chan;
	}
    }

    if (!audio_buf[0] && !audio_buf[1]) {
	audio_buf[0] = malloc (SIZE_PCM_FRAME);
	audio_buf[1] = malloc (SIZE_PCM_FRAME);
	if (!audio_buf[0] || !audio_buf[1]) {
	    fprintf(stderr, "[%s] [%s:%d] malloc failed\n", MOD_NAME, __FILE__, __LINE__);
	    return (-1);
	}
    }
    if (verbose & TC_DEBUG)
      printf("[%s] changing audio bufsize (%d) -> (%d)\n", MOD_NAME, vob->im_a_size, vob->ex_a_size);
    
    return(0);
  }

  //----------------------------------
  //
  // filter close
  //
  //----------------------------------

  
  if(ptr->tag & TC_FILTER_CLOSE) {
    return(0);
  }
  
  //----------------------------------
  //
  // filter frame routine
  //
  //----------------------------------

  // tag variable indicates, if we are called before
  // transcodes internal video/audo frame processing routines
  // or after and determines video/audio context

  /* pass-through */
  if (ptr->id == 0)
      return 0;

  /* in:      T1 B1 | T2 B2 | T3 B3 | T4 B4 */
  /* out: T1 B1 | T2 B2 | T2 B3 | T3 B4 | T4 B4 */

  // rearrange the audio buffers, do not add or delete data
  // 4*8008 -> 5*6408

  /* XXX: Is this really working??  appearantly -- tibit*/
  if(ptr->tag & TC_POST_S_PROCESS && ptr->tag & TC_AUDIO) {
    int mod = ptr->id % 4;
    int ex = vob->ex_a_size; // 6408
    //    int diff = im - ex;
    int diff = ex/4;

    switch (mod) {
      case 1:
	  tc_memcpy (audio_buf[0], ptr->audio_buf+ex, 1*diff);
	  ptr->audio_size=ex;
	break;
      case 2:
	  tc_memcpy (audio_buf[0]+1*diff , ptr->audio_buf           , ex-1*diff);
	  tc_memcpy (audio_buf[1]        , ptr->audio_buf+ex-1*diff , 2*diff);
	  tc_memcpy (ptr->audio_buf      , audio_buf[0]             , ex);
	  ptr->audio_size=ex;
	break;
      case 3:
	  tc_memcpy (audio_buf[1]+2*diff , ptr->audio_buf          , ex-2*diff);
	  tc_memcpy (audio_buf[0]        , ptr->audio_buf+ex-2*diff, 3*diff);
	  tc_memcpy (ptr->audio_buf      , audio_buf[1]            , ex);
	  ptr->audio_size=ex;
	break;
      case 0:
	if (!(ptr->attributes & TC_FRAME_WAS_CLONED)) {
	    ptr->attributes |= TC_FRAME_IS_CLONED;

	    if (verbose & TC_DEBUG)
	      printf("[A] frame cloned (%d)\n", ptr->id);

	    tc_memcpy (audio_buf[0]+3*diff , ptr->audio_buf          , ex-3*diff);
	    tc_memcpy (audio_buf[1]        , ptr->audio_buf+ex-3*diff, 4*diff);
	    tc_memcpy (ptr->audio_buf      , audio_buf[0]            , ex);
	    ptr->audio_size=ex;
	} else {
	  //	    tc_memcpy (audio_buf[1]+4*diff , ptr->audio_buf          , ex-4*diff);
	    tc_memcpy (ptr->audio_buf      , audio_buf[1]            , ex);
	    ptr->audio_size=ex;
	  
	}
	break;
    }
    
  }
  
  return(0);
}
