/* Subroutine:  psx_malloc( size, pntr, status )
*+
*  Name:
*     PSX_MALLOC

*  Purpose:
*     Allocate virtual memory

*  Language:
*     ANSI C

*  Invocation:
*     CALL PSX_MALLOC( SIZE, PNTR, STATUS )

*  Description:
*     The routine allocates an amount of virtual memory specified by
*     SIZE. The memory returned is not initialized. The unit of SIZE is
*     the amount of storage required to store a single character. A
*     pointer to the allocated storage is returned in PNTR. This
*     pointer can be passed on to other subroutines using the %VAL
*     construct. If the storage cannot be allocated, then PNTR is set
*     to zero, STATUS is set to PSX__NOALL and an error is reported.

*  Arguments:
*     SIZE = INTEGER (Given)
*        The amount of virtual memory to be allocated.
*     PNTR = POINTER (Returned)
*        A pointer to the allocated storage.
*     STATUS = INTEGER (Given and Returned)
*        The global status.

*  Examples:
*        CALL PSX_MALLOC( 40, PNTR, STATUS )
*        CALL SUB1( %VAL(PNTR), 10, STATUS )
*           ...
*        SUBROUTINE SUB1( ARRAY, N, STATUS )
*        INTEGER N
*        INTEGER ARRAY( N )
*           ...
*
*        Allocates 40 bytes and uses this as a 10 element integer array.
*
*        The call to PSX_MALLOC allocates forty bytes of storage. The
*        pointer to this storage is then passed to subroutine SUB1,
*        where it is accessed as an array of integers.
*        We assume SUB1 returns without action if STATUS is bad.
*
*        Note that here the program needs to know that an INTEGER variable
*        is stored in four bytes. THIS IS NOT PORTABLE. In such a case
*        it is better to use PSX_CALLOC or to use the symbolic
*        constants NUM_NB<T> defined in the file PRM_PAR to determine
*        the number of bytes per unit of storage. (See SUN/39 for a
*        description of these constants).

*  Notes:
*     -  Storage allocated by PSX_MALLOC should be returned by a call to
*        PSX_FREE when it is no longer needed.
*     -  PNTR is declared to be of type POINTER. This is usually
*        represented in FORTRAN as an INTEGER, although any type that
*        uses the same amount of storage would be just as good.
*        The pointer will have been registered for C and FORTRAN use,
*        according to the scheme described in SUN/209, allowing its use
*        where pointers are longer than INTEGERs. For portability, the
*        construct %VAL(CNF_PVAL(PNTR)), rather than simply %VAL(PNTR),
*        should be used to pass the pointer to the subroutine. Function
*        CNF_PVAL is described in SUN/209 Section `Pointers'.
*     -  If several calls to PSX_MALLOC are made, the space returned by
*        each call is completely separate from that made by any other
*        call. In particular, the program should not assume that the
*        space returned by successive calls is contiguous.

*  External Routines Used:
*     cnf: cnfMalloc, cnfFptr

*  References:
*     -  POSIX standard (1988), section 8.1
*     -  ANSI C standard (1989), section 4.10.3.3
      
*  Copyright:
*     Copyright (C) 1991 Science & Engineering Research Council

*  Authors:
*     PMA: Peter Allan (Starlink, RAL)
*     RFWS: R.F. Warren-Smith (Starlink, RAL)
*     AJC: A.J. Chipperfield (Starlink, RAL)
*     {enter_new_authors_here}

*  History:
*     5-MAR-1991 (PMA):
*        Original version.
*     8-APR-1991 (PMA):
*        Added call to ems_rep_c to report errors.
*     15-APR-1991 (PMA):
*        Changed calls to ems to calls to psx1.
*     3-MAY-1991 (PMA):
*        Ensure that a value is returned in PNTR, even on an error.
*     27-JUN-1991 (PMA):
*        Changed IMPORT and EXPORT macros to GENPTR.
*     14-APR-1993 (PMA):
*        Cast the temporary pointer to an F77 pointer.
*     14-APR-199 (RFWS):
*        Use CNF for memory allocation.
*     23-JUN-2000 (AJC):
*        Improve documentation re pointers
*        Tidy refs to CNF routines
*     {enter_further_changes_here}

*  Bugs:
*     {note_any_bugs_here}

*-
-----------------------------------------------------------------------------
*/


/* Global Constants:		.					    */

#include <stdlib.h>		 /* Standard C library			    */
#include "f77.h"		 /* C - Fortran interface		    */
#include "psx_err.h"		 /* PSX error values			    */
#include "psx1.h"		 /* Internal PSX routines		    */
#include "sae_par.h"		 /* ADAM constants			    */


F77_SUBROUTINE(psx_malloc)( INTEGER(size), POINTER(pntr), INTEGER(status) )
{

/* Pointers to Arguments:						    */

   GENPTR_INTEGER(size)
   GENPTR_POINTER(pntr)
   GENPTR_INTEGER(status)

/* Local Variables:							    */

   void *temp;			 /* Temporary return value from malloc.	    */
   char errbuf[100];		 /* Buffer for error message		    */

/* Check inherited global status.					    */

   if( *status != SAI__OK ) return;

/* Allocate the space (use CNF so that the resulting pointer can be used    */
/* from both C and Fortran).						    */

   temp = cnfMalloc( (size_t)( *size ) );

/* Check that the space was allocated.					    */

   if( temp != 0 )

/* Copy the pointer to the allocated storage space to the subroutine	    */
/* argument, converting to a Fortran pointer.				    */

   {
      *pntr = cnfFptr( temp );
   }
   else

/* Set STATUS to an error code and report the error.			    */

   {
      *pntr = (F77_POINTER_TYPE)0;
      *status = PSX__NOALL;
      sprintf( errbuf, 
         "Failed to allocate space with malloc. %d bytes requested",
         *size );
      psx1_rep_c( "PSX_MALLOC_NOALL", errbuf, status );
   }

}
