// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef UI_VIEWS_METADATA_VIEW_FACTORY_H_
#define UI_VIEWS_METADATA_VIEW_FACTORY_H_

#include <functional>
#include <map>
#include <memory>
#include <utility>

#include "base/strings/string16.h"
#include "ui/base/class_property.h"
#include "ui/views/metadata/type_conversion.h"
#include "ui/views/metadata/view_factory_internal.h"
#include "ui/views/views_export.h"

namespace views {

template <typename Builder>
class BaseViewBuilderT : public internal::ViewBuilderCore {
 public:
  using ViewClass_ = typename internal::ViewClassTrait<Builder>::ViewClass_;
  BaseViewBuilderT() = default;
  BaseViewBuilderT(BaseViewBuilderT&&) = default;
  BaseViewBuilderT& operator=(BaseViewBuilderT&&) = default;
  ~BaseViewBuilderT() override = default;

  Builder& CopyAddressTo(ViewClass_** view_address) {
    view_address_ = view_address;
    return *static_cast<Builder*>(this);
  }

  template <typename Child>
  Builder& AddChild(Child& child) {
    children_.push_back(child);
    return *static_cast<Builder*>(this);
  }

  Builder& AddChildren(
      const std::initializer_list<
          std::reference_wrapper<internal::ViewBuilderCore>>& children) {
    children_.insert(children_.end(), children.begin(), children.end());
    return *static_cast<Builder*>(this);
  }

  std::unique_ptr<ViewClass_> Build() {
    auto view = std::make_unique<ViewClass_>();
    SetProperties(view.get());
    CreateChildren(view.get());
    if (view_address_)
      *view_address_ = view.get();
    return view;
  }

  template <typename T>
  Builder& SetProperty(const ui::ClassProperty<T>* property,
                       metadata::ArgType<T> value) {
    auto setter =
        std::make_unique<internal::ClassPropertySetter<ViewClass_, T>>(
            property, std::move(value));
    internal::ViewBuilderCore::AddPropertySetter(std::move(setter));
    return *static_cast<Builder*>(this);
  }

 protected:
  std::unique_ptr<View> DoBuild() override { return Build(); }

  ViewClass_** view_address_ = nullptr;
};

// Example of builder class generated by the following macros.
//
// template <typename Builder, typename ViewClass>
// class ViewBuilderT : public BaseViewBuilderT<Builder, ViewClass> {
//  public:
//   ViewBuilderT() = default;
//   ViewBuilderT(const ViewBuilderT&) = default;
//   ~ViewBuilderT() override = default;
//   ViewBuilderT& operator=(const ViewBuilderT&) = default;
//
//   Builder& SetEnabled(bool value) {
//     auto setter = std::make_unique<
//         PropertySetter<ViewClass, bool, decltype(&ViewClass::SetEnabled),
//         &ViewClass::SetEnabled>>(value);
//     ViewBuilderCore::AddPropertySetter(std::move(setter));
//     return *static_cast<Builder*>(this);
//   }
//
//   Builder& SetVisible(bool value) {
//     auto setter = std::make_unique<
//         PropertySetter<ViewClass, bool, &ViewClass::SetVisible>>(value);
//     ViewBuilderCore::AddPropertySetter(std::move(setter));
//     return *static_cast<Builder*>(this);
//   }
// };
//
// class VIEWS_EXPORT ViewBuilderTest
//     : public ViewBuilderT<ViewBuilderTest, View> {};
//
// template <typename Builder, typename ViewClass>
// class LabelButtonBuilderT : public ViewBuilderT<Builder, ViewClass> {
//  public:
//   LabelButtonBuilderT() = default;
//   LabelButtonBuilderT(LabelButtonBuilderT&&) = default;
//   ~LabelButtonBuilderT() override = default;
//   LabelButtonBuilderT& operator=(LabelButtonBuilderT&&) = default;
//
//   Builder& SetIsDefault(bool value) {
//     auto setter = std::make_unique<
//         PropertySetter<ViewClass, bool, decltype(&ViewClass::SetIsDefault),
//         &ViewClass::SetIsDefault>>(value);
//     ViewBuilderCore::AddPropertySetter(std::move(setter));
//     return *static_cast<Builder*>(this);
//   }
// };
//
// class VIEWS_EXPORT LabelButtonBuilderTest
//     : public LabelButtonBuilderT<LabelButtonBuilderTest, LabelButton> {};

#define BEGIN_VIEW_BUILDER(export, view_class, ancestor)                    \
  template <typename BuilderT>                                              \
  class export view_class##BuilderT : public ancestor##BuilderT<BuilderT> { \
   public:                                                                  \
    using ViewClass_ = view_class;                                          \
                                                                            \
    view_class##BuilderT() = default;                                       \
    view_class##BuilderT(view_class##BuilderT&&) = default;                 \
    view_class##BuilderT& operator=(view_class##BuilderT&&) = default;      \
    ~view_class##BuilderT() override = default;

#define VIEW_BUILDER_PROPERTY(property_type, property_name)                   \
  BuilderT& Set##property_name(                                               \
      ::views::metadata::ArgType<property_type> value) {                      \
    auto setter = std::make_unique<::views::internal::PropertySetter<         \
        ViewClass_, property_type, decltype(&ViewClass_::Set##property_name), \
        &ViewClass_::Set##property_name>>(std::move(value));                  \
    ::views::internal::ViewBuilderCore::AddPropertySetter(std::move(setter)); \
    return *static_cast<BuilderT*>(this);                                     \
  }

#define VIEW_BUILDER_METHOD(method_name)                                      \
  BuilderT& method_name() {                                                   \
    auto caller = std::make_unique<::views::internal::ClassMethodCaller<      \
        ViewClass_, decltype(&ViewClass_::method_name),                       \
        &ViewClass_::method_name>>();                                         \
    ::views::internal::ViewBuilderCore::AddPropertySetter(std::move(caller)); \
    return *static_cast<BuilderT*>(this);                                     \
  }

#define VIEW_BUILDER_VIEW_TYPE_PROPERTY(property_type, property_name)         \
  BuilderT& Set##property_name(                                               \
      std::reference_wrapper<::views::internal::ViewBuilderCore> value) {     \
    auto setter = std::make_unique<::views::internal::ViewBuilderSetter<      \
        ViewClass_, property_type,                                            \
        decltype(&ViewClass_::Set##property_name<property_type>),             \
        &ViewClass_::Set##property_name<property_type>>>(value);              \
    ::views::internal::ViewBuilderCore::AddPropertySetter(std::move(setter)); \
    return *static_cast<BuilderT*>(this);                                     \
  }

#define VIEW_BUILDER_VIEW_PROPERTY(property_type, property_name)              \
  BuilderT& Set##property_name(                                               \
      std::reference_wrapper<::views::internal::ViewBuilderCore> value) {     \
    auto setter = std::make_unique<::views::internal::ViewBuilderSetter<      \
        ViewClass_, property_type, decltype(&ViewClass_::Set##property_name), \
        &ViewClass_::Set##property_name>>(value);                             \
    ::views::internal::ViewBuilderCore::AddPropertySetter(std::move(setter)); \
    return *static_cast<BuilderT*>(this);                                     \
  }

#define VIEW_BUILDER_PROPERTY_DEFAULT(property_type, property_name, default)  \
  BuilderT& Set##property_name(                                               \
      ::views::metadata::ArgType<property_type> value = default) {            \
    auto setter = std::make_unique<::views::internal::PropertySetter<         \
        ViewClass_, property_type, decltype(&ViewClass_::Set##property_name), \
        &ViewClass_::Set##property_name>>(std::move(value));                  \
    ::views::internal::ViewBuilderCore::AddPropertySetter(std::move(setter)); \
    return *static_cast<BuilderT*>(this);                                     \
  }

// Turn off clang-format due to it messing up the following macro. Places the
// semi-colon on a separate line.
// clang-format off

#define END_VIEW_BUILDER(export, view_class) \
  };                                         \
                                             \
  template <>                                \
  class export Builder<view_class>           \
      : public view_class##BuilderT<views::Builder<view_class>>{};

// clang-format on

}  // namespace views

#endif  // UI_VIEWS_METADATA_VIEW_FACTORY_H_
