// Copyright 2018 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "ui/ozone/platform/scenic/scenic_surface.h"

#include <lib/ui/scenic/cpp/commands.h>
#include <lib/ui/scenic/cpp/view_token_pair.h>
#include <lib/zx/eventpair.h>

#include "ui/ozone/platform/scenic/scenic_gpu_host.h"
#include "ui/ozone/platform/scenic/scenic_surface_factory.h"

namespace ui {

ScenicSurface::ScenicSurface(
    ScenicSurfaceFactory* scenic_surface_factory,
    gfx::AcceleratedWidget window,
    scenic::SessionPtrAndListenerRequest sesion_and_listener_request)
    : scenic_session_(std::move(sesion_and_listener_request)),
      shape_(&scenic_session_),
      scenic_surface_factory_(scenic_surface_factory),
      window_(window) {
  // Setting alpha to 0 makes this transparent.
  scenic::Material transparent_material(&scenic_session_);
  transparent_material.SetColor(0, 0, 0, 0);
  shape_.SetShape(scenic::Rectangle(&scenic_session_, 1.f, 1.f));
  shape_.SetMaterial(transparent_material);
  scenic_surface_factory->AddSurface(window, this);
  scenic_session_.SetDebugName("Chromium ScenicSurface");
}

ScenicSurface::~ScenicSurface() {
  DCHECK_CALLED_ON_VALID_THREAD(thread_checker_);
  scenic_surface_factory_->RemoveSurface(window_);
}

bool ScenicSurface::SetTextureToNewImagePipe(
    fidl::InterfaceRequest<fuchsia::images::ImagePipe2> image_pipe_request) {
  DCHECK_CALLED_ON_VALID_THREAD(thread_checker_);
  uint32_t image_pipe_id = scenic_session_.AllocResourceId();
  scenic_session_.Enqueue(scenic::NewCreateImagePipe2Cmd(
      image_pipe_id, std::move(image_pipe_request)));
  scenic::Material image_material(&scenic_session_);
  image_material.SetTexture(image_pipe_id);
  shape_.SetMaterial(image_material);
  scenic_session_.ReleaseResource(image_pipe_id);
  scenic_session_.Present2(
      /*requested_presentation_time=*/0,
      /*requested_prediction_span=*/0,
      [](fuchsia::scenic::scheduling::FuturePresentationTimes info) {});
  return true;
}

void ScenicSurface::SetTextureToImage(const scenic::Image& image) {
  DCHECK_CALLED_ON_VALID_THREAD(thread_checker_);
  scenic::Material image_material(&scenic_session_);
  image_material.SetTexture(image);
  shape_.SetMaterial(image_material);
}

mojo::PlatformHandle ScenicSurface::CreateView() {
  DCHECK_CALLED_ON_VALID_THREAD(thread_checker_);

  // Scenic will associate the View and ViewHolder regardless of which it
  // learns about first, so we don't need to synchronize View creation with
  // attachment into the scene graph by the caller.
  auto tokens = scenic::ViewTokenPair::New();
  parent_ = std::make_unique<scenic::View>(
      &scenic_session_, std::move(tokens.view_token), "chromium surface");
  parent_->AddChild(shape_);

  scenic_session_.Present2(
      /*requested_presentation_time=*/0,
      /*requested_prediction_span=*/0,
      [](fuchsia::scenic::scheduling::FuturePresentationTimes info) {});
  return mojo::PlatformHandle(std::move(tokens.view_holder_token.value));
}

}  // namespace ui
