// Copyright 2019 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "ui/native_theme/native_theme_win.h"

#include "testing/gtest/include/gtest/gtest.h"
#include "ui/gfx/color_palette.h"

namespace ui {

using PrefScheme = NativeTheme::PreferredColorScheme;
using SystemThemeColor = NativeTheme::SystemThemeColor;

class TestNativeThemeWin : public NativeThemeWin {
 public:
  TestNativeThemeWin() : NativeThemeWin(false, false) {}
  ~TestNativeThemeWin() override = default;

  // NativeTheme:
  void SetSystemColor(SystemThemeColor system_color, SkColor color) {
    system_colors_[system_color] = color;
  }

  DISALLOW_COPY_AND_ASSIGN(TestNativeThemeWin);
};

TEST(NativeThemeWinTest, CalculatePreferredColorScheme) {
  TestNativeThemeWin theme;

  theme.set_high_contrast(false);
  theme.set_use_dark_colors(true);
  EXPECT_EQ(theme.CalculatePreferredColorScheme(), PrefScheme::kDark);

  theme.set_use_dark_colors(false);
  EXPECT_EQ(theme.CalculatePreferredColorScheme(), PrefScheme::kLight);

  theme.set_high_contrast(true);
  theme.SetSystemColor(SystemThemeColor::kWindow, SK_ColorBLACK);
  EXPECT_EQ(theme.CalculatePreferredColorScheme(), PrefScheme::kDark);

  theme.SetSystemColor(SystemThemeColor::kWindow, SK_ColorWHITE);
  EXPECT_EQ(theme.CalculatePreferredColorScheme(), PrefScheme::kLight);

  theme.SetSystemColor(SystemThemeColor::kWindow, SK_ColorBLUE);
  EXPECT_EQ(theme.CalculatePreferredColorScheme(), PrefScheme::kDark);

  theme.SetSystemColor(SystemThemeColor::kWindow, SK_ColorYELLOW);
  EXPECT_EQ(theme.CalculatePreferredColorScheme(), PrefScheme::kLight);

  theme.set_high_contrast(false);
  EXPECT_EQ(theme.CalculatePreferredColorScheme(), PrefScheme::kLight);
}

TEST(NativeThemeWinTest, GetDefaultSystemColorScheme) {
  using ColorScheme = NativeTheme::ColorScheme;

  TestNativeThemeWin theme;
  theme.set_high_contrast(false);
  theme.set_use_dark_colors(true);
  EXPECT_EQ(theme.GetDefaultSystemColorScheme(), ColorScheme::kDark);

  theme.set_use_dark_colors(false);
  EXPECT_EQ(theme.GetDefaultSystemColorScheme(), ColorScheme::kLight);

  theme.set_high_contrast(true);
  theme.SetSystemColor(SystemThemeColor::kWindow, SK_ColorBLACK);
  theme.SetSystemColor(SystemThemeColor::kWindowText, SK_ColorWHITE);
  EXPECT_EQ(theme.GetDefaultSystemColorScheme(),
            ColorScheme::kPlatformHighContrast);

  theme.SetSystemColor(SystemThemeColor::kWindow, SK_ColorWHITE);
  theme.SetSystemColor(SystemThemeColor::kWindowText, SK_ColorBLACK);
  EXPECT_EQ(theme.GetDefaultSystemColorScheme(),
            ColorScheme::kPlatformHighContrast);

  theme.SetSystemColor(SystemThemeColor::kWindowText, SK_ColorBLUE);
  EXPECT_EQ(theme.GetDefaultSystemColorScheme(),
            ColorScheme::kPlatformHighContrast);

  theme.set_high_contrast(false);
  EXPECT_EQ(theme.GetDefaultSystemColorScheme(), ColorScheme::kLight);
}

TEST(NativeThemeWinTest, GetPlatformHighContrastColor) {
  using ColorId = NativeTheme::ColorId;

  // These specific colors don't matter, but should be unique.
  constexpr SkColor kWindowTextColor = SK_ColorGREEN;
  constexpr SkColor kHighlightColor = SK_ColorYELLOW;
  constexpr SkColor kHighlightTextColor = SK_ColorBLUE;

  TestNativeThemeWin theme;
  theme.SetSystemColor(SystemThemeColor::kWindowText, kWindowTextColor);
  theme.SetSystemColor(SystemThemeColor::kHighlight, kHighlightColor);
  theme.SetSystemColor(SystemThemeColor::kHighlightText, kHighlightTextColor);

  // Test that we get regular colors when HC is off.
  theme.set_high_contrast(false);
  EXPECT_NE(theme.GetSystemColor(ColorId::kColorId_LabelEnabledColor),
            kWindowTextColor);
  EXPECT_NE(theme.GetSystemColor(ColorId::kColorId_ProminentButtonColor),
            kHighlightColor);
  EXPECT_NE(theme.GetSystemColor(ColorId::kColorId_TextOnProminentButtonColor),
            kHighlightTextColor);

  // Test that we get HC colors when HC is on.
  theme.set_high_contrast(true);
  EXPECT_EQ(theme.GetSystemColor(ColorId::kColorId_LabelEnabledColor),
            kWindowTextColor);
  EXPECT_EQ(theme.GetSystemColor(ColorId::kColorId_ProminentButtonColor),
            kHighlightColor);
  EXPECT_EQ(theme.GetSystemColor(ColorId::kColorId_TextOnProminentButtonColor),
            kHighlightTextColor);
}

TEST(NativeThemeWinTest, GetPlatformHighContrastColorScheme) {
  using HCColorScheme = NativeTheme::PlatformHighContrastColorScheme;

  TestNativeThemeWin theme;
  theme.set_high_contrast(false);
  theme.set_preferred_color_scheme(PrefScheme::kDark);
  EXPECT_EQ(theme.GetPlatformHighContrastColorScheme(), HCColorScheme::kNone);

  theme.set_preferred_color_scheme(PrefScheme::kLight);
  EXPECT_EQ(theme.GetPlatformHighContrastColorScheme(), HCColorScheme::kNone);

  theme.set_high_contrast(true);
  theme.set_preferred_color_scheme(PrefScheme::kDark);
  EXPECT_EQ(theme.GetPlatformHighContrastColorScheme(), HCColorScheme::kDark);

  theme.set_preferred_color_scheme(PrefScheme::kLight);
  EXPECT_EQ(theme.GetPlatformHighContrastColorScheme(), HCColorScheme::kLight);

  theme.set_high_contrast(false);
  EXPECT_EQ(theme.GetPlatformHighContrastColorScheme(), HCColorScheme::kNone);
}

}  // namespace ui
