      SUBROUTINE PAR_GDRVI( PARAM, MAXVAL, VMIN, VMAX, VALUES,
     :                        ACTVAL, STATUS )

*+
*  Name:
*     PAR_GDRVx

*  Purpose:
*     Obtains a vector of values within a given range from a parameter.

*  Language:
*     Starlink Fortran 77

*  Invocation:
*     CALL PAR_GDRVx( PARAM, MAXVAL, VMIN, VMAX, VALUES, ACTVAL,
*                     STATUS )

*  Description:
*     This routine obtains up to a given number of values from a
*     parameter.  All the values must be within a supplied range of
*     acceptable values.

*  Arguments:
*     PARAM = CHARACTER * ( * ) (Given)
*        The name of the parameter.
*     MAXVAL = INTEGER (Given)
*        The maximum number of values required.  A PAR__ERROR status is
*        returned when the number of values requested is less than one.
*     VMIN = ? (Given)
*        The value immediately above a range wherein the obtained
*        values cannot lie.  Thus if VMAX is greater than VMIN, VMIN
*        is the minimum allowed for the obtained values.  However,
*        should VMAX be less than VMIN, all values are acceptable
*        except those between VMAX and VMIN exclusive.
*     VMAX = ? (Given)
*        The value immediately below a range wherein the obtained
*        values cannot lie.  Thus if VMAX is greater than VMIN, VMAX
*        is the maximum allowed for the obtained values.  However,
*        should VMAX be less than VMIN, all values are acceptable
*        except those between VMAX and VMIN exclusive.
*     VALUES( MAXVAL ) = ? (Returned)
*        The values associated with the parameter.  They will only be
*        valid if STATUS is not set to an error value.
*     ACTVAL = INTEGER (Returned)
*        The actual number of values obtained.
*     STATUS = INTEGER (Given and Returned)
*        The global status.

*  Notes:
*     -  There is a routine for each of the data types double precision,
*     integer, and real: replace "x" in the routine name by D, I, or R
*     respectively as appropriate.  The VMIN, VMAX, and VALUES arguments
*     all must have the corresponding data type.
*     -  Should too many values be read, the parameter system will
*     repeat the get in order to obtain a permitted number of values.
*     -  If any of the values violates the constraint, the user is
*     informed of the constraint and prompted for another vector of
*     values.  This is not achieved through the MIN/MAX system.

*  Algorithm:
*     -  If the number of values is not positive then report the error
*     and exit.
*     -  Find whether an inclusion or exclusion constraint is requested.
*     -  Loop until acceptable values are obtained or an error occurs.
*     Get up to the maximum number of values from the parameter system.
*     Find their maximum and minimum.  Test that the extremes lie
*     within the acceptable limits.  If they do not, report the fact
*     and the limits.

*  Authors:
*     MJC: Malcolm J. Currie  (STARLINK)
*     {enter_new_authors_here}

*  History:
*     1991 January 2 (MJC):
*        Original based upon PAR_GDR0.
*     1992 November 18 (MJC):
*        Permitted the limits to be reversed in order to specify an
*        exclusion range.
*     1997 June 23 (MJC):
*        Moved a status check to before checking the range of supplied
*        values.  The old code caused problems on Linux.
*     {enter_further_changes_here}

*  Bugs:
*     {note_any_bugs_here}

*-

*  Type Definitions:
      IMPLICIT NONE            ! Switch off default typing

*  Global Constants:
      INCLUDE 'SAE_PAR'        ! Environment constants
      INCLUDE 'PAR_ERR'        ! Parameter-system error constants

*  Arguments Given:
      CHARACTER * ( * )
     :  PARAM                  ! Parameter name associated with value
                               ! to be obtained

      INTEGER
     :  MAXVAL                 ! Maximum number of values to obtain

      INTEGER
     :  VMIN,                  ! Minimum acceptable value for values to
                               ! be obtained
     :  VMAX                   ! Maximum acceptable value for values to
                               ! be obtained

*  Arguments Returned:
      INTEGER
     :  VALUES( MAXVAL )       ! Values---only valid if STATUS does not
                               ! have an error

      INTEGER
     :  ACTVAL                 ! Number of values obtained

*  Status:
      INTEGER STATUS           ! Global status

*  Local Variables:
      LOGICAL                  ! True if:
     :  EXCLUD,                ! The range is an exclusion zone
     :  NOTOK                  ! Unacceptable values obtained

      INTEGER
     :  I                      ! Loop counter

      INTEGER
     :  VRANGE( 2 )            ! Minimum and maximum of values read

*.

*  Check the inherited status.
      IF ( STATUS .NE. SAI__OK ) RETURN

*  Ensure that the number of values needed is positive.
      IF ( MAXVAL .LT. 1 ) THEN

*  Too few values requested.
         STATUS = PAR__ERROR
         CALL MSG_SETC( 'PARAM', PARAM )
         CALL ERR_REP( 'PAR_GDRVx_TOOFEW',
     :     'A non-positive number of values was requested for '/
     :     /'parameter ^PARAM. (Probable programming error.)', STATUS )

*  Exit the routine.
         GOTO 999

      END IF

*  Find whether the range is inclusive or exclusive depending on the
*  polarity of the two values.
      EXCLUD = VMIN .GT. VMAX

*  Start a new error context.
      CALL ERR_MARK

*  Loop to obtain the values of the parameter.
*  ===========================================

*  Initialise NOTOK to start off the loop.
      NOTOK = .TRUE.

  100 CONTINUE

*  The loop will keep going as long as suitable values have not be read
*  and there is no error.
         IF ( .NOT. NOTOK .OR. ( STATUS .NE. SAI__OK ) ) GOTO 140

*  Get up to MAXVAL values from the parameter system.
         CALL PAR_GETVI( PARAM, MAXVAL, VALUES, ACTVAL, STATUS )

*  Check for an error.
         IF ( STATUS .EQ. SAI__OK ) THEN

*  Obtain the range of the values given.
*  =====================================

*  Initialise the minimum and maximum values.
            VRANGE( 1 ) = VALUES( 1 )
            VRANGE( 2 ) = VALUES( 1 )

*  Search all the remaining values obtained for their minimum and
*  maximum.
            IF ( ACTVAL .GT. 1 ) THEN
               DO 120 I = 2, ACTVAL
                  VRANGE( 1 ) = MIN( VRANGE( 1 ), VALUES( I ) )
                  VRANGE( 2 ) = MAX( VRANGE( 2 ), VALUES( I ) )
  120          CONTINUE
            END IF

*  Validate the values.
*  ====================

*  Check that the valus are within the specified include or exclude
*  range.
            IF ( EXCLUD ) THEN
               NOTOK = ( VRANGE( 2 ) .LT. VMIN ) .AND.
     :                 ( VRANGE( 1 ) .GT. VMAX )
            ELSE
               NOTOK = ( VRANGE( 1 ) .LT. VMIN ) .OR.
     :                 ( VRANGE( 2 ) .GT. VMAX )
            END IF

*  The value is not within the constraints, so report as an error,
*  including full information using tokens. 
            IF ( NOTOK ) THEN
               STATUS = PAR__ERROR
               CALL MSG_SETC( 'PARAM', PARAM )
               CALL MSG_SETI( 'MIN', VMIN )
               CALL MSG_SETI( 'MAX', VMAX )
               IF ( EXCLUD ) THEN
                  CALL MSG_SETC( 'XCLD', 'outside' )
               ELSE
                  CALL MSG_SETC( 'XCLD', 'in' )
               END IF

               CALL ERR_REP( 'PAR_GDRVx_OUTR',
     :           'At least one of the values is outside the allowed '/
     :           /'range for parameter ^PARAM.  Give values ^XCLD the '/
     :           /'range ^MIN to ^MAX please.', STATUS )

*  Note that the error is flushed immediately as we are in a loop.
               CALL ERR_FLUSH( STATUS )

*  Cancel the parameter to enable a retry to get all values within the
*  range.
               CALL PAR_CANCL( PARAM, STATUS )

            ELSE

*  The values must be satisfactory so terminate the loop.
               NOTOK = .FALSE.
            END IF

         END IF

*  Go to the head of the main loop.
         GOTO 100

*  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^

*  Come here when the main loop has been exited.
  140 CONTINUE

*  Release the new error context.
      CALL ERR_RLSE

  999 CONTINUE

      END
