/*
 * Copyright (c) 2017, 2019, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

/*
 * The certificates and corresponding private keys used by the test.
 * All of certificates uses relative weak key size and hash algorithm, then
 * all JDK releases can load them. Accordingly, a custom java.security file is
 * provided to make sure such weak key sizes and algorithms are not blocked by
 * any JDK build.
 */
public enum Cert {

    // This certificate is generated by the below command:
    // openssl genpkey -algorithm rsa -pkeyopt rsa_keygen_bits:2048 \
    //     -pkeyopt rsa_keygen_pubexp:65537 -out key.pem
    // openssl req -x509 -new -days 7300 -key key.pem \
    //     -subj "/CN=RSA-2048-SHA256" -sha256 -out cer.pem
    RSA_2048_SHA256(
            KeyAlgorithm.RSA,
            "-----BEGIN CERTIFICATE-----\n" +
            "MIIDFTCCAf2gAwIBAgIUe8nlNUPJa9Iy57Cy5JM49bCzWdkwDQYJKoZIhvcNAQEL\n" +
            "BQAwGjEYMBYGA1UEAwwPUlNBLTIwNDgtU0hBMjU2MB4XDTE5MDIyNzA3NDkwMVoX\n" +
            "DTM5MDIyMjA3NDkwMVowGjEYMBYGA1UEAwwPUlNBLTIwNDgtU0hBMjU2MIIBIjAN\n" +
            "BgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA1Clo5Prh1AdHSdM7G85B6K20bjSn\n" +
            "bydcWxa7vQDEgFid1Ne8XRbugv5i8I7kGv2sTl99fopHeJcXHJvQGg7KVPgZqH0Z\n" +
            "S7ZImlT5f4FYFj8sKnM5wx2P2AxcbO8ktSox0qIgtsHsCd7SusczylqEvSUrcqEe\n" +
            "V58LtoWH+trsWoSBDlHRew2eD6ZGyQTM8VFqbt9oF2XXW22JiuP+cSvb+p5qSCy5\n" +
            "dGpdPCJpPB/9HpChZl/r+VsqpbHwUPEVu9/FG0SVjpcqvJojYrgglb1PvJxLqceN\n" +
            "DPOirwxnnEdiu5j0xC6RhOkbcxTGtS0VgEEC1+HyY+KeauZJOrw2x1ZmxQIDAQAB\n" +
            "o1MwUTAdBgNVHQ4EFgQUSSj0EFZWTSFr91nTUE2MrJdrJGowHwYDVR0jBBgwFoAU\n" +
            "SSj0EFZWTSFr91nTUE2MrJdrJGowDwYDVR0TAQH/BAUwAwEB/zANBgkqhkiG9w0B\n" +
            "AQsFAAOCAQEAW9uuS2ZpG1ytpNA0g20m29R/DVSnygdfp8r/xeaWgdol4H2syPzg\n" +
            "xok3PLkxkSpBv6CgIPXBzy/iXSMlkX0mUKEO3aQnJ8MoZ5Tzu3Bkp2fTugRQuTRi\n" +
            "iNWQjsMoupsrTXIZhJ64jkDCnlFADPAdvVqQV01yZcKW98sj3TyaT7TJuYX9mU+V\n" +
            "OuICkS1LE5NssuyLodxpfqpjBMtVovSKZ57JvO36G6riftnjr3FBf8ukWFK2/UfP\n" +
            "DaHyFQ+NewbjPy7N+taFlLHS7ELwZVQQ42t8JeHRuF5IVvlp1UjTgXC5NuhOBwQY\n" +
            "2dXFFroT0vXetn7Fr51zENPP3/TGeaoQnw==\n" +
            "-----END CERTIFICATE-----",
            "308204be020100300d06092a864886f70d0101010500048204a8308204a40201" +
            "000282010100d42968e4fae1d4074749d33b1bce41e8adb46e34a76f275c5b16" +
            "bbbd00c480589dd4d7bc5d16ee82fe62f08ee41afdac4e5f7d7e8a477897171c" +
            "9bd01a0eca54f819a87d194bb6489a54f97f8158163f2c2a7339c31d8fd80c5c" +
            "6cef24b52a31d2a220b6c1ec09ded2bac733ca5a84bd252b72a11e579f0bb685" +
            "87fadaec5a84810e51d17b0d9e0fa646c904ccf1516a6edf681765d75b6d898a" +
            "e3fe712bdbfa9e6a482cb9746a5d3c22693c1ffd1e90a1665febf95b2aa5b1f0" +
            "50f115bbdfc51b44958e972abc9a2362b82095bd4fbc9c4ba9c78d0cf3a2af0c" +
            "679c4762bb98f4c42e9184e91b7314c6b52d15804102d7e1f263e29e6ae6493a" +
            "bc36c75666c502030100010282010028f1f4f47c16a93cde5d390ee746df2170" +
            "a4a9c02fb01c008ef3cc37a5b646aed387083baa1b8adc6d0bdb3138849d006b" +
            "ffb1d0820f590e8fbf4db2d3d496e7df19d4929017348ebe7a37cc8bc1dc4944" +
            "d4cc781157db32eeefc7763fb756f55699438701d5f3f1b4e9a7182fad5880c8" +
            "73a223c61f52ea87c72d7f14511906af61d7fa190b02854471d4bdb77dac34ef" +
            "46a3af3f39dff1c8844cad7f74f9936fbcc22bed6b139f47dc215048ddf02f60" +
            "a24703b292be106ea4f01ec0839466666d9c3dc8488b353dccdd5f90bd4b5bb9" +
            "4493b7da219ec4962fe6a427f6d69e2764065212c5accdbed3aa36a18d540e55" +
            "192e63db9f6bdfc90ec52b89714d0102818100f7c35a70ee7d6aabd7feb590f6" +
            "30ce9c28299f3431ebcd3c89ec9424cf68c626ee4f6ff0748ffc0ad810a9f6dd" +
            "2b203c8fa7f516483545822e6c963b9f6a1687aca663be061aadcca920b09699" +
            "bd7d2e8973bafe9ef11e19a27c10befe3e8919c141d04e5aab2990cc061c6798" +
            "5d3da742a3c8c62b68a8ccb4af21c1c935bdcd02818100db37101251d805b8d6" +
            "12749d8780cce9e4ff8fc58313e4192fbf9018dc2a8875beff70ea2ebaa24058" +
            "d3de6eab4be66b0bb81c04f0fa29bad0f787776ed2e6ab7b7d9814ce7721cadd" +
            "cc3f500ddfd73ae9def4d92a79e4a1662da16dbfc52d60507308286cf592ed8b" +
            "9911089c4ec7dba3fcd64926b55d137d41f6de454694d902818100af6b582077" +
            "2ac318d2401bcb7c13886555a64a7b09115be98df9bbd5e827d58c00d4ab7bc2" +
            "fba2c706bd9da9146491596f98ca553160ce4ae295ad349fa4dc38c94bb178fc" +
            "176d9066faa72ca9c358db572462741e92b6ee0d75ebe15e5f66709ebcfb404e" +
            "bfbb1932eaecb7885013f3d5a1e2e83419d0d1c6e7ec6da9096ccd0281810099" +
            "81fc922797f3a1d4dec5a4ce8fc66effba6aae7034cca54a8785dbb2c96217ba" +
            "315c9bd12f469172e2a2bfb2da8ab769547ae286f157a987cddea2270c2f15e4" +
            "7b35b554439e79564a4207c83f7893bbd43277a4c408f370ff012d3e7e506142" +
            "d4dae09c3477b83aea6c40305d069d6b3f91bb560ce8e9cdec1478dfe0263902" +
            "818002b66c71380142c3e606bfc598b4060f6833ac80e16d08aea40f4837191d" +
            "34a3c85b91c3043c6ebb2c651a7fbb89539f5621820f792a5279a947c51f47b7" +
            "1b6051c5a81d2d1c30dfa1f93cb57af1d7ee7862e8d90e33bd5c80f14aa9471b" +
            "a2ea7aacddbb44d1a5e60f5fac437ca50cd56e237936fd3e9d034efc3e3c6710" +
            "4c08"),

    // This certificate is generated by the below command:
    // openssl genpkey -algorithm rsa -pkeyopt rsa_keygen_bits:2048 \
    //     -pkeyopt rsa_keygen_pubexp:65537 -out key.pem
    // openssl req -x509 -new -days 7300 -key key.pem \
    //     -subj "/CN=EXAMPLE" -sha256 -out cer.pem
    EXAMPLE_RSA_2048_SHA256(
            KeyAlgorithm.RSA,
            "-----BEGIN CERTIFICATE-----\n" +
            "MIIDBTCCAe2gAwIBAgIUD+8I14TmOfEfxtD6hgnhhK8ARCAwDQYJKoZIhvcNAQEL\n" +
            "BQAwEjEQMA4GA1UEAwwHRVhBTVBMRTAeFw0xOTAyMjcwODAzNDhaFw0zOTAyMjIw\n" +
            "ODAzNDhaMBIxEDAOBgNVBAMMB0VYQU1QTEUwggEiMA0GCSqGSIb3DQEBAQUAA4IB\n" +
            "DwAwggEKAoIBAQChKYq85df7kUnf35qAfxW/OnqCgn/5FNwlWAwHLlEiHpK+k7jD\n" +
            "8S6LVbw55I/4J3lehIHcIapGdmqh9ijUc2aNxTJ33z+/TTu2n+KlWmGj0G7ovTXk\n" +
            "TbWptdgk5ro8DCr8I8YcvwdLekwH4AkRL6jSyiqsqlGZYLIxDd4l0CwSt5orbu/y\n" +
            "+2UtM4DEOEswrxdP9UAd+W0On4AWaFIEbfuFaLZXHadvKxidnaCmudOJry6NjFWn\n" +
            "+3PmIWNhZJitD0gq8FG3pvY502fLqHX95pigWCkDtrDNiqReXgVvZFWPaSMs065y\n" +
            "n2ClShbzTs8pqJp8oBde9Iwi3RKwkew8I2iJAgMBAAGjUzBRMB0GA1UdDgQWBBTL\n" +
            "3w5XucuEre5nQiaKnqi4s7ldBjAfBgNVHSMEGDAWgBTL3w5XucuEre5nQiaKnqi4\n" +
            "s7ldBjAPBgNVHRMBAf8EBTADAQH/MA0GCSqGSIb3DQEBCwUAA4IBAQBo51E5U5H7\n" +
            "kXkI1LaGDs35dOOtrPQQQ7DzhQACUDfyYB8/BBUlYHPegS2ca/l1mkD2P/GNyZbN\n" +
            "r4eRoOujfU59gOwH4+MEs/4zUKicajTGUPCbJ56heipHunHzj+2nj3ok5PI0MoI1\n" +
            "soQfRV1FshfWAni7E49L1QI/PQrQ2cPR/1dvkB06JCIn0GoTxd8E76VCJz576xnd\n" +
            "XgjiadVbjMYDH9XJEy6eQT6vY5WeGOQC2XwSE3mS6G+Z9jlwIswa8RrKGtKqFGK1\n" +
            "6yse9zfJv64X8tQwnrkatCx4shJwDUet6wJQzm0/NMPfquoxz7QHF2NsLlNky+fY\n" +
            "fZjMnoO3J1nV\n" +
            "-----END CERTIFICATE-----",
            "308204bd020100300d06092a864886f70d0101010500048204a7308204a30201" +
            "000282010100a1298abce5d7fb9149dfdf9a807f15bf3a7a82827ff914dc2558" +
            "0c072e51221e92be93b8c3f12e8b55bc39e48ff827795e8481dc21aa46766aa1" +
            "f628d473668dc53277df3fbf4d3bb69fe2a55a61a3d06ee8bd35e44db5a9b5d8" +
            "24e6ba3c0c2afc23c61cbf074b7a4c07e009112fa8d2ca2aacaa519960b2310d" +
            "de25d02c12b79a2b6eeff2fb652d3380c4384b30af174ff5401df96d0e9f8016" +
            "6852046dfb8568b6571da76f2b189d9da0a6b9d389af2e8d8c55a7fb73e62163" +
            "616498ad0f482af051b7a6f639d367cba875fde698a0582903b6b0cd8aa45e5e" +
            "056f64558f69232cd3ae729f60a54a16f34ecf29a89a7ca0175ef48c22dd12b0" +
            "91ec3c236889020301000102820100655c9e60ce62b85b99ce0f71ba2db3fcd1" +
            "07ea7baf8776823b8e940a142c7d3c23696fb97eab7b6db11fb07dbbbb0500c5" +
            "dcab5c4b642feb1c87ff2d90e97fefdcbe303c9e7870580535ac33f9937d9783" +
            "9a281ef41798114448cc74bd5f34fbf8177bebea8de8ffe33ff4bd5f2ccd8ebe" +
            "0e7708ac47be54749bd7438b199d2f134b71efc513827f260c0f74f1fc32f45b" +
            "e5d510844777fcd2a486bc02c080d120d1c32336000ece743ea755f79f60a44a" +
            "5e619ceb1caa873d847715616874d13c2ff1fe9f9f81d8fc83e83fb035bce8d9" +
            "ed8f5caa41626d323551311b1d8d8f06785e3700d45e4d771157b22826efe553" +
            "7a5892ad3bf3f915ec25342a8c7a3d02818100d19c03d857442bbaedb41b741e" +
            "8e93d295940fdfc455898463ad96b0089ee68d90b787848b7aed6bb735c7e4b9" +
            "7b22e867000d8e4b4ede4b155c34fd88c10244917912c048d023757bd758a117" +
            "764aa80434c5c9636ec125574667ffe01af856f4517d06b6831ad50f16b26bba" +
            "67a7125e158988c98b817dbb0928efa00c3ed702818100c4d49f7f3bf36586aa" +
            "519bf2841c459c1863e71c08a9111344e51fcf5ff4267420fd9ffc9f72288e44" +
            "b56bdae4eaa669e5e350afe4b4402be4af54d5dbc8b5dc5f5b6bb79df4fd17a5" +
            "225287947783b5327b5dedf02733fb534514cc05fde1dcfceb8b537ad3c163a8" +
            "8f36a60e2fb17fa6d9a0f3fca444f349eed9f07823879f02818100a5e9eb753c" +
            "261ec338d23e84dc8718e53036e195cacfb6294fc920a4b83e26da59799c5043" +
            "238b789ead784b48b1fa40a0fefebbea4a44548454d730f4256a8921e906f9a2" +
            "e8f59851ed741f16f63043ec0865a2720d41df2fc4f01f2ea1ca7ef1a6eae2fc" +
            "66ac3f8750fceb9ec1db1203dce25f9ec0c93fdf6371beb31dde430281807852" +
            "be59ea4d25504847f13c34948fdd176fe2f4d93a790cbd7e0f8f16ca4ac38cf3" +
            "5e5cf11fb93917398c805896353ae164af8b8714c571cfaf7afded086a5c1812" +
            "ebeb686d3e56b9051d4c726f091db8897fe7177aefa500c7672a3db370e245de" +
            "bbe24160b784f3a2f0b65c4fbd831a7d498e3d70321243acf69fb0e18f630281" +
            "8065f0a2f257f8bf1d57e3f1b72c9a664ca92630985ee5ba35438e57a1df67a6" +
            "f6b380907f5b7f9bdd2ddc63385615c5ca3c0dcbedfdc3f18433160855824712" +
            "eaaeb318774478427dfb58135715cf82730a743dd8448984450905c28a6a97a0" +
            "5f4aaad616978c07c5957c4f1945073f333df4337557bd6b754953f71df7a03c" +
            "ec"),

    // This certificate is generated by the below commands:
    // openssl genpkey -genparam -algorithm dsa -pkeyopt dsa_paramgen_bits:2048 \
    //     -pkeyopt dsa_paramgen_q_bits:256 -out param.pem
    // openssl genpkey -paramfile param.pem -out key.pem
    // openssl req -x509 -new -key key.pem -days 7300 \
    //     -subj "/CN=DSA-2048-SHA256" -sha256 -out cer.pem
    DSA_2048_SHA256(
            KeyAlgorithm.DSA,
            "-----BEGIN CERTIFICATE-----\n" +
            "MIIEezCCBCCgAwIBAgIUYbBHFrJkO9EokOROEScK0dr3aVowCwYJYIZIAWUDBAMC\n" +
            "MBoxGDAWBgNVBAMMD0RTQS0yMDQ4LVNIQTI1NjAeFw0xOTAxMTYwNjA4NTJaFw0z\n" +
            "OTAxMTEwNjA4NTJaMBoxGDAWBgNVBAMMD0RTQS0yMDQ4LVNIQTI1NjCCA0cwggI5\n" +
            "BgcqhkjOOAQBMIICLAKCAQEAnGTeaC+MitQykXl2UcGtJhkelRhBalZuzR9fh8Xf\n" +
            "swSj2auVzhjkQfns6E14X72XLYDwHPQilh4ZSiC1cX7bpvohJ7CjbouGnT78Gwf0\n" +
            "pfVMRBRfbNiwBQzFtzehAqlLPXyHWCPxAzW0EgzZh8asIofBEGmGh6sxlpXbEcCS\n" +
            "Ra5Q2vdDRVksm4APHKwbS1F6L/0QDvjDAudWyLFdFMSBuoyeFwCf7Gr5lcVi538h\n" +
            "MliXSAhJ7SXe4B9K/5/VicuzAffE+EZsZZuJKeQ4oHz8xtycdMcW86Iag1i/POzD\n" +
            "0Z7c5p9j1zDgms1ZRSz4fd7YgNGjmmf6dYJlZmCX4w9YiwIhALnVATHxZmk7G1kf\n" +
            "LGX7SMeflAhA/D3EPA5QWdgTc0orAoIBAGYTWMnKHBF4oMKUsx6lF6KvSrqFQ0+j\n" +
            "mWN1RNAGiVYm/Js9sc7jolZCbVTWM7GblO2DxiKed3FtcL3xw4P6l3GU7kFthsjh\n" +
            "bHbMG58s5JVboLX50wZo1uBOb0kRcZYjYTfUwbBYpGNnl7xfQsZ/Bxq/wzyn4gxb\n" +
            "+C0pu/vzmko+opKfFr9a2EL+ADvQhPd6y/L0YcpiTihvfXDWu+n3bNlwhUZYevux\n" +
            "SPVkQH3h5YEqXePF7UeY506/2sr41/xCbCkuH+Ob77Cy1IjMqr4OpXzj6wCSjlFy\n" +
            "Re66yqsjGpuBeTtsn9lKmlFVl4QUdw7XYbRubafNFEdd5IazMflCcgMDggEGAAKC\n" +
            "AQEAnF3/TT0nYzfwY6/+lmoqoBtGU4of7RK4U3Jdhf6dj755GwgsPGAz3Rzh/hq/\n" +
            "B3fuyrtnE/aU0EK1dtRNqnKeFsMCURBi3sWYYe7vamw5TukFMSUOuH76v75b26c+\n" +
            "nTfZF30iDz+66Z5gcOSwhz9aLCnF/wYwijnzagnCF4m0Nhwsjtz4qECMPfL/Rkx8\n" +
            "s83tUF53mx5o8WCYMNr4yJGcHEkautvZN05zwoxZTPKZNngWeCUAY07XwOYHTUvx\n" +
            "C9Sxiom9k1pZPbHZnGpUx96at2dMoeTeHR6xF/0QpiIb4wLpOg/CRxsopBmhEl0S\n" +
            "BEXYNIcrXLwVBLvJD6H44DxRiqNTMFEwHQYDVR0OBBYEFCr6C8Bl7wjz5L3cYMG3\n" +
            "/ZFe7Ee0MB8GA1UdIwQYMBaAFCr6C8Bl7wjz5L3cYMG3/ZFe7Ee0MA8GA1UdEwEB\n" +
            "/wQFMAMBAf8wCwYJYIZIAWUDBAMCA0gAMEUCIArByiqjCG1ZuOVY91OPa9g8q60y\n" +
            "/BSRWRgikEuq3AbgAiEAoZoKXW80FTMxv/9tFy0N7OrUV4rc9+AUBSTuRCuTAk4=\n" +
            "-----END CERTIFICATE-----",
            "308202640201003082023906072a8648ce3804013082022c02820101009c64de" +
            "682f8c8ad43291797651c1ad26191e9518416a566ecd1f5f87c5dfb304a3d9ab" +
            "95ce18e441f9ece84d785fbd972d80f01cf422961e194a20b5717edba6fa2127" +
            "b0a36e8b869d3efc1b07f4a5f54c44145f6cd8b0050cc5b737a102a94b3d7c87" +
            "5823f10335b4120cd987c6ac2287c110698687ab319695db11c09245ae50daf7" +
            "4345592c9b800f1cac1b4b517a2ffd100ef8c302e756c8b15d14c481ba8c9e17" +
            "009fec6af995c562e77f21325897480849ed25dee01f4aff9fd589cbb301f7c4" +
            "f8466c659b8929e438a07cfcc6dc9c74c716f3a21a8358bf3cecc3d19edce69f" +
            "63d730e09acd59452cf87dded880d1a39a67fa758265666097e30f588b022100" +
            "b9d50131f166693b1b591f2c65fb48c79f940840fc3dc43c0e5059d813734a2b" +
            "02820100661358c9ca1c1178a0c294b31ea517a2af4aba85434fa399637544d0" +
            "06895626fc9b3db1cee3a256426d54d633b19b94ed83c6229e77716d70bdf1c3" +
            "83fa977194ee416d86c8e16c76cc1b9f2ce4955ba0b5f9d30668d6e04e6f4911" +
            "7196236137d4c1b058a4636797bc5f42c67f071abfc33ca7e20c5bf82d29bbfb" +
            "f39a4a3ea2929f16bf5ad842fe003bd084f77acbf2f461ca624e286f7d70d6bb" +
            "e9f76cd9708546587afbb148f564407de1e5812a5de3c5ed4798e74ebfdacaf8" +
            "d7fc426c292e1fe39befb0b2d488ccaabe0ea57ce3eb00928e517245eebacaab" +
            "231a9b81793b6c9fd94a9a5155978414770ed761b46e6da7cd14475de486b331" +
            "f942720304220220509dd213cc9769e93825063a4a60500f67c4b979f6504b2f" +
            "ccdbefb3ab8fe6da"),

    // This certificate is generated by the below commands:
    // openssl genpkey -genparam -algorithm dsa -pkeyopt dsa_paramgen_bits:2048 \
    //     -pkeyopt dsa_paramgen_q_bits:256 -out param.pem
    // openssl genpkey -paramfile param.pem -out key.pem
    // openssl req -x509 -new -key key.pem -days 7300 \
    //     -subj "/CN=EXAMPLE" -sha256 -out cer.pem
    EXAMPLE_DSA_2048_SHA256(
            KeyAlgorithm.DSA,
            "-----BEGIN CERTIFICATE-----\n" +
            "MIIEaTCCBA+gAwIBAgIUckU2myqkWak+Svv0FFJX91vv1jMwCwYJYIZIAWUDBAMC\n" +
            "MBIxEDAOBgNVBAMMB0VYQU1QTEUwHhcNMTkwMTE2MDYwODUzWhcNMzkwMTExMDYw\n" +
            "ODUzWjASMRAwDgYDVQQDDAdFWEFNUExFMIIDRjCCAjkGByqGSM44BAEwggIsAoIB\n" +
            "AQDGmXUxMKzLIVUQUy4lu4fbnv8fRXtWIbv1jxWATCNFiFt4cUxrr7GVC16MGrgm\n" +
            "FtMIHXWwFlq2wTL0RhVp2WPIPQzxCwQFivQjpdewVCB1BoJ1ozlvQjU4hri5Ymdt\n" +
            "ebe90uT8NsvrQrSKYCl+/pPNiZKT8oX1aKzRYPQLn0bVXUicWlACoLfHGM4irjEC\n" +
            "4JezC/tdLleeNYNpy2/dKYu/atyN/u0d+dPRLWBCw6/qCRn1yRAv04GC3WYBlSXz\n" +
            "f9OKlCH5kfm9sLyatz/RWDqOb/YWW2Rk7UTKAnoTAyB+I9yUXg6Gad2csNkxXv55\n" +
            "9oJAhdhsOS5cdBoqlZEahIFBAiEAjiMCVBGpnAxjb2dXM/Eec7EfThflQXbl33Zn\n" +
            "Uq3AAsUCggEAaBNP9Uttfs1eV/38aurLd3T1OiM0CF7DPxE0qpSM9dQz9cKZajIE\n" +
            "lsVTGOLBC5/+VSat9t1VG+JoyvSspkvk97/mxx0WCz/QAYTdwCXVuMuSv+EqeOMP\n" +
            "lCxEdbOS8pfD8shOK+pnDSHMJvURYxB+fJkHHeXfwesH3632Vq0GlJ8PgXH5NLHM\n" +
            "MWv7oZjyZMnGWDq2taJcZZG5ETNda2fATNCF9Al430MUxie2Sp50vA1KEtyUqMu+\n" +
            "CLpyOynPHi96TWHNfD23TmKFVN9Uh2nUNIpUk8NMKBwg2O7FvvNnKfbl44ikuCnc\n" +
            "06U7SdF3y8NRdwyayMI3BkOsV8mkoMwUgAOCAQUAAoIBADK2c1Gl3+6zrFU8Uhro\n" +
            "5lFnDy3UYXINpdtWR/8T0FJ6YMax70VCkyxucq3AUnmq9AQtFijjmXDnxzD5g7IS\n" +
            "zOU/28Kg1Mmw26uzrpUZPiixNU7v/xzE37M0Ig3VCEQ9mw57/yv8gwZurHtsokGt\n" +
            "k0lzH/krhYPCOpskg6NulGq5lGsnNVdPkSkiAWZFHTysgKgxvMUxXj0bUm4ZyNw6\n" +
            "Lp2bFHKbmSeTy3OLo/Kvd7BIjhV+Bi5LFh7h8spf6SC7qyEm0X7Keu+61xkFHU8a\n" +
            "aghnGQYwuzpTp+hsbAJdHwVLw9ElqEoaVp3rAmxtsdzqhZSxcMZoypbjEJpiZ1/v\n" +
            "fQCjUzBRMB0GA1UdDgQWBBRTPXqHl7VKr5U94p9c882ge9DOXzAfBgNVHSMEGDAW\n" +
            "gBRTPXqHl7VKr5U94p9c882ge9DOXzAPBgNVHRMBAf8EBTADAQH/MAsGCWCGSAFl\n" +
            "AwQDAgNHADBEAiAOQxtKiMImXKWQvUKZxoUc+chXRxAj3UD3Zj7RksPF2AIgf8HG\n" +
            "hOnr3hzIej3Da2Ty9RTittcgZ14boraO/Vlx9Jo=\n" +
            "-----END CERTIFICATE-----",
            "308202640201003082023906072a8648ce3804013082022c0282010100c69975" +
            "3130accb215510532e25bb87db9eff1f457b5621bbf58f15804c2345885b7871" +
            "4c6bafb1950b5e8c1ab82616d3081d75b0165ab6c132f4461569d963c83d0cf1" +
            "0b04058af423a5d7b0542075068275a3396f42353886b8b962676d79b7bdd2e4" +
            "fc36cbeb42b48a60297efe93cd899293f285f568acd160f40b9f46d55d489c5a" +
            "5002a0b7c718ce22ae3102e097b30bfb5d2e579e358369cb6fdd298bbf6adc8d" +
            "feed1df9d3d12d6042c3afea0919f5c9102fd38182dd66019525f37fd38a9421" +
            "f991f9bdb0bc9ab73fd1583a8e6ff6165b6464ed44ca027a1303207e23dc945e" +
            "0e8669dd9cb0d9315efe79f6824085d86c392e5c741a2a95911a848141022100" +
            "8e23025411a99c0c636f675733f11e73b11f4e17e54176e5df766752adc002c5" +
            "0282010068134ff54b6d7ecd5e57fdfc6aeacb7774f53a2334085ec33f1134aa" +
            "948cf5d433f5c2996a320496c55318e2c10b9ffe5526adf6dd551be268caf4ac" +
            "a64be4f7bfe6c71d160b3fd00184ddc025d5b8cb92bfe12a78e30f942c4475b3" +
            "92f297c3f2c84e2bea670d21cc26f51163107e7c99071de5dfc1eb07dfadf656" +
            "ad06949f0f8171f934b1cc316bfba198f264c9c6583ab6b5a25c6591b911335d" +
            "6b67c04cd085f40978df4314c627b64a9e74bc0d4a12dc94a8cbbe08ba723b29" +
            "cf1e2f7a4d61cd7c3db74e628554df548769d4348a5493c34c281c20d8eec5be" +
            "f36729f6e5e388a4b829dcd3a53b49d177cbc351770c9ac8c2370643ac57c9a4" +
            "a0cc1480042202201ba738489c54dddd5ffbf0009ef9d11851182235a251a410" +
            "4a6354181ecd0348"),

    // This certificate is generated by the below commands:
    // openssl genpkey -algorithm ec -pkeyopt ec_paramgen_curve:P-256 \
    //     -pkeyopt ec_param_enc:named_curve -out key.pem
    // openssl req -x509 -new -days 7300 -key key.pem \
    //     -subj "/CN=ECDSA-SECP256-SHA256" -sha256 -out cer.pem
    ECDSA_PRIME256V1_SHA256(
            KeyAlgorithm.EC,
            "-----BEGIN CERTIFICATE-----\n" +
            "MIIBkzCCATmgAwIBAgIUVW+Rj8muf1DO8yUB9NSEDkD8oYowCgYIKoZIzj0EAwIw\n" +
            "HzEdMBsGA1UEAwwURUNEU0EtU0VDUDI1Ni1TSEEyNTYwHhcNMTkwMjI3MTEwNzA0\n" +
            "WhcNMzkwMjIyMTEwNzA0WjAfMR0wGwYDVQQDDBRFQ0RTQS1TRUNQMjU2LVNIQTI1\n" +
            "NjBZMBMGByqGSM49AgEGCCqGSM49AwEHA0IABJPHqflVA59hR/sBM64OOY2/PTTx\n" +
            "kZZhKcVV8vEkWRWvDV2u2F+lbRQoEoe8bwfGgQgGJIdc+dz9/TVAaYlitaKjUzBR\n" +
            "MB0GA1UdDgQWBBRS9gbMeeA7j7QdipPufKn3jI3hKTAfBgNVHSMEGDAWgBRS9gbM\n" +
            "eeA7j7QdipPufKn3jI3hKTAPBgNVHRMBAf8EBTADAQH/MAoGCCqGSM49BAMCA0gA\n" +
            "MEUCIDH0b3EewcoZUeSo0c2pNSWGCeRlZI49dASDbZ3A0jdTAiEAy/dM9LwYvyLl\n" +
            "yuWq4yTouCdzfQwR9QXg3ohRMhnASlg=\n" +
            "-----END CERTIFICATE-----",
            "308187020100301306072a8648ce3d020106082a8648ce3d030107046d306b02" +
            "01010420ae670b91bae99a9752f2b7e26ab9c0e98636f0b0040d78f2ea4081f8" +
            "e57c72e0a1440342000493c7a9f955039f6147fb0133ae0e398dbf3d34f19196" +
            "6129c555f2f1245915af0d5daed85fa56d14281287bc6f07c681080624875cf9" +
            "dcfdfd3540698962b5a2"),

    // This certificate is generated by the below commands:
    // openssl genpkey -algorithm ec -pkeyopt ec_paramgen_curve:P-256 \
    //     -pkeyopt ec_param_enc:named_curve -out key.pem
    // openssl req -x509 -new -days 7300 -key key.pem \
    //     -subj "/CN=EXAMPLE" -sha256 -out cer.pem
    EXAMPLE_ECDSA_PRIME256V1_SHA256(
            KeyAlgorithm.EC,
            "-----BEGIN CERTIFICATE-----\n" +
            "MIIBeTCCAR+gAwIBAgIUH6kQ0NfopvszxUwZ58KhMicqgCwwCgYIKoZIzj0EAwIw\n" +
            "EjEQMA4GA1UEAwwHRVhBTVBMRTAeFw0xOTAyMjcxMTA5MTJaFw0zOTAyMjIxMTA5\n" +
            "MTJaMBIxEDAOBgNVBAMMB0VYQU1QTEUwWTATBgcqhkjOPQIBBggqhkjOPQMBBwNC\n" +
            "AASbW2bDwNxTHAzN7aW/OD/ywfa0A4bPKF3Qw4U4nLFBHhbbEmDrIkRWqU56UUDt\n" +
            "fnTZnBCJtm4sH8o9D1D9UZVFo1MwUTAdBgNVHQ4EFgQUEEpzWKgPritmUQNEcQhz\n" +
            "bB+5KuUwHwYDVR0jBBgwFoAUEEpzWKgPritmUQNEcQhzbB+5KuUwDwYDVR0TAQH/\n" +
            "BAUwAwEB/zAKBggqhkjOPQQDAgNIADBFAiBjeGB0oc6t2fWOaviIMfqRqta64nl6\n" +
            "Gj8I/JfDH97P1wIhAJ5IC9cxVTiPL/QTxUxRRlTYUboL/+ck1XR9JbZjd/ar\n" +
            "-----END CERTIFICATE-----",
            "308187020100301306072a8648ce3d020106082a8648ce3d030107046d306b02" +
            "010104205dfd6695d259d4047433c0b4520bedcf95130c5c08ba149caddad70d" +
            "b3b66c1ba144034200049b5b66c3c0dc531c0ccdeda5bf383ff2c1f6b40386cf" +
            "285dd0c385389cb1411e16db1260eb224456a94e7a5140ed7e74d99c1089b66e" +
            "2c1fca3d0f50fd519545"),

    // This certificate is generated by the below commands:
    // openssl genpkey -algorithm ec -pkeyopt ec_paramgen_curve:P-256 \
    //     -pkeyopt ec_param_enc:named_curve -out key.pem
    // openssl req -new -key key.pem \
    //     -subj "/CN=EC-RSA-SECP256-SHA256" -sha256 -out csr.pem
    // openssl x509 -req -CAcreateserial -days 7300 -in csr.pem -sha256 \
    //     -CA CA.cer -CAkey CA.key -out cer.pem
    // Actually the CA is RSA_2048_SHA256
    EC_RSA_PRIME256V1_SHA256(
            KeyAlgorithm.EC,
            "-----BEGIN CERTIFICATE-----\n" +
            "MIIB9TCB3gIUeM/cp7ZB4C1Gi/elzfbpdCrNPtAwDQYJKoZIhvcNAQELBQAwGjEY\n" +
            "MBYGA1UEAwwPUlNBLTIwNDgtU0hBMjU2MB4XDTE5MTIxNTEzMDQxMFoXDTM5MTIx\n" +
            "MDEzMDQxMFowIDEeMBwGA1UEAwwVRUMtUlNBLVNFQ1AyNTYtU0hBMjU2MFkwEwYH\n" +
            "KoZIzj0CAQYIKoZIzj0DAQcDQgAE/kcsoE9AIKIjGv0IlkSSQvSwRpoO97UJlUks\n" +
            "F0+MLaw66yOxFvVT7182/oKdJ8HiwZfxmiGEswSumT8yDdMnFzANBgkqhkiG9w0B\n" +
            "AQsFAAOCAQEAuBUW4HsZ0ICYW6KtKLr9LlZfQdhv9Bc/nU836ASd8cV+PQHH3NSM\n" +
            "1sUFWqq7ivVxvlNTcXThOrTleK8QN6388K4NRUFEXzROZk9Qo3q6lLpnfWiCIqOF\n" +
            "DzQerjcUH9L/TgEtAm83tDq0ZgO7G59X8thTE9GjhXJ6bITK7TMh/VtsGzE7cW8r\n" +
            "kXBNwflorhsR7rywKsASz5snYPYoTM0NZL9T093sxtO86B5sMlfX2rR+/BDQPDsX\n" +
            "bCnXQNvn3tPuFuUZV1IAmrxG+xLxEaH0vi8eU7N2xMvPQKHGrprk4TvxM/d1Z7xY\n" +
            "lWJXWvlhmPI3hb2ngLHEFa/gElpcdWi8Tw==\n" +
            "-----END CERTIFICATE-----",
            "308187020100301306072a8648ce3d020106082a8648ce3d030107046d306b02" +
            "010104207ce96cbc7fe72e987c6f194afab1b219d26eed446e6b462ed51a8f2a" +
            "c0619c53a14403420004fe472ca04f4020a2231afd0896449242f4b0469a0ef7" +
            "b50995492c174f8c2dac3aeb23b116f553ef5f36fe829d27c1e2c197f19a2184" +
            "b304ae993f320dd32717"),

    // This certificate is generated by the below commands:
    // openssl genpkey -algorithm ec -pkeyopt ec_paramgen_curve:P-256 \
    //     -pkeyopt ec_param_enc:named_curve -out key.pem
    // openssl req -new -key key.pem -subj "/CN=EXAMPLE" -sha256 -out csr.pem
    // openssl x509 -req -CAcreateserial -days 7300 -in csr.pem -sha256 \
    //     -CA CA.cer -CAkey CA.key -out cer.pem
    // Actually the CA is EXAMPLE_RSA_2048_SHA256
    EXAMPLE_EC_RSA_PRIME256V1_SHA256(
            KeyAlgorithm.EC,
            "-----BEGIN CERTIFICATE-----\n" +
            "MIIB3zCByAIUWm9wgVB1TgdT5lpGNNkWBzuclKQwDQYJKoZIhvcNAQELBQAwEjEQ\n" +
            "MA4GA1UEAwwHRVhBTVBMRTAeFw0xOTAyMjcwODA0MTNaFw0zOTAyMjIwODA0MTNa\n" +
            "MBIxEDAOBgNVBAMMB0VYQU1QTEUwWTATBgcqhkjOPQIBBggqhkjOPQMBBwNCAASp\n" +
            "6YAqTEEjuMlG+vKl8XPo2T2wgqY6t+j1R5ySC0YiGesfrwVLTM4V+Ey9PKHoEIVK\n" +
            "kWNUF5Sb2JdrYIuzb5WdMA0GCSqGSIb3DQEBCwUAA4IBAQBPrIScxw5Nx4DhT5GL\n" +
            "ngyNBOun0yAwqrxQ3LPheMuN7CH4qehFPDx8MHhmFFjEIDKVRbEEgxiCJAgca7qD\n" +
            "uLCfESM8KU4bkV4Pjx7/OEQZ3AkQ0UwDvDr/DypPg7TLLyF979OQo+fEaqWKH8Q4\n" +
            "8Ot8+VUFuwpYhHQlkoPnwFKIuCfDGwYmmHP2btlZ5qBuDDzdo1JVGF8pJ943cfA8\n" +
            "zRBJGKw8MMJXlfk3yiDSKMji0106SFuGwFJfkrdUnZ+hpeJ7rrrqW7jwLIil8PKf\n" +
            "Z41UjYM4Ut/6O5SFqueBsC6yxfzrJbd8UZ7ZkfagWMr/AXLK1Sm3ICSPHsQW30mH\n" +
            "uX+T\n" +
            "-----END CERTIFICATE-----",
            "308187020100301306072a8648ce3d020106082a8648ce3d030107046d306b02" +
            "01010420f1f944e1fc4bd7013b157db5fed23b84a4a1cd3d1a22f40746353185" +
            "c0d8684da14403420004a9e9802a4c4123b8c946faf2a5f173e8d93db082a63a" +
            "b7e8f5479c920b462219eb1faf054b4cce15f84cbd3ca1e810854a9163541794" +
            "9bd8976b608bb36f959d");

    public final KeyAlgorithm keyAlgorithm;
    public final String certMaterials;
    public final String privKeyMaterials;

    private Cert(
            KeyAlgorithm keyAlgorithm,
            String certMaterials,
            String privKeyMaterials) {
        this.keyAlgorithm = keyAlgorithm;
        this.certMaterials = certMaterials;
        this.privKeyMaterials = privKeyMaterials;
    }

    // Two certificates (mainCert and exampleCert) are selected to respect the
    // specified cipher suite. SNI-associated cases specify exampleCert as desired.
    public static Cert[] getCerts(CipherSuite cipherSuite) {
        Cert mainCert = Cert.ECDSA_PRIME256V1_SHA256;
        Cert exampleCert = Cert.EXAMPLE_ECDSA_PRIME256V1_SHA256;
        if (cipherSuite.keyExAlgorithm == KeyExAlgorithm.ECDHE_RSA
                || cipherSuite.keyExAlgorithm == KeyExAlgorithm.DHE_RSA
                || cipherSuite.keyExAlgorithm == KeyExAlgorithm.RSA) {
            mainCert = Cert.RSA_2048_SHA256;
            exampleCert = Cert.EXAMPLE_RSA_2048_SHA256;
        } else if (cipherSuite.keyExAlgorithm == KeyExAlgorithm.ECDH_RSA) {
            mainCert = Cert.EC_RSA_PRIME256V1_SHA256;
            exampleCert = Cert.EXAMPLE_EC_RSA_PRIME256V1_SHA256;
        } else if (cipherSuite.keyExAlgorithm == KeyExAlgorithm.DHE_DSS) {
            mainCert = Cert.DSA_2048_SHA256;
            exampleCert = Cert.EXAMPLE_DSA_2048_SHA256;
        }
        System.out.printf("mainCert=%s, exampleCert=%s%n",
                mainCert, exampleCert);
        return new Cert[] { mainCert, exampleCert };
    }
}
