/* -*-mode:c; c-style:k&r; c-basic-offset:4; -*- */
/* Balsa E-Mail Client
 *
 * Copyright (C) 1997-2018 Stuart Parmenter and others,
 *                         See the file AUTHORS for a list.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <https://www.gnu.org/licenses/>.
 */

#if defined(HAVE_CONFIG_H) && HAVE_CONFIG_H
#	include "config.h"
#endif                          /* HAVE_CONFIG_H */

#if defined ENABLE_AUTOCRYPT

#include <stdlib.h>
#include <glib/gi18n.h>
#include <glib/gstdio.h>
#include <sqlite3.h>
#include "libbalsa-gpgme.h"
#include "libbalsa-gpgme-keys.h"
#include "libbalsa-gpgme-widgets.h"
#include "identity.h"
#include "geometry-manager.h"
#include "autocrypt.h"


#ifdef G_LOG_DOMAIN
#  undef G_LOG_DOMAIN
#endif
#define G_LOG_DOMAIN "autocrypt"


/* the autocrypt SQL table contains the following data:
 * addr: email address
 * last_seen: time_t value when the last message from addr has been seen
 * ac_timestamp: time_t value when the last message with a valid Autocrypt header from addr has been seen
 * pubkey: raw (binary) public key data
 * fingerprint: the fingerprint of pubkey, stored to avoid frequently importing pubkey into a temporary context
 * expires: the expiry time of pubkey (0 for never), stored to avoid frequently importing pubkey into a temporary context
 * prefer_encrypt: TRUE (1) if the prefer-encrypt=mutual attribute was given in the latest Autocrypt header
 *
 * notes: SQLite stores BOOLEAN as INTEGER
 *        We do not support key gossip, so storing everything in a flat table is sufficient */
#define DB_SCHEMA								\
	"PRAGMA auto_vacuum = 1;"					\
	"CREATE TABLE autocrypt("					\
		"addr TEXT PRIMARY KEY NOT NULL, "		\
		"last_seen BIGINT, "					\
		"ac_timestamp BIGINT, "					\
		"pubkey BLOB NOT NULL, "				\
		"fingerprint TEXT NOT NULL, "			\
		"expires BIGINT NOT NULL, "				\
		"prefer_encrypt BOOLEAN DEFAULT 0);"


#define NUM_QUERIES								6U


struct _AutocryptData {
	gchar *addr;
	time_t last_seen;
	time_t ac_timestamp;
	GBytes *keydata;
	gchar *fingerprint;
	time_t expires;
	gboolean prefer_encrypt;
};

typedef struct _AutocryptData AutocryptData;


enum {
	AC_ADDRESS_COLUMN = 0,
	AC_LAST_SEEN_COLUMN,
	AC_TIMESTAMP_COLUMN,
	AC_PREFER_ENCRYPT_COLUMN,
	AC_KEY_PTR_COLUMN,
	AC_DB_VIEW_COLUMNS
};


static void autocrypt_close(void);
static AutocryptData *scan_autocrypt_headers(GList * const  header_list,
											 const gchar   *from_addr)
	G_GNUC_WARN_UNUSED_RESULT;
static AutocryptData *parse_autocrypt_header(const gchar *value)
	G_GNUC_WARN_UNUSED_RESULT;
static gboolean eval_autocrypt_attr(const gchar   *attr,
									const gchar   *value,
									gboolean      *seen,
									AutocryptData *target);
static void add_or_update_user_info(const AutocryptData  *user_info,
									time_t                date_header,
									gboolean              update,
									GError              **error);
static void update_last_seen(const gchar  *addr,
							 time_t        date_header,
							 GError      **error);
static AutocryptData *autocrypt_user_info(const gchar  *mailbox,
										  GError      **error)
	G_GNUC_WARN_UNUSED_RESULT;
static void autocrypt_free(AutocryptData *data);
static AutocryptRecommend autocrypt_check_ia_list(gpgme_ctx_t           gpgme_ctx,
												  InternetAddressList  *recipients,
												  time_t                ref_time,
												  GList               **missing_keys,
												  GError              **error);
static gboolean key_button_event_press_cb(GtkWidget      *widget,
						 				  GdkEventButton *event,
										  gpointer        data);


static sqlite3 *autocrypt_db = NULL;
static sqlite3_stmt *query[NUM_QUERIES] = { NULL, NULL, NULL, NULL, NULL, NULL };
G_LOCK_DEFINE_STATIC(db_mutex);


/* documentation: see header file */
gboolean
autocrypt_init(GError **error)
{
	static const gchar * const prepare_statements[NUM_QUERIES] = {
		"SELECT * FROM autocrypt WHERE LOWER(addr) = ?",
		"INSERT INTO autocrypt VALUES (?1, ?2, ?2, ?3, ?4, ?5, ?6)",
		"UPDATE autocrypt SET last_seen = MAX(?2, last_seen), ac_timestamp = ?2, pubkey = ?3, fingerprint = ?4,"
		" expires = ?5, prefer_encrypt = ?6 WHERE addr = ?1",
		"UPDATE autocrypt SET last_seen = ?2 WHERE addr = ?1 AND last_seen < ?2 AND ac_timestamp < ?2",
		"SELECT pubkey FROM autocrypt WHERE fingerprint LIKE ?",
		"SELECT addr, last_seen, ac_timestamp, prefer_encrypt, pubkey FROM autocrypt ORDER BY LOWER(addr) ASC"
	};
	gboolean result;

	G_LOCK(db_mutex);
	if (autocrypt_db == NULL) {
		gchar *db_path;
		gboolean require_init;
		int sqlite_res;

		/* ensure that the config folder exists, otherwise Balsa will throw an error on first use */
		libbalsa_assure_balsa_dir();

		db_path = g_build_filename(g_get_home_dir(), ".balsa", "autocrypt.db", NULL);
		require_init = (g_access(db_path, R_OK + W_OK) != 0);
		sqlite_res = sqlite3_open(db_path, &autocrypt_db);
		if (sqlite_res == SQLITE_OK) {
			guint n;

			/* write the schema if the database is new */
			if (require_init) {
				sqlite_res = sqlite3_exec(autocrypt_db, DB_SCHEMA, NULL, NULL, NULL);
			}

			/* always vacuum the database */
			if (sqlite_res == SQLITE_OK) {
				sqlite_res = sqlite3_exec(autocrypt_db, "VACUUM", NULL, NULL, NULL);
			}

			/* prepare statements */
			for (n = 0U; (sqlite_res == SQLITE_OK) && (n < NUM_QUERIES); n++) {
				sqlite_res = sqlite3_prepare_v2(autocrypt_db, prepare_statements[n], -1, &query[n], NULL);
			}
		}
		G_UNLOCK(db_mutex);

		/* error checks... */
		if (sqlite_res != SQLITE_OK) {
			g_set_error(error, AUTOCRYPT_ERROR_QUARK, sqlite_res, _("cannot initialise Autocrypt database “%s”: %s"), db_path,
				sqlite3_errmsg(autocrypt_db));
			autocrypt_close();
			result = FALSE;
		} else {
			atexit(autocrypt_close);
			result = TRUE;
		}
		g_free(db_path);
	} else {
		G_UNLOCK(db_mutex);
		result = TRUE;
	}

	return result;
}


/* documentation: see header file */
void
autocrypt_from_message(LibBalsaMessage  *message,
					   GError          **error)
{
	const gchar *from_addr;
	AutocryptData *autocrypt;
        LibBalsaMessageHeaders *headers;

	g_return_if_fail(LIBBALSA_IS_MESSAGE(message));
        headers = libbalsa_message_get_headers(message);
	g_return_if_fail(headers != NULL);
	g_return_if_fail(headers->from != NULL);
	g_return_if_fail(headers->content_type != NULL);
	g_return_if_fail(autocrypt_db != NULL);

	// FIXME - we should ignore spam - how can we detect it?

	/* check for content types which shall be ignored
	 * Note: see Autocrypt Level 1 standard, section 2.3 (https://autocrypt.org/level1.html#updating-autocrypt-peer-state) for
	 *       details about this and the following checks which may result in completely ignoring the message. */
	if (autocrypt_ignore(headers->content_type)) {
		g_debug("ignore %s/%s", g_mime_content_type_get_media_type(headers->content_type),
			g_mime_content_type_get_media_subtype(headers->content_type));
		return;
	}

	/* check for exactly one From: mailbox address - others shall be ignored */
	if ((internet_address_list_length(headers->from) != 1) ||
		!INTERNET_ADDRESS_IS_MAILBOX(internet_address_list_get_address(headers->from, 0))) {
		g_debug("require exactly one From: address, ignored");
		return;
	}

	/* ignore messages without a Date: header or with a date in the future */
	if ((headers->date == 0) || (headers->date > time(NULL))) {
		g_debug("no Date: header or value in the future, ignored");
		return;
	}

	/* get the From: address (is a mailbox, checked above) */
	from_addr =
		internet_address_mailbox_get_addr(INTERNET_ADDRESS_MAILBOX(internet_address_list_get_address(headers->from, 0)));
	g_debug("message from '%s', date %ld", from_addr, headers->date);

	/* scan for Autocrypt headers */
	autocrypt = scan_autocrypt_headers(headers->user_hdrs, from_addr);

    /* update the database */
    G_LOCK(db_mutex);
    if (autocrypt != NULL) {
    	AutocryptData *db_info;

    	db_info = autocrypt_user_info(autocrypt->addr, error);
    	if (db_info != NULL) {
    		if (headers->date > db_info->ac_timestamp) {
        		add_or_update_user_info(autocrypt, headers->date, TRUE, error);
    		} else {
    			g_info("message timestamp %ld not newer than autocrypt db timestamp %ld, ignore message",
    				(long) headers->date, (long) db_info->ac_timestamp);
    		}
    		autocrypt_free(db_info);
    	} else {
    		add_or_update_user_info(autocrypt, headers->date, FALSE, error);
    	}
    	autocrypt_free(autocrypt);
    } else {
    	update_last_seen(from_addr, headers->date, error);
    }
    G_UNLOCK(db_mutex);
}


/* documentation: see header file */
gchar *
autocrypt_header(LibBalsaIdentity *identity, GError **error)
{
	const gchar *mailbox;
	gchar *use_fpr = NULL;
	gchar *result = NULL;
        InternetAddress *ia;
        const gchar *force_gpg_key_id;
        AutocryptMode autocrypt_mode;

	g_return_val_if_fail(identity != NULL, NULL);
	autocrypt_mode = libbalsa_identity_get_autocrypt_mode(identity);
	g_return_val_if_fail(autocrypt_mode != AUTOCRYPT_DISABLE, NULL);

	ia = libbalsa_identity_get_address(identity);
	mailbox = internet_address_mailbox_get_addr(INTERNET_ADDRESS_MAILBOX(ia));

	/* no key fingerprint has been passed - try to find the fingerprint of a secret key matching the passed mailbox */
        force_gpg_key_id = libbalsa_identity_get_force_gpg_key_id(identity);
	if ((force_gpg_key_id == NULL) || (force_gpg_key_id[0] == '\0')) {
		gpgme_ctx_t ctx;

		ctx = libbalsa_gpgme_new_with_proto(GPGME_PROTOCOL_OpenPGP, NULL, NULL, error);
		if (ctx != NULL) {
			GList *keys = NULL;

			libbalsa_gpgme_list_keys(ctx, &keys, NULL, mailbox, TRUE, FALSE, FALSE, error);
			if (keys != NULL) {
				gpgme_key_t key = (gpgme_key_t) keys->data;

				if ((key != NULL) && (key->subkeys != NULL)) {
					use_fpr = g_strdup(key->subkeys->fpr);
				}
				g_list_free_full(keys, (GDestroyNotify) gpgme_key_release);
			}
			gpgme_release(ctx);
		}

		if (use_fpr == NULL) {
			g_set_error(error, AUTOCRYPT_ERROR_QUARK, -1,
				_("No usable private key for “%s” found! Please create a key or disable Autocrypt."), mailbox);
		} else {
			g_debug("found fingerprint %s for '%s'", use_fpr, mailbox);
		}
	} else {
		use_fpr = g_strdup(force_gpg_key_id);
	}

	if (use_fpr != NULL) {
		gchar *keydata;

		keydata = libbalsa_gpgme_export_autocrypt_key(use_fpr, mailbox, error);
		g_free(use_fpr);
		if (keydata != NULL) {
			GString *buffer;
			gssize ins_fws;

			buffer = g_string_new(NULL);
			g_string_append_printf(buffer, "addr=%s;", mailbox);
			if (autocrypt_mode == AUTOCRYPT_PREFER_ENCRYPT) {
				g_string_append(buffer, "prefer-encrypt=mutual;");
			}
			g_string_append_printf(buffer, "keydata=%s", keydata);
			for (ins_fws = 66; ins_fws < (gssize) buffer->len; ins_fws += 78) {
				g_string_insert(buffer, ins_fws, "\n\t");
			}
			result = g_string_free(buffer, FALSE);
			g_free(keydata);
		}
	}

	return result;
}


/* documentation: see header file */
gboolean
autocrypt_ignore(GMimeContentType *content_type)
{
	g_return_val_if_fail(GMIME_IS_CONTENT_TYPE(content_type), TRUE);

	return g_mime_content_type_is_type(content_type, "multipart", "report") ||
		g_mime_content_type_is_type(content_type, "text", "calendar");
}


/* documentation: see header file */
GBytes *
autocrypt_get_key(const gchar *fingerprint, GError **error)
{
	gchar *param;
	int sqlite_res;
	GBytes *result = NULL;

	g_return_val_if_fail(fingerprint != NULL, NULL);

	/* prepend SQL "LIKE" wildcard */
	param = g_strconcat("%", fingerprint, NULL);

	sqlite_res = sqlite3_bind_text(query[4], 1, param, -1, SQLITE_STATIC);
	if (sqlite_res == SQLITE_OK) {
		sqlite_res = sqlite3_step(query[4]);
		if (sqlite_res == SQLITE_ROW) {
			result = g_bytes_new(sqlite3_column_blob(query[4], 0), sqlite3_column_bytes(query[4], 0));
			sqlite_res = sqlite3_step(query[4]);
		}

		if (sqlite_res != SQLITE_DONE) {
			g_set_error(error, AUTOCRYPT_ERROR_QUARK, sqlite_res, _("error reading Autocrypt data for “%s”: %s"), fingerprint,
				sqlite3_errmsg(autocrypt_db));
			if (result != NULL) {
				g_bytes_unref(result);
				result = NULL;
			}
		}
	} else {
		g_set_error(error, AUTOCRYPT_ERROR_QUARK, sqlite_res, _("error reading Autocrypt data for “%s”: %s"), fingerprint,
			sqlite3_errmsg(autocrypt_db));
	}
	sqlite3_reset(query[4]);
	g_free(param);

	return result;
}


/* documentation: see header file */
AutocryptRecommend
autocrypt_recommendation(InternetAddressList *recipients, GList **missing_keys, GError **error)
{
	AutocryptRecommend result;
	gpgme_ctx_t gpgme_ctx;

	g_return_val_if_fail(IS_INTERNET_ADDRESS_LIST(recipients), AUTOCRYPT_ENCR_DISABLE);

    /* create the gpgme context and set the protocol */
    gpgme_ctx = libbalsa_gpgme_new_with_proto(GPGME_PROTOCOL_OpenPGP, NULL, NULL, error);
    if (gpgme_ctx == NULL) {
    	result = AUTOCRYPT_ENCR_ERROR;
    } else {
    	result = autocrypt_check_ia_list(gpgme_ctx, recipients, time(NULL), missing_keys, error);
    	gpgme_release(gpgme_ctx);

    	if ((result == AUTOCRYPT_ENCR_ERROR) && (missing_keys != NULL) && (*missing_keys != NULL)) {
    		g_list_free_full(*missing_keys, (GDestroyNotify) g_bytes_unref);
    		*missing_keys = NULL;
    	}
    }

	return result;
}


/* documentation: see header file */
void
autocrypt_db_dialog_run(const gchar *date_string, GtkWindow *parent)
{
	GtkWidget *dialog;
	GtkWidget *vbox;
    GtkWidget *label;
    GtkWidget *scrolled_window;
    GtkWidget *tree_view;
    GtkListStore *model;
    GtkTreeSelection *selection;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
    GList *keys = NULL;
	int sqlite_res;

	dialog = gtk_dialog_new_with_buttons(_("Autocrypt database"), parent,
		GTK_DIALOG_DESTROY_WITH_PARENT | libbalsa_dialog_flags(), _("_Close"), GTK_RESPONSE_CLOSE, NULL);
	geometry_manager_attach(GTK_WINDOW(dialog), "AutocryptDB");

    vbox = gtk_box_new(GTK_ORIENTATION_VERTICAL, 12);
    gtk_container_add(GTK_CONTAINER(gtk_dialog_get_content_area(GTK_DIALOG(dialog))), vbox);
    gtk_widget_set_vexpand (vbox, TRUE);
    label = gtk_label_new(_("Double-click key to show details"));
    gtk_widget_set_halign(label, GTK_ALIGN_START);
    gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, TRUE, 0);

    scrolled_window = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolled_window), GTK_SHADOW_ETCHED_IN);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start(GTK_BOX(vbox), scrolled_window, TRUE, TRUE, 0);

    model = gtk_list_store_new(AC_DB_VIEW_COLUMNS, G_TYPE_STRING,	/* address */
    	G_TYPE_STRING,												/* formatted last seen timestamp */
		G_TYPE_STRING,												/* formatted last Autocrypt message timestamp */
		G_TYPE_BOOLEAN,												/* user prefers encrypted messages */
		G_TYPE_POINTER);											/* key */

    tree_view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(model));
    g_signal_connect(tree_view, "button_press_event", G_CALLBACK(key_button_event_press_cb), dialog);
    gtk_container_add(GTK_CONTAINER(scrolled_window), tree_view);
    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(tree_view));
    gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);

    /* add the keys */
    sqlite_res = sqlite3_step(query[5]);
    while (sqlite_res == SQLITE_ROW) {
    	gchar *last_seen_buf;
    	gchar *last_ac_buf;
    	GBytes *key;
        GtkTreeIter iter;

    	last_seen_buf = libbalsa_date_to_utf8(sqlite3_column_int64(query[5], 1), date_string);
    	last_ac_buf = libbalsa_date_to_utf8(sqlite3_column_int64(query[5], 2), date_string);
    	key = g_bytes_new(sqlite3_column_blob(query[5], 4), sqlite3_column_bytes(query[5], 4));
    	keys = g_list_prepend(keys, key);

		gtk_list_store_append(model, &iter);
		gtk_list_store_set(model, &iter,
			AC_ADDRESS_COLUMN, sqlite3_column_text(query[5], 0),
			AC_LAST_SEEN_COLUMN, last_seen_buf,
			AC_TIMESTAMP_COLUMN, last_ac_buf,
			AC_PREFER_ENCRYPT_COLUMN, sqlite3_column_int(query[5], 3),
			AC_KEY_PTR_COLUMN, key,
			-1);
		g_free(last_seen_buf);
		g_free(last_ac_buf);

    	sqlite_res = sqlite3_step(query[5]);
    }
    sqlite3_reset(query[5]);

    /* set up the tree view */
    g_object_unref(model);

	renderer = gtk_cell_renderer_text_new();
	column = gtk_tree_view_column_new_with_attributes(_("Mailbox"), renderer, "text", 0, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), column);
	gtk_tree_view_column_set_resizable(column, TRUE);

	renderer = gtk_cell_renderer_text_new();
	column = gtk_tree_view_column_new_with_attributes(_("Last seen"), renderer, "text", 1, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), column);
	gtk_tree_view_column_set_resizable(column, TRUE);

	renderer = gtk_cell_renderer_text_new();
	column = gtk_tree_view_column_new_with_attributes(_("Last Autocrypt message"), renderer, "text", 2, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), column);
	gtk_tree_view_column_set_resizable(column, TRUE);

	renderer = gtk_cell_renderer_toggle_new();
	column = gtk_tree_view_column_new_with_attributes(_("Prefer encryption"), renderer, "active", 3, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), column);
	gtk_tree_view_column_set_resizable(column, TRUE);
	gtk_widget_show_all(vbox);

	(void) gtk_dialog_run(GTK_DIALOG(dialog));
	gtk_widget_destroy(dialog);
	g_list_free_full(keys, (GDestroyNotify) g_bytes_unref);
}


static AutocryptRecommend
autocrypt_check_ia_list(gpgme_ctx_t           gpgme_ctx,
						InternetAddressList  *recipients,
						time_t                ref_time,
						GList               **missing_keys,
						GError              **error)
{
	AutocryptRecommend result = AUTOCRYPT_ENCR_AVAIL_MUTUAL;
	gint i;

	for (i = 0; (result > AUTOCRYPT_ENCR_DISABLE) && (i < internet_address_list_length(recipients)); i++) {
    	InternetAddress *ia = internet_address_list_get_address(recipients, i);

    	/* check all entries in the list, handle groups recursively */
    	if (INTERNET_ADDRESS_IS_GROUP(ia)) {
    		result = autocrypt_check_ia_list(gpgme_ctx, INTERNET_ADDRESS_GROUP(ia)->members, ref_time, missing_keys, error);
    	} else {
    		AutocryptData *autocrypt_user;
    		const gchar *mailbox;

    		mailbox = INTERNET_ADDRESS_MAILBOX(ia)->addr;
    		autocrypt_user = autocrypt_user_info(mailbox, NULL);
    		if (autocrypt_user == NULL) {
        		GList *keys = NULL;

    			/* check if we have a public key, keep the state if we found one, disable if not */
        		if (libbalsa_gpgme_list_keys(gpgme_ctx, &keys, NULL, mailbox, FALSE, FALSE, FALSE, error)) {
        			if (keys != NULL) {
        				g_list_free_full(keys, (GDestroyNotify) gpgme_key_unref);
        				g_debug("'%s': found in public key ring, overall status %d", mailbox, result);
        			} else {
        				result = AUTOCRYPT_ENCR_DISABLE;
        				g_debug("'%s': not in Autocrypt db or public key ring, overall status %d", mailbox, result);
        			}
        		} else {
        			result = AUTOCRYPT_ENCR_ERROR;
        		}
    		} else {
    			/* we found Autocrypt data for this user */
    			if ((autocrypt_user->expires > 0) && (autocrypt_user->expires <= ref_time)) {
    				result = AUTOCRYPT_ENCR_DISABLE;		/* key has expired */
    			} else if (autocrypt_user->ac_timestamp < (autocrypt_user->last_seen - (35 * 24 * 60 * 60))) {
    				result = MIN(result, AUTOCRYPT_ENCR_DISCOURAGE);	/* Autocrypt timestamp > 35 days older than last seen */
    			} else if (autocrypt_user->prefer_encrypt) {
    				result = MIN(result, AUTOCRYPT_ENCR_AVAIL_MUTUAL);	/* user requested "prefer-encrypt=mutual" */
    			} else {
    				result = MIN(result, AUTOCRYPT_ENCR_AVAIL);			/* user did not request "prefer-encrypt=mutual" */
    			}

    			/* check if the Autocrypt key is already in the key ring, add it to the list of missing ones otherwise */
    			if (missing_keys != NULL) {
            		GList *keys = NULL;

            		if (libbalsa_gpgme_list_keys(gpgme_ctx, &keys, NULL, autocrypt_user->fingerprint, FALSE, FALSE, FALSE, error)) {
            			if (keys != NULL) {
            				g_list_free_full(keys, (GDestroyNotify) gpgme_key_unref);
            			} else {
            				*missing_keys = g_list_prepend(*missing_keys, g_bytes_ref(autocrypt_user->keydata));
            			}
            		} else {
            			result = AUTOCRYPT_ENCR_ERROR;
            		}
    			}
    			autocrypt_free(autocrypt_user);
    			g_debug("'%s': found in Autocrypt db, overall status %d", mailbox, result);
    		}
    	}
	}

	return result;
}


static void
autocrypt_free(AutocryptData *data)
{
	if (data != NULL) {
		g_free(data->addr);
		g_free(data->fingerprint);
		if (data->keydata) {
			g_bytes_unref(data->keydata);
		}
		g_free(data);
	}
}


static void
autocrypt_close(void)
{
	guint n;

	g_debug("closing Autocrypt database");
	G_LOCK(db_mutex);
	for (n = 0U; n < NUM_QUERIES; n++) {
		sqlite3_finalize(query[n]);
		query[n] = NULL;
	}
	sqlite3_close(autocrypt_db);
	autocrypt_db = NULL;
	G_UNLOCK(db_mutex);
}


/** \brief Extract Autocrypt data from message headers
 *
 * \param header_list list of headers pointing to gchar** (name, value) pairs
 * \param from_addr sender mailbox extracted from the From: header
 * \return the data extracted from the Autocrypt header, or NULL if no valid data is present
 *
 * The following rules apply according to the Autocrypt Level 1 standard:
 * - invalid Autocrypt headers are just discarded, but checking for more Autocrypt headers continues (see section 2.1 <em>The
 *   Autocrypt Header</em>, https://autocrypt.org/level1.html#the-autocrypt-header);
 * - if the \em addr attribute of an otherwise valid Autocrypt header does not match the mailbox extracted from the From: message
 *   header, the Autocrypt header shall be treated as being invalid and discarded (see section 2.1);
 * - if more than one valid Autocrypt header is present, \em all Autocrypt headers shall be discarded (see section 2.3 <em>Updating
 *   Autocrypt Peer State</em>, https://autocrypt.org/level1.html#updating-autocrypt-peer-state);
 *
 * Thus, this function returns a newly allocated Autocrypt data structure iff the passed headers list contains exactly \em one valid
 * Autocrypt header.
 */
static AutocryptData *
scan_autocrypt_headers(GList * const header_list, const gchar *from_addr)
{
	GList *header;
	AutocryptData *result = NULL;

	for (header = header_list; header != NULL; header = header->next) {
		const gchar **header_parts = (const gchar **) header->data;

		if ((g_ascii_strcasecmp(header_parts[0], "Autocrypt") == 0) && (header_parts[1] != NULL)) {
			AutocryptData *new_data;

			new_data = parse_autocrypt_header(header_parts[1]);
			if (new_data != NULL) {
				if (result == NULL) {
			    	if (g_ascii_strcasecmp(new_data->addr, from_addr) != 0) {
			    		g_info("Autocrypt header for '%s' in message from '%s', ignore header", new_data->addr, from_addr);
			    		autocrypt_free(new_data);
			    	} else {
			    		result = new_data;
			    	}
				} else {
					g_info("more than one valid Autocrypt header");
					autocrypt_free(result);
					autocrypt_free(new_data);
					return NULL;
				}
			}
		}
	}

	return result;
}


static AutocryptData *
parse_autocrypt_header(const gchar *value)
{
	gchar **attributes;
	gboolean attr_seen[3] = { FALSE, FALSE, FALSE };
	AutocryptData *new_data;
	gint n;
	gboolean broken;

	new_data = g_new0(AutocryptData, 1U);
	attributes = g_strsplit(value, ";", -1);
	if (attributes == NULL) {
		g_info("empty Autocrypt header");
		broken = TRUE;
	} else {
		broken = FALSE;
	}

	for (n = 0; !broken && (attributes[n] != NULL); n++) {
		gchar **items;

		items = g_strsplit(attributes[n], "=", 2);
		if ((items == NULL) || (items[0] == NULL) || (items[1] == NULL)) {
			g_info("bad Autocrypt header attribute");
			broken = TRUE;
		} else {
			broken = !eval_autocrypt_attr(g_strstrip(items[0]), g_strstrip(items[1]), attr_seen, new_data);
		}
		g_strfreev(items);
	}
	g_strfreev(attributes);

	if (!broken) {
		if (!attr_seen[0] || !attr_seen[2]) {
			g_info("missing mandatory Autocrypt header attribute");
			broken = TRUE;
		}
	}

	/* try to import the key into a temporary context */
	if (!broken) {
		gboolean success = FALSE;
		gpgme_ctx_t ctx;

		ctx = libbalsa_gpgme_new_with_proto(GPGME_PROTOCOL_OpenPGP, NULL, NULL, NULL);
		if (ctx != NULL) {
			gchar *temp_dir = NULL;

			if (!libbalsa_mktempdir(&temp_dir)) {
				g_warning("Failed to create a temporary folder");
			} else {
				GList *keys = NULL;
				GError *error = NULL;
				guint bad_keys = 0U;

				success = libbalsa_gpgme_ctx_set_home(ctx, temp_dir, &error) &&
					libbalsa_gpgme_import_bin_key(ctx, new_data->keydata, NULL, &error) &&
					libbalsa_gpgme_list_keys(ctx, &keys, &bad_keys, NULL, FALSE, FALSE, FALSE, &error);
				if (success && (keys != NULL) && (keys->next == NULL)) {
					gpgme_key_t key = (gpgme_key_t) keys->data;

					if ((key != NULL) && (key->subkeys != NULL)) {
						new_data->fingerprint = g_strdup(key->subkeys->fpr);
						new_data->expires = key->subkeys->expires;
					}
				} else {
					g_warning("Failed to import or list key data for '%s': %s (%u keys, %u bad)", new_data->addr,
						(error != NULL) ? error->message : "unknown", (keys != NULL) ? g_list_length(keys) : 0U, bad_keys);
				}
				g_clear_error(&error);

				g_list_free_full(keys, (GDestroyNotify) gpgme_key_release);
				libbalsa_delete_directory_contents(temp_dir);
				g_rmdir(temp_dir);
			}

			gpgme_release(ctx);
		}
	}

	/* check if a broken header has been detected, or if importing the key failed */
	if (broken || (new_data->fingerprint == NULL)) {
		autocrypt_free(new_data);
		new_data = NULL;
	} else {
		g_debug("valid Autocrypt header for '%s', prefer encrypt %d, key fingerprint %s", new_data->addr, new_data->prefer_encrypt,
			new_data->fingerprint);
	}

	return new_data;
}


static gboolean
eval_autocrypt_attr(const gchar *attr, const gchar *value, gboolean *seen, AutocryptData *target)
{
	gboolean result = FALSE;

	if (seen[2]) {
		g_info("broken Autocrypt header, extra attribute after keydata");
	} else if (strcmp(attr, "addr") == 0) {
		if (seen[0]) {
			g_info("duplicated Autocrypt header attribute 'addr'");
		} else {
			seen[0] = TRUE;
			/* note: not exactly the canonicalisation as required by the Autocrypt standard, but should work in all practical use
			 * cases... */
			target->addr = g_ascii_strdown(value, -1);
			result = TRUE;
		}
	} else if (strcmp(attr, "prefer-encrypt") == 0) {
		if (seen[1]) {
			g_info("duplicated Autocrypt header attribute 'addr'");
		} else {
			seen[1] = TRUE;
			if (strcmp(value, "mutual") == 0) {
				target->prefer_encrypt = TRUE;
				result = TRUE;
			} else {
				g_info("bad value '%s' for Autocrypt header attribute 'prefer-encrypt'", value);
			}
		}
	} else if (strcmp(attr, "keydata") == 0) {
		guchar *data;
		gsize len;

		seen[2] = TRUE;
		data = g_base64_decode(value, &len);
		if (data == NULL) {
			g_info("invalid keydata in Autocrypt header");
		} else {
			target->keydata = g_bytes_new_take(data, len);
			result = TRUE;
		}
	} else if (attr[0] == '_') {
		g_debug("ignoring non-critical Autocrypt header attribute '%s'", attr);
		result = TRUE;		/* note that this is no error */
	} else {
		g_info("unexpected Autocrypt header attribute '%s'", attr);
	}

	return result;
}


static AutocryptData *
autocrypt_user_info(const gchar *mailbox, GError **error)
{
	int sqlite_res;
	AutocryptData *user_info = NULL;

	g_return_val_if_fail((mailbox != NULL) && (autocrypt_db != NULL), NULL);

	sqlite_res = sqlite3_bind_text(query[0], 1, mailbox, -1, SQLITE_STATIC);
	if (sqlite_res == SQLITE_OK) {
		sqlite_res = sqlite3_step(query[0]);
		if (sqlite_res == SQLITE_ROW) {
			user_info = g_new0(AutocryptData, 1U);
			user_info->addr = g_strdup((const gchar *) sqlite3_column_text(query[0], 0));
			user_info->last_seen = sqlite3_column_int64(query[0], 1);
			user_info->ac_timestamp = sqlite3_column_int64(query[0], 2);
			user_info->keydata = g_bytes_new(sqlite3_column_blob(query[0], 3), sqlite3_column_bytes(query[0], 3));
			user_info->fingerprint = g_strdup((const gchar *) sqlite3_column_text(query[0], 4));
			user_info->expires = sqlite3_column_int64(query[0], 5);
			user_info->prefer_encrypt = (sqlite3_column_int(query[0], 6) != 0);
			sqlite_res = sqlite3_step(query[0]);
		}

		if (sqlite_res != SQLITE_DONE) {
			g_set_error(error, AUTOCRYPT_ERROR_QUARK, sqlite_res, _("error reading Autocrypt data for “%s”: %s"), mailbox,
				sqlite3_errmsg(autocrypt_db));
			autocrypt_free(user_info);
			user_info = NULL;
		}
	} else {
		g_set_error(error, AUTOCRYPT_ERROR_QUARK, sqlite_res, _("error reading Autocrypt data for “%s”: %s"), mailbox,
			sqlite3_errmsg(autocrypt_db));
	}
	sqlite3_reset(query[0]);

	return user_info;
}


static void
add_or_update_user_info(const AutocryptData *user_info, time_t date_header, gboolean update, GError **error)
{
	guint query_idx;
	gconstpointer keyvalue;
	gsize keysize;

	query_idx = update ? 2 : 1;
	keyvalue = g_bytes_get_data(user_info->keydata, &keysize);
	if ((sqlite3_bind_text(query[query_idx], 1, user_info->addr, -1, SQLITE_STATIC) != SQLITE_OK) ||
		(sqlite3_bind_int64(query[query_idx], 2, date_header) != SQLITE_OK) ||
		(sqlite3_bind_blob(query[query_idx], 3, keyvalue, keysize, SQLITE_STATIC) != SQLITE_OK) ||
		(sqlite3_bind_text(query[query_idx], 4, user_info->fingerprint, -1, SQLITE_STATIC) != SQLITE_OK) ||
		(sqlite3_bind_int64(query[query_idx], 5, user_info->expires) != SQLITE_OK) ||
		(sqlite3_bind_int(query[query_idx], 6, user_info->prefer_encrypt) != SQLITE_OK) ||
		(sqlite3_step(query[query_idx]) != SQLITE_DONE)) {
		g_set_error(error, AUTOCRYPT_ERROR_QUARK, -1,
                        update ? _("update user “%s” failed: %s") : _("insert user “%s” failed: %s"),
			user_info->addr, sqlite3_errmsg(autocrypt_db));
	} else {
		g_debug("%s user '%s': %d", update ? "updated" : "inserted", user_info->addr, sqlite3_changes(autocrypt_db));
	}
	sqlite3_reset(query[query_idx]);
}


static void
update_last_seen(const gchar *addr, time_t date_header, GError **error)
{
	if ((sqlite3_bind_text(query[3], 1, addr, -1, SQLITE_STATIC) != SQLITE_OK) ||
		(sqlite3_bind_int64(query[3], 2, date_header) != SQLITE_OK) ||
		(sqlite3_step(query[3]) != SQLITE_DONE)) {
		g_set_error(error, AUTOCRYPT_ERROR_QUARK, -1, _("update user “%s” failed: %s"), addr, sqlite3_errmsg(autocrypt_db));
	} else {
		g_debug("updated last_seen for '%s': %d", addr, sqlite3_changes(autocrypt_db));
	}
	sqlite3_reset(query[3]);
}


static gboolean
key_button_event_press_cb(GtkWidget      *widget,
						  GdkEventButton *event,
						  gpointer        data)
{
    GtkTreeView *tree_view = GTK_TREE_VIEW(widget);
    GtkTreeSelection *selection = gtk_tree_view_get_selection(tree_view);
    GtkTreePath *path;
    GtkTreeIter iter;
    GtkTreeModel *model;

    g_return_val_if_fail(event != NULL, FALSE);
    if ((event->type != GDK_2BUTTON_PRESS) || event->window != gtk_tree_view_get_bin_window(tree_view)) {
        return FALSE;
    }

    if (gtk_tree_view_get_path_at_pos(tree_view, event->x, event->y, &path, NULL, NULL, NULL)) {
        if (!gtk_tree_selection_path_is_selected(selection, path)) {
            gtk_tree_view_set_cursor(tree_view, path, NULL, FALSE);
            gtk_tree_view_scroll_to_cell(tree_view, path, NULL, FALSE, 0, 0);
        }
        gtk_tree_path_free(path);
    }

    /* note: silently ignore all errors below... */
    if (gtk_tree_selection_get_selected(selection, &model, &iter)) {
		gpgme_ctx_t ctx;

		ctx = libbalsa_gpgme_new_with_proto(GPGME_PROTOCOL_OpenPGP, NULL, NULL, NULL);
		if (ctx != NULL) {
			gchar *temp_dir = NULL;

			if (libbalsa_mktempdir(&temp_dir)) {
				GBytes *key;
				GList *keys = NULL;
				gboolean success;

				gtk_tree_model_get(model, &iter, AC_KEY_PTR_COLUMN, &key, -1);
				success = libbalsa_gpgme_ctx_set_home(ctx, temp_dir, NULL) &&
					libbalsa_gpgme_import_bin_key(ctx, key, NULL, NULL) &&
					libbalsa_gpgme_list_keys(ctx, &keys, NULL, NULL, FALSE, FALSE, TRUE, NULL);
				if (success && (keys != NULL)) {
			    	GtkWidget *dialog;

			    	dialog = libbalsa_key_dialog(GTK_WINDOW(data), GTK_BUTTONS_CLOSE, (gpgme_key_t) keys->data, GPG_SUBKEY_CAP_ALL,
			    		NULL, NULL);
			    	(void) gtk_dialog_run(GTK_DIALOG(dialog));
			    	gtk_widget_destroy(dialog);
			    	g_list_free_full(keys, (GDestroyNotify) gpgme_key_release);
				}
				libbalsa_delete_directory_contents(temp_dir);
				g_rmdir(temp_dir);
			}

			gpgme_release(ctx);
		}
    }

    return TRUE;
}

#endif  /* ENABLE_AUTOCRYPT */
